/*******************************************************************************
 * Copyright (c) 2015, 2016 Bosch Software Innovations GmbH and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v1.0 which accompany this distribution.
 *   
 * The Eclipse Public License is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * The Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *   
 * Contributors:
 * Bosch Software Innovations GmbH - Please refer to git log
 *******************************************************************************/
package org.eclipse.vorto.codegen.protobuf.templates

import java.util.List
import org.eclipse.vorto.codegen.api.IFileTemplate
import org.eclipse.vorto.codegen.api.InvocationContext
import org.eclipse.vorto.codegen.utils.Utils
import org.eclipse.vorto.core.api.model.datatype.DictionaryPropertyType
import org.eclipse.vorto.core.api.model.datatype.ObjectPropertyType
import org.eclipse.vorto.core.api.model.datatype.PrimitivePropertyType
import org.eclipse.vorto.core.api.model.datatype.PrimitiveType
import org.eclipse.vorto.core.api.model.datatype.Property
import org.eclipse.vorto.core.api.model.datatype.PropertyType
import org.eclipse.vorto.core.api.model.functionblock.FunctionblockModel
import org.eclipse.vorto.core.api.model.functionblock.Param
import org.eclipse.vorto.core.api.model.functionblock.PrimitiveParam
import org.eclipse.vorto.core.api.model.functionblock.RefParam
import org.eclipse.vorto.core.api.model.functionblock.ReturnType
import org.eclipse.vorto.core.api.model.functionblock.ReturnPrimitiveType
import org.eclipse.vorto.core.api.model.functionblock.ReturnObjectType

class ProtobufTemplate implements IFileTemplate<FunctionblockModel> {

	override getFileName(FunctionblockModel element) {
		return element.name + ".proto"
	}

	override getPath(FunctionblockModel element) {
		return "";
	}

	override getContent(FunctionblockModel element, InvocationContext context) {
		'''
			// Generated by Vorto from «element.namespace».«element.name»:«element.version»
			syntax = "proto3";
			
			package «element.name.toLowerCase»
			
			service «element.name» {
				
				«IF element.functionblock.operations != null»
					«FOR operation : element.functionblock.operations»
						«IF operation.params != null»
							rpc «operation.name»(«operation.name.toFirstUpper»Request) returns «operation.name.toFirstUpper»Response {}
						«ENDIF»
					«ENDFOR»
				«ENDIF»
				«IF element.functionblock.status != null»
				rpc getStatus(StatusRequest) returns StatusResponse {}
				«ENDIF»
			}
			
			«IF element.functionblock.operations != null»
				«FOR operation : element.functionblock.operations»
					«IF operation.params != null»
						message «operation.name.toFirstUpper»Request {
							«generateMessagePropertiesFromParams(operation.params)»
						}
					«ENDIF»
					message «operation.name.toFirstUpper»Response {
						«generateMessagePropertiesFromReturnType(operation.returnType)»
					}
				«ENDFOR»
			«ENDIF»
			
			«IF element.functionblock.status != null»
				message StatusRequest {
					// empty
				}
				
				message StatusResponse {
					«generateMessageProperties(element.functionblock.status.properties)»
				}
			«ENDIF»
«««			
«««			«IF element.functionblock.configuration != null»
«««				message «element.name»Configuration {
«««					«generateMessageProperties(element.functionblock.configuration.properties)»
«««				}
«««			«ENDIF»
			
			«FOR entity : Utils.getReferencedEntities(element.functionblock)»
			message «entity.name» {
				«generateMessageProperties(entity.properties)»
			}
			«ENDFOR»
			
			«FOR enumeration : Utils.getReferencedEnums(element.functionblock)»
			enum «enumeration.name» {
				
				«IF enumeration.enums != null»
					«var counter = 0»
					«FOR literal : enumeration.enums»
						«literal.name» = «counter++»;
					«ENDFOR»
				«ENDIF»
			}
			«ENDFOR»
		'''
	}

	def generateMessageProperties(List<Property> properties) {
		'''
			«IF properties != null»
				«var counter = 1»
				«FOR property : properties»
					«IF property.multiplicity»repeated «ENDIF»«type(property.type)» «property.name» = «counter++»;
				«ENDFOR»
			«ENDIF»
		'''
	}

	def generateMessagePropertiesFromReturnType(ReturnType returnType) {
		'''
			«IF returnType != null && returnType instanceof ReturnPrimitiveType»
				«IF returnType.multiplicity»repeated «ENDIF»«toProtoPrimitive((returnType as ReturnPrimitiveType).returnType)» value = 1;
			«ENDIF»
			«IF returnType != null && returnType instanceof ReturnObjectType»
				«IF returnType.multiplicity»repeated «ENDIF»«(returnType as ReturnObjectType).returnType.name» value = 1;
			«ENDIF»
		'''
	}

	def generateMessagePropertiesFromParams(List<Param> params) {
		'''
			«IF params != null»
				«var counter = 1»
				«FOR param : params»
					«IF param.multiplicity»repeated «ENDIF»«type(param)» «param.name» = «counter++»;
				«ENDFOR»
			«ENDIF»
		'''
	}

	def String toProtoPrimitive(PrimitiveType primitiveType) {
		switch (primitiveType) {
			case (PrimitiveType.DOUBLE): {
				return "double"
			}
			case (PrimitiveType.FLOAT): {
				return "float"
			}
			case (PrimitiveType.INT): {
				return "int32"
			}
			case (PrimitiveType.LONG): {
				return "int64"
			}
			case (PrimitiveType.BOOLEAN): {
				return "bool"
			}
			case (PrimitiveType.BASE64_BINARY): {
				return "bytes"
			}
			case (PrimitiveType.STRING): {
				return "string"
			}
			default: {
				return "string"
			}
		}
	}
	
	def String type(Param param) {
		if (param instanceof PrimitiveParam) {
			return toProtoPrimitive((param as PrimitiveParam).getType);
		} else {
			return (param as RefParam).getType().name
		}
	}
	
	def String type(PropertyType type) {
		if (type instanceof PrimitivePropertyType) {
			return toProtoPrimitive((type as PrimitivePropertyType).getType);
		} else if (type instanceof DictionaryPropertyType){
			return "map<"+type((type as DictionaryPropertyType).keyType)+","+type((type as DictionaryPropertyType).valueType)+">";
		} else {
			return (type as ObjectPropertyType).getType().name
		}
	}
	
}
