/*******************************************************************************
 * Copyright (c) 2009, 2010 Obeo.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Obeo - initial API and implementation
 *******************************************************************************/
package org.eclipse.acceleo.internal.compatibility.parser.ast.ocl.environment;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.acceleo.common.utils.AcceleoNonStandardLibrary;
import org.eclipse.acceleo.internal.parser.ast.ocl.environment.AcceleoEnvironment;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EEnumLiteral;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EParameter;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.ECrossReferenceAdapter;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.ocl.AbstractTypeChecker;
import org.eclipse.ocl.Environment;
import org.eclipse.ocl.TypeChecker;
import org.eclipse.ocl.ecore.CallOperationAction;
import org.eclipse.ocl.ecore.Constraint;
import org.eclipse.ocl.ecore.EcoreFactory;
import org.eclipse.ocl.ecore.PrimitiveType;
import org.eclipse.ocl.ecore.SendSignalAction;
import org.eclipse.ocl.ecore.StringLiteralExp;
import org.eclipse.ocl.ecore.TypeExp;
import org.eclipse.ocl.expressions.CollectionKind;
import org.eclipse.ocl.options.ParsingOptions;
import org.eclipse.ocl.types.CollectionType;
import org.eclipse.ocl.types.TupleType;
import org.eclipse.ocl.utilities.TypedElement;

/**
 * This class will not compile under Eclipse Ganymede with OCL 1.2 installed. It requires OCL 1.3 and
 * shouldn't be called or instantiated under previous versions.
 * <p>
 * This code is meant to be called in Eclipse 3.5 and later only, and will <em>not</em> compile in earlier
 * versions. That is expected and will not provoke runtime errors.
 * </p>
 * 
 * @author <a href="mailto:laurent.goubet@obeo.fr">Laurent Goubet</a>
 */
public class AcceleoEnvironmentGalileo extends AcceleoEnvironment {
	/** Type checker created for this environment. */
	private AcceleoTypeChecker typeChecker;

	/**
	 * Delegates instantiation to the super constructor.
	 * 
	 * @param parent
	 *            Parent for this Acceleo environment.
	 */
	public AcceleoEnvironmentGalileo(
			Environment<EPackage, EClassifier, EOperation, EStructuralFeature, EEnumLiteral, EParameter, EObject, CallOperationAction, SendSignalAction, Constraint, EClass, EObject> parent) {
		super(parent);
		setOption(ParsingOptions.USE_BACKSLASH_ESCAPE_PROCESSING, Boolean.TRUE);
	}

	/**
	 * Delegates instantiation to the super-constructor.
	 * 
	 * @param oclEnvironmentResource
	 *            resource used to keep the OCL environment.
	 */
	public AcceleoEnvironmentGalileo(Resource oclEnvironmentResource) {
		super(oclEnvironmentResource);
		setOption(ParsingOptions.USE_BACKSLASH_ESCAPE_PROCESSING, Boolean.TRUE);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.ocl.AbstractEnvironment#createTypeChecker()
	 */
	@Override
	public TypeChecker<EClassifier, EOperation, EStructuralFeature> createTypeChecker() {
		if (typeChecker == null) {
			typeChecker = new AcceleoTypeChecker(this);
		}
		return typeChecker;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.acceleo.internal.parser.ast.ocl.environment.AcceleoEnvironment#dispose()
	 */
	@Override
	public void dispose() {
		super.dispose();
		typeChecker.dispose();
	}

	/**
	 * This will allow us to type our standard and non standard operations.
	 * 
	 * @author <a href="mailto:laurent.goubet@obeo.fr">Laurent Goubet</a>
	 */
	public class AcceleoTypeChecker extends AbstractTypeChecker<EClassifier, EOperation, EStructuralFeature, EParameter> {
		/**
		 * As we will infer type information for the non standard library, we'll have to keep references to
		 * the modified types.
		 */
		private final Map<EClassifier, Set<EClassifier>> alteredTypes = new HashMap<EClassifier, Set<EClassifier>>();

		/** This will allow us to maintain the subtypes hierarchy of our metamodel. */
		private final Map<EClassifier, Set<EClassifier>> subTypes = new HashMap<EClassifier, Set<EClassifier>>();

		/**
		 * This will map a duet EClassifier-featureName to the actual EStructuralFeature in the subtypes
		 * hierarchy. Keys of this map will be <code>EClassifier.hashCode() + featureName.hasCode()</code>.
		 */
		private final Map<Long, EStructuralFeature> hierarchyFeatureCache = new HashMap<Long, EStructuralFeature>();

		/**
		 * Delegates instantiation to the super constructor.
		 * 
		 * @param environment
		 *            The environment to which belongs this checker.
		 */
		public AcceleoTypeChecker(
				Environment<EPackage, EClassifier, EOperation, EStructuralFeature, EEnumLiteral, EParameter, EObject, CallOperationAction, SendSignalAction, Constraint, EClass, EObject> environment) {
			super(environment);
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see org.eclipse.ocl.AbstractTypeChecker#getOperations(java.lang.Object)
		 */
		@Override
		public List<EOperation> getOperations(EClassifier owner) {
			final List<EOperation> result = new ArrayList<EOperation>(super.getOperations(owner));
			if (!(owner instanceof PrimitiveType)) {
				result.addAll(getUMLReflection().getOperations(EcorePackage.eINSTANCE.getEObject()));
			}
			return result;
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see org.eclipse.ocl.AbstractTypeChecker#getResultType(java.lang.Object, java.lang.Object,
		 *      java.lang.Object, java.util.List)
		 */
		@Override
		public EClassifier getResultType(Object problemObject, EClassifier owner, EOperation operation,
				List<? extends TypedElement<EClassifier>> args) {
			EClassifier type = super.getResultType(problemObject, owner, operation, args);
			if (args.size() == 0 || operation.getEAnnotation("MTL non-standard") == null) { //$NON-NLS-1$
				return type;
			}

			final String operationName = operation.getName();
			// Handles all operations which can return a typed sequence as their result.
			if (args.get(0) instanceof TypeExp) {
				boolean isParameterizedCollection = AcceleoNonStandardLibrary.OPERATION_EOBJECT_EALLCONTENTS
						.equals(operationName);
				isParameterizedCollection = isParameterizedCollection
						|| AcceleoNonStandardLibrary.OPERATION_EOBJECT_ECONTENTS.equals(operationName);
				isParameterizedCollection = isParameterizedCollection
						|| AcceleoNonStandardLibrary.OPERATION_COLLECTION_FILTER.equals(operationName);
				isParameterizedCollection = isParameterizedCollection
						|| AcceleoNonStandardLibrary.OPERATION_EOBJECT_ANCESTORS.equals(operationName);
				isParameterizedCollection = isParameterizedCollection
						|| AcceleoNonStandardLibrary.OPERATION_EOBJECT_SIBLINGS.equals(operationName);
				isParameterizedCollection = isParameterizedCollection
						|| AcceleoNonStandardLibrary.OPERATION_EOBJECT_EINVERSE.equals(operationName);
				isParameterizedCollection = isParameterizedCollection
						|| AcceleoNonStandardLibrary.OPERATION_EOBJECT_PRECEDINGSIBLINGS
								.equals(operationName);
				isParameterizedCollection = isParameterizedCollection
						|| AcceleoNonStandardLibrary.OPERATION_EOBJECT_FOLLOWINGSIBLINGS
								.equals(operationName);

				if (isParameterizedCollection) {
					final org.eclipse.ocl.ecore.CollectionType alteredSequence = (org.eclipse.ocl.ecore.CollectionType)EcoreUtil
							.copy(type);
					alteredSequence.setElementType(((TypeExp)args.get(0)).getReferredType());
					Set<EClassifier> altered = alteredTypes.get(type);
					if (altered == null) {
						altered = new HashSet<EClassifier>();
						alteredTypes.put(type, altered);
					}
					altered.add(alteredSequence);
					type = alteredSequence;
				} else if (AcceleoNonStandardLibrary.OPERATION_OCLANY_CURRENT.equals(operationName)
						|| AcceleoNonStandardLibrary.OPERATION_EOBJECT_ECONTAINER.equals(operationName)) {
					type = ((TypeExp)args.get(0)).getReferredType();
				}
			} else if (args.get(0) instanceof StringLiteralExp
					&& AcceleoNonStandardLibrary.OPERATION_EOBJECT_EGET.equals(operationName)) {
				final String featureName = ((StringLiteralExp)args.get(0)).getStringSymbol();

				EStructuralFeature feature = null;
				if (owner instanceof EClass) {
					for (EStructuralFeature childFeature : ((EClass)owner).getEAllStructuralFeatures()) {
						if (childFeature.getName().equals(featureName)) {
							feature = childFeature;
							break;
						}
					}
				}

				if (feature == null) {
					createSubTypesHierarchy(owner);

					feature = findFeatureInSubTypesHierarchy(owner, featureName);
				}

				if (feature != null) {
					type = inferTypeFromFeature(feature);
					final Long key = Long.valueOf(owner.hashCode() + featureName.hashCode());
					if (!hierarchyFeatureCache.containsKey(key)) {
						hierarchyFeatureCache.put(key, feature);
					}
				}
			}

			return type;
		}

		/**
		 * Gets rid of caches. This is set as protected to be accessible from the enclosing environment.
		 */
		protected void dispose() {
			for (Set<EClassifier> alteredTypesValuesSet : alteredTypes.values()) {
				alteredTypesValuesSet.clear();
			}
			for (Set<EClassifier> subTypesValuesSet : subTypes.values()) {
				subTypesValuesSet.clear();
			}
			alteredTypes.clear();
			subTypes.clear();
			hierarchyFeatureCache.clear();
		}

		/**
		 * Searches the given type for a feature named <code>featureName</code>.
		 * 
		 * @param type
		 *            The type in which to search the feature.
		 * @param featureName
		 *            Name of the sought feature.
		 * @return The feature named <code>featureName</code> in <code>type</code> if it exists,
		 *         <code>null</code> otherwise.
		 */
		private EStructuralFeature findFeatureInType(EClassifier type, String featureName) {
			final Long key = Long.valueOf(type.hashCode() + featureName.hashCode());
			if (hierarchyFeatureCache.containsKey(key)) {
				return hierarchyFeatureCache.get(key);
			}

			EStructuralFeature feature = null;
			for (EObject child : type.eContents()) {
				if (child instanceof EStructuralFeature
						&& ((EStructuralFeature)child).getName().equals(featureName)) {
					feature = (EStructuralFeature)child;
					hierarchyFeatureCache.put(key, feature);
					break;
				}
			}

			return feature;
		}

		/**
		 * Goes down the <code>base</code> classifier's subtypes hierachy in search of a feature named
		 * <code>featureName</code> and returns it.
		 * 
		 * @param base
		 *            The starting point of the hierachy lookup.
		 * @param featureName
		 *            Name of the sought feature.
		 * @return The feature named <code>featureName</code> in <code>base</code>'s hierarchy.
		 */
		private EStructuralFeature findFeatureInSubTypesHierarchy(EClassifier base, String featureName) {
			EStructuralFeature feature = null;
			for (EClassifier subType : subTypes.get(base)) {
				feature = findFeatureInType(subType, featureName);
				if (feature == null) {
					feature = findFeatureInSubTypesHierarchy(subType, featureName);
				}
				if (feature != null) {
					break;
				}
			}
			return feature;
		}

		/**
		 * Tries and determine the static type of the given <code>feature</code>'s value.
		 * 
		 * @param feature
		 *            Feature we need a static type of.
		 * @return The determined type for this feature.
		 */
		@SuppressWarnings("unchecked")
		private EClassifier inferTypeFromFeature(EStructuralFeature feature) {
			EClassifier type = feature.getEType();
			// FIXME handle lists
			if (feature.isMany()) {
				if (feature.isOrdered() && feature.isUnique()) {
					type = EcoreFactory.eINSTANCE.createOrderedSetType();
				} else if (feature.isOrdered() && !feature.isUnique()) {
					type = EcoreFactory.eINSTANCE.createSequenceType();
				} else if (!feature.isOrdered() && feature.isUnique()) {
					type = EcoreFactory.eINSTANCE.createSetType();
				} else {
					type = EcoreFactory.eINSTANCE.createBagType();
				}
				((CollectionType<EClassifier, EOperation>)type).setElementType(feature.getEType());
			}
			return type;
		}

		/**
		 * Creates and stores the subtypes hierarchy of <code>classifier</code>.
		 * 
		 * @param classifier
		 *            The classifier we need the subtypes hierarchy of.
		 */
		private void createSubTypesHierarchy(EClassifier classifier) {
			if (subTypes.get(classifier) == null) {
				final Set<EClassifier> hierarchy = new HashSet<EClassifier>();

				ECrossReferenceAdapter referencer = getCrossReferencer(classifier);
				for (EStructuralFeature.Setting setting : referencer.getInverseReferences(classifier, false)) {
					if (setting.getEStructuralFeature() == EcorePackage.eINSTANCE.getEClass_ESuperTypes()) {
						EClassifier subType = (EClassifier)setting.getEObject();
						hierarchy.add(subType);
						createSubTypesHierarchy(subType);
					}
				}

				subTypes.put(classifier, hierarchy);
			}
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see org.eclipse.ocl.AbstractTypeChecker#exactTypeMatch(java.lang.Object, java.lang.Object)
		 */
		@Override
		public boolean exactTypeMatch(EClassifier type1, EClassifier type2) {
			boolean match = false;
			Set<EClassifier> alteredType1 = alteredTypes.get(type1);
			Set<EClassifier> alteredType2 = alteredTypes.get(type2);
			if (alteredType1 != null) {
				for (EClassifier alteredType : alteredType1) {
					if (alteredType == type2) {
						match = true;
						break;
					}
				}
			}
			if (!match && alteredType2 != null) {
				for (EClassifier alteredType : alteredType2) {
					if (alteredType == type1) {
						match = true;
						break;
					}
				}
			}
			if (!match) {
				return super.exactTypeMatch(type1, type2);
			}
			return true;
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see org.eclipse.ocl.AbstractTypeChecker#resolve(java.lang.Object)
		 */
		@Override
		protected EClassifier resolve(EClassifier type) {
			return getEnvironment().getTypeResolver().resolve(type);
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see org.eclipse.ocl.AbstractTypeChecker#resolveCollectionType(org.eclipse.ocl.expressions.CollectionKind,
		 *      java.lang.Object)
		 */
		@Override
		protected CollectionType<EClassifier, EOperation> resolveCollectionType(CollectionKind kind,
				EClassifier elementType) {
			return getEnvironment().getTypeResolver().resolveCollectionType(kind, elementType);
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see org.eclipse.ocl.AbstractTypeChecker#resolveTupleType(org.eclipse.emf.common.util.EList)
		 */
		@Override
		protected TupleType<EOperation, EStructuralFeature> resolveTupleType(
				EList<? extends TypedElement<EClassifier>> parts) {
			return getEnvironment().getTypeResolver().resolveTupleType(parts);
		}

		/**
		 * This will retrieve (and create if needed) a cross referencer adapter for the resource
		 * <code>scope</code> is in.
		 * 
		 * @param scope
		 *            Object that will serve as the scope of our new cross referencer.
		 * @return The <code>scope</code>'s resource cross referencer.
		 */
		private ECrossReferenceAdapter getCrossReferencer(EObject scope) {
			ECrossReferenceAdapter referencer = null;
			for (Adapter adapter : scope.eResource().eAdapters()) {
				if (adapter instanceof ECrossReferenceAdapter) {
					referencer = (ECrossReferenceAdapter)adapter;
					break;
				}
			}
			if (referencer == null) {
				referencer = new ECrossReferenceAdapter();
				scope.eResource().eAdapters().add(referencer);
			}
			return referencer;
		}
	}
}
