/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.seatunnel.transform.embedding;

import org.apache.seatunnel.shade.com.fasterxml.jackson.core.type.TypeReference;
import org.apache.seatunnel.shade.com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.seatunnel.shade.com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.seatunnel.shade.com.google.common.collect.Lists;

import org.apache.seatunnel.transform.nlpmodel.embedding.remote.custom.CustomModel;
import org.apache.seatunnel.transform.nlpmodel.embedding.remote.doubao.DoubaoModel;
import org.apache.seatunnel.transform.nlpmodel.embedding.remote.openai.OpenAIModel;
import org.apache.seatunnel.transform.nlpmodel.embedding.remote.qianfan.QianfanModel;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class EmbeddingRequestJsonTest {

    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    @Test
    void testOpenAIRequestJson() throws IOException {
        OpenAIModel model =
                new OpenAIModel(
                        "apikey", "modelName", "https://api.openai.com/v1/chat/completions", 1);
        ObjectNode node =
                model.createJsonNodeFromData(
                        new Object[] {
                            "Determine whether someone is Chinese or American by their name"
                        });
        Assertions.assertEquals(
                "{\"model\":\"modelName\",\"input\":\"Determine whether someone is Chinese or American by their name\"}",
                OBJECT_MAPPER.writeValueAsString(node));
        model.close();
    }

    @Test
    void testDoubaoRequestJson() throws IOException {
        DoubaoModel model =
                new DoubaoModel(
                        "apikey", "modelName", "https://api.doubao.io/v1/chat/completions", 1);
        ObjectNode node =
                model.createJsonNodeFromData(
                        new Object[] {
                            "Determine whether someone is Chinese or American by their name"
                        });
        Assertions.assertEquals(
                "{\"model\":\"modelName\",\"input\":[\"Determine whether someone is Chinese or American by their name\"]}",
                OBJECT_MAPPER.writeValueAsString(node));
        model.close();
    }

    @Test
    void testQianfanRequestJson() throws IOException {
        QianfanModel model =
                new QianfanModel(
                        "apikey",
                        "secretKey",
                        "modelName",
                        "https://api.qianfan.io/v1/chat/completions",
                        1,
                        "xxxx",
                        "xxxxxxx");
        ObjectNode node =
                model.createJsonNodeFromData(
                        new Object[] {
                            "Determine whether someone is Chinese or American by their name"
                        });
        Assertions.assertEquals(
                "{\"input\":[\"Determine whether someone is Chinese or American by their name\"]}",
                OBJECT_MAPPER.writeValueAsString(node));
        model.close();
    }

    @Test
    void testCustomRequestJson() throws IOException {
        Map<String, String> header = new HashMap<>();
        header.put("Content-Type", "application/json");
        header.put("Authorization", "Bearer " + "apikey");
        Map<String, Object> body = new HashMap<>();
        body.put("model1", "${model}");
        body.put("input1", Lists.newArrayList("${input}", "${input}"));

        CustomModel model =
                new CustomModel(
                        "modelName",
                        "https://api.custom.com/v1/chat/completions",
                        header,
                        body,
                        "$.data[*].embedding",
                        1);
        ObjectNode node =
                model.createJsonNodeFromData(
                        new Object[] {
                            "Determine whether someone is Chinese or American by their name"
                        });
        Assertions.assertEquals(
                "{\"model1\":\"modelName\",\"input1\":[\"Determine whether someone is Chinese or American by their name\"]}",
                OBJECT_MAPPER.writeValueAsString(node));
        model.close();
    }

    @Test
    void testCustomParseResponse() {
        CustomModel customModel =
                new CustomModel(
                        "modelName",
                        "https://api.custom.com/v1/chat/completions",
                        new HashMap<>(),
                        new HashMap<>(),
                        "$.data[*].embedding",
                        1);
        List<List<Float>> lists =
                OBJECT_MAPPER.convertValue(
                        customModel.parseResponse(
                                "{\"created\":1725001256,\"id\":\"02172500125677376580aba8475a41c550bbf05104842f0405ef5\",\"data\":[{\"embedding\":[-1.625,0.07958984375,-1.5703125,-3.03125,-1.4609375,3.46875,-0.73046875,-2.578125,-0.66796875,1.71875,0.361328125,2,5.125,2.25,4.6875,1.4921875,-0.77734375,-0.466796875,0.0439453125,-2.46875,3.59375,4.96875,2.34375,-5.34375,0.11083984375,-5.875,3.0625,4.09375,3.4375,0.2265625,9,-1.9296875,2.25,0.765625,3.671875,-2.484375,-1.171875,-1.6171875,4.1875,2.390625,-6.90625,0.369140625,0.259765625,3.671875,-2.9375,-1.9140625,-0.71875,-1.6640625,0.29296875,0.396484375,-4.625,-1.9921875,5.15625,-1.3984375,3.015625,-3.203125,-1.453125,4,-8.75,-5.625,1.0546875,-3.28125,-1.2265625,0.287109375,2.09375,4.6875,0.1572265625,0.42578125,0.79296875,3.234375,-0.169921875,0.9296875,7.40625,-3.296875,5.53125,3.890625,0.62109375,1.1171875,-0.373046875,4.125,-2.78125,0.333984375,3.9375,4.59375,6,1.53125,-0.373046875,1.109375,-4.0625,1.96875,1.421875,0.57421875,-0.56640625,-2.390625,0.734375,1.1875,-2.859375,-6.09375,-5.96875,1.8359375,-3,0.80859375,-0.130859375,-5.3125,-2.859375,1.484375,-4.53125,-6.90625,-2.25,0.7734375,-1.2734375,1.1484375,3.421875,-3.484375,2.65625,1.3359375,1.1484375,-4.09375,-5.625,2.625,-0.283203125,-3.46875,2.3125,-0.220703125,4.21875,3.75,-0.37109375,0.9609375,7.25,-0.87890625,7.03125,2.34375,4.5,-1.609375,-6.46875,-6.125,-2.59375,2.234375,3.78125,1.3046875,-5.5,1.953125,-3.421875,-5.9375,3.25,-3.4375,-8.3125,-2.546875,3.640625,0.267578125,-0.220703125,0.294921875,-0.4140625,2.515625,-1.0546875,-5.21875,6.6875,3.640625,0.2314453125,-2.5,1,1.6640625,0.59765625,2.75,1.1328125,1.1328125,-4.96875,4.53125,-0.349609375,3.390625,-0.193359375,7.625,2.921875,-3.484375,4.1875,8.5,-1.9140625,6.3125,2.5625,3.0625,0.40234375,0.76953125,-4.78125,3.53125,-2.765625,0.1591796875,-0.1025390625,-3.875,2.203125,0.03076171875,1.765625,1.859375,2.15625,-1.2578125,-4.40625,-0.62890625,4.4375,-1.78125,2.671875,2.765625,-1.7890625,-8.3125,-0.02197265625,1.640625,-3.96875,-3.15625,2.796875,1.1875,2,1.15625,2.359375,1.3984375,4.21875,-2.953125,8.5,3.46875,3.578125,0.90625,-1.8828125,2.15625,3.921875,4.125,-0.9609375,-2.171875,2.328125,2.921875,1.9765625,1.0703125,4.03125,6.28125,-3.59375,-0.94921875,5.6875,-1.9140625,-5.1875,-4.25,-7.71875,1.7109375,-1.59375,3.765625,-5.3125,-3.9375,-3.796875,2.90625,2.859375,-2.203125,-1.78125,-3.796875,0.1708984375,-5.15625,0.298828125,1.828125,-7.1875,1.6953125,6.125,2.671875,0.1728515625,3.375,0.609375,-4.78125,0.40625,-3.875,-6.4375,0.6953125,1.171875,-2.140625,5.8125,-1.640625,5.90625,-0.1650390625,4.9375,-2.421875,1.609375,-3.171875,-4.71875,7.6875,-1.09375,-1.9296875,0.033447265625,-3.46875,-2.671875,-8.875,2.4375,-1.1015625,4.40625,-3.53125,1.546875,2.359375,-3.15625,10.625,7.46875,-3.0625,-0.044677734375,0.90234375,-5.28125,-3,-1.2890625,0.59375,-6.34375,-1.8203125,5.40625,5.78125,-1.578125,2.46875,-2.171875,-1.71875,-0.38671875,-2.21875,-0.150390625,4.65625,-3.46875,1.5625,4.4375,-2.609375,1.6875,-2.828125,-6.03125,5.15625,-2.296875,-1.65625,-2.3125,-4.75,-3.3125,-3.703125,-1.9296875,-6.59375,3.640625,-0.62890625,4.8125,0.11279296875,2.515625,0.9921875,-3.03125,-5.40625,7.5625,-1.765625,4.4375,4.25,-0.140625,3.671875,-2.984375,-2.734375,2.203125,-6.96875,-1.1640625,2.390625,1.3515625,-1.75,2.96875,-3.75,-0.109375,2.5,0.796875,5.21875,7.8125,-4,1.171875,0.435546875,1.2734375,-3.015625,1.90625,-1.21875,5.9375,-0.31640625,-4.03125,-3.59375,1.09375,4.65625,-0.81640625,-2.046875,0.80859375,-5.375,2,-2.265625,5.34375,-0.46875,-1.3359375,-2.953125,-4.875,-0.53515625,-3,1.8203125,-2.59375,-1.4765625,6.28125,2.09375,0.1318359375,2.40625,-0.09130859375,-2.421875,-1.78125,1.59375,0.48828125,-0.310546875,-0.2353515625,0.1748046875,0.4453125,2.078125,-1.046875,1.46875,0.6953125,-0.52734375,-0.19140625,-2.28125,-0.515625,0.47265625,-1.2421875,-8.3125,1.1875,2.015625,-4.46875,3.734375,1.453125,-2.8125,-2.78125,5.875,-0.38671875,1.171875,-6.5,1.8046875,-2.15625,4,3.375,-0.78125,0.87890625,-1.796875,-1.265625,2.59375,3.96875,1.7421875,2.296875,2.78125,-5.8125,-2.046875,-0.1201171875,-4.1875,3.96875,-3.484375,-4.125,1.21875,3.484375,0.3828125,3.8125,1.90625,-8.3125,-2.15625,2.578125,2.578125,-1.34375,-3.359375,4.71875,-1.640625,-3.484375,2.046875,3.0625,-1.03125,-2.96875,6.96875,3.703125,-0.29296875,-0.423828125,2.640625,-1.265625,3.9375,-0.314453125,-4.15625,-2.171875,0.2734375,6.375,-6.21875,-6.3125,4.6875,-0.053466796875,0.045166015625,2.765625,2.953125,1.078125,-0.453125,1.96875,-6.71875,-3.375,-4.1875,2.515625,-0.5390625,-1.9296875,-4.03125,1.1953125,8.1875,1.0078125,0.80859375,-1.15625,-1.53125,2.875,-3.921875,1.953125,4.09375,6.59375,-4.5625,-1.2109375,3.5,-8.1875,0.294921875,-3.453125,-0.9921875,-2.015625,1.5,0.6328125,-4.90625,-2.765625,1.0546875,4.25,-2.390625,-5.96875,7.15625,-5.4375,-3.953125,1.15625,-0.017822265625,2.90625,2.78125,-2.21875,3.390625,1.9453125,2.171875,1.8671875,-1.125,-3.65625,-1.359375,0.96484375,2.5625,-2.9375,1.2734375,4.15625,-6,-0.2021484375,-1.8515625,-0.56640625,-1.671875,1.546875,5.8125,-0.640625,1.140625,-5.71875,-0.40625,0.5390625,-1.6640625,0.3203125,-2.375,4.9375,-2.453125,-1.59375,0.1669921875,1.6796875,-0.81640625,1.765625,-3.125,-1.234375,0.84375,-0.96484375,0.232421875,-0.01300048828125,-6.03125,4.25,5.625,0.65625,-1.6015625,1.390625,5.65625,3.0625,0.287109375,-0.08203125,4.15625,-1.5703125,-0.609375,-6.34375,2.203125,-3.84375,-2.53125,-3.390625,3.15625,4.59375,-4.46875,5.0625,-3.09375,3.328125,-0.65625,1.8515625,-9.375,1.609375,-1.515625,-2.5625,-2.953125,0.734375,2.375,1.3515625,0.390625,1.8671875,0.07080078125,1.328125,3.6875,0.2421875,0.73828125,3.1875,1.65625,2.75,2.859375,-2.8125,-7.75,1.53125,-1.1015625,-1.6875,6.3125,3.03125,-2.46875,0.77734375,-0.34765625,-1.78125,-1.4453125,3.40625,3.140625,-3.96875,3.984375,-3.21875,5.375,-2.890625,2.90625,-2.375,-6.1875,-2.4375,0.34375,-4.46875,-2.421875,3.40625,-1.2578125,4.59375,4.125,-6,0.003936767578125,1.1484375,2.359375,4.09375,0.5703125,-1.328125,-6.03125,4.5,3.234375,-2.140625,5.03125,-2.640625,0.041748046875,-0.90234375,4.375,-2.125,-0.1630859375,2.421875,-2.078125,1.1328125,-3.53125,1.0234375,-0.2734375,-9.125,-6.03125,0.73828125,-0.87109375,6.59375,-0.65625,-2.109375,-3.359375,2.40625,-0.0157470703125,5.96875,2.390625,3.078125,5.65625,5.09375,-1.5859375,1.78125,-0.921875,-8.0625,7.0625,-5.71875,-2.375,2.359375,2.65625,-1.453125,-1.2265625,1.984375,-2.125,-5.46875,-5.25,-1.78125,-4.28125,3.375,-2.09375,1.984375,-0.75,-5.0625,1.46875,-1.8671875,-2.875,-1.859375,2.609375,-5.5,2.484375,5.65625,1.875,-0.94921875,3.890625,4.125,0.8984375,-2.796875,0.95703125,-7.9375,1.7890625,3.453125,-1.9296875,-0.69140625,-5.84375,2.171875,-3.4375,2.921875,0.890625,-2.203125,-2.375,-1.6328125,-2.65625,0.8515625,-7.28125,2.484375,1.6484375,-0.8359375,-0.859375,0.232421875,1.921875,0.73046875,-0.30078125,1.515625,4.9375,0.7109375,-0.43359375,-3.140625,-2.796875,-0.2431640625,2.265625,-2.53125,6.875,-0.54296875,-1.5625,3.96875,0.44921875,-3.640625,-4.25,4.375,-1.875,0.45703125,-1.2265625,5.65625,0.298828125,3.921875,-1.703125,-2.8125,-3.328125,1.7578125,3.3125,-1.6875,-3.234375,2.09375,2.375,5.40625,-3.234375,-7.09375,1.984375,4.125,-0.8046875,-2.71875,8.6875,-1.296875,-2.625,-3,-3.78125,1.359375,1.515625,2.875,0.11279296875,-1.5859375,1.078125,3.46875,-1.390625,0.6328125,0.24609375,-3.765625,3.515625,0.380859375,2.609375,-0.80078125,-2.484375,-2.15625,-1.3203125,0.02490234375,4.03125,8.25,-1.5234375,-1.1953125,1.2109375,0.3125,-1.7421875,5.625,-0.76953125,5.90625,1.15625,0.1640625,1.390625,0.82421875,-0.322265625,3.21875,-4.65625,-4.5,-1.765625,3.171875,-4.3125,-1.4375,-2.546875,-0.9140625,4.28125,0.609375,-3.171875,3.671875,0.48046875,-0.9140625,-4,-2.4375,-5.34375,-1.96875,0.828125,1.953125,-2.140625,-2.59375,-0.353515625,4.78125,-4.09375,-3.921875,0.03173828125,1.8359375,1.3984375,-0.65234375,-1.15625,0.1611328125,0.50390625,2.90625,-1.875,-3.40625,0.498046875,8.75,3.90625,-4.53125,0.67578125,-0.765625,1.8359375,-5.3125,-2.15625,-0.6796875,-1.8984375,-3.046875,-1.7734375,-1.390625,-2.71875,-2.015625,5.84375,-3.28125,0.55859375,0.8046875,3.984375,0.99609375,3.015625,0.458984375,5.3125,3.1875,-1.2421875,-5.84375,-1.3828125,-0.04052734375,-5.75,-1.8828125,3.234375,6,3.171875,1.5703125,-2.828125,0.033203125,-0.953125,0.640625,5.3125,-5.75,-3.78125,-1.984375,-7.9375,-6.84375,-3.859375,-2.65625,-3.15625,-6.84375,-0.9765625,-1.375,-7.1875,-1.1328125,-2.109375,-1.546875,-1,0.640625,4.625,-4.65625,2.3125,3.703125,2.6875,3.0625,-2.28125,3.34375,0.474609375,-1.46875,0.34765625,-2.03125,5.25,-1.4609375,5.875,3.984375,-0.87890625,-3.8125,4.46875,4.40625,5.90625,-4.875,-3.53125,-2.53125,-1.8125,-0.39453125,-1.2421875,2.203125,-3.828125,-3.59375,-1.0859375,-3.453125,0.1845703125,5.625,0.421875,5.3125,-1.3671875,0.30859375,1.5234375,2.953125,0.1064453125,2.59375,1.5546875,-4.46875,3.609375,-0.81640625,1.390625,0.8359375,-2.78125,2.125,-1.6875,0.365234375,2.234375,3.875,10.4375,1.15625,2.328125,-0.09326171875,-0.76171875,-2.609375,-2.96875,2.40625,1.6796875,1.4921875,-3.65625,0.74609375,-0.8828125,2.03125,-0.306640625,-16.875,-3.328125,-5.53125,-2.109375,4.625,-1.0546875,-1.984375,1.0625,3.6875,2.671875,7.09375,-1.484375,4.03125,-1.078125,-0.7265625,2.515625,-4.3125,1.578125,3.6875,1.890625,4.625,1.7734375,-1.8125,-2.828125,6.9375,5.0625,-4.5,0.193359375,5.09375,-1.3515625,-1.140625,4.40625,-2.96875,2.078125,-4.75,3.078125,7.09375,2.75,-2.953125,-4.125,-2.375,-2.0625,1.0234375,3.046875,-2.578125,1.578125,2.921875,-5.65625,2.28125,2.28125,-0.259765625,-3.484375,-0.37109375,2.71875,1.625,-0.158203125,-4.5,2.5625,0.98828125,3.84375,4.8125,-2.796875,-2.140625,2.34375,2.90625,2.1875,1.5546875,2.578125,2.8125,-1.8515625,-2.984375,0.310546875,-1.328125,-0.0234375,-1.9765625,0.83984375,3.65625,2.046875,-4.5625,2.171875,2.234375,-2.109375,-0.0439453125,-4.0625,-3.5,2.09375,-2.21875,-2.5,0.703125,-2.953125,-1.28125,3.234375,-4.6875,4.1875,-2.484375,8.75,-0.53125,-1.8203125,1.171875,-3.0625,4.78125,-2.484375,-3.453125,3.765625,-2.6875,1.5625,-3.828125,1.9296875,-1.765625,1.2421875,5.0625,-4.65625,-2.0625,0.53125,3.265625,-2.875,-2.296875,0.29296875,3.859375,0.123046875,-4.46875,4.09375,-2.796875,3.96875,-3.890625,1.875,-4.46875,-0.5078125,2.140625,0.3203125,4.84375,5.03125,-5.34375,-4.96875,-1.3203125,-5.03125,-4.875,-4.5625,5.03125,-2.625,-0.75,1.046875,2.109375,-0.130859375,1.890625,-1.8125,2.53125,6.53125,-2.09375,0.87890625,-0.41015625,-0.412109375,-4.09375,-2.421875,-4.46875,6.40625,0.43359375,1.2578125,3.734375,-1.7109375,2.953125,1.8125,-1.1171875,-1.7109375,2.15625,1.859375,-2.015625,-2.25,1.7734375,-3.578125,4.15625,-3.328125,-3.28125,-4.71875,2.953125,1.40625,-0.287109375,1.5703125,3.53125,1.578125,3.171875,-4.34375,-3.125,5.78125,3.453125,-2.046875,4.3125,-1.2265625,-1.84375,0.640625,2.625,0.12890625,-3.25,-4.6875,5.28125,2.65625,2.015625,-4.4375,-5.75,-3.625,4.0625,4.59375,-0.78125,-2.484375,-2.03125,-3.75,1.6875,-4.15625,2.734375,-1.65625,-3.453125,-0.89453125,3.71875,2.453125,-4.15625,2.09375,0.82421875,-2.03125,0.052978515625,4.4375,1.734375,-3.71875,1.375,-0.349609375,-1.75,-7,3.59375,-2.625,-0.427734375,-4.40625,-3.84375,-3.265625,-3.796875,0.74609375,2.65625,1.6171875,3.609375,-0.7890625,3.890625,2.796875,-0.8671875,-0.43359375,2.15625,-1.7578125,-3.640625,2.375,-4.65625,-3.5,1.3984375,-7.1875,-1.5,5.0625,-2.625,4.0625,-1.171875,3.796875,-1.453125,-2.9375,-4,-1.3046875,0.91796875,6.59375,0.64453125,-0.91796875,0.64453125,2.703125,2.1875,-2.296875,-1.015625,-1.9921875,5,-0.298828125,2.953125,-5.125,-5.03125,5.375,-1.1328125,-4.46875,-0.5546875,-3.09375,1.5703125,5.34375,0.765625,-4.46875,-2.421875,-6.75,2.8125,-1.6171875,3.109375,-5.59375,0.87109375,-4.875,2.53125,4.46875,-7.21875,-3.1875,2.4375,3,5.1875,1.84375,-2.625,-6.21875,2.21875,0.306640625,-1.90625,-4.09375,-2.34375,-1.3046875,-3.875,4.4375,-2.328125,2.546875,-3.875,-2.40625,0.80078125,0.34765625,1,0.828125,1.4453125,-0.859375,3.03125,1.109375,5.15625,1.1953125,-3.8125,2.734375,4.21875,0.345703125,-1.2109375,2.0625,-0.79296875,-2.8125,2.109375,2.96875,-2.90625,5.15625,3.359375,4.3125,-5.53125,-2.875,1.515625,3.515625,-2.75,1.7109375,-4.9375,0.7265625,3.71875,-0.4765625,1.34375,0.049560546875,2.796875,-1.421875,-1.7890625,1.5,2.3125,4.21875,1.6875,3.015625,3.3125,-1.1640625,3.546875,-0.375,-1.2265625,-1.59375,3.609375,-3.015625,-2.546875,-4.625,1.046875,-1.796875,4.75,2.515625,1.1484375,0.8984375,-1.4140625,-2.328125,0.037841796875,-5.78125,-1.5859375,0.11669921875,3.015625,-0.83984375,0.84375,-0.82421875,0.96484375,4.0625,0.0400390625,4.25,-2.28125,1.3515625,1,1.5625,-2.8125,3.15625,-2.609375,-0.142578125,1.578125,-2.875,3.75,-4.3125,-1.359375,-2.578125,-0.69140625,2.84375,3.75,-4.75,-5.5625,0.84765625,0.380859375,5.125,3.0625,-3.140625,-0.93359375,0.73046875,0.0303955078125,4.3125,0.85546875,2.703125,-4.28125,5.625,5.90625,0.4296875,0.76953125,-0.9140625,-1.71875,-4.5,3.828125,-0.4609375,2.21875,-1.9453125,2.5,4.15625,1.8984375,3.984375,-5.75,2.953125,0.2734375,3.890625,-0.76171875,-3.90625,0.337890625,1.96875,0.69140625,-0.70703125,3.578125,0.046142578125,0.765625,-2.734375,4.28125,4.3125,2.578125,-4.40625,1.921875,-2.90625,1.7734375,-3.90625,1.1484375,-5.625,1.65625,2.703125,5.34375,-1.9296875,-6.1875,4.5,1.5625,-0.9140625,-3.953125,4.65625,4.5625,2.484375,-5.15625,-2.375,1.625,-1.328125,-0.26171875,-5.25,3.328125,2.0625,-3.609375,-3.71875,1.6171875,1.046875,-3.1875,-3.71875,-3.34375,1.9609375,2.5625,3.609375,1.59375,-2.484375,4.125,-0.80078125,1.9140625,4.78125,-1.09375,0.140625,3.171875,-3.578125,2.640625,-0.6640625,-2.65625,-1.4375,0.47265625,-2.46875,2.6875,-2.515625,-1.0234375,-2.09375,-0.138671875,-0.5078125,1.5,4.15625,-3.09375,0.158203125,4.4375,-1.96875,-3,-1.9609375,2.09375,-1.7734375,-1.09375,-1.8984375,3.3125,1.9765625,0.8671875,0.2890625,0.66796875,-1.9765625,-3.640625,-4.90625,2.0625,-4.0625,3.59375,-0.84765625,-6.21875,1.515625,3.890625,3.640625,-0.2734375,-2.046875,0.875,3.78125,0.07470703125,-1.078125,-1.4921875,3.671875,-2.796875,-3.6875,2.75,2.78125,-5.40625,1.7890625,-4.28125,-2.265625,-0.98046875,4.46875,0.173828125,-2.25,-2.875,-3.84375,-1.7421875,-1.6171875,-3.21875,1.9140625,1.7421875,2.671875,1.09375,1.4375,-3.5,2.59375,19.125,0.0101318359375,-8.4375,1.3515625,-3.625,4.4375,4.65625,1.8125,0.423828125,-1.5,0.62890625,4.21875,0.609375,0.5390625,-2.390625,0.984375,-0.79296875,2.078125,-3.703125,-3.109375,-2.265625,-1.0234375,-0.328125,1.9765625,2.5,2.375,0.8046875,-2.265625,1.2734375,-3.390625,-4.375,-4.71875,3.765625,-2.921875,3.125,-3.171875,4.65625,-0.7890625,-3.3125,-2.984375,-3.296875,-2.796875,2.375,-0.12255859375,-3.21875,5.21875,0.1982421875,0.2138671875,-1.1796875,-0.130859375,-4.34375,-1.4453125,-2.5,6.3125,1.0625,-6.15625,-0.5703125,-3.203125,-3.546875,-1.375,2.9375,-0.53515625,1.7578125,2.71875,-1.9453125,-2.640625,-3.046875,0.49609375,1.0078125,-3,-4.84375,0.2119140625,1.2265625,1.3515625,1.609375,-4.84375,2.46875,2.140625,2.171875,1.75,0.67578125,-0.60546875,-2.46875,-2.234375,-0.9453125,1.2421875,-3.15625,0.006744384765625,3.359375,-1.765625,8.375,-8.3125,5.8125,5.15625,-2.0625,-0.470703125,1.5,-0.30859375,-2.421875,-0.2294921875,0.95703125,1.8828125,4.84375,-0.68359375,4.625,1.359375,0.373046875,0.83203125,2.640625,4.34375,0.7578125,3.109375,-0.412109375,-2,2.15625,-0.08349609375,-3.140625,-3,-3.703125,-2.5625,3.6875,1.7890625,-3.296875,0.89453125,-7.5,-5.40625,-2.25,-7.625,4.34375,-1.34375,-0.14453125,3.515625,-2.46875,-1.2109375,-2.46875,-3.921875,1.265625,3.65625,1.4375,-1.46875,-5.03125,2.59375,3.890625,-2.765625,-2.4375,0.353515625,-4.21875,4.4375,-0.376953125,3.9375,-2.09375,3.96875,3.234375,-2.203125,-6.875,5.15625,-3.6875,-4.34375,-6.625,-2.90625,-4.9375,-3.34375,0.412109375,-0.9453125,-0.5703125,-1.3046875,3.21875,-0.65234375,-1.6796875,3.171875,3.453125,-4.4375,-1.2578125,0.828125,1.1796875,-4.375,0.1787109375,4,0.53515625,1.328125,-0.546875,0.271484375,-0.5546875,-3.859375,-0.2216796875,0.86328125,-4.53125,-1.3828125,-0.60546875,-5.46875,-1.3515625,-1.2890625,-3.734375,2.9375,2.40625,-3.984375,0.875,-2.953125,-0.9765625,-1.6328125,-1.25,3.96875,1.6953125,0.0072021484375,5.875,-0.921875,-3.46875,-3.140625,-0.26953125,0.2265625,-2.09375,7.0625,-1.09375,0.30078125,-6.03125,5.34375,2.359375,1.6640625,-0.99609375,4.625,4.25,-2.484375,-4,0.89453125,3.0625,4.1875,-4.28125,3.953125,0.6328125,-0.74609375,-1.53125,2.015625,-1.1796875,1.03125,-1.6484375,-5.4375,0.3671875,1.8125,-0.326171875,1.546875,4.03125,-3.34375,0.484375,2.5,-1.4140625,3.34375,4.25,-1.7890625,1.09375,2.171875,5.34375,-1.5625,0.98828125,-5.09375,-3.625,-2.640625,-2.46875,3.109375,-2.515625,0.09033203125,0.21484375,-3.921875,3.125,-4.1875,1.2109375,1.3671875,1.1875,-5.4375,4.59375,3.890625,-2.8125,3.328125,-5.125,-1.9765625,-1.4296875,2.34375,-2.71875,-5.875,3.125,3.453125,-1.515625,3.546875,2.265625,-0.52734375,1.9375,-2.859375,2.703125,-3.359375,4.75,1.2734375,3.09375,3.65625,-0.255859375,-0.1044921875,-5.75,-0.3359375,-0.77734375,-2.234375,6.1875,-3.84375,0.19921875,4.25,6.4375,-10.5,-1.5078125,0.7265625,0.2890625,3.921875,5.0625,0.09814453125,0.68359375,3.109375,1.015625,2.671875,0.0257568359375,-0.4765625,-4,5.15625,0.2314453125,-4.6875,3.1875,3.984375,-2.609375,3.4375,-2.375,-3.734375,-0.07568359375,2.75,-5.3125,1.9296875,4.625,-1.6484375,2.875,3.734375,-1.34375,3.875,-1.9921875,-11.3125,-1.53125,3.296875,5.71875,0.80859375,1.7578125,0.48046875,-2.015625,1.4765625,-0.5546875,0.71484375,-0.7578125,-11.1875,0.9765625,-3,-0.09765625,-1.9453125,-3.8125,-2.5,4.375,1.65625,1.1015625,3.328125,2.84375,0.84375,4.5625,0.11279296875,-5.84375,1.1484375,1.7578125,-4.8125,-0.59765625,3.234375,1.125,-1.859375,-2.515625,3.78125,-1.7421875,-0.69921875,5.8125,3.765625,1.578125,-1.84375,-5.03125,0.984375,-3.375,-1.9140625,1.1953125,-0.384765625,2.8125,-2.203125,2.828125,1.1171875,-3.75,-4.15625,-2.25,-3.5625,1.5,2.671875,2.171875,-2.609375,-1.7265625,2.8125,2.5,-0.455078125,-1.546875,2.1875,-0.1884765625,-2.984375,-1.4765625,2.0625,-4.46875,-2.90625,4.0625,1.8359375,0.443359375,-0.7734375,-3.140625,2.171875,1.734375,-1.8515625,-1.84375,-1.234375,2.15625,5.34375,-2.484375,-5.6875,-1.2734375,0.1806640625,-4.375,-3.5625,0.89453125,-1.15625,0.75,3.09375,-2.25,1.1875,4.6875,-1.3359375,-3.875,3.53125,4.4375,-2.671875,-0.75,-0.458984375,-2.53125,3.8125,5,-1.2421875,-2.109375,-0.50390625,-2.734375,-4.90625,1.0234375,2.421875,-3.34375,-10.125,6.46875,3.671875,5.40625,1.546875,-2.59375,3.8125,-1.6953125,3.703125,-0.423828125,0.82421875,1.515625,-7.59375,-2.40625,-2.0625,-5.0625,0.59375,-0.345703125,-4.75,1.4921875,6.25,-2.15625,-1.8671875,-2.703125,-3.9375,4.28125,-3.484375,-5.9375,1.984375,-7.4375,1.4609375,-1.9609375,3.265625,-5.875,1.8359375,-0.017333984375,2.046875,-0.5859375,-0.671875,-2.328125,1.1953125,-2.65625,3.625,0.7890625,3.9375,-0.365234375,2.90625,-1.2421875,0.314453125,-3.265625,1.6640625,1.7109375,0.60546875,0.384765625,2.296875,-2.28125,-0.8046875,-1.0546875,1.046875,2.796875,0.61328125,-0.625,0.10693359375,4.21875,-0.6484375,2.03125,-2.3125,-0.173828125,-1.015625,-0.224609375,0.74609375,-0.86328125,0.0145263671875,0.1318359375,1.7109375,1.421875,0.486328125,-0.19921875,0.140625,1.2734375,1.015625,1.5625,-1.65625,-0.45703125,-0.435546875,-0.0206298828125,1.828125,1.734375,-2.734375,1.65625,-2.09375,-0.6875,-0.2421875,2.125,1.1015625,0.1064453125,1.59375,-1.875,1.828125,0.15234375,-1.2421875,1.25,-0.765625,-2.265625,2.34375,-2.109375,-0.921875,0.6640625,-1.2734375,-1.4765625,-0.73828125,2.21875,-0.84375,1.328125,-1.171875,-0.181640625,0.306640625,-1.171875,0.279296875,0.94140625,1.171875,-3.921875,3.15625,1.2421875,0.52734375,-0.1630859375,1.0390625,-1.46875,-0.08447265625,1.0390625,-0.37109375,0.921875,1.859375,-1.8046875,0.54296875,-0.8203125,-1.09375,1.1640625,1.515625,0.54296875,-1.65625,-1,1.5234375,1.4453125,-1.1953125,0.359375,-0.062255859375,-2.09375,3.03125,1.21875,-3.15625,-0.357421875,-0.169921875,0.546875,-0.73828125,-0.126953125,1.046875,-2.75,-0.2314453125,0.2421875,0.306640625,-1.1328125,1.8984375,0.00469970703125,3.9375,0.8515625,1.1328125,1.1875,1.3984375,2.046875,-1.3515625,0.25390625,-0.9921875,3.234375,-0.373046875,0.8828125,1.3828125,-1.921875,-0.484375,-0.81640625,0.61328125,1.4296875,-0.70703125,-0.404296875,2.53125,1.625,0.494140625,2.375,-2.03125,0.33984375,0.291015625,-0.68359375,-1.625,1.625,-0.478515625,0.349609375,-2.0625,-1.25,-0.1484375,-0.44140625,0.67578125,0.3671875,0.4921875,0.236328125,1.1953125,0.5078125,-2.375,1.3671875,-0.341796875,0.6328125,-1.7265625,-1.328125,0.84375,-0.08935546875,1.0625,0.90625,1.984375,2.828125,1.109375,-1.3671875,1.03125,1.0625,1.75,0.263671875,-1.234375,-0.09228515625,-0.13671875,0.271484375,0.58203125,-0.9375,-1.28125,0.4609375,-0.95703125,-0.1552734375,-1.5703125,3.375,-0.9609375,-1.1796875,-0.419921875,-1.5,0.58984375,-1.3125,1,-1.578125,2.484375,1.34375,3.34375,1.4296875,-0.671875,-0.984375,0.30859375,0.72265625,-0.337890625,-0.06982421875,-1.125,-0.44921875,-0.62890625,5.40625,0.263671875,1.0390625,-2.03125,3.296875,0.68359375,-0.10986328125,-1.078125,-0.2412109375,-2.078125,-0.13671875,-1.4375,-1.390625,0.29296875,-1.1484375,-4.0625,-2.703125,-0.302734375,0.77734375,-1.640625,-0.0390625,3.890625,0.375,1.2890625,1.5,2.640625,0.19140625,-1.78125,-0.5859375,1.6328125,-1.234375,2,0.8125,-1.9453125,-2.78125,-0.3671875,-2.328125,-1.9453125,-0.59375,-0.8046875,1.9921875,-0.265625,-0.03515625,-1.3125,-1.5234375,-3.03125,-0.458984375,-0.1279296875,2.375,1.53125,0.67578125,-0.55078125,-0.4296875,0.515625,-1.75,0.6640625,-1.65625,4.25,-0.326171875,-1.4296875,2.53125,0.396484375,3.140625,0.859375,-1.3671875,-1.8828125,-0.828125,0.45703125,0.7109375,3.0625,-0.2578125,0.6328125,0.57421875,-0.85546875,0.5625,1.0234375,-0.296875,-4.84375,-1.578125,-0.486328125,2.59375,-1.2109375,0.09765625,2.59375,-0.87109375,-0.7890625,-1.7421875,-2.34375,-0.2490234375,-0.82421875,0.8046875,2.078125,-0.7265625,-0.10400390625,-0.703125,-1.046875,0.46875,-1.7734375,1.09375,-0.30859375,0.0181884765625,0.2734375,-2.703125,-0.470703125,0.67578125,-1.921875,-1.0078125,1.6328125,0.2021484375,1.359375,1.6796875,-1.6015625,1.5703125,0.6484375,-2.859375,-0.63671875,-0.8359375,1.34375,0.0556640625,0.4375,1.765625,-1.1484375,-1.90625,-1.453125,0.57421875,0.84375,-0.349609375,0.251953125,-0.0927734375,0.416015625,-0.40625,-2.71875,-0.48046875,0.4140625,-0.2109375,0.96484375,1.0859375,1.453125,1.15625,1.375,-0.478515625,1.375,-1.8828125,1.6484375,0.9921875,-2.171875,0.5859375,2.03125,-2.125,0.314453125,1.1796875,-0.4921875,-0.72265625,-0.80078125,0.5546875,-0.52734375,0.58203125,-0.52734375,1.9453125,1.71875,-0.328125,1.453125,-2.203125,-2.09375,-2.625,0.2177734375,-0.82421875,0.3359375,-2.203125,1.375,-1.7578125,-0.072265625,-0.4765625,-0.38671875,-1.9453125,1.5625,1.7578125,0.4453125,0.640625,0.0255126953125,-0.5703125,3.796875,-1.0703125,-0.1201171875,0.93359375,1.15625,-2.078125,3.484375,0.5234375,2.109375,0.0037078857421875,1.3359375,-0.796875,1.25,0.1455078125,0.86328125,0.478515625,1.828125,0.31640625,-0.296875,-0.154296875,-1.53125,-1.1640625,0.6484375,1.0703125,-5.375,0.86328125,0.890625,0.48828125,0.84765625,-2.828125,1.1015625,0.4765625,3.296875,-0.00408935546875,-0.40234375,3.421875,0.61328125,-1.46875,1.1875,0.953125,0.0771484375,-2.78125,-1.171875,-0.86328125,2.9375,-1.0703125,0.1015625,-0.279296875,-0.90625,3.046875,0.6796875,-1.6640625,1.453125,0.443359375,-0.439453125,-1.453125,-3.40625,-0.1689453125,1.71875,-0.9453125,2.234375,0.158203125,0.87109375,0.66796875,-1.640625,1,0.265625,0.267578125,-0.90625,1.75,-0.2041015625,-1.59375,1.65625,-1.1484375,-1.78125,2.421875,1.6953125,-2.328125,0.027587890625,-0.494140625,-0.3203125,-0.01953125,0.58203125,-2.28125,0.546875,0.62109375,0.90625,-0.921875,-1.53125,2.484375,1.890625,2.953125,2.359375,-0.90234375,0.171875,-2.234375,0.33984375,-0.45703125,-0.87109375,0.08251953125,1.8671875,-1.0078125,1.5703125,-0.30078125,0.921875,-1.8046875,1.609375,2.703125,0.92578125,0.40625,-0.26171875,-0.322265625,-1.8671875,-0.5,-2.296875,0.62109375,0.6953125,1.1640625,0.1376953125,-1.4296875,1.5390625],\"index\":0,\"object\":\"embedding\"},{\"embedding\":[-2.28125,-0.7734375,-0.8359375,-2.3125,3.046875,4.125,-1.0390625,-2.890625,0.0103759765625,1.9296875,0.1015625,1.75,2.4375,2.015625,5.09375,1.203125,-2.140625,-2.828125,-1.328125,-4.6875,1.0078125,6.8125,0.578125,-4.71875,-0.80859375,-6.25,1.578125,4.25,4.46875,-1.0078125,8,-2.3125,2.546875,-0.00555419921875,1.5625,-1.8671875,-2.375,-2.53125,5.25,-0.69140625,-2.96875,-0.68359375,1.6171875,2.96875,-3.015625,-1.734375,0.4140625,-2.9375,2.53125,-1.6640625,-4.5625,-1.9296875,3.234375,-2.734375,2.359375,-4.125,-3.046875,4.5,-5.875,-2.984375,-1.8515625,-2.8125,-0.7734375,0.46484375,1.3984375,5.28125,0.68359375,-1.3359375,0.51171875,8.625,-0.055908203125,3.578125,6.5,-2.390625,6.34375,5.5625,0.7265625,1.578125,-2.921875,4.90625,-2.953125,-0.62890625,2.453125,3.46875,4.5625,2.671875,-1.9140625,0.859375,-3.03125,1.703125,1.96875,0.59375,-1.4140625,-3.140625,-1.2109375,1.2890625,-3.21875,-6.5625,-6.78125,2.765625,-0.78515625,-0.3515625,1.8125,-4.53125,-5.03125,2.171875,-1.8515625,-5.46875,-1.78125,0.380859375,2.640625,1.65625,3.640625,-2.140625,2.46875,1.21875,4.28125,-2.796875,-4.40625,2.796875,-2.0625,-1.9765625,4.28125,-0.6796875,4.4375,4.28125,-4.03125,-0.01416015625,5.53125,-1.4609375,7.25,3.578125,3.6875,-2.375,-8.0625,-4.71875,-1.9453125,3.71875,4.3125,4.40625,-5.03125,3.21875,-3.734375,-6.625,4.1875,-3.4375,-6.4375,-3.15625,3.859375,-1.9140625,-1.78125,1.8046875,0.5,2.3125,-1.2421875,-4.375,4.0625,3.875,0.1259765625,-1.0546875,2.015625,3.328125,1.1484375,1.7265625,1.8046875,-0.462890625,-5.625,3.6875,-1.0390625,2.5625,0.90625,10.4375,4.28125,-4.5625,1.9765625,8.625,-1.328125,8.625,1.4609375,2.203125,0.81640625,-0.640625,-2.90625,4.53125,-2.15625,1.5,0.12255859375,-5.6875,3.140625,1.2890625,1.578125,1.5625,2.71875,-1,-4.84375,-1.8671875,3.484375,-2.578125,3.4375,0.1025390625,-1.40625,-7.375,1.4921875,1.5546875,-4.71875,-3.765625,2.703125,-1.71875,3.078125,-0.380859375,2.265625,0.24609375,3.21875,-2.0625,7.65625,2.640625,2.734375,2.046875,1.8359375,2.46875,4.53125,3.484375,1.8359375,-2.078125,-0.83984375,2.03125,5.8125,0.439453125,3.75,8.6875,0.251953125,0.408203125,6.84375,-2.515625,-1.78125,-3.578125,-3.78125,1.6015625,-0.279296875,2.671875,-5.65625,-4.0625,-2.328125,2.984375,3.515625,-3.359375,-2.34375,-2.703125,-0.51171875,-6.4375,1.484375,3.671875,-9.0625,1.8828125,5.625,3.96875,1.984375,1.265625,-0.33203125,-4.125,0.333984375,-2.4375,-5.875,-0.58203125,1.890625,-2.390625,5.09375,-1.5546875,3.515625,-0.7421875,5.1875,-2.28125,-0.0927734375,-3.046875,-4.3125,8.8125,-0.232421875,-1.90625,1.0703125,-3.078125,-3.5625,-10.25,2.5,1.1171875,4.96875,-2.921875,1.40625,0.40234375,-3.640625,12.75,3.90625,-1.8203125,1.9921875,-0.63671875,-6.03125,-1.984375,-2.046875,2.046875,-5.59375,1.84375,3.6875,4.5,-1.9296875,3.4375,-1.7421875,-0.9296875,-1.109375,-4.5625,-1.9375,2.671875,-3.765625,2.34375,9.625,-4.75,2.03125,-2.109375,-6.1875,4.75,-0.03662109375,-0.11376953125,-2.140625,-5.125,-1.9921875,-2.78125,-1.4296875,-6.65625,4.96875,-0.984375,5.375,0.97265625,3,3.296875,-4.1875,-5.03125,8.4375,-1.5,3.296875,5.71875,0.55078125,0.68359375,-3.515625,-4.6875,2.46875,-5.46875,0.953125,5.71875,3.328125,-1.640625,1.0234375,-6.21875,2.40625,2.328125,-0.68359375,6.53125,6.90625,-2.265625,2.78125,1.9140625,-0.71484375,-2.28125,-0.2294921875,-1.078125,6.34375,1.1875,-3.890625,-3.796875,-0.5859375,5.03125,-2.375,0.7734375,-1.21875,-4.15625,2.59375,-1.15625,3.6875,0.91796875,0.90625,-1.8046875,-5.125,0.087890625,-2.625,0.29296875,-1.7734375,-3.28125,4.25,1.515625,-0.484375,1.59375,0.67578125,-3.53125,-0.46484375,0.59765625,-1.15625,0.65625,2.5625,-0.5703125,-0.984375,1.5546875,-0.3828125,-2.21875,1.0546875,-1.2734375,2.40625,-6.9375,-0.6484375,-0.2490234375,-2.125,-8.375,-0.4765625,1.0703125,-3.78125,2.71875,1.96875,-1.2578125,-3.0625,4.4375,1.421875,1.8671875,-6.90625,2.15625,-1.8828125,3.328125,2.140625,-1.7421875,0.59375,-1.4296875,-2.765625,4.375,3.546875,-0.69921875,3.453125,0.68359375,-3.265625,-3.625,0.1630859375,-4.90625,4.75,-0.236328125,-1.859375,5.21875,2.203125,-1.5,1.625,0.98828125,-6.28125,-4.78125,2.96875,3.171875,-3.078125,-3.96875,0.470703125,-1.4296875,-4.4375,3.078125,3.84375,-1.1171875,-2.8125,3.40625,4.375,-2.203125,0.0830078125,1.1171875,0.52734375,2.703125,-1.9375,-3.140625,-0.1103515625,0.130859375,4.71875,-5.8125,-6.84375,3.015625,-2.875,0.2001953125,1.15625,4.5625,0.46875,-1.8984375,-1.9296875,-3.0625,-3.46875,-2.828125,3.53125,-1.078125,-2.53125,-2.90625,0.29296875,8.3125,1.90625,0.369140625,-2.375,-0.11572265625,2.453125,-1.71875,0.50390625,4.4375,7.90625,-4.03125,-0.63671875,3.53125,-8.125,0.94921875,-1.375,-1.15625,-0.94921875,2.3125,2.1875,-6.25,-0.7890625,0.0115966796875,5.03125,-3.453125,-3.828125,5.15625,-4.8125,-3.09375,1.859375,-0.6875,4.0625,1.296875,-1.34375,2.875,2.984375,2.65625,1.8203125,-2.53125,-3.640625,-3.3125,1.2890625,2.265625,-2.234375,2.296875,4,-5.4375,0.90234375,-2.25,-0.6953125,-0.212890625,-0.515625,5.90625,2.125,2.25,-6.09375,1.2578125,0.50390625,-0.416015625,-0.7421875,-1.1484375,6.71875,-0.5,-0.2294921875,0.94921875,2.09375,-1.1953125,1.640625,-3.796875,-2.453125,-3.109375,-1.796875,-1.0234375,-4.03125,-5.5,4.4375,6,-1.234375,-1.6796875,2.171875,5.5,3.984375,-0.84375,1.515625,3.421875,-2.5,0.23828125,-5.40625,2.609375,-7.84375,-2.53125,-1.6875,2.921875,3.75,-4.15625,3.765625,-2.578125,2.4375,-1.4375,4.4375,-10.5625,2.046875,-2.15625,-2.796875,-2.28125,-0.57421875,3.171875,-0.44921875,2.109375,1.3671875,-0.75,3.953125,5.46875,-1.5,1.765625,2.1875,2.46875,-0.5859375,2.515625,-2.125,-8.25,1.3125,-1.1484375,1.09375,7.5625,1.9375,-1.7734375,2.46875,0.88671875,-1.5703125,-1.7265625,4.0625,3.015625,-1.546875,4.25,-3.90625,5.40625,-3.28125,1.7265625,-3.265625,-6.15625,0.279296875,1.9296875,-5.5625,-4.09375,2.859375,0.216796875,5.78125,3.421875,-5.375,1.21875,-0.41796875,1.109375,2,0.30078125,-0.03759765625,-4.75,3.921875,4.1875,-2.40625,7.03125,-1.5703125,-1.6484375,-1.1171875,2.40625,-1.7734375,0.373046875,1.84375,0.287109375,-0.78125,-3.484375,0.96484375,0.5703125,-6.625,-7.21875,1.7265625,-1.7734375,7.0625,0.73046875,-0.859375,-3.15625,2,1.5546875,6.375,3.3125,3.765625,4.5,3.765625,-2.390625,2.671875,-3.6875,-6.09375,7,-6.53125,-1.8515625,1.015625,0.859375,-0.2578125,-1.0234375,-0.3515625,-0.71484375,-3.484375,-6.09375,-2.359375,-1.875,2.015625,-1.6484375,2.203125,0.57421875,-4.09375,-0.5703125,-1.6484375,-1.6875,-1.6640625,4.15625,-5.625,1.484375,5.71875,2.046875,-1.5234375,4.15625,3.09375,-0.47265625,-4.78125,0.7109375,-6.875,1.6015625,1.46875,-0.6015625,0.50390625,-8,2.03125,-2.4375,3.5,-0.671875,-0.05078125,-1.265625,-3.296875,-1.3984375,-0.91796875,-5.40625,-0.171875,1.6953125,1.125,-1.8359375,0.671875,3.078125,-0.52734375,0.384765625,-1.125,2.046875,0.40625,2.34375,-4.78125,-2.90625,1.28125,0.9140625,-2.03125,6.53125,0.91796875,0.79296875,3.546875,1.7265625,-5.5,-5.78125,3.921875,-2.8125,-1.796875,-3.25,2.421875,-1.359375,6.53125,-2.21875,-5.53125,-3.703125,1.6484375,3.15625,-2.609375,-3.09375,4.78125,1.8359375,2.765625,-2.15625,-7.5,1.609375,0.98828125,-0.146484375,-1.140625,8.625,-1.9296875,-0.4765625,-4.4375,-3.234375,2.046875,0.875,2.046875,-0.76171875,-1.2734375,0.69921875,0.4765625,-2.34375,-0.55078125,0.6015625,-2.546875,1.75,0.07177734375,4.875,-2.53125,0.3984375,-1.2734375,-0.50390625,-0.10009765625,4.3125,8.75,-1.765625,-0.96875,0.35546875,2.984375,-3.59375,6.6875,1.3515625,7.75,-1.1640625,0.25,1.03125,0.375,-2.171875,4.59375,-5.25,-2.84375,-1.890625,1.21875,-2.5625,0.671875,-3.984375,-0.498046875,4.40625,-0.455078125,-0.007568359375,2.609375,0.79296875,-0.201171875,-3.09375,-1.3125,-4.71875,-2.515625,-0.14453125,2.03125,-3.03125,-0.4921875,-0.33984375,5.84375,-0.357421875,-1.4453125,-2.59375,1.53125,1.859375,1.171875,-0.8046875,0.255859375,0.58984375,3.3125,-1.015625,-4.34375,-0.94921875,8.4375,4.21875,-6.875,1.5703125,-0.43359375,1.4453125,-4.8125,-1.4609375,-2.15625,-1.4921875,-4.1875,1.1328125,0.419921875,-3,-0.06494140625,4.5,-1.2890625,-0.15625,3.46875,4.0625,0.478515625,2.96875,-2.125,4.375,2.21875,-2.09375,-5.96875,-1.703125,0.48046875,-2.75,-1.4140625,2.03125,6.15625,0.55859375,2.625,-1.0625,2.28125,-1.6953125,3.78125,5.125,-4.59375,-2.703125,-2.3125,-9.5625,-4.03125,-1.7421875,-2.921875,-5.34375,-4.25,-0.86328125,-1.2421875,-8,0.0966796875,-2.234375,-3.265625,1.4453125,2.953125,1.7578125,-5.75,3.125,4.125,2.578125,2.546875,0.84765625,5.46875,-0.050537109375,-2.96875,1.4453125,-3.4375,4.15625,-1.03125,3.546875,6.25,-0.453125,-4.96875,4.78125,2.96875,5.53125,-7.375,-2.625,-0.337890625,-1.671875,-0.458984375,-1.7578125,2.546875,-4.5,-5.5,1.078125,-3.203125,1.2265625,4.6875,-0.8046875,6.78125,1.6328125,0.419921875,2.140625,2.71875,0.62109375,0.169921875,1.7421875,-5.9375,3.234375,-2.171875,3.265625,-0.296875,-1.5234375,2.734375,-0.7578125,-0.310546875,2.8125,2.734375,10.3125,0.515625,4,-2.3125,0.63671875,-1.7265625,-0.2392578125,2.25,2.015625,0.79296875,-1.4765625,0.7890625,-0.44921875,0.478515625,-0.4609375,-13.25,-1.9609375,-7.25,-1.9296875,7.0625,-2.1875,-1.9921875,1.4296875,2.6875,3.484375,5.125,-0.58984375,3.375,-0.60546875,0.80859375,5.96875,-4.25,1.03125,3.359375,2.546875,5.21875,0.154296875,-0.44921875,-3.203125,8,2.25,-1.4140625,0.8359375,2.796875,-1.3046875,-2.34375,3.09375,-3.171875,2.96875,-4.9375,0.5859375,4.15625,0.65625,-3.890625,-3.4375,-2,-0.62890625,1.3828125,1.375,-2.59375,0.18359375,0.94921875,-4.1875,3.328125,-0.59375,0.140625,-5.53125,1.03125,4.65625,0.703125,-0.109375,-1.8515625,1.4453125,-0.8984375,4.3125,2.78125,-2.734375,0.2734375,2.21875,1.7421875,-0.125,1.03125,1.1328125,2.921875,-3.09375,-0.353515625,-0.44140625,-1.625,1.4765625,-3.1875,1.6640625,3.203125,1.3984375,-3.984375,2.21875,0.79296875,-0.11669921875,2.96875,-5.125,-1.9921875,-1.1015625,-0.71484375,-4.0625,-0.9140625,-4.375,-0.1455078125,5.46875,-5,3.4375,-2.515625,8.1875,0.1298828125,-1.421875,1.2890625,-2.828125,2.59375,-3.390625,-1.234375,3.484375,-0.92578125,2.125,-3.546875,1.8984375,-2.078125,-0.46484375,6.09375,-3.953125,-1.9765625,0.7421875,3.21875,-5.0625,-3.296875,0.1611328125,0.8515625,0.009765625,-1.8984375,1.4765625,-2.03125,4.4375,-4.75,3.390625,-4.65625,-3.90625,0.28125,0.07568359375,7.90625,4.25,-3.796875,-3.421875,-0.6015625,-7.0625,-3.421875,-3.859375,6.65625,-0.52734375,0.96875,2.078125,2.390625,-0.01031494140625,1.46875,-2.96875,3.203125,5.28125,0.294921875,3.046875,2.1875,-1.125,-4.40625,0.3125,-3.171875,7.0625,3.0625,0.404296875,3,-1.8984375,1.484375,-1.03125,-1.0625,-2.828125,2.171875,1.71875,-2.5,-3.28125,1.046875,-3.859375,0.72265625,-5.40625,-2.578125,-5.3125,2.765625,2.3125,-0.81640625,-0.7578125,4.4375,0.318359375,3.328125,-5.53125,-3.890625,3.8125,0.9765625,0.333984375,2.84375,-0.6796875,-5.03125,-0.9375,0.201171875,1.9140625,-4.1875,-3.609375,3.328125,2.46875,0.283203125,-3.9375,-4.40625,-3.453125,2.390625,4.1875,-0.96484375,0.353515625,0.06005859375,-1.53125,2.171875,-2.65625,4.5,-3.109375,-4.15625,-0.47265625,0.734375,3.578125,-3.203125,-1.0703125,1.4296875,-3.4375,0.7578125,1.2734375,-0.11279296875,-1.9453125,3.171875,-2,-3.65625,-5.4375,5.78125,-2.0625,0.45703125,-3.875,-2.65625,-3.1875,-1.421875,-0.6640625,1.7421875,0.0703125,5.78125,-0.63671875,2.8125,0.478515625,-0.8828125,0.0712890625,3.453125,-0.271484375,-2.90625,1.8359375,-4.59375,-4.65625,0.7578125,-8.0625,-2.0625,2.90625,-2.40625,2.671875,-2.671875,2.375,-1.1015625,-2.21875,-1.8203125,-0.8203125,0.83984375,5.375,2.171875,0.2216796875,0.38671875,1.8984375,0.859375,-1.109375,-1.8515625,-0.25,5.34375,0.62109375,2.765625,-3.359375,-2.34375,4.46875,-0.59375,-3.75,0.8984375,-0.357421875,0.6640625,4.5625,0.9609375,-3.796875,-2.9375,-6.15625,4.03125,0.73828125,1.828125,-4.625,1.5,-3.0625,0.1748046875,2.03125,-6.5625,-2.546875,3.328125,2.828125,5.46875,1.328125,-2.421875,-4.53125,2.203125,-0.396484375,-1.6171875,-2.234375,-1.7265625,-0.96875,-3.765625,4.125,-2.515625,4.25,-1.3359375,-2.8125,-0.8671875,0.61328125,-0.203125,0.47265625,-0.353515625,-0.88671875,4.0625,-0.3515625,7,2.171875,-4.0625,4.59375,2.515625,0.412109375,-1.5625,3.75,-1.109375,-2.3125,3.921875,2.890625,-4.0625,4.96875,2.125,3.375,-3.46875,-2.1875,-0.9921875,4.5625,0.287109375,1.28125,-4.34375,0.1630859375,4.0625,-0.1884765625,0.8671875,-1.765625,0.3046875,0.65234375,0.52734375,2,1.921875,3.4375,-0.52734375,1,-0.92578125,-1.2265625,2.328125,-0.1328125,-0.703125,-1.8828125,3.21875,-1.6953125,-1.875,-6,1.2421875,-3.46875,2.21875,3.1875,2.875,2.234375,-2.828125,-1.625,-2.640625,-5.25,-3.140625,1.75,1.09375,-1.75,1.875,-0.1181640625,2.546875,5.84375,0.130859375,4.6875,-3.109375,2.5,1.140625,0.875,0.046630859375,4.3125,-1.8203125,-2.21875,3.640625,-4.46875,3.71875,-4.53125,-3.078125,-0.63671875,-0.10986328125,2.640625,6.625,-4.5625,-3.953125,5.21875,1.328125,4.59375,3.78125,-2.078125,-1.484375,0.79296875,1.3515625,5.46875,0.93359375,2.953125,-2.734375,6.9375,5.65625,0.90625,2.359375,0.166015625,-2.6875,-6.4375,5.125,1.3984375,1.984375,-2.375,1.6875,3.109375,0.1533203125,3.640625,-5.5,0.8671875,1.2109375,0.90625,0.5234375,-3.15625,0.103515625,2.640625,0.33203125,-1.6875,5.84375,0.97265625,4.125,-0.72265625,3.34375,2.328125,3.703125,-2.03125,1.5234375,-3.46875,3.578125,-1.3984375,2.15625,-5.5,1.0546875,3.640625,4.3125,-1.625,-3.5625,2.21875,0.275390625,-0.5,-4.46875,4.21875,3.59375,2.5625,-6.9375,-3.328125,-0.05029296875,0.2060546875,1.234375,-3.484375,1.171875,1.6796875,-4.625,-3.265625,1.296875,1.625,-5.65625,-6.0625,-3.203125,1.65625,1.3203125,3.1875,3.21875,-0.8203125,3.40625,-0.55078125,3.046875,4.28125,-1.1328125,1.5546875,0.9375,-2.75,4.125,-0.263671875,-2.671875,1.5546875,-0.50390625,-2.140625,0.50390625,-2.296875,-1.0703125,-4.21875,-0.85546875,2.328125,-1.09375,5.125,-3.96875,0.30078125,3.609375,-1.4375,-2.28125,-2.65625,0.5703125,-2.921875,-2.578125,-1.9140625,3.609375,2.984375,2.046875,0.58203125,-0.6015625,-3.265625,-6.40625,-5.65625,3.578125,-2.515625,2.859375,0.439453125,-4.25,2.078125,2.8125,1.78125,-0.1640625,-0.55859375,2.765625,4.59375,0.455078125,-1.7265625,-0.466796875,3.609375,-4.5625,-3.78125,0.515625,1,-3.171875,2.28125,-3.125,-1.8359375,0.79296875,4.5,-0.5078125,-2.859375,-1.75,-2.40625,-2.875,-3.03125,-2.859375,2.5625,1.859375,3.296875,0.1689453125,-0.421875,-5,3.71875,16.875,0.9375,-4.71875,2.421875,-3.140625,2.65625,3.171875,4.8125,-1.7109375,-1.96875,-2.1875,1.765625,0.01031494140625,1.4140625,-2.140625,1.7421875,1.9921875,-0.48828125,-4.125,-1.9765625,-1.328125,0.84765625,-0.7578125,2.96875,0.408203125,2.265625,-0.734375,-0.259765625,0.2333984375,-3.234375,-4.46875,-4.4375,2.265625,-1.7578125,4.75,-4.25,5.375,0.1845703125,-2.9375,-2.09375,-3.296875,-3.171875,1.0234375,-0.75,-1.9453125,4.34375,-0.72265625,1.09375,0.37890625,-0.337890625,-3.546875,-3.046875,-2.6875,7.25,0.62890625,-5.71875,-1.546875,-4.84375,-4.5625,0.58984375,2.796875,-2.328125,1.6328125,1.453125,-1.828125,-2.171875,-1.953125,0.85546875,3,-5.125,-5.625,0.13671875,1.5546875,3.359375,2.796875,-4.0625,1.5703125,5.3125,2.6875,0.69140625,-0.75,1.4453125,-1.3828125,-2.5,-0.91015625,1.4609375,-4.03125,1.109375,1.4453125,-4.875,11.25,-8.625,4.8125,4.0625,-4.75,-0.1865234375,2.796875,1.796875,-1.6796875,-0.169921875,2.953125,2.453125,3.359375,-0.306640625,6.09375,1.5234375,0.388671875,0.73828125,2.9375,3.578125,2.4375,2.9375,-0.828125,-1.9609375,1.3046875,1.7734375,-2.484375,-3.46875,-1.4609375,-4.4375,6,1.6171875,-2.765625,-1.2578125,-10.5,-3.421875,-2.328125,-5.84375,4.5,-2.65625,2.46875,3.421875,-0.609375,-1.078125,-2.53125,-5,2.296875,4.0625,0.208984375,-0.3984375,-6.0625,2.84375,3.546875,-3.984375,-2.09375,1.4453125,-3.265625,3.296875,-0.1923828125,4.9375,-3.578125,3.9375,2.03125,-2.546875,-5.8125,3.171875,-3.765625,-2.234375,-5.3125,-2.453125,-2.078125,-3.328125,-0.6171875,-0.35546875,-2.078125,-1.03125,1.6171875,-0.60546875,-3.15625,2.921875,2.96875,-4.375,-2.625,0.58203125,0.73046875,-4.28125,1.1875,5.1875,-0.54296875,1.5,0.55078125,0.078125,-0.3203125,-4.34375,0.81640625,1.71875,-4.03125,-0.71875,-1.359375,-2.828125,-2.4375,-2.78125,-3.375,3.875,3.59375,-5.0625,1.9609375,-0.34765625,0.014892578125,-1.4453125,-1.546875,6.4375,2.234375,-1.6484375,5.59375,1.03125,-4.15625,-2,-2.046875,-1.1484375,-1.2734375,6.3125,1.2578125,2.375,-5.90625,7.53125,2.453125,1.7265625,-0.43359375,2.34375,1.6796875,-3.71875,-5.40625,2.46875,2.75,3.84375,-4.59375,0.6328125,0.53515625,0.53125,-4.28125,1.90625,-0.259765625,0.482421875,-3.140625,-7.59375,-0.109375,0.90625,-1.8828125,1.5234375,4.25,-2.96875,1.3828125,0.95703125,-0.58984375,3.640625,3.28125,-2.828125,1.90625,-0.1904296875,2.625,-2.34375,1.4921875,-3.71875,-4.96875,-3.109375,-1.765625,1.8828125,-2.625,0.67578125,-0.357421875,-4.1875,2.109375,-2.25,1.125,1.09375,0.2578125,-6.25,3.984375,5.1875,-4.15625,4.4375,-5.53125,-2.4375,-1.640625,2.21875,-1.9140625,-6.46875,2.0625,4.5,-3.390625,2.203125,3.546875,-1.625,-0.4453125,-2.25,5.3125,-1.015625,4.78125,-0.6953125,3.953125,3.9375,-1.28125,-0.061279296875,-5.125,0.470703125,-2.28125,-3.84375,5.53125,-1.921875,2.46875,5.21875,4.9375,-9,-1.96875,0.54296875,-0.1845703125,3.578125,3.109375,-1.3671875,1.0234375,0.028076171875,-0.30859375,4.4375,-0.9296875,-1.46875,-3.65625,4.96875,-0.1728515625,-4.0625,2.984375,2.609375,-4.15625,4.34375,-2.75,-2.6875,-0.6875,-0.1396484375,-5.625,1.8046875,2.6875,-0.92578125,3.4375,3.109375,1.203125,3.59375,-2.640625,-10.0625,0.0703125,2.75,5.3125,1.7265625,2.3125,0.0859375,-1.0625,3.640625,-4.5625,0.46875,-1.484375,-9.5,0.255859375,-4.15625,-1.609375,-3.453125,-1.4921875,-1.9453125,3.90625,1.3984375,-0.8515625,3.5,2.921875,0.453125,4.15625,-0.361328125,-3.578125,1.2734375,1.75,-5.28125,-1.90625,4.8125,3.578125,-2.203125,-2.0625,3.84375,-4.28125,-0.70703125,4.3125,4.28125,2.15625,-0.828125,-3.234375,2.84375,-2.546875,-2.828125,1.703125,-3.421875,2.453125,-1.4375,2.578125,1.296875,-2.640625,-2.03125,-4.15625,-2.71875,3.484375,0.28515625,0.9765625,-2.265625,-1.1171875,3.234375,3.5625,-2.359375,-2.109375,2.796875,-1.3515625,-4.28125,-1.0859375,1.0859375,-5.90625,-2.609375,2.734375,3.4375,-2.5625,-3.5625,-2.125,1.6171875,1.3046875,-0.8984375,-0.1318359375,-3.53125,2.65625,5.0625,-2.9375,-3.75,-1.6171875,-0.486328125,-5.03125,-3.609375,-0.1767578125,1.140625,-0.73046875,3.890625,-1.40625,0.47265625,4.4375,-3.65625,-3.21875,3.96875,3.359375,-3.203125,-1.46875,2.25,-3.375,1.03125,5.4375,-2.390625,-2.234375,0.41796875,-2.171875,-4.28125,2.34375,1.2265625,-3.734375,-7.875,5.96875,1.0703125,4.34375,4.125,-3.90625,4.0625,-4.6875,1.8828125,-1.265625,1.015625,1.3828125,-5.65625,-1.1875,-2.5,-3.5,0.5390625,-1.734375,-3.5625,0.66015625,8.0625,-1.328125,-2.59375,-2.953125,-3.515625,3.3125,-4.15625,-7.625,0.1181640625,-7.34375,1.734375,-2.1875,1.75,-5.59375,1.9140625,-1.078125,1.734375,-2.984375,0.27734375,-0.384765625,1.21875,0.54296875,4.6875,1.2109375,1.984375,-0.1484375,2.71875,0.0791015625,1.875,-1.453125,-0.4921875,1.21875,-1.234375,0.33203125,0.69921875,-2.734375,0.1708984375,-1.7578125,-0.263671875,-1.015625,1.7578125,2.9375,-0.640625,-0.291015625,-1.6875,1.703125,-4.5,1.3125,-1.796875,0.859375,-0.78515625,-1.0078125,1.9609375,-2.328125,1.6640625,1.015625,1.640625,0.01068115234375,-1.5,2.234375,2.6875,-0.031982421875,-2.328125,-1.8046875,-0.55859375,-1.7421875,1.7421875,0.55078125,-2.0625,2.9375,-1.640625,-0.41015625,0.890625,1.7265625,0.44140625,-1.6484375,2.40625,-1.8671875,1.2890625,1.0859375,-1.5234375,2.609375,0.63671875,1.03125,1.2734375,0.9765625,-2,0.64453125,0.2578125,-1.4375,-0.291015625,3.484375,-1.7265625,0.31640625,-1.078125,-0.5625,1.0859375,-0.8671875,1.2109375,0.15625,-0.396484375,-2.75,2.640625,-2.125,-1.2578125,-0.42578125,0.29296875,-0.5703125,0.8984375,0.08935546875,1.2109375,-0.29296875,2.28125,-0.73828125,2.171875,-0.020263671875,-0.2060546875,1.3359375,3.421875,-1.984375,0.7421875,-2.0625,-1.1328125,1.3203125,-0.3046875,1.15625,-0.93359375,-2,1.2421875,1.1328125,-2.984375,-0.734375,2.265625,-0.189453125,-1.1328125,-0.609375,1.2265625,-0.75390625,-0.38671875,0.419921875,-0.89453125,2,3.265625,-1.0625,2.5,-1.453125,0.396484375,0.73046875,1.046875,2.3125,0.07958984375,-2.34375,-0.9296875,2.71875,-1.4375,0.37109375,0.890625,-1.53125,-0.1396484375,1.3359375,0.5703125,1.640625,-0.06982421875,-1.859375,-0.330078125,-0.6796875,1.609375,1.65625,-1.6875,0.68359375,-1.8359375,-0.53125,-1.015625,2.765625,-1.7578125,-2.140625,-0.78515625,-1.1015625,-0.83203125,-0.498046875,0.11962890625,-0.1298828125,0.60546875,1.125,1.5,0.4296875,-0.609375,1.4375,-0.08056640625,0.68359375,-1.1875,-1.5234375,1.484375,1.2421875,2.34375,-1.359375,1.34375,0.9296875,0.8828125,-1.1796875,1.9453125,-0.5234375,0.314453125,0.010986328125,-0.1181640625,1.40625,2.21875,0.318359375,0.5859375,-0.1328125,1.40625,0.69921875,1.375,-1.3046875,-2.203125,-1.0078125,-1.4296875,-2.125,0.361328125,-0.0615234375,-1.3046875,-0.1904296875,0.034912109375,-0.86328125,1.375,1.1796875,1.5390625,-0.828125,-0.58203125,0.1787109375,-0.328125,0.25390625,0.8828125,-0.8046875,-0.78125,-1.1171875,-2.0625,1.578125,0.88671875,-1.09375,-0.2890625,2.0625,-1.5,1.0078125,-2.78125,0.55078125,-1.828125,-0.341796875,0.0859375,-3.265625,0.34765625,-0.12451171875,-2.15625,-3.078125,-1.75,-0.85546875,-2.375,-0.3203125,4,-0.81640625,-1.21875,2.03125,0.08203125,-1.0078125,-0.94921875,1.7578125,2.84375,-0.8203125,3.859375,0.349609375,-0.16015625,-1.3984375,-1.265625,0.52734375,-1.2890625,0.294921875,-0.84765625,-0.8046875,-1.6796875,-3.109375,0.05859375,-4.1875,-2.125,0.1337890625,0.90625,1.890625,-0.08447265625,-0.7421875,-0.56640625,-0.96875,2.796875,-0.267578125,0.18359375,1.4375,0.27734375,0.46875,-1.4140625,0.92578125,-0.84375,2.953125,-1.171875,-0.50390625,-2.65625,-1.5546875,-4.1875,1.453125,2.484375,0.421875,2.96875,1.3671875,-0.5546875,-2.5625,0.07421875,0.00909423828125,-4.75,-0.373046875,-0.7265625,0.07275390625,-1.4140625,-0.7109375,-0.1318359375,-0.609375,-1.328125,-0.51953125,-1.828125,-0.271484375,-2.28125,2.984375,1.7890625,1.875,2.3125,0.3125,-0.31640625,1.1875,2.359375,1.1484375,0.6953125,0.255859375,0.408203125,-1.09375,2.09375,0.337890625,0.4609375,-1.2265625,0.2275390625,1.1875,2.5625,1.734375,-0.76171875,0.85546875,0.328125,-1.9140625,-1.40625,0.31640625,0.296875,1.140625,0.333984375,1.03125,-1.2890625,0.416015625,-0.6875,0.9453125,1.7578125,-1.953125,1.109375,-0.134765625,0.1787109375,-1.5,1.203125,1.15625,1.8203125,-0.48046875,2.140625,1.1640625,0.48828125,1.8515625,2.609375,-0.361328125,1.421875,-0.86328125,1.953125,0.51953125,-2.484375,3.15625,-0.34375,-0.47265625,-0.56640625,1.2890625,1.359375,-0.60546875,-0.25,-0.38671875,2.015625,0.52734375,0.14453125,1.8828125,0.67578125,-0.546875,-0.77734375,-0.6015625,-1.09375,-2.328125,-1.0078125,-3.0625,-0.37109375,-0.9375,1.765625,-0.828125,-1.484375,-0.142578125,1.390625,-0.02099609375,1.3203125,1.6171875,-1.0859375,2.09375,0.154296875,0.1962890625,0.89453125,-0.97265625,-1.2421875,1.15625,0.82421875,-0.59765625,4.625,0.1962890625,2.28125,-0.65625,-1.0390625,-0.78515625,3.59375,-0.44921875,-0.4375,-1.6953125,1.140625,-0.296875,-1.25,-0.76953125,-1.3984375,-0.9765625,1.78125,-0.87109375,-3.234375,-2.171875,0.330078125,-1.875,0.48828125,-1.859375,-1.0390625,2.40625,1.734375,-0.63671875,0.216796875,1.125,-1.0234375,0.58984375,-0.4296875,0.3515625,1.6015625,-1.2109375,1.765625,0.5859375,2.796875,-3.921875,-0.298828125,2.171875,1.578125,-0.458984375,-1.015625,-0.51171875,2.109375,0.369140625,-0.018798828125,-0.50390625,-4.46875,0.0135498046875,-0.043212890625,-3.21875,-0.09423828125,0.4921875,1.2421875,0.6640625,-3.15625,0.73046875,-1.5078125,-1.6328125,3.46875,-0.55078125,-0.41796875,0.58203125,1.1640625,-0.83203125,-0.84765625,1.53125,0.17578125,-3.484375,-1.1015625,-0.1591796875,-0.875,0.59765625,0.01373291015625,0.099609375,0.546875,-0.36328125,-1.171875,-1.1328125,-0.33984375,-0.08056640625,1.015625,4,1.1484375,1.265625,1.2109375,-2.125,4.5625,-2.515625,-0.96484375,1.1015625,1.3515625,-1.1796875,3.921875,1.109375,0.2265625,-2,0.55859375,2.96875,0.765625,0.9453125,0.671875,1.28125,1.7421875,1.78125,-1,-1.8671875,1.5,-0.35546875,-2.5,0.012451171875,0.2578125],\"index\":1,\"object\":\"embedding\"}],\"model\":\"doubao-embedding-text-240715\",\"object\":\"list\",\"usage\":{\"prompt_tokens\":7,\"total_tokens\":7}}"),
                        new TypeReference<List<List<Float>>>() {});
        Assertions.assertEquals(2, lists.size());
        Assertions.assertEquals(2560, lists.get(0).size());
    }
}
