#!/usr/bin/env python3
#
# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
#

import optparse
import signal
import sys

from proton import symbol
from proton.handlers import MessagingHandler
from proton.reactor import Container


class FailoverServer(MessagingHandler):
    def __init__(self, address):
        super(FailoverServer, self).__init__()
        self.listener = None
        self.address = address

    def on_start(self, event):
        self.listener = event.container.listen(self.address)

    def stop(self):
        if self.listener:
            self.listener.close()

    def on_connection_opening(self, event):
        # Sends an empty failover list.
        # This will test the case where we deliberately send an empty failover list so that the router
        # receiving this open frame will clean out its failover list.
        event.connection.properties = {
            symbol('failover-server-list'): []
        }


parser = optparse.OptionParser(usage="usage: %prog [options]",
                               description="Testing Router failover support")

parser.add_option("-a", "--address", default="localhost:55671",
                  help="address to listen on (default %default)")

opts, args = parser.parse_args()

handler = FailoverServer(opts.address)


def sigterm_handler(_signo, _stack_frame):
    sys.exit(0)


signal.signal(signal.SIGTERM, sigterm_handler)

Container(handler).run()
