/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.storageengine;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.FileUtils;
import org.apache.iotdb.common.rpc.thrift.TFlushReq;
import org.apache.iotdb.common.rpc.thrift.TSStatus;
import org.apache.iotdb.common.rpc.thrift.TSetConfigurationReq;
import org.apache.iotdb.common.rpc.thrift.TSetTTLReq;
import org.apache.iotdb.common.rpc.thrift.TTimePartitionSlot;
import org.apache.iotdb.commons.concurrent.ExceptionalCountDownLatch;
import org.apache.iotdb.commons.concurrent.IoTDBThreadPoolFactory;
import org.apache.iotdb.commons.concurrent.ThreadName;
import org.apache.iotdb.commons.concurrent.threadpool.ScheduledExecutorUtil;
import org.apache.iotdb.commons.conf.CommonDescriptor;
import org.apache.iotdb.commons.conf.ConfigurationFileUtils;
import org.apache.iotdb.commons.conf.TrimProperties;
import org.apache.iotdb.commons.consensus.DataRegionId;
import org.apache.iotdb.commons.consensus.index.ProgressIndex;
import org.apache.iotdb.commons.exception.IllegalPathException;
import org.apache.iotdb.commons.exception.ShutdownException;
import org.apache.iotdb.commons.exception.StartupException;
import org.apache.iotdb.commons.file.SystemFileFactory;
import org.apache.iotdb.commons.service.IService;
import org.apache.iotdb.commons.service.ServiceType;
import org.apache.iotdb.commons.utils.PathUtils;
import org.apache.iotdb.commons.utils.TestOnly;
import org.apache.iotdb.commons.utils.TimePartitionUtils;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.exception.DataRegionException;
import org.apache.iotdb.db.exception.StorageEngineException;
import org.apache.iotdb.db.exception.TsFileProcessorException;
import org.apache.iotdb.db.exception.WriteProcessRejectException;
import org.apache.iotdb.db.exception.runtime.StorageEngineFailureException;
import org.apache.iotdb.db.pipe.agent.PipeDataNodeAgent;
import org.apache.iotdb.db.queryengine.plan.analyze.cache.schema.DataNodeTTLCache;
import org.apache.iotdb.db.queryengine.plan.planner.plan.node.load.LoadTsFilePieceNode;
import org.apache.iotdb.db.queryengine.plan.scheduler.load.LoadTsFileScheduler;
import org.apache.iotdb.db.service.metrics.FileMetrics;
import org.apache.iotdb.db.service.metrics.WritingMetrics;
import org.apache.iotdb.db.storageengine.buffer.BloomFilterCache;
import org.apache.iotdb.db.storageengine.buffer.ChunkCache;
import org.apache.iotdb.db.storageengine.buffer.TimeSeriesMetadataCache;
import org.apache.iotdb.db.storageengine.dataregion.DataRegion;
import org.apache.iotdb.db.storageengine.dataregion.compaction.repair.UnsortedFileRepairTaskScheduler;
import org.apache.iotdb.db.storageengine.dataregion.compaction.schedule.CompactionScheduleTaskManager;
import org.apache.iotdb.db.storageengine.dataregion.flush.CloseFileListener;
import org.apache.iotdb.db.storageengine.dataregion.flush.CompressionRatio;
import org.apache.iotdb.db.storageengine.dataregion.flush.FlushListener;
import org.apache.iotdb.db.storageengine.dataregion.flush.TsFileFlushPolicy;
import org.apache.iotdb.db.storageengine.dataregion.wal.WALManager;
import org.apache.iotdb.db.storageengine.dataregion.wal.exception.WALException;
import org.apache.iotdb.db.storageengine.dataregion.wal.recover.WALRecoverManager;
import org.apache.iotdb.db.storageengine.load.LoadTsFileManager;
import org.apache.iotdb.db.storageengine.load.limiter.LoadTsFileRateLimiter;
import org.apache.iotdb.db.storageengine.rescon.memory.SystemInfo;
import org.apache.iotdb.db.utils.ThreadUtils;
import org.apache.iotdb.rpc.RpcUtils;
import org.apache.iotdb.rpc.TSStatusCode;
import org.apache.tsfile.exception.write.PageException;
import org.apache.tsfile.utils.FilePathUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StorageEngine
implements IService {
    private static final Logger LOGGER = LoggerFactory.getLogger(StorageEngine.class);
    private static final IoTDBConfig CONFIG = IoTDBDescriptor.getInstance().getConfig();
    private static final WritingMetrics WRITING_METRICS = WritingMetrics.getInstance();
    private static final String systemDir = FilePathUtils.regularizePath((String)CONFIG.getSystemDir()) + "databases";
    private final ConcurrentHashMap<DataRegionId, DataRegion> dataRegionMap = new ConcurrentHashMap();
    private final ConcurrentHashMap<DataRegionId, DataRegion> deletingDataRegionMap = new ConcurrentHashMap();
    private AtomicInteger readyDataRegionNum;
    private final AtomicBoolean isReadyForReadAndWrite = new AtomicBoolean();
    private final AtomicBoolean isReadyForNonReadWriteFunctions = new AtomicBoolean();
    private ScheduledExecutorService seqMemtableTimedFlushCheckThread;
    private ScheduledExecutorService unseqMemtableTimedFlushCheckThread;
    private final TsFileFlushPolicy fileFlushPolicy = new TsFileFlushPolicy.DirectFlushPolicy();
    private ExecutorService cachedThreadPool;
    private final List<CloseFileListener> customCloseFileListeners = new ArrayList<CloseFileListener>();
    private final List<FlushListener> customFlushListeners = new ArrayList<FlushListener>();
    private int recoverDataRegionNum = 0;
    private final LoadTsFileManager loadTsFileManager = new LoadTsFileManager();

    private StorageEngine() {
    }

    public static StorageEngine getInstance() {
        return InstanceHolder.INSTANCE;
    }

    private static void initTimePartition() {
        TimePartitionUtils.setTimePartitionInterval((long)CommonDescriptor.getInstance().getConfig().getTimePartitionInterval());
    }

    public static void blockInsertionIfReject() throws WriteProcessRejectException {
        long startTime = System.currentTimeMillis();
        while (SystemInfo.getInstance().isRejected()) {
            try {
                TimeUnit.MILLISECONDS.sleep(CONFIG.getCheckPeriodWhenInsertBlocked());
                if (System.currentTimeMillis() - startTime <= (long)CONFIG.getMaxWaitingTimeWhenInsertBlocked()) continue;
                throw new WriteProcessRejectException("System rejected over " + (System.currentTimeMillis() - startTime) + "ms");
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    public boolean isReadyForReadAndWrite() {
        return this.isReadyForReadAndWrite.get();
    }

    public boolean isReadyForNonReadWriteFunctions() {
        return this.isReadyForNonReadWriteFunctions.get();
    }

    private void asyncRecoverDataRegion() throws StartupException {
        long startRecoverTime = System.currentTimeMillis();
        this.isReadyForNonReadWriteFunctions.set(false);
        this.isReadyForReadAndWrite.set(false);
        this.cachedThreadPool = IoTDBThreadPoolFactory.newCachedThreadPool((String)ThreadName.STORAGE_ENGINE_CACHED_POOL.getName());
        LinkedList<Future<Void>> futures = new LinkedList<Future<Void>>();
        this.asyncRecover(futures);
        if (!CONFIG.getDataRegionConsensusProtocolClass().equals("org.apache.iotdb.consensus.ratis.RatisConsensus")) {
            try {
                WALRecoverManager.getInstance().recover();
            }
            catch (WALException e) {
                LOGGER.error("Fail to recover wal.", (Throwable)((Object)e));
            }
        }
        Thread recoverEndTrigger = new Thread(() -> {
            this.checkResults(futures, "StorageEngine failed to recover.");
            this.isReadyForReadAndWrite.set(true);
            LOGGER.info("Storage Engine recover cost: {}s.", (Object)((System.currentTimeMillis() - startRecoverTime) / 1000L));
        }, ThreadName.STORAGE_ENGINE_RECOVER_TRIGGER.getName());
        recoverEndTrigger.start();
    }

    private void asyncRecover(List<Future<Void>> futures) {
        Map<String, List<DataRegionId>> localDataRegionInfo = this.getLocalDataRegionInfo();
        localDataRegionInfo.values().forEach(list -> this.recoverDataRegionNum += list.size());
        this.readyDataRegionNum = new AtomicInteger(0);
        WALRecoverManager.getInstance().setAllDataRegionScannedLatch(new ExceptionalCountDownLatch(this.recoverDataRegionNum));
        for (Map.Entry<String, List<DataRegionId>> entry : localDataRegionInfo.entrySet()) {
            String sgName = entry.getKey();
            for (DataRegionId dataRegionId : entry.getValue()) {
                Callable<Void> recoverDataRegionTask = () -> {
                    DataRegion dataRegion;
                    try {
                        dataRegion = this.buildNewDataRegion(sgName, dataRegionId);
                    }
                    catch (DataRegionException e) {
                        LOGGER.error("Failed to recover data region {}[{}]", new Object[]{sgName, dataRegionId.getId(), e});
                        return null;
                    }
                    this.dataRegionMap.put(dataRegionId, dataRegion);
                    LOGGER.info("Data regions have been recovered {}/{}", (Object)this.readyDataRegionNum.incrementAndGet(), (Object)this.recoverDataRegionNum);
                    return null;
                };
                futures.add(this.cachedThreadPool.submit(recoverDataRegionTask));
            }
        }
    }

    public Map<String, List<DataRegionId>> getLocalDataRegionInfo() {
        File system = SystemFileFactory.INSTANCE.getFile(systemDir);
        File[] sgDirs = system.listFiles();
        HashMap<String, List<DataRegionId>> localDataRegionInfo = new HashMap<String, List<DataRegionId>>();
        if (sgDirs == null) {
            return localDataRegionInfo;
        }
        for (File sgDir : sgDirs) {
            if (!sgDir.isDirectory()) continue;
            String sgName = sgDir.getName();
            ArrayList<DataRegionId> dataRegionIdList = new ArrayList<DataRegionId>();
            for (File dataRegionDir : Objects.requireNonNull(sgDir.listFiles())) {
                if (!dataRegionDir.isDirectory()) continue;
                dataRegionIdList.add(new DataRegionId(Integer.parseInt(dataRegionDir.getName())));
            }
            localDataRegionInfo.put(sgName, dataRegionIdList);
        }
        return localDataRegionInfo;
    }

    public void start() throws StartupException {
        this.recoverDataRegionNum = 0;
        StorageEngine.initTimePartition();
        try {
            FileUtils.forceMkdir((File)SystemFileFactory.INSTANCE.getFile(systemDir));
        }
        catch (IOException e) {
            throw new StorageEngineFailureException(e);
        }
        this.asyncRecoverDataRegion();
        this.startTimedService();
        while (!this.isReadyForReadAndWrite.get()) {
            try {
                TimeUnit.MILLISECONDS.sleep(100L);
            }
            catch (InterruptedException e) {
                LOGGER.warn("Storage engine failed to set up.", (Throwable)e);
                Thread.currentThread().interrupt();
                return;
            }
        }
        this.asyncRecoverTsFileResource();
    }

    private void startTimedService() {
        if (CONFIG.isEnableTimedFlushSeqMemtable()) {
            this.seqMemtableTimedFlushCheckThread = IoTDBThreadPoolFactory.newSingleThreadScheduledExecutor((String)ThreadName.TIMED_FLUSH_SEQ_MEMTABLE.getName());
            ScheduledExecutorUtil.safelyScheduleAtFixedRate((ScheduledExecutorService)this.seqMemtableTimedFlushCheckThread, this::timedFlushSeqMemTable, (long)CONFIG.getSeqMemtableFlushCheckInterval(), (long)CONFIG.getSeqMemtableFlushCheckInterval(), (TimeUnit)TimeUnit.MILLISECONDS);
            LOGGER.info("start sequence memtable timed flush check thread successfully.");
        }
        if (CONFIG.isEnableTimedFlushUnseqMemtable()) {
            this.unseqMemtableTimedFlushCheckThread = IoTDBThreadPoolFactory.newSingleThreadScheduledExecutor((String)ThreadName.TIMED_FLUSH_UNSEQ_MEMTABLE.getName());
            ScheduledExecutorUtil.safelyScheduleAtFixedRate((ScheduledExecutorService)this.unseqMemtableTimedFlushCheckThread, this::timedFlushUnseqMemTable, (long)CONFIG.getUnseqMemtableFlushCheckInterval(), (long)CONFIG.getUnseqMemtableFlushCheckInterval(), (TimeUnit)TimeUnit.MILLISECONDS);
            LOGGER.info("start unsequence memtable timed flush check thread successfully.");
        }
    }

    private void timedFlushSeqMemTable() {
        for (DataRegion dataRegion : this.dataRegionMap.values()) {
            if (dataRegion == null) continue;
            dataRegion.timedFlushSeqMemTable();
        }
    }

    private void timedFlushUnseqMemTable() {
        for (DataRegion dataRegion : this.dataRegionMap.values()) {
            if (dataRegion == null) continue;
            dataRegion.timedFlushUnseqMemTable();
        }
    }

    private void asyncRecoverTsFileResource() {
        LinkedList<Future<Void>> futures = new LinkedList<Future<Void>>();
        long startRecoverTime = System.currentTimeMillis();
        for (DataRegion dataRegion : this.dataRegionMap.values()) {
            List<Callable<Void>> asyncTsFileResourceRecoverTasks;
            if (dataRegion == null || (asyncTsFileResourceRecoverTasks = dataRegion.getAsyncTsFileResourceRecoverTaskList()) == null) continue;
            Callable<Void> taskOfRegion = () -> {
                for (Callable task : asyncTsFileResourceRecoverTasks) {
                    task.call();
                }
                dataRegion.clearAsyncTsFileResourceRecoverTaskList();
                dataRegion.initCompactionSchedule();
                return null;
            };
            futures.add(this.cachedThreadPool.submit(taskOfRegion));
        }
        Thread recoverEndTrigger = new Thread(() -> {
            this.checkResults(futures, "async recover tsfile resource meets error.");
            this.recoverRepairData();
            this.isReadyForNonReadWriteFunctions.set(true);
            LOGGER.info("TsFile Resource recover cost: {}s.", (Object)((System.currentTimeMillis() - startRecoverTime) / 1000L));
        }, ThreadName.STORAGE_ENGINE_RECOVER_TRIGGER.getName());
        recoverEndTrigger.start();
    }

    public void stop() {
        for (DataRegion dataRegion : this.dataRegionMap.values()) {
            if (dataRegion == null) continue;
            CompactionScheduleTaskManager.getInstance().unregisterDataRegion(dataRegion);
        }
        this.syncCloseAllProcessor();
        ThreadUtils.stopThreadPool(this.seqMemtableTimedFlushCheckThread, ThreadName.TIMED_FLUSH_SEQ_MEMTABLE);
        ThreadUtils.stopThreadPool(this.unseqMemtableTimedFlushCheckThread, ThreadName.TIMED_FLUSH_UNSEQ_MEMTABLE);
        if (this.cachedThreadPool != null) {
            this.cachedThreadPool.shutdownNow();
        }
        this.dataRegionMap.clear();
    }

    public void shutdown(long milliseconds) throws ShutdownException {
        try {
            for (DataRegion dataRegion : this.dataRegionMap.values()) {
                if (dataRegion == null) continue;
                CompactionScheduleTaskManager.getInstance().unregisterDataRegion(dataRegion);
            }
            this.forceCloseAllProcessor();
        }
        catch (TsFileProcessorException e) {
            throw new ShutdownException((Throwable)((Object)e));
        }
        this.shutdownTimedService(this.seqMemtableTimedFlushCheckThread, "SeqMemtableTimedFlushCheckThread");
        this.shutdownTimedService(this.unseqMemtableTimedFlushCheckThread, "UnseqMemtableTimedFlushCheckThread");
        this.cachedThreadPool.shutdownNow();
        this.dataRegionMap.clear();
    }

    private void shutdownTimedService(ScheduledExecutorService pool, String poolName) {
        if (pool != null) {
            pool.shutdownNow();
            try {
                pool.awaitTermination(30L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                LOGGER.warn("{} still doesn't exit after 30s", (Object)poolName);
                Thread.currentThread().interrupt();
            }
        }
    }

    public ServiceType getID() {
        return ServiceType.STORAGE_ENGINE_SERVICE;
    }

    public DataRegion buildNewDataRegion(String databaseName, DataRegionId dataRegionId) throws DataRegionException {
        LOGGER.info("construct a data region instance, the database is {}, Thread is {}", (Object)databaseName, (Object)Thread.currentThread().getId());
        DataRegion dataRegion = new DataRegion(systemDir + File.separator + databaseName, String.valueOf(dataRegionId.getId()), this.fileFlushPolicy, databaseName);
        WRITING_METRICS.createFlushingMemTableStatusMetrics(dataRegionId);
        WRITING_METRICS.createDataRegionMemoryCostMetrics(dataRegion);
        WRITING_METRICS.createActiveMemtableCounterMetrics(dataRegionId);
        dataRegion.setCustomFlushListeners(this.customFlushListeners);
        dataRegion.setCustomCloseFileListeners(this.customCloseFileListeners);
        return dataRegion;
    }

    @TestOnly
    public synchronized void reset() {
        this.dataRegionMap.clear();
    }

    public void syncCloseAllProcessor() {
        LOGGER.info("Start closing all database processor");
        ArrayList tasks = new ArrayList();
        for (DataRegion dataRegion : this.dataRegionMap.values()) {
            if (dataRegion == null) continue;
            tasks.add(this.cachedThreadPool.submit(() -> {
                dataRegion.syncCloseAllWorkingTsFileProcessors();
                return null;
            }));
        }
        this.checkResults(tasks, "Failed to sync close processor.");
    }

    public void forceCloseAllProcessor() throws TsFileProcessorException {
        LOGGER.info("Start force closing all database processor");
        ArrayList tasks = new ArrayList();
        for (DataRegion dataRegion : this.dataRegionMap.values()) {
            if (dataRegion == null) continue;
            tasks.add(this.cachedThreadPool.submit(() -> {
                dataRegion.forceCloseAllWorkingTsFileProcessors();
                return null;
            }));
        }
        this.checkResults(tasks, "Failed to force close processor.");
    }

    public void syncCloseProcessorsInDatabase(String databaseName) {
        ArrayList tasks = new ArrayList();
        for (DataRegion dataRegion : this.dataRegionMap.values()) {
            if (dataRegion == null || !dataRegion.getDatabaseName().equals(databaseName)) continue;
            tasks.add(this.cachedThreadPool.submit(() -> {
                dataRegion.syncCloseAllWorkingTsFileProcessors();
                return null;
            }));
        }
        this.checkResults(tasks, "Failed to sync close processor.");
    }

    public void syncCloseProcessorsInRegion(List<String> dataRegionIds) {
        ArrayList tasks = new ArrayList();
        for (DataRegion dataRegion : this.dataRegionMap.values()) {
            if (dataRegion == null || !dataRegionIds.contains(dataRegion.getDataRegionId())) continue;
            tasks.add(this.cachedThreadPool.submit(() -> {
                dataRegion.syncCloseAllWorkingTsFileProcessors();
                return null;
            }));
        }
        this.checkResults(tasks, "Failed to sync close processor.");
    }

    public void syncCloseProcessorsInDatabase(String databaseName, boolean isSeq) {
        ArrayList tasks = new ArrayList();
        for (DataRegion dataRegion : this.dataRegionMap.values()) {
            if (!dataRegion.getDatabaseName().equals(databaseName)) continue;
            tasks.add(this.cachedThreadPool.submit(() -> {
                dataRegion.syncCloseWorkingTsFileProcessors(isSeq);
                return null;
            }));
        }
        this.checkResults(tasks, "Failed to close database processor.");
    }

    private <V> void checkResults(List<Future<V>> tasks, String errorMsg) {
        for (Future<V> task : tasks) {
            try {
                task.get();
            }
            catch (ExecutionException e) {
                throw new StorageEngineFailureException(errorMsg, e);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new StorageEngineFailureException(errorMsg, e);
            }
        }
    }

    public void mergeAll() throws StorageEngineException {
        if (CommonDescriptor.getInstance().getConfig().isReadOnly()) {
            throw new StorageEngineException("Current system mode is read only, does not support merge");
        }
        this.dataRegionMap.values().forEach(DataRegion::compact);
    }

    public boolean repairData() throws StorageEngineException {
        if (CommonDescriptor.getInstance().getConfig().isReadOnly()) {
            throw new StorageEngineException("Current system mode is read only, does not support merge");
        }
        if (!CompactionScheduleTaskManager.getRepairTaskManagerInstance().markRepairTaskStart()) {
            return false;
        }
        LOGGER.info("start repair data");
        ArrayList<DataRegion> dataRegionList = new ArrayList<DataRegion>(this.dataRegionMap.values());
        this.cachedThreadPool.submit(new UnsortedFileRepairTaskScheduler(dataRegionList, false));
        return true;
    }

    public void stopRepairData() throws StorageEngineException {
        CompactionScheduleTaskManager.RepairDataTaskManager repairDataTaskManager = CompactionScheduleTaskManager.getRepairTaskManagerInstance();
        if (!CompactionScheduleTaskManager.getRepairTaskManagerInstance().hasRunningRepairTask()) {
            return;
        }
        LOGGER.info("stop repair data");
        try {
            repairDataTaskManager.markRepairTaskStopping();
            repairDataTaskManager.abortRepairTask();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public void recoverRepairData() {
        ArrayList<DataRegion> dataRegionList = new ArrayList<DataRegion>(this.dataRegionMap.values());
        String repairLogDirPath = IoTDBDescriptor.getInstance().getConfig().getSystemDir() + File.separator + "repair";
        File repairLogDir = new File(repairLogDirPath);
        if (!repairLogDir.exists() || !repairLogDir.isDirectory()) {
            return;
        }
        File[] files = repairLogDir.listFiles();
        List fileList = Stream.of(files == null ? new File[]{} : files).filter(f -> {
            String fileName = f.getName();
            return f.isFile() && ("repair-data.progress".equals(fileName) || "repair-data.stopped".equals(fileName));
        }).collect(Collectors.toList());
        if (!fileList.isEmpty()) {
            CompactionScheduleTaskManager.getRepairTaskManagerInstance().markRepairTaskStart();
            this.cachedThreadPool.submit(new UnsortedFileRepairTaskScheduler(dataRegionList, true));
        }
    }

    public void operateFlush(TFlushReq req) {
        if (req.getRegionIds() != null && !req.getRegionIds().isEmpty()) {
            StorageEngine.getInstance().syncCloseProcessorsInRegion(req.getRegionIds());
        } else if (req.storageGroups == null || req.storageGroups.isEmpty()) {
            StorageEngine.getInstance().syncCloseAllProcessor();
            WALManager.getInstance().syncDeleteOutdatedFilesInWALNodes();
        } else {
            for (String databaseName : req.storageGroups) {
                if (req.isSeq == null) {
                    StorageEngine.getInstance().syncCloseProcessorsInDatabase(databaseName);
                    continue;
                }
                StorageEngine.getInstance().syncCloseProcessorsInDatabase(databaseName, Boolean.parseBoolean(req.isSeq));
            }
        }
    }

    public void clearCache() {
        ChunkCache.getInstance().clear();
        TimeSeriesMetadataCache.getInstance().clear();
        BloomFilterCache.getInstance().clear();
    }

    public TSStatus setConfiguration(TSetConfigurationReq req) {
        TSStatus tsStatus = new TSStatus(TSStatusCode.SUCCESS_STATUS.getStatusCode());
        Map newConfigItems = req.getConfigs();
        if (newConfigItems.isEmpty()) {
            return tsStatus;
        }
        TrimProperties newConfigProperties = new TrimProperties();
        newConfigProperties.putAll(newConfigItems);
        URL configFileUrl = IoTDBDescriptor.getPropsUrl("iotdb-system.properties");
        if (configFileUrl == null || !new File(configFileUrl.getFile()).exists()) {
            String msg = "Unable to find the configuration file. Some modifications are made only in memory.";
            tsStatus = RpcUtils.getStatus((TSStatusCode)TSStatusCode.EXECUTE_STATEMENT_ERROR, (String)msg);
            LOGGER.warn(msg);
            try {
                IoTDBDescriptor.getInstance().loadHotModifiedProps(newConfigProperties);
                IoTDBDescriptor.getInstance().reloadMetricProperties(newConfigProperties);
            }
            catch (Exception e) {
                return RpcUtils.getStatus((TSStatusCode)TSStatusCode.EXECUTE_STATEMENT_ERROR, (String)e.getMessage());
            }
            return tsStatus;
        }
        try {
            ConfigurationFileUtils.updateConfiguration((File)new File(configFileUrl.getFile()), (Properties)newConfigProperties, mergedProperties -> {
                try {
                    IoTDBDescriptor.getInstance().loadHotModifiedProps(mergedProperties);
                }
                catch (Exception e) {
                    throw new IllegalArgumentException(e);
                }
            });
        }
        catch (Exception e) {
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            return RpcUtils.getStatus((TSStatusCode)TSStatusCode.EXECUTE_STATEMENT_ERROR, (String)e.getMessage());
        }
        return tsStatus;
    }

    public void registerFlushListener(FlushListener listener) {
        this.customFlushListeners.add(listener);
    }

    public void registerCloseFileListener(CloseFileListener listener) {
        this.customCloseFileListeners.add(listener);
    }

    private void makeSureNoOldRegion(DataRegionId regionId) {
        while (this.deletingDataRegionMap.containsKey(regionId)) {
            DataRegion oldRegion = this.deletingDataRegionMap.get(regionId);
            if (oldRegion == null) continue;
            oldRegion.waitForDeleted();
        }
    }

    public void createDataRegion(DataRegionId regionId, String databaseName) throws DataRegionException {
        this.makeSureNoOldRegion(regionId);
        AtomicReference<Object> exceptionAtomicReference = new AtomicReference<Object>(null);
        this.dataRegionMap.computeIfAbsent(regionId, region -> {
            try {
                return this.buildNewDataRegion(databaseName, (DataRegionId)region);
            }
            catch (DataRegionException e) {
                exceptionAtomicReference.set(e);
                return null;
            }
        });
        if (exceptionAtomicReference.get() != null) {
            throw (DataRegionException)exceptionAtomicReference.get();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteDataRegion(DataRegionId regionId) {
        if (!this.dataRegionMap.containsKey(regionId) || this.deletingDataRegionMap.containsKey(regionId)) {
            return;
        }
        DataRegion region = this.deletingDataRegionMap.computeIfAbsent(regionId, k -> this.dataRegionMap.remove(regionId));
        if (region != null) {
            LOGGER.info("Removing data region {}", (Object)regionId);
            region.markDeleted();
            try {
                region.abortCompaction();
                region.syncDeleteDataFiles();
                region.deleteFolder(systemDir);
                region.deleteDALFolderAndClose();
                PipeDataNodeAgent.receiver().pipeConsensus().releaseReceiverResource(regionId);
                switch (CONFIG.getDataRegionConsensusProtocolClass()) {
                    case "org.apache.iotdb.consensus.iot.IoTConsensus": 
                    case "org.apache.iotdb.consensus.iot.IoTConsensusV2": {
                        WALManager.getInstance().deleteWALNode(region.getDatabaseName() + "-" + region.getDataRegionId());
                        for (String dataDir : CONFIG.getLocalDataDirs()) {
                            File regionSnapshotDir = new File(dataDir + File.separator + "snapshot", region.getDatabaseName() + "-" + regionId.getId());
                            if (!regionSnapshotDir.exists()) continue;
                            try {
                                FileUtils.deleteDirectory((File)regionSnapshotDir);
                            }
                            catch (IOException e) {
                                LOGGER.error("Failed to delete snapshot dir {}", (Object)regionSnapshotDir, (Object)e);
                            }
                        }
                        break;
                    }
                    case "org.apache.iotdb.consensus.simple.SimpleConsensus": {
                        WALManager.getInstance().deleteRegionAndMayDeleteWALNode(region.getDatabaseName(), region.getDataRegionId());
                        break;
                    }
                }
                WRITING_METRICS.removeDataRegionMemoryCostMetrics(regionId);
                WRITING_METRICS.removeFlushingMemTableStatusMetrics(regionId);
                WRITING_METRICS.removeActiveMemtableCounterMetrics(regionId);
                FileMetrics.getInstance().deleteRegion(region.getDatabaseName(), region.getDataRegionId());
                CompressionRatio.getInstance().removeDataRegionRatio(String.valueOf(regionId.getId()));
                LOGGER.info("Removed data region {}", (Object)regionId);
            }
            catch (Exception e) {
                LOGGER.error("Error occurs when deleting data region {}-{}", new Object[]{region.getDatabaseName(), region.getDataRegionId(), e});
            }
            finally {
                this.deletingDataRegionMap.remove(regionId);
            }
        }
    }

    public boolean runIfAbsent(DataRegionId regionId, Runnable runnable) {
        AtomicBoolean result = new AtomicBoolean(false);
        this.dataRegionMap.computeIfAbsent(regionId, k -> {
            runnable.run();
            result.set(true);
            return null;
        });
        return result.get();
    }

    public boolean runIfPresent(DataRegionId regionId, Consumer<DataRegion> consumer) {
        AtomicBoolean result = new AtomicBoolean(false);
        this.dataRegionMap.computeIfPresent(regionId, (id, region) -> {
            consumer.accept((DataRegion)region);
            result.set(true);
            return region;
        });
        return result.get();
    }

    public DataRegion getDataRegion(DataRegionId regionId) {
        return this.dataRegionMap.get(regionId);
    }

    public List<DataRegion> getAllDataRegions() {
        return new ArrayList<DataRegion>(this.dataRegionMap.values());
    }

    public List<DataRegionId> getAllDataRegionIds() {
        return new ArrayList<DataRegionId>(this.dataRegionMap.keySet());
    }

    public int getDataRegionNumber() {
        return this.dataRegionMap.size();
    }

    public void setDataRegion(DataRegionId regionId, DataRegion newRegion) {
        if (this.dataRegionMap.containsKey(regionId)) {
            DataRegion oldRegion = this.dataRegionMap.get(regionId);
            oldRegion.markDeleted();
            oldRegion.abortCompaction();
            oldRegion.syncCloseAllWorkingTsFileProcessors();
        }
        WRITING_METRICS.createFlushingMemTableStatusMetrics(regionId);
        WRITING_METRICS.createDataRegionMemoryCostMetrics(newRegion);
        WRITING_METRICS.createActiveMemtableCounterMetrics(regionId);
        this.dataRegionMap.put(regionId, newRegion);
    }

    public TSStatus setTTL(TSetTTLReq req) throws IllegalPathException {
        String[] path = PathUtils.splitPathToDetachedNodes((String)((String)req.getPathPattern().get(0)));
        long ttl = req.getTTL();
        boolean isDataBase = req.isDataBase;
        if (ttl == -1L) {
            DataNodeTTLCache.getInstance().unsetTTLForTree(path);
            if (isDataBase) {
                String[] pathWithWildcard = Arrays.copyOf(path, path.length + 1);
                pathWithWildcard[pathWithWildcard.length - 1] = "**";
                DataNodeTTLCache.getInstance().unsetTTLForTree(pathWithWildcard);
            }
        } else {
            DataNodeTTLCache.getInstance().setTTLForTree(path, ttl);
            if (isDataBase) {
                String[] pathWithWildcard = Arrays.copyOf(path, path.length + 1);
                pathWithWildcard[pathWithWildcard.length - 1] = "**";
                DataNodeTTLCache.getInstance().setTTLForTree(pathWithWildcard, ttl);
            }
        }
        return RpcUtils.getStatus((TSStatusCode)TSStatusCode.SUCCESS_STATUS);
    }

    public TsFileFlushPolicy getFileFlushPolicy() {
        return this.fileFlushPolicy;
    }

    public TSStatus writeLoadTsFileNode(DataRegionId dataRegionId, LoadTsFilePieceNode pieceNode, String uuid) {
        TSStatus status = new TSStatus();
        if (CommonDescriptor.getInstance().getConfig().isReadOnly()) {
            status.setCode(TSStatusCode.SYSTEM_READ_ONLY.getStatusCode());
            status.setMessage("Current system mode is read only, does not support load file");
            return status;
        }
        LoadTsFileRateLimiter.getInstance().acquire(pieceNode.getDataSize());
        DataRegion dataRegion = this.getDataRegion(dataRegionId);
        if (dataRegion == null) {
            LOGGER.warn("DataRegion {} not found on this DataNode when writing piece nodeof TsFile {} (maybe due to region migration), will skip.", (Object)dataRegionId, (Object)pieceNode.getTsFile());
            return RpcUtils.SUCCESS_STATUS;
        }
        try {
            this.loadTsFileManager.writeToDataRegion(dataRegion, pieceNode, uuid);
        }
        catch (IOException | PageException e) {
            LOGGER.warn("IO error when writing piece node of TsFile {} to DataRegion {}.", new Object[]{pieceNode.getTsFile(), dataRegionId, e});
            status.setCode(TSStatusCode.LOAD_FILE_ERROR.getStatusCode());
            status.setMessage(e.getMessage());
            return status;
        }
        catch (Exception e) {
            LOGGER.warn("Exception occurred when writing piece node of TsFile {} to DataRegion {}.", new Object[]{pieceNode.getTsFile(), dataRegionId, e});
            status.setCode(TSStatusCode.LOAD_FILE_ERROR.getStatusCode());
            status.setMessage(e.getMessage());
            return status;
        }
        return RpcUtils.SUCCESS_STATUS;
    }

    public TSStatus executeLoadCommand(LoadTsFileScheduler.LoadCommand loadCommand, String uuid, boolean isGeneratedByPipe, Map<TTimePartitionSlot, ProgressIndex> timePartitionProgressIndexMap) {
        TSStatus status = new TSStatus();
        try {
            switch (loadCommand) {
                case EXECUTE: {
                    if (this.loadTsFileManager.loadAll(uuid, isGeneratedByPipe, timePartitionProgressIndexMap)) {
                        status = RpcUtils.SUCCESS_STATUS;
                        break;
                    }
                    status.setCode(TSStatusCode.LOAD_FILE_ERROR.getStatusCode());
                    status.setMessage(String.format("No load TsFile uuid %s recorded for execute load command %s.", new Object[]{uuid, loadCommand}));
                    break;
                }
                case ROLLBACK: {
                    if (this.loadTsFileManager.deleteAll(uuid)) {
                        status = RpcUtils.SUCCESS_STATUS;
                        break;
                    }
                    status.setCode(TSStatusCode.LOAD_FILE_ERROR.getStatusCode());
                    status.setMessage(String.format("No load TsFile uuid %s recorded for execute load command %s.", new Object[]{uuid, loadCommand}));
                    break;
                }
                default: {
                    status.setCode(TSStatusCode.ILLEGAL_PARAMETER.getStatusCode());
                    status.setMessage(String.format("Wrong load command %s.", new Object[]{loadCommand}));
                    break;
                }
            }
        }
        catch (Exception e) {
            LOGGER.error("Execute load command {} error.", (Object)loadCommand, (Object)e);
            status.setCode(TSStatusCode.LOAD_FILE_ERROR.getStatusCode());
            status.setMessage(e.getMessage());
        }
        return status;
    }

    public void rebootTimedService() throws ShutdownException {
        LOGGER.info("Start rebooting all timed service.");
        this.stopTimedServiceAndThrow(this.seqMemtableTimedFlushCheckThread, "SeqMemtableTimedFlushCheckThread");
        this.stopTimedServiceAndThrow(this.unseqMemtableTimedFlushCheckThread, "UnseqMemtableTimedFlushCheckThread");
        LOGGER.info("Stop all timed service successfully, and now restart them.");
        this.startTimedService();
        LOGGER.info("Reboot all timed service successfully");
    }

    private void stopTimedServiceAndThrow(ScheduledExecutorService pool, String poolName) throws ShutdownException {
        if (pool != null) {
            pool.shutdownNow();
            try {
                pool.awaitTermination(30L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                LOGGER.warn("{} still doesn't exit after 30s", (Object)poolName);
                throw new ShutdownException((Throwable)e);
            }
        }
    }

    public void getDiskSizeByDataRegion(Map<Integer, Long> dataRegionDisk, List<Integer> dataRegionIds) {
        this.dataRegionMap.forEach((dataRegionId, dataRegion) -> {
            if (dataRegionIds.contains(dataRegionId.getId())) {
                dataRegionDisk.put(dataRegionId.getId(), dataRegion.countRegionDiskSize());
            }
        });
    }

    public static File getDataRegionSystemDir(String dataBaseName, String dataRegionId) {
        return SystemFileFactory.INSTANCE.getFile(systemDir + File.separator + dataBaseName, dataRegionId);
    }

    public Runnable executeCompactFileTimeIndexCache() {
        return () -> {
            if (!this.isReadyForNonReadWriteFunctions()) {
                return;
            }
            for (DataRegion dataRegion : this.dataRegionMap.values()) {
                if (dataRegion == null) continue;
                dataRegion.compactFileTimeIndexCache();
            }
        };
    }

    static class InstanceHolder {
        private static final StorageEngine INSTANCE = new StorageEngine();

        private InstanceHolder() {
        }
    }
}

