/*
 * Decompiled with CFR 0.152.
 */
package org.apache.empire.db.validation;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import org.apache.empire.commons.StringUtils;
import org.apache.empire.data.DataType;
import org.apache.empire.db.DBColumn;
import org.apache.empire.db.DBCommandExpr;
import org.apache.empire.db.DBDatabase;
import org.apache.empire.db.DBRelation;
import org.apache.empire.db.DBRowSet;
import org.apache.empire.db.DBTable;
import org.apache.empire.db.DBTableColumn;
import org.apache.empire.db.DBView;
import org.apache.empire.exceptions.InternalException;
import org.apache.empire.exceptions.NotSupportedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DBModelParser {
    protected static final Logger log = LoggerFactory.getLogger(DBModelParser.class);
    protected final String catalog;
    protected final String schema;
    protected final String remoteName;
    protected DBDatabase remoteDb = null;
    protected final Map<String, DBRowSet> tableMap = new HashMap<String, DBRowSet>();
    private String standardIdentityColumnName = null;
    private String standardTimestampColumnName = null;

    public DBModelParser(String catalog, String schema) {
        this.catalog = catalog;
        this.schema = schema;
        StringBuilder b = new StringBuilder();
        if (StringUtils.isNotEmpty(catalog)) {
            b.append(catalog);
        }
        if (StringUtils.isNotEmpty(schema)) {
            if (b.length() > 0) {
                b.append(".");
            }
            b.append(schema);
        }
        if (b.length() == 0) {
            b.append("[Unknown]");
        }
        this.remoteName = b.toString();
    }

    public String getCatalog() {
        return this.catalog;
    }

    public String getSchema() {
        return this.schema;
    }

    public void setStandardIdentityColumnName(String standardIdentityColumnName) {
        this.standardIdentityColumnName = standardIdentityColumnName;
    }

    public void setStandardTimestampColumnName(String standardTimestampColumnName) {
        this.standardTimestampColumnName = standardTimestampColumnName;
    }

    public DBDatabase getDatabase() {
        return this.remoteDb;
    }

    public void parseModel(Connection conn) {
        try {
            this.remoteDb = this.createRemoteDatabase();
            DatabaseMetaData dbMeta = conn.getMetaData();
            this.populateRemoteDatabase(dbMeta);
        }
        catch (SQLException e) {
            log.error("checkModel failed for {}", (Object)this.remoteName);
            throw new InternalException(e);
        }
        finally {
            this.tableMap.clear();
        }
    }

    protected void populateRemoteDatabase(DatabaseMetaData dbMeta) throws SQLException {
        int count = this.collectTablesAndViews(dbMeta, null);
        log.info("{} tables and views added for schema \"{}\"", (Object)count, (Object)this.remoteName);
        count = this.collectColumns(dbMeta);
        log.info("{} columns added for schema \"{}\"", (Object)count, (Object)this.remoteName);
        count = this.collectPrimaryKeys(dbMeta);
        log.info("{} primary keys added for schema \"{}\"", (Object)count, (Object)this.remoteName);
        count = this.collectForeignKeys(dbMeta);
        log.info("{} foreign keys added for schema \"{}\"", (Object)count, (Object)this.remoteName);
    }

    protected boolean isSystemTable(String tableName, ResultSet tableMeta) {
        return tableName.indexOf(36) >= 0;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected int collectTablesAndViews(DatabaseMetaData dbMeta, String tablePattern) throws SQLException {
        this.tableMap.clear();
        try (ResultSet dbTables = dbMeta.getTables(this.catalog, this.schema, tablePattern, new String[]{"TABLE", "VIEW"});){
            int count = 0;
            while (dbTables.next()) {
                String tableName = dbTables.getString("TABLE_NAME");
                String tableType = dbTables.getString("TABLE_TYPE");
                if (this.isSystemTable(tableName, dbTables)) {
                    log.info("Ignoring system table " + tableName);
                    continue;
                }
                if ("VIEW".equalsIgnoreCase(tableType)) {
                    this.addView(tableName);
                } else {
                    this.addTable(tableName);
                }
                ++count;
            }
            int n = count;
            return n;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected int collectColumns(DatabaseMetaData dbMeta) throws SQLException {
        int count = 0;
        for (DBRowSet t : this.getTables()) {
            try (ResultSet dbColumns = dbMeta.getColumns(this.catalog, this.schema, t.getName(), null);){
                while (dbColumns.next()) {
                    this.addColumn(t, dbColumns);
                    ++count;
                }
            }
        }
        return count;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected int collectColumns(DatabaseMetaData dbMeta, String tablePattern) throws SQLException {
        try (ResultSet dbColumns = dbMeta.getColumns(this.catalog, this.schema, tablePattern, null);){
            int count = 0;
            while (dbColumns.next()) {
                String tableName = dbColumns.getString("TABLE_NAME");
                DBRowSet t = this.getTable(tableName);
                if (t == null) {
                    log.info("Table \"{}\" for column \"{}\" is not available!", (Object)tableName, (Object)dbColumns.getString("COLUMN_NAME"));
                    continue;
                }
                this.addColumn(t, dbColumns);
                ++count;
            }
            int n = count;
            return n;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected int collectPrimaryKeys(DatabaseMetaData dbMeta) throws SQLException {
        int count = 0;
        for (DBRowSet rs : this.getTables()) {
            if (!(rs instanceof DBTable)) continue;
            DBTable t = (DBTable)rs;
            ArrayList<String> pkCols = new ArrayList<String>();
            try (ResultSet primaryKeys = dbMeta.getPrimaryKeys(this.catalog, this.schema, t.getName());){
                while (primaryKeys.next()) {
                    String columnName = primaryKeys.getString("COLUMN_NAME");
                    int keyIndex = primaryKeys.getInt("KEY_SEQ") - 1;
                    if (keyIndex < pkCols.size()) {
                        pkCols.set(keyIndex, columnName);
                        continue;
                    }
                    if (keyIndex > pkCols.size()) {
                        while (keyIndex > pkCols.size()) {
                            pkCols.add(null);
                        }
                    }
                    pkCols.add(columnName);
                }
                if (pkCols.size() <= 0) continue;
                DBColumn[] keys = new DBColumn[pkCols.size()];
                for (int i = 0; i < keys.length; ++i) {
                    keys[i] = t.getColumn(((String)pkCols.get(i)).toUpperCase());
                }
                t.setPrimaryKey(keys);
                ++count;
            }
        }
        return count;
    }

    protected int collectForeignKeys(DatabaseMetaData dbMeta) throws SQLException {
        int count = 0;
        for (DBRowSet t : this.getTables()) {
            if (!(t instanceof DBTable)) continue;
            count += this.collectForeignKeys(dbMeta, t.getName());
        }
        return count;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected int collectForeignKeys(DatabaseMetaData dbMeta, String tablePattern) throws SQLException {
        try (ResultSet foreignKeys = dbMeta.getImportedKeys(this.catalog, this.schema, tablePattern);){
            int count = 0;
            while (foreignKeys.next()) {
                String fkTable = foreignKeys.getString("FKTABLE_NAME");
                String fkColumn = foreignKeys.getString("FKCOLUMN_NAME");
                String pkTable = foreignKeys.getString("PKTABLE_NAME");
                String pkColumn = foreignKeys.getString("PKCOLUMN_NAME");
                String fkName = foreignKeys.getString("FK_NAME");
                DBTableColumn c1 = (DBTableColumn)this.getTable(fkTable).getColumn(fkColumn.toUpperCase());
                DBTableColumn c2 = (DBTableColumn)this.getTable(pkTable).getColumn(pkColumn.toUpperCase());
                DBRelation relation = this.remoteDb.getRelation(fkName);
                if (relation == null) {
                    this.addRelation(fkName, c1.referenceOn(c2));
                    ++count;
                    continue;
                }
                DBRelation.DBReference[] refs = relation.getReferences();
                this.remoteDb.removeRelation(relation);
                DBRelation.DBReference[] newRefs = new DBRelation.DBReference[refs.length + 1];
                DBRelation.DBReference newRef = new DBRelation.DBReference(c1, c2);
                for (int i = 0; i < refs.length; ++i) {
                    newRefs[i] = refs[i];
                }
                newRefs[newRefs.length - 1] = newRef;
                this.addRelation(fkName, newRefs);
            }
            int n = count;
            return n;
        }
    }

    protected final Collection<DBRowSet> getTables() {
        return this.tableMap.values();
    }

    protected final DBRowSet getTable(String tableName) {
        return this.tableMap.get(tableName.toUpperCase());
    }

    protected DBDatabase createRemoteDatabase() {
        return new RemoteDatabase();
    }

    protected void addTable(String tableName) {
        this.tableMap.put(tableName.toUpperCase(), new DBTable(tableName, this.remoteDb));
    }

    protected void addView(String viewName) {
        this.tableMap.put(viewName.toUpperCase(), new RemoteView(viewName, this.remoteDb));
    }

    protected void addRelation(String relName, DBRelation.DBReference ... references) {
        this.remoteDb.addRelation(relName, references);
    }

    protected DBColumn addColumn(DBRowSet t, ResultSet rs) throws SQLException {
        DBColumn col;
        String name = rs.getString("COLUMN_NAME");
        DataType empireType = this.getEmpireDataType(rs.getInt("DATA_TYPE"));
        double colSize = this.getColumnSize(empireType, rs);
        boolean required = this.isColumnRequired(rs);
        Object defaultValue = this.getColumnDefault(rs);
        if (t instanceof DBTable) {
            boolean timestampColumn = false;
            if (empireType == DataType.INTEGER && this.isIdentityColumn(rs)) {
                empireType = DataType.AUTOINC;
            }
            if (empireType.isDate() && (timestampColumn = this.isTimestampColumn(rs))) {
                empireType = DataType.TIMESTAMP;
            }
            col = ((DBTable)t).addColumn(name, empireType, colSize, required, defaultValue);
            if (empireType == DataType.AUTOINC) {
                ((DBTable)t).setPrimaryKey(col);
            }
            if (timestampColumn) {
                t.setTimestampColumn(col);
            }
            log.debug("Added table column {}.{} of type {}", new Object[]{t.getName(), name, empireType});
        } else if (t instanceof DBView) {
            col = ((RemoteView)t).addColumn(name, empireType, colSize, false);
            log.debug("Added view column {}.{} of type {}", new Object[]{t.getName(), name, empireType});
        } else {
            log.warn("Unknown Object Type {}", (Object)t.getClass().getName());
            col = null;
        }
        return col;
    }

    protected double getColumnSize(DataType empireType, ResultSet rs) throws SQLException {
        double colSize = rs.getInt("COLUMN_SIZE");
        if (empireType == DataType.DECIMAL || empireType == DataType.FLOAT) {
            int decimalDig = rs.getInt("DECIMAL_DIGITS");
            if (decimalDig > 0) {
                try {
                    int intSize = rs.getInt("COLUMN_SIZE");
                    colSize = Double.parseDouble(String.valueOf(intSize) + '.' + decimalDig);
                }
                catch (Exception e) {
                    String name = rs.getString("COLUMN_NAME");
                    log.error("Failed to parse decimal digits for column " + name);
                }
            }
            if (colSize < 1.0) {
                empireType = DataType.INTEGER;
            }
        } else if (empireType.isDate()) {
            colSize = 0.0;
        } else if (empireType == DataType.INTEGER || empireType == DataType.CLOB || empireType == DataType.BLOB) {
            colSize = 0.0;
        }
        return colSize;
    }

    protected boolean isColumnRequired(ResultSet rs) throws SQLException {
        return rs.getString("IS_NULLABLE").equalsIgnoreCase("NO");
    }

    protected Object getColumnDefault(ResultSet rs) throws SQLException {
        return rs.getString("COLUMN_DEF");
    }

    protected boolean isIdentityColumn(ResultSet rs) {
        try {
            return this.standardIdentityColumnName != null && this.standardIdentityColumnName.equalsIgnoreCase(rs.getString("COLUMN_NAME"));
        }
        catch (SQLException e) {
            return false;
        }
    }

    protected boolean isTimestampColumn(ResultSet rs) {
        try {
            return this.standardTimestampColumnName != null && this.standardTimestampColumnName.equalsIgnoreCase(rs.getString("COLUMN_NAME"));
        }
        catch (SQLException e) {
            return false;
        }
    }

    protected DataType getEmpireDataType(int sqlType) {
        DataType empireType = DataType.UNKNOWN;
        switch (sqlType) {
            case -6: 
            case -5: 
            case 4: 
            case 5: {
                empireType = DataType.INTEGER;
                break;
            }
            case -9: 
            case 12: {
                empireType = DataType.VARCHAR;
                break;
            }
            case 91: {
                empireType = DataType.DATE;
                break;
            }
            case 92: {
                empireType = DataType.TIME;
                break;
            }
            case 93: {
                empireType = DataType.DATETIME;
                break;
            }
            case -15: 
            case 1: {
                empireType = DataType.CHAR;
                break;
            }
            case 6: 
            case 7: 
            case 8: {
                empireType = DataType.FLOAT;
                break;
            }
            case 2: 
            case 3: {
                empireType = DataType.DECIMAL;
                break;
            }
            case -7: 
            case 16: {
                empireType = DataType.BOOL;
                break;
            }
            case -16: 
            case -1: 
            case 2005: {
                empireType = DataType.CLOB;
                break;
            }
            case -4: 
            case -3: 
            case -2: 
            case 2004: {
                empireType = DataType.BLOB;
                break;
            }
            default: {
                empireType = DataType.UNKNOWN;
                log.warn("SQL column type " + sqlType + " not supported.");
            }
        }
        log.debug("Mapping date type " + String.valueOf(sqlType) + " to " + (Object)((Object)empireType));
        return empireType;
    }

    private static class RemoteView
    extends DBView {
        public RemoteView(String name, DBDatabase db) {
            super(name, db);
        }

        public DBColumn addColumn(String columnName, DataType dataType, double size, boolean dummy) {
            return super.addColumn(columnName, dataType, size);
        }

        @Override
        public DBCommandExpr createCommand() {
            throw new NotSupportedException(this, "createCommand");
        }
    }

    private static class RemoteDatabase
    extends DBDatabase {
        private RemoteDatabase() {
        }
    }
}

