/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.core.journal.impl;

import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.util.Collection;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Stream;
import org.apache.activemq.artemis.api.core.ActiveMQIOErrorException;
import org.apache.activemq.artemis.core.io.SequentialFile;
import org.apache.activemq.artemis.core.io.SequentialFileFactory;
import org.apache.activemq.artemis.core.journal.impl.JournalFile;
import org.apache.activemq.artemis.core.journal.impl.JournalFileImpl;
import org.apache.activemq.artemis.core.journal.impl.JournalImpl;
import org.apache.activemq.artemis.journal.ActiveMQJournalBundle;
import org.apache.activemq.artemis.journal.ActiveMQJournalLogger;
import org.apache.activemq.artemis.utils.ThreadDumpUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JournalFilesRepository {
    private static final Logger logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final boolean CHECK_CONSISTENCE = false;
    private final SequentialFileFactory fileFactory;
    private final JournalImpl journal;
    private final BlockingDeque<JournalFile> dataFiles = new LinkedBlockingDeque<JournalFile>();
    private final Queue<JournalFile> freeFiles = new ConcurrentLinkedQueue<JournalFile>();
    private final BlockingQueue<JournalFile> openedFiles = new LinkedBlockingQueue<JournalFile>();
    private final AtomicLong nextFileID = new AtomicLong(0L);
    private final int maxAIO;
    private final int minFiles;
    private final int poolSize;
    private final int fileSize;
    private final String filePrefix;
    private final String fileExtension;
    private final int userVersion;
    private final AtomicInteger freeFilesCount = new AtomicInteger(0);
    private final int journalFileOpenTimeout;
    private final int maxAtticFiles;
    private Executor openFilesExecutor;
    private final Runnable pushOpenRunnable = new Runnable(){

        @Override
        public void run() {
            try {
                JournalFilesRepository.this.pushOpenedFile();
            }
            catch (Exception e) {
                ActiveMQJournalLogger.LOGGER.errorPushingFile(e);
                JournalFilesRepository.this.fileFactory.onIOError(e, "unable to open ");
            }
        }
    };

    public JournalFilesRepository(SequentialFileFactory fileFactory, JournalImpl journal, String filePrefix, String fileExtension, int userVersion, int maxAIO, int fileSize, int minFiles, int poolSize, int journalFileOpenTimeout, int maxAtticFiles) {
        if (filePrefix == null) {
            throw new IllegalArgumentException("filePrefix cannot be null");
        }
        if (fileExtension == null) {
            throw new IllegalArgumentException("fileExtension cannot be null");
        }
        if (maxAIO <= 0) {
            throw new IllegalArgumentException("maxAIO must be a positive number");
        }
        this.fileFactory = fileFactory;
        this.maxAIO = maxAIO;
        this.filePrefix = filePrefix;
        this.fileExtension = fileExtension;
        this.minFiles = minFiles;
        this.fileSize = fileSize;
        this.poolSize = poolSize;
        this.userVersion = userVersion;
        this.journal = journal;
        this.journalFileOpenTimeout = journalFileOpenTimeout;
        this.maxAtticFiles = maxAtticFiles;
    }

    public int getPoolSize() {
        return this.poolSize;
    }

    public void setExecutor(Executor fileExecutor) {
        this.openFilesExecutor = fileExecutor;
    }

    public void clear() throws Exception {
        this.dataFiles.clear();
        this.freeFiles.clear();
        this.freeFilesCount.set(0);
        for (JournalFile file : this.openedFiles) {
            try {
                file.getFile().close();
            }
            catch (Exception e) {
                ActiveMQJournalLogger.LOGGER.errorClosingFile(e);
            }
        }
        this.openedFiles.clear();
    }

    public int getMaxAIO() {
        return this.maxAIO;
    }

    public String getFileExtension() {
        return this.fileExtension;
    }

    public String getFilePrefix() {
        return this.filePrefix;
    }

    public void calculateNextfileID(List<JournalFile> files) {
        for (JournalFile file : files) {
            long fileIdFromFile = file.getFileID();
            long fileIdFromName = JournalFilesRepository.getFileNameID(this.filePrefix, file.getFile().getFileName());
            this.setNextFileID(Math.max(fileIdFromName, fileIdFromFile));
        }
    }

    public void setNextFileID(long targetUpdate) {
        long current;
        do {
            if ((current = this.nextFileID.get()) < targetUpdate) continue;
            return;
        } while (!this.nextFileID.compareAndSet(current, targetUpdate));
    }

    public void ensureMinFiles() throws Exception {
        int filesToCreate = this.minFiles - (this.dataFiles.size() + this.freeFilesCount.get());
        if (filesToCreate > 0) {
            for (int i = 0; i < filesToCreate; ++i) {
                this.freeFiles.add(this.createFile(false, false, true, false, -1L));
                this.freeFilesCount.getAndIncrement();
            }
        }
    }

    public void openFile(JournalFile file, boolean multiAIO) throws Exception {
        if (multiAIO) {
            file.getFile().open();
        } else {
            file.getFile().open(1, false);
        }
        file.getFile().position(file.getFile().calculateBlockStart(16));
    }

    public JournalFile[] getDataFilesArray() {
        return this.dataFiles.toArray(new JournalFile[this.dataFiles.size()]);
    }

    public JournalFile pollLastDataFile() {
        return (JournalFile)this.dataFiles.pollLast();
    }

    public void removeDataFile(JournalFile file) {
        if (!this.dataFiles.remove(file)) {
            ActiveMQJournalLogger.LOGGER.couldNotRemoveFile(file);
        }
        this.removeNegatives(file);
    }

    public int getDataFilesCount() {
        return this.dataFiles.size();
    }

    public int getJournalFileOpenTimeout() {
        return this.journalFileOpenTimeout;
    }

    public Collection<JournalFile> getDataFiles() {
        return this.dataFiles;
    }

    public void clearDataFiles() {
        this.dataFiles.clear();
    }

    public void addDataFileOnTop(JournalFile file) {
        this.dataFiles.addFirst(file);
    }

    public String debugFiles() {
        StringBuilder buffer = new StringBuilder();
        buffer.append("**********\nCurrent File = " + String.valueOf(this.journal.getCurrentFile()) + "\n");
        buffer.append("**********\nDataFiles:\n");
        for (JournalFile file : this.dataFiles) {
            buffer.append(file.toString() + "\n");
        }
        buffer.append("*********\nFreeFiles:\n");
        for (JournalFile file : this.freeFiles) {
            buffer.append(file.toString() + "\n");
        }
        return buffer.toString();
    }

    public synchronized void checkDataFiles() {
        long seq = -1L;
        for (JournalFile file : this.dataFiles) {
            if (file.getFileID() <= seq) {
                ActiveMQJournalLogger.LOGGER.checkFiles();
                logger.info(this.debugFiles());
                ActiveMQJournalLogger.LOGGER.seqOutOfOrder();
                throw new IllegalStateException("Sequence out of order");
            }
            if (this.journal.getCurrentFile() != null && this.journal.getCurrentFile().getFileID() <= file.getFileID()) {
                ActiveMQJournalLogger.LOGGER.checkFiles();
                logger.info(this.debugFiles());
                ActiveMQJournalLogger.LOGGER.currentFile(file.getFileID(), this.journal.getCurrentFile().getFileID(), file.getFileID(), this.journal.getCurrentFile() == file);
            }
            if (this.journal.getCurrentFile() == file) {
                throw new RuntimeException("Check failure! Current file listed as data file!");
            }
            seq = file.getFileID();
        }
        long lastFreeId = -1L;
        for (JournalFile file : this.freeFiles) {
            if (file.getFileID() <= lastFreeId) {
                ActiveMQJournalLogger.LOGGER.checkFiles();
                logger.info(this.debugFiles());
                ActiveMQJournalLogger.LOGGER.fileIdOutOfOrder();
                throw new RuntimeException("Check failure!");
            }
            lastFreeId = file.getFileID();
            if (file.getFileID() >= seq) continue;
            ActiveMQJournalLogger.LOGGER.checkFiles();
            logger.info(this.debugFiles());
            ActiveMQJournalLogger.LOGGER.fileTooSmall();
        }
    }

    public void addDataFileOnBottom(JournalFile file) {
        this.dataFiles.add(file);
    }

    public int getFreeFilesCount() {
        return this.freeFilesCount.get();
    }

    public synchronized void addFreeFile(JournalFile file, boolean renameTmp) throws Exception {
        this.addFreeFile(file, renameTmp, true);
    }

    public synchronized void addFreeFile(JournalFile file, boolean renameTmp, boolean checkDelete) throws Exception {
        if (logger.isDebugEnabled()) {
            logger.debug("Adding free file {}, renameTMP={}, checkDelete={}", new Object[]{file, renameTmp, checkDelete});
        }
        long calculatedSize = 0L;
        try {
            calculatedSize = file.getFile().size();
        }
        catch (Exception e) {
            throw new IllegalStateException(e.getMessage() + " file: " + String.valueOf(file));
        }
        if (calculatedSize != (long)this.fileSize) {
            this.damagedFile(file);
        } else if (!checkDelete || this.freeFilesCount.get() + this.dataFiles.size() + 1 + this.openedFiles.size() < this.poolSize || this.poolSize < 0) {
            if (logger.isTraceEnabled()) {
                logger.trace("Re-initializing file {} as checkDelete={}, freeFilesCount={}, dataFiles.size={} openedFiles={}, poolSize={}", new Object[]{file, checkDelete, this.freeFilesCount, this.dataFiles.size(), this.openedFiles, this.poolSize});
            }
            JournalFile jf = this.reinitializeFile(file);
            if (renameTmp) {
                jf.getFile().renameTo(JournalImpl.renameExtensionFile(jf.getFile().getFileName(), ".tmp"));
            }
            this.freeFiles.add(jf);
            this.freeFilesCount.getAndIncrement();
        } else {
            logger.debug("Deleting file {}", (Object)file.getFile());
            if (logger.isTraceEnabled()) {
                logger.trace("DataFiles.size() = {}", (Object)this.dataFiles.size());
                logger.trace("openedFiles.size() = {}", (Object)this.openedFiles.size());
                logger.trace("minfiles = {}, poolSize = {}", (Object)this.minFiles, (Object)this.poolSize);
                logger.trace("Free Files = {}", (Object)this.freeFilesCount.get());
                logger.trace("File {} being deleted as freeFiles.size() + dataFiles.size() + 1 + openedFiles.size() () < minFiles ({})", new Object[]{file, this.freeFilesCount.get() + this.dataFiles.size() + 1 + this.openedFiles.size(), this.minFiles});
            }
            file.getFile().delete();
        }
    }

    private void damagedFile(JournalFile file) throws Exception {
        if (file.getFile().isOpen()) {
            file.getFile().close(false, false);
        }
        if (file.getFile().exists()) {
            Path journalPath = file.getFile().getJavaFile().toPath();
            Path atticPath = journalPath.getParent().resolve("attic");
            Files.createDirectories(atticPath, new FileAttribute[0]);
            if (this.listFiles(atticPath) < this.maxAtticFiles) {
                ActiveMQJournalLogger.LOGGER.movingFileToAttic(file.getFile().getFileName());
                Files.move(journalPath, atticPath.resolve(journalPath.getFileName()), StandardCopyOption.REPLACE_EXISTING);
            } else {
                ActiveMQJournalLogger.LOGGER.deletingFile(file);
                Files.delete(journalPath);
            }
        }
    }

    private int listFiles(Path path) throws IOException {
        try (Stream<Path> files = Files.list(path);){
            int n = files.mapToInt(e -> 1).sum();
            return n;
        }
    }

    public Collection<JournalFile> getFreeFiles() {
        return this.freeFiles;
    }

    public JournalFile getFreeFile() {
        JournalFile file = this.freeFiles.remove();
        this.freeFilesCount.getAndDecrement();
        return file;
    }

    public int getOpenedFilesCount() {
        return this.openedFiles.size();
    }

    public JournalFile openFileCMP() throws Exception {
        JournalFile file = this.openFile();
        SequentialFile sequentialFile = file.getFile();
        sequentialFile.close();
        sequentialFile.renameTo(sequentialFile.getFileName() + ".cmp");
        return file;
    }

    public JournalFile openFile() throws InterruptedException, ActiveMQIOErrorException {
        JournalFile nextFile;
        if (logger.isTraceEnabled()) {
            logger.trace("enqueueOpenFile with openedFiles.size={}", (Object)this.openedFiles.size());
        }
        if ((nextFile = (JournalFile)this.openedFiles.poll()) == null) {
            this.pushOpen();
            nextFile = this.openedFiles.poll(this.journalFileOpenTimeout, TimeUnit.SECONDS);
        } else if (this.openedFiles.isEmpty()) {
            this.pushOpen();
        }
        if (nextFile == null) {
            ActiveMQJournalLogger.LOGGER.cantOpenFileTimeout(this.journalFileOpenTimeout);
            logger.warn(ThreadDumpUtil.threadDump((String)ActiveMQJournalBundle.BUNDLE.threadDumpAfterFileOpenTimeout()));
            try {
                nextFile = this.takeFile(true, true, true, false);
            }
            catch (Exception e) {
                this.fileFactory.onIOError(e, "unable to open ");
                this.fileFactory.activateBuffer(this.journal.getCurrentFile().getFile());
                throw ActiveMQJournalBundle.BUNDLE.fileNotOpened();
            }
        }
        logger.trace("Returning file {}", (Object)nextFile);
        return nextFile;
    }

    private void pushOpen() {
        if (this.openFilesExecutor == null) {
            this.pushOpenRunnable.run();
        } else {
            this.openFilesExecutor.execute(this.pushOpenRunnable);
        }
    }

    public synchronized void pushOpenedFile() throws Exception {
        JournalFile nextOpenedFile = this.takeFile(true, true, true, false);
        logger.trace("pushing openFile {}", (Object)nextOpenedFile);
        if (!this.openedFiles.offer(nextOpenedFile)) {
            ActiveMQJournalLogger.LOGGER.failedToAddFile(nextOpenedFile);
        }
    }

    public void closeFile(JournalFile file, boolean block) throws Exception {
        this.fileFactory.deactivateBuffer();
        file.getFile().close(true, block);
        if (!this.dataFiles.contains(file)) {
            this.dataFiles.add(file);
        }
    }

    private JournalFile takeFile(boolean keepOpened, boolean multiAIO, boolean initFile, boolean tmpCompactExtension) throws Exception {
        JournalFile nextFile = null;
        nextFile = this.freeFiles.poll();
        if (nextFile != null) {
            this.freeFilesCount.getAndDecrement();
        }
        if (nextFile == null) {
            nextFile = this.createFile(keepOpened, multiAIO, initFile, tmpCompactExtension, -1L);
        } else {
            if (tmpCompactExtension) {
                SequentialFile sequentialFile = nextFile.getFile();
                sequentialFile.renameTo(sequentialFile.getFileName() + ".cmp");
            }
            if (keepOpened) {
                this.openFile(nextFile, multiAIO);
            }
        }
        return nextFile;
    }

    public JournalFile createRemoteBackupSyncFile(long fileID) throws Exception {
        return this.createFile(false, false, true, false, fileID);
    }

    private JournalFile createFile(boolean keepOpened, boolean multiAIO, boolean init, boolean tmpCompact, long fileIdPreSet) throws Exception {
        if (System.getSecurityManager() == null) {
            return this.createFile0(keepOpened, multiAIO, init, tmpCompact, fileIdPreSet);
        }
        try {
            return AccessController.doPrivileged(() -> this.createFile0(keepOpened, multiAIO, init, tmpCompact, fileIdPreSet));
        }
        catch (PrivilegedActionException e) {
            throw this.unwrapException(e);
        }
    }

    private RuntimeException unwrapException(PrivilegedActionException e) throws Exception {
        Throwable c = e.getCause();
        if (c instanceof RuntimeException) {
            RuntimeException runtimeException = (RuntimeException)c;
            throw runtimeException;
        }
        if (c instanceof Error) {
            Error error = (Error)c;
            throw error;
        }
        throw new RuntimeException(c);
    }

    private JournalFile createFile0(boolean keepOpened, boolean multiAIO, boolean init, boolean tmpCompact, long fileIdPreSet) throws Exception {
        long fileID = fileIdPreSet != -1L ? fileIdPreSet : this.generateFileID();
        String fileName = this.createFileName(tmpCompact, fileID);
        logger.trace("Creating file {}", (Object)fileName);
        String tmpFileName = fileName + ".tmp";
        SequentialFile sequentialFile = this.fileFactory.createSequentialFile(tmpFileName);
        sequentialFile.open(1, false);
        if (init) {
            sequentialFile.fill(this.fileSize);
            JournalImpl.initFileHeader(this.fileFactory, sequentialFile, this.userVersion, fileID);
        }
        long position = sequentialFile.position();
        sequentialFile.close(false, false);
        logger.trace("Renaming file {} as {}", (Object)tmpFileName, (Object)fileName);
        sequentialFile.renameTo(fileName);
        if (keepOpened) {
            if (multiAIO) {
                sequentialFile.open();
            } else {
                sequentialFile.open(1, false);
            }
            sequentialFile.position(position);
        }
        return new JournalFileImpl(sequentialFile, fileID, 2);
    }

    private String createFileName(boolean tmpCompact, long fileID) {
        String fileName = tmpCompact ? this.filePrefix + "-" + fileID + "." + this.fileExtension + ".cmp" : this.filePrefix + "-" + fileID + "." + this.fileExtension;
        return fileName;
    }

    private long generateFileID() {
        return this.nextFileID.incrementAndGet();
    }

    public static long getFileNameID(String filePrefix, String fileName) {
        try {
            return Long.parseLong(fileName.substring(filePrefix.length() + 1, fileName.indexOf(46)));
        }
        catch (Throwable e) {
            try {
                return Long.parseLong(fileName.substring(fileName.lastIndexOf("-") + 1, fileName.indexOf(46)));
            }
            catch (Throwable e2) {
                ActiveMQJournalLogger.LOGGER.errorRetrievingID(fileName, e);
                return 0L;
            }
        }
    }

    private JournalFile reinitializeFile(JournalFile file) throws Exception {
        long newFileID = this.generateFileID();
        SequentialFile sf = file.getFile();
        sf.open(1, false);
        int position = JournalImpl.initFileHeader(this.fileFactory, sf, this.userVersion, newFileID);
        JournalFileImpl jf = new JournalFileImpl(sf, newFileID, 2);
        sf.position(position);
        sf.close(false, false);
        this.removeNegatives(file);
        return jf;
    }

    public void removeNegatives(JournalFile file) {
        this.dataFiles.forEach(f -> f.fileRemoved(file));
    }

    public String toString() {
        return "JournalFilesRepository(dataFiles=" + String.valueOf(this.dataFiles) + ", freeFiles=" + String.valueOf(this.freeFiles) + ", openedFiles=" + String.valueOf(this.openedFiles) + ")";
    }
}

