/*
 * XMMS meta-input plugin: A unix program interface for xmms input plugins
 *
 * Copyright (C) 2000 Mikael Bouillot <mikael dot bouillot at bigfoot dot com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <pthread.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <gtk/gtk.h>
#include <signal.h>
#include <sys/wait.h>
#include <errno.h>

#include "xmms/plugin.h"

typedef struct String_Link_s
{
	struct String_Link_s *next;
	char *str;
}
String_Link;

typedef struct Player_s
{
	char *file;
	String_Link magic;
	String_Link extensions;
}
Player;

InputPlugin *get_iplugin_info (void);
static void init (void);
static int is_our_file (char *filename);
static void *play_loop(void *arg);
static void play (char *filename);
static void stop (void);
static void pause (short p);
static int get_time (void);
static void get_song_info (char *filename, char **title_real, int *len_real);
static void seek (int sec);
static void clean_up (void);
static void strlnk_add (char *str, String_Link *link);
static int strlnk_search (char *str, String_Link *there);
static char *magic_get (char *filename);
static void about_box (void);

InputPlugin meta_input_ip =
{
	NULL,
	NULL,
	"~Meta-input plugin",
	init,
	about_box,
	NULL,
	is_our_file,
	NULL,
	play,
	stop,
	pause,
	seek,
	NULL,
	get_time,
	NULL,
	NULL,
	clean_up,
	NULL,
	NULL,
	NULL,
	NULL,
	get_song_info,
	NULL,
	NULL
};

static pthread_t decode_thread;
static gboolean audio_error = FALSE;

static FILE *file;
int going, eof;
char data [2048];
int written;
int playtime;
int datapipe;
Player players [32];
int numPlayers;
int playerPID;

static GtkWidget *about_window = NULL;
char about_window_text[] = "Meta-Input Plugin\nby Mikael Bouillot\n<mikael.bouillot@bigfoot.com>\n\nAvailable players:\n";

InputPlugin *get_iplugin_info (void)
{
	return &meta_input_ip;
}

/*
 *  We look for all available players and store them in the
 * "players" global.
 */
static void init (void)
{
    char filename [256];
    DIR *dir;
    struct dirent *ent;
    struct stat statbuf;
	
	players [0].file = strdup ("unknown");
	players [0].magic.next = NULL;
	players [0].extensions.next = NULL;
	numPlayers = 1;
	
	dir = opendir (PLAYER_DIR);
	if (!dir)
		return;
	
	while ((ent = readdir (dir)) != NULL)
	{
		sprintf (filename, "%s/%s", PLAYER_DIR, ent->d_name);
		if (stat (filename, &statbuf) != -1 && S_ISREG (statbuf.st_mode))
		{
			players [numPlayers].file = strdup (filename);
			players [numPlayers].magic.next = NULL;
			players [numPlayers].extensions.next = NULL;
			numPlayers++;
		}
	}
	
	closedir (dir);
}

/*
 *  Which player knows how to play this file? Let's ask them.
 * New: Full caching of the file extensions and magic types
 * added. Works *much* faster now.
 */
static int whose_file (char *filename)
{
	char command [256];
	int a;
	char *ext, *magic;
	
	ext = rindex (filename, '.');
	if (ext)
	{
		ext++;
		if (strlen (ext) > 5)
		{
			ext = NULL;
		}
	}
	
	if (ext)
	{
		for (a = 0; a < numPlayers; a++)
		{
			if (strlnk_search (ext, &(players [a].extensions)))
			{
				if (a == 0)
					return -1;
				else
					return a;
			}
		}
	}
	
	magic = magic_get (filename);
	
	if (magic)
	{
		for (a = 0; a < numPlayers; a++)
		{
			if (strlnk_search (magic, &(players [a].magic)))
			{
				if (a == 0)
					return -1;
				else
					return a;
			}
		}
	}
	
	for (a = 1; a < numPlayers; a++)
	{
		sprintf (command, "%s isOurFile \"%s\"", players [a].file, filename);
		if (system (command) == 0)
		{
			if (ext)
			{
				strlnk_add (ext, &(players [a].extensions));
			}
			if (magic)
			{
				strlnk_add (magic, &(players [a].magic));
			}
			return a;
		}
	}
	
	if (ext)
	{
		strlnk_add (ext, &(players [0].extensions));
	}
	if (magic)
	{
		strlnk_add (magic, &(players [0].magic));
	}
	return -1;
}

/*
 *  It's our file if one player knows how to play it.
 */
static int is_our_file (char *filename)
{
	int a;
	
	if (whose_file (filename) != -1)
		return TRUE;
	else
		return FALSE;
}

/*
 *  The play loop. Executed in a separate thread, it just shovels
 * what comes from the data pipe to the vis and output plugins.
 * When the pipe is closed, we know the player died.
 */
static void *play_loop (void *arg)
{
	int a, nread;
	
	while (going)
	{
		if (!eof)
		{
			while (written < 2048)
			{
				nread = read (datapipe, data + written, 2048 - written);
				if (nread == 0 || nread == -1)
				{
					eof = 1;
					break;
				}
				written += nread;
			}

			meta_input_ip.add_vis_pcm (meta_input_ip.output->written_time(), FMT_S16_LE, 2, written, data);

			while (going && meta_input_ip.output->buffer_free() < written)
				xmms_usleep (10000);

			if (going)
				meta_input_ip.output->write_audio (data, written);
			
			written = 0;
		}
		else	/* eof */
		{
			xmms_usleep (10000);
		}
	}
	
	pthread_exit (NULL);
}

/*
 *  We are asked to play a file, so we check what player can
 * play it, then we create the data pipe, we fork the player
 * with his standard output plugged into our pipe and we launch
 * the playing thread.
 */
static void play (char *filename)
{
	FILE *file;
	char *name, *temp;
	unsigned long len;
	int rate;
	int filedes [2];
	int ret;
	int player;
	int a;
	
	audio_error = FALSE;
	
	player = whose_file (filename);
	
	if (player != -1)
	{
		written = 0;
		going = 1;
		eof = 0;
		if (meta_input_ip.output->open_audio (FMT_S16_LE, 44100, 2) == 0)
		{
			audio_error = TRUE;
			return;
		}

		pipe (filedes);

		ret = fork();

		if (ret == -1)
		{
			audio_error = TRUE;
			return;
		}

		if (ret == 0)	// Child process
		{
			// Set group ID for easy killing of the player later...
			setpgrp();
			
			close (filedes [0]);
			close (0);
			dup2 (filedes [1], 1);
			close (filedes [1]);
			
			execl (players [player].file, "foo", "play", filename, NULL);
			
			exit (-1);
		}
		
		// Parent process
		
		playerPID = ret;
		
		close (filedes [1]);

		datapipe = filedes [0];

		temp = strrchr (filename, '/');
		if (!temp)
			temp = filename;
		else
			temp++;
		name = malloc (strlen (temp) + 1);
		strcpy (name, temp);
		if (strrchr (name, '.') != NULL)
			*strrchr (name, '.') = '\0';

		meta_input_ip.set_info (name, -1, 333000 /* bps */, 44100, 2 /* stereo */);
		free (name);

		pthread_create (&decode_thread, NULL, play_loop, NULL);
	}
}

/*
 *  We have to stop playing. If we *are* playing, we suicide
 * the playing thread, we close the pipe, and we wait for both
 * the playing thread and the player to die.
 */
static void stop (void)
{
	int ret;
	
	if (going)
	{
		going = 0;
		close (datapipe);
		pthread_join (decode_thread, NULL);
		meta_input_ip.output->close_audio();
		usleep (10000);
		
		// Now, the player should have died a natural death by SIGPIPE.
		// But for some reason, some programs don't (notably ogg123).
		// So if they are still alive, help them a little...
		
		// Note: it's not only that the playing program doesn't respond
		// to SIGPIPE... It does the same thing with mpg123, which dies
		// very well when its pipe is broken on the command line. It
		// used to work well, but it seems the player doesn't get the
		// signal anymore. If someone know why, feel free to enlighten
		// me.
		ret = waitpid (playerPID, NULL, WNOHANG);
		if (ret == 0)
		{
			killpg (playerPID, SIGKILL);
			ret = waitpid ( - playerPID, NULL, 0);
			// Note: I expected to have to wait for all the processes
			// spawned by the player, but it seems that when I kill the
			// player itself, its children are transfered to init (if not
			// dead) and I cannot wait for them... I'm a bit fuzzy on this
			// child waiting thing.
			while (ret > 0)
			{
				ret = waitpid ( - playerPID, NULL, 0);
			}
		}
	}
}

static void pause (short p)
{
	meta_input_ip.output->pause (p);
}

static int get_time (void)
{
	if (audio_error)
		return -2;
	if (!going || (eof && !meta_input_ip.output->buffer_playing()))
		return -1;
	else
		return meta_input_ip.output->output_time();
}

/*
 *  Generic info about one of our files. This ought to be
 * asked to the concerned player, so it's just done this
 * way until the player interface is improved.
 */
static void get_song_info (char *filename, char **title_real, int *len_real)
{
	FILE *file;
	char *name, *temp;
	int playtime, fadetime;

	temp = strrchr (filename, '/');
	if (!temp)
		temp = filename;
	else
		temp++;
	name = malloc (strlen (temp) + 1);
	strcpy (name, temp);
	if (strrchr (name, '.') != NULL)
		*strrchr (name, '.') = '\0';
	(*title_real) = name;
	(*len_real) = -1;
}

/*
 *  This is just an empty function for now, to keep XMMS from crashing
 * when seeking during a song.
 */
static void seek (int sec)
{
}

static void clean_up (void)
{
	// should free() all the malloc()ed things here.
	// and strdup()ed, also...
}

//*****************************************************************************

static void strlnk_add (char *str, String_Link *there)
{
	String_Link *link;
	
	link = (String_Link *) malloc (sizeof (String_Link));
	link->str = strdup (str);
	
	link->next = there->next;
	there->next = link;
}

static int strlnk_search (char *str, String_Link *there)
{
	String_Link *link;
	
	link = there->next;
	
	while (link != NULL)
	{
		if (strcmp (str, link->str) == 0)
		{
			return 1;
		}
		
		link = link->next;
	}
	
	return 0;
}

// TODO: fermer stderr
static char *magic_get (char *filename)
{
	static int inited = 0;
	static int mime_supported = 0;
	char buf[256];
	int filedes [2];
	int ret;
	int nread;
	char *ptr;
	
	if (! inited)
	{
		inited = 1;
		ret = system ("file -i /dev/zero");
		if (ret == 0)
		{
			mime_supported = 1;
		}
		else
		{
			mime_supported = 0;
		}
	}
	
	pipe (filedes);

	ret = fork();

	if (ret == -1)
	{
		return NULL;
	}

	if (ret == 0)	// Child process
	{
		close (filedes [0]);
		close (0);
		dup2 (filedes [1], 1);
		close (filedes [1]);
		
		if (mime_supported)
		{
			execlp ("file", "file", "-biL", filename, NULL);
		}
		else
		{
			execlp ("file", "file", "-bL", filename, NULL);
		}

		exit (-1);
	}

	// Parent process
	close (filedes [1]);
	nread = read (filedes [0], buf, 256);
	close (filedes [0]);
	wait (NULL);
	ptr = index (buf, '\n');
	*ptr = '\0';
	
	if (strcmp (buf, "data") == 0)
	{
		return NULL;
	}
	
	// quick hack to get "Standard MIDI" without the "using 20 tracks"...
	if (! mime_supported)
	{
		buf[13] = '\0';
	}
	
	return strdup (buf);
}

/*  About box courtesy of James Mitchell.
 * (adapted from the MikMod plugin)
 */
static void about_box (void)
{
	GtkWidget *dialog_vbox1;
	GtkWidget *hbox1;
	GtkWidget *label1;
	GtkWidget *dialog_action_area1;
	GtkWidget *about_exit;
	char *p = NULL, *q;
	int a;

	if (!about_window)
	{
		about_window = gtk_dialog_new();
		gtk_object_set_data(GTK_OBJECT(about_window), "about_window", about_window);
		gtk_window_set_title(GTK_WINDOW(about_window), "About Meta-Plugin");
		gtk_window_set_policy(GTK_WINDOW(about_window), FALSE, FALSE, FALSE);
		gtk_window_set_position(GTK_WINDOW(about_window), GTK_WIN_POS_MOUSE);
		gtk_signal_connect(GTK_OBJECT(about_window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &about_window);
		gtk_container_border_width(GTK_CONTAINER(about_window), 10);

		dialog_vbox1 = GTK_DIALOG(about_window)->vbox;
		gtk_object_set_data(GTK_OBJECT(about_window), "dialog_vbox1", dialog_vbox1);
		gtk_widget_show(dialog_vbox1);
		gtk_container_border_width(GTK_CONTAINER(dialog_vbox1), 5);

		hbox1 = gtk_hbox_new(FALSE, 0);
		gtk_object_set_data(GTK_OBJECT(about_window), "hbox1", hbox1);
		gtk_widget_show(hbox1);
		gtk_box_pack_start(GTK_BOX(dialog_vbox1), hbox1, TRUE, TRUE, 0);
		gtk_container_border_width(GTK_CONTAINER(hbox1), 5);
		gtk_widget_realize(about_window);

		for (a = 1; a < numPlayers; a++)
		{
			if (!p)
			{
				p = strdup(players[a].file);
			}
			else
			{
				q = malloc(strlen(p)+strlen(players[a].file)+2);
				strcpy(q,p);
				strcat(q,"\n");
				strcat(q,players[a].file);
				p=q;
			}
		}
		q =malloc(strlen(p)+strlen(about_window_text)+1);
		strcpy(q,about_window_text);
		strcat(q,p);
		p = q;
		q=NULL;

		label1 = gtk_label_new(p);
		gtk_object_set_data(GTK_OBJECT(about_window), "label1", label1);
		gtk_widget_show(label1);
		gtk_box_pack_start(GTK_BOX(hbox1), label1, TRUE, TRUE, 0);

		dialog_action_area1 = GTK_DIALOG(about_window)->action_area;
		gtk_object_set_data(GTK_OBJECT(about_window), "dialog_action_area1", dialog_action_area1);
		gtk_widget_show(dialog_action_area1);
		gtk_container_border_width(GTK_CONTAINER(dialog_action_area1), 10);

		about_exit = gtk_button_new_with_label("Ok");
		gtk_signal_connect_object(GTK_OBJECT(about_exit), "clicked",
			GTK_SIGNAL_FUNC(gtk_widget_destroy),
			GTK_OBJECT(about_window));

		gtk_object_set_data(GTK_OBJECT(about_window), "about_exit", about_exit);
		gtk_widget_show(about_exit);
		gtk_box_pack_start(GTK_BOX(dialog_action_area1), about_exit, TRUE, TRUE, 0);



		gtk_widget_show(about_window);
	}
	else
	{
		gdk_window_raise(about_window->window);
	}
}
