

/*
 *  Author: Arvin Schnell
 */


#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <errno.h>
#include <X11/Xlib.h>
#include <Xm/Xm.h>
#include <iostream>

using std::cerr;

#include "Sample.h"
#include "utils.h"


Sample sample;


Sample::Sample ()
    : buffer (0),
      pcm (0),
      file (0)
{
}


bool
Sample::open ()
{
    // check length and resolution

    if (!one_bit_set (length) || length < 128) {
	cerr << "error: illegal sample length (must be a 2^n)\n";
	return false;
    }

    if (resolution != 16 && resolution != 32) {
	cerr << "error: illegal sample resolution (must 16 or 32)\n";
	return false;
    }

    // allocate memory

    buffer = new int32_t[2 * length];

    // open DSP

    switch (source)
    {
	case SOUNDCARD:

	    pcm = PCM::create (device, PCM::DEVICE, PCM::READ, resolution == 16 ?
			       PCM::S16 : PCM::S32, 2, rate, length);
	    if (!pcm) {
		delete[] buffer;
		buffer = 0;
		return false;
	    }

	    break;

	case DISKFILE:

	    pcm = PCM::create (device, PCM::DEVICE, PCM::WRITE, resolution == 16 ?
			       PCM::S16 : PCM::S32, 2, rate, length);
	    if (!pcm) {
		delete[] buffer;
		buffer = 0;
		return false;
	    }

	    file = PCM::create (filename, PCM::FILE, PCM::READ, PCM::S16, 2, 44100);
	    if (!file) {
		delete pcm;
		pcm = 0;
		delete[] buffer;
		buffer = 0;
		return false;
	    }

	    break;
    }

    pcm->info ();

    frames_per_second = (double) (rate) / length;
    frame_count = -1;		// yes!

    return true;
}


void
Sample::calc_dc ()
{
    int64_t left = 0;
    int64_t right = 0;

    int32_t* tmp1 = buffer;
    for (unsigned int i = 0; i < length; i++) {
	left += *tmp1++;
	right += *tmp1++;
    }

    dc[0] = left / (signed) length;
    dc[1] = right / (signed) length;
}


void
Sample::expand ()
{
    int16_t* tmp1 = (int16_t*)(buffer) + 2 * length;
    int32_t* tmp2 = buffer + 2 * length;
    for (unsigned int i = 0; i < length; i++)
    {
	*--tmp2 = (*--tmp1) << 16;
	*--tmp2 = (*--tmp1) << 16;
    }
}


bool
Sample::shot ()
{
    switch (source)
    {
	case SOUNDCARD:

	    pcm->read (buffer, length);

	    if (resolution == 16)
		expand ();

	    break;

	case DISKFILE:

	    size_t i = file->read (buffer, length);

	    // there might still be something to play but we can only handle
	    // complete blocks
	    if (i != length)
		return false;

	    if (resolution == 16) {
		pcm->write (buffer, length);
		expand ();
	    } else {
		expand ();
		pcm->write (buffer, length);
	    }

	    break;
    }

    calc_dc ();

    frame_count++;

    return true;
}


bool
Sample::close ()
{
    if (pcm) {
	delete pcm;
	pcm = 0;
    }

    switch (source)
    {
	case SOUNDCARD:
	    break;

	case DISKFILE:
	    if (file) {
		delete file;
		file = 0;
	    }
	    break;
    }

    if (buffer) {
	delete[] buffer;
	buffer = 0;
    }

    return true;
}
