/*-------------------------------------------------------*/
/* term.c       ( NTHU CS MapleBBS Ver 2.36 )            */
/*-------------------------------------------------------*/
/* target : termcap I/O control routines                 */
/* create : 95/03/29                                     */
/* update : 95/12/15                                     */
/*-------------------------------------------------------*/


#include "bbs.h"
#include <sys/ioctl.h>

#ifdef HP_UX
#define O_HUPCL 01
#define O_XTABS 02
#endif

#ifdef LINUX
#include <linux/termios.h>
#define stty(fd, data) tcsetattr( fd, TCSETS, data )
#define gtty(fd, data) tcgetattr( fd, data )
#endif

#ifndef TANDEM
#define TANDEM  0x00000001
#endif

#ifndef CBREAK
#define CBREAK  0x00000002
#endif

#ifdef LINUX
struct termios tty_state, tty_new;
#else
struct sgttyb tty_state, tty_new;
#endif


/* ----------------------------------------------------- */
/* basic tty control                                     */
/* ----------------------------------------------------- */


void
get_tty()
{
  if (gtty (1, &tty_state) < 0)
  {
     fprintf (stderr, "gtty failed\n");
     exit (-1);
  }
}


void
init_tty()
{
  if (gtty(1, &tty_state) < 0)
  {
    fprintf(stderr, "gtty failed\n");
    exit(-1);
  }
  memcpy(&tty_new, &tty_state, sizeof(tty_new));

#ifdef  LINUX

  tty_new.c_lflag &= ~(ICANON | ECHO | RAW | ISIG);
  tcsetattr(1, TCSANOW, &tty_new);
  restore_tty();

#else

  tty_new.sg_flags |= RAW;

#ifdef  HP_UX
  tty_new.sg_flags &= ~(O_HUPCL | O_XTABS | LCASE | ECHO | CRMOD);
#else
  tty_new.sg_flags &= ~(TANDEM | CBREAK | LCASE | ECHO | CRMOD);
#endif

  stty(1, &tty_new);
#endif
}


#ifdef LINUX
reset_tty()
{
   system("stty -raw echo");
}
restore_tty()
{
   system("stty raw -echo");
}
#else
void
reset_tty()
{
  stty(1, &tty_state);
}
void
restore_tty()
{
  stty(1, &tty_new);
}

#endif



/* ----------------------------------------------------- */
/* init tty control code                                 */
/* ----------------------------------------------------- */


#define TERMCOMSIZE (40)

int dumb_term = YEA;

char clearbuf[TERMCOMSIZE];
int clearbuflen;

char cleolbuf[TERMCOMSIZE];
int cleolbuflen;

char cursorm[TERMCOMSIZE];
char *cm;

char scrollrev[TERMCOMSIZE];
int scrollrevlen;

char strtstandout[TERMCOMSIZE];
int strtstandoutlen;

char endstandout[TERMCOMSIZE];
int endstandoutlen;

int t_lines = 24;
int b_lines = 23;
int p_lines = 20;
int t_columns = 80;

int automargins;

char *outp;
int *outlp;


static void
outcf(ch)
  char ch;
{
  if (*outlp < TERMCOMSIZE)
  {
    (*outlp)++;
    *outp++ = ch;
  }
}


int
term_init(term)
  char *term;
{
  extern char PC, *UP, *BC;
  extern short ospeed;
  static char UPbuf[TERMCOMSIZE];
  static char BCbuf[TERMCOMSIZE];
  static char buf[1024];
  char sbuf[2048];
  char *sbp, *s;
  char *tgetstr();

#ifdef LINUX
  ospeed = cfgetospeed(&tty_state);
#else
  ospeed = tty_state.sg_ospeed;
#endif

  if (tgetent(buf, term) != 1)
    return NA;

  sbp = sbuf;
  s = tgetstr("pc", &sbp);      /* get pad character */
  if (s)
    PC = *s;

  t_lines = tgetnum("li");
  t_columns = tgetnum("co");
  automargins = tgetflag("am");

  outp = clearbuf;              /* fill clearbuf with clear screen command */
  outlp = &clearbuflen;
  clearbuflen = 0;
  sbp = sbuf;
  s = tgetstr("cl", &sbp);
  if (s)
    tputs(s, t_lines, outcf);

  outp = cleolbuf;              /* fill cleolbuf with clear to eol command */
  outlp = &cleolbuflen;
  cleolbuflen = 0;
  sbp = sbuf;
  s = tgetstr("ce", &sbp);
  if (s)
    tputs(s, 1, outcf);

  outp = scrollrev;
  outlp = &scrollrevlen;
  scrollrevlen = 0;
  sbp = sbuf;
  s = tgetstr("sr", &sbp);
  if (s)
    tputs(s, 1, outcf);

  outp = strtstandout;
  outlp = &strtstandoutlen;
  strtstandoutlen = 0;
  sbp = sbuf;
  s = tgetstr("so", &sbp);
  if (s)
    tputs(s, 1, outcf);

  outp = endstandout;
  outlp = &endstandoutlen;
  endstandoutlen = 0;
  sbp = sbuf;
  s = tgetstr("se", &sbp);
  if (s)
    tputs(s, 1, outcf);

  sbp = cursorm;
  cm = tgetstr("cm", &sbp);
  if (cm)
    dumb_term = NA;
  else
  {
    dumb_term = YEA;
    t_lines = 24;
    t_columns = 80;
  }

  sbp = UPbuf;
  UP = tgetstr("up", &sbp);
  sbp = BCbuf;
  BC = tgetstr("bc", &sbp);

  b_lines = t_lines - 1;
  p_lines = t_lines - 4;
  return YEA;
}


void
do_move(destcol, destline)
  int destcol, destline;
{
  extern void ochar();

  tputs(tgoto(cm, destcol, destline), 0, ochar);
}
