/* tn5250 -- an implentation of the 5250 telnet protocol.
 * Copyright (C) 1997 Michael Madore
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include <assert.h>
#include "displaybuf.h"
#include "formattable.h"
#include "ctype.h"
#include "codes5250.h"
#include "utility.h"

FormatTable::FormatTable ()
{
   memset(screen_map, 0xFF, sizeof(screen_map));
   curfield = 0;
   numfields = 0;
}

FormatTable::~FormatTable ()
{
}

void FormatTable::AddField(unsigned char FFW1, unsigned char FFW2, 
     unsigned char FCW1, unsigned char FCW2, unsigned char Attr, 
     unsigned char Length1, unsigned char Length2, 
     unsigned char StartRow, unsigned char StartCol)
{

   int col, row;
   int count;
   int cur;   
   int done;

   LOG (("FormatTable::AddField: entered.\n"));
   LOG (("FormatTable::AddField: field format word = 0x%02X%02X\n",
      FFW1, FFW2));
   LOG (("FormatTable::AddField: field control word = 0x%02X%02X\n",
      FCW1, FCW2));
   LOG (("FormatTable::AddField: L1 = %X; L2 = %X\n", Length1, Length2));
   LOG (("FormatTable::AddField: numfields = %d\n", numfields));

   cur = 0;
   done = 0;
   while(!done && cur < numfields)
   {
      if((fields[cur].start_col == StartCol) &&
         (fields[cur].start_row == StartRow))
      {
         fields[cur].FFW[0] = FFW1;
         fields[cur].FFW[1] = FFW2;

         fields[cur].attribute = Attr;

         done = 1;
      }
      cur++;
   }

   if(!done)
   {
      fields[numfields].FFW[0] = FFW1;
      fields[numfields].FFW[1] = FFW2;

      fields[numfields].FCW[0] = FCW1;
      fields[numfields].FCW[1] = FCW2;

      fields[numfields].attribute = Attr;

      fields[numfields].length = Length1 << 8;
      fields[numfields].length = fields[numfields].length + Length2;

      fields[numfields].MDT = IsFieldModified(numfields);
      LOG(("MDT == %d\n", fields[numfields].MDT));

      fields[numfields].start_row = StartRow;
      fields[numfields].start_col = StartCol;
      fields[numfields].start_pos = (StartRow - 1) * 80 + StartCol - 1;
 

      fields[numfields].end_col = ((fields[numfields].length+
         fields[numfields].start_col - 1) % 80);

      fields[numfields].end_pos = fields[numfields].start_pos+
         fields[numfields].length - 1;

      fields[numfields].end_row = (fields[numfields].end_pos) / 80 + 1;
      if(fields[numfields].end_col == 0)
         fields[numfields].end_col=80;

      fields[numfields].numrows = fields[numfields].end_row -
         fields[numfields].start_row + 1;

      for(row = StartRow - 1, col = StartCol - 1, count = 0;
         count < fields[numfields].length; count++)
      {
         screen_map[col][row] = numfields;
         if (++col == 80) {
	    col = 0;
	    row++;
	 }
      }


      int bufsize = 120;
      if (fields[numfields].length > 120) {
         bufsize = fields[numfields].length;
      }

      fields[numfields].data = new unsigned char[bufsize];

      memset(fields[numfields].data, '\0', fields[numfields].length); 

      numfields++;
   }
}

void FormatTable::DumpMap()
{
#ifndef NDEBUG
   int col, row;
  
   LOG (("FormatTable::DumpMap: "));
   for(row = 0; row < 25; row++) {
      for(col = 0; col < 80; col++)
	 LOG (("%X:", screen_map[col][row]));
      LOG (("\n"));
   }
#endif
}

void FormatTable::DumpField(int number)
{
#ifndef NDEBUG
   int curchar;

   LOG (("FormatTable::DumpField: "));
   for(curchar = 0; curchar < fields[number].length; curchar++)
      LOG (("%c", ebcdic2ascii (fields[number].data[curchar])));
   LOG (("\n"));
#endif
}

int FormatTable::NumFields()
{
   return(numfields);
}

void FormatTable::SetCurField(int CurField)
{
   LOG (("FormatTable::SetCurField: CurField = %d; numfields = %d\n",
   	CurField, numfields));
   curfield = CurField;
}

int FormatTable::CurField()
{
   return(curfield);
}

int FormatTable::FirstField()
{

   if(numfields > 0)
   {
      curfield = 0;
      while(IsBypass(curfield))
      {
         if (curfield == NumFields()-1) 
         {
            curfield = -1;
            break;
         }
         else
            curfield++;
      }
   }
   else {
      curfield = -1;
   }
   
   return(curfield);
}

int FormatTable::NextField ()
{
   int oldcurfield = curfield;

   LOG (("FormatTable::NextField: numfields = %d; curfield = %d\n",
      numfields, curfield));

   assert (curfield != -1);

   if(numfields > 0) {
      do {
	 curfield = (curfield + 1) % numfields;
      } while (IsBypass(curfield) && curfield != oldcurfield);
   }
   else
      curfield = -1;

   return(curfield);
}

int FormatTable::NextField (int x, int y)
{
   int nx = x, ny = y;
   if (FieldNum (x, y) >= 0)
      return NextField ();

   do {
      if (++nx == 80) {
	 nx = 0;
	 if (++ny == 24)
	    ny = 0;
      }
      curfield = FieldNum (nx, ny);
   } while (curfield < 0 && !(x == nx && y == ny));

   if (IsBypass (curfield))
      return NextField ();

   return curfield;
}

int FormatTable::PrevField ()
{
   int oldcurfield = curfield;

   LOG (("FormatTable::PrevField: numfields = %d; curfield = %d\n",
      numfields, curfield)); 

   assert (curfield != -1);

   if(numfields > 0) {
      do {
	 if (--curfield < 0)
	    curfield = numfields-1;
      } while (IsBypass(curfield));
   }
   else
      curfield = -1;

   return(curfield);
}

int FormatTable::PrevField (int x, int y)
{
   int nx = x, ny = y;

   if (FieldNum (x, y) >= 0)
      return PrevField ();

   do {
      if (--x < 0) {
	 x = 79;
	 if (--y < 0)
	    y = 23;
      }
      curfield = FieldNum (x, y);
   } while (curfield < 0 && !(nx == x && ny == y));

   if (IsBypass (curfield))
      return PrevField ();

   return curfield;
}

int FormatTable::FieldNum(int X, int Y)
{
   if (screen_map[X][Y] == 0xFF)
   	return -1;
   return screen_map[X][Y];
}

int FormatTable::IsBypass(int FieldNumber)
{
   return(fields[FieldNumber].FFW[0] & BYPASS);
}

int FormatTable::IsDupEnable(int FieldNumber)
{
   return(fields[FieldNumber].FFW[0] & DUP_ENABLE);
}

int FormatTable::IsFieldModified(int FieldNumber)
{
   return(fields[FieldNumber].FFW[0] & MODIFIED);
}

int FormatTable::IsAlphaShift(int FieldNumber)
{
   return(fields[FieldNumber].FFW[0] & FIELD_MASK == ALPHA_SHIFT);
}

int FormatTable::IsAlphaOnly(int FieldNumber)
{
   return(fields[FieldNumber].FFW[0] & FIELD_MASK == ALPHA_ONLY);
}

int FormatTable::IsNumShift(int FieldNumber)
{
   return(fields[FieldNumber].FFW[0] & FIELD_MASK == NUM_SHIFT);
}

int FormatTable::IsNumOnly(int FieldNumber)
{
   return(fields[FieldNumber].FFW[0] & FIELD_MASK == NUM_ONLY);
}

int FormatTable::IsKataShift(int FieldNumber)
{
   return(fields[FieldNumber].FFW[0] & FIELD_MASK == KATA_SHIFT);
}

int FormatTable::IsDigitOnly(int FieldNumber)
{
   return(fields[FieldNumber].FFW[0] & FIELD_MASK == DIGIT_ONLY);
}

int FormatTable::IsMagReader(int FieldNumber)
{
   return(fields[FieldNumber].FFW[0] & FIELD_MASK == MAG_READER);
}

int FormatTable::IsSignedNum(int FieldNumber)
{
   return((fields[FieldNumber].FFW[0] & FIELD_MASK) == SIGNED_NUM);
}

unsigned char FormatTable::FieldType(int FieldNumber)
{
   return(fields[FieldNumber].FFW[0] & FIELD_MASK);
}

char *FormatTable::FieldDescription(int FieldNumber)
{

  switch (fields[FieldNumber].FFW[0] & FIELD_MASK) {
  case (ALPHA_SHIFT):
    return("Alpha Shift");
    break;
  case (DUP_ENABLE):
    return("Dup Enabled");
    break;
  case (ALPHA_ONLY):
    return("Alpha Only");
    break;
  case (NUM_SHIFT):
    return("Numeric Shift");
    break;
  case (NUM_ONLY):
    return("Numeric Only");
    break;
  case (KATA_SHIFT):
    return("Katakana");
    break;
  case (DIGIT_ONLY):
    return("Digits Only");
    break;
  case (MAG_READER):
    return("Mag Reader I/O Field");
    break;
  case (SIGNED_NUM):
    return("Signed Numeric");
    break;
  default:
     return("");
     break;
  }
}

/* FFW byte 2 */

int FormatTable::IsAutoEnter(int FieldNumber)
{
   return(fields[FieldNumber].FFW[1] & AUTO_ENTER);
}

int FormatTable::IsFer(int FieldNumber)
{
   return(fields[FieldNumber].FFW[1] & FER);
}

int FormatTable::IsMonocase(int FieldNumber)
{
   return(fields[FieldNumber].FFW[1] & MONOCASE);
}

int FormatTable::IsReserved(int FieldNumber)
{
   return(fields[FieldNumber].FFW[1] & RESERVED);
}

int FormatTable::IsMandatory(int FieldNumber)
{
   return(fields[FieldNumber].FFW[1] & MANDATORY);
}

int FormatTable::IsNoAdjust(int FieldNumber)
{
   return(fields[FieldNumber].FFW[1] & MAND_FILL_MASK == NO_ADJUST);
}

int FormatTable::IsMF_1(int FieldNumber)
{
   return(fields[FieldNumber].FFW[1] & MAND_FILL_MASK == MF_RESERVED_1);
}

int FormatTable::IsMF_2(int FieldNumber)
{
   return(fields[FieldNumber].FFW[1] & MAND_FILL_MASK == MF_RESERVED_2);
}

int FormatTable::IsMF_3(int FieldNumber)
{
   return(fields[FieldNumber].FFW[1] & MAND_FILL_MASK == MF_RESERVED_3);
}

int FormatTable::IsMF_4(int FieldNumber)
{
   return(fields[FieldNumber].FFW[1] & MAND_FILL_MASK == MF_RESERVED_4);
}

int FormatTable::IsRightZero(int FieldNumber)
{
   return(fields[FieldNumber].FFW[1] & MAND_FILL_MASK == RIGHT_ZERO);
}

int FormatTable::IsRightBlank(int FieldNumber) 
{
   return(fields[FieldNumber].FFW[1] & MAND_FILL_MASK == RIGHT_BLANK);
}

int FormatTable::IsMandatoryFill(int FieldNumber) 
{
   return(fields[FieldNumber].FFW[1] & MANDATORY_FILL);
}

unsigned char FormatTable::AdjustFill(int FieldNumber)
{
   return(fields[FieldNumber].FFW[1] & MAND_FILL_MASK);
}

char * FormatTable::AdjustDescription(int FieldNumber)
{

    switch (fields[FieldNumber].FFW[1] & MAND_FILL_MASK) {
    case (NO_ADJUST):
      return("No Adjust");
      break;
    case (MF_RESERVED_1):
      return("Reserved 1");
      break;
    case (MF_RESERVED_2):
      return("Reserved 2");
      break;
    case (MF_RESERVED_3):
      return("Reserved 3");
      break;
    case (MF_RESERVED_4):
      return("Reserved 4");
      break;
    case (RIGHT_ZERO):
      return("Right Adjust, Zero Fill");
      break;
    case (RIGHT_BLANK):
      return("Right Adjust, Blank Fill");
      break;
    case (MANDATORY_FILL):
      return("Mandatory Fill");
      break;
    default:
      return("");
      break;
    }
}

int FormatTable::Attribute(int FieldNumber)
{
   return(fields[FieldNumber].attribute);
}

int FormatTable::StartRow(int FieldNumber)
{
   return(fields[FieldNumber].start_row);
}

int FormatTable::StartCol(int FieldNumber)
{
   return(fields[FieldNumber].start_col);
}

int FormatTable::EndRow(int FieldNumber)
{
   return(fields[FieldNumber].end_row);
}

int FormatTable::EndCol(int FieldNumber)
{
   return(fields[FieldNumber].end_col);
}

int FormatTable::NumRows(int FieldNumber)
{
   return(fields[FieldNumber].numrows);
}

int FormatTable::Length(int FieldNumber)
{
   return(fields[FieldNumber].length);
}

int FormatTable::IsFull(int FieldNumber)
{
   return(fields[FieldNumber].data[fields[FieldNumber].length-1] != 0);
}

int FormatTable::Save()
{
   memcpy(savebuffer.fields, fields, sizeof(fields));
   memcpy(savebuffer.screen_map, screen_map, sizeof(screen_map));
   savebuffer.numfields = numfields;
   
   return(formatbuffer.SetBuffer(&savebuffer, sizeof(savebuffer)));
}

void FormatTable::Restore(int formatnum)
{
   formatbuffer.GetBuffer(&savebuffer, formatnum); 
   memcpy(fields, savebuffer.fields, sizeof(fields));
   memcpy(screen_map, savebuffer.screen_map, sizeof(screen_map));
   numfields = savebuffer.numfields;
}

void FormatTable::Clear()
{
   int curfield;
   
   numfields = 0;
   for(curfield = 0; curfield < numfields; curfield++) {
      fields[curfield].MDT = 0;
      delete fields[curfield].data;
   }   

   MasterMDT = 0;
   message_line = 24;

   memset(screen_map, 0xFF, sizeof(screen_map));
   curfield = 0;

   LOG (("FormatTable::Clear: entered.\n"));
}

int FormatTable::IsModified(int FieldNumber)
{
   return(fields[FieldNumber].MDT);
}

void FormatTable::ClearModified(int FieldNumber)
{
   fields[FieldNumber].MDT = 0;
}

/* CountLeft returns the number of characters in the specified
   field which are left of the specified cursor position */
int FormatTable::CountLeft(int FieldNumber, int X, int Y)
{
   return((Y * 80 + X) - fields[FieldNumber].start_pos);
}

/* CountRight returns the number of characters in the specified
   field which are right of the specified cursor position */
int FormatTable::CountRight(int FieldNumber, int X, int Y)
{
   return(fields[FieldNumber].end_pos - (Y * 80 + X));
}

/* CountEOF returns the number of characters in the specified
   field up to and including the last non-blank or non-null */
int FormatTable::CountEOF(int FieldNumber)
{
   int count;
   unsigned char c;

   for(count = fields[FieldNumber].length; count > 0; count--) {
      c = fields[FieldNumber].data[count-1];
      if(c != '\0' && c != ' ') break;
   }
   return(count);
}

void FormatTable::AddChar(int X, int Y, unsigned char Data)
{
   int fieldnum;
   int ins_loc;

   fieldnum = screen_map[X][Y];
   ins_loc = CountLeft(fieldnum, X, Y);

   fields[fieldnum].data[ins_loc] = Data;

   fields[fieldnum].MDT = 1;
}

void FormatTable::PutChar(int X, int Y, unsigned char Data)
{
   int fieldnum;
   int ins_loc;

   fieldnum = screen_map[X][Y];
   ins_loc = CountLeft(fieldnum, X, Y);

   fields[fieldnum].data[ins_loc] = Data;

}

void FormatTable::InsChar(int X, int Y, unsigned char Data)
{
   int fieldnum;
   int ins_loc;
   int shiftcount;

   fieldnum = screen_map[X][Y];

   ins_loc = CountLeft(fieldnum, X, Y);

   shiftcount = CountRight(fieldnum, X, Y);

   if(shiftcount > 0) {
      memmove(fields[fieldnum].data+ins_loc+1,
         fields[fieldnum].data+ins_loc, shiftcount);
   }

   fields[fieldnum].data[ins_loc] = Data;
   fields[fieldnum].MDT = 1;
}

void FormatTable::DelChar(int X, int Y)
{
   int fieldnum;
   int ins_loc;
   int shiftcount;

   fieldnum = screen_map[X][Y];

   ins_loc = CountLeft(fieldnum, X, Y);

   shiftcount = CountRight(fieldnum, X, Y);

   if(shiftcount > 0) {
      memmove(fields[fieldnum].data+ins_loc,
         fields[fieldnum].data+ins_loc+1, shiftcount);
   }
   fields[fieldnum].data[ins_loc+shiftcount] = 0x00;

   fields[fieldnum].MDT = 1;
}

void FormatTable::FieldChar(int FieldNumber, int Position, unsigned char Data)
{
   fields[FieldNumber].data[Position] = Data;
}


unsigned char FormatTable::GetChar(int FieldNumber, int Position)
{
   return(fields[FieldNumber].data[Position]);
}

void FormatTable::SetMDT()
{
   MasterMDT = 1;
}

void FormatTable::ClearMDT()
{
   MasterMDT = 0;
}

int FormatTable::IsMDTSet()
{
   return(MasterMDT);
}

void FormatTable::SetMessageLine(int row)
{
   message_line = row;
}

int FormatTable::GetMessageLine()
{
   return(message_line);
}

int FormatTable::FieldExit(DisplayBuffer& disp_buf, int X, int Y)
{

   LOG(("FormatTable::FieldExit Current Field: %d, Column %d, Row %d\n",
	   curfield, X, Y));

   /* Print the Information about this field for reference */
   LOG(("SR %d, SC %d, ER %d, EC %d, Length %d, CountLeft %d, CountRight %d\n",
	   StartRow(curfield), StartCol(curfield), 
	   EndRow(curfield), EndCol(curfield), Length(curfield),
	   CountLeft(curfield, X, Y), CountRight(curfield, X, Y)));
   DumpField(curfield);

   /* Note as you read this code column and Row are numbered as 1 offset
      where as X and Y are 0 offset in the upper left corner */
    
   switch(FieldType(curfield)) {
   case ALPHA_SHIFT :
     LOG(("Processing Alpha Shift\n"));
     /* accept all characters */
     ProcessAdjust(disp_buf, X, Y);
     break;
   case ALPHA_ONLY :
     LOG(("Processing Alpha Only\n"));
     ProcessAdjust(disp_buf, X, Y);
     break;
   case NUM_SHIFT :
     LOG(("Processing Numeric Shift\n"));
     ProcessAdjust(disp_buf, X, Y);
     break;
   case NUM_ONLY :
     LOG(("Processing Numberic Only\n"));
     ProcessAdjust(disp_buf, X, Y);
     break;
   case KATA_SHIFT :
     LOG(("Processing Katakana NOT CODED\n"));
     break;
   case DIGIT_ONLY :
     LOG(("Processing Digit Only\n"));
     ProcessAdjust(disp_buf, X, Y);
     break;
   case MAG_READER :
     LOG(("Processing Mag Reader NOT CODED\n"));
     break;
   case SIGNED_NUM :
     LOG(("Processing Signed Numberic\n"));
     ShiftRightFill(CountLeft(curfield, X, Y), ascii2ebcdic(' '));
     disp_buf.add (fields[curfield].start_row - 1,
		   fields[curfield].start_col - 1,
		   fields[curfield].data, fields[curfield].length);
   }
   
   fields[curfield].MDT = 1;

   return (0);
}

void FormatTable::ClearFieldBufferToEOF (int pos, unsigned char fill_char)
{
  int i;

  for (i = pos; i < fields[curfield].length; i++) {
    fields[curfield].data[i] = fill_char;
  }
}

void FormatTable::ProcessAdjust(DisplayBuffer& disp_buf, int X, int Y)
{

   switch(AdjustFill(curfield)) {
   case NO_ADJUST :
      LOG(("Processing Adjust/Edit: No_Adjust\n"));
      ClearFieldBufferToEOF(CountLeft(curfield, X, Y), ascii2ebcdic('\0'));
      ClearScreenToEOF(disp_buf, CountRight(curfield, X, Y));
      break;
   case RIGHT_ZERO :
      LOG(("Processing Adjust/Edit: Right Zero Fill\n"));
      ShiftRightFill(CountLeft(curfield, X, Y), ascii2ebcdic('0'));
      disp_buf.add (fields[curfield].start_row - 1,
		    fields[curfield].start_col - 1, fields[curfield].data,
		    fields[curfield].length);
      break;
   case RIGHT_BLANK :
      LOG(("Processing Adjust/Edit: Right Blank Fill\n"));
      ShiftRightFill(CountLeft(curfield, X, Y), ascii2ebcdic(' '));
      disp_buf.add (fields[curfield].start_row - 1,
		    fields[curfield].start_col - 1,
		    fields[curfield].data, fields[curfield].length);
      break;
   case MANDATORY_FILL :
      LOG(("Processing Adjust/Edit: Manditory Fill  NOT CODED\n"));
      break;
   }

   DumpField(curfield);
  
}

void FormatTable::ClearScreenToEOF(DisplayBuffer& disp_buf, int rightcount)
{
   for (int i = 0; i <= rightcount; i++)
      disp_buf.add (0x40);  /* EBCDIC blank */
}

void FormatTable::ShiftRightFill(int leftcount, unsigned char fill_char)
{

   int o, n;

   n = Length(curfield) - 1;
   if (IsNumOnly(curfield) || IsSignedNum(curfield)) n--;

   for (o = leftcount - 1; o >= 0; o--) {

     fields[curfield].data[n] = fields[curfield].data[o];

     n--;
   }

   /* fill the balance with 'fill_char' */
   for (; n >= 0; n--){
     fields[curfield].data[n] = fill_char;
   }
}

void FormatTable::SetMinusZoneScreen(DisplayBuffer& disp_buf, int curfld)
{
   unsigned char lastchar;

   /* Set the Zone field */
   lastchar = fields[curfld].data[fields[curfld].length - 2];
   if (ebcdic2ascii(lastchar) >= '0' || ebcdic2ascii(lastchar) <= 9) {
     lastchar = 0xD0 | lastchar & 0x0F;
   }
   else {
     LOG(("Error: Tried to set Zone field for Negative number on non digit.\n"));
   }
   fields[curfld].data[fields[curfld].length - 2] = lastchar;

   /* fix the image on the screen */
   disp_buf.cursor (fields[curfld].end_row - 1, fields[curfld].end_col - 1);
   disp_buf.add (ascii2ebcdic('-'));
}
