/*
 * Copyright (C) 1998
 *	Sony Computer Science Laboratories Inc.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY SONY CSL AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL SONY CSL OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: stat.c,v 1.6 2000/06/30 11:16:26 kjc Exp kjc $
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/queue.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <math.h>

#include "tcpdstat.h"

struct timeval start_time, end_time;
int read_count = 0;
int caplen_total = 0;
int caplen_max = 0;
double rate_mean = 0;
double rate_var = 0;
int rate_count = 0;

int use_ipflow = 1;
char *pktlen_filename = NULL;
FILE *pktlen_file = NULL;
int debug = 0;

static char *file = "-";	/* stdin is the default input */

struct pkt_cnt tcpdstat[PROTOTYPE_MAX];

#define PKTSIZE_BUCKETS		16
static int pktsize_buckets[PKTSIZE_BUCKETS];

static void show_stat(void);
static void sigint_handler(int sig);
static void pktsize_print(void);
static void write_pktlen_dist(FILE *fp);

static void
sigint_handler(int sig)
{
    fprintf(stderr, "got signal %d. exiting...\n", sig);
    exit(1);
}

static void
usage(void)
{
    fprintf(stderr, "usage: tcpdstat [-dn] [-c count] [-w len_file] [dumpfile]\n");
    fprintf(stderr, "            -d: debug\n");
    fprintf(stderr, "            -n: no flow info\n");
    fprintf(stderr, "            -c: exit after \"count\" packets\n");
    fprintf(stderr, "            -l: write packet length distributions to a file\n");
    exit(1);
}


int main(argc, argv)
    int argc;
    char **argv;
{
    int fd, ch, len;

    while ((ch = getopt(argc, argv, "c:dnw:")) != EOF) {
	switch (ch) {
	case 'c':
	    read_count = atoi(optarg);
	    break;
	case 'd':
	    debug = 1;
	    break;
	case 'n':
	    use_ipflow = 0;
	    break;
	case 'w':
	    pktlen_filename = optarg;
	    break;
	default:
	    usage();
	}
    }
    argc -= optind;
    argv += optind;
    if (argc > 0)
	file = argv[0];

    signal(SIGINT, sigint_handler);
    signal(SIGTERM, sigint_handler);

    printf("\nDumpFile:  %s\n", file);
    if (strcmp(file, "-") != 0) {
	struct stat statbuf;

	if (stat(file, &statbuf) == 0)
	    printf("FileSize: %.2fMB\n", (double)statbuf.st_size/1024/1024);
    }

    if (pktlen_filename != NULL) {
	if ((pktlen_file = fopen(pktlen_filename, "w")) == NULL) {
	    fprintf(stderr, "can't open packet lenght output file: %s!\n",
		    pktlen_filename);
	    exit(1);
	}
    }

    fd = open_dump(file);

    do {
	len = read_dump();
    } while (len > 0);

    close_dump();

    show_stat();

    if (pktlen_file != NULL) {
	write_pktlen_dist(pktlen_file);
	fclose(pktlen_file);
    }

    ipflow_destroy();
    
    return (0);
}

/* the names must match "enum protos" */
struct {
    char *name;
    int level;
} protos[] = {
	{"total    ",	0}, 
        {"ip       ",	1},
        {" tcp     ",	2},
        {"  http(s)",	3},
        {"  http(c)",	3},
        {"  squid  ",	3},
        {"  smtp   ",	3},
        {"  nntp   ",	3},
        {"  ftp    ",	3},
        {"  pop3   ",	3},
        {"  imap   ",	3},
        {"  telnet ",	3},
        {"  ssh    ",	3},
        {"  dns    ",	3},
        {"  bgp    ",	3},
        {"  napster",	3},
        {"  realaud",	3},
        {"  rtsp   ",	3},
        {"  icecast",	3},
        {"  hotline",	3},
        {"  other  ",	3},
        {" udp     ",	2},
        {"  dns    ",	3},
        {"  rip    ",	3},
        {"  mcast  ",	3},
        {"  realaud",	3},
        {"  halflif",	3},
        {"  starcra",	3},
        {"  everque",	3},
        {"  unreal ",	3},
        {"  quake  ",	3},
        {"  suseeme",	3},
        {"  other  ",	3},
        {" icmp    ",	2},
        {" igmp    ",	2},
        {" ospf    ",	2},
        {" ipip    ",	2},
        {" ipsec   ",	2},
        {" ip6     ",	2},
        {" other   ",	2},
        {" frag    ",	2},
        {"ip6      ",	1},
        {" tcp6    ",	2},
        {"  http(s)",	3},
        {"  http(c)",	3},
        {"  squid  ",	3},
        {"  smtp   ",	3},
        {"  nntp   ",	3},
        {"  ftp    ",	3},
        {"  pop3   ",	3},
        {"  imap   ",	3},
        {"  telnet ",	3},
        {"  ssh    ",	3},
        {"  dns    ",	3},
        {"  bgp    ",	3},
        {"  napster",	3},
        {"  realaud",	3},
        {"  rtsp   ",	3},
        {"  icecast",	3},
        {"  hotline",	3},
        {"  other  ",	3},
        {" udp6    ",	2},
        {"  dns    ",	3},
        {"  rip    ",	3},
        {"  mcast  ",	3},
        {"  realaud",	3},
        {"  halflif",	3},
        {"  starcra",	3},
        {"  everque",	3},
        {"  unreal ",	3},
        {"  quake  ",	3},
        {"  suseeme",	3},
        {"  other  ",	3},
        {" icmp6   ",	2},
        {" ospf6   ",	2},
        {" ip4     ",	2},
        {" ip6     ",	2},
        {" ipsec6  ",	2},
        {" other6  ",	2},
        {" frag6   ",	2},
        {"other    ",	1},
        {NULL,		-1}
};

static void show_stat(void)
{
    int i;
    time_t t;
    struct tm *tm;
    double sec, stddev;
    double frac_pkts, frac_bytes, avg_bytes;

    /* create Id from start_time. provide time info.  */
    t = start_time.tv_sec;
    tm = localtime(&t);
    printf("Id: %.4d%.2d%.2d%.2d%.2d\n", tm->tm_year + 1900, tm->tm_mon + 1,
	   tm->tm_mday, tm->tm_hour, tm->tm_min);
    printf("StartTime: %s", ctime(&t));
    t = end_time.tv_sec;
    printf("EndTime:   %s", ctime(&t));
    sec = (double)(end_time.tv_sec - start_time.tv_sec)
	    + (double)(end_time.tv_usec - start_time.tv_usec) / 1000000.0;
    printf("TotalTime: %.2f seconds\n", sec);

    printf("TotalCapSize: %.2fMB  CapLen: %d bytes\n",
	   (double)caplen_total/1024/1024, caplen_max);

    printf("# of packets: %d", tcpdstat[TOTAL].packets);
    if (tcpdstat[TOTAL].bytes > 1024*1024)
	    printf(" (%.2fMB)\n",
		   (double)tcpdstat[TOTAL].bytes/(1024*1024));
    else 
	    printf(" (%.2fKB)\n", (double)tcpdstat[TOTAL].bytes / 1024);

    stddev = sqrt(rate_var / (rate_count - 1));
    printf("AvgRate: ");
    if (rate_mean > 1000000.0)
	    printf("%.2fMbps  stddev:%.2fM\n",
		   rate_mean / 1000000.0, stddev / 1000000.0);
    else if (rate_mean > 1000.0)
	    printf("%.2fKbps  stddev:%.2fK\n",
		   rate_mean / 1000.0, stddev / 1000.0);
    else
	    printf("%.2fbps  stddev:%.2f\n",
		   rate_mean, stddev);
    printf("\n");

    if (use_ipflow)
	    ipflow_show();

    pktsize_print();

    printf("\n### Protocol Breakdown ###\n");
    printf("<<<<\n");
    printf("     protocol\t\tpackets\t\t\tbytes\t\tbytes/pkt\n");
    printf("------------------------------------------------------------------------\n");
    for (i = 0; i < PROTOTYPE_MAX; i++) {
	if (tcpdstat[i].packets != 0) {
	    frac_pkts = (double)tcpdstat[i].packets
		/ (double)tcpdstat[TOTAL].packets * 100.0;
	    frac_bytes = (double)tcpdstat[i].bytes
		/ (double)tcpdstat[TOTAL].bytes * 100.0;
	    avg_bytes = (double)tcpdstat[i].bytes
		/ (double)tcpdstat[i].packets;
	    printf("[%d] %s %12d (%6.2f%%) %16qd (%6.2f%%)  %8.2f\n",
		   protos[i].level,
		   protos[i].name,
		   tcpdstat[i].packets, frac_pkts,
		   tcpdstat[i].bytes, frac_bytes,
		   avg_bytes);
	}
    }
    printf(">>>>\n");
    printf("\n\n");
}

#define MAX_PACKET_SIZE		9180  /* MTU for ATM */
static int pktlen_dist[MAX_PACKET_SIZE+1];

void pktsize_add(int size)
{
    int n = 0;

    if (pktlen_file != NULL) {
	int len = size;
	if (len > MAX_PACKET_SIZE)
	    len = MAX_PACKET_SIZE;
	pktlen_dist[len]++;
    }

    while (size > 1) {
	size >>= 1;
	n++;
    }
    if (n >= PKTSIZE_BUCKETS)
	n = PKTSIZE_BUCKETS - 1;
    pktsize_buckets[n]++;
}

static void pktsize_print(void)
{
    int i;

    printf("### Packet Size Distribution (including MAC headers) ###\n");
    printf("<<<<\n");
    for (i = 0; i < PKTSIZE_BUCKETS; i++)
	if (pktsize_buckets[i] > 0)
	    printf(" [%5d-%5d]: %10d\n",
		   1 << i, (1 << (i+1))-1, pktsize_buckets[i]);
    printf(">>>>\n");
    printf("\n");
}

static void write_pktlen_dist(FILE *fp)
{
    int len, count;

    fprintf(fp, "# packet_size  count  bytes\n");
    for (len = 0; len <= MAX_PACKET_SIZE; len++) {
	if ((count = pktlen_dist[len]) != 0)
	    fprintf(fp, "%d %d %lld\n", len,  count, (long long)len*count);
    }
}
