/*
 * stegserver.c 
 *
 * Main code/starting point for stegserver
 *
 * Copyright (c) 2003 Todd MacDermid <tmacd@synacklabs.net> 
 *
 */

#include <sys/types.h>
#include <sys/stat.h>

#include <netinet/in.h>

#include <dnet.h>
#include <fcntl.h>
#include <pcap.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <packetp.h>

#include "crypto.h"
#include "session.h"
#include "sha1.h"
#include "stegtunnel.h"


/* usage: prints the usage string and then exits */

void 
usage(char *prog_name) 
{
  printf("usage: %s [-dhnrV] [-f <filename>] [-p <proxyIp>]\n", prog_name);
  exit(0);
}

/*
 * inbound: this function handles packets destined back to us. It is called 
 * by the processPacket function in tunnel.c. It will update the packet and 
 * the st_ctx appropriately. If a proxy IP is being used, it will then write 
 * the resultant packet to the network. We cannot use a simplified tunnel in
 * that situation because there are multiple potential target IP addresses to 
 * write to on the outbound leg.
 */

int 
inbound(struct packetp_ctx *pp_ctx, uint8_t *packet, void *voidstate) 
{
  struct stegt_ctx *st_ctx;
  struct stegt_file *file_ctx;
  struct stegt_session *session;
  struct ip_hdr *ip_header;
  struct tcp_hdr *tcp_header;
  struct addr verbose_addr;
  char verbose_buf[80];  
  uint8_t extracted[4];
  uint32_t ack_num;
  uint32_t tmp_seq;
  uint32_t tmp_seqoff;
  uint32_t tmp_loc;
  int sendbytes;
  int i;
  
  st_ctx = voidstate;

  ip_header = (struct ip_hdr *)packet;

  if(st_ctx->verbose) {
    memcpy(&(verbose_addr.addr_ip), &(ip_header->ip_src), IP_ADDR_LEN);
    verbose_addr.addr_type = ADDR_TYPE_IP;
    verbose_addr.addr_bits = 32;
    addr_ntop(&verbose_addr, verbose_buf, 80);
  }

  if (ip_header->ip_p != IP_PROTO_TCP) {
    if(st_ctx->verbose)
      fprintf(stderr, 
	      "Received non-TCP packet from %s, continuing...\n", 
	      verbose_buf);
    return(0); /* Firewalled if using a proxy, no effect otherwise */
  }

  tcp_header = (struct tcp_hdr *) (packet + (ip_header->ip_hl * 4));

  if((session = 
      stegt_session_find(packet, st_ctx, IN, pp_ctx)) == NULL) {
    if(tcp_header->th_flags & TH_SYN) {
      if(st_ctx->verbose)
	fprintf(stderr, "Created new session from %s\n", verbose_buf);
      if((session = 
	  stegt_session_add(pp_ctx, st_ctx, packet)) == NULL) {
	fprintf(stderr, "Error adding new session from %s\n", verbose_buf);
	return(-1);
      }
      sequence_decrypt(packet, st_ctx, extracted);
      if((extracted[0] == 0) && (extracted[1] == 0) &&
	 (extracted[2] == 0) && (extracted[3] == 0)) {
	session->keyed = 1;
	if(st_ctx->verbose) {
	  fprintf(stderr, "%s: New session keyed\n", verbose_buf);
	}
	if(st_ctx->file_server) {
	  if((file_ctx = stegt_file_find(packet, st_ctx, IN)) == NULL) {
	    if((file_ctx = stegt_file_ropen(st_ctx->served_name)) == NULL) {
	      fprintf(stderr, "Error opening file: %s\n", st_ctx->served_name);
	      session->keyed = 0;
	      return(-1);
	    }
	    stegt_file_add(st_ctx, session, file_ctx);
	    session->keyed = 1;
	  } else {
	    if(file_ctx->active_session == NULL) {
	      session->keyed = 1;
	      file_ctx->active_session = session;
	      if(file_ctx->need_resynch) {
		file_ctx->need_resynch = 0;
	      }
	      session->cur_file = file_ctx;
	    } else {
	      session->keyed = 0;
	    }
	  }
	}
      } else if((extracted[0] == 0) && (extracted[1] == 0) &&
		(extracted[2] == 0) && (extracted[3] == 1)) {
	
	/* Not supported yet, server receiving files. */

      }
      session->tcp_state = TCP_SYN_RECEIVED;
    } else {
      return(0);
    }
  } else {
    if(session->keyed) {
      ipid_decrypt(packet, st_ctx, extracted);
      if(st_ctx->file_server) {
	if((file_ctx = session->cur_file) != NULL) {
	  if((extracted[0] != 0) || (extracted[1] != 0)) {
	    if((file_ctx->pot_resynch[0] != 0) || 
	       (file_ctx->pot_resynch[1] != 0)) {
	      if((memcmp(file_ctx->pot_resynch, extracted, 2)) == 0) {
		tmp_loc = file_ctx->file_loc % 64;
		if(tmp_loc >= ((extracted[0] & 0x1F) << 1)) {
		  file_ctx->file_loc = file_ctx->file_loc & 0xFFFFFFE0;
		  file_ctx->file_loc += ((extracted[0] & 0x1F) << 1);
		} else {
		  file_ctx->file_loc = file_ctx->file_loc & 0xFFFFFFE0;
		  file_ctx->file_loc +=  ((extracted[0] & 0x1F) << 1);
		  file_ctx->file_loc -= 64;
		}
		tmp_seqoff = tcp_header->th_ack % 16384;
		tmp_seq = ((extracted[0] & 0xE0) << 3);
		tmp_seq |= extracted[1];
		tmp_seq = (tmp_seq << 3);
		if(tmp_seqoff <= tmp_seq) {
		  file_ctx->synch_seq = ntohl(tcp_header->th_ack) & 0xFFFFC000;
		  file_ctx->synch_seq += tmp_seq;
		} else {
		  file_ctx->synch_seq = ntohl(tcp_header->th_ack) & 0xFFFFC000;
		  file_ctx->synch_seq += tmp_seq;
		  file_ctx->synch_seq += 16384;
		}
		file_ctx->need_resynch = 1;
		for(i = 0; i < 2; i++) {
		  file_ctx->old_resynch[i] = file_ctx->pot_resynch[i];
		  file_ctx->pot_resynch[i] = 0;
		}
		lseek(file_ctx->fd, file_ctx->file_loc, SEEK_SET);
		stegt_file_read(file_ctx);
	      }
	    } 
	    if((memcmp(&(file_ctx->old_resynch), extracted, 2)) != 0) {
	      memcpy(&(file_ctx->pot_resynch), extracted, 2);
	    }	    
	  } else {
	    file_ctx->pot_resynch[0] = 0;
	    file_ctx->pot_resynch[1] = 0;
	  }
	}
      } else {
	if(extracted[0] != 0) fprintf(stdout, "%c", extracted[0]);
	if(extracted[1] != 0) fprintf(stdout, "%c", extracted[1]);
      }
      ack_num = htonl(ntohl(tcp_header->th_ack) + session->seq_offset);
      memcpy(&(tcp_header->th_ack), &(ack_num), 4);
    }
    if (ntohs(tcp_header->th_flags) & TH_RST) {
      if(st_ctx->verbose)
	fprintf(stderr, "%s: session closed (reset)\n", verbose_buf);
      stegt_session_del(st_ctx, pp_ctx);
    } else if (tcp_header->th_flags & TH_FIN) {
      if(session->tcp_state == TCP_ESTABLISHED) {
	session->tcp_state = TCP_FIN_RECEIVED;
      }
      else if (session->tcp_state == TCP_FIN_SENT) {
	if(tcp_header->th_flags & TH_ACK) {
	  session->tcp_state = TCP_FINACK_RECEIVED;
	} else {
	  session->tcp_state = TCP_FINACK_SENT;
	}
      }
    } else if (tcp_header->th_flags & TH_ACK) {
      if(session->tcp_state == TCP_SYNACK_SENT)
	session->tcp_state = TCP_ESTABLISHED;
      else if (session->tcp_state == TCP_FINACK_SENT){
	if(st_ctx->verbose)
	  fprintf(stderr, "%s: session closed (regular close)\n", verbose_buf);
	stegt_session_del(st_ctx, pp_ctx);
      }
    }
  }
  
  if(pp_ctx->wedge_type == PP_FAKEIP) {
    tcp_header->th_sport = htons(session->proxy_port);
    memcpy(&(ip_header->ip_src), &(pp_ctx->proxy_ip.addr_ip), IP_ADDR_LEN);
    memcpy(&(ip_header->ip_dst), &(pp_ctx->my_ip.addr_ip), IP_ADDR_LEN);
    sendbytes = ntohs(ip_header->ip_len);
    ip_checksum(packet, sendbytes);
    sendbytes = ip_send(pp_ctx->ip_handle, packet, sendbytes);
  }

  return(0);
}

/* outbound: this function handles packets destined outward. It is called 
 * by the processPacket function in tunnel.c. It will update the packet and 
 * the st_ctx appropriately. If a proxy IP is being used, it will then write 
 * the resultant packet to the network. We cannot use a simplified tunnel in
 * that situation because there are multiple potential target IP addresses to 
 * write to.  
 */ 

int 
outbound(struct packetp_ctx *pp_ctx, uint8_t *packet, void *voidstate) 
{
  struct stegt_ctx *st_ctx;
  struct stegt_session *session;
  struct stegt_file *file_ctx;
  struct ip_hdr *ip_header;
  struct tcp_hdr *tcp_header;
  struct addr verbose_addr;
  char verbose_buf[80];
  uint32_t seq_num;
  uint8_t cipher_stream[4];
  uint8_t readbuf[2];
  int i, sendbytes;

  st_ctx = voidstate;

  ip_header = (struct ip_hdr *)packet;

  if(st_ctx->verbose) {
    memcpy(&(verbose_addr.addr_ip), &(ip_header->ip_src), IP_ADDR_LEN);
    verbose_addr.addr_type = ADDR_TYPE_IP;
    verbose_addr.addr_bits = 32;
    addr_ntop(&verbose_addr, verbose_buf, 80);
  }

  if (ip_header->ip_p != IP_PROTO_TCP) {
    if(st_ctx->verbose)
      fprintf(stderr, 
	      "Received non-TCP packet from %s, continuing...\n", 
	      verbose_buf);
    return(0);
  }
  
  tcp_header = (struct tcp_hdr *) (packet + (ip_header->ip_hl * 4));
  
  if((session = stegt_session_find(packet, st_ctx, OUT, pp_ctx)) == NULL) {
    if(st_ctx->verbose) {
      fprintf(stderr, "Outbound packet to unknown destination\n");
    }
    return(0);
  } else {
    if(pp_ctx->wedge_type == PP_FAKEIP) {
      memcpy(&(tcp_header->th_dport), &(session->remote_port), 
	     sizeof(uint16_t));
      memcpy(&(ip_header->ip_src), &(pp_ctx->proxy_ip.addr_ip), IP_ADDR_LEN);
      memcpy(&(ip_header->ip_dst), &(session->remote_ip.addr_ip), IP_ADDR_LEN);
      addr_ntop(&(session->remote_ip), verbose_buf, 80);
    }
    
    /* OOOI! Don't forget to check for retransmit!!! */

    if(session->keyed) {
      if(tcp_header->th_flags & TH_SYN) {
	if(tcp_header->th_flags & TH_ACK) {
	  session->tcp_state = TCP_SYNACK_SENT;
	  sha_seq_stream(packet, st_ctx, cipher_stream);
	  if(st_ctx->file_server) {
	    cipher_stream[3] ^= 0x01;
	  }
	  memcpy(&seq_num, cipher_stream, 4);
	  session->seq_offset = ntohl(tcp_header->th_seq) - ntohl(seq_num);
	  memcpy(&(tcp_header->th_seq), &seq_num, 4);
	} else {
	  return(0);
	}
      } else {
	seq_num = htonl(ntohl(tcp_header->th_seq) - session->seq_offset);
	memcpy(&(tcp_header->th_seq), &seq_num, 4);
	sha_ipid_stream(packet, st_ctx, cipher_stream);
	if(st_ctx->file_server) {
	  if((file_ctx = session->cur_file) != NULL) {
	    if(file_ctx->need_resynch) {
	      if(ntohl(seq_num) > file_ctx->synch_seq) {
		file_ctx->need_resynch = 0;
	      } else {
		rand_get(st_ctx->rand_handle, cipher_stream, 2);
	      }
	    } 
	    if(!(file_ctx->need_resynch)) {
	      stegt_file_output(file_ctx, readbuf, 2);
	 
	      for(i = 0; i < 2; i++) {
		cipher_stream[i] = cipher_stream[i] ^ readbuf[i];
	      }
	    }
	  }
	} else if(st_ctx->repeat) {
	  for(i = 0; i < 2; i++) {
	    cipher_stream[i] =
	      (st_ctx->message)[session->msg_offset]^cipher_stream[i];
	    session->msg_offset++;
	    if((st_ctx->message)[session->msg_offset] == 0) {
	      session->msg_offset = 0;
	    }
	  }
	} else {	    
	  if(read(0, readbuf, 2) > 0) {
	    for(i = 0; i < 2; i++) {
	      cipher_stream[i] = cipher_stream[i] ^ readbuf[i];
	    }
	  }
	}
	memcpy(&(ip_header->ip_id), cipher_stream, 2);

	if (tcp_header->th_flags & TH_RST) {
	  if(st_ctx->verbose)
	    fprintf(stderr, "%s: session closed (reset)\n", verbose_buf);
	  stegt_session_del(st_ctx, pp_ctx);
	} else if (tcp_header->th_flags & TH_FIN) {
	  if(session->tcp_state == TCP_ESTABLISHED) {
	    session->tcp_state = TCP_FIN_SENT;
	  } else if (session->tcp_state == TCP_FIN_RECEIVED) {
	    session->tcp_state = TCP_FINACK_SENT;
	  }
	} else if (tcp_header->th_flags & TH_ACK) {
	  if(session->tcp_state == TCP_SYNACK_RECEIVED)
	    session->tcp_state = TCP_ESTABLISHED;
	  else if (session->tcp_state == TCP_FINACK_RECEIVED) {
	    if(st_ctx->verbose)
	      fprintf(stderr, 
		      "%s: session closed (regular close)\n", 
		      verbose_buf);
	    stegt_session_del(st_ctx, pp_ctx);
	  }
	}
      }
    }

    if(pp_ctx->wedge_type == PP_FAKEIP) {
      sendbytes = ntohs(ip_header->ip_len);
      ip_checksum(packet, sendbytes);
      sendbytes = ip_send(pp_ctx->ip_handle, packet, sendbytes);
    }
  }
  return(0);
}

/*
 * main: The starting point. Interpret command line options, initialize data 
 * structures, then call packetp_start, which only returns if something's
 * gone wrong.
 */

int main(int argc, char **argv) {
  int mode = SEQ_AND_IPID;
  struct packetp_ctx *pp_ctx;
  struct stegt_ctx *st_ctx;
  int got_proxy;
  int daemon;
  int c;

  if(getuid() != 0) {
    fprintf(stderr, "Gotta be root to run %s\n", argv[0]);
    return(-1);
  }

  /*
   * ARGUMENTS:
   *
   * -p proxyIp: used to set up the tunnel through a nonexistent 
   * IP address on the local subnet.
   *
   * -h: Print help message
   *
   * -V: Verbose mode. 
   *
   * -r: repeating mode. Read in input up front, and simply repeat that
   * to clients.
   *
   * -n NAT mode: Weakens the nonce in order to traverse NAT
   *
   * -f filename: file mode. serve filename to clients.
   *
   * -d: Daemon mode. fork into background. Must be in repeating or file mode!
   *
   * -v: Print version
   */

  st_ctx = stegt_ctx_init(mode);

  if((pp_ctx = packetp_init()) == NULL) 
    return(-1);

  got_proxy = 0;
  daemon = 0;

  while((c=getopt(argc, argv, "p:f:hnvVrd")) != -1) {
    switch(c) {
    case 'p':
      if ((addr_pton(optarg, &(pp_ctx->proxy_ip))) < 0) {
        fprintf(stderr, "Cannot resolve silent IP: %s\n", optarg);
	return(-1);
      }
      got_proxy = 1;
      break;

    case 'h':
      usage(argv[0]);
      break;

    case 'V':
      st_ctx->verbose = 1;
      break;

    case 'r':
      st_ctx->repeat = 1;
      break;

    case 'f':
      st_ctx->file_server = 1;
      strncpy(st_ctx->served_name, optarg, 79);
      break;

    case 'n':
      st_ctx->nat = 1;
      break;

    case 'v':
      version(argv[0]);
      break;

    case 'd':
      daemon = 1;
      break;

    }
  }
  
  if(got_proxy) {
    pp_ctx->wedge_type = PP_FAKEIP;
    pp_ctx->simplified = 0;
  }

  if(daemon && (!((st_ctx->repeat) || (st_ctx->file_server)))) {
    fprintf(stderr, "daemon mode requires input. Use -r or -f\n");
    usage(argv[0]);
  }

  if((st_ctx->repeat) && (st_ctx->file_server)) {
    fprintf(stderr, "Cannot use both repeat mode and file mode\n");
    usage(argv[0]);
  }

  if(st_ctx->repeat) {
    printf("Enter served message: ");
    fflush(stdout);
    fgets(st_ctx->message, 65534, stdin);
  }

  if(get_passphrase(st_ctx) < 0) {
    fprintf(stderr, "get_passphrase failed\n");
    return(-1);
  }

  if (fcntl(0, F_SETFL, O_NONBLOCK) < 0) {
    fprintf(stderr, "fcntl error\n");
    return(-1);
  } 

  if(daemon) {
    if(fork != 0) {
      exit(0);
    }
  }
  
  st_ctx->server = 1;

  packetp_start(pp_ctx, inbound, outbound, st_ctx);
  return(0);
}

