/*
 * forkarp.c
 *
 * Code that handles the forking of a child process to listen for and respond
 * to arp requests for the silent IP address.
 *
 * Also contains code to determine the hardware addresses of systems
 * that may or may not be in the current arp table. This necessitates
 * effectively maintaining a parallel arp table. Kind of icky, but
 * there it is.
 *
 * Copyright (c) 2002, 2003 Todd MacDermid <tmacd@synacklabs.net>
 *
 */

#include <dnet.h>
#include <pcap.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "packetp.h"

/* arp_handle and arp_entry are global so they can be accessed by the exit
 * handler to clean things up */

arp_t *arp_handle;
struct arp_entry arp_entry;

pid_t arp_pid; /* To be killed off on exit by the main program's exit handler */

/* kill_arpd - registered as an exit handler by the parent thread, to kill
 * the arping child off when it is done. 
 */

void kill_arpd() {
  kill(arp_pid, SIGINT);
}

/* remove_arp_entry - registered as an exit handler by the arping child,
 * to clean up the arp table when it is killed
 */

void remove_arp_entry() {
  arp_delete(arp_handle, &arp_entry);
  arp_close(arp_handle);
}

/* call_exit simply exits out, used as a signal handler for SIGINT and SIGTERM,
 * to ensure that the exit handlers get called when those signals received
 */

void call_exit(int throwaway) {
  exit(0);
}

void 
clean_arp(struct packetp_ctx *ctx) 
{
  struct arp_node *arp_item;
  time_t now;

  pthread_mutex_lock(&(ctx->arptab_mutex));

  if(hfirst(ctx->arp_table)) {
    now = time(NULL);
    do {
      arp_item = hstuff(ctx->arp_table);
      if (now - arp_item->set_time > 600) {
	free(hkey(ctx->arp_table));
	free(arp_item);
	hdel(ctx->arp_table);
      }  
    }
    while(hnext(ctx->arp_table));
  }
  pthread_mutex_unlock(&(ctx->arptab_mutex));
}

/*
 * packetp_get_mac_nonblock attempts to return a MAC address for a given
 * IP address, using both the kernel's arp table and the context's arp
 * table. Returns instantly, with a -1 if not found, or 0 if found.
 * 
 * The arp_entry should contain the IP address desired. 
 *
 * There originally was a blocking version of this, which turned into
 * dead code and was removed. The name is still accurate, so I kept it.
 */

int
packetp_get_mac_nonblock(struct packetp_ctx *ctx, struct arp_entry *entry) 
{
  struct arp_node *arp_item;
  time_t now;
  
  if((arp_get(ctx->arp_handle, entry)) == 0) 
    return(0);
  
  pthread_mutex_lock(&(ctx->arptab_mutex));
  if((hfind(ctx->arp_table, &(entry->arp_pa.addr_ip), IP_ADDR_LEN)) == TRUE) {
    arp_item = ctx->arp_table->ipos->stuff;
    now = time(NULL);
    if(((now - arp_item->set_time) < 600) && 
       ((now - arp_item->used_time) < 120)) {
      arp_item->used_time = now;
      memcpy(&(entry->arp_ha), &(arp_item->mac), sizeof(struct addr));
      pthread_mutex_unlock(&(ctx->arptab_mutex));
      return(0);
    } 
  }
  pthread_mutex_unlock(&(ctx->arptab_mutex));
  return(-1);
}

/* packetp_arp_wait runs as a separate thread when packetp_frame_send
 * is unable to determine an arp entry for the next hop on the way out.
 * it will periodically send arp requests for the next hop, and hope
 * that something updates one of the arp tables, so that it can send
 * its frame. It will time out eventually. 
 *
 * arg needs to be freed before it returns, along with arg->frame, as 
 * they were allocated by packetp_frame_send.
 */

void *
packetp_arp_wait(void *arg)
{
  int timer;
  struct timespec sleeptime;
  uint8_t packet[PACKETSIZE];
  struct thread_arg *targ;
  struct arp_hdr *arp_header;
  struct arp_ethip *arp_message;
  struct arp_entry dest_arp;
  struct eth_hdr *eth_header;
  struct ip_hdr *ip_header;
  int i;

  targ = (struct thread_arg *)arg;
  
  memcpy(&(dest_arp.arp_pa), &(targ->next_hop), sizeof(struct addr));
  
  memset(packet, 0x00, PACKETSIZE);
  eth_header = (struct eth_hdr *)packet;
  memset(eth_header->eth_dst.data, 0xFF, ETH_ADDR_LEN);
  memcpy(eth_header->eth_src.data, 
	 &(targ->rnode->interface.intf_link_addr.addr_eth), 
	 ETH_ADDR_LEN);
  eth_header->eth_type = htons(ETH_TYPE_ARP);
  
  arp_header = (struct arp_hdr *)(packet + sizeof(struct eth_hdr));
  arp_header->ar_hrd = htons(ARP_HRD_ETH);
  arp_header->ar_pro = htons(ARP_PRO_IP);
  arp_header->ar_hln = ETH_ADDR_LEN;
  arp_header->ar_pln = IP_ADDR_LEN;
  arp_header->ar_op = htons(ARP_OP_REQUEST);
  arp_message = (struct arp_ethip *)((uint8_t *)arp_header + 
				     sizeof(struct arp_hdr));
  memcpy(&(arp_message->ar_sha), 
	 &(targ->rnode->interface.intf_link_addr.addr_eth), 
	 ETH_ADDR_LEN);
  memcpy(&(arp_message->ar_spa), 
	 &(targ->rnode->interface.intf_addr.addr_ip), 
	 IP_ADDR_LEN);
  memcpy(&(arp_message->ar_tpa), &(targ->next_hop.addr_ip), IP_ADDR_LEN); 

  timer = 0;
  sleeptime.tv_sec = 0;
  sleeptime.tv_nsec = 10000000;

  while(timer < ARP_TIMEOUT) {
    if((timer % 100) == 0) { 
      eth_send(targ->rnode->eth_handle, packet, 
	       sizeof(struct eth_hdr) + sizeof(struct arp_hdr) +
	       2*IP_ADDR_LEN + 2* ETH_ADDR_LEN); 
    }
    nanosleep(&sleeptime, NULL);
    if(packetp_get_mac_nonblock(targ->ctx, &dest_arp) == 0) { 
      eth_header = (struct eth_hdr *)targ->frame;
      ip_header = (struct ip_hdr *)(targ->frame + ETH_HDR_LEN);
      
      memcpy(&(eth_header->eth_dst), 
	     &(dest_arp.arp_ha.addr_eth), 
	     ETH_ADDR_LEN);
      eth_send(targ->rnode->eth_handle, targ->frame, 
	       ntohs(ip_header->ip_len)+ETH_HDR_LEN);
      timer += ARP_TIMEOUT;
    }
    timer++;
  }

  free(targ->frame);
  free(targ);
  return(NULL);
}

/*
 * packetp_arp_update updates the internal packetp arp table with arp
 * responses. Must lock mutexes to play nicely with packetp_get_mac_nonblock
 */

void 
packetp_arp_update(struct packetp_ctx *ctx, uint8_t *packet)
{
  struct arp_hdr *arp_header;
  struct arp_ethip *arp_message;
  struct arp_node *arp_item;
  uint8_t ip_key[IP_ADDR_LEN];
  uint8_t *new_key;
  time_t now;

  arp_header = (struct arp_hdr *)(packet + sizeof(struct eth_hdr));
  arp_message = (struct arp_ethip *)((uint8_t *)arp_header 
				     + sizeof(struct arp_hdr));

  if(arp_header->ar_op == htons(ARP_OP_REPLY)) {
    pthread_mutex_lock(&(ctx->arptab_mutex));
    now = time(NULL);
    memcpy(ip_key, arp_message->ar_spa, IP_ADDR_LEN);

    if((hfind(ctx->arp_table, ip_key, IP_ADDR_LEN)) == TRUE) {
      arp_item = ctx->arp_table->ipos->stuff;
      arp_item->set_time = now;
      arp_item->used_time = now;
      memcpy(&(arp_item->mac.addr_eth), arp_message->ar_sha, ETH_ADDR_LEN);
    } else {
      if((arp_item = (struct arp_node *)
	  calloc(1, sizeof(struct arp_node))) == NULL) {
	pthread_mutex_unlock(&(ctx->arptab_mutex));
	return;
      }
      if((new_key = (uint8_t *)calloc(1, IP_ADDR_LEN)) == NULL) {
	free(arp_item);
	pthread_mutex_unlock(&(ctx->arptab_mutex));
	return;
      }

      memcpy(new_key, ip_key, IP_ADDR_LEN);
      arp_item->set_time = now;
      arp_item->used_time = now;
      memcpy(&(arp_item->mac.addr_eth), arp_message->ar_sha, ETH_ADDR_LEN);
      arp_item->mac.addr_type = ADDR_TYPE_ETH;
      arp_item->mac.addr_bits = ETH_ADDR_BITS;
      hadd(ctx->arp_table, new_key, IP_ADDR_LEN, arp_item);
    }
    pthread_mutex_unlock(&(ctx->arptab_mutex));
  }
  return;
}

/* forkarp_arp_loop - This is where the child spends most of its time, sniffing
 * packets, and responding with arp replies for arp requests that fall
 * within the netmask given in the silent_ip originally passed to 
 * packetp_fork_arpd
 */

void forkarp_arp_loop(pcap_t *pcap_handle, eth_t *ethernet_handle, 
	     eth_addr_t *my_mac, struct addr *silent_ip) {
  uint8_t *in_packet;
  uint8_t out_packet[PACKETSIZE];
  uint8_t mask[4];
  struct pcap_pkthdr packet_head;
  struct arp_hdr *in_arp_header, *out_arp_header;
  struct arp_ethip *in_arp_message, *out_arp_message;
  struct eth_hdr *eth_hdr_out;
  int i, match;

  for(i = 0; ((i+1)*8) <= silent_ip->addr_bits; i ++) 
    mask[i] = 0xFF;

  if((silent_ip->addr_bits % 8) != 0) 
    *(mask + (silent_ip->addr_bits / 8)) = 
      (~0 << (8 - (silent_ip->addr_bits % 8)));
  
  while(1) {
    in_packet = NULL;
    while((in_packet = (uint8_t *)pcap_next(pcap_handle, &packet_head)) != NULL) {
      in_packet += ETH_HDR_LEN;
      in_arp_header = (struct arp_hdr *)in_packet;
      if(in_arp_header->ar_op == htons(ARP_OP_REQUEST)) {
	in_arp_message = (struct arp_ethip *)(in_packet + sizeof(struct arp_hdr));
	match = 1;
	for(i = 0; i < IP_ADDR_LEN; i++) {
	  if (!(((in_arp_message->ar_tpa)[i] & mask[i]) ==
		((silent_ip->addr_data8)[i] & mask[i]))) {
	    match = 0;
	  }
	}
	if(match) {
	  memset(out_packet, 0x00, PACKETSIZE);
	  eth_hdr_out = (struct eth_hdr *)out_packet;
	  memcpy(eth_hdr_out->eth_dst.data, in_arp_message->ar_sha, ETH_ADDR_LEN);
	  memcpy(eth_hdr_out->eth_src.data, my_mac, ETH_ADDR_LEN);
	  
	  eth_hdr_out->eth_type = htons(ETH_TYPE_ARP);
	  out_arp_header = (struct arp_hdr *)(out_packet + sizeof(struct eth_hdr));
	  out_arp_header->ar_hrd = htons(ARP_HRD_ETH);
	  out_arp_header->ar_pro = htons(ARP_PRO_IP);
	  out_arp_header->ar_hln = ETH_ADDR_LEN;
	  out_arp_header->ar_pln = IP_ADDR_LEN;
	  out_arp_header->ar_op = htons(ARP_OP_REPLY);
	  out_arp_message = (struct arp_ethip *)((uint8_t *)out_arp_header + sizeof(struct arp_hdr));
	  
	  memcpy(&(out_arp_message->ar_sha), my_mac, ETH_ADDR_LEN);
	  memcpy(&(out_arp_message->ar_spa), &(in_arp_message->ar_tpa), IP_ADDR_LEN);
	  memcpy(&(out_arp_message->ar_tha), &(in_arp_message->ar_sha), ETH_ADDR_LEN);
	  memcpy(&(out_arp_message->ar_tpa), &(in_arp_message->ar_spa), IP_ADDR_LEN); 
	  eth_send(ethernet_handle, out_packet, 
		   sizeof(struct eth_hdr) + sizeof(struct arp_hdr) +
		   2*IP_ADDR_LEN + 2* ETH_ADDR_LEN); 
	}
      }
    }
  }
}

/* open_arp_pcap gets a pcap pointer for the given interface, and sets the
 * filter for "arp"
 */

pcap_t *open_arp_pcap(char *interface) {
  pcap_t *pcap_handle;
  bpf_u_int32 my_addr, my_netmask;
  struct bpf_program filter_code;
  char errbuf[PCAP_ERRBUF_SIZE];
  int ret;

  if((pcap_lookupnet(interface, &my_addr, &my_netmask, errbuf)) < 0) {
    fprintf(stderr,"pcap_lookupnet failed: %s\n", errbuf);
    return(NULL);
  }
  if((pcap_handle = pcap_open_live(interface, PACKETSIZE, 0, PACKETWAIT,
			       errbuf)) == NULL) {
    fprintf(stderr,"pcap_open_live failed:%s\n", errbuf);
    return(NULL);
  }
  if ((pcap_compile(pcap_handle, &filter_code, "arp", 1, my_netmask)) < 0) {
    fprintf(stderr, "pcap_compile failed: %s\n", errbuf);
    return(NULL);
  }
  ret = pcap_setfilter(pcap_handle, &filter_code);
  pcap_freecode(&filter_code);
  if(ret < 0) {
    fprintf(stderr, "pcap_setfilter failed: %s\n", pcap_geterr(pcap_handle));
    return(NULL);
  }

  return(pcap_handle);
}

/* This is the entry point to the arp routines, and handles the
 * initialization of the necessary pointers, registers appropriate
 * exit handlers for parent and child, sets signal handlers, and
 * the parent returns, the child calls arp_loop, which never returns
 */

/* Hey, there's no reason this shouldn't be IPV6 compatible... */
/* Ummm... Do it in a later version ^_^;; */
/* Well, and libdnet probably needs IPV6 support, too... Boy, like
 * I'm running short on things to do or something. :P
 *
 * (note to self: libdnet is merging in IPV6 support now, so get off
 * lazy ass and code).
 */

int packetp_fork_arpd(struct addr *silent_ip, arpd_end_handler callback, char *interface) {
  pid_t pid;
  pcap_t *pcap_handle;
  char errbuf[PCAP_ERRBUF_SIZE];
  char *local_intf;
  eth_t *ethernet_handle;
  eth_addr_t my_mac;

  if(interface) {
    local_intf = interface;
  } else {
    if((local_intf = pcap_lookupdev(errbuf)) == NULL) {
      fprintf(stderr,"pcap_lookupdev failed: %s\n", errbuf);
      return(-1);
    }
  }

  pid = fork();
  if (pid != 0) {
    arp_pid = pid;
    signal(SIGCHLD, callback);
    signal(SIGINT, exit);
    signal(SIGTERM, exit);
    atexit(kill_arpd);
    return(pid);
  } 

  if((pcap_handle = open_arp_pcap(local_intf)) == NULL) {
    exit(-1);
  }
  
  if ((ethernet_handle=eth_open(local_intf)) == NULL) {
    fprintf(stderr, "Unable to open ethernet interface");
    exit(-1);
  }
  
  if (eth_get(ethernet_handle, &my_mac) < 0) {
    fprintf(stderr,"Unable to obtain hardware address");
    exit(-1);
  }

  if ((arp_handle = arp_open()) == NULL) {
    fprintf(stderr, "Unable to open arp cache");
    exit(-1);
  }
  
  memcpy(&(arp_entry.arp_pa), silent_ip, sizeof(struct addr));
  memcpy(&(arp_entry.arp_ha.addr_eth), &my_mac, ETH_ADDR_LEN);
  arp_entry.arp_ha.addr_type = ADDR_TYPE_ETH;
  arp_entry.arp_ha.addr_bits = ETH_ADDR_LEN * 8;
  if ((arp_add(arp_handle, &arp_entry)) < 0) {
    fprintf(stderr, "Error adding entry to arp table");
    exit(-1);
  }

  atexit(remove_arp_entry);
  signal(SIGINT, call_exit);  /* Die calling exit() so our exit handler */
  signal(SIGTERM, call_exit); /* gets to run and clean the arp table */

  forkarp_arp_loop(pcap_handle, ethernet_handle, &my_mac, silent_ip);

  return(0); /* Never reached */
}
