const builtin = @import("builtin");
const std = @import("std");
const mem = std.mem;
const fs = std.fs;
const Target = std.build.Target;
const program_name = @import("src/docs.zig").program_name;
const sep = fs.path.sep_str;
const join = fs.path.join;

pub fn build(b: *std.Build) !void {
    const optimize = b.standardOptimizeOption(.{});
    const target = b.standardTargetOptions(.{});
    const exe = b.addExecutable(.{
        .optimize = optimize,
        .target = target,
        .name = program_name,
        .root_source_file = .{ .path = "src" ++ sep ++ "main.zig" },
        .single_threaded = true,
        .strip = b.option(
            bool,
            "strip",
            "Omit debug symbols",
        ) orelse (optimize != .Debug),
    });

    exe.pie = b.option(
        bool,
        "pie",
        "Enable Position Independent Executable (PIE)",
    ) orelse (target.result.os.tag == .openbsd);

    const build_options = b.addOptions();
    build_options.addOption(
        bool,
        "embed_wordlist",
        b.option(
            bool,
            "embed-wordlist",
            "Embed a default English wordlist",
        ) orelse true,
    );
    exe.root_module.addOptions("build-options", build_options);

    const unit_tests = b.addTest(.{
        .root_source_file = .{ .path = "src" ++ sep ++ "tests.zig" },
        .optimize = optimize,
        .target = target,
    });

    const docs = b.addExecutable(.{
        .optimize = optimize,
        .target = target,
        .name = program_name,
        .root_source_file = .{ .path = "src" ++ sep ++ "docs.zig" },
    });

    inline for (.{
        "clap",
        "ziglyph",
    }) |d| {
        const dep = b.dependency(d, .{
            .optimize = optimize,
            .target = target,
        });
        exe.root_module.addImport(d, dep.module(d));
        unit_tests.root_module.addImport(d, dep.module(d));
        docs.root_module.addImport(d, dep.module(d));
    }

    b.installArtifact(exe);

    const run_cmd = b.addRunArtifact(exe);
    run_cmd.step.dependOn(&exe.step);

    if (b.args) |args| {
        run_cmd.addArgs(args);
    }

    const run_step = b.step("run", "Run " ++ program_name);
    run_step.dependOn(&run_cmd.step);

    const run_tests = b.addRunArtifact(unit_tests);
    const test_step = b.step("test", "Run all " ++ program_name ++ " tests");
    test_step.dependOn(&run_tests.step);

    const run_docs_cmd = b.addRunArtifact(docs);
    const run_docs_step = b.step("doc", "Run documentation generator");
    run_docs_step.dependOn(&run_docs_cmd.step);

    const man_path = b.pathJoin(&.{ manDir(b), "man1", program_name ++ ".1" });

    b.installFile(program_name ++ ".1", man_path);
}

fn manDir(b: *std.Build) []const u8 {
    return b.option(
        []const u8,
        "man-dir",
        "Directory in which to install man-files, relative to prefix (default: share/man)",
    ) orelse b.pathJoin(&.{ "share", "man" });
}
