'From VisualWorks®, Release 5i.2 of December 4, 2000 on January 16, 2001 at 1:09:37 pm'!


Smalltalk defineClass: #IsdExecd
	superclass: #{Core.Object}
	indexedType: #none
	private: false
	instanceVariableNames: 'socket serverPort '
	classInstanceVariableNames: ''
	imports: ''
	category: 'Isect'!

IsdExecd comment:
'IsdExecd is a remote execution daemon for smalltalk isectd workers.  In typical isectd configurations a program called isdexecd starts child processes.  This doesn''t work for smalltalk applications.  Instead, I''m thinking a thread in the smalltalk VM will wait for messages to fork isd workers inside the smalltalk VM.

To start the daemon, execute the following code:

[IsdExecd new start] fork.

To stop the daemon, execute ''isdexecd stop'' from the command line.

Currently, the daemon only knows how to fork smalltalk children, and not regular
operating system children.'!

!IsdExecd methodsFor: 'constants'!

ISDXSERVICE
	^IPSocketAddress servicePortByName:'isdexec'.! !

!IsdExecd methodsFor: 'accessing'!

serverPort
	serverPort isNil ifTrue:[
		self serverPort: self ISDXSERVICE.
	].

	^serverPort!

serverPort: anInteger
	serverPort := anInteger.
	^self serverPort.!

socket
	socket isNil ifTrue:[
		self socket: (SocketAccessor newTCPserverAtPort:(self serverPort)).
		self socket listenFor:5.
		self socket soReuseaddr: true.
	].
	^socket!

socket: anObject
	socket := anObject! !

!IsdExecd methodsFor: 'documentation'!

readme
"
IsdExecd is a remote execution daemon for smalltalk isectd workers.  In typical isectd configurations a program called isdexecd starts child processes.  This doesn't work for smalltalk applications.  Instead, I'm thinking a thread in the smalltalk VM will wait for messages to fork isd workers inside the smalltalk VM.

To start the daemon, execute the following code:

[IsdExecd new start] fork.

To stop the daemon, execute 'isdexecd stop' from the command line.

Currently, the daemon only knows how to fork smalltalk children, and not regular
operating system children.
"! !

!IsdExecd methodsFor: 'behavior'!

getCommandLine: aSocketAccessor
"
according to the documentation for connections, closing the connection stream 
closes the accept()ed socket.
"
	|  connection stream commandLine  |

			connection := ExternalConnection ioAccessor: aSocketAccessor.
			stream := connection readAppendStream.
			commandLine := stream upToEnd.
			stream close.
			self logRequest: commandLine.
			^commandLine!

isStopCommand: aString
	^(self stopCommands includes: aString)!

logRequest: aString
			Transcript beginEntry; cr; nextPutAll: aString; cr; endEntry.!

makeArgArray: aString 
	| stream args |

	stream := ReadStream on: aString.
	args := OrderedCollection new.
	[stream atEnd] whileFalse: [args add: (stream upTo: Character space)].
	^args!

start
	^self startAtPort: self ISDXSERVICE!

startAtPort: anInteger 
	self serverPort: anInteger.
	self logRequest: 'IsdExecd starting'.
	[self workLoop: self socket]
		valueNowOrOnUnwindDo: [self socket close]!

startWorker:commandLine
"
commandLine (a String or something similar) should contain a valid request from isectd to start a worker,
and in this case, a smalltalk worker.
"
	| args isdWorkerClass isdWorker |

	args := self makeArgArray:commandLine.

	isdWorkerClass :=  Smalltalk at:(args first) ifAbsent:nil.

	isdWorkerClass isNil ifTrue:[
		self logRequest:(args first),' is not a valid Class name.'.
	] ifFalse: [
		isdWorker := isdWorkerClass new.
		(isdWorker isKindOf:IsdWorker) ifTrue:[
			[isdWorker startWith: (args copyFrom: 2 to:(args size))] fork.
		] ifFalse:[
			self logRequest:(args first), ' is not an IsdWorker subclass.'.
		].
	].!

stopCommands
	^#('halt' 'stop' 'quit' 'end')!

workLoop: aSocketAccessor 
	"
This loop basically continues until it's stopped.  On 11-3-1999 I added
code to isdexecd (the program) to send a stop command to isdexecd
if it were running.  When this smalltalk process is running instead
the stop command will kill it as well.
"

	| clientSocket commandLine |
	
	[aSocketAccessor isActive 
		ifTrue: [clientSocket := aSocketAccessor accept]
		ifFalse: [clientSocket := nil].
	clientSocket isNil] 
			whileFalse: 
				[commandLine := self getCommandLine: clientSocket.
				(self isStopCommand: commandLine) 
					ifTrue: [self socket close]
					ifFalse: [self startWorker: commandLine]].
	self logRequest: 'IsdExecd stopping.'! !


Smalltalk defineClass: #IsdIO
	superclass: #{External.ExternalInterface}
	indexedType: #none
	private: false
	instanceVariableNames: ''
	classInstanceVariableNames: ''
	imports: '
			IsdIODictionary.*
			'
	category: 'Isect'
	attributes: #(
			#(#includeFiles #())
			#(#includeDirectories #())
			#(#libraryFiles #('libisdio.so'))
			#(#libraryDirectories #('/usr/local/lib'))
			#(#beVirtual false)
			#(#optimizationLevel #debug))!

IsdIO comment:
'IsdIO is the interface to the shared library.  The shared library implements the IO interface to isectd.  Currently, the only interesting routine that remains in here is isdAttach, which connects clients and workers to isectd.

Originally, the send and recv methods were also implemented here, but I had difficulty getting the DLL/C interface to work without crashing the VM.  I tried everything, I thought.  Eventually, I decided I was more interested in getting the system working than I was in banging my head against a wall.  I removed the vestigal methods from the class to make the system easier to read through.'!

!IsdIO methodsFor: 'types'!

ByteString
	<C: typedef void * ByteString>!

int32_t
	<C: typedef void * int32_t>!

int64_t
	<C: typedef void * int64_t>!

isdBadge
	<C: typedef struct  {
			long id;
			long hostpid;
			long priority;
			long blockable;
			char service[MAXSERVICENAMELEN];
		} isdBadge>!

isdHeader
	<C: typedef struct  {
			long len;
			long sequence;
			short reply;
			short error;
			short command;
			short version;
			long workerid;
			short more;
			short unused[17];
			float aligner;
		} isdHeader>!

u_int32_t
	<C: typedef void * u_int32_t>! !

!IsdIO methodsFor: 'testing-private'!

isdInit
	<C: void isdInit(void)>
	^self externalAccessFailedWith: _errorCode!

isdTest:arg1
	<C: void isdTest(unsigned char *arg1)>
	^self externalAccessFailedWith: _errorCode! !

!IsdIO methodsFor: 'procedures'!

isdAttach: arg1 with: arg2 with:arg3 with:arg4
"
SOCKET Xport
isdAttach(const char *hostname, const char *service, char *wid, int priority)
"
	<C: int isdAttach(char *arg1, char *arg2, char *arg3, int arg4)>
	^self externalAccessFailedWith: _errorCode! !

!IsdIO methodsFor: 'defines'!

ISDARGHOST
	<C: #define ISDARGHOST 1
	>!

ISDARGMAX
	<C: #define ISDARGMAX 4
	>!

ISDARGSERVICE
	<C: #define ISDARGSERVICE 2
	>!

ISDARGWID
	<C: #define ISDARGWID 3
	>!

isdHeadlessArgvStart
	^3!

ISDHOSTNAME
	^'isectd'!

ISDSERVICE
	^'op'!

ISDXSERVICE
	<C: #define ISDXSERVICE "isdexec"
	>!

MAXSERVICENAMELEN
	<C: #define MAXSERVICENAMELEN 32>! !


Smalltalk defineClass: #IsdSmalltalkCompiler
	superclass: #{Kernel.Compiler}
	indexedType: #none
	private: false
	instanceVariableNames: ''
	classInstanceVariableNames: ''
	imports: ''
	category: 'Isect'!


Smalltalk.Core defineClass: #IsdMessage
	superclass: #{Core.Object}
	indexedType: #none
	private: false
	instanceVariableNames: 'header message '
	classInstanceVariableNames: ''
	imports: ''
	category: 'Isect'!

Core.IsdMessage comment:
'IsdMessage is what clients and workers receive from an isdRecv.  Each message includes two distinct pieces, a header and the message.  The header, unless a client/worker has something specific they need to do with it, can be left alone, but should be returned to isectd in the reply (isdSend).'!

!Core.IsdMessage methodsFor: 'accessing'!

header
	^header!

header: anIsdHeader
	^header := anIsdHeader!

message
	^message!

message: aSequenceableCollection
	^message := aSequenceableCollection! !


Smalltalk.Core defineClass: #IsdHeader
	superclass: #{Core.UninterpretedBytes}
	indexedType: #bytes
	private: false
	instanceVariableNames: ''
	classInstanceVariableNames: ''
	imports: ''
	category: 'Isect'!

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!


!Core.IsdHeader class methodsFor: 'instance creation'!

new
	^self from:(ByteArray new:64)! !

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!


!Core.IsdHeader methodsFor: 'accessing-depreciated'!

workerid
	^self unsignedLongAt:17 bigEndian:true.!

workerid:anUnsignedLong
	^self unsignedLongAt:17 put:anUnsignedLong bigEndian:true.! !

!Core.IsdHeader methodsFor: 'accessing'!

command
	^self shortAt:13 bigEndian:true.!

command:aShort
	^self shortAt:13 put:aShort bigEndian:true.!

err
	"
	this used to be error, but I got yelled at by 
	the VM for implementing an 'error:' method, 
	so I renamed it--shorter. What else? 
	"

	^self shortAt: 11 bigEndian: true!

err: aShort 
	"
	this used to be error:, but I got yelled at by 
	the VM for implementing an 'error:' method, 
	so I renamed it--shorter. What else? 
	"

	^self
		shortAt: 11
		put: aShort
		bigEndian: true!

len
	^self longAt:1 bigEndian:true.!

len: aLong
	^self longAt:1 put:aLong bigEndian:true.!

more
	^self shortAt:21 bigEndian:true.!

more: aShort 
	^self
		shortAt: 21
		put: aShort
		bigEndian: true.!

reply
	^self shortAt:9 bigEndian:true.!

reply: aShort 
	^self
		shortAt: 9
		put: aShort
		bigEndian: true!

sequence
	^self longAt:5 bigEndian:true.!

sequence: aLong
	^self longAt:5 put:aLong bigEndian:true.!

version
	^self shortAt:15 bigEndian:true.!

version:aShort
	^self shortAt:15 put:aShort bigEndian:true.! !


Smalltalk.VisualWave defineClass: #IsdClientResponder
	superclass: #{VisualWave.WebResponder}
	indexedType: #none
	private: false
	instanceVariableNames: ''
	classInstanceVariableNames: ''
	imports: ''
	category: 'Isect'!

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!


!VisualWave.IsdClientResponder class methodsFor: 'utility'!

registerForConfigurationIn: aList
	
	^aList add: self; yourself! !

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!


VisualWave.IsdClientResponder comment:
'This is a test class to see if I can get a smalltalk program running in TinyHTTPServer.  I figure if I can figure out how to get an IsdClient working in Wave then I could eventually do a lot of things.  

As of 5/ 16/2000, the responder only talk to the ''op'' service.  I send it URLs like:

http://localhost:8008/IsdClient?workers
http://localhost:8008/IsdClient?workers+busy
http://localhost:8008/IsdClient?workers%20busy
http://localhost:8008/IsdClient?client+165

URL decoding is accomplished with "URLEncoder new decode: aString."  I discovered it''s best not to use spaces in the URL.  Whatever follows the spaces doesn''t normally get to isectd.  It''s best to use %20 or + if you need to embed spaces.


'!

!VisualWave.IsdClientResponder methodsFor: 'accessing'!

defaultPath
	^'IsdClient' copy! !

!VisualWave.IsdClientResponder methodsFor: 'public interface'!

answerFor: aWebRequest 
	| isdClient isdMessage answer |
	isdClient := IsdClient new.
	isdClient isdLogin.
	answer := URLEncoder new decode: aWebRequest QUERY_STRING.

	isdClient isdSend: answer len: -1.
	isdMessage := isdClient isdRecv.
	isdClient isdLogout.
	^SimpleAnswer plainTextWith: isdMessage message asString! !


Smalltalk defineClass: #IsdClient
	superclass: #{Core.Object}
	indexedType: #none
	private: false
	instanceVariableNames: 'isdLib isdSocket isdHeader isdBuffer isdSocketAccessor isdSocketAddress '
	classInstanceVariableNames: ''
	imports: ''
	category: 'Isect'!

IsdClient comment:
'IsdClient is an abstract class which supplies much of the protocol necessary for implementing isectd clients and servers.

Though this distribution doesn''t include any client classes, all the worker classes are subclasses of IsdClient.  The major difference between client and workers is client use isdLogin to connect to isectd as clients, and workers use connect: (ultimately isdAttach) to connect to isectd as workers.  Workers reply on a special port to isectd with a worker ID they''re started with, and clients connect on the well known port, advertised in /etc/services, called isectd.

I''m thinking the routines I tested in this class for sending and receiving messages w/ isectd should be moved to IsdIO, even though they really aren''t interfacing with the shared library.  The class IsdIO represents the IO, or interface classes, and it makes sense these should be collected there and not sprinkled throught the package.

'!

!IsdClient methodsFor: 'behavior'!

isdAttach: hostname service: aString workerID: aStringWID priority: anInteger 
	| osHandle |
	osHandle := self isdLib
				isdAttach: hostname
				with: aString
				with: aStringWID
				with: anInteger.
	self isdSocketAccessor: (SocketAccessor new handleValue: osHandle).
	^self isdSocketAddress!

isdLogin
	^self isdLogin:nil.!

isdLogin:hostname
	^self isdLogin:hostname service:(self isdLib ISDSERVICE).!

isdLogin:hostname service:aString 
	^self isdLogin:hostname service:aString priority: 2.!

isdLogin: hostname service: aString priority: anInteger 
	^self
		isdAttach: hostname
		service: aString
		workerID: nil
		priority: anInteger!

isdLogout
	^self isdSocketAccessor close.!

isdRecv
	| byteCount header headerBytes messageBytes message |
	header := IsdHeader new.
	headerBytes := ByteArray new: header sizeInBytes.
	self isdSocketAccessor readWait.
	byteCount := self isdSocketAccessor receiveFrom: self isdSocketAddress buffer: headerBytes.
	byteCount == header sizeInBytes
		ifTrue: 
			[header
				replaceBytesFrom: 1
				to: header sizeInBytes
				with: headerBytes
				startingAt: 1.
			messageBytes := ByteArray new: header len.
			byteCount := self isdSocketAccessor receiveFrom: self isdSocketAddress buffer: messageBytes.
			message := IsdMessage new.
			message header: header.
			byteCount == header len
				ifTrue: 
					[message message: messageBytes.
					^message]].
	^nil!

isdSend: aByteString
	| header |

	header := IsdHeader new.

	header len: aByteString size.

	self isdSocketAccessor sendTo:(self isdSocketAddress) buffer:(header asByteArray).
	self isdSocketAccessor sendTo:(self isdSocketAddress) buffer:aByteString.


	^aByteString size.!

isdSend: aByteString header:anIsdHeader

	anIsdHeader len: aByteString size.

	self isdSocketAccessor sendTo:(self isdSocketAddress) buffer:(anIsdHeader asByteArray).
	self isdSocketAccessor sendTo:(self isdSocketAddress) buffer:aByteString.

	^aByteString size.!

isdSend: aByteString len: anInteger 
"
Yes, I know I can get the size of the buffer by using the size message, but sometimes I know I'm sending a string that needs to be null terminated, and Smalltalk doesn't generally null-terminate string unless it's crossing over into a C function.  Sine the C API includes a len: parameter with -1 I thought it would solve a problem of mine (in IsdClientResponder) of having to make sure the URL was nul terminated.
"
	| header aString |
	header := IsdHeader new.
	anInteger < 0
		ifTrue: 
			[aString := ByteString fromBytes: (aByteString copyWith: (Character value: 0)).
			header len: aString size]
		ifFalse: 
			[aString := aByteString.
			header len: anInteger].
	self isdSocketAccessor sendTo: self isdSocketAddress buffer: header asByteArray.
	self isdSocketAccessor sendTo: self isdSocketAddress buffer: aString.
	^aByteString size! !

!IsdClient methodsFor: 'initialization'!

finish
	ObjectMemory quit.!

start
	^self subclassResponsibility!

workLoop
	^self subclassResponsibility! !

!IsdClient methodsFor: 'accessing'!

isdLib
	isdLib isNil ifTrue:[
		self isdLib:(IsdIO new)].

	^isdLib.!

isdLib: anIsdIO
	isdLib := anIsdIO.
	^isdLib.!

isdSocketAccessor
	^isdSocketAccessor.!

isdSocketAccessor: aSocketAccessor
	isdSocketAccessor := aSocketAccessor.
	^isdSocketAccessor.!

isdSocketAddress
	isdSocketAddress isNil ifTrue: [
		self isdSocketAddress: self isdSocketAccessor getPeer.
	].
	^isdSocketAddress!

isdSocketAddress: aSocketAddress
	^isdSocketAddress := aSocketAddress! !


Smalltalk defineClass: #IsdWorker
	superclass: #{Smalltalk.IsdClient}
	indexedType: #none
	private: false
	instanceVariableNames: ''
	classInstanceVariableNames: ''
	imports: ''
	category: 'Isect'!

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!


!IsdWorker class methodsFor: 'initialization'!

initialize
	ObjectMemory removeDependent: self.
	ObjectMemory addDependent: self.!

update: anAspect with: arguments from: anObject
"
	(anObject == ObjectMemory and: [anAspect == #earlySystemInstallation.]) ifTrue: [
		[IsdEcho new start] forkAt: ProcessorUserSchedulingPriority].
"! !

!IsdWorker class methodsFor: 'deployment'!

saveHeadlessImage
	HeadlessImage default transcriptFilename: ('/tmp/', self name, '.tr').
	HeadlessImage default startupFilename: nil.
	^HeadlessImage default saveHeadless: (self name)! !

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!


IsdWorker comment:
'IsdWorker is the abstract superclass of all isectd-compatible workers.  It is a subclass of IsdClient.

To be started by isdexecd (the smalltalk version of the remote execution daemon) a class must be a subclass of IsdWorker.  This is how the library knows it can correctly start up the worker classes.

Classes subclassed from IsdWorker, like IsdEcho, need to implement at least on method, called workLoop, which is invoked by isdExecd when the class starts.  IsdEcho loops on an isdRecv, and when it receives a message calls work:.  If this philosophy works for you, as it does for IsdEat, the only thing you need to implement, as IsdEat does, is work:.

work: receives an IsdMessage, which includes the header and message.'!

!IsdWorker methodsFor: 'initialization'!

init
"
For subclasses of IsdWorker, override this message with
any initialization required before connecting to to isectd.

Return True if everything's cool, or False if things are
not cool.
"
	^self subclassResponsibility!

start
" Whether this works or not, we're going to terminate the image using 'self finish'"
	| argv isdArgv |

	HeadlessImage default isHeadless ifTrue: [
		argv := CEnvironment commandLine.
		isdArgv := argv copyFrom: (self isdLib isdHeadlessArgvStart) to: (argv size).
		Transcript cr; nextPutAll: 'starting worker'; cr; flush.

		(isdArgv size >= 4) ifTrue: [
			[ self startWith: isdArgv ] valueNowOrOnUnwindDo: [ self finish ].
		].
	] ifFalse: [
		Transcript cr; nextPutAll: 'not running a headless image'; cr; flush.
	].!

startWith:anArray
	| socketAddress  |
		(self init) ifTrue: [	
			socketAddress := self connect:anArray.
			(socketAddress isNil) ifFalse: [
				Transcript cr; nextPutAll: 'socketAddress = ', socketAddress printString; cr; flush.
				[ self workLoop ] valueNowOrOnUnwindDo: [ self isdLogout ].
			]
		]!

workLoop
	^self subclassResponsibility! !

!IsdWorker methodsFor: 'behavior'!

connect
	| argv isdArgv |

	argv := CEnvironment commandLine.
	self halt.
	isdArgv := argv copyFrom: (self isdLib isdHeadlessArgvStart) to: (argv size).
	^self connect:isdArgv!

connect:anArray

	(anArray size >= (self isdLib ISDARGMAX)) ifTrue: [
		^self isdAttach: (anArray at: 1) service: (anArray at: 2) workerID: (anArray at: 3) priority: 0.
	].

	^nil!

isdLogout
	super isdLogout.
	Transcript nextPutAll: 'worker logged out'; cr; flush.
	^nil! !

!IsdWorker methodsFor: 'documentation'!

headlessImages
"
	For any subclass of IsdWorker, send the class the message 'saveHeadlessImage'
	and it will create an image named after the class, wherever images are being saved.

	The following command, when executed in a workspace, creates ~/IsdWorker.im..
		IsdWorker saveHeadlessImage.

	.. and this command will save a headless image named ~/IsdEcho.im..
		IsdEcho saveHeadlessImage.

	Because the method is implemented in IsdWorker, which should be the parent of 
	most (if not all) workers, this will work.  The only thing it doesn't do is
	modify the code for HeadlessImage>>#returnFromSnapshot to fork-off the proper
	worker class.

	In my system at home, I use the following:
		[IsdEcho new start] fork.
	just before returnFromSnapshot starts playing with the Transcript.
	Come to think of it, I should probably move it to the bottom.

	To start a headlessImage from isectd, I created the following scripts:
		#!!/bin/sh
		VISUALWORKS=/share/local/vwnc31;export VISUALWORKS
		PATH=$PATH:$VISUALWORKS/bin;export PATH
		$VISUALWORKS/bin/visualnc IsdEcho.im $*
	Which works pretty well, except that when the script is run under
	isdexecd I can't gurantee isdEcho.im will be in the working
	directory.  You should edit the script however necessary
	for your situation so that a complete path is provided for
	the image.
"! !


Smalltalk defineClass: #IsdEcho
	superclass: #{Smalltalk.IsdWorker}
	indexedType: #none
	private: false
	instanceVariableNames: ''
	classInstanceVariableNames: ''
	imports: ''
	category: 'Isect'!

IsdEcho comment:
'IsdEcho is a subclassof IsdWorker.  The only thing it does is echo back to the client whatever it receives.  Pretty simple.  Subclasses that do something different, have only to override the ''work:'' method, and do whatever it is that''s unique to what that worker is intended to do.'!

!IsdEcho methodsFor: 'behavior'!

init
	^true!

work: anIsdMessage

	self isdSend:(anIsdMessage message) header: (anIsdMessage header).!

workLoop
	| message |
	[(message := self isdRecv) isNil] whileFalse: [
		self work:message.
	].! !


Smalltalk defineClass: #IsdWorkerReplyStreamAbstract
	superclass: #{Smalltalk.IsdEcho}
	indexedType: #none
	private: false
	instanceVariableNames: 'replyStream '
	classInstanceVariableNames: ''
	imports: ''
	category: 'Isect'!

!IsdWorkerReplyStreamAbstract methodsFor: 'documentation'!

readme
"
<Note>
After replying back to the client, if the worker is FINISHED
with that work unit it should rewind the replyStream using something
like:

	self replyStream reset.

</Note>

WorkerReplyStreamAbstract was created shortly after starting my
XMLTest worker.  Because I wanted a stream to write my replies
on I initially subclassed IsdEat, but later decided there was a lot
of protocol in IsdEat that wasn't necessary in IsdXMLTest and
would lead to confusion over the reasons XMLTest was subclassed
from Eat.

So instead of wonder about it, I decided to create an abstract class
with the replyStream instance variable and allow IsdEat and 
IsdXMLTest to both subclass from it.
"! !

!IsdWorkerReplyStreamAbstract methodsFor: 'accessing'!

replyStream
	replyStream isNil ifTrue: [
		replyStream := WriteStream on: (String new: 128).
	].

	^replyStream! !


Smalltalk defineClass: #IsdEat
	superclass: #{Smalltalk.IsdWorkerReplyStreamAbstract}
	indexedType: #none
	private: false
	instanceVariableNames: 'allPacks yourPacks allBytes yourBytes '
	classInstanceVariableNames: ''
	imports: ''
	category: 'Isect'!

IsdEat comment:
'This is an example of a subclass off IseEcho.  What it attempts to do, but is incomplete at doing, is count the number of bytes it receives.  One of the things that makes it different is it watches the ''more'' variable to make sure it doesn''t report back prematurely--before the client is done sending its entire message.'!

!IsdEat methodsFor: 'behavior'!

init
	self allBytes: 0.
	self allPacks: 0.

	self yourInit.

	^super init!

work: anIsdMessage
	| messageLength more |

	messageLength := anIsdMessage header len.
	more := anIsdMessage header more.

	self allPacks: (self allPacks) + 1.
	self allBytes: (self allBytes) + messageLength.
	self yourPacks: (self yourPacks) + 1.
	self yourBytes: (self yourBytes) + messageLength.

	(more == 0) ifTrue: [
		self replyStream nextPutAll:' AllPacks   AllBytes  YourPacks  YourBytes'; nextPut: Character lf;
                      nextPutAll:'---------  ---------  ---------  ---------'; nextPut: Character lf.
		anIsdMessage header len: (self replyStream contents size).
		self isdSend: (self replyStream contents) header:anIsdMessage header.
		self yourInit.
	]!

yourInit
	self yourBytes: 0.
	self yourPacks: 0.

	self replyStream reset.! !

!IsdEat methodsFor: 'accessing'!

allBytes
	^allBytes!

allBytes: anInteger
	allBytes := anInteger!

allPacks
	^allPacks!

allPacks: anInteger
	allPacks := anInteger!

replyStream
	replyStream isNil ifTrue: [
		replyStream := WriteStream on: (String new).
	].

	^replyStream!

yourBytes
	^yourBytes!

yourBytes: anInteger
	yourBytes := anInteger!

yourPacks
	^yourPacks!

yourPacks: anInteger
	yourPacks := anInteger! !


Smalltalk defineClass: #IsdSmalltalkWorker
	superclass: #{Smalltalk.IsdWorkerReplyStreamAbstract}
	indexedType: #none
	private: false
	instanceVariableNames: 'compiler '
	classInstanceVariableNames: ''
	imports: ''
	category: 'Isect'!

!IsdSmalltalkWorker methodsFor: 'behavior'!

work: anIsdMessage 
	| result |

"subclasses of IsdWorkerReplyStreamAbstract should rewind
the replyStream before writing fresh information to it."
	self replyStream reset.

	[result := IsdSmalltalkCompiler evaluate: (anIsdMessage message copyWithout:0) asString] on: Error 
		do: [:ex | 
			self replyStream
				nextPutAll: ex signal description; crlf; 
				nextPutAll: ex messageText; crlf.].

	result printOn: self replyStream.
	self replyStream crlf; nextPutAll:'Done.'; crlf.

	anIsdMessage header len: self replyStream contents size.
	self isdSend: self replyStream contents header: anIsdMessage header.! !


#{IsdWorker} initialize!

