//==============================================================================================
//  geometry.h                                                                     Font3D
//----------------------------------------------------------------------------------------------
//
//  Copyright (c) 1994-1996 by Todd A. Prater                                   Version 1.60
//  All rights reserved.
//
//----------------------------------------------------------------------------------------------
//
//  Permission to copy and distribute Font3D in its entirety, for noncommercial purposes,
//  is hereby granted without fee, provided that this license information and copyright 
//  notice appear in all copies. 
//
//  If you redistribute Font3D, the entire contents of this distribution must be distributed,
//  including the readme.txt, and register.txt files, and the complete set of documentation,
//  both ASCII text, and PostScript files. 
//
//  The software may be modified for your own purposes, but modified versions may not be
//  distributed without prior consent of the author.
//
//  This software is provided 'asis', without any express or implied warranty.  In no event
//  will the author be held liable for any damages arising from the use of this software.  
//
//  If you would like to do something with Font3D that this copyright prohibits (such as 
//  distributing it with a commercial product, using portions of the source in some other
//  program, distributing registered copies, etc.), please contact the author (preferably
//  via email).  Arrangements can probably be worked out.
//
//==============================================================================================

#ifndef __GEOMETRY_H__
#define __GEOMETRY_H__

#include <iostream.h>

#include "vector.h"

#define  TRUE          1
#define  FALSE         0

#define  CLOCKWISE     1
#define  ANTICLOCKWISE 2

#define BIG              1.0e30
#define MIN_SHRINK_ANGLE   0.15

#define GEOM_ERR_NoError          0
#define GEOM_ERR_NoPolyFound      1


//----------------------------------------------------------------------------------------------
//  TRIANGLE
//----------------------------------------------------------------------------------------------

   class TRIANGLE
   {
      public:  vector v1,v2,v3;
               vector n1,n2,n3;

               TRIANGLE(void) {}
               TRIANGLE(vector _v1, vector _v2, vector _v3)
               {
                  v1=_v1; v2=_v2; v3=_v3;
                  n1=vector(0,0,1);
                  n2=vector(0,0,1); 
                  n3=vector(0,0,1);
               }

               TRIANGLE(vector _v1, vector _v2, vector _v3,
                        vector _n1, vector _n2, vector _n3)
               {
	          v1=_v1; v2=_v2; v3=_v3;
                  n1=_n1; n2=_n2; n3=_n3;
               }

               TRIANGLE(const TRIANGLE& t)
               {
                  v1=t.v1; v2=t.v2; v3=t.v3;
                  n1=t.n1; n2=t.n2; n3=t.n3;
               }

               vector Normal(void) 
               { 
                  return ~((v3-v1)^(v2-v1)); 
               }

               void Output(ostream& s, int format);

               int Orientation(void)
               {
                  if (((v2.x-v1.x)*(v3.y-v1.y)
                      -(v3.x-v1.x)*(v2.y-v1.y)) >= 0.0)
                     return ANTICLOCKWISE;
                  else
                     return CLOCKWISE;
               }
   };

   ostream& operator << (ostream& s, const TRIANGLE& t);


//----------------------------------------------------------------------------------------------
//  TRIANGLELISTLINK
//----------------------------------------------------------------------------------------------

   class TRIANGLELISTLINK
   {
      private: TRIANGLE*         obj;
               TRIANGLELISTLINK* next;

      public:  TRIANGLELISTLINK (void)
               {
                  obj = NULL;
                  next = NULL;
               }

               TRIANGLELISTLINK (TRIANGLE* object, TRIANGLELISTLINK* tonext)
               {
                  obj  = object;
                  next = tonext;
               }

               TRIANGLE* Obj(void) 
               { 
                  return obj; 
               };

               TRIANGLELISTLINK* Next(void)
               { 
                  return next; 
               };

               void setObj(TRIANGLE* object) 
               {
                  obj = object; 
               };

               void setNext(TRIANGLELISTLINK* n)  
               { 
                  next = n; 
               };

   };


//----------------------------------------------------------------------------------------------
//  TRIANGLELIST
//----------------------------------------------------------------------------------------------

   class TRIANGLELIST
   {
      private: int               count;
               TRIANGLELISTLINK* first;
               TRIANGLELISTLINK* current;
               TRIANGLELISTLINK* last;

      public:  TRIANGLELIST(void)
               {
                  count = 0;
                  first = NULL;
                  current = NULL;
                  last = NULL;
               }

               void Empty(void);
               int Add(TRIANGLE* t);

               int Count(void)  
               { 
                  return count; 
               };

               TRIANGLE* First(void)  
               { 
                  return first->Obj();
               };

               TRIANGLE* Current(void)  
               { 
                  return current->Obj();
               };

               TRIANGLE* Last(void)  
               { 
                  return last->Obj();
               };

               TRIANGLE* Next(void)
               { 
                  return current->Next()->Obj();
               };

               void gotoFirst(void)
               { 
                  current = first;
               };

               void gotoLast(void)
               { 
                  current = last;
               };

               void gotoNext(void)  
               {
                  if (current!=last)
                     current = current->Next();
               };

   };


//----------------------------------------------------------------------------------------------
//  POLYGON
//----------------------------------------------------------------------------------------------

   class POLYGON
   {
       public:
           int      numpoints;
           vector*  pointlist;
           int      orientation;

           int      findOrientation(void);
           int      findDeterminant(int p1, int p2, int p3);
           int      noneInside(int p1, int p2, int p3, int n, int* vlist);



           POLYGON(void);
           POLYGON(int n, vector* p);
           POLYGON(const POLYGON& P);

           int NumPoints(void) { return numpoints; }
           void Correct(void);
           int Triangulate(TRIANGLELIST& trianglelist, int verbose);
           int isInside(POLYGON& p);
           void Combine(POLYGON& p);
           void Shrink(POLYGON& p, double shrinkFactor);
           void Shrink(double shrinkFactor);
           void SetDepth(double d);
           void Translate(vector offset);

   };

   ostream& operator << (ostream& s, const POLYGON& p);


   vector ApproximateQuadraticSpline(vector cp1, vector cp2, vector cp3, double t);


#endif
