/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : DPSample.inl
|
| Revision    : 1.0
| Date        : 04/03/96
|
| Object      : DPSample
|
| Description : DPSample inline methods for handling audio samples
|
| (c) Richard Kent 1996
|
| $Id: DPSampleInline.h,v 1.1 2003/09/10 00:06:25 rk Exp $
|
****************************************************************************/

/*---------------------------------------------------------------------------
| Modifiers
---------------------------------------------------------------------------*/

void DPSample::setChanged (int newChanged)
{
//cerr << "DPSample::setChanged (" << newChanged<< ")" << endl;
  changed = newChanged;
}

void DPSample::setFormat (int newFormat)
{
//cerr << "DPSample::setFormat (" << newFormat << ")" << endl;
  if (format != newFormat) changed = TRUE;
  format = newFormat;
}

void DPSample::setCompression (int newCompression)
{
//cerr << "DPSample::setCompression (" << newCompression << ")" << endl;
  if (compression != newCompression) changed = TRUE;
  compression = newCompression;
}

void DPSample::setRate (double newRate)
{
//cerr << "DPSample::setRate (" << newRate << ")" << endl;
  if (rate != newRate) changed = TRUE;
  rate = newRate;
}

void DPSample::setFilename (char *newFilename)
{
//cerr << "DPSample::setFilename (" << newFilename << ")" << endl;
  if (strcmp (filename,newFilename)) changed = TRUE;
  delete [] filename;
  filename = new char [strlen (newFilename) + 1];
  strcpy (filename,newFilename);
}

void DPSample::setName (char *newName)
{
//cerr << "DPSample::setName (" << newName << ")" << endl;
  if (!newName) newName = "";
  if (strcmp (name,newName)) changed = TRUE;
  delete [] name;
  name = new char [strlen (newName) + 1];
  strcpy (name,newName);
}

void DPSample::setCopyright (char *newCopyright)
{
//cerr << "DPSample::setCopyright (" << newCopyright << ")" << endl;
  if (!newCopyright) newCopyright = "";
  if (strcmp (copyright,newCopyright)) changed = TRUE;
  delete [] copyright;
  copyright = new char [strlen (newCopyright) + 1];
  strcpy (copyright,newCopyright);
}

void DPSample::setAuthor (char *newAuthor)
{
//cerr << "DPSample::setAuthor (" << newAuthor << ")" << endl;
  if (!newAuthor) newAuthor = "";
  if (strcmp (author,newAuthor)) changed = TRUE;
  delete [] author;
  author = new char [strlen (newAuthor) + 1];
  strcpy (author,newAuthor);
}

void DPSample::setAnnotation (char *newAnnotation)
{
//cerr << "DPSample::setAnnotation (" << newAnnotation << ")" << endl;
  if (!newAnnotation) newAnnotation = "";
  if (strcmp (annotation,newAnnotation)) changed = TRUE;
  delete [] annotation;
  annotation = new char [strlen (newAnnotation) + 1];
  strcpy (annotation,newAnnotation);
}

void DPSample::setSusLoopMode (int newSusLoopMode)
{
//cerr << "DPSample::setSusLoopMode (" << newSusLoopMode << ")" << endl;
  if (susLoopMode != newSusLoopMode) changed = TRUE;
  susLoopMode = newSusLoopMode;
}

void DPSample::setSusLoopStart (int newSusLoopStart)
{
//cerr << "DPSample::setSusLoopStart (" << newSusLoopStart << ") ";
  if (susLoopStart != newSusLoopStart) changed = TRUE;
  if (newSusLoopStart < 0)
    susLoopStart = 0;
  else if (newSusLoopStart > frames)
    susLoopStart = frames;
  else
    susLoopStart = newSusLoopStart;
  if (susLoopStart > susLoopEnd)
    susLoopStart = susLoopEnd;
//cerr << "[" << susLoopStart << "," << susLoopEnd << "]" << endl;
}

void DPSample::setSusLoopEnd (int newSusLoopEnd)
{
//cerr << "DPSample::setSusLoopEnd (" << newSusLoopEnd << ") ";
  if (susLoopEnd != newSusLoopEnd) changed = TRUE;
  if (newSusLoopEnd < 0)
    susLoopEnd = 0;
  else if (newSusLoopEnd > frames)
    susLoopEnd = frames;
  else
    susLoopEnd = newSusLoopEnd;
  if (susLoopEnd < susLoopStart)
    susLoopEnd = susLoopStart;
//cerr << "[" << susLoopStart << "," << susLoopEnd << "]" << endl;
}

void DPSample::setSusLoop (int newSusLoopStart,int newSusLoopEnd)
{
//cerr << "DPSample::setSusLoop ";
  if (susLoopStart != newSusLoopStart) changed = TRUE;
  else if (susLoopEnd != newSusLoopEnd) changed = TRUE;
  if (newSusLoopStart < 0)
    susLoopStart = 0;
  else if (newSusLoopStart > frames)
    susLoopStart = frames;
  else
    susLoopStart = newSusLoopStart;
  if (newSusLoopEnd < 0)
    susLoopEnd = 0;
  else if (newSusLoopEnd > frames)
    susLoopEnd = frames;
  else
    susLoopEnd = newSusLoopEnd;
  if (susLoopEnd < susLoopStart)
    susLoopEnd = susLoopStart;
//cerr << "[" << susLoopStart << "," << susLoopEnd << "]" << endl;
}

void DPSample::setRelLoopMode (int newRelLoopMode)
{
//cerr << "DPSample::setRelLoopMode (" << newRelLoopMode << ")" << endl;
  if (relLoopMode != newRelLoopMode) changed = TRUE;
  relLoopMode = newRelLoopMode;
}

void DPSample::setRelLoopStart (int newRelLoopStart)
{
//cerr << "DPSample::setRelLoopStart (" << newRelLoopStart << ") ";
  if (relLoopStart != newRelLoopStart) changed = TRUE;
  if (newRelLoopStart < 0)
    relLoopStart = 0;
  else if (newRelLoopStart > frames)
    relLoopStart = frames;
  else
    relLoopStart = newRelLoopStart;
  if (relLoopStart > relLoopEnd)
    relLoopStart = relLoopEnd;
//cerr << "[" << relLoopStart << "," << relLoopEnd << "]" << endl;
}

void DPSample::setRelLoopEnd (int newRelLoopEnd)
{
//cerr << "DPSample::setRelLoopEnd (" << newRelLoopEnd << ") ";
  if (relLoopEnd != newRelLoopEnd) changed = TRUE;
  if (newRelLoopEnd < 0)
    relLoopEnd = 0;
  else if (newRelLoopEnd > frames)
    relLoopEnd = frames;
  else
    relLoopEnd = newRelLoopEnd;
  if (relLoopEnd < relLoopStart)
    relLoopEnd = relLoopStart;
//cerr << "[" << relLoopStart << "," << relLoopEnd << "]" << endl;
}

void DPSample::setRelLoop (int newRelLoopStart,int newRelLoopEnd)
{
//cerr << "DPSample::setRelLoop ";
  if (relLoopStart != newRelLoopStart) changed = TRUE;
  else if (relLoopEnd != newRelLoopEnd) changed = TRUE;
  if (newRelLoopStart < 0)
    relLoopStart = 0;
  else if (newRelLoopStart > frames)
    relLoopStart = frames;
  else
    relLoopStart = newRelLoopStart;
  if (newRelLoopEnd < 0)
    relLoopEnd = 0;
  else if (newRelLoopEnd > frames)
    relLoopEnd = frames;
  else
    relLoopEnd = newRelLoopEnd;
  if (relLoopEnd < relLoopStart)
    relLoopEnd = relLoopStart;
//cerr << "[" << relLoopStart << "," << relLoopEnd << "]" << endl;
}

void DPSample::setRangeValid (int newRangeValid)
{
//cerr << "DPSample::setRangeValid (" << newRangeValid << ") ";
  rangeValid = newRangeValid;
}

void DPSample::setRangeStart (int newRangeStart)
{
//cerr << "DPSample::setRangeStart (" << newRangeStart << ") ";
  rangeValid = TRUE;
  if (newRangeStart < 0)
    rangeStart = 0;
  else if (newRangeStart > frames)
    rangeStart = frames;
  else
    rangeStart = newRangeStart;
  if (rangeStart > rangeEnd)
  {
    int temp  = rangeEnd;
    rangeEnd   = rangeStart;
    rangeStart = temp;
  }
//cerr << "[" << rangeStart << "," << rangeEnd << "]" << endl;
}

void DPSample::setRangeEnd (int newRangeEnd)
{
//cerr << "DPSample::setRangeEnd (" << newRangeEnd << ") ";
  rangeValid = TRUE;
  if (newRangeEnd < 0)
    rangeEnd = 0;
  else if (newRangeEnd > frames)
    rangeEnd = frames;
  else
    rangeEnd = newRangeEnd;
  if (rangeEnd < rangeStart)
  {
    int temp  = rangeEnd;
    rangeEnd   = rangeStart;
    rangeStart = temp;
  }
//cerr << "[" << rangeStart << "," << rangeEnd << "]" << endl;
}

void DPSample::setRange (int newRangeStart,int newRangeEnd)
{
//cerr << "DPSample::setRange ";
  rangeValid = TRUE;
  if (newRangeStart < 0)
    rangeStart = 0;
  else if (newRangeStart > frames)
    rangeStart = frames;
  else
    rangeStart = newRangeStart;
  if (newRangeEnd < 0)
    rangeEnd = 0;
  else if (newRangeEnd > frames)
    rangeEnd = frames;
  else
    rangeEnd = newRangeEnd;
  if (rangeEnd < rangeStart)
  {
    int temp  = rangeEnd;
    rangeEnd   = rangeStart;
    rangeStart = temp;
  }
//cerr << "[" << rangeStart << "," << rangeEnd << "]" << endl;
}

void DPSample::setDisplayStart (int newDisplayStart)
{
//cerr << "DPSample::setDisplayStart (" << newDisplayStart << ") ";
  if (newDisplayStart < 0)
    displayStart = 0;
  else if (newDisplayStart > frames)
    displayStart = frames;
  else
    displayStart = newDisplayStart;
  if (displayStart > displayEnd)
    displayEnd = displayStart;
//cerr << "[" << displayStart << "," << displayEnd << "]" << endl;
}

void DPSample::setDisplayEnd (int newDisplayEnd)
{
//cerr << "DPSample::setDisplayEnd (" << newDisplayEnd << ")" << endl;
  if (newDisplayEnd < 0)
    displayEnd = 0;
  else if (newDisplayEnd > frames)
    displayEnd = frames;
  else
    displayEnd = newDisplayEnd;
  if (displayEnd < displayStart)
    displayStart = displayEnd;
//cerr << "[" << displayStart << "," << displayEnd << "]" << endl;
}

void DPSample::setDisplay (int newDisplayStart,int newDisplayEnd)
{
//cerr << "DPSample::setDisplay ";
  if (newDisplayStart < 0)
    displayStart = 0;
  else if (newDisplayStart > frames)
    displayStart = frames;
  else
    displayStart = newDisplayStart;
  if (newDisplayEnd < 0)
    displayEnd = 0;
  else if (newDisplayEnd > frames)
    displayEnd = frames;
  else
    displayEnd = newDisplayEnd;
  if (displayStart > displayEnd)
    displayEnd = displayStart;
//cerr << "[" << displayStart << "," << displayEnd << "]" << endl;
}

void DPSample::setDisplayVertLow (int newDisplayVertLow)
{
//cerr << "DPSample::setDisplayVertLow (" << newDisplayVertLow << ") ";
  if (newDisplayVertLow < -MAX23)
    displayVertLow = -MAX23;
  else if (newDisplayVertLow > MAX23_1)
    displayVertLow = MAX23_1;
  else
    displayVertLow = newDisplayVertLow;
  if (displayVertLow > displayVertHigh)
    displayVertHigh = displayVertLow;
//cerr << "[" << displayVertLow << "," << displayVertHigh << "]" << endl;
}

void DPSample::setDisplayVertHigh (int newDisplayVertHigh)
{
//cerr << "DPSample::setDisplayVertHigh (" << newDisplayVertHigh << ")" << endl;
  if (newDisplayVertHigh < -MAX23)
    displayVertHigh = -MAX23;
  else if (newDisplayVertHigh > MAX23_1)
    displayVertHigh = MAX23_1;
  else
    displayVertHigh = newDisplayVertHigh;
  if (displayVertHigh < displayVertLow)
    displayVertLow = displayVertHigh;
//cerr << "[" << displayVertLow << "," << displayVertHigh << "]" << endl;
}

void DPSample::setDisplayVert (
  int newDisplayVertLow,
  int newDisplayVertHigh)
{
//cerr << "DPSample::setDisplayVert ";
  if (newDisplayVertLow < -MAX23)
    displayVertLow = -MAX23;
  else if (newDisplayVertLow > MAX23_1)
    displayVertLow = MAX23_1;
  else
    displayVertLow = newDisplayVertLow;
  if (newDisplayVertHigh < -MAX23)
    displayVertHigh = -MAX23;
  else if (newDisplayVertHigh > MAX23_1)
    displayVertHigh = MAX23_1;
  else
    displayVertHigh = newDisplayVertHigh;
  if (displayVertLow > displayVertHigh)
    displayVertHigh = displayVertLow;
//cerr << "[" << displayVertLow << "," << displayVertHigh << "]" << endl;
}

void DPSample::setRedraw (int newRedraw)
{
//cerr << "DPSample::setRedraw (" << newRedraw << ")" << endl;
  redraw = newRedraw;
}

void DPSample::setCurrentPos (int newCurrentPos)
{
//cerr << "DPSample::setCurrentPos (" << newCurrentPos << ")" << endl;
  if (newCurrentPos < 0 || newCurrentPos > frames)
    currentPos = -1;
  else
    currentPos = newCurrentPos;
}

/*---------------------------------------------------------------------------
| Accessors
---------------------------------------------------------------------------*/

int   DPSample::getValid () const           { return valid;           }
int   DPSample::getChanged () const         { return changed;         }
int   DPSample::getFormat () const          { return format;          }
int   DPSample::getCompression () const     { return compression;     }
double DPSample::getRate () const            { return rate;            }
int   DPSample::getWidth () const           { return width;           }
int   DPSample::getChannels () const        { return channels;        }
int   DPSample::getFrames () const          { return frames;          }
int   DPSample::getSamples () const         { return samples;         }
int   DPSample::getBytes () const           { return bytes;           }
int   DPSample::getAESused () const         { return AESused;         }
unsigned char *DPSample::getAES ()           { return AES;             }
char   *DPSample::getFilename () const       { return filename;        }
char   *DPSample::getName () const           { return name;            }
char   *DPSample::getCopyright () const      { return copyright;       }
char   *DPSample::getAuthor () const         { return author;          }
char   *DPSample::getAnnotation () const     { return annotation;      }
int   DPSample::getSusLoopMode () const     { return susLoopMode;     }
int   DPSample::getSusLoopStart () const    { return susLoopStart;    }
int   DPSample::getSusLoopEnd () const      { return susLoopEnd;      }
int   DPSample::getRelLoopMode () const     { return relLoopMode;     }
int   DPSample::getRelLoopStart () const    { return relLoopStart;    }
int   DPSample::getRelLoopEnd () const      { return relLoopEnd;      }
int   DPSample::getRangeValid () const      { return rangeValid;      }
int   DPSample::getRangeStart () const      { return rangeStart;      }
int   DPSample::getRangeEnd () const        { return rangeEnd;        }
int   DPSample::getDisplayStart () const    { return displayStart;    }
int   DPSample::getDisplayEnd () const      { return displayEnd;      }
int   DPSample::getDisplayVertLow () const  { return displayVertLow;  }
int   DPSample::getDisplayVertHigh () const { return displayVertHigh; }
int   DPSample::getRedraw () const          { return redraw;          }
int   DPSample::getCurrentPos () const      { return currentPos;      }

int DPSample::getLzero (int frame)
{
  int i=frame;
  int found=FALSE;
  int sign1;
  int sign2;
  int sign3;
  int sign4;
  if (frame == 0) return 0;
  if (channels == 1)
  {
    --i;
    sign1 = sign (getFrame24 (i,0));
    while (i > 0 && !found)
    {
      --i;
      if (sign (getFrame24 (i,0)) != sign1) found = TRUE;
    }
    if (found) return i + 1;
    if (sign (getFrame24 (0,0)) == 0) return 0;
    return frame;
  }
  else if (channels == 2)
  {
    --i;
    sign1 = sign (getFrame24 (i,0));
    sign2 = sign (getFrame24 (i,1));
    while (i > 0 && !found)
    {
      --i;
      if (sign (getFrame24 (i,0)) != sign1) found = TRUE;
      if (sign (getFrame24 (i,1)) != sign2) found = TRUE;
    }
    if (found) return i + 1;
    if (sign (getFrame24 (0,0)) == 0) return 0;
    if (sign (getFrame24 (0,1)) == 0) return 0;
    return frame;
  }
  else
  {
    --i;
    sign1 = sign (getFrame24 (i,0));
    sign2 = sign (getFrame24 (i,1));
    sign3 = sign (getFrame24 (i,2));
    sign4 = sign (getFrame24 (i,3));
    while (i > 0 && !found)
    {
      --i;
      if (sign (getFrame24 (i,0)) != sign1) found = TRUE;
      if (sign (getFrame24 (i,1)) != sign2) found = TRUE;
      if (sign (getFrame24 (i,2)) != sign3) found = TRUE;
      if (sign (getFrame24 (i,3)) != sign4) found = TRUE;
    }
    if (found) return i + 1;
    if (sign (getFrame24 (0,0)) == 0) return 0;
    if (sign (getFrame24 (0,1)) == 0) return 0;
    if (sign (getFrame24 (0,2)) == 0) return 0;
    if (sign (getFrame24 (0,3)) == 0) return 0;
    return frame;
  }
}

int DPSample::getL (int frame,int scrollFrames)
{
  int i=frame;
  i -= scrollFrames;
  if (i > 0) return i;
  else return 0;
}

int DPSample::getR (int frame,int scrollFrames)
{
  int i=frame;
  i += scrollFrames;
  if (i < frames) return i;
  else return frames;
}

int DPSample::getRzero  (int frame)
{
  int i=frame;
  int found=FALSE;
  int sign1;
  int sign2;
  int sign3;
  int sign4;
  if (frame == frames) return frames;
  if (channels == 1)
  {
    sign1 = sign (getFrame24 (i,0));
    while (i < (frames - 1) && !found)
    {
      ++i;
      if (sign (getFrame24 (i,0)) != sign1) found = TRUE;
    }
    if (found) return i;
    if (sign (getFrame24 (frames - 1,0)) == 0) return frames;
    return frame;
  }
  else if (channels == 2)
  {
    sign1 = sign (getFrame24 (i,0));
    sign2 = sign (getFrame24 (i,1));
    while (i < (frames - 1) && !found)
    {
      ++i;
      if (sign (getFrame24 (i,0)) != sign1) found = TRUE;
      if (sign (getFrame24 (i,1)) != sign2) found = TRUE;
    }
    if (found) return i;
    if (sign (getFrame24 (frames - 1,0)) == 0) return frames;
    if (sign (getFrame24 (frames - 1,1)) == 0) return frames;
    return frame;
  }
  else
  {
    sign1 = sign (getFrame24 (i,0));
    sign2 = sign (getFrame24 (i,1));
    sign3 = sign (getFrame24 (i,2));
    sign4 = sign (getFrame24 (i,3));
    while (i < (frames - 1) && !found)
    {
      ++i;
      if (sign (getFrame24 (i,0)) != sign1) found = TRUE;
      if (sign (getFrame24 (i,1)) != sign2) found = TRUE;
      if (sign (getFrame24 (i,2)) != sign3) found = TRUE;
      if (sign (getFrame24 (i,3)) != sign4) found = TRUE;
    }
    if (found) return i;
    if (sign (getFrame24 (frames - 1,0)) == 0) return frames;
    if (sign (getFrame24 (frames - 1,1)) == 0) return frames;
    if (sign (getFrame24 (frames - 1,2)) == 0) return frames;
    if (sign (getFrame24 (frames - 1,3)) == 0) return frames;
    return frame;
  }
}

/*---------------------------------------------------------------------------
| Frame functions
---------------------------------------------------------------------------*/

int DPSample::getFrame24 (int frame,int channel)
{
  // return frame as 24 bit number stored in int
  if (frame < 0 || frame >= frames) return 0;
  if (channels == 1)
    channel = 0;
  else if (channels == 2)
    channel = channel % 2;
  else
    channel = channel % 4;
  if (width == 1)
    return 65536 * ((signed char *) data) [frame*channels + channel];
  else
    return 256 * ((short *) data) [frame*channels + channel];
}

int DPSample::getFrame24Extra (int frame,int channel)
{
  // return frame as 24 bit number stored in int
  if (frame < 0) return 0;
  if (channels == 1)
    channel = 0;
  else if (channels == 2)
    channel = channel % 2;
  else
    channel = channel % 4;
  if (frame < frames)
  {
    if (width == 1)
      return 65536 * ((signed char *) data) [frame*channels + channel];
    else
      return 256 * ((short *) data) [frame*channels + channel];
  }
  
  extraListEntry *e;
  
  e = extra;
  while (e)
  {
    if ((frame >= e->start) && (frame <= e->end))
    {
      if (width == 1)
        return 65536 * ((signed char *) e->data)
          [(frame - e->start)*channels + channel];
      else
        return 256 * ((short *) e->data)
          [(frame - e->start)*channels + channel];
    }
    e = e->next;
  }
  return 0;
}

void DPSample::setFrame24 (int frame,int channel,int newFrame)
{
  // store frame given as 24 bit number
  changed = TRUE;
  if (frame < 0 || frame >= frames) return;
  if (channels == 1)
    channel = 0;
  else if (channels == 2)
    channel = channel % 2;
  else
    channel = channel % 4;
  if (width == 1)
    ((signed char *) data) [frame*channels + channel] = newFrame / 65536;
  else
    ((short *) data) [frame*channels + channel] = newFrame / 256;
}

void DPSample::setFrame24Extra (int frame,int channel,int newFrame)
{
  // store frame given as 24 bit number
  changed = TRUE;
  if (frame < 0) return;
  if (channels == 1)
    channel = 0;
  else if (channels == 2)
    channel = channel % 2;
  else
    channel = channel % 4;
  if (frame < frames)
  {
    if (width == 1)
      ((signed char *) data) [frame*channels + channel] = newFrame / 65536;
    else
      ((short *) data) [frame*channels + channel] = newFrame / 256;
    return;
  }

  extraListEntry *e;

  if (!extra) e = expand (1.0);
  else e = extra;
  
  while (e)
  {
    if ((frame >= e->start) && (frame <= e->end))
    {
      if (width == 1)
        ((signed char *) e->data)
          [(frame - e->start)*channels + channel] = newFrame / 65536;
      else
        ((short *) e->data)
          [(frame - e->start)*channels + channel] = newFrame / 256;
      return;
    }
    e = e->next;
    if (!e) e = expand (1.0);
  }
}

short DPSample::getFrame16 (int frame,int channel)
{
  // return frame as 16 bit number stored in short
  if (frame < 0 || frame >= frames) return 0;
  if (channels == 1)
    channel = 0;
  else if (channels == 2)
    channel = channel % 2;
  else
    channel = channel % 4;
  if (width == 1)
    return 256 * ((signed char *) data) [frame*channels + channel];
  else
    return ((short *) data) [frame*channels + channel];
}

short DPSample::getFrame16Extra (int frame,int channel)
{
  return ((short)((getFrame24Extra (frame,channel)) / 256));
}

void DPSample::setFrame16 (int frame,int channel,short newFrame)
{
  // store frame given as 16 bit number
  changed = TRUE;
  if (frame < 0 || frame >= frames) return;
  if (channels == 1)
    channel = 0;
  else if (channels == 2)
    channel = channel % 2;
  else
    channel = channel % 4;
  if (width == 1)
    ((signed char *) data) [frame*channels + channel] = newFrame / 256;
  else
    ((short *) data) [frame*channels + channel] = newFrame;
}

void DPSample::setFrame16Extra (int frame,int channel,short newFrame)
{
  setFrame24Extra (frame,channel,((int) newFrame) * 256);
}

double DPSample::getFrameDb (int frame,int channel)
{
  // return frame as double value [-1.0,1.0]
  return (((double) getFrame24 (frame,channel)) / MAX23);
}

double DPSample::getFrameDbExtra (int frame,int channel)
{
  // return frame as double value [-1.0,1.0]
  return (((double) getFrame24Extra (frame,channel)) / MAX23);
}

void DPSample::setFrameDb (int frame,int channel,double newFrame)
{
  // store frame given as double value [-1.0,1.0]
  if (newFrame > 1.0) newFrame = 1.0;
  else if (newFrame < -1.0) newFrame = -1.0;
  int value = (int)(MAX23 * newFrame);
  if (value < -MAX23) value = -MAX23;
  else if (value > MAX23_1) value = MAX23_1;
  setFrame24 (frame,channel,value);
}

void DPSample::setFrameDbExtra (int frame,int channel,double newFrame)
{
  // store frame given as double value [-1.0,1.0]
  if (newFrame > 1.0) newFrame = 1.0;
  else if (newFrame < -1.0) newFrame = -1.0;
  int value = (int)(MAX23 * newFrame);
  if (value < -MAX23) value = -MAX23;
  else if (value > MAX23_1) value = MAX23_1;
  setFrame24Extra (frame,channel,value);
}

int DPSample::sign (int x)
{
  if (x > 0) return 1;
  if (x < 0) return -1;
  return 0;
}

/*---------------------------------------------------------------------------
| Time functions
---------------------------------------------------------------------------*/

int DPSample::timeToFrame (double time)
{
//cerr << "DPSample::timeToFrame" << endl;
  return (int) (rate * time);
}

double DPSample::frameToTime (int frame)
{
//cerr << "DPSample::frameToTime" << endl;
  return ((double) frame) / rate;
}

/***************************************************************************/
