/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : DPSampleXforms.cc
|
| Object      : Xforms sample
|
| Description : Xforms sample object
|
| (c) Richard Kent 1996
|
| $Id: DPSampleXforms.cc,v 1.1 2003/09/10 00:06:24 rk Exp $
|
****************************************************************************/

static char DPSampleXForms_cc [] = "$Id: DPSampleXforms.cc,v 1.1 2003/09/10 00:06:24 rk Exp $";

#include "forms.h"
#include "DPSample.h"
#include "DPSampleXforms.h"

typedef struct
{
  DPSample  *sample;
  int       scroll;               // Whether to scroll when playing & recording
  int       axis;                 // Whether to display the axis
  int       edit;                 // 0=Left 1=Right 2=Both (stereo samples only)
                                  // 2=All  3=Three 4=Four (quadro samples only)
  int       frames;               // Whether to display in frames or time
  int       freehand;             // Whether freehand drawing mode is on
  int       drawmode;             // 0=dots 1=lines 2=filled
  int       drawaccuracy;         // 0=quick 1=accurate
  int       undomode;             // 0=no undo 1=undo
  int       autowindows;          // close / reopen windows when doing DSP
  int       autoglobals;          // automatically adjust audio globals
  int       autoeditmode;         // Whether range select chooses mode
  int       soxcompatible;        // sox compatibility - affects
                                  // + saving of copyright chunk in AIFF files
  int       tooltips;             // show tooltips
  int       rangealt;             // use alternative range marking method
  int       rangecross;           // use crossover when range marking
  char      tempdir [TEMPDIRLEN]; // directory for temp saves
  int       vertscroll;           // show vertical scrollbar
  int       inputtextsize;        // input box text size
  int       borderwidth;          // overall border width
  int       basicwavs;            // save basic WAVs (no looping, comments, etc)
  char      audiodev [DEVICELEN]; // device for audio
  char      mixerdev [DEVICELEN]; // device for mixer
  int       rangeMark;            // Initial range mark
  int       rangeGrab;            // 0=none 1=start 2=end
  int       oldPosx;              // Previous play or record position
  int       grabSusStart;         // Sustain start marker grabbed
  int       grabSusEnd;           // Sustain end marker grabbed
  int       grabRelStart;         // Release start marker grabbed
  int       grabRelEnd;           // Release end marker grabbed
  int       oldframe;             // Previous frame for freehand
  int       oldval;               // Previous sample value for freehand
  int       oldchan;              // Previous channel for freehand
} sampleSpec;

#define SPEC ((sampleSpec *)ob->spec)

#define TOPIXELX(frame)                                                     \
  (int)((long long) objLeftX +                                              \
  ((((long long)(frame) - displayStart) *                                   \
  (objWidth - 1)) / displayDuration))

#define TOPIXELY(samp24,sampBot)                                            \
  (int) rint ((double) sampBot -                                           \
  ((double) samp24 - displayVertLow) *                                      \
  (sampleHeight - 1) / displayHeight)

#define TOFRAME(pixelx)                                                     \
  (int)((long long) displayStart +                                         \
  ((pixelx) * (long long) displayDuration / (objWidth - 1)))

#define TOFRAMEROUND(pixelx)                                                \
  (int)((long long) displayStart +                                         \
  (((pixelx) * (long long) displayDuration) + ((objWidth - 1) / 2)) /       \
  (objWidth - 1))

#define TOSAMPLE(pixely,sampBot)                                            \
  (int) rint (((double) sampBot - pixely) * displayHeight /                \
  (sampleHeight - 1) + displayVertLow)

#define ROUNDSAMPLE(samp24)                                                 \
  if (samp24 > 0) samp24 += (MAX23 / width);                                \
  else samp24 -= (MAX23 / width)

/*---------------------------------------------------------------------------
| FUNCTION fl_create_sample
---------------------------------------------------------------------------*/
extern FL_OBJECT *fl_create_sample (
  int type,
  FL_Coord x,
  FL_Coord y, 
  FL_Coord w,
  FL_Coord h,
  const char *label)
{
  FL_OBJECT *ob;
  
  ob = fl_make_object (FL_SAMPLE,type,x,y,w,h,label,handle_sample);
  ob->boxtype = FL_FLAT_BOX;
  ob->spec = fl_calloc (1,sizeof (sampleSpec));
  
  // Set defaults
  SPEC->scroll         = 0;
  SPEC->axis           = 1;
  SPEC->edit           = 2;
  SPEC->frames         = 1;
  SPEC->freehand       = 0;
  SPEC->drawmode       = 1;
  SPEC->drawaccuracy   = 0;
  SPEC->undomode       = 1;
  SPEC->autowindows    = 1;
  SPEC->autoglobals    = 1;
  SPEC->autoeditmode   = 1;
  SPEC->soxcompatible  = 0;
  SPEC->tooltips       = 1;
  SPEC->rangealt       = 0;
  SPEC->rangecross     = 0;
  strcpy (SPEC->tempdir,"/tmp");
  SPEC->vertscroll     = 0;
  SPEC->inputtextsize  = FL_SMALL_SIZE;
  SPEC->borderwidth    = FL_BOUND_WIDTH;
  SPEC->basicwavs      = 0;
  #ifdef NETBSD
  strcpy (SPEC->audiodev,"/dev/audio");
  strcpy (SPEC->mixerdev,"/dev/mixer");
  #else
  strcpy (SPEC->audiodev,"/dev/dsp");
  strcpy (SPEC->mixerdev,"/dev/mixer");
  #endif
  SPEC->rangeMark      = -1;
  SPEC->rangeGrab      = 0;
  SPEC->oldPosx        = -1;
  SPEC->grabSusStart   = 0;
  SPEC->grabSusEnd     = 0;
  SPEC->grabRelStart   = 0;
  SPEC->grabRelEnd     = 0;
  SPEC->oldframe       = -1;
  SPEC->oldval         = 0;
  SPEC->oldchan        = 0;
  return ob;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_add_sample
---------------------------------------------------------------------------*/
extern FL_OBJECT *fl_add_sample (
  int type,
  FL_Coord x,
  FL_Coord y, 
  FL_Coord w,
  FL_Coord h,
  const char *label)
{
  FL_OBJECT *ob = fl_create_sample (type,x,y,w,h,label);
  fl_add_object (fl_current_form,ob);
  return ob;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample
---------------------------------------------------------------------------*/
extern DPSample *fl_get_sample (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->sample;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample
---------------------------------------------------------------------------*/
extern void fl_set_sample (FL_OBJECT *ob,DPSample *newSample)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->sample       = newSample;
  SPEC->freehand     = 0;
  SPEC->rangeMark    = -1;
  SPEC->oldPosx      = -1;
  SPEC->grabSusStart = 0;
  SPEC->grabSusEnd   = 0;
  SPEC->grabRelStart = 0;
  SPEC->grabRelEnd   = 0;
  SPEC->oldframe     = -1;
  SPEC->oldval       = 0;
  SPEC->oldchan      = 0;
  fl_set_button (mainForm->manualEdit,0);
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_scroll
---------------------------------------------------------------------------*/
extern int fl_get_sample_scroll (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->scroll;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_scroll
---------------------------------------------------------------------------*/
extern void fl_set_sample_scroll (FL_OBJECT *ob,int newScroll)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->scroll = newScroll;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_axis
---------------------------------------------------------------------------*/
extern int fl_get_sample_axis (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->axis;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_axis
---------------------------------------------------------------------------*/
extern void fl_set_sample_axis (FL_OBJECT *ob,int newAxis)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->axis = newAxis;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_edit
---------------------------------------------------------------------------*/
extern int fl_get_sample_edit (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->edit;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_edit
---------------------------------------------------------------------------*/
extern void fl_set_sample_edit (FL_OBJECT *ob,int newEdit)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->edit = newEdit;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_frames
---------------------------------------------------------------------------*/
extern int fl_get_sample_frames (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->frames;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_frames
---------------------------------------------------------------------------*/
extern void fl_set_sample_frames (FL_OBJECT *ob,int newFrames)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->frames = newFrames;
}


/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_freehand
---------------------------------------------------------------------------*/
extern int fl_get_sample_freehand (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->freehand;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_freehand
---------------------------------------------------------------------------*/
extern void fl_set_sample_freehand (FL_OBJECT *ob,int newFreehand)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->freehand = newFreehand;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_drawmode
---------------------------------------------------------------------------*/
extern int fl_get_sample_drawmode (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->drawmode;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_drawmode
---------------------------------------------------------------------------*/
extern void fl_set_sample_drawmode (FL_OBJECT *ob,int newDrawmode)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->drawmode = newDrawmode;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_drawaccuracy
---------------------------------------------------------------------------*/
extern int fl_get_sample_drawaccuracy (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->drawaccuracy;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_drawaccuracy
---------------------------------------------------------------------------*/
extern void fl_set_sample_drawaccuracy (FL_OBJECT *ob,int newDrawaccuracy)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->drawaccuracy = newDrawaccuracy;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_undomode
---------------------------------------------------------------------------*/
extern int fl_get_sample_undomode (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->undomode;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_undomode
---------------------------------------------------------------------------*/
extern void fl_set_sample_undomode (FL_OBJECT *ob,int newUndomode)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->undomode = newUndomode;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_autowindows
---------------------------------------------------------------------------*/
extern int fl_get_sample_autowindows (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->autowindows;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_autowindows
---------------------------------------------------------------------------*/
extern void fl_set_sample_autowindows (FL_OBJECT *ob,int newAutowindows)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->autowindows = newAutowindows;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_autoglobals
---------------------------------------------------------------------------*/
extern int fl_get_sample_autoglobals(FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->autoglobals;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_autoglobals
---------------------------------------------------------------------------*/
extern void fl_set_sample_autoglobals(FL_OBJECT *ob,int newAutoglobals)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->autoglobals = newAutoglobals;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_autoeditmode
---------------------------------------------------------------------------*/
extern int fl_get_sample_autoeditmode(FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->autoeditmode;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_autoeditmode
---------------------------------------------------------------------------*/
extern void fl_set_sample_autoeditmode(FL_OBJECT *ob,int newAutoeditmode)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->autoeditmode = newAutoeditmode;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_soxcompatible
---------------------------------------------------------------------------*/
extern int fl_get_sample_soxcompatible (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->soxcompatible;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_soxcompatible
---------------------------------------------------------------------------*/
extern void fl_set_sample_soxcompatible (FL_OBJECT *ob,int newSoxcompatible)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->soxcompatible = newSoxcompatible;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_tooltips
---------------------------------------------------------------------------*/
extern int fl_get_sample_tooltips (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->tooltips;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_tooltips
---------------------------------------------------------------------------*/
extern void fl_set_sample_tooltips (FL_OBJECT *ob,int newTooltips)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->tooltips = newTooltips;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_rangealt
---------------------------------------------------------------------------*/
extern int fl_get_sample_rangealt (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->rangealt;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_rangealt
---------------------------------------------------------------------------*/
extern void fl_set_sample_rangealt (FL_OBJECT *ob,int newRangealt)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->rangealt = newRangealt;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_rangecross
---------------------------------------------------------------------------*/
extern int fl_get_sample_rangecross (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->rangecross;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_rangecross
---------------------------------------------------------------------------*/
extern void fl_set_sample_rangecross (FL_OBJECT *ob,int newRangecross)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->rangecross = newRangecross;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_tempdir
---------------------------------------------------------------------------*/
extern const char *fl_get_sample_tempdir (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return (const char *) SPEC->tempdir;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_tempdir
---------------------------------------------------------------------------*/
extern void fl_set_sample_tempdir (FL_OBJECT *ob,const char *newTempdir)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  strncpy (SPEC->tempdir,(char *)newTempdir,TEMPDIRLEN);
  SPEC->tempdir[TEMPDIRLEN-1] = 0;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_vertscroll
---------------------------------------------------------------------------*/
extern int fl_get_sample_vertscroll (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->vertscroll;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_vertscroll
---------------------------------------------------------------------------*/
extern void fl_set_sample_vertscroll (FL_OBJECT *ob,int newVertscroll)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->vertscroll = newVertscroll;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_inputtextsize
---------------------------------------------------------------------------*/
extern int fl_get_sample_inputtextsize (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->inputtextsize;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_inputtextsize
---------------------------------------------------------------------------*/
extern void fl_set_sample_inputtextsize (FL_OBJECT *ob,int newInputtextsize)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->inputtextsize = newInputtextsize;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_borderwidth
---------------------------------------------------------------------------*/
extern int fl_get_sample_borderwidth (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->borderwidth;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_borderwidth
---------------------------------------------------------------------------*/
extern void fl_set_sample_borderwidth (FL_OBJECT *ob,int newBorderwidth)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->borderwidth = newBorderwidth;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_basicwavs
---------------------------------------------------------------------------*/
extern int fl_get_sample_basicwavs (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return SPEC->basicwavs;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_basicwavs
---------------------------------------------------------------------------*/
extern void fl_set_sample_basicwavs (FL_OBJECT *ob,int newBasicwavs)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  SPEC->basicwavs = newBasicwavs;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_audio_device
---------------------------------------------------------------------------*/
extern const char *fl_get_sample_audio_device (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return (const char *) SPEC->audiodev;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_audio_device
---------------------------------------------------------------------------*/
extern void fl_set_sample_audio_device (FL_OBJECT *ob,const char *newAudioDevice)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  strncpy (SPEC->audiodev,(char *)newAudioDevice,DEVICELEN);
  SPEC->audiodev[DEVICELEN-1] = 0;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_get_sample_mixer_device
---------------------------------------------------------------------------*/
extern const char *fl_get_sample_mixer_device (FL_OBJECT *ob)
{
  if (!ob || ob->objclass != FL_SAMPLE) return 0;
  return (const char *) SPEC->mixerdev;
}

/*---------------------------------------------------------------------------
| FUNCTION fl_set_sample_mixer_device
---------------------------------------------------------------------------*/
extern void fl_set_sample_mixer_device (FL_OBJECT *ob,const char *newMixerDevice)
{
  if (!ob || ob->objclass != FL_SAMPLE) return;
  strncpy (SPEC->mixerdev,(char *)newMixerDevice,DEVICELEN);
  SPEC->mixerdev[DEVICELEN-1] = 0;
}

/*---------------------------------------------------------------------------
| FUNCTION handle_sample
---------------------------------------------------------------------------*/

int handle_sample (
  FL_OBJECT *ob,
  int event,
  FL_Coord mx,
  FL_Coord my,
  int key,
  void *)
{
  DPSample *sample  = SPEC->sample;
  int *rangeMark   = &SPEC->rangeMark;
  int *oldPosx      = &SPEC->oldPosx;

  switch (event)
  {
    case FL_DRAW:
    {
      int displayStart    = sample->getDisplayStart ();
      int displayEnd      = sample->getDisplayEnd ();
      int displayDuration = displayEnd - displayStart;
      int displayVertLow  = sample->getDisplayVertLow ();
      int displayVertHigh = sample->getDisplayVertHigh ();
      int displayHeight   = displayVertHigh - displayVertLow;
      int rangeStart      = sample->getRangeStart ();
      int rangeEnd        = sample->getRangeEnd ();
      int objLeftX        = ob->x + BORDERX;
      int objRightX       = ob->x + ob->w - 1 - BORDERX;
      int objTopY         = ob->y + BORDERY;
      int objBotY         = ob->y + ob->h - 1 - BORDERY;
      int objWidth        = ob->w-(2*BORDERX);
//    int objHeight       = ob->h-(2*BORDERY);
      int first           = TRUE;
      int mono            = sample->getChannels () == 1;
      int stereo          = sample->getChannels () == 2;
      int quadro          = sample->getChannels () == 4;
//    int width           = (sample->getWidth () == 1) ? MAX8 : MAX16;
      int currentPos      = 0;
      int frame;
      int sample24L;
      int sample24R;
      int sample243;
      int sample244;
      int i;
      int px;
      int pyL;
      int pyR;
      int py3;
      int py4;
      int oldframe        = -1;
      int oldpx           = -1;
      int oldpyL          = -1;
      int oldpyR          = -1;
      int oldpy3          = -1;
      int oldpy4          = -1;
      int oldpyLmax       = -1;
      int oldpyRmax       = -1;
      int oldpy3max       = -1;
      int oldpy4max       = -1;
//    Legacy code for direct X drawing
//    GC gc                = (fl_state [fl_get_vclass()].gc)[15];
//    Window win           = fl_winget ();
//    Display *display     = fl_get_display ();
      FL_COLOR drawColour;
      int sampleTopL       = 0;
      int sampleBotL       = 0;
      int sampleTopR       = 0;
      int sampleBotR       = 0;
      int sampleTop3       = 0;
      int sampleBot3       = 0;
      int sampleTop4       = 0;
      int sampleBot4       = 0;
      int sampleHeight     = 0;
      int sampleCentreL    = 0;
      int sampleCentreR    = 0;
      int sampleCentre3    = 0;
      int sampleCentre4    = 0;
      
      if (sample->getRedraw ())
      {
        int scroll=0;
        
        // Remove old position if valid
        if (*oldPosx != -1)
        {
          fl_drawmode (GXxor);
          fl_line (*oldPosx,objTopY,*oldPosx,objBotY,FL_YELLOW);
          fl_drawmode (GXcopy);
          *oldPosx = -1;
        }
        
        currentPos = sample->getCurrentPos ();

        // Scroll display if required
        if (fl_get_sample_scroll (ob) && currentPos != -1)
        {
          int displayMiddle = (displayStart + displayEnd) / 2;
          if (currentPos < displayStart)
          {
            sample->scrollL (displayMiddle - currentPos);
            scroll = 1;
            displayStart    = sample->getDisplayStart ();
            displayEnd      = sample->getDisplayEnd ();
            displayDuration = displayEnd - displayStart;
          }
          else if (currentPos > displayEnd)
          {
            sample->scrollR (currentPos - displayMiddle);
            scroll = 1;
            displayStart    = sample->getDisplayStart ();
            displayEnd      = sample->getDisplayEnd ();
            displayDuration = displayEnd - displayStart;
          }
        }
        
        // Draw current position if valid
        if (!scroll && sample->getRedraw () != 2)
        {
          if (currentPos >= displayStart && currentPos <= displayEnd)
          {
            *oldPosx = TOPIXELX (currentPos);
            fl_drawmode (GXxor);
            fl_line (*oldPosx,objTopY,*oldPosx,objBotY,FL_YELLOW);
            fl_drawmode (GXcopy);
          }
          return 0;
        }
      }

      // Draw normal box (xforms)
      fl_drw_box 
        (ob->boxtype,ob->x,ob->y,ob->w,ob->h,ob->col1,ob->bw);

      // Calculate sample boundaries, draw frames, central lines and axis
      if (mono)
      {
        if (SPEC->axis)
        {
          // Mono - with axis markings
          
          sampleTopL   = objTopY;
          sampleBotL   = objBotY - AXISSPACE - MONOGAP;
          sampleHeight = sampleBotL - sampleTopL;
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopL,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleBotL + MONOGAP,
            objWidth,AXISSPACE,FL_BLACK,ob->bw);
          sampleCentreL = TOPIXELY (0,sampleBotL);
          fl_set_clipping (objLeftX,sampleTopL,objWidth,sampleHeight);
          fl_line (objLeftX,sampleCentreL,objRightX,sampleCentreL,FL_BLACK);
          fl_unset_clipping ();
          
          int axis = objBotY - (AXISSPACE * 2 / 3);
          fl_line (objLeftX,axis,objRightX,axis,FL_BLACK);
          int space;
          if (displayDuration < 10)
          {
            space = 1;
          }
          else
          {
            int x = displayDuration / 10;
            int rounding =
              (int) rint (pow (10.0,rint (log10 (displayDuration)) - 1.0));
            space = rounding * (int) rint ((double)x / rounding + 0.5);
          }
          int frame = 0;
          char tempString [50];
          while (frame < displayStart) frame += space;
          while (frame < displayEnd)
          {
            px = TOPIXELX (frame);
            fl_line (px,axis - AXISTICKS,px,axis + AXISTICKS,FL_BLACK);
            if (SPEC->frames)
              sprintf (tempString,"%d",frame);
            else
              sprintf (tempString,"%.3f",frame / sample->getRate ());
            if (px < objRightX - 50)
              fl_drw_text (FL_ALIGN_LEFT,px,axis,
                50,14,FL_BLACK,FL_NORMAL_STYLE,AXISTEXTSIZE,tempString);
            frame += space;
          }
        }
        else
        {
          // Mono - no axis markings

          sampleTopL   = objTopY;
          sampleBotL   = objBotY;
          sampleHeight = sampleBotL - sampleTopL;
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopL,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          sampleCentreL = TOPIXELY (0,sampleBotL);
          fl_set_clipping (objLeftX,sampleTopL,objWidth,sampleHeight);
          fl_line (objLeftX,sampleCentreL,objRightX,sampleCentreL,FL_BLACK);
          fl_unset_clipping ();
        }
      }
      else if (stereo)
      {
        if (SPEC->axis)
        {
          // Stereo - with axis markings
          
          sampleHeight = (objBotY - objTopY - AXISSPACE) / 2;
          sampleTopL   = objTopY;
          sampleBotL   = sampleTopL + sampleHeight;
          sampleBotR   = objBotY;
          sampleTopR   = sampleBotR - sampleHeight;
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopL,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopR,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          sampleCentreL = TOPIXELY (0,sampleBotL);
          sampleCentreR = TOPIXELY (0,sampleBotR);
          fl_set_clipping (objLeftX,sampleTopL,objWidth,sampleHeight);
          fl_line (objLeftX,sampleCentreL,objRightX,sampleCentreL,FL_BLACK);
          fl_set_clipping (objLeftX,sampleTopR,objWidth,sampleHeight);
          fl_line (objLeftX,sampleCentreR,objRightX,sampleCentreR,FL_BLACK);
          fl_unset_clipping ();
          
          int axis = sampleTopR - (AXISSPACE * 2 / 3);
          fl_line (objLeftX,axis,objRightX,axis,FL_BLACK);
          int space;
          if (displayDuration < 10)
          {
            space = 1;
          }
          else
          {
            int x = displayDuration / 10;
            int rounding =
              (int) rint (pow (10.0,rint (log10 (displayDuration)) - 1.0));
            space = rounding * (int) rint ((double)x / rounding + 0.5);
          }
          int frame = 0;
          char tempString [50];
          while (frame < displayStart) frame += space;
          while (frame < displayEnd)
          {
            px = TOPIXELX (frame);
            fl_line (px,axis - AXISTICKS,px,axis + AXISTICKS,FL_BLACK);
            if (SPEC->frames)
              sprintf (tempString,"%d",frame);
            else
              sprintf (tempString,"%.3f",frame / sample->getRate ());
            if (px < objRightX - 50)
              fl_drw_text (FL_ALIGN_LEFT,px,axis,
                50,14,FL_BLACK,FL_NORMAL_STYLE,AXISTEXTSIZE,tempString);
            frame += space;
          }
        }
        else
        {
          // Stereo - no axis markings

          sampleHeight = (objBotY - objTopY - STEREOGAP) / 2;
          sampleTopL   = objTopY;
          sampleBotL   = sampleTopL + sampleHeight;
          sampleBotR   = objBotY;
          sampleTopR   = sampleBotR - sampleHeight;
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopL,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopR,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          sampleCentreL = TOPIXELY (0,sampleBotL);
          sampleCentreR = TOPIXELY (0,sampleBotR);
          fl_set_clipping (objLeftX,sampleTopL,objWidth,sampleHeight);
          fl_line (objLeftX,sampleCentreL,objRightX,sampleCentreL,FL_BLACK);
          fl_set_clipping (objLeftX,sampleTopR,objWidth,sampleHeight);
          fl_line (objLeftX,sampleCentreR,objRightX,sampleCentreR,FL_BLACK);
          fl_unset_clipping ();
        }
      }
      else
      {
        if (SPEC->axis)
        {
          // Quadro - with axis markings
          
          sampleHeight = (objBotY - objTopY - AXISSPACE - 2 * QUADROGAP) / 4;
          sampleTopL   = objTopY;
          sampleBotL   = objTopY + sampleHeight;
          sampleTopR   = sampleBotL + QUADROGAP;
          sampleBotR   = sampleTopR + sampleHeight;
          sampleBot4   = objBotY;
          sampleTop4   = sampleBot4 - sampleHeight;
          sampleBot3   = sampleTop4 - QUADROGAP;
          sampleTop3   = sampleBot3 - sampleHeight;
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopL,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopR,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTop3,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTop4,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          sampleCentreL = TOPIXELY (0,sampleBotL);
          sampleCentreR = TOPIXELY (0,sampleBotR);
          sampleCentre3 = TOPIXELY (0,sampleBot3);
          sampleCentre4 = TOPIXELY (0,sampleBot4);
          fl_set_clipping (objLeftX,sampleTopL,objWidth,sampleHeight);
          fl_line (objLeftX,sampleCentreL,objRightX,sampleCentreL,FL_BLACK);
          fl_set_clipping (objLeftX,sampleTopR,objWidth,sampleHeight);
          fl_line (objLeftX,sampleCentreR,objRightX,sampleCentreR,FL_BLACK);
          fl_set_clipping (objLeftX,sampleTop3,objWidth,sampleHeight);
          fl_line (objLeftX,sampleCentre3,objRightX,sampleCentre3,FL_BLACK);
          fl_set_clipping (objLeftX,sampleTop4,objWidth,sampleHeight);
          fl_line (objLeftX,sampleCentre4,objRightX,sampleCentre4,FL_BLACK);
          fl_unset_clipping ();
          
          int axis = sampleTop3 - (AXISSPACE * 2 / 3);
          fl_line (objLeftX,axis,objRightX,axis,FL_BLACK);
          int space;
          if (displayDuration < 10)
          {
            space = 1;
          }
          else
          {
            int x = displayDuration / 10;
            int rounding =
              (int) rint (pow (10.0,rint (log10 (displayDuration)) - 1.0));
            space = rounding * (int) rint ((double)x / rounding + 0.5);
          }
          int frame = 0;
          char tempString [50];
          while (frame < displayStart) frame += space;
          while (frame < displayEnd)
          {
            px = TOPIXELX (frame);
            fl_line (px,axis - AXISTICKS,px,axis + AXISTICKS,FL_BLACK);
            if (SPEC->frames)
              sprintf (tempString,"%d",frame);
            else
              sprintf (tempString,"%.3f",frame / sample->getRate ());
            if (px < objRightX - 50)
              fl_drw_text (FL_ALIGN_LEFT,px,axis,
                50,14,FL_BLACK,FL_NORMAL_STYLE,AXISTEXTSIZE,tempString);
            frame += space;
          }
        }
        else
        {
          // Quadro - no axis markings

          sampleHeight = (objBotY - objTopY - 3 * QUADROGAP) / 4;
          sampleTopL   = objTopY;
          sampleBotL   = objTopY + sampleHeight;
          sampleTopR   = sampleBotL + QUADROGAP;
          sampleBotR   = sampleTopR + sampleHeight;
          sampleBot4   = objBotY;
          sampleTop4   = sampleBot4 - sampleHeight;
          sampleBot3   = sampleTop4 - QUADROGAP;
          sampleTop3   = sampleBot3 - sampleHeight;
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopL,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTopR,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTop3,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          fl_drw_frame (FL_ENGRAVED_FRAME,objLeftX,sampleTop4,
            objWidth,sampleHeight,FL_BLACK,ob->bw);
          sampleCentreL = TOPIXELY (0,sampleBotL);
          sampleCentreR = TOPIXELY (0,sampleBotR);
          sampleCentre3 = TOPIXELY (0,sampleBot3);
          sampleCentre4 = TOPIXELY (0,sampleBot4);
          fl_set_clipping (objLeftX,sampleTopL,objWidth,sampleHeight);
          fl_line (objLeftX,sampleCentreL,objRightX,sampleCentreL,FL_BLACK);
          fl_set_clipping (objLeftX,sampleTopR,objWidth,sampleHeight);
          fl_line (objLeftX,sampleCentreR,objRightX,sampleCentreR,FL_BLACK);
          fl_set_clipping (objLeftX,sampleTop3,objWidth,sampleHeight);
          fl_line (objLeftX,sampleCentre3,objRightX,sampleCentre3,FL_BLACK);
          fl_set_clipping (objLeftX,sampleTop4,objWidth,sampleHeight);
          fl_line (objLeftX,sampleCentre4,objRightX,sampleCentre4,FL_BLACK);
          fl_unset_clipping ();
        }
      }
      
      // Draw range box if appropriate
      if (sample->getValid () && displayDuration != 0
        && sample->getRangeValid ())
      {
        if (rangeStart <= displayEnd && rangeEnd >= displayStart)
        {
          if (rangeStart < displayStart) rangeStart = displayStart;
          if (rangeEnd > displayEnd)     rangeEnd   = displayEnd;
          int px1 = TOPIXELX (rangeStart);
          int px2 = TOPIXELX (rangeEnd);
          if (mono || (SPEC->edit == 2 || SPEC->edit == 0))
          {
            fl_rectf (px1,sampleTopL,px2-px1+1,sampleHeight,FL_BOTTOM_BCOL);
            fl_line (px1,sampleCentreL,px2,sampleCentreL,FL_BLACK);
          }
          if ((stereo || quadro) && (SPEC->edit == 2 || SPEC->edit == 1))
          {
            fl_rectf (px1,sampleTopR,px2-px1+1,sampleHeight,FL_BOTTOM_BCOL);
            fl_line (px1,sampleCentreR,px2,sampleCentreR,FL_BLACK);
          }
          if (quadro && (SPEC->edit == 2 || SPEC->edit == 3))
          {
            fl_rectf (px1,sampleTop3,px2-px1+1,sampleHeight,FL_BOTTOM_BCOL);
            fl_line (px1,sampleCentre3,px2,sampleCentre3,FL_BLACK);
          }
          if (quadro && (SPEC->edit == 2 || SPEC->edit == 4))
          {
            fl_rectf (px1,sampleTop4,px2-px1+1,sampleHeight,FL_BOTTOM_BCOL);
            fl_line (px1,sampleCentre4,px2,sampleCentre4,FL_BLACK);
          }
        }
      }
      
      // This ensures that sliders aren't redrawn (causes problems)
      fl_set_clipping (0,0,0,0);
      
      // Update scroll and zoom sliders if necessary
      if (!sample->getValid () || sample->getFrames() == displayDuration)
      {
        /* Note zoom form must not be last !! */
        fl_set_slider_value (mainForm->zoomBarSlider,0.0);
        fl_set_slider_size (mainForm->scrollBarSlider,1.0);
        fl_set_slider_value (mainForm->scrollBarSlider,0.5);
      }
      else
      {
        double size = (double) displayDuration / (double) sample->getFrames();
        double pos = (double) displayStart /
          ((double) sample->getFrames () - (double) displayDuration);
        double zoom = 1.0 - (double) displayDuration /
          (double) sample->getFrames ();
        
        /* Note zoom form must not be last !! */
        fl_set_slider_value (mainForm->zoomBarSlider,zoom);
        fl_set_slider_size (mainForm->scrollBarSlider,size);
        fl_set_slider_value (mainForm->scrollBarSlider,pos);
      }

      // Update vert scroll and zoom sliders if necessary
      if (!sample->getValid () || displayHeight == MAX24_1)
      {
        /* Note zoom form must not be last !! */
        fl_set_slider_size (mainForm->vertScrollBarSlider,1.0);
        fl_set_slider_value (mainForm->vertScrollBarSlider,0.5);
        fl_set_slider_value (mainForm->vertZoomBarSlider,0.0);
      }
      else
      {
        double vertSize = (double) displayHeight / (double) MAX24_1;
        double vertPos  = (double) (displayVertLow + MAX23) /
          ((double) MAX24_1 - (double) displayHeight);
        double vertZoom = 1.0 - (double) displayHeight / (double) MAX24_1;
        fl_set_slider_size (mainForm->vertScrollBarSlider,vertSize);
        fl_set_slider_value (mainForm->vertScrollBarSlider,vertPos);
        fl_set_slider_value (mainForm->vertZoomBarSlider,vertZoom);
      }
      
      // Return clipping
      fl_unset_clipping ();
      
      // Don't proceed if no valid sample
      if (!sample->getValid () || displayDuration == 0 || displayHeight == 0)
        return 0;
      
      // Calculate grid locking factor
      int grid = displayDuration / (int) objWidth;
      
      int rangeColour;

      // Draw sample waveform
      if (SPEC->drawaccuracy == 0)
      {
        // Quick method
        
        for (i=0; i<objWidth; i++)
        {
          // Calculate frame
          frame = TOFRAME (i);
          if (frame >= displayEnd) frame = displayEnd - 1;

          // Grid based locking
          if (grid > 1) frame = frame - (frame % grid);

          // Set colour inversion if within range bounds
          if (sample->getRangeValid () && frame >= rangeStart
            && frame <= rangeEnd-1)
            rangeColour = 1;
          else
            rangeColour = 0;

          // Retrieve frame values
          sample24L = sample->getFrame24 (frame,0);

          // Calculate x and y coordinates
          px = objLeftX + i;
          pyL = TOPIXELY (sample24L,sampleBotL);

          if (stereo)
          {
            // Stereo

            sample24R = sample->getFrame24 (frame,1);
            pyR = TOPIXELY (sample24R,sampleBotR);

            if (rangeColour && SPEC->edit != 1)
              drawColour = FL_CYAN;
            else
              drawColour = FL_BLACK;

            fl_set_clipping (objLeftX,sampleTopL,objWidth,sampleHeight);
            
            // Draw (dots, lines or filled) for left
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
                fl_point (px,pyL,drawColour);
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              if (first)
                fl_point (px,pyL,drawColour);
              else
                fl_line (oldpx+1,oldpyL,px,pyL,drawColour);
            }
            else
            {
              // Filled
              fl_line (px,sampleCentreL,px,pyL,drawColour);
            }

            if (rangeColour && SPEC->edit != 0)
              drawColour = FL_CYAN;
            else
              drawColour = FL_BLACK;

            fl_set_clipping (objLeftX,sampleTopR,objWidth,sampleHeight);
            
            // Draw (dots, lines or filled) for right
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
                fl_point (px,pyR,drawColour);
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              if (first)
              {
                fl_point (px,pyR,drawColour);
                first = FALSE;
              }
              else
              {
                fl_line (oldpx+1,oldpyR,px,pyR,drawColour);
              }
            }
            else
            {
              // Filled
              fl_line (px,sampleCentreR,px,pyR,drawColour);
            }

            fl_unset_clipping ();
            
            oldframe = frame;
            oldpx    = px;
            oldpyL   = pyL;
            oldpyR   = pyR;
          }
          else if (mono)
          {
            // Mono

            if (rangeColour)
              drawColour = FL_CYAN;
            else
              drawColour = FL_BLACK;

            fl_set_clipping (objLeftX,sampleTopL,objWidth,sampleHeight);
            
            // Draw (dots, lines or filled)
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
              {
                fl_point (px,pyL,drawColour);
              }
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              if (first)
              {
                fl_point (px,pyL,drawColour);
                first = FALSE;
              }
              else
              {
                fl_line (oldpx+1,oldpyL,px,pyL,drawColour);
              }
            }
            else
            {
              // Filled
              fl_line (px,sampleCentreL,px,pyL,drawColour);
            }
            
            fl_unset_clipping ();
            
            oldframe = frame;
            oldpx    = px;
            oldpyL   = pyL;
          }
          else
          {
            // Quadro

            sample24R = sample->getFrame24 (frame,1);
            sample243 = sample->getFrame24 (frame,2);
            sample244 = sample->getFrame24 (frame,3);

            pyR = TOPIXELY (sample24R,sampleBotR);
            py3 = TOPIXELY (sample243,sampleBot3);
            py4 = TOPIXELY (sample244,sampleBot4);

            if (rangeColour && (SPEC->edit == 2 || SPEC->edit == 0))
              drawColour = FL_CYAN;
            else
              drawColour = FL_BLACK;

            fl_set_clipping (objLeftX,sampleTopL,objWidth,sampleHeight);
            
            // Draw (dots, lines or filled) for left
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
                fl_point (px,pyL,drawColour);
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              if (first)
                fl_point (px,pyL,drawColour);
              else
                fl_line (oldpx+1,oldpyL,px,pyL,drawColour);
            }
            else
            {
              // Filled
              fl_line (px,sampleCentreL,px,pyL,drawColour);
            }

            if (rangeColour && (SPEC->edit == 2 || SPEC->edit == 1))
              drawColour = FL_CYAN;
            else
              drawColour = FL_BLACK;

            fl_set_clipping (objLeftX,sampleTopR,objWidth,sampleHeight);
            
            // Draw (dots, lines or filled) for right
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
                fl_point (px,pyR,drawColour);
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              if (first)
                fl_point (px,pyR,drawColour);
              else
                fl_line (oldpx+1,oldpyR,px,pyR,drawColour);
            }
            else
            {
              // Filled
              fl_line (px,sampleCentreR,px,pyR,drawColour);
            }

            if (rangeColour && (SPEC->edit == 2 || SPEC->edit == 3))
              drawColour = FL_CYAN;
            else
              drawColour = FL_BLACK;

            fl_set_clipping (objLeftX,sampleTop3,objWidth,sampleHeight);
            
            // Draw (dots, lines or filled) for channel 3
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
                fl_point (px,py3,drawColour);
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              if (first)
                fl_point (px,py3,drawColour);
              else
                fl_line (oldpx+1,oldpy3,px,py3,drawColour);
            }
            else
            {
              // Filled
              fl_line (px,sampleCentre3,px,py3,drawColour);
            }

            if (rangeColour && (SPEC->edit == 2 || SPEC->edit == 4))
              drawColour = FL_CYAN;
            else
              drawColour = FL_BLACK;

            fl_set_clipping (objLeftX,sampleTop4,objWidth,sampleHeight);
            
            // Draw (dots, lines or filled) for channel 4
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
                fl_point (px,py4,drawColour);
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              if (first)
              {
                fl_point (px,py4,drawColour);
                first = FALSE;
              }
              else
              {
                fl_line (oldpx+1,oldpy4,px,py4,drawColour);
              }
            }
            else
            {
              // Filled
              fl_line (px,sampleCentre4,px,py4,drawColour);
            }

            fl_unset_clipping ();
            
            oldframe = frame;
            oldpx    = px;
            oldpyL   = pyL;
            oldpyR   = pyR;
            oldpy3   = py3;
            oldpy4   = py4;
          }
        }
      }
      else
      {
        // Accurate method
        
        for (i=0; i<objWidth; i++)
        {
          int nextframe;
          int fLmin;
          int fLmax;
          int fRmin;
          int fRmax;
          int f3min;
          int f3max;
          int f4min;
          int f4max;
          int pyLmin;
          int pyLmax;
          int pyRmin;
          int pyRmax;
          int py3min;
          int py3max;
          int py4min;
          int py4max;
          
          // Calculate frame
          frame = TOFRAME (i);
          nextframe = TOFRAME (i+1);
          if (frame >= displayEnd) frame = displayEnd - 1;
          if (nextframe >= displayEnd) nextframe = displayEnd - 1;

          // Grid based locking
          if (grid > 1)
          {
            frame = frame - (frame % grid);
            nextframe = nextframe - (nextframe % grid);
          }

          // Set colour inversion if within range bounds
          if (sample->getRangeValid () && frame >= rangeStart
            && frame <= rangeEnd-1)
            rangeColour = 1;
          else
            rangeColour = 0;

          // Retrieve frame values
          fLmin = fLmax = sample24L = sample->getFrame24 (frame,0);
          for (int tempi=frame+1; tempi<nextframe; tempi++)
          {
            sample24L = sample->getFrame24 (tempi,0);
            if (sample24L < fLmin) fLmin = sample24L;
            else if (sample24L > fLmax) fLmax = sample24L;
          }

          // Calculate x and y coordinates
          px = objLeftX + i;
          pyLmin = TOPIXELY (fLmin,sampleBotL);
          pyLmax = TOPIXELY (fLmax,sampleBotL);

          if (stereo)
          {
            // Stereo

            fRmin = fRmax = sample24R = sample->getFrame24 (frame,1);
            for (int tempi=frame+1; tempi<nextframe; tempi++)
            {
              sample24R = sample->getFrame24 (tempi,1);
              if (sample24R < fRmin) fRmin = sample24R;
              else if (sample24R > fRmax) fRmax = sample24R;
            }

            pyRmin = TOPIXELY (fRmin,sampleBotR);
            pyRmax = TOPIXELY (fRmax,sampleBotR);

            if (rangeColour && SPEC->edit != 1)
              drawColour = FL_CYAN;
            else
              drawColour = FL_BLACK;

            fl_set_clipping (objLeftX,sampleTopL,objWidth,sampleHeight);
            
            // Draw (dots, lines or filled) for left
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
              {
                fl_point (px,pyLmin,drawColour);
                fl_point (px,pyLmax,drawColour);
              }
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              fl_line (px,pyLmin,px,pyLmax,drawColour);
              if (!first)
              {
                if (grid < 1)
                {
                  fl_line (oldpx+1,oldpyL,px,pyLmax,drawColour);
                }
                else
                {
                  if (pyLmax < oldpyL)
                    fl_line (oldpx+1,pyLmax,px,oldpyL,drawColour);
                  else if (oldpyLmax < pyLmin)
                    fl_line (oldpx+1,oldpyLmax,px,pyLmin,drawColour);
                }
              }
            }
            else
            {
              // Filled
              fl_line (px,sampleCentreL,px,pyLmin,drawColour);
              fl_line (px,sampleCentreL,px,pyLmax,drawColour);
            }

            if (rangeColour && SPEC->edit != 0)
              drawColour = FL_CYAN;
            else
              drawColour = FL_BLACK;

            fl_set_clipping (objLeftX,sampleTopR,objWidth,sampleHeight);
            
            // Draw (dots, lines or filled) for right
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
              {
                fl_point (px,pyRmin,drawColour);
                fl_point (px,pyRmax,drawColour);
              }
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              fl_line (px,pyRmin,px,pyRmax,drawColour);
              if (!first)
              {
                if (grid < 1)
                {
                  fl_line (oldpx+1,oldpyR,px,pyRmax,drawColour);
                }
                else
                {
                  if (pyRmax < oldpyR)
                    fl_line (oldpx+1,pyRmax,px,oldpyR,drawColour);
                  else if (oldpyRmax < pyRmin)
                    fl_line (oldpx+1,oldpyRmax,px,pyRmin,drawColour);
                }
              }
              else
              {
                first = FALSE;
              }
            }
            else
            {
              // Filled
              fl_line (px,sampleCentreR,px,pyRmin,drawColour);
              fl_line (px,sampleCentreR,px,pyRmax,drawColour);
            }

            fl_unset_clipping ();
            
            oldframe  = frame;
            oldpx     = px;
            oldpyL    = pyLmin;
            oldpyR    = pyRmin;
            oldpyLmax = pyLmax;
            oldpyRmax = pyRmax;
          }
          else if (mono)
          {
            // Mono

            if (rangeColour)
              drawColour = FL_CYAN;
            else
              drawColour = FL_BLACK;

            fl_set_clipping (objLeftX,sampleTopL,objWidth,sampleHeight);
            
            // Draw (dots, lines or filled)
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
              {
                fl_point (px,pyLmin,drawColour);
                fl_point (px,pyLmax,drawColour);
              }
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              fl_line (px,pyLmin,px,pyLmax,drawColour);
              if (!first)
              {
                if (grid < 1)
                {
                  fl_line (oldpx+1,oldpyL,px,pyLmax,drawColour);
                }
                else
                {
                  if (pyLmax < oldpyL)
                    fl_line (oldpx+1,pyLmax,px,oldpyL,drawColour);
                  else if (oldpyLmax < pyLmin)
                    fl_line (oldpx+1,oldpyLmax,px,pyLmin,drawColour);
                }
              }
              else
              {
                first = FALSE;
              }
            }
            else
            {
              // Filled
              fl_line (px,sampleCentreL,px,pyLmin,drawColour);
              fl_line (px,sampleCentreL,px,pyLmax,drawColour);
            }
            
            fl_unset_clipping ();
            
            oldframe  = frame;
            oldpx     = px;
            oldpyL    = pyLmin;
            oldpyLmax = pyLmax;
          }
          else
          {
            // Quadro
            
            int tempi;

            fRmin = fRmax = sample24R = sample->getFrame24 (frame,1);
            for (tempi=frame+1; tempi<nextframe; tempi++)
            {
              sample24R = sample->getFrame24 (tempi,1);
              if (sample24R < fRmin) fRmin = sample24R;
              else if (sample24R > fRmax) fRmax = sample24R;
            }

            f3min = f3max = sample243 = sample->getFrame24 (frame,2);
            for (tempi=frame+1; tempi<nextframe; tempi++)
            {
              sample243 = sample->getFrame24 (tempi,2);
              if (sample243 < f3min) f3min = sample243;
              else if (sample243 > f3max) f3max = sample243;
            }

            f4min = f4max = sample244 = sample->getFrame24 (frame,3);
            for (tempi=frame+1; tempi<nextframe; tempi++)
            {
              sample244 = sample->getFrame24 (tempi,3);
              if (sample244 < f4min) f4min = sample244;
              else if (sample244 > f4max) f4max = sample244;
            }

            pyRmin = TOPIXELY (fRmin,sampleBotR);
            pyRmax = TOPIXELY (fRmax,sampleBotR);
            py3min = TOPIXELY (f3min,sampleBot3);
            py3max = TOPIXELY (f3max,sampleBot3);
            py4min = TOPIXELY (f4min,sampleBot4);
            py4max = TOPIXELY (f4max,sampleBot4);

            if (rangeColour && (SPEC->edit == 2 || SPEC->edit == 0))
              drawColour = FL_CYAN;
            else
              drawColour = FL_BLACK;

            fl_set_clipping (objLeftX,sampleTopL,objWidth,sampleHeight);
            
            // Draw (dots, lines or filled) for left
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
              {
                fl_point (px,pyLmin,drawColour);
                fl_point (px,pyLmax,drawColour);
              }
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              fl_line (px,pyLmin,px,pyLmax,drawColour);
              if (!first)
              {
                if (grid < 1)
                {
                  fl_line (oldpx+1,oldpyL,px,pyLmax,drawColour);
                }
                else
                {
                  if (pyLmax < oldpyL)
                    fl_line (oldpx+1,pyLmax,px,oldpyL,drawColour);
                  else if (oldpyLmax < pyLmin)
                    fl_line (oldpx+1,oldpyLmax,px,pyLmin,drawColour);
                }
              }
            }
            else
            {
              // Filled
              fl_line (px,sampleCentreL,px,pyLmin,drawColour);
              fl_line (px,sampleCentreL,px,pyLmax,drawColour);
            }

            if (rangeColour && (SPEC->edit == 2 || SPEC->edit == 1))
              drawColour = FL_CYAN;
            else
              drawColour = FL_BLACK;

            fl_set_clipping (objLeftX,sampleTopR,objWidth,sampleHeight);
            
            // Draw (dots, lines or filled) for right
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
              {
                fl_point (px,pyRmin,drawColour);
                fl_point (px,pyRmax,drawColour);
              }
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              fl_line (px,pyRmin,px,pyRmax,drawColour);
              if (!first)
              {
                if (grid < 1)
                {
                  fl_line (oldpx+1,oldpyR,px,pyRmax,drawColour);
                }
                else
                {
                  if (pyRmax < oldpyR)
                    fl_line (oldpx+1,pyRmax,px,oldpyR,drawColour);
                  else if (oldpyRmax < pyRmin)
                    fl_line (oldpx+1,oldpyRmax,px,pyRmin,drawColour);
                }
              }
            }
            else
            {
              // Filled
              fl_line (px,sampleCentreR,px,pyRmin,drawColour);
              fl_line (px,sampleCentreR,px,pyRmax,drawColour);
            }

            if (rangeColour && (SPEC->edit == 2 || SPEC->edit == 3))
              drawColour = FL_CYAN;
            else
              drawColour = FL_BLACK;

            fl_set_clipping (objLeftX,sampleTop3,objWidth,sampleHeight);
            
            // Draw (dots, lines or filled) for channel 3
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
              {
                fl_point (px,py3min,drawColour);
                fl_point (px,py3max,drawColour);
              }
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              fl_line (px,py3min,px,py3max,drawColour);
              if (!first)
              {
                if (grid < 1)
                {
                  fl_line (oldpx+1,oldpy3,px,py3max,drawColour);
                }
                else
                {
                  if (py3max < oldpy3)
                    fl_line (oldpx+1,py3max,px,oldpy3,drawColour);
                  else if (oldpy3max < py3min)
                    fl_line (oldpx+1,oldpy3max,px,py3min,drawColour);
                }
              }
            }
            else
            {
              // Filled
              fl_line (px,sampleCentre3,px,py3min,drawColour);
              fl_line (px,sampleCentre3,px,py3max,drawColour);
            }

            if (rangeColour && (SPEC->edit == 2 || SPEC->edit == 4))
              drawColour = FL_CYAN;
            else
              drawColour = FL_BLACK;

            fl_set_clipping (objLeftX,sampleTop4,objWidth,sampleHeight);
            
            // Draw (dots, lines or filled) for channel 4
            if (SPEC->drawmode == 0)
            {
              // Dots
              if (frame != oldframe)
              {
                fl_point (px,py4min,drawColour);
                fl_point (px,py4max,drawColour);
              }
            }
            else if (SPEC->drawmode == 1)
            {
              // Lines
              fl_line (px,py4min,px,py4max,drawColour);
              if (!first)
              {
                if (grid < 1)
                {
                  fl_line (oldpx+1,oldpy4,px,py4max,drawColour);
                }
                else
                {
                  if (py4max < oldpy4)
                    fl_line (oldpx+1,py4max,px,oldpy4,drawColour);
                  else if (oldpy4max < py4min)
                    fl_line (oldpx+1,oldpy4max,px,py4min,drawColour);
                }
              }
              else
              {
                first = FALSE;
              }
            }
            else
            {
              // Filled
              fl_line (px,sampleCentre4,px,py4min,drawColour);
              fl_line (px,sampleCentre4,px,py4max,drawColour);
            }

            fl_unset_clipping ();
            
            oldframe  = frame;
            oldpx     = px;
            oldpyL    = pyLmin;
            oldpyR    = pyRmin;
            oldpy3    = py3min;
            oldpy4    = py4min;
            oldpyLmax = pyLmax;
            oldpyRmax = pyRmax;
            oldpy3max = py3max;
            oldpy4max = py4max;
          }
        }
      }
      
      // Draw loop markers

      if (sample->getSusLoopMode () != AF_LOOP_MODE_NOLOOP)
      {
        int susLoopStart = sample->getSusLoopStart ();
        int susLoopEnd   = sample->getSusLoopEnd ();
        int loopStart = TOPIXELX (susLoopStart);
        int loopEnd   = TOPIXELX (susLoopEnd);
        int loopTop   = objTopY - 10;

        if (susLoopStart >= displayStart && susLoopStart <= displayEnd)
        {
          fl_line (loopStart,loopTop,loopStart,objBotY,FL_RED);
          fl_rect (loopStart-9,loopTop,9,9,FL_RED);
          if (SPEC->grabSusStart)
            fl_rectf (loopStart-8,loopTop+1,8,8,FL_YELLOW);
        }

        if (susLoopEnd >= displayStart && susLoopEnd <= displayEnd)
        {
          fl_line (loopEnd,loopTop,loopEnd,objBotY,FL_RED);
          fl_rect (loopEnd,loopTop,9,9,FL_RED);
          if (SPEC->grabSusEnd)
            fl_rectf (loopEnd+1,loopTop+1,8,8,FL_YELLOW);
        }
      }

      if (sample->getRelLoopMode () != AF_LOOP_MODE_NOLOOP)
      {
        int relLoopStart = sample->getRelLoopStart ();
        int relLoopEnd   = sample->getRelLoopEnd ();
        int loopStart = TOPIXELX (sample->getRelLoopStart ());
        int loopEnd   = TOPIXELX (sample->getRelLoopEnd ());
        int loopBot   = objBotY + 1;

        if (relLoopStart >= displayStart && relLoopStart <= displayEnd)
        {
          fl_line (loopStart,objTopY,loopStart,objBotY+10,FL_GREEN);
          fl_rect (loopStart-9,loopBot,9,9,FL_GREEN);
          if (SPEC->grabRelStart)
            fl_rectf (loopStart-8,loopBot+1,8,8,FL_YELLOW);
        }

        if (relLoopEnd >= displayStart && relLoopEnd <= displayEnd)
        {
          fl_line (loopEnd,objTopY,loopEnd,objBotY+10,FL_GREEN);
          fl_rect (loopEnd,loopBot,9,9,FL_GREEN);
          if (SPEC->grabRelEnd)
            fl_rectf (loopEnd+1,loopBot+1,8,8,FL_YELLOW);
        }
      }
      
      // Draw current position line
      if (sample->getRedraw () &&
        currentPos >= displayStart && currentPos <= displayEnd)
      {
        *oldPosx = TOPIXELX (currentPos);
        fl_drawmode (GXxor);
        fl_line (*oldPosx,objTopY,*oldPosx,objBotY,FL_YELLOW);
        fl_drawmode (GXcopy);
      }
      
      return 0;
    }
      
    case FL_PUSH:
    {
      int displayStart    = sample->getDisplayStart ();
      int displayEnd      = sample->getDisplayEnd ();
      int displayDuration = displayEnd - displayStart;
      int displayVertLow  = sample->getDisplayVertLow ();
      int displayVertHigh = sample->getDisplayVertHigh ();
      int displayHeight   = displayVertHigh - displayVertLow;
      int  width           = (sample->getWidth () == 1) ? MAX8 : MAX16;
      int  objLeftX        = ob->x + BORDERX;
      int  objWidth        = ob->w-(2*BORDERX);
      int  objTopY         = ob->y + BORDERY;
      int  objBotY         = ob->y + ob->h - 1 - BORDERY;      
      int frame;
      int  mono;
      int  stereo;
      int  quadro;
      int val;

      if (!sample->getValid () || displayDuration == 0) return 0;

      if (sample->getSusLoopMode () != AF_LOOP_MODE_NOLOOP
        && my < objTopY)
      {
        int loopStart = sample->getSusLoopStart ();
        int loopEnd   = sample->getSusLoopEnd ();
        int pixelStart = TOPIXELX (loopStart) + 1;
        int pixelEnd   = TOPIXELX (loopEnd) + 1;
        if (mx >= pixelStart-9 && mx <= pixelStart)
          SPEC->grabSusStart = pixelStart - mx + 1;
        else if (mx >= pixelEnd && mx <= pixelEnd+9)
          SPEC->grabSusEnd = mx - pixelEnd + 1;
        fl_redraw_object (ob);
        return 0;
      }

      if (sample->getRelLoopMode () != AF_LOOP_MODE_NOLOOP
        && my > objBotY)
      {
        int loopStart = sample->getRelLoopStart ();
        int loopEnd   = sample->getRelLoopEnd ();
        int pixelStart = TOPIXELX (loopStart) + 1;
        int pixelEnd   = TOPIXELX (loopEnd) + 1;
        if (mx >= pixelStart-9 && mx <= pixelStart)
          SPEC->grabRelStart = pixelStart - mx + 1;
        else if (mx >= pixelEnd && mx <= pixelEnd+9)
          SPEC->grabRelEnd = mx - pixelEnd + 1;
        fl_redraw_object (ob);
        return 0;
      }

      if (my >= objTopY && my <= objBotY)
      {
        if (SPEC->freehand)
        {
          if (displayDuration > objWidth)
          {
            fl_set_button (mainForm->manualEdit,0);
            SPEC->freehand = FALSE;
          }
          else
          {
            int sampleTopL   = 0;
            int sampleBotL   = 0;
            int sampleTopR   = 0;
            int sampleBotR   = 0;
            int sampleTop3   = 0;
            int sampleBot3   = 0;
            int sampleTop4   = 0;
            int sampleBot4   = 0;
            int sampleHeight = 0;
            
            mono   = sample->getChannels () == 1;
            stereo = sample->getChannels () == 2;
            quadro = sample->getChannels () == 4;
            frame  = TOFRAME (mx - objLeftX);
            if (mono)
            {
              if (SPEC->axis)
              {
                sampleTopL   = objTopY;
                sampleBotL   = objBotY - AXISSPACE - MONOGAP;
                sampleHeight = sampleBotL - sampleTopL;
              }
              else
              {
                sampleTopL   = objTopY;
                sampleBotL   = objBotY;
                sampleHeight = sampleBotL - sampleTopL;
              }
            }
            else if (stereo)
            {
              if (SPEC->axis)
              {
                sampleHeight = (objBotY - objTopY - AXISSPACE) / 2;
                sampleTopL   = objTopY;
                sampleBotL   = sampleTopL + sampleHeight;
                sampleBotR   = objBotY;
                sampleTopR   = sampleBotR - sampleHeight;
              }
              else
              {
                sampleHeight = (objBotY - objTopY - STEREOGAP) / 2;
                sampleTopL   = objTopY;
                sampleBotL   = sampleTopL + sampleHeight;
                sampleBotR   = objBotY;
                sampleTopR   = sampleBotR - sampleHeight;
              }
            }
            else
            {
              if (SPEC->axis)
              {
                sampleHeight =
                  (objBotY - objTopY - AXISSPACE - 2 * QUADROGAP) / 4;
                sampleTopL   = objTopY;
                sampleBotL   = objTopY + sampleHeight;
                sampleTopR   = sampleBotL + QUADROGAP;
                sampleBotR   = sampleTopR + sampleHeight;
                sampleBot4   = objBotY;
                sampleTop4   = sampleBot4 - sampleHeight;
                sampleBot3   = sampleTop4 - QUADROGAP;
                sampleTop3   = sampleBot3 - sampleHeight;
              }
              else
              {
                sampleHeight = (objBotY - objTopY - 3 * QUADROGAP) / 4;
                sampleTopL   = objTopY;
                sampleBotL   = objTopY + sampleHeight;
                sampleTopR   = sampleBotL + QUADROGAP;
                sampleBotR   = sampleTopR + sampleHeight;
                sampleBot4   = objBotY;
                sampleTop4   = sampleBot4 - sampleHeight;
                sampleBot3   = sampleTop4 - QUADROGAP;
                sampleTop3   = sampleBot3 - sampleHeight;
              }
            }
            
            if (my >= sampleTopL && my <= sampleBotL)
            {
              val = TOSAMPLE (my,sampleBotL);
              ROUNDSAMPLE(val);
              if (val < -MAX23) val = -MAX23;
              else if (val > MAX23_1) val = MAX23_1;
              sample->setFrame24 (frame,0,val);
              sample->setChanged (TRUE);
              SPEC->oldframe = frame;
              SPEC->oldval   = val;
              SPEC->oldchan  = 0;
            }
            else if ((stereo || quadro) &&
              (my >= sampleTopR && my <= sampleBotR))
            {
              val = TOSAMPLE (my,sampleBotR);
              ROUNDSAMPLE(val);
              if (val < -MAX23) val = -MAX23;
              else if (val > MAX23_1) val = MAX23_1;
              sample->setFrame24 (frame,1,val);
              sample->setChanged (TRUE);
              SPEC->oldframe = frame;
              SPEC->oldval   = val;
              SPEC->oldchan  = 1;
            }
            else if (quadro && (my >= sampleTop3 && my <= sampleBot3))
            {
              val = TOSAMPLE (my,sampleBot3);
              ROUNDSAMPLE(val);
              if (val < -MAX23) val = -MAX23;
              else if (val > MAX23_1) val = MAX23_1;
              sample->setFrame24 (frame,2,val);
              sample->setChanged (TRUE);
              SPEC->oldframe = frame;
              SPEC->oldval   = val;
              SPEC->oldchan  = 2;
            }
            else if (quadro && (my >= sampleTop4 && my <= sampleBot4))
            {
              val = TOSAMPLE (my,sampleBot4);
              ROUNDSAMPLE(val);
              if (val < -MAX23) val = -MAX23;
              else if (val > MAX23_1) val = MAX23_1;
              sample->setFrame24 (frame,3,val);
              sample->setChanged (TRUE);
              SPEC->oldframe = frame;
              SPEC->oldval   = val;
              SPEC->oldchan  = 3;
            }
            
            fl_redraw_object (ob);
            return 0;
          }
        }
        
        *rangeMark = TOFRAMEROUND (mx - objLeftX);
        if (*rangeMark < 0) *rangeMark = 0;
        else if (*rangeMark > displayEnd) *rangeMark = displayEnd;
        
        if (SPEC->edit != 2 && 
          SPEC->autoeditmode && sample->getChannels () != 1)
        {
          int  sampleTopL   = 0;
          int  sampleBotL   = 0;
          int  sampleTopR   = 0;
          int  sampleBotR   = 0;
          int  sampleTop3   = 0;
          int  sampleBot3   = 0;
          int  sampleTop4   = 0;
          int  sampleBot4   = 0;
          int  sampleHeight = 0;

          stereo = sample->getChannels () == 2;
          quadro = sample->getChannels () == 4;

          if (stereo)
          {
            if (SPEC->axis)
            {
              sampleHeight = (objBotY - objTopY - AXISSPACE) / 2;
              sampleTopL   = objTopY;
              sampleBotL   = sampleTopL + sampleHeight;
              sampleBotR   = objBotY;
              sampleTopR   = sampleBotR - sampleHeight;
            }
            else
            {
              sampleHeight = (objBotY - objTopY - STEREOGAP) / 2;
              sampleTopL   = objTopY;
              sampleBotL   = sampleTopL + sampleHeight;
              sampleBotR   = objBotY;
              sampleTopR   = sampleBotR - sampleHeight;
            }
          }
          else
          {
            if (SPEC->axis)
            {
              sampleHeight =
                (objBotY - objTopY - AXISSPACE - 2 * QUADROGAP) / 4;
              sampleTopL   = objTopY;
              sampleBotL   = objTopY + sampleHeight;
              sampleTopR   = sampleBotL + QUADROGAP;
              sampleBotR   = sampleTopR + sampleHeight;
              sampleBot4   = objBotY;
              sampleTop4   = sampleBot4 - sampleHeight;
              sampleBot3   = sampleTop4 - QUADROGAP;
              sampleTop3   = sampleBot3 - sampleHeight;
            }
            else
            {
              sampleHeight = (objBotY - objTopY - 3 * QUADROGAP) / 4;
              sampleTopL   = objTopY;
              sampleBotL   = objTopY + sampleHeight;
              sampleTopR   = sampleBotL + QUADROGAP;
              sampleBotR   = sampleTopR + sampleHeight;
              sampleBot4   = objBotY;
              sampleTop4   = sampleBot4 - sampleHeight;
              sampleBot3   = sampleTop4 - QUADROGAP;
              sampleTop3   = sampleBot3 - sampleHeight;
            }
          }

          if (stereo)
          {
            if (my >= sampleTopL && my <= sampleBotL)
            {
              fl_set_choice (mainForm->editMode,1);
              SPEC->edit = 0;
            }
            else if (my >= sampleTopR && my <= sampleBotR)
            {
              fl_set_choice (mainForm->editMode,2);
              SPEC->edit = 1;
            }
          }
          else
          {
            if (my >= sampleTopL && my <= sampleBotL)
            {
              fl_set_choice (mainForm->editMode,1);
              SPEC->edit = 0;
            }
            else if (my >= sampleTopR && my <= sampleBotR)
            {
              fl_set_choice (mainForm->editMode,2);
              SPEC->edit = 1;
            }
            else if (my >= sampleTop3 && my <= sampleBot3)
            {
              fl_set_choice (mainForm->editMode,3);
              SPEC->edit = 3;
            }
            else if (my >= sampleTop4 && my <= sampleBot4)
            {
              fl_set_choice (mainForm->editMode,4);
              SPEC->edit = 4;
            }
          }
        }

        if (SPEC->rangealt)
        {
          if (key == 1)
          {
            // Dragging start - as if end was first point placed
            // Hence setting of rangeMark

            SPEC->rangeGrab = 1;
            if (!sample->getRangeValid ())
            {
              sample->setRange (*rangeMark,sample->getFrames ());
              *rangeMark = sample->getRangeEnd ();
            }
            else if (*rangeMark > sample->getRangeEnd ())
            {
              if (SPEC->rangecross)
              {
                sample->setRangeStart (*rangeMark);
                *rangeMark = sample->getRangeStart (); // Crossed over
              }
              else
              {
                sample->setRangeStart (sample->getRangeEnd ());
                *rangeMark = sample->getRangeEnd ();
              }
            }
            else
            {
              sample->setRangeStart (*rangeMark);
              *rangeMark = sample->getRangeEnd ();
            }
          }
          else if (key == 3)
          {
            // Dragging end - as if start was first point placed
            // Hence setting of rangeMark

            SPEC->rangeGrab = 2;
            if (!sample->getRangeValid ())
            {
              sample->setRange (0,*rangeMark);
              *rangeMark = sample->getRangeStart ();
            }
            else if (*rangeMark < sample->getRangeStart ())
            {
              if (SPEC->rangecross)
              {
                sample->setRangeEnd (*rangeMark);
                *rangeMark = sample->getRangeEnd (); // Crossed over
              }
              else
              {
                sample->setRangeEnd (sample->getRangeStart ());
                *rangeMark = sample->getRangeStart ();
              }
            }
            else
            {
              sample->setRangeEnd (*rangeMark);
              *rangeMark = sample->getRangeStart ();
            }
          }
        }
        else
        {
          if (key == 1)
          {
            SPEC->rangeGrab = 0;
            sample->setRange (*rangeMark,*rangeMark);
          }
          else if (key == 3)
          {
            if (!sample->getRangeValid ())
            {
              SPEC->rangeGrab = 0;
              sample->setRange (*rangeMark,*rangeMark);
            }
            else
            {
              int curRangeStart = sample->getRangeStart ();
              int curRangeEnd   = sample->getRangeEnd ();
              
              // Catch special case
              if (curRangeStart == curRangeEnd &&
                curRangeStart == *rangeMark)
              {
                SPEC->rangeGrab = 0;
                sample->setRange (*rangeMark,*rangeMark);
              }
              else if (*rangeMark <= ((curRangeStart + curRangeEnd) / 2))
              {
                SPEC->rangeGrab = 1;
                sample->setRangeStart (*rangeMark);
                
                // Dragging start - as if end was first placed
                *rangeMark = sample->getRangeEnd ();
              }
              else
              {
                SPEC->rangeGrab = 2;
                sample->setRangeEnd (*rangeMark);
                
                // Dragging end - as if start was first placed
                *rangeMark = sample->getRangeStart ();
              }
            }
          }
        }
        
        updateRangeDetails ();
        fl_redraw_object (ob);
        return 0;
      }
      return 0;
    }
      
    case FL_MOUSE:
    {
      int displayStart    = sample->getDisplayStart ();
      int displayEnd      = sample->getDisplayEnd ();
      int displayDuration = displayEnd - displayStart;
      int displayVertLow  = sample->getDisplayVertLow ();
      int displayVertHigh = sample->getDisplayVertHigh ();
      int displayHeight   = displayVertHigh - displayVertLow;
      int width           = (sample->getWidth () == 1) ? MAX8 : MAX16;
      int objLeftX   = ob->x + BORDERX;
//    int objRightX  = ob->x + ob->w - 1 - BORDERX;
      int objTopY    = ob->y + BORDERY;
      int objBotY    = ob->y + ob->h - 1 - BORDERY;
      int objWidth   = ob->w-(2*BORDERX);
      int frame;
      int mono;
      int stereo;
      int quadro;
      int mxx;
      int val;
      
      if (!sample->getValid () || displayDuration == 0) return 0;
      
      if (SPEC->grabSusStart)
        mxx = mx - objLeftX + SPEC->grabSusStart - 1;
      else if (SPEC->grabSusEnd)
        mxx = mx - objLeftX - SPEC->grabSusEnd + 1;
      else if (SPEC->grabRelStart)
        mxx = mx - objLeftX + SPEC->grabRelStart - 1;
      else if (SPEC->grabRelEnd)
        mxx = mx - objLeftX - SPEC->grabRelEnd + 1;
      else if (SPEC->rangeMark != -1)
        mxx = mx - objLeftX;
      else if (SPEC->freehand)
        mxx = mx - objLeftX;
      else return 0;

      if (mxx < - BORDERX / 2)
      {
        sample->scrollL (displayDuration / 30 + 1);
        frame = sample->getDisplayStart ();
        updateDisplayDetails ();
        fl_redraw_object (mainForm->scrollBarSlider);
        fl_redraw_object (mainForm->zoomBarSlider);
      }
      else if (mxx < 0)
      {
        sample->scrollL (displayDuration / 200 + 1);
        frame = sample->getDisplayStart ();
        updateDisplayDetails ();
        fl_redraw_object (mainForm->scrollBarSlider);
        fl_redraw_object (mainForm->zoomBarSlider);
      }
      else if (mxx > objWidth + BORDERX / 2)
      {
        sample->scrollR (displayDuration / 30 + 1);
        frame = sample->getDisplayEnd ();
        updateDisplayDetails ();
        fl_redraw_object (mainForm->scrollBarSlider);
        fl_redraw_object (mainForm->zoomBarSlider);
      }
      else if (mxx >= objWidth)
      {
        sample->scrollR (displayDuration / 200 + 1);
        frame = sample->getDisplayEnd ();
        updateDisplayDetails ();
        fl_redraw_object (mainForm->scrollBarSlider);
        fl_redraw_object (mainForm->zoomBarSlider);
      }
      else
      {
        if (SPEC->grabSusStart || SPEC->grabSusEnd ||
          SPEC->grabRelStart || SPEC->grabRelEnd)
          frame = TOFRAMEROUND (mxx);
        else if (SPEC->freehand)
          frame = TOFRAME (mxx);
        else
          frame = TOFRAMEROUND (mxx);
      }
        
      char tempString [50];
      sprintf (tempString,"%d",frame);
      fl_set_object_label (displayForm->positionText,tempString);
      sprintf (tempString,"%.3f",frame / sample->getRate ());
      fl_set_object_label (displayForm->timeText,tempString);
      
      if (SPEC->grabSusStart)
      {
        sample->setSusLoopStart (frame);
        updateLoopDetails ();
        fl_redraw_object (ob);
        return 0;
      }

      if (SPEC->grabSusEnd)
      {
        sample->setSusLoopEnd (frame);
        updateLoopDetails ();
        fl_redraw_object (ob);
        return 0;
      }

      if (SPEC->grabRelStart)
      {
        sample->setRelLoopStart (frame);
        updateLoopDetails ();
        fl_redraw_object (ob);
        return 0;
      }

      if (SPEC->grabRelEnd)
      {
        sample->setRelLoopEnd (frame);
        updateLoopDetails ();
        fl_redraw_object (ob);
        return 0;
      }

      if (SPEC->freehand)
      {
        if (displayDuration > objWidth)
        {
          fl_set_button (mainForm->manualEdit,0);
          SPEC->freehand = FALSE;
        }
        else
        {
          int  sampleTopL   = 0;
          int  sampleBotL   = 0;
          int  sampleTopR   = 0;
          int  sampleBotR   = 0;
          int  sampleTop3   = 0;
          int  sampleBot3   = 0;
          int  sampleTop4   = 0;
          int  sampleBot4   = 0;
          int  sampleHeight = 0;

          mono   = sample->getChannels () == 1;
          stereo = sample->getChannels () == 2;
          quadro = sample->getChannels () == 4;
          if (mono)
          {
            if (SPEC->axis)
            {
              sampleTopL   = objTopY;
              sampleBotL   = objBotY - AXISSPACE - MONOGAP;
              sampleHeight = sampleBotL - sampleTopL;
            }
            else
            {
              sampleTopL   = objTopY;
              sampleBotL   = objBotY;
              sampleHeight = sampleBotL - sampleTopL;
            }
          }
          else if (stereo)
          {
            if (SPEC->axis)
            {
              sampleHeight = (objBotY - objTopY - AXISSPACE) / 2;
              sampleTopL   = objTopY;
              sampleBotL   = sampleTopL + sampleHeight;
              sampleBotR   = objBotY;
              sampleTopR   = sampleBotR - sampleHeight;
            }
            else
            {
              sampleHeight = (objBotY - objTopY - STEREOGAP) / 2;
              sampleTopL   = objTopY;
              sampleBotL   = sampleTopL + sampleHeight;
              sampleBotR   = objBotY;
              sampleTopR   = sampleBotR - sampleHeight;
            }
          }
          else
          {
            if (SPEC->axis)
            {
              sampleHeight =
                (objBotY - objTopY - AXISSPACE - 2 * QUADROGAP) / 4;
              sampleTopL   = objTopY;
              sampleBotL   = objTopY + sampleHeight;
              sampleTopR   = sampleBotL + QUADROGAP;
              sampleBotR   = sampleTopR + sampleHeight;
              sampleBot4   = objBotY;
              sampleTop4   = sampleBot4 - sampleHeight;
              sampleBot3   = sampleTop4 - QUADROGAP;
              sampleTop3   = sampleBot3 - sampleHeight;
            }
            else
            {
              sampleHeight = (objBotY - objTopY - 3 * QUADROGAP) / 4;
              sampleTopL   = objTopY;
              sampleBotL   = objTopY + sampleHeight;
              sampleTopR   = sampleBotL + QUADROGAP;
              sampleBotR   = sampleTopR + sampleHeight;
              sampleBot4   = objBotY;
              sampleTop4   = sampleBot4 - sampleHeight;
              sampleBot3   = sampleTop4 - QUADROGAP;
              sampleTop3   = sampleBot3 - sampleHeight;
            }
          }

          if (SPEC->oldframe != -1 && SPEC->oldchan == 0)
          {
            if (my < sampleTopL)
              val = displayVertHigh;
            else if (my > sampleBotL)
              val = displayVertLow;
            else
              val = TOSAMPLE (my,sampleBotL);
            ROUNDSAMPLE(val);
            if (val < -MAX23) val = -MAX23;
            else if (val > MAX23_1) val = MAX23_1;
            sample->setFrame24 (frame,0,val);
            sample->setChanged (TRUE);
            if (SPEC->oldframe != -1 && SPEC->oldframe != frame
            && SPEC->oldchan == 0)
            {
              int i;
              int j;
              int l;
              double x;
              double y;
              if (frame > SPEC->oldframe)
              {
                l = frame - SPEC->oldframe;
                for (i=SPEC->oldframe+1,j=1; i<frame; i++,j++)
                {
                  x = j / l;
                  y = 1.0 - x;
                  sample->setFrame24 (i,0,(int) (x*val + y*SPEC->oldval));
                  sample->setChanged (TRUE);
                }
              }
              else
              {
                l = SPEC->oldframe - frame;
                for (i=frame+1,j=1; i<SPEC->oldframe; i++,j++)
                {
                  x = j / l;
                  y = 1.0 - x;
                  sample->setFrame24 (i,0,(int) (x*SPEC->oldval + y*val));
                  sample->setChanged (TRUE);
                }
              }
            }
            SPEC->oldframe = frame;
            SPEC->oldval   = val;
            SPEC->oldchan  = 0;
          }
          else if ((stereo || quadro) &&
            (SPEC->oldframe != -1 && SPEC->oldchan == 1))
          {
            if (my < sampleTopR)
              val = displayVertHigh;
            else if (my > sampleBotR)
              val = displayVertLow;
            else
              val = TOSAMPLE (my,sampleBotR);
            ROUNDSAMPLE(val);
            if (val < -MAX23) val = -MAX23;
            else if (val > MAX23_1) val = MAX23_1;
            sample->setFrame24 (frame,1,val);
            sample->setChanged (TRUE);
            if (SPEC->oldframe != -1 && SPEC->oldframe != frame
            && SPEC->oldchan == 1)
            {
              int i;
              int j;
              int l;
              double x;
              double y;
              if (frame > SPEC->oldframe)
              {
                l = frame - SPEC->oldframe;
                for (i=SPEC->oldframe+1,j=1; i<frame; i++,j++)
                {
                  x = j / l;
                  y = 1.0 - x;
                  sample->setFrame24 (i,1,(int) (x*val + y*SPEC->oldval));
                  sample->setChanged (TRUE);
                }
              }
              else
              {
                l = SPEC->oldframe - frame;
                for (i=frame+1,j=1; i<SPEC->oldframe; i++,j++)
                {
                  x = j / l;
                  y = 1.0 - x;
                  sample->setFrame24 (i,1,(int) (x*SPEC->oldval + y*val));
                  sample->setChanged (TRUE);
                }
              }
            }
            SPEC->oldframe = frame;
            SPEC->oldval   = val;
            SPEC->oldchan  = 1;
          }
          else if (quadro &&
            (SPEC->oldframe != -1 && SPEC->oldchan == 2))
          {
            if (my < sampleTop3)
              val = displayVertHigh;
            else if (my > sampleBot3)
              val = displayVertLow;
            else
              val = TOSAMPLE (my,sampleBot3);
            ROUNDSAMPLE(val);
            if (val < -MAX23) val = -MAX23;
            else if (val > MAX23_1) val = MAX23_1;
            sample->setFrame24 (frame,2,val);
            sample->setChanged (TRUE);
            if (SPEC->oldframe != -1 && SPEC->oldframe != frame
            && SPEC->oldchan == 2)
            {
              int i;
              int j;
              int l;
              double x;
              double y;
              if (frame > SPEC->oldframe)
              {
                l = frame - SPEC->oldframe;
                for (i=SPEC->oldframe+1,j=1; i<frame; i++,j++)
                {
                  x = j / l;
                  y = 1.0 - x;
                  sample->setFrame24 (i,2,(int) (x*val + y*SPEC->oldval));
                  sample->setChanged (TRUE);
                }
              }
              else
              {
                l = SPEC->oldframe - frame;
                for (i=frame+1,j=1; i<SPEC->oldframe; i++,j++)
                {
                  x = j / l;
                  y = 1.0 - x;
                  sample->setFrame24 (i,2,(int) (x*SPEC->oldval + y*val));
                  sample->setChanged (TRUE);
                }
              }
            }
            SPEC->oldframe = frame;
            SPEC->oldval   = val;
            SPEC->oldchan  = 2;
          }
          else if (quadro &&
            (SPEC->oldframe != -1 && SPEC->oldchan == 3))
          {
            if (my < sampleTop4)
              val = displayVertHigh;
            else if (my > sampleBot4)
              val = displayVertLow;
            else
              val = TOSAMPLE (my,sampleBot4);
            ROUNDSAMPLE(val);
            if (val < -MAX23) val = -MAX23;
            else if (val > MAX23_1) val = MAX23_1;
            sample->setFrame24 (frame,3,val);
            sample->setChanged (TRUE);
            if (SPEC->oldframe != -1 && SPEC->oldframe != frame
            && SPEC->oldchan == 3)
            {
              int i;
              int j;
              int l;
              double x;
              double y;
              if (frame > SPEC->oldframe)
              {
                l = frame - SPEC->oldframe;
                for (i=SPEC->oldframe+1,j=1; i<frame; i++,j++)
                {
                  x = j / l;
                  y = 1.0 - x;
                  sample->setFrame24 (i,3,(int) (x*val + y*SPEC->oldval));
                  sample->setChanged (TRUE);
                }
              }
              else
              {
                l = SPEC->oldframe - frame;
                for (i=frame+1,j=1; i<SPEC->oldframe; i++,j++)
                {
                  x = j / l;
                  y = 1.0 - x;
                  sample->setFrame24 (i,3,(int) (x*SPEC->oldval + y*val));
                  sample->setChanged (TRUE);
                }
              }
            }
            SPEC->oldframe = frame;
            SPEC->oldval   = val;
            SPEC->oldchan  = 3;
          }
          else
          {
            SPEC->oldframe = -1;
            SPEC->oldval   = 0;
            SPEC->oldchan  = 0;
          }

          fl_redraw_object (ob);
          return 0;
        }
      }
          
      if (SPEC->rangeMark != -1)
      {
        if (SPEC->rangeGrab == 1)
        {
          if (frame >= *rangeMark)
          {
            if (SPEC->rangecross)
            {
              if (frame >= *rangeMark)
                sample->setRange (*rangeMark,frame);
              else
                sample->setRange (frame,*rangeMark);
            }
            else
              sample->setRangeStart (sample->getRangeEnd ());
          }
          else
          {
            sample->setRange (frame,*rangeMark);
          }
        }
        else if (SPEC->rangeGrab == 2)
        {
          if (frame <= *rangeMark)
          {
            if (SPEC->rangecross)
            {
              if (frame >= *rangeMark)
                sample->setRange (*rangeMark,frame);
              else
                sample->setRange (frame,*rangeMark);
            }
            else
              sample->setRangeEnd (sample->getRangeStart ());
          }
          else
          {
            sample->setRange (*rangeMark,frame);
          }
        }
        else
        {
          if (frame >= *rangeMark)
            sample->setRange (*rangeMark,frame);
          else
            sample->setRange (frame,*rangeMark);
        }

        updateRangeDetails ();
        fl_redraw_object (ob);
        return 0;
      }
      return 0;
    }
    
    case FL_RELEASE:
    {
      SPEC->rangeMark    = -1;
      SPEC->rangeGrab    = 0;
      SPEC->grabSusStart = 0;
      SPEC->grabSusEnd   = 0;
      SPEC->grabRelStart = 0;
      SPEC->grabRelEnd   = 0;
      SPEC->oldframe     = -1;
      SPEC->oldval       = 0;
      SPEC->oldchan      = 0;
      fl_redraw_object (ob);
      return 0;
    }
    
    case FL_ENTER :
    case FL_MOTION :
    {
      int displayStart    = sample->getDisplayStart ();
      int displayEnd      = sample->getDisplayEnd ();
      int displayDuration = displayEnd - displayStart;

      int objLeftX   = ob->x + BORDERX;
      int objRightX  = ob->x + ob->w - 1 - BORDERX;
      int objTopY    = ob->y + BORDERY;
      int objBotY    = ob->y + ob->h - 1 - BORDERY;      
      int objWidth   = ob->w-(2*BORDERX);
      int frame;
      
      if (!sample->getValid () || displayDuration == 0) return 0;
      
      if (mx < objLeftX || mx > objRightX
        || my < objTopY || my > objBotY)
      {
        fl_set_object_label (displayForm->positionText,"");
        fl_set_object_label (displayForm->timeText,"");
       return 0;
      }

      frame = TOFRAME (mx - objLeftX);
      char tempString [50];
      sprintf (tempString,"%d",frame);
      fl_set_object_label (displayForm->positionText,tempString);
      sprintf (tempString,"%.3f",frame / sample->getRate ());
      fl_set_object_label (displayForm->timeText,tempString);
      return 0;
    }
    
    case FL_LEAVE :
    {
      fl_set_object_label (displayForm->positionText,"");
      fl_set_object_label (displayForm->timeText,"");
      return 0;
    }
    
    case FL_FREEMEM:
    {
      fl_free ((sampleSpec *)ob->spec);
      return 0;
    }
  }
  return 0;
}

/***************************************************************************/
