/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : DPSample.h
|
| Object      : DPSample
|
| Description : Prototypes for the DPSample class
|
| (c) Richard Kent 1996
|
| $Id: DPSample.h,v 1.1 2003/09/10 00:06:25 rk Exp $
|
****************************************************************************/

class DPSample;

#ifndef _DPSample_h
#define _DPSample_h

#include "DPDefs.h"

// Stop and release callback functions
void DPSampleStopCatch (int);
void DPSampleReleaseCatch (int);

// Replacement audiofile void handler
void DPSampleAudiofileError (long,const char *);

// Prototypes for general routines
int getInputRate ();
int getOutputRate ();
void limitQueueSize (int channels,int *queueSize);

// Dynamic memory structure
typedef struct extraListEntry
{
  char *data;
  int start;
  int end;
  extraListEntry *next;
} extraListEntry;

class DPSample
{
  public:
  
    // Constructor, destructor and copy methods
    DPSample ();
    ~DPSample ();
    DPSample (DPSample &newSample);
    DPSample &operator= (DPSample &newSample);
    
    // Functions
    int clone (DPSample &newSample);
    int cloneSimple (DPSample &newSample);
    int cloneData (DPSample &newSample);
    char *loadAIFF (char *newFilename);
    char *loadWAV (char *newFilename);
    char *loadRAW (
      char *newFilename,
      int rawChannels,
      int rawDataFormat,
      int rawSignedData,
      int rawEndian);
    char *load (
      char *newFilename,
      int sampleFormat,
      int rawChannels,
      int rawDataFormat,
      int rawSignedData,
      int rawEndian);
    char *saveAIFF (
      char *newFilename,
      int rangeonly,
      int soxcompatible,
      int replaceFilename);
    char *saveWAV (
      char *newFilename,
      int rangeonly,
      int soxcompatible,
      int replaceFilename);
    char *saveRAW (
      char *newFilename,
      int rangeonly,
      int soxcompatible,
      int replaceFilename,
      int rawDataFormat,
      int rawSignedData,
      int rawEndian);
    char *save (
      char *newFilename,
      int rangeonly,
      int soxcompatible,
      int replaceFilename,
      int sampleFormat,
      int rawDataFormat,
      int rawSignedData,
      int rawEndian);
    char *play (int playMode,FL_OBJECT *object);
    char *record (int recordMode,int trigger,FL_OBJECT *object);
    void clear ();
    int fresh (
      double newRate,
      int newWidth,
      int newChannels,
      int frames);
    void basicWavs (int set);

    // Range functions
    void rangeClear ();
    void rangeDisplay ();
    void rangeAll ();
    void setRangeSusLoop ();
    void setRangeRelLoop ();
    void setSusLoopRange ();
    void setRelLoopRange ();
    
    // Time functions
    inline int   timeToFrame (double time);
    inline double frameToTime (int frame);    

    // Zoom functions
    void zoomIn ();
    void zoomOut ();
    void zoomAll ();
    void zoomRange ();
    void zoomWidth (int width);
    
    // Scroll functions
    void scrollLstart ();
    void scrollL (int scrollFrames);
    void scrollR (int scrollFrames);
    void scrollRend ();

    // Vertical zoom functions
    void zoomVertIn (int minHeight);
    void zoomVertOut ();
    void zoomVertAll ();
    void zoomVertHeight (int height);
    
    // Vertical scroll functions
    void scrollVertBottom ();
    void scrollVertDown (int scrollAmount);
    void scrollVertCentre ();
    void scrollVertUp (int scrollAmount);
    void scrollVertTop ();

    // Modifiers
    inline void setChanged (int newChanged);
    inline void setFormat (int newFormat);
    inline void setCompression (int newCompression);
    inline void setRate (double newRate);
    inline void setFilename (char *newFilename);
    inline void setName (char *newName);
    inline void setCopyright (char *newCopyright);
    inline void setAuthor (char *newAuthor);
    inline void setAnnotation (char *newAnnotation);
    inline void setSusLoopMode (int newSusLoopMode);
    inline void setSusLoopStart (int newSusLoopStart);
    inline void setSusLoopEnd (int newSusLoopEnd);
    inline void setSusLoop (int newSusLoopStart,int newSusLoopEnd);
    inline void setRelLoopMode (int newRelLoopMode);
    inline void setRelLoopStart (int newRelLoopStart);
    inline void setRelLoopEnd (int newRelLoopEnd);
    inline void setRelLoop (int newRelLoopStart,int newRelLoopEnd);
    inline void setRangeValid (int newRangeValid);
    inline void setRangeStart (int newRangeStart);
    inline void setRangeEnd (int newRangeEnd);
    inline void setRange (int newRangeStart,int newRangeEnd);
    inline void setDisplayStart (int newDisplayStart);
    inline void setDisplayEnd (int newDisplayEnd);
    inline void setDisplay (int newDisplayStart,int newDisplayEnd);
    inline void setDisplayVertLow (int newDisplayVertLow);
    inline void setDisplayVertHigh (int newDisplayVertHigh);
    inline void setDisplayVert (
      int newDisplayVertLow,
      int newDisplayVertHigh);
    inline void setRedraw (int newRedraw);
    inline void setCurrentPos (int newCurrentPos);
   
    // Accessors
    inline int    getValid () const;
    inline int    getChanged () const;
    inline int    getFormat () const;
    inline int    getCompression () const;
    inline double getRate () const;
    inline int    getWidth () const;
    inline int    getChannels () const;
    inline int    getFrames () const;
    inline int    getSamples () const;
    inline int    getBytes () const;
    inline int    getAESused () const;
    inline unsigned char *getAES ();
    inline char   *getFilename () const;
    inline char   *getName () const;
    inline char   *getCopyright () const;
    inline char   *getAuthor () const;
    inline char   *getAnnotation () const;
    inline int    getSusLoopMode () const;
    inline int    getSusLoopStart () const;
    inline int    getSusLoopEnd () const;
    inline int    getRelLoopMode () const;
    inline int    getRelLoopStart () const;
    inline int    getRelLoopEnd () const;
    inline int    getRangeValid () const;
    inline int    getRangeStart () const;
    inline int    getRangeEnd () const;
    inline int    getDisplayStart () const;
    inline int    getDisplayEnd () const;
    inline int    getDisplayVertLow () const;
    inline int    getDisplayVertHigh () const;
    inline int    getRedraw () const;
    inline int    getCurrentPos () const;
    inline int    getLzero (int frame);
    inline int    getL (int frame,int scrollFrames);
    inline int    getR (int frame,int scrollFrames);
    inline int    getRzero (int frame);
    
    // Frame functions
    inline int    getFrame24 (int frame,int channel);
    inline void   setFrame24 (int frame,int channel,int newFrame);
    inline short  getFrame16 (int frame,int channel);
    inline void   setFrame16 (int frame,int channel,short newFrame);
    inline double getFrameDb (int frame,int channel);
    inline void   setFrameDb (int frame,int channel,double newFrame);
    inline int    sign (int x);
    int           getFrame24Interp (double frame,int channel);
    short         getFrame16Interp (double frame,int channel);
    double        getFrameDbInterp (double frame,int channel);
    inline int    getFrame24Extra (int frame,int channel);
    inline short  getFrame16Extra (int frame,int channel);
    inline double getFrameDbExtra (int frame,int channel);
    inline void   setFrame24Extra (int frame,int channel,int newFrame);
    inline void   setFrame16Extra (int frame,int channel,short newFrame);
    inline void   setFrameDbExtra (int frame,int channel,double newFrame);
    
    // Conversion functions
    void to8bit   ();
    void to16bit  ();
    void toMono   (int convertMode);
    void toStereo (int convertMode);
    void toQuadro (int convertMode);
    
    // Trim function
    void trim ();

    // Cut & paste functions
    void cut (DPSample *clip,int mode);
    void del (int mode);
    void copy (DPSample *clip,int mode);
    void paste (DPSample *clip,int mode);
    void keep (DPSample *clip,int mode);
    void overwrite (DPSample *clip,int mode);
    void mix (
      DPSample *clip,
      double buffVol,
      double clipVol,
      int    mixin,
      int    mixout,
      int    mode);
    void overwriteRange (DPSample *clip,int mode);
    void mixRange (
      DPSample *clip,
      double buffVol,
      double clipVol,
      int    mixin,
      int    mixout,
      int    mode);
    
    // Range functions
    void reverse (int mode);
    void invert (int mode);
    void zero (int mode);
    void channelChange (int channelMode);
    void duplicate (int times,int mode);
    void panorama (int amount,int ltor,int rtol,int panChannels);
    void adjustDC (int amount,int mode);
    void addWorkspace (int newFrames,int mode);
    
    // Amplitude functions
    void percentage (
      double lamount,
      double ramount,
      double amount3,
      double amount4,
      int wrap,
      int mode);
    void normalise (int mode);
    void fadeIn (int mode);
    void fadeOut (int mode);
    void bounce (int bounceMode);
    void ramp (double startamount,double endamount,int wrap,int mode);
    void draw (double *amounts,int *frames,int n,int wrap,int mode);
    void limitValue (int *value,int wrap);
    
    // Resampling functions
    void nonTimeStretch (int newFrames,double newRate,int mode);
    int timeStretch (
      double pitchChange,
      double framesChange,
      double newRate,
      int    lowestFreq,
      int    quality,
      int    mode);
    
    // Dynamic memory functions
    extraListEntry *expand (double newTime);
    int consolidate ();
    void removeExtra ();
        
  private:
  
    // Data
    int    valid;           // true if valid sample held
    int    changed;         // true if sample changed and not saved
    int    format;          // original file format
    int    compression;     // original compresion used
    double rate;            // hertz
    int    width;           // 1=8bit=char 2=16bit=short
    int    channels;        // 1=mono 2=stereo 4=quadro
    int    frames;          // number of sample frames
    int    samples;         // frames  * channels
    int    bytes;           // samples * width
    int    AESused;         // Whether AES data bytes present
    unsigned char AES [24]; // AES data bytes;
    char   *filename;
    char   *name;
    char   *copyright;
    char   *author;
    char   *annotation;
    int    susLoopMode;     // AIFF loop mode
    int    susLoopStart;    // frames
    int    susLoopEnd;      // frames
    int    relLoopMode;     // AIFF loop mode
    int    relLoopStart;    // frames
    int    relLoopEnd;      // frames
    int    rangeValid;      // 1=on 0=off
    int    rangeStart;      // frames
    int    rangeEnd;        // frames
    int    displayStart;    // frames
    int    displayEnd;      // frames
    int    displayVertLow;  // 24 bit signed sample value
    int    displayVertHigh; // 24 bit signed sample value
    int    redraw;          // 0=full 1=line only 2=full + line
    int    currentPos;      // current play/record frame (-1 for none)
    char   *data;           // (width=1)=>(char*) (width=2)=>(short*)
    extraListEntry *extra;
};

// Include inline methods
#include "DPSampleInline.h"

// Resample headers
#include "DPResample.h"

#endif // _DPSample_h

/***************************************************************************/
