/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : DPMixPlay.cc
|
| Object      : None
|
| Description : Functions for the mix play operation
|
| (c) Richard Kent 1996
|
| $Id: DPMixPlay.cc,v 1.1 2003/09/10 00:06:25 rk Exp $
|
****************************************************************************/

static char DPMixPlay_cc [] = "$Id: DPMixPlay.cc,v 1.1 2003/09/10 00:06:25 rk Exp $";

#include "DPMixPlay.h"

// defines

#define MIXPORTNAME         "MixTichPort"
#define MIXCHECKTIME        0.20
#define MIXPLAYQUEUETIME    0.50
#define MIXFILEBUFFERTIME   2.00

// #define SNEAKYCHECK

// Global variables used

playEntry    playList [MIXCHANNELS];
double       mixplayMasterVol;
double       mixplayMasterPan;
double       mixplayMasterVolL; // For efficiency
double       mixplayMasterVolR; // For efficiency
int          DPMixPlayStop;
AFfilehandle mixfile;

char *mixPlay ()
{
  ALconfig audioConfig;
  ALport   audioPort;
  int      queueSize;
  int      channels;
  
  int    intPlayRate;
  double playRate;
  
  intPlayRate = getOutputRate ();
  if (intPlayRate == AL_RATE_UNDEFINED)
    playRate = 48000.0;
  else
    playRate = intPlayRate;
  
  int i;
  
  #ifndef NOFORMS
  for (i=0; i<15; i++)
    fl_set_object_color (mixplayForm->playLight [i],FL_COL1,FL_COL1);
  #endif
  
  // Disable error handler
  ALseterrorhandler (0);

  // Set up new audio configuration 
  audioConfig = ALnewconfig ();
  if (!audioConfig)
    return "Unable to create new audio configuration";
  
  // Number of channels governed by global channel count
  #ifndef NOFORMS
  channels = globalChannels;
  #else
  channels = 2;
  #endif
  
  // Set sample format and width
  ALsetsampfmt (audioConfig,AL_SAMPFMT_TWOSCOMP);
  ALsetwidth (audioConfig,AL_SAMPLE_16);

  // Set number of channels
  ALsetchannels (audioConfig,channels);

  // Set queue size (MIXPLAYQUEUETIME seconds at playback rate)
  queueSize = (int) (playRate * channels * MIXPLAYQUEUETIME);
  limitQueueSize (channels,&queueSize);
  
  if (ALsetqueuesize (audioConfig,queueSize))
  {
    ALfreeconfig (audioConfig);
    return "Unable to establish audio queue";
  }

  // Open new audio port with given configuration
  audioPort = ALopenport (MIXPORTNAME,"w",audioConfig);
  if (!audioPort)
  {
    ALfreeconfig (audioConfig);
    return "Unable to open audio port";
  }
  
  int  checkReset = (int) (playRate * MIXCHECKTIME);
  int  check      = 1;
  int  sum [4];
  short samps [4];
  int  overloaded;
  
  int bufferSizeFrames = (int) (playRate * MIXFILEBUFFERTIME);
  int bufferSizeSamps  = bufferSizeFrames * channels;
  short *buffer;
  
  if (bufferSizeSamps < 4) bufferSizeSamps = 4;
  buffer = new short [bufferSizeSamps];
  DPMixPlayStop  = 0;
  int cur        = 0;
  int end        = bufferSizeSamps;
  int quadro     = (channels == 4);
  int anyPlayed;
  
  #ifdef SNEAKYCHECK
  XEvent newEvent;
  #endif

  while (!DPMixPlayStop)
  {
    // Calculate sample
    sum [0] = 0;
    sum [1] = 0;
    sum [2] = 0;
    sum [3] = 0;
    anyPlayed = 0;
    for (i=0; i<MIXCHANNELS; i++)
    {
      if (playList [i].sample && (playList [i].frame) >= 0)
      {
        sum [0] = (int) (sum [0] + playList [i].volL *
          (playList [i].sample)->getFrame16 (playList [i].frame,0));
        sum [1] = (int) (sum [1] + playList [i].volR *
          (playList [i].sample)->getFrame16 (playList [i].frame,1));

        if (quadro)
        {
          sum [2] = (int) (sum [2] + playList [i].volL *
            (playList [i].sample)->getFrame16 (playList [i].frame,2));
          sum [3] = (int) (sum [3] + playList [i].volR *
            (playList [i].sample)->getFrame16 (playList [i].frame,3));
        }
        anyPlayed++;
      }
    }
    
    sum [0] = (int) (sum [0] * mixplayMasterVolL);
    sum [1] = (int) (sum [1] * mixplayMasterVolR);
    
    if (sum [0] > MAX15_1)
      sum [0] = MAX15_1;
    else if (sum [0] < -MAX15)
      sum [0] = -MAX15;

    if (sum [1] > MAX15_1)
      sum [1] = MAX15_1;
    else if (sum [1] < -MAX15)
      sum [1] = -MAX15;
    
    samps [0] = (short) sum [0];
    samps [1] = (short) sum [1];
    
    overloaded = 0;
    
    if (samps [0] > 30720 || samps [0] < -30720) overloaded++;
    if (samps [1] > 30720 || samps [1] < -30720) overloaded++;
    
    if (quadro)
    {
      sum [2] = (int) (sum [2] * mixplayMasterVolL);
      sum [3] = (int) (sum [3] * mixplayMasterVolR);

      if (sum [2] > MAX15_1)
        sum [2] = MAX15_1;
      else if (sum [2] < -MAX15)
        sum [2] = -MAX15;

      if (sum [3] > MAX15_1)
        sum [3] = MAX15_1;
      else if (sum [3] < -MAX15)
        sum [3] = -MAX15;

      samps [2] = (short) sum [2];
      samps [3] = (short) sum [3];
      
      if (samps [2] > 30720 || samps [2] < -30720) overloaded++;
      if (samps [3] > 30720 || samps [3] < -30720) overloaded++;
    }
    
    #ifndef NOFORMS
    if (overloaded)
      fl_set_object_color (mixplayForm->overloaded,FL_RED,FL_RED);
    else
      fl_set_object_color (mixplayForm->overloaded,FL_COL1,FL_COL1);
    #endif
    
    ALwritesamps (audioPort,samps,channels);

    if (mixfile && buffer)
    {
      buffer [cur++] = samps [0];
      buffer [cur++] = samps [1];
      if (quadro)
      {
        buffer [cur++] = samps [2];
        buffer [cur++] = samps [3];
      }
      
      if (cur == end)
      {
        if (AFwriteframes (mixfile,AF_DEFAULT_TRACK,buffer,cur/channels)
          < cur/channels)
        {
          AFclosefile (mixfile);
          mixfile = 0;
        }
        cur = 0;
      }
    }

    // Update play list
    DPSample *sample;
    int susLoopMode;
    int susLoopStart;
    int susLoopEnd;
    int relLoopMode;
    int relLoopStart;
    int relLoopEnd;
    int frame;

    #ifndef NOFORMS
    if (anyPlayed)
      fl_set_object_color
      (mixplayForm->playLight [8],FL_RED,FL_RED);
    else
      fl_set_object_color
      (mixplayForm->playLight [8],FL_COL1,FL_COL1);
    #endif
    
    for (i=0; i<MIXCHANNELS; i++)
    {
      sample = playList [i].sample;
      frame  = playList [i].frame;
      if (sample && frame >= 0)
      {
        #ifndef NOFORMS
        if (frame == 0)
        {
          fl_set_object_color
          (mixplayForm->playLight [i],FL_RED,FL_RED);
          if (i < 4)
          {
            if (i < 2)
              fl_set_object_color
              (mixplayForm->playLight [i+9],FL_RED,FL_RED);
            fl_set_object_color
            (mixplayForm->playLight [i+11],FL_RED,FL_RED);
          }
        }
        #endif
        
        if (!(playList [i].release))
        {
          susLoopMode  = sample->getSusLoopMode ();
          susLoopStart = sample->getSusLoopStart ();
          susLoopEnd   = sample->getSusLoopEnd ();
          if (susLoopMode == AF_LOOP_MODE_FORW && frame == susLoopEnd - 1)
            frame = susLoopStart - 1;
          else if (susLoopMode == AF_LOOP_MODE_FORWBAKW)
          {
            if (frame == susLoopEnd - 1) playList [i].inc = -1;
            else if (frame == susLoopStart) playList [i].inc = 1;
          }
        }
        else if (playList [i].release == 1)
        {
          relLoopMode  = sample->getRelLoopMode ();
          relLoopStart = sample->getRelLoopStart ();
          relLoopEnd   = sample->getRelLoopEnd ();
          if (relLoopMode == AF_LOOP_MODE_FORW && frame == relLoopEnd - 1)
            frame = relLoopStart - 1;
          else if (relLoopMode == AF_LOOP_MODE_FORWBAKW)
          {
            if (frame == relLoopEnd - 1) playList [i].inc = -1;
            else if (frame == relLoopStart) playList [i].inc = 1;
          }
        }
        playList [i].frame = frame + playList [i].inc;
        if (playList [i].frame >= sample->getFrames ())
        {
          playList [i].frame = -1;
          #ifndef NOFORMS
          fl_set_object_color
          (mixplayForm->playLight [i],FL_COL1,FL_COL1);
          if (i < 4)
          {
            if (i < 2)
              fl_set_object_color
              (mixplayForm->playLight [i+9],FL_COL1,FL_COL1);
            fl_set_object_color
            (mixplayForm->playLight [i+11],FL_COL1,FL_COL1);
          }
          #endif
        }
      }
    }
    
    // Check xforms if required
    
    if (!--check)
    {
      check = checkReset;
      #ifndef NOFORMS
      #ifdef SNEAKYCHECK
      if (XCheckWindowEvent (fl_display,
        mixplayForm->mixplayForm->window,~(int)0,&newEvent) ||
        XCheckWindowEvent (fl_display,
        mixerForm->mixerForm->window,~(int)0,&newEvent))
      {
        XPutBackEvent (fl_display,&newEvent);
        fl_check_forms ();
      }
      #else
      fl_check_forms ();
      #endif // SNEAKYCHECK
      #endif // NOFORMS
    }
  }

  // Write out remainder

  if (mixfile && buffer)
  {
    if (cur > 0)
      AFwriteframes (mixfile,AF_DEFAULT_TRACK,buffer,cur/channels);
    AFclosefile (mixfile);
    mixfile = 0;
  }

  // Wait until port cleared 
  if (!DPMixPlayStop)
  {
    #ifdef LINUX
    ALflush (audioPort,0);
    #else
    while (ALgetfilled (audioPort)) {}
    #endif
  }

  // Close the port, free the configuration and return
  ALcloseport (audioPort);
  ALfreeconfig (audioConfig);  
  
  delete [] buffer;
  
  #ifndef NOFORMS
  fl_set_button (mixplayForm->save,0);
  #endif
  
  return 0;
}

/***************************************************************************/
