/*
 * Copyright (c) 2005 Jacob Meuser <jakemsr@jakemsr.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 * $Id: bsdav_video_hw.c,v 1.4 2005/11/20 00:35:17 jakemsr Exp $
 */

#include "includes.h"

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#include <err.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>

#include "bsdav.h"


int
bsdav_get_vid_formats(int fd)
{
struct meteor_pixfmt pixfmt;
int	i, id;

	for (i = 0; ; i++) {
		pixfmt.index = i;
		if (ioctl(fd, METEORGSUPPIXFMT, &pixfmt) < 0)
			break;
		switch(pixfmt.type) {
		case METEOR_PIXTYPE_RGB:
			if (pixfmt.Bpp == 2) {
				if ((pixfmt.masks[0] == 0x7c00) &&
				    (pixfmt.swap_bytes == 1)) {
					id = bsdav_find_vid_fmt("rv15");
					bsdav_vid_fmts[id].bktr_idx = i;
				} else if ((pixfmt.masks[0] == 0xf800) &&
				    (pixfmt.swap_bytes == 1)) {
					id = bsdav_find_vid_fmt("rv16");
					bsdav_vid_fmts[id].bktr_idx = i;
				}
			}
			break;
		case METEOR_PIXTYPE_YUV:
			if (pixfmt.Bpp == 2) {
				if (pixfmt.swap_bytes == 1) {
					id = bsdav_find_vid_fmt("yv16");
					bsdav_vid_fmts[id].bktr_idx = i;
				}
			}
			break;
		case METEOR_PIXTYPE_YUV_PACKED:
			if (pixfmt.Bpp == 2) {
				if (pixfmt.swap_bytes == 1) {
					id = bsdav_find_vid_fmt("yuy2");
					bsdav_vid_fmts[id].bktr_idx = i;
				} else {
					id = bsdav_find_vid_fmt("uyvy");
					bsdav_vid_fmts[id].bktr_idx = i;
				}
			}
			break;
		case METEOR_PIXTYPE_YUV_12:
			if (pixfmt.Bpp == 2) {
				if (pixfmt.swap_bytes == 1) {
					id = bsdav_find_vid_fmt("i420");
					bsdav_vid_fmts[id].bktr_idx = i;
				}
			}
			break;
		default:
			break;
		}
	}

	return(0);
}


int
bsdav_list_bktr_formats(char *dev, int fd)
{
#ifdef BT848_GETDEV
struct bktr_info bktrinfo;
#endif
int	 i;
int	 dev_was_open = 0;


	if (fd < 0) {
		if ((fd = open(dev, O_RDONLY)) < 0) {
			warn("%s", dev);
			close(fd);
			return(1);
		}
	} else
		dev_was_open = 1;

	if (bsdav_get_vid_formats(fd) != 0) {
		warnx("get video formats failed");
		return(1);
	}

#ifdef BT848_GETDEV
	/* get the name of the device */
	if (ioctl(fd, BT848_GETDEV, &bktrinfo) < 0)
		warn("BT848_GETDEV");
#endif

	if (dev_was_open != 1)
		close(fd);

	printf("Video Formats\n");
#ifdef BT848_GETDEV
	printf("  %s: %s\n", dev, bktrinfo.card_name);
#else
	printf("  %s\n", dev);
#endif
	printf("    %-12s  %16s\n", "name", "bits per pixel");
	for (i = 0; bsdav_vid_fmts[i].name; i++) {
		if (bsdav_vid_fmts[i].bktr_idx != -1)
			printf("    %12s  %16d\n", bsdav_vid_fmts[i].name,
			    bsdav_vid_fmts[i].bpp);
	}

	return(0);
}


int
bsdav_bktr_init(int fd, int norm, int *width, int *height, int fields,
    int format, int source, int *fps, struct bsdav_crop *crop)
{
struct bktr_capture_area caparea;
struct meteor_geomet geomet;
int c;

	/* norm */
	if (ioctl(fd, BT848SFMT, &bsdav_vid_nrms[norm].bktr_id) < 0) {
		warn("BT848SFMT");
		return(1);
	}

        /* geometry */

	/* use the capture area size as default width and height */
	if (ioctl(fd, BT848_GCAPAREA, &caparea) < 0) {
		warn("BT848_GCAPAREA");
		return(1);
	}

	/* cropping */
	caparea.x_offset = crop->l;
	caparea.y_offset = crop->t;
	caparea.x_size -= (crop->l + crop->r);
	caparea.y_size -= (crop->t + crop->b);

	if (ioctl(fd, BT848_SCAPAREA, &caparea) < 0) {
		warn("BT848_SCAPAREA");
		return(1);
	}

	/* use the full capture area */

	if (*width == 0)
		geomet.columns = caparea.x_size;
	else
		geomet.columns = *width;
	if (*height == 0)
		geomet.rows = caparea.y_size;
	else
		geomet.rows = *height;

	/* single field capture means every other row is missing */

	if (fields != 0) {
		geomet.rows = geomet.rows / 2;
		geomet.oformat |= fields;
	}

	/* driver only supports 1 frame */
	geomet.frames = 1;

	/* actual format will be set later */
	geomet.oformat = 0;

	if (ioctl(fd, METEORSETGEO, &geomet) < 0) {
		warn("METEORSETGEO");
		return(1);
	}

	/* set these to the real values, since they may have changed */
	*width = geomet.columns;
	*height = geomet.rows;

	bsdav_get_vid_formats(fd);

	/* encoding */
	if (ioctl(fd, METEORSACTPIXFMT,
	    &bsdav_vid_fmts[format].bktr_idx) < 0) {
		warn("METEORGACTPIXFMT");
		return(1);
	}

	/* source */
	if (ioctl(fd, METEORSINPUT, &bsdav_vid_srcs[source].bktr_id) < 0) {
		warn("METEORSINPUT");
		return(1);
	}

	/* fps */
	if (*fps > 0) {
		if  (ioctl(fd, METEORSFPS, fps) < 0) {
			warn("METEORSFPS");
			return(1);
		}
	} else {
		if  (ioctl(fd, METEORGFPS, fps) < 0) {
			warn("METEORGFPS");
			return(1);
		}
	}

	/* clear buffer at start of capture */
	c = 1;
	if (ioctl(fd, BT848SCBUF, &c) < 0) {
		warn("BT848SCBUF");
		return(1);
	}

	return(0);
}


size_t
bsdav_map_vid_buffer(uint8_t **buf, int fd, int w, int h, int f)
{
size_t	 buf_size;

	buf_size = w * h * bsdav_vid_fmts[f].bpp / 8;

	*buf = mmap(0, buf_size, PROT_READ, MAP_SHARED, fd, 0);
	if (*buf == MAP_FAILED) {
		warn("init_vid_buffer: buf");
		return(0);
	}
	return(buf_size);
}


void
bsdav_unmap_vid_buffer(uint8_t **buf, size_t buf_size)
{
	if (*buf != MAP_FAILED)
		munmap(*buf, buf_size);
	*buf = MAP_FAILED;
}
