/*
 * Copyright (c) 2005 Jacob Meuser <jakemsr@jakemsr.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 * $Id: bsdav_frames.c,v 1.7 2006/03/24 10:22:44 jakemsr Exp $
 */

#include "includes.h"

#include <err.h>
#include <stdlib.h>

#include "bsdav.h"


int
bsdav_write_frame_header(FILE *file, struct bsdav_frame_header *frmhdr)
{
	if (fprintf(file, BSDAV_FRMHDR_SFMT, frmhdr->tssec, frmhdr->tsusec,
	    frmhdr->type, frmhdr->size, frmhdr->num) != BSDAV_FRMHDR_SIZE)
		return(1);

	return(0);
}


int
bsdav_read_frame_header(FILE *file, struct bsdav_frame_header *frmhdr)
{
	if (fscanf(file, BSDAV_FRMHDR_SFMT, &frmhdr->tssec, &frmhdr->tsusec,
	    &frmhdr->type, &frmhdr->size, &frmhdr->num) != BSDAV_FRMHDR_NMEM)
		return(1);

	return(0);
}


void
bsdav_dump_frame_header(struct bsdav_frame_header *frmhdr)
{
	printf("frame header:\n");
	printf("  tssec= %ld\n", frmhdr->tssec);
	printf("  tsusec= %ld\n", frmhdr->tsusec);
	printf("  type = %d\n", frmhdr->type);
	printf("  size = %d\n", frmhdr->size);
	printf("  num = %d\n", frmhdr->num);

	return;
}


int
bsdav_read_frame_data(FILE *file, uint8_t *buffer, size_t len, int skip)
{
int	 ret = 0;

	if (len == 0)
		return(ret);

	if (skip == 1) {
		if (fseek(file, len, SEEK_CUR) != 0) {
			warn("fseek error");
			ret = 1;
		}
	} else {
		if (fread(buffer, len, 1, file) != 1) {
			warn("fread error");
			ret = 1;
		}
	}

	return(ret);
}


int
bsdav_write_frame_data(FILE *file, uint8_t *buffer, size_t len, int skip)
{
int	 ret = 0;

	if (len == 0)
		return(ret);

	if (skip == 1) {
		if (fseek(file, len, SEEK_CUR) != 0)
			ret = 1;
	} else {
		if (fwrite(buffer, len, 1, file) != 1)
			ret = 1;
	}

	return(ret);
}


double
bsdav_probe_frame_rate(FILE *file, long end_pos)
{
struct bsdav_stream_header strhdr;
struct bsdav_frame_header frmhdr;
uint8_t	*buffer;
long	 pos = 0;
double	 fps = 0;
int	 start_frame = -1;
int 	 start_ts = 0;
int	 end_frame = -1;
int	 end_ts = 0;
int	 error = 0;

	/* read stream header */ 
	if (bsdav_read_stream_header(file, &strhdr) != 0) {
		warnx("failed to read bsdav stream header");
		return(0);
	}

	if ((buffer = malloc(strhdr.vidmfs)) == NULL) {
		warn("could not allocate %d bytes", strhdr.vidmfs);
		error = 1;
	}

	while ((pos < end_pos) && (error == 0)) {
		if (bsdav_read_frame_header(file, &frmhdr) != 0) {
			warnx("failed to read bsdav frame header");
			error = 1;
			break;
		}
		if (frmhdr.type == BSDAV_FRMTYP_VIDEO) {
			if (start_frame == -1) {
				start_frame = frmhdr.num;
				start_ts = frmhdr.tssec * 1000000 + frmhdr.tsusec;
			}
			end_frame = frmhdr.num;
			end_ts = frmhdr.tssec * 1000000 + frmhdr.tsusec;
		}
		if (frmhdr.size > 0) {
			if (bsdav_read_frame_data(file, buffer, (size_t)frmhdr.size, 1) != 0) {
				warnx("failed to read bsdav frame data");
				error = 1;
				break;
			}
		}
		pos = ftell(file);
	}

	free(buffer);

	if ((error == 0) && (end_frame != start_frame)) {
		fps = (double)(end_frame - start_frame) * 1000000 /
			(end_ts - start_ts);
	}

	return(fps);
}
