/*
 * Copyright (c) 2005 Jacob Meuser <jakemsr@jakemsr.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 *  $Id: bsdavrec.c,v 1.24 2006/03/30 09:21:26 jakemsr Exp $
 */

#include "includes.h"

#include <sys/param.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/ioctl.h>

#include <ctype.h>
#include <errno.h>
#include <err.h>
#include <fcntl.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "bsdav.h"

#define BKTR_WIDTH_MAX		768
#define BKTR_HEIGHT_MAX		576
#define BKTR_FPS_MAX		30

extern int errno;

volatile sig_atomic_t bsdavrec_quit;
volatile sig_atomic_t bktr_buf_waiting;


char	*bktr_dev;
int	 bktr_fd;
uint8_t	*vid_buffer;
size_t	 vid_buffer_size;
int	 vid_fmt;
int	 vid_width;
int	 vid_height;

char	*audio_dev;
int	 audio_fd;
uint8_t	*aud_buffer;
size_t	 aud_buffer_size;
int	 aud_fmt;
int	 aud_channels;
int	 aud_rate;

int	 bktr_source;
int	 bktr_norm;
int	 bktr_fps;
int	 bktr_fields;

char	*outfile;
FILE	*output;
int	 do_video;
int	 do_audio;
int	 verbose;

extern char *__progname;

void	 usage(void);
void	 catchsignal(int);
int	 audio_grab_buf(void);
int	 stream(void);
void	 cleanup(int);


/* getopt externs */
extern char *optarg;
extern int opterr;
extern int optind;
extern int optopt;
extern int optreset;


void
catchsignal(int signal)
{
	switch (signal) {
	case SIGUSR1:
		bktr_buf_waiting++;
		break;
	case SIGALRM:
		warnx("sigalrm!");
		/* FALLTHROUGH */
	case SIGINT:
		bsdavrec_quit = 1;
		break;
	default:
		break;
	}
}


void
usage(void)
{
	printf("usage: %s [-AVlv] [-C channels] [-E encoding] [-e encoding]\n"
	       "          [-F file] [-f file] [-h height] [-i field]\n"
	       "          [-n norm] [-o output] [-p pidfile]\n"
	       "          [-R rate] [-r rate] [-s source] [-w width]\n",
	    __progname);

	exit(1);
}


void
cleanup(int excode)
{

	if (vid_buffer != MAP_FAILED) {
		munmap(vid_buffer, vid_buffer_size);
		vid_buffer = MAP_FAILED;
	}

	if (output != NULL) {
		fclose(output);
		output = NULL;
	}

	if (bktr_fd >= 0) {
		close(bktr_fd);
		bktr_fd = -1;
	}

	if (audio_fd >= 0) {
		close(audio_fd);
		audio_fd = -1;
	}

	if (excode != 0)
		exit(excode);
}


int
audio_grab_buf(void)
{
size_t	left;
off_t	offset;
ssize_t	received;

	for (left = aud_buffer_size, offset = 0; left > 0;) {
		received = read(audio_fd, aud_buffer + offset, left);
		if (received == 0) {
			/* EOF from /dev/audio ? */
			warnx("audio grab: received == 0");
		}
		if (received < 0) {
			if (errno == EINTR) {
				received = 0;
				/* warnx("audio grab: interrupted"); */
			} else {
				warn("audio grab");
				return(1);
			}
		}
		if (received > left) {
			warnx("read returns more bytes than requested");
			warnx("requested: %llu, returned: %lld",
			    (unsigned long long)left, (long long)received);
			return(1);
		}
		offset += received;
		left -= received;
	}
	return(left);
}


int
write_frame(struct bsdav_frame_header *frmhdr, uint8_t *data)
{
struct timeval tv_now;

	gettimeofday(&tv_now, NULL);
	frmhdr->tssec = tv_now.tv_sec;
	frmhdr->tsusec = tv_now.tv_usec;

	if (bsdav_write_frame_header(output, frmhdr) != 0) {
		warnx("write frame header failed");
		return(1);
	}
	if (bsdav_write_frame_data(output, data, frmhdr->size, 0) != 0) {
		warnx("write frame data failed");
		return(1);
	}

	return(0);
}


int
stream(void)
{
struct bsdav_stream_header strhdr;
struct bsdav_frame_header frmhdr;
struct sigaction act;
struct itimerval tim;
struct timeval start_ts;
struct timeval last_vid_ts;
struct timeval last_aud_ts;
struct timeval vid_tv;
struct timeval aud_tv;
#if defined(HAVE_SUN_AUDIO)
audio_info_t audio_if;
#endif
long	 aud_usec;
long	 vid_usec;
double	 aud_usecs_per_frame;
double	 vid_usecs_per_frame;
useconds_t sleep_usecs;
long	 vid_bufs_written;
long	 aud_bufs_written;
int	 sequence;
int	 samples_per_buffer;
uint	 c;
int	 ret;
int	 count;
int	 got_frame;
int	 audio_buf_waiting;

	samples_per_buffer = 2048;
	sequence = 100;
	ret = 0;
	count = 0;
	bsdavrec_quit = 0;
	vid_bufs_written = 0;
	aud_bufs_written = 0;
	bktr_buf_waiting = 0;
	audio_buf_waiting = 0;
	vid_usecs_per_frame = 0;
	aud_usecs_per_frame = 0;

	if (do_video == 1) {

		/* We should really use the exact bits per pixel.  Since we only
		 * have integers in the kernel, and the bktr developers decided
		 * to use bytes per pixel, the bit-depth information for
		 * METEOR_PIXTYPE_YUV_12 as returned by bktr is inaccurate.
		 * It's 12 bits, or 1.5 bytes.  bktr says it's 2 bytes.
		 * Use bits per pixels here, and we are still only using
		 * integers. 
		 */
		if (verbose > 1)
			warnx("video bits per pixel = %d",
			    bsdav_vid_fmts[vid_fmt].bpp);

		vid_buffer_size = vid_width * vid_height *
		    bsdav_vid_fmts[vid_fmt].bpp / 8;

		if (verbose > 1)
			warnx("video buffer size: %llu", (unsigned long long)vid_buffer_size);

		if ((vid_buffer = mmap(0, vid_buffer_size, PROT_READ,
		    MAP_SHARED, bktr_fd, 0)) == MAP_FAILED) {
			warn("mmap vid buffer, size = %llu",
			    (unsigned long long)vid_buffer_size);
			return(1);
		}

		/* close enough */
		vid_usecs_per_frame = 1000000 / bktr_fps;

		if (verbose > 2)
			warnx("vid_usecs_per_frame = %f", vid_usecs_per_frame);
	}

	if (do_audio == 1) {

		aud_buffer_size = samples_per_buffer *
		    ((bsdav_aud_fmts[aud_fmt].bps / 8) * aud_channels);

		if (verbose > 1) {
			warnx("audio buffer size: %llu", (unsigned long long)aud_buffer_size);
			warnx("audio samples per buffer: %d",
			    samples_per_buffer);
		}

		if ((aud_buffer = (uint8_t *)malloc(aud_buffer_size)) == NULL) {
			warn("malloc audio buffer, size = %llu",
			    (unsigned long long)aud_buffer_size);
			return(1);
		}

		aud_usecs_per_frame = (double)aud_buffer_size * 1000000 * 8 /
		    (aud_rate * bsdav_aud_fmts[aud_fmt].bps * aud_channels);

		if (verbose > 2)
			warnx("aud_usecs_per_frame = %f", aud_usecs_per_frame);
	}

	sleep_usecs = ((aud_usecs_per_frame < vid_usecs_per_frame) ?
	    aud_usecs_per_frame : vid_usecs_per_frame) / 8;

	if (verbose > 2)
		warnx("sleep_usecs = %u", sleep_usecs);

	/* signal handler setup */
	memset(&act, 0, sizeof(act));
	sigemptyset(&act.sa_mask);
	act.sa_handler = catchsignal;
	if (do_video == 1)
		sigaction(SIGUSR1, &act, NULL);
	sigaction(SIGINT, &act, NULL);
	sigaction(SIGALRM, &act, NULL);

	if ((output = fopen(outfile, "w")) == NULL) {
		warn("outfile: %s", outfile);
		return(1);
	}

	/* stream header */
	strhdr.vidfmt = vid_fmt;
	strhdr.vidwth = vid_width;
	strhdr.vidhgt = vid_height;
	strhdr.vidmfs = vid_buffer_size;
	strhdr.audfmt = aud_fmt;
	strhdr.audchn = aud_channels;
	strhdr.audsrt = aud_rate;
	strhdr.audmfs = aud_buffer_size;

	/* timeout timer */
	memset(&tim, 0, sizeof(tim));
	tim.it_value.tv_sec = 1;
	setitimer(ITIMER_REAL, &tim, NULL);

	if (do_video == 1) {
		/* signal when frame captured */
		c = SIGUSR1;
		if (ioctl(bktr_fd, METEORSSIGNAL, &c) < 0) {
			warn("METEORSSIGNAL");
			return(1);
		}

		c = METEOR_CAP_CONTINOUS;
		if (ioctl(bktr_fd, METEORCAPTUR, &c) < 0) {
			warn("METEORCAPTUR");
			return(1);
		}
	} else
		bktr_buf_waiting = 1;

	if (do_audio == 1) {
#if defined(HAVE_SUN_AUDIO)
		AUDIO_INITINFO(&audio_if);
		audio_if.record.pause = 0;
		if (ioctl(audio_fd, AUDIO_SETINFO, &audio_if) < 0) {
			warn("AUDIO_SETINFO");
			return(1);
		}
#elif defined(HAVE_OSS_AUDIO)
		audio_grab_buf();
#endif
	} else
		audio_buf_waiting = 1;

	/* Wait for first signal from both bktr and audio, or until
	 * our timeout timer has gone off.  Notice that if we're not
	 * doing either audio or video, we said there's a buffer of
	 * it waiting above.  This makes the following work, and
	 * they get set to 0 just after this anyway.
	 */

	while ((audio_buf_waiting == 0) || (bktr_buf_waiting == 0)) {
		if (do_audio == 1)
			audio_buf_waiting = bsdav_get_audhw_buf_pos(audio_fd,
			    BSDAV_AUDMODE_REC) / aud_buffer_size;
		if (bsdavrec_quit == 1)
			break;
		usleep(sleep_usecs);
	}

	/* reset the timeout timer */
	setitimer(ITIMER_REAL, &tim, NULL);

	if (bsdavrec_quit == 1) {
		warnx("timeout");
		ret = 1;
	} else {
		if (do_audio == 1) {
			while (audio_buf_waiting > 0) {
				audio_grab_buf();
				audio_buf_waiting--;
			}
		}
		bktr_buf_waiting = 0;
	}

	gettimeofday(&start_ts, NULL);

	strhdr.tssec = start_ts.tv_sec;
	strhdr.tsusec = start_ts.tv_usec;

	if (bsdav_write_stream_header(output, &strhdr) != 0) {
		warnx("failed to write stream header");
		return(1);
	}

	while (bsdavrec_quit == 0) {

		/* reset the timeout timer */
		setitimer(ITIMER_REAL, &tim, NULL);

		got_frame = 0;

		/* clear out the frame header */
		memset(&frmhdr, 0, sizeof(struct bsdav_frame_header));

		if (do_audio == 1)
			audio_buf_waiting = bsdav_get_audhw_buf_pos(audio_fd,
			    BSDAV_AUDMODE_REC) / aud_buffer_size;
		while ((audio_buf_waiting == 0) && (bktr_buf_waiting == 0) &&
		    (bsdavrec_quit == 0)) {
			if (do_audio == 1)
				audio_buf_waiting = bsdav_get_audhw_buf_pos(audio_fd,
				    BSDAV_AUDMODE_REC) / aud_buffer_size;
			usleep(sleep_usecs);
		}

		if (bsdavrec_quit == 1)
			break;

		if (bktr_buf_waiting > 0) {
			frmhdr.type = BSDAV_FRMTYP_VIDEO;
			frmhdr.size = vid_buffer_size;
			frmhdr.num = vid_bufs_written;
			for (c = 0; (c < bktr_buf_waiting) && (c < 3); c++) {
				if (write_frame(&frmhdr, vid_buffer) != 0) {
					bsdavrec_quit = 1;
					break;
				}
				bktr_buf_waiting--;
				vid_bufs_written++;
				last_vid_ts.tv_sec = frmhdr.tssec;
				last_vid_ts.tv_usec = frmhdr.tsusec;
				got_frame++;
			}
			if (c > 1) {
				c -= 1;
				warnx("missed/cloned %d video frames at "
				    "frame %ld  ", c, vid_bufs_written - c);
			}
		}

		if (bsdavrec_quit == 1)
			break;

		while (audio_buf_waiting > 0) {
			if (audio_grab_buf() > 0)
				warnx("incomplete audio read   ");
			audio_buf_waiting--;
			frmhdr.type = BSDAV_FRMTYP_AUDIO;
			frmhdr.size = aud_buffer_size;
			frmhdr.num = aud_bufs_written;
			if (write_frame(&frmhdr, aud_buffer) != 0) {
				bsdavrec_quit = 1;
				break;
			}
			aud_bufs_written++;
			last_aud_ts.tv_sec = frmhdr.tssec;
			last_aud_ts.tv_usec = frmhdr.tsusec;
			got_frame++;
		}

		if (bsdavrec_quit == 1)
			break;

		if (got_frame > 0)
			count++;

		if (count == sequence) {
			count = 0;

			timersub(&last_vid_ts, &start_ts, &vid_tv);
			timersub(&last_aud_ts, &start_ts, &aud_tv);
			vid_usec = vid_tv.tv_sec * 1000000 + vid_tv.tv_usec;
			aud_usec = aud_tv.tv_sec * 1000000 + aud_tv.tv_usec;

			if (verbose > 1) {
				fprintf(stderr, "%s: sec: %08.3f,", __progname,
				    ((do_video == 1) ? vid_usec : aud_usec) /
				    1000000.0);
				if (do_video == 1)
					fprintf(stderr,
					    " vf: %06ld, fps: %0.3f,",
				    	    vid_bufs_written,
					    vid_bufs_written * 1000000.0 /
					    vid_usec);
				if (do_audio == 1)
					fprintf(stderr,
					    " as: %09ld, srate: %ld",
				    	    aud_bufs_written *
					    samples_per_buffer,
					    (long)(aud_bufs_written *
					    samples_per_buffer * 1000000 /
					    aud_usec));
				fprintf(stderr, "\r");
				fflush(stderr);
			}
		}
	}

	/*  last character was probably a carriage return */
	if (verbose > 1)
		fprintf(stderr, "\n");

	/* close down the signal generation */

	tim.it_value.tv_sec = 0;
	setitimer(ITIMER_REAL, &tim, NULL);

	if (do_video == 1) {
		c = METEOR_SIG_MODE_MASK;
		if (ioctl(bktr_fd, METEORSSIGNAL, &c) < 0) {
			warn("METEORSSIGNAL");
			ret = 1;
		}

		c = METEOR_CAP_STOP_CONT;
		if (ioctl(bktr_fd, METEORCAPTUR, &c) < 0) {
			warn("METEORCAPTUR");
			ret = 1;
		}
	}

	act.sa_handler = SIG_DFL;
	if (do_video == 1)
		sigaction(SIGUSR1, &act, NULL);
	sigaction(SIGALRM, &act, NULL);
	sigaction(SIGINT, &act, NULL);

	fclose(output);

	timersub(&last_vid_ts, &start_ts, &vid_tv);
	vid_usec = vid_tv.tv_sec * 1000000 + vid_tv.tv_usec;

	timersub(&last_aud_ts, &start_ts, &aud_tv);
	aud_usec = aud_tv.tv_sec * 1000000 + aud_tv.tv_usec;

	if (verbose > 0) {
		warnx("total time: %f", ((do_video == 1) ? vid_usec : aud_usec) /
		    1000000.0);
		if (do_video == 1) {
			warnx("video: total frames: %ld", vid_bufs_written);
			warnx("video: average fps: %f",
			    (double)vid_bufs_written * 1000000 / vid_usec);
		}
		if (do_audio == 1) {
			warnx("audio: total samples: %ld",
			    aud_bufs_written * samples_per_buffer);
			warnx("audio: average rate: %f Hz",
			    (aud_bufs_written * samples_per_buffer) * 1000000.0 /
			    aud_usec);
		}
	}

	if (do_video == 1) {
		munmap(vid_buffer, vid_buffer_size);
		vid_buffer = MAP_FAILED;
	}

	return(ret);
}


int
main(int argc, char *argv[])
{
struct bsdav_crop crop;
const char *errstr;
char	*pidfile;
int	 lflag;
int	 ch;


	/* defaults */
	bktr_dev = DEFAULT_VIDEO_DEVICE;
	audio_dev = DEFAULT_AUDIO_DEVICE;
	outfile = NULL;
	bktr_fields = 0;
	bktr_source = BSDAV_VIDSOURCE_TUNER;
	bktr_norm = BSDAV_VIDNORM_NTSC;
	vid_fmt = BSDAV_VIDFMT_UYVY;
	aud_fmt = BSDAV_AUDFMT_SLLE;
	aud_rate = 48000;
	aud_channels = 2;
	verbose = 0;
	lflag = 0;
	pidfile = NULL;

	/* for cleanup() */
	bktr_fd = -1;
	audio_fd = -1;
	vid_buffer = MAP_FAILED;  /* mmap() */
	aud_buffer = NULL;	  /* malloc() */
	output = NULL;

	do_video = 1;
	do_audio = 1;

	while ((ch = getopt(argc, argv,
	    "AVvlC:E:e:F:f:h:i:n:o:p:R:r:s:w:")) != -1) {
		switch (ch) {
		case 'A':
			do_audio = 0;
			break;
		case 'C':
			aud_channels = (int)strtonum(optarg, 0, 2, &errstr);
			if (errstr != NULL)
				errx(1, "channels is %s: %s", errstr, optarg);
			break;
		case 'E':
			aud_fmt = bsdav_find_aud_fmt(optarg);
			if (aud_fmt == -1)
				errx(1, "invalid audio encoding: %s", optarg);
			break;
		case 'e':
			vid_fmt = bsdav_find_vid_fmt(optarg);
			if (vid_fmt == -1)
				errx(1, "invalid video encoding: %s", optarg);
			break;
		case 'F':
			audio_dev = optarg;
			break;
		case 'f':
			bktr_dev = optarg;
			break;
		case 'h':
			vid_height = (int)strtonum(optarg, 0, BKTR_HEIGHT_MAX,
			    &errstr);
			if (errstr != NULL)
				errx(1, "height is %s: %s", errstr, optarg);
			break;
		case 'i':
			if (strncmp(optarg, "odd", 3) == 0) {
				bktr_fields = METEOR_GEO_ODD_ONLY;
			} else if (strncmp(optarg, "even", 3) == 0) {
				bktr_fields = METEOR_GEO_EVEN_ONLY;
			} else
				errx(1, "invalid fields: %s", optarg);
			break;
		case 'l':
			lflag++;
			break;
		case 'n':
			bktr_norm = bsdav_find_vid_norm(optarg);
			if (bktr_norm < 0)
				errx(1, "invalid video norm: %s", optarg);
			break;
		case 'o':
			outfile = optarg;
			break;
		case 'p':
			pidfile = optarg;
			break;
		case 'R':
			aud_rate = (int)strtonum(optarg, 0, 96000, &errstr);
			if (errstr != NULL)
				errx(1, "rate is %s: %s", errstr, optarg);
			break;
		case 'r':
			bktr_fps = (int)strtonum(optarg, 0, BKTR_FPS_MAX, &errstr);
			if (errstr != NULL)
				errx(1, "rate is %s: %s", errstr, optarg);
			break;
		case 's':
			bktr_source = bsdav_find_vid_source(optarg);
			if (bktr_source < 0)
				errx(1, "invalid video source: %s", optarg);
			break;
		case 'V':
			do_video = 0;
			break;
		case 'v':
			verbose++;
			break;
		case 'w':
			vid_width = (int)strtonum(optarg, 0, BKTR_WIDTH_MAX,
			    &errstr);
			if (errstr != NULL)
				errx(1, "width is %s: %s", errstr, optarg);
			break;
		default:
			usage();
		}
	}
	argc -= optind;
	argv += optind;

	if (lflag > 0) {
		printf("\n");
		bsdav_list_audio_formats(audio_dev, audio_fd);
		printf("\n");
		bsdav_list_bktr_formats(bktr_dev, bktr_fd);
		printf("\n");
	} else {
		if (outfile == NULL) {
			warnx("no output file specified, exiting");
			cleanup(1);
		}

		if ((do_audio == 0) && (do_video == 0)) {
			warnx("recording neither audio nor video, exiting");
			cleanup(1);
		}

		if (do_video == 1) {
			if ((bktr_fd = open(bktr_dev, O_RDONLY)) < 0) {
				warn("%s", bktr_dev);
				cleanup(1);
			}
			memset(&crop, 0, sizeof(struct bsdav_crop));
			if (bsdav_bktr_init(bktr_fd, bktr_norm, &vid_width,
			    &vid_height, bktr_fields, vid_fmt, bktr_source,
			    &bktr_fps, &crop) != 0)
				cleanup(1);

			if (verbose > 0) {
				warnx("video width = %d", vid_width);
				warnx("video height = %d", vid_height);
				warnx("video encoding = %s",
				    bsdav_vid_fmts[vid_fmt].name);
				warnx("video fps = %d", bktr_fps);
			}
			if (verbose > 1) {
				warnx("video norm = %s",
				    bsdav_vid_nrms[bktr_norm].name);
				warnx("video source = %s",
				    bsdav_vid_srcs[bktr_source].name);
				if (bktr_fields == METEOR_GEO_EVEN_ONLY)
					warnx("video fields = even only");
				else if (bktr_fields == METEOR_GEO_ODD_ONLY)
					warnx("video fields = odd only");
				else
					warnx("video fields = interlaced");
			}
		}
		if (do_audio == 1) {
			if ((audio_fd = open(audio_dev, O_RDONLY)) < 0) {
				warn("%s", audio_dev);
				cleanup(1);
			}
			if (bsdav_audio_init(audio_fd, BSDAV_AUDMODE_REC,
			    aud_fmt, aud_channels, aud_rate) != 0) {
				warnx("could not set audio format");
				cleanup(1);
			}
			if (verbose > 1) {
				warnx("audio format = %s",
				    bsdav_aud_fmts[aud_fmt].name);
				warnx("audio channels = %d", aud_channels);
				warnx("audio sample rate = %d", aud_rate);
			}
		}
		if (pidfile != NULL) {
			if (bsdav_write_pid(pidfile) != 0) {
				warnx("couldn't write pid file");
				cleanup(1);
			}
		}
		if (stream() != 0)
			cleanup(1);
	}

	cleanup(0);

	return (0);
}
