/* b2mng
 * Creates MNG animations from Blinkenlights movies.
 *
 * Copyright (C) 2002  Sven Neumann <sven@gimp.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <zlib.h>

#define MNG_SUPPORT_WRITE
#define MNG_ACCESS_CHUNKS
#include <libmng.h>

#include <blib/blib.h>

#include "bmovie-mng.h"


typedef struct
{
  FILE   *stream;
} UserData;

static mng_bool
mng_cb_writedata (mng_handle  hMNG,
                  mng_ptr     pBuf,
                  mng_uint32  iSize,
                  mng_uint32 *iWritten)
{ 
  UserData *data = (UserData *) mng_get_userdata (hMNG);

  *iWritten = fwrite (pBuf, 1, iSize, data->stream);

  return MNG_TRUE;
}

static mng_ptr
mng_cb_alloc (mng_size_t iSize)
{
  return (mng_ptr) g_malloc0 (iSize);
}

static void
mng_cb_free (mng_ptr    pPtr,
             mng_size_t iSize)
{
  g_free (pPtr);
}

static mng_bool
mng_cb_openstream (mng_handle hMNG)
{
  return MNG_TRUE;
}

static mng_bool
mng_cb_closestream (mng_handle hMNG)
{
  return MNG_TRUE;
}

static gpointer
zlib_compress_frame (const guchar *data,
                     gint          width,
                     gint          height,
                     gint          rowstride,
                     gulong       *len)
{
  static gchar  *z_in  = NULL;
  static gchar  *z_out = NULL;
  static gulong  z_len = 0;
  gulong  size;
  gint    row;
  guchar *dest;

  size = (width + 1) * height;

  if (size > z_len)
    {
      z_len = size;
      z_in  = g_realloc (z_in,  z_len);
      z_out = g_realloc (z_out, z_len + 20);
    }

  *len = z_len + 20;

  for (row = 0, dest = z_in; row < height; row++)
    {
      *dest = 0;  /* zero filter-byte */
      dest++;

      memcpy (dest, data, width);

      data += rowstride;
      dest += width;
    }
  
  compress2 (z_out, len, z_in, z_len, Z_BEST_COMPRESSION);

  return z_out;
}

static void
mng_put_simple_frame_data (mng_handle    hMNG,
                           BMovie       *movie,
                           BRectangle   *rect,
                           const guchar *data)
{
  gpointer    idat;
  gulong      len;
  BRectangle  area = { 0, 0, movie->width, movie->height };

  if (rect)
    area = *rect;

  if (!area.w || !area.h)
    return;

  if (area.x || area.y)
    mng_putchunk_defi (hMNG,
                       0, MNG_DONOTSHOW_VISIBLE, MNG_ABSTRACT,
                       MNG_TRUE, area.x, area.y,
                       MNG_FALSE, 0, 0, 0, 0);

  mng_putchunk_ihdr (hMNG,
                     area.w, area.h, 8, MNG_COLORTYPE_GRAY,
                     MNG_COMPRESSION_DEFLATE, MNG_FILTER_ADAPTIVE,
                     MNG_INTERLACE_NONE);
  
  idat = zlib_compress_frame (data + area.y * movie->width + area.x,
                              area.w, area.h, movie->width, &len);
  
  mng_putchunk_idat (hMNG, len, idat);
  mng_putchunk_iend (hMNG);
}

static void
b_movie_create_simple_mng (BMovie     *movie,
                           mng_handle  hMNG)
{
  GList       *list;
  BMovieFrame *frame;
  gint         delay;

  /* create a new MNG */
  mng_create (hMNG);

  /* the MNG header */
  mng_putchunk_mhdr (hMNG,
                     movie->width, movie->height,
                     1000,
                     movie->n_frames + 1, movie->n_frames, movie->duration,
                     (MNG_SIMPLICITY_VALID |
                      MNG_SIMPLICITY_SIMPLEFEATURES |
                      MNG_SIMPLICITY_COMPLEXFEATURES));

  /* a comment */
  mng_putchunk_text (hMNG,
                     5, "title", 19, "Blinkenlights Movie");

  /* to loop or not to loop */
  if (movie->loop)
    mng_putchunk_term (hMNG,
                       MNG_TERMACTION_REPEAT, MNG_ITERACTION_CLEAR, 0,
                       1024); /* quite often */
  else
    mng_putchunk_term (hMNG,
                       MNG_TERMACTION_CLEAR, MNG_ITERACTION_CLEAR, 0, 0);

  /* a black background */
  mng_putchunk_back (hMNG,
                     0x00, 0x00, 0x00, MNG_BACKGROUNDCOLOR_MANDATORY,
                     0, MNG_BACKGROUNDIMAGE_NOTILE);

  /* the first frame */
  list  = movie->frames;
  frame = list->data;
  delay = frame->duration;

  mng_putchunk_fram (hMNG,
                     MNG_FALSE, MNG_FRAMINGMODE_1,
                     0, MNG_NULL,
                     MNG_CHANGEDELAY_DEFAULT,
                     MNG_CHANGETIMOUT_NO,
                     MNG_CHANGECLIPPING_NO,
                     MNG_CHANGESYNCID_NO,
                     delay, 0,
                     0, 0, 0, 0, 0, 0, 0);

  mng_put_simple_frame_data (hMNG, movie, NULL, frame->data);

  /* subsequent frames */
  for (list = list->next; list; list = list->next)
    {
      frame = list->data;

      if (frame->duration != delay)
        {
          delay = frame->duration;
          mng_putchunk_fram (hMNG,
                             MNG_FALSE, MNG_FRAMINGMODE_NOCHANGE,
                             0, MNG_NULL,
                             MNG_CHANGEDELAY_DEFAULT,
                             MNG_CHANGETIMOUT_NO,
                             MNG_CHANGECLIPPING_NO,
                             MNG_CHANGESYNCID_NO,
                             delay, 0,
                             0, 0, 0, 0, 0, 0, 0);
        }

      mng_put_simple_frame_data (hMNG, movie, NULL, frame->data);
    }

  /* the MNG footer */
  mng_putchunk_mend (hMNG);
}

static void
b_movie_create_themed_mng (BMovie     *movie,
                           BTheme     *theme,
                           mng_handle  hMNG)
{
  g_printerr ("Sorry, themed MNG animations are not yet supported!\n");
  return b_movie_create_simple_mng (movie, hMNG);
}

gboolean
b_movie_save_as_mng (BMovie  *movie,
                     BTheme  *theme,
                     FILE    *stream,
                     GError **error)
{
  mng_handle   hMNG;
  UserData     data;

  g_return_val_if_fail (B_IS_MOVIE (movie), FALSE);
  g_return_val_if_fail (theme == NULL || B_IS_THEME (theme), FALSE);

  data.stream = stream;
  
  hMNG = mng_initialize ((mng_ptr) &data, mng_cb_alloc, mng_cb_free, MNG_NULL);

  if (!hMNG)
    goto error;
  
  if (mng_setcb_openstream  (hMNG, mng_cb_openstream)  ||
      mng_setcb_closestream (hMNG, mng_cb_closestream) ||
      mng_setcb_writedata   (hMNG, mng_cb_writedata))
    goto error;

  if (theme)
    b_movie_create_themed_mng (movie, theme, hMNG);
  else
    b_movie_create_simple_mng (movie, hMNG);

  if (mng_write (hMNG))
    goto error;

  mng_cleanup (&hMNG);

  return TRUE;
  
 error:
  g_set_error (error, 0, 0,
               "Unexpected problems creating the MNG.");
  return FALSE;
}
