/* bsend - plays Blinkenlights movie files over the net
 *
 * Copyright (c) 2001-2002  Sven Neumann <sven@gimp.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef HAVE_GETOPT_LONG
#define _GNU_SOURCE
#include <getopt.h>
#endif

#include <blib/blib.h>


#ifdef HAVE_GETOPT_LONG
static const struct option options[] =
{
  { "loop",    no_argument, NULL, 'l' },
  { "help",    no_argument, NULL, 'h' },
  { "version", no_argument, NULL, 'v' },
  { NULL,      0,           NULL,  0  }
};
#endif

static const gchar *option_str = "lhv";


static void
usage (const gchar *name)
{
  g_printerr ("\n");
  g_printerr ("bsend - reads Blinkenlights movies and sends them over the net.\n"); 
  g_printerr ("Check http://www.blinkenlights.de/ for more information.\n\n");
  g_printerr ("Usage: %s [options] <filename> <host1> <host2:port> ...\n",
              name);
  g_printerr ("Options:\n");
  g_printerr ("   -l, --loop       repeat the movie in an endless loop\n");
  g_printerr ("   -h, --help       output usage information\n");
  g_printerr ("   -v, --version    output version information\n");
  g_printerr ("\n");

#ifndef HAVE_GETOPT_LONG
  g_printerr ("This version of bsend was compiled without support for long command-line\n");
  g_printerr ("options. Only the short, one-letter options will work.\n\n");
#endif
}


int 
main (int   argc,
      char *argv[])
{
  BSender     *sender;
  BMovie      *movie;
  const gchar *filename;
  GError      *error = NULL;
  GList       *list;
  gboolean     loop  = FALSE;
  gint         c, i, n;

  g_type_init ();

#ifdef HAVE_GETOPT_LONG
  while ((c = getopt_long (argc, argv, option_str, options, NULL)) >= 0)
#else
  while ((c = getopt (argc, argv, option_str)) >= 0)
#endif
    {
      switch (c)
        {
        case 'l':
          loop = TRUE;
          break;

        case 'h':
          usage (argv[0]);
          return EXIT_SUCCESS;

        case 'v':
          g_printerr ("bsend (%s version %s)\n", PACKAGE, VERSION);
          return EXIT_SUCCESS;

        default:
          usage (argv[0]);
          return EXIT_FAILURE;
        }
    }

  if (argc - optind < 1)
    {
      usage (argv[0]);
      return EXIT_FAILURE;
    }

  sender = b_sender_new ();

  n = 0;
  if (argc - optind == 1)
    {
      g_print ("No host specified, using localhost.\n");
      if (b_sender_add_recipient (sender,
                                  "localhost", MCU_LISTENER_PORT, NULL))
        n++;
    }
  else
    {
      for (i = optind + 1; i < argc; i++)
        {
          gchar *colon = strrchr (argv[i], ':');
          gint   port  = MCU_LISTENER_PORT;

          if (colon)
            {
              b_parse_int (colon + 1, &port);
              *colon = '\0';
            }
          
          if (b_sender_add_recipient (sender, argv[i], port, &error))
            n++;
          else
            g_printerr ("Skipping %s: %s\n", argv[i], error->message);
          
          g_clear_error (&error);
        }
    }

  if (!n)
    {
      g_printerr ("All hosts failed, exiting\n");
      g_object_unref (sender);
      return EXIT_FAILURE;
    }

  filename = argv[optind];

  if (strcmp (filename, "-") == 0)
    {
      filename = "<stdin>";
      movie = b_movie_new_from_fd (0, &error);
    }
  else
    {
      movie = b_movie_new_from_file (filename, FALSE, &error);
    }

  if (!movie)
    {
      g_printerr ("Error opening '%s': %s\n", filename, error->message);
      return EXIT_FAILURE;
    }

  g_print ("Loaded '%s' (%dx%d) channels=%d maxval=%d (%d frames, %d.%d s)\n",
           movie->title ? movie->title : b_object_get_name (B_OBJECT (movie)),
           movie->width, movie->height, movie->channels, movie->maxval,
           movie->n_frames,
           movie->duration / 1000, (movie->duration % 1000) / 10);


  b_sender_configure (sender,
                      movie->width, movie->height,
                      movie->channels, movie->maxval);

  g_print ("Sending to %d host(s) ...\n", n);

  do
    for (list = movie->frames; list; list = list->next)
      {
        BMovieFrame *frame = list->data;
        
        b_sender_send_frame (sender, frame->data);
        
        usleep (frame->duration * 1000);
      }
  while (loop);

  g_object_unref (movie);
  g_object_unref (sender);

  return EXIT_SUCCESS;
}
