/* Restore depleted /dev/random entropy pool with disk activity. */

#ifndef lint
static char vcid[] = "$Revision: 1.9 $";
#endif /* lint */

/* Author:   Peter Hendrickson <pdh@wiredyne.com>
 *
 * License:  This code is in the public domain.  Any of this code may be
 *           freely reproduced or adapted, even without mentioning the
 *           source.
 *
 * Warranty: IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR
 *           DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 *           DAMAGES ARISING OUT OF THE USE OF THIS CODE, EVEN IF THE
 *           AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *           THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 *           INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *           MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE
 *           CODE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THERE
 *           IS NO OBLIGATION WHATSOEVER TO PROVIDE MAINTENANCE,
 *           SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/rnd.h>
#include <sys/syslimits.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/utsname.h>
#include <sys/wait.h>
#include <syslog.h>
#include <unistd.h>

const char gRequiredOS[] = "NetBSD";
const char gRandomDevice[] = "/dev/random";
const long gMaxEntropy = RND_POOLBITS;
const long gCriticalEntropyThreshold = RND_POOLBITS / 10;
const long gMaxPathLen = PATH_MAX;

long verbose_flag = 0;
long syslog_flag = 0;
long no_stderr_flag = 0;
long silent_flag = 0;
long version_flag = 0;
long which_find_flag = 0;
long help_flag = 0;
long ignore_OS_flag = 0;

long report_interval = 3 * 60; /* Three minutes. */

#if !defined(FIND)
const char default_find_executable[] = "FIND";
#else
const char default_find_executable[] = "/usr/bin/find";
#endif

char *find_executable = (char *) NULL;

struct search_dir {
  char *dir;
  struct search_dir *next;
};

struct search_dir *root_search_dir = (struct search_dir *) NULL;
struct search_dir *current_search_dir = (struct search_dir *) NULL;

void parse_command_line_args (int argc, char *argv[]);
long entropy_available(const char *device);
void restore_entropy();
void restore_entropy_one_shot();
void nullify_fd (int fd);
void exec_find (const char *directory);
void kill_process (pid_t pid);
int find_done (pid_t pid);
void log_info (char *fmt, ... );
void log_warn (char *fmt, ... );
void log_err (char *fmt, ... );
void check_os ();
int is_directory (const char *path);
void setup_find_executable ();
void print_help ();

main(int argc, char *argv[])
{
  if( argc == 1 ) {
    fprintf(stderr, "usage: %s [switches] --search-dir dir [--search-dir dir ...]\n",
	    argv[0]);
    fprintf(stderr, "(Try --help.)\n");
    exit(1);
  }

  parse_command_line_args( argc, argv );

  if( version_flag ) {
    fprintf(stderr, "%s: %s\n", argv[0], vcid);
    exit(0);
  }

  if( which_find_flag ) {
    fprintf(stderr, "%s: %s\n", argv[0], default_find_executable);
    exit(0);
  }

  if( help_flag ) {
    print_help();
    exit(0);
  }

  if( ! ignore_OS_flag ) {
    check_os();
  }
  setup_find_executable();

  if( root_search_dir == (struct search_dir *) NULL ) {
    log_err("No directory specified with --search-dir.  Exiting.\n");
    exit(1);
  }
  current_search_dir = root_search_dir;

  log_info("Starting program.\n", "foo");
  while(1) {
    if( entropy_available( gRandomDevice ) < gCriticalEntropyThreshold ) {
      time_t then, now, elapsed;

      then = time( (time_t *) NULL );
      restore_entropy();
      now = time( (time_t *) NULL );
      elapsed = now - then;
      log_info("Stirred bits for %ld seconds.\n", (long) elapsed);
    }
    sleep(1);
  }
}

void parse_command_line_args (int argc, char *argv[])
{
  int i;

  for( i=1; i < argc; i++ ) {
    if( strcmp("--verbose", argv[i]) == 0 ) {
      verbose_flag = 1;
    }
    else if( strcmp("--version", argv[i]) == 0 ) {
      version_flag = 1;
    }
    else if( strcmp("--which-find", argv[i]) == 0 ) {
      which_find_flag = 1;
    }
    else if( strcmp("--help", argv[i]) == 0 ) {
      help_flag = 1;
    }
    else if( strcmp("--syslog", argv[i]) == 0 ) {
      syslog_flag = 1;
    }
    else if( strcmp("--no-stderr", argv[i]) == 0 ) {
      no_stderr_flag = 1;
    }
    else if( strcmp("--silent", argv[i]) == 0 ) {
      silent_flag = 1;
    }
    else if( strcmp("--ignore-OS", argv[i]) == 0 ) {
      ignore_OS_flag = 1;
    }
    else if( strcmp("--set-find", argv[i]) == 0 ) {
      size_t arglength;

      i++;
      if( i >= argc ) {
	log_err("--set-find present, but value not supplied.  Exiting.\n");
	exit(1);
      }

      arglength = strlen(argv[i]);
      if( arglength > PATH_MAX ) {
	log_err("--set-find argument is longer than maximum path length of %ld characters.  Exiting.\n",
		(long) PATH_MAX);
	exit(1);
      }

      find_executable = (char *) calloc( arglength + 1, sizeof(char) );
      if( find_executable == (char *) NULL ) {
	log_err("Could not allocate memory to store find_executable.  Exiting.\n");
	exit(1);
      }

      strncpy(find_executable, argv[i], arglength);
      find_executable[arglength] = '\0';
    }
    else if( strcmp("--search-dir", argv[i]) == 0 ) {
      size_t arglength;
      struct search_dir *new_search_dir;

      i++;
      if( i >= argc ) {
	log_err("--search-dir present, but value not supplied.  Exiting.\n");
	exit(1);
      }

      arglength = strlen(argv[i]);
      if( arglength > PATH_MAX ) {
	log_err("--search-dir argument is longer than maximum path length of %ld characters.  Exiting.\n",
		(long) PATH_MAX);
	exit(1);
      }

      if( ! is_directory( argv[i] ) ) {
	log_err("--search-dir argument \"%s\" is not a directory.\n", argv[i]);
	exit(1);
      }

      if( *(argv[i]) == '-' ) {
	log_err("Search directories starting with '-' are not permitted.\n");
	exit(1);
      }

      new_search_dir = (struct search_dir *) calloc( (size_t) 1, sizeof(struct search_dir) );
      if( new_search_dir == (struct search_dir *) NULL ) {
	log_err("Could not allocate memory for struct search_dir.  Exiting.\n");
	exit(1);
      }

      new_search_dir->dir = (char *) calloc( arglength + 1, sizeof(char) );
      if( new_search_dir->dir == (char *) NULL ) {
	log_err("Could not allocate memory for search_dir string.  Exiting.\n");
	exit(1);
      }

      strncpy(new_search_dir->dir, argv[i], arglength + 1);

      /* Puts the new search dir at end of list so directories
       * will be searched in the order given on command line. */
      new_search_dir->next = (struct search_dir *) NULL;
      if( root_search_dir == (struct search_dir *) NULL ) {
	root_search_dir = new_search_dir;
      }
      else {
	struct search_dir *tail;

	tail = root_search_dir;
	while( tail->next != (struct search_dir *) NULL )
	  tail = tail->next;

	tail->next = new_search_dir;
      }
    }
    else if( strcmp("--report-interval", argv[i]) == 0 ) {
      long new_interval;
      char **endptr;

      i++;
      if( i >= argc ) {
	log_err("--report-interval present, but value not supplied.  Exiting.\n");
	exit(1);
      }

      if( *argv[i] == '\0' ) {
	log_err("Report interval should be a positive integer.  Exiting.\n");
	exit(1);
      }

      new_interval = strtol(argv[i], endptr, 10);
      if( errno == ERANGE ) {
	log_err("Report interval should be a positive integer.  Exiting.\n");
	exit(1);
      }

      if( **endptr != '\0' ) {
	log_err("Report interval should be a positive integer.  Exiting.\n");
	exit(1);
      }

      report_interval = new_interval;
    }
    else {
      log_err("Unknown argument: %s (Try --help.)\n", argv[i]);
      exit(1);
    }
  }
}

long entropy_available( const char *device )
{
  int fd, result;
  u_int32_t entropy;

  fd = open(device, O_RDONLY);
  if( fd == -1 ) {
    log_err("Could not open device %s for reading.  Exiting.\n", device);
    exit(1);
  }

  result = ioctl( fd, RNDGETENTCNT, &entropy );
  if( result == -1 ) {
    log_err("Call to ioctl() failed.  Exiting.\n");
    exit(1);
  }

  result = close(fd);
  if( result == -1 ) {
    log_err("close() returned an error value.  Exiting.  (%m)\n");
    exit(1);
  }
  else if( result != 0 ) {
    log_err("close() returned an unexpected value of %d.  Exiting.\n", result);
    exit(1);
  }

  return (long) entropy;
}

/* Generates entropy by wearing out the disk. ;-) */
void restore_entropy ()
{
  /* Keep gathering entropy until we are near the top. */
  while( entropy_available(gRandomDevice) < gMaxEntropy ) {
    restore_entropy_one_shot();
  }
}

void restore_entropy_one_shot ()
{
  pid_t pid;

  pid = fork();
  if( pid > 0 ) {
    time_t start, now, elapsed;
    long msgs = 1;

    /* Parent */
    if( current_search_dir->next == (struct search_dir *) NULL )
      current_search_dir = root_search_dir;
    else
      current_search_dir = current_search_dir->next;

    start = time( (time_t *) NULL );
    while(1) {
      if( find_done(pid) ) {
	return;
      }

      /* If we have enough entropy, kill the find and return. */
      if( entropy_available(gRandomDevice) >= gMaxEntropy ) {
	kill_process(pid);
	return;
      }

      now = time( (time_t *) NULL );
      elapsed = now - start;
      if( elapsed >= (msgs * report_interval) ) {
	msgs = elapsed / report_interval + 1;
	log_info("Entropy generation in progress. (%ld seconds)\n", (long) elapsed);
      }

      sleep(1);
    }
  }
  else if( pid == 0 ) {
    /* Child */
    exec_find(current_search_dir->dir);
  }
  else {
    /* Error */
    log_err("Fork returned an error.  Exiting.\n");
    exit(1);
  }
}

int find_done (pid_t pid)
{
  int status;
  pid_t pid_child;

  pid_child = wait3( &status, WNOHANG, 0 );
  if( pid_child == 0 ) {
    /* Process not done yet, most common occurence. */
    return 0;
  }
  else if( pid_child > 0 ) {
    if( pid_child != pid ) {
      log_err("Expected pid %ld to exit, but got pid %ld instead.  Exiting.\n",
	      (long) pid, (long) pid_child);
      exit(1);
    }

    if( ! WIFEXITED(status) ) {
      log_warn("Find process does not seem to have exited normally.");
    }

    return 1;
  }
  else if( pid_child == -1 ) {
    log_err("Expected to have a child process either running or exiting.  Exiting.\n");
    exit(1);
  }
  else {
    log_err("wait3() returned %ld, which we believe can't happen.  Exiting",
	    (long) pid_child);
    exit(1);
  }
}

void kill_process (pid_t pid) {
  int result, status;
  pid_t exiting_pid;

  result = kill( pid, SIGTERM );
  if( result != 0 ) {
    log_err("Unable to kill process (pid = %ld).  Exiting.\n",
	    (long) pid);
    exit(1);
  }

  exiting_pid = wait3( &status, 0, 0 );
  if( exiting_pid > 0 ) {
    if( exiting_pid != pid ) {
      log_err("Expected pid %ld to exit, but got pid %ld instead.  Exiting.\n",
	      (long) pid, (long) exiting_pid);
      exit(1);
    }

    return;
  }
  else {
    log_err("Killed pid %ld but wait didn't find it.  Exiting.\n", (long) pid);
    exit(1);
  }
}

void exec_find (const char *directory) {
  int result;
  char dircpy[gMaxPathLen + 1];
  char *const args[4] = { find_executable, "-x", dircpy, (char *) NULL };
  size_t length;

  length = strlen(directory);
  if( length > gMaxPathLen ) {
    log_err("exec_find(): directory is longer than %ld.  Exiting.\n", gMaxPathLen);
    exit(1);
  }

  strncpy(dircpy, directory, gMaxPathLen);

  /* Redirect stderr and stdout to /dev/null. */
  nullify_fd( (int) STDOUT_FILENO );
  nullify_fd( (int) STDERR_FILENO );

  execv( find_executable, args );
  log_err("Failed to exec find (errno = %d).  Exiting.\n", errno);
  exit(1);
}

void nullify_fd (int fd)
{
  int result;
  int new_fd;

  result = close(fd);
  if ( result == -1 ) {
    log_err("Couldn't close file descriptor %d.  Exiting.\n", fd);
    exit(1);
  }

  new_fd = open( "/dev/null", O_WRONLY, 0666 );
  if( new_fd != fd  ) {
    log_err("Got back %d as a file descriptor instead of fd.\n", new_fd, fd);
    exit(1);
  }
}

void log_info ( char *fmt, ... )
{
  va_list ap;

  va_start(ap, fmt);
  if( !silent_flag )
    if( verbose_flag ) {
      if( syslog_flag )
	vsyslog(LOG_INFO, fmt, ap);
      if( ! no_stderr_flag )
	vfprintf(stderr, fmt, ap);
    }
  va_end(ap);
}

void log_warn ( char *fmt, ... )
{
  va_list ap;

  va_start(ap, fmt);
  if( !silent_flag ) {
    if( syslog_flag )
      vsyslog(LOG_WARNING, fmt, ap);
    if( ! no_stderr_flag )
      vfprintf(stderr, fmt, ap);
  }
  va_end(ap);
}

void log_err ( char *fmt, ... )
{
  va_list ap;

  va_start(ap, fmt);
  if( !silent_flag ) {
    if( syslog_flag )
      vsyslog(LOG_ERR, fmt, ap);
    if( ! no_stderr_flag )
      vfprintf(stderr, fmt, ap);
  }
  va_end(ap);
}

void check_os ()
{
  int result;
  struct utsname sysinfo;

  result = uname( &sysinfo );
  if( result != 0 ) {
    log_err("Error calling uname (errno = %d).  Exiting.\n", errno);
    exit(1);
  }

  if( strcmp( gRequiredOS, sysinfo.sysname ) != 0 ) {
    log_err("This code currently only runs safely under %s\n", gRequiredOS);
    exit(1);
  }
}

int is_directory (const char *path)
{
  int result;
  struct stat fileinfo;

  result = stat(path, &fileinfo);
  if( result != 0 ) {
    if( errno == ENOENT ) {
      log_err("File %s does not exist.  Exiting.\n", path);
    }
    else {
      log_err("Could not stat %s (errno = %d).  Exiting.\n", path, errno);
    }
    exit(1);
  }

  if( fileinfo.st_mode & S_IFDIR ) {
    return 1;
  }
  else {
    return 0;
  }
}

void setup_find_executable ()
{
  if( find_executable == (char *) NULL ) {
    size_t findlen;

    findlen = strlen(default_find_executable);

    find_executable = (char *) calloc( findlen  + 1, sizeof(char) );
    if( find_executable == NULL ) {
      log_err("Could not allocate memory to copy executable name.  Exiting.\n");
      exit(1);
    }

    strncpy(find_executable, default_find_executable, findlen + 1);
  }
}

void print_help()
{
    fprintf(stderr, "Basic help message - see bitstir(8) for more detail:\n");
    fprintf(stderr, "  --search-dir dir: Directory to search (at least one required)\n");
    fprintf(stderr, "  --set-find file: Choose alternate find program.\n");
    fprintf(stderr, "  --which-find: Print location of default find executable.\n");
    fprintf(stderr, "  --ignore-OS: run on any OS (use with care)\n");
    fprintf(stderr, "  --syslog: write messages to syslog\n");
    fprintf(stderr, "  --verbose: print more stuff\n");
    fprintf(stderr, "  --no-stderr: write no messages to standard error\n");
    fprintf(stderr, "  --silent: write no messages at all\n");
    fprintf(stderr, "  --report-interval seconds: set new interval for reports\n");
    fprintf(stderr, "  --version: print the version\n");
    fprintf(stderr, "  --help: print this message\n");
}
