/* $Id: util.c,v 1.8 2004/03/06 22:30:15 andrewbaker Exp $ */
/*
** Copyright (C) 2001 Martin Roesch <roesch@sourcefire.com>
** Copyright (C) 2001-2004 Andrew R. Baker <andrewb@sourcefire.com>
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef SOLARIS
    #include <strings.h>
#endif
#include <syslog.h>
#include <stdarg.h>
#include <sys/types.h>
#include <dirent.h>
#include <ctype.h>
#include <sys/time.h>
#include <time.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <errno.h>

#include "barnyard.h"
#include "strlcpyu.h"
#include "strlcatu.h"
#include "util.h"

#define FRAME_SIZE        66
#define C_OFFSET          49

char *data_dump_buffer;     /* printout buffer for PrintNetData */
int dump_ready;         /* flag to indicate status of printout buffer */
int dump_size;          /* size of printout buffer */
char *protocol_names[256];


/*
 * Function: LogMessage(const char *, ...)
 *
 * Purpose: Print a message to stdout or with logfacility.
 *
 * Arguments: format => the formatted error string to print out
 *            ... => format commands/fillers
 *
 * Returns: void function
 */
void LogMessage(const char *format,...)
{
    char buf[STD_BUF+1];
    va_list ap;

    va_start(ap, format);

    if(pv.daemon_flag)
    {
        vsnprintf(buf, STD_BUF, format, ap);
        syslog(LOG_DAEMON | LOG_NOTICE, "%s", buf);
    }
    else
    {
        vfprintf(stderr, format, ap);
    }
    
    va_end(ap);

    return;
}


/*
 * Function: FatalError(const char *, ...)
 *
 * Purpose: When a fatal error occurs, this function prints the error message
 *          and cleanly shuts down the program
 *
 * Arguments: format => the formatted error string to print out
 *            ... => format commands/fillers
 *
 * Returns: void function
 */
void FatalError(const char *format,...)
{
    char buf[STD_BUF+1];
    va_list ap;

    va_start(ap, format);

    if(pv.daemon_flag)
    {
        vsnprintf(buf, STD_BUF, format, ap);
        syslog(LOG_CONS | LOG_DAEMON | LOG_ERR, "FATAL ERROR: %s", buf);
    }
    else
    {
        vfprintf(stderr, format, ap);
        fprintf(stderr,"Fatal Error, Quitting..\n");
    }

    CleanExit(1);
}



void FreeToks(char **toks, int num_toks)
{
    do
    {
        num_toks--;
        free(toks[num_toks]);
    } while(num_toks);
}


char *ZapWhitespace(char *data)
{
    while(isspace((int)*data)) data++;
    
    return data;
}


void *SafeAlloc(u_long size)
{
    void *p;

    p = calloc(size, sizeof(char));

    if(p == NULL)
    {
        FatalError("ERROR => Unable to allocate memory!\n");
    }

    return p;
}



char *ProcessFileOption(char *filespec)
{
    char *filename;
    char buffer[STD_BUF];

    if(filespec == NULL)
    {
        FatalError("ERROR: no arguement in this file option, remove extra ':' at the end of the alert option\n");
    }

    /* look for ".." in the string and complain and exit if it is found */
    if(strstr(filespec, "..") != NULL)
    {
        FatalError("ERROR: file definition contains \"..\".  Do not do that!\n");
    }

    if(filespec[0] == '/')
    {
        /* absolute filespecs are saved as is */
        filename = strdup(filespec);
    }
    else
    {
        /* relative filespec is considered relative to the log directory */
        /* or /var/log if the log directory has not been set */
        if(pv.log_dir)
        {
            strlcpy(buffer, pv.log_dir, STD_BUF);
        }
        else
        {
            strlcpy(buffer, "/var/log/snort", STD_BUF);
        }

        strlcat(buffer, "/", STD_BUF - strlen(buffer));
        strlcat(buffer, filespec, STD_BUF - strlen(buffer));
        filename = strdup(buffer);
    }
#ifdef DEBUG
    LogMessage("ProcessFileOption: %s\n", filename);
#endif
    return filename;
}

int BuildFilePath(char *filename, char *default_basedir, char **filepath)
{
    if(filename[0] == '/')
    {
        /* Absolute path */
        if(!(*filepath = strdup(filename)))
        {
            FatalError("Out of memory (wanted %u bytes)\n",
                    strlen(filename) + 1);
            return -1;
        }
    }
    else
    {
        size_t len = strlen(default_basedir) + 1 + strlen(filename) + 1;
        if(!(*filepath = (char *)calloc(len, sizeof(char))))
        {
            FatalError("Out of memory (wanted %u bytes)\n", len);
            return -1;
        }
        snprintf(*filepath, len, "%s/%s", default_basedir, filename);
    }       

    return 0;
}


int strip(char *data)
{
    int size;
    char *end;
    char *idx;

    idx = data;
    end = data + strlen(data);
    size = end - idx;

    while(*idx == ' ') 
    {
        idx++;
        size--;
    }

    while(*(end-1) == ' ')
    {
        *(--end) = 0;
        size--;
    }

    while(idx != end)
    {
        if((*idx == '\n') ||
           (*idx == '\r'))
        {
            *idx = 0;
            size--;
        }
        if(*idx == '\t')
        {
            *idx = ' ';
        }
        idx++;
    }

    return size;
}


void InitProtoNames()
{
    int i;
    struct protoent *pt;
    unsigned char *tmp;
    u_char protoname[11];

    for(i = 0; i < 256; i++)
    {
        pt = getprotobynumber(i);

        if(pt)
        {
            protocol_names[i] = strdup(pt->p_name);

            tmp = protocol_names[i];

            for(tmp = protocol_names[i]; *tmp != 0; tmp++)
                *tmp = (unsigned char) toupper(*tmp);
        }
        else
        {
            snprintf(protoname, 10, "PROTO%03d", i);
            protocol_names[i] = strdup(protoname);
        }
    }
}



char *fasthex(u_char *xdata, int length)
{
    char conv[] = "0123456789ABCDEF";
    char *retbuf = NULL; 
    char *index;
    char *end;
    char *ridx;

    index = xdata;
    end = xdata + length;
    retbuf = (char *) calloc((length*2)+1, sizeof(char));
    ridx = retbuf;

    while(index < end)
    {
        *ridx++ = conv[((*index & 0xFF)>>4)];
        *ridx++ = conv[((*index & 0xFF)&0x0F)];
        index++;
    }

    return retbuf;
}



void PrintNetData(FILE * fp, u_char * start, const int len)
{
    char *end;          /* ptr to buffer end */
    int i;          /* counter */
    int j;          /* counter */
    int dbuf_size;      /* data buffer size */
    int done;           /* flag */
    char *data;         /* index pointer */
    char *frame_ptr;        /* we use 66 byte frames for a printed line */
    char *d_ptr;        /* data pointer into the frame */
    char *c_ptr;        /* char pointer into the frame */
    char conv[] = "0123456789ABCDEF";   /* xlation lookup table */

    /* initialization */
    done = 0;

    if(start == NULL)
    {
        LogMessage("Got NULL ptr in PrintNetData()\n");
        return;
    }
    /* zero, print a <CR> and get out */
    if(!len)
    {
        fputc('\n', fp);
        return;
    }
    /*
     *      * if we've already prepared this particular data buffer, just print it
     *           * out again to save time
     *                */
    if(dump_ready)
    {
        fwrite(data_dump_buffer, dump_size, 1, fp);
        fflush(fp);
        return;
    }
    end = (char*) (start + (len - 1));    /* set the end of buffer ptr */

    /*if(len > pv.mtus[0])*/
    if(len > 65535)
    {
        dbuf_size = FRAME_SIZE + FRAME_SIZE + 1;

        /* dbuf_size = 66 + 67; */
        end =  (char*) (start + 15);
    }
    else
    {
        /* figure out how big the printout data buffer has to be */
        dbuf_size = ((len / 16) * FRAME_SIZE) + FRAME_SIZE + 1;
    }

    /* generate the buffer */
    data_dump_buffer = (char *) malloc(dbuf_size);

    /* make sure it got allocated properly */
    if(data_dump_buffer == NULL)
    {
        perror("PrintNetData()");
        FatalError("Failed allocating %X bytes! (Length: %X)\n",
                dbuf_size, len);
    }

    /* clean it out */
    memset(data_dump_buffer, 0x20, dbuf_size);


    /* set the byte buffer pointer to step thru the data buffer */
    data = (char*) start;

    /* set the frame pointer to the start of the printout buffer */
    frame_ptr = data_dump_buffer;

    /* initialize counters and frame index pointers */
    i = 0;
    j = 0;

    /* loop thru the whole buffer */
    while(!done)
    {
        d_ptr = frame_ptr;
        c_ptr = (frame_ptr + C_OFFSET);

        /* process 16 bytes per frame */
        for(i = 0; i < 16; i++)
        {
            /*
             * look up the ASCII value of the first nybble of the current
             * data buffer
             */
            *d_ptr = conv[((*data & 0xFF) >> 4)];
            d_ptr++;

            /* look up the second nybble */
            *d_ptr = conv[((*data & 0xFF) & 0x0F)];
            d_ptr++;

            /* put a space in between */
            *d_ptr = 0x20;
            d_ptr++;

            /* print out the char equivalent */
            if(*data > 0x1F && *data < 0x7F)
                *c_ptr = (char) (*data & 0xFF);
            else
                *c_ptr = 0x2E;

            c_ptr++;

            /* increment the pointer or finish up */
            if(data < end)
                data++;
            else
            {
                /* finish up the buffer printout and set the "ready" flags */
                done = 1;
                dump_ready = 1;

                *c_ptr = '\n';
                c_ptr++;
                *c_ptr = '\n';
                c_ptr++;
                *c_ptr = 0;

                dump_size = (int) (c_ptr - data_dump_buffer);
                fwrite(data_dump_buffer, dump_size, 1, fp);
                return;
            }
        }

        *c_ptr = '\n';
        frame_ptr += FRAME_SIZE;
    }
}


void ClearDumpBuf()
{
    if(data_dump_buffer != NULL && dump_ready)
        free(data_dump_buffer);
    else
        return;

    data_dump_buffer = NULL;

    dump_ready = 0;
}


void GoDaemon(void)
{
    pid_t fs;
    int fd;

    LogMessage("Initializing daemon mode\n");

    fs = fork();

    if(fs > 0)
        exit(0);                /* parent */

    if(fs < 0)
    {
        perror("fork");
        CleanExit(1);
    }
    chdir("/");
    setpgid(0, getpid());
    if((fd = open("/dev/null", O_RDWR)) >= 0)
    {
        dup2(fd, 0);
        dup2(fd, 1);
        dup2(fd, 2);
    }

    return;
}

size_t RenderTimestamp(time_t timet, char *timebuf, size_t len)
{
    struct tm *lt;

    if(pv.localtime)
    {
        lt = localtime(&timet);
        return strftime(timebuf, len, "%Y-%m-%d %H:%M:%S %z", lt);
    }
        
    lt = gmtime(&timet);

    return strftime(timebuf, len, "%Y-%m-%d %H:%M:%S", lt);



}

static char tmpbuf[256];

int RenderTimeval(struct timeval *tv, char *timebuf, size_t len)
{
    struct tm *lt;
    time_t timet;
    size_t size;
    int written;

    timet = tv->tv_sec;
    
    if(pv.localtime)
        lt = localtime(&timet);
    else
        lt = gmtime(&timet);

    if((size = strftime(tmpbuf, len, "%m/%d/%y-%H:%M:%S", lt)) == 0)
    {
        LogMessage("ERROR: tmpbuf too small in util.c:RenderTimeval(...)\n");
        /* not enough space in the buffer */
        return -1;
    }

    /* add in the microseconds */
    written = snprintf(timebuf, len, "%s.%06u", tmpbuf, (u_int) tv->tv_usec);
    /* different lib versions behave differently */
    if(written == -1 || written >= len)  
    {
        /* not enough space, buffer was truncated */
        return -1;
    }
    
    return written;
}

int CreatePidFile(char *filename)
{
    int pid_file;
    char pid_buffer[12];
    struct flock lock;

    if((pid_file = open(filename, O_CREAT | O_RDWR,
                    S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)) == -1)
    {
        /* XXX: more robust error checking */
        LogMessage("Failed to open pid file %s\n", filename);
        return -1;
    }

    /* mutual exclusion */
    lock.l_type = F_WRLCK;
    lock.l_start = 0;
    lock.l_whence = SEEK_SET;
    lock.l_len = 0;
    if(fcntl(pid_file, F_SETLK, &lock) == -1)
    {
        /* failed to lock file */
        if(errno == EACCES || errno == EAGAIN)
            LogMessage("Unable to lock pid file, already locked\n");
        else
            LogMessage("Failed to lock pid file: %s\n", strerror(errno));
        return -1;
    }

    snprintf(pid_buffer, 11, "%d\n", (int) getpid());
    pid_buffer[11] = '\0';
    ftruncate(pid_file, 0);
    write(pid_file, pid_buffer, strlen(pid_buffer));

    return 0;
}


int String2Long(char *string, long *result)
{
    long value;
    char *endptr;
    if(!string)
        return -1;

    value = strtol(string, &endptr, 10);
    
    while(isspace(*endptr))
        endptr++;
    
    if(*endptr != '\0')
        return -1;
    
    if(result)
        *result = value;

    return 0;
}

int String2ULong(char *string, unsigned long *result)
{
    unsigned long value;
    char *endptr;
    if(!string)
        return -1;

    value = strtoul(string, &endptr, 10);
    
    while(isspace(*endptr))
        endptr++;
    
    if(*endptr != '\0')
        return -1;
    
    if(result)
        *result = value;

    return 0;
}

int StripWhitespace(char **data)
{
    char *string;
    int len;

    if(!data || !*data)
        return -1;

    string = *data;

    while(isspace(*string))
        string++;

    len = strlen(string);

    while(len)
    {
        if(isspace(string[len-1]))
            string[len-1] = '\0';
        else
            break;
        len--;
    }
    *data = string;
    return 0;
}

