/* $Id: op_plugbase.c,v 1.11 2004/03/29 00:14:19 andrewbaker Exp $ */
/*
** Copyright (C) 2001-2002  Andrew R. Baker <andrewb@snort.org>
** Copyright (C) 2001 Martin Roesch <roesch@sourcefire.com>
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "op_plugbase.h"
#include "util.h"

/* Output plug-in include files */
#include "op_fast.h"
#include "op_logdump.h"
#include "op_alert_syslog.h"
#include "op_log_pcap.h"
#if defined(ENABLE_MYSQL) || defined(ENABLE_POSTGRES)
#include "op_acid_db.h"
#endif
#ifdef ENABLE_MYSQL
#include "op_sguil.h"
#endif
#include "op_alert_csv.h"
#include "op_alert_syslog2.h"
#include "op_alert_console.h"

/* ----------------------- Global Data --------------------------*/
OutputPluginListNode *outputPlugins = NULL;

/* ----------------------- Global Functions --------------------------*/
void LoadOutputPlugins()
{
    OpAlertFast_Init();
    OpAlertSyslog_Init();
    OpLogDump_Init();
    OpLogPcap_Init();
#if defined(ENABLE_MYSQL) || defined(ENABLE_POSTGRES)
    OpAcidDb_Init();
#endif
#ifdef ENABLE_MYSQL
    OpSguil_Init();
#endif
    OpAlertCSV_Init();
    OpAlertSyslog2_Init();
    OpAlertConsole_Init();
    return;
}


OutputPlugin *RegisterOutputPlugin(char *name, char *type)
{
    OutputPlugin *outputPlugin = SafeAlloc(sizeof(OutputPlugin));
    
    outputPlugin->name = strdup(name);  /* XXX error check */
    outputPlugin->type = strdup(type);  /* XXX error check */

    AddOutputPlugin(&outputPlugins, outputPlugin);
    
    return outputPlugin;
}

/* XXX: should not allow OutputPlugins with duplicate names */
int AddOutputPlugin(OutputPluginListNode **list, OutputPlugin *outputPlugin)
{
    OutputPluginListNode *current, *newNode;

    /* create a new node */
    newNode = SafeAlloc(sizeof(OutputPluginListNode));
    newNode->outputPlugin = outputPlugin;
    newNode->next = NULL;

    /* add node to list */
    if(*list == NULL)
    {
        *list = newNode;
        return 0;
    }

    current = *list;
    while(current->next != NULL)
        current = current->next;
    current->next = newNode;

    return 0;
}

void FreeOutputPluginList(OutputPluginListNode *list, int free_plugins)
{
    OutputPluginListNode *current;
    while(list != NULL)
    {
        current = list;
        if(free_plugins)
            OutputPluginFree(current->outputPlugin);
        list = current->next;
        free(current);
    }
} 
        
void CallOutputPlugins(OutputPluginListNode *list, void *data)
{
    OutputPlugin *outputPlugin;
    if(data == NULL)
        return;

    while(list != NULL)
    {
        outputPlugin = list->outputPlugin;
        if(outputPlugin && outputPlugin->outputFunc)
            outputPlugin->outputFunc(outputPlugin->data, data);
        list = list->next;
    }
}

OutputPlugin *LookupOutputPlugin(char *name)
{
    OutputPluginListNode *opln = outputPlugins;

    while(opln != NULL)
    {
        OutputPlugin *op = opln->outputPlugin;
        if(op && strcasecmp(name, op->name) == 0)
            return op;
        opln = opln->next;
    }
    return NULL;
}

/* ------------------ Output Plugin Functions ----------------------*/
int OutputPluginSetup(OutputPlugin *outputPlugin, char *args)
{
    if(outputPlugin == NULL)
        return 1;

    if(outputPlugin->configured)
        return 0;

    if(outputPlugin->setupFunc)

        outputPlugin->setupFunc(outputPlugin, args);

    outputPlugin->configured = 1;
    return 0;
}

int OutputPluginExit(OutputPlugin *outputPlugin)
{
    if(outputPlugin == NULL)
        return 1;

    if(outputPlugin->configured == 0)
        return 0;

    if(outputPlugin->started == 1)
        OutputPluginStop(outputPlugin);

    if(outputPlugin->exitFunc != NULL)
       outputPlugin->exitFunc(outputPlugin);

    if(outputPlugin->data == NULL) 
        free(outputPlugin->data);
    outputPlugin->data = NULL;
    outputPlugin->configured = 0;
    return 0;
}

int OutputPluginStart(OutputPlugin *outputPlugin, void *spool_header)
{
    if(outputPlugin == NULL || outputPlugin->configured == 0)
        return 1;

    if(outputPlugin->started == 1)
        return 0;

    if(outputPlugin->startFunc != NULL)
        outputPlugin->startFunc(outputPlugin, spool_header);

    outputPlugin->started = 1;
    return 0;
}

int OutputPluginStop(OutputPlugin *outputPlugin)
{
    if(outputPlugin == NULL)
        return 1;

    if(outputPlugin->started == 0)
        return 0;

    if(outputPlugin->stopFunc != NULL)
        outputPlugin->stopFunc(outputPlugin);

    outputPlugin->started = 0;
    return 0;
}

int OutputPluginRestart(OutputPlugin *outputPlugin, void *spool_header)
{
    if(outputPlugin == NULL || outputPlugin->configured == 0)
        return 1;
    if(outputPlugin->started == 0)
        return 1;
    OutputPluginStop(outputPlugin);
    OutputPluginStart(outputPlugin, spool_header);
    return 0;
}

void OutputPluginFree(OutputPlugin *outputPlugin)
{
    if(outputPlugin == NULL)
        return;

    OutputPluginExit(outputPlugin);
    
    if(outputPlugin->name)
        free(outputPlugin->name);
    if(outputPlugin->type)
        free(outputPlugin->type);
    free(outputPlugin);
}
