/*  SpiralLoops
 *  Copyleft (C) 2000 David Griffiths <dave@pawfal.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include "SpiralLoops.h"

#include <FL/Fl.H>
#include <FL/Enumerations.H>
#include <FL/fl_file_chooser.h>
#include <fstream>
#include <strstream>
#include <string>
#include <time.h>
#include <stdlib.h>

#include <dlfcn.h>
#include <stdio.h>

#include "SpiralSound/Output.h"
#include "Loop.h"
#include <list>

static const int FILE_VERSION = 1;

LoopWin::~LoopWin()
{	
	delete m_LoopGUI;
	m_Loop->Clear();
	delete m_Loop;
} 
		
Loops::Loops():
m_PluginOpen(false),
m_PluginLinked(false),
m_PluginPaused(false),
m_PluginWindow(NULL),
m_DelayGUI(&m_Delay),
m_ReverbGUI(&m_Reverb),
m_Out(2),
m_OutGUI(&m_Out),
m_MasterLength(0),
m_MasterSpeed(1),
m_Magic(0),
m_NextLoopId(0)
{ 	
	AllocMem();
}

Loops::~Loops()
{ 
	if (m_PluginOpen)
	{
		ClosePlugin();
	}
	
	// Todo: delete all loops from the list
}

void Loops::AllocMem()
{
	m_Databuf.Allocate(SpiralLoopsInfo::BUFSIZE);
}

Fl_Window *Loops::CreateWindow()
{
 Fl_Window* w;
  { Fl_Window* o = Window = new Fl_Window(835, 135, "SpiralLoops 2.0.0");
    w = o;
	o->color(SpiralLoopsInfo::GUIBG_COLOUR);
 
	// put part guis here
	CreateGUI(5, 5, "Main");
	m_DelayGUI.CreateGUI(210,5,"Delay");
	m_ReverbGUI.CreateGUI(210,70,"Reverb");	
	m_Scope.CreateGUI(515,13,"Scope");	
	m_OutGUI.CreateGUI(745,13,"Output");
  }  
	
	//Set up for stereo	
	m_Out.SetNumChannels(2);
	m_Scope.SetNumChannels(2);
	m_Delay.SetNumChannels(2);
	m_Reverb.SetNumChannels(2);
	
  return w;
}

void Loops::DoIdle()
{
	m_Out.ClearBuffer();
	
	// see if any need deleting
	for (map<int,LoopWin*>::iterator i=m_LoopWinMap.begin(); 
		 i!=m_LoopWinMap.end(); i++)
	{		
		if (i!=m_LoopWinMap.end() && i->second->m_Loop->Delete())
		{
			delete (i->second);
			m_LoopWinMap.erase(i);
			break;
		}
	}
	
	// Do the trigger thing
	CheckTriggers();

	// run the Loops!
	for (map<int,LoopWin*>::iterator i=m_LoopWinMap.begin(); 
		 i!=m_LoopWinMap.end(); i++)
	{			
		if (i->second->m_Loop->GetOutput(m_Databuf))
		{
			m_Out.Send(m_Databuf, i->second->m_Loop->GetLeftVol(),i->second->m_Loop->GetRightVol());
		}
	}
	
	if (m_PluginOpen && m_PluginLinked && !m_PluginPaused)  
	{
		m_Out.Send(*RunPlugin());
	}
	
	m_Delay.GetOutput(m_Out.GetBuffer());
	m_Reverb.GetOutput(m_Out.GetBuffer());
	m_Scope.Display(m_Out.GetBuffer().GetBuffer());
	m_Out.Play();
}

void Loops::AddLoop()
{
	LoopWin *nlw;
	nlw = new LoopWin;
	nlw->m_Loop = new Loop(this);
	nlw->m_Loop->SetId(m_NextLoopId);
	if (m_PluginBuffer) nlw->m_Loop->SetRecordingSource(m_PluginBuffer->GetBuffer());
	nlw->m_LoopGUI = new LoopGUI(nlw->m_Loop, 
							     nlw->m_Loop->GetDelayPtr(), 
								 nlw->m_Loop->GetReverbPtr());
	char Name[64];
	sprintf(Name,"SpiralLoop %d",m_NextLoopId);
	nlw->m_LoopGUI->CreateGUI(Name);
	nlw->m_LoopGUI->StopUpdate(!m_PluginPaused);
	nlw->m_Loop->SetGUI(nlw->m_LoopGUI);
	m_LoopWinMap[m_NextLoopId]=nlw;
	
	m_NextLoopId++;
}

void Loops::ClearLoops()
{
	for (map<int,LoopWin*>::iterator i=m_LoopWinMap.begin(); 
		 i!=m_LoopWinMap.end(); i++)
	{			
		delete i->second;
	}
	
	m_LoopWinMap.clear();
}

void Loops::SetMaster (int id, float s, float sp) 
{
	m_MasterLoopId=id; 
	m_MasterLength=s;
	m_MasterSpeed=sp;
	m_Magic=m_MasterLength/m_MasterSpeed;
	
	for(map<int,LoopWin*>::iterator i=m_LoopWinMap.begin();
		i!=m_LoopWinMap.end(); i++)
	{
		if (i->second->m_Loop->GetId()!=m_MasterLoopId)
		{
			i->second->m_LoopGUI->SetMasterStatus(false);
		}
	}
}

void Loops::CheckTriggers()
{
	// Get all the loops to be triggered
	vector<int> Loops;
	
	for(map<int,LoopWin*>::iterator i=m_LoopWinMap.begin();
		i!=m_LoopWinMap.end(); i++)
	{
		i->second->m_LoopGUI->CheckTriggers(i->second->m_Loop->GetCurrentAngle(),Loops);
	}
	
	// Trigger all the loops we've found
	for(vector<int>::iterator i=Loops.begin();
		i!=Loops.end(); i++)
	{
		map<int,LoopWin*>::iterator f=m_LoopWinMap.find(*i);
		
		if (f!=m_LoopWinMap.end())
		{
			f->second->m_Loop->Trigger();
			f->second->m_LoopGUI->Trigger();
		}
	}
	
}

istream &operator>>(istream &s, Loops &o)
{	
	string dummy;
	int ver;
	s>>dummy>>dummy>>dummy>>dummy>>ver;
	
	if (ver>FILE_VERSION)
	{
		cerr<<"Bad file, or more recent version."<<endl;
		return s;
	}
	
	s>>o.m_Delay>>o.m_Reverb>>o.m_MasterLength>>o.m_MasterSpeed>>o.m_Magic>>o.m_MasterLoopId;
	
	// load the loops
	int Num, ID;
	s>>Num;
	
	for(int n=0; n<Num; n++)
	{
		s>>ID; // load the id
		
		LoopWin *nlw;
		nlw = new LoopWin;
		nlw->m_Loop = new Loop(&o);
		nlw->m_LoopGUI = new LoopGUI(nlw->m_Loop, 
								     nlw->m_Loop->GetDelayPtr(), 
									 nlw->m_Loop->GetReverbPtr());
							
		nlw->m_Loop->SetGUI(nlw->m_LoopGUI);
		 
		// read the data
		s>>*nlw->m_Loop;		
		
		// set up the plugin
		if (o.m_PluginBuffer) nlw->m_Loop->SetRecordingSource(o.m_PluginBuffer->GetBuffer());
									 									 
		char Name[64];
		sprintf(Name,"SpiralLoop %d",ID);
		nlw->m_LoopGUI->CreateGUI(Name);
		nlw->m_LoopGUI->StopUpdate(!o.m_PluginPaused);
		o.m_LoopWinMap[ID]=nlw;
		
		// read in the gui data
		s>>*nlw->m_LoopGUI;		
		
		if (ID>=o.m_NextLoopId) o.m_NextLoopId=ID+1;
	}
	return s;
}

ostream &operator<<(ostream &s, Loops &o)
{
	s<<"SpiralLoops WorkSpace File Ver 1"<<endl;
	s<<o.m_Delay<<" "<<o.m_Reverb<<" "<<o.m_MasterLength<<" "<<o.m_MasterSpeed<<" "
	 <<o.m_Magic<<" "<<o.m_MasterLoopId<<" ";
	
	// save out the loops
	s<<o.m_LoopWinMap.size()<<" ";
	
	for(map<int,LoopWin*>::iterator i=o.m_LoopWinMap.begin();
		i!=o.m_LoopWinMap.end(); i++)
	{
		s<<i->first<<" "; // save the id
		s<<*i->second->m_Loop<<" "; // save the loop
		s<<*i->second->m_LoopGUI<<" "; // save the loopGUI
	}
	return s;
}

/////////////// Plugin handling functions ///////////////////////////

int Loops::OpenPlugin(const char *PluginName)
{
        handle = dlopen (PluginName, RTLD_NOW);
        
        if (!handle)
        {
        	cerr<<"Error loading plugin:"<<endl;
            fputs(dlerror(), stderr);
            return 0;
        }
		
		m_PluginOpen=true;
		
        return 1;
}

int Loops::LinkPlugin()
{       
        Destroy = (void (*)()) dlsym(handle, "Destroy__Fv");
		Create  = (Fl_Window *(*)()) dlsym(handle, "Create__Fv");
        Run     = (Sample *(*)()) dlsym(handle, "Run__Fv");
        
                
        if ((error = dlerror()) != NULL)
        {
        	cerr<<"Error linking to plugin:"<<endl;
		    fputs(error, stderr);
            return 0;
        }       
        
		m_PluginLinked=true;
		
        return 1;
}

int Loops::ClosePlugin()
{
	if (m_PluginOpen)
    {
		DestroyPlugin();
		
		m_PluginOpen=false;
		m_PluginLinked=false;
		
		return dlclose(handle);
	}
	
	return -1;
}

///////////////// CopyBuffer stuff /////////////////////////////////

void Loops::Cut(Sample &buf, int start, int end)
{
	// remove the old copybuffer
	m_CopyBuffer.Clear();
	buf.GetRegion(m_CopyBuffer,start,end);
	buf.Remove(start,end);
}

void Loops::Copy(Sample buf, int start, int end)
{	
	// remove the old copybuffer
	m_CopyBuffer.Clear();
	buf.GetRegion(m_CopyBuffer,start,end);
}

void Loops::Paste(Sample &buf, int start)
{	
	buf.Insert(m_CopyBuffer, start);
}

void Loops::PasteMix(Sample &buf, int start)
{	
	buf.Mix(m_CopyBuffer, start);
}

///////////////// Main GUI stuff /////////////////////////////////
   
void Loops::CreateGUI(int xoff, int yoff, char *name)
{
	 Fl_Group* o = GUIMainGroup = new Fl_Group(xoff, yoff, 200, 125, name);
      o->type(1);
	  o->box(FL_UP_BOX);
      o->labeltype(FL_ENGRAVED_LABEL);
      o->align(FL_ALIGN_TOP_LEFT|FL_ALIGN_INSIDE);
	  o->user_data((void*)(this));
	  
      Load = new Fl_Button(5+xoff, 20+yoff, 90, 20, "Load Workspace");
      Load->labelsize(10);
      Load->type(0);		 
	  Load->callback((Fl_Callback*)cb_Load);

	  Save = new Fl_Button(5+xoff, 40+yoff, 90, 20, "Save Workspace");
      Save->labelsize(10);
      Save->type(0);		 
	  Save->callback((Fl_Callback*)cb_Save); 	  
	  
	  NewLoop = new Fl_Button(5+xoff, 60+yoff, 90, 20, "New Loop");
      NewLoop->labelsize(10);
      NewLoop->type(0);		 
	  NewLoop->callback((Fl_Callback*)cb_NewLoop);
 
 	  LoadPlugin = new Fl_Button(5+xoff, 80+yoff, 90, 20, "Load Plugin");
      LoadPlugin->labelsize(10);
      LoadPlugin->type(0);		 
	  LoadPlugin->callback((Fl_Callback*)cb_LoadPlugin);
 
 	  QuitPlugin = new Fl_Button(5+xoff, 100+yoff, 90, 20, "Pause Plugin");
      QuitPlugin->labelsize(10);
      QuitPlugin->type(1);		 
	  QuitPlugin->callback((Fl_Callback*)cb_ClosePlugin);
 
 	  SnapPoints = new Fl_Knob(xoff+125, yoff+5, 40, 40, "Snap Points");
	  SnapPoints->color(SpiralLoopsInfo::GUI_COLOUR);
      SnapPoints->labelsize(10);
	  SnapPoints->maximum(50);
      SnapPoints->step(1);
      SnapPoints->value(2);
      SnapPoints->callback((Fl_Callback*)cb_SnapPoints);
	  
	  Snap = new Fl_Button(100+xoff, 80+yoff, 90, 20, "Snap");
      Snap->labelsize(10);
      Snap->type(1);		 
	  Snap->callback((Fl_Callback*)cb_Snap);

	  SnapOutput = new Fl_Value_Output(105, 63, 90, 15);
	  SnapOutput->labelsize(10);
	  SnapOutput->textsize(10);
	  
      o->end();
    
}

inline void Loops::cb_NewLoop_i(Fl_Button* o, void* v)
{AddLoop();}
void Loops::cb_NewLoop(Fl_Button* o, void* v)
{((Loops*)(o->parent()->user_data()))->cb_NewLoop_i(o,v);}

inline void Loops::cb_LoadPlugin_i(Fl_Button* o, void* v)
{
	char *fn=fl_file_chooser("Pick a plugin", "*.so", NULL);
		
	if (fn && fn!="")
	{
		OpenPlugin(fn);
		LinkPlugin();  	
		m_PluginWindow = CreatePlugin();
		
		m_PluginWindow->show();

		// run once, fill buffer and get location
		m_PluginBuffer=RunPlugin();

		// notify all loops of new source
		for (map<int,LoopWin*>::iterator i=m_LoopWinMap.begin();
			 i!=m_LoopWinMap.end(); i++)
		{
			i->second->m_Loop->SetRecordingSource(m_PluginBuffer->GetBuffer());
		}
	}
}
void Loops::cb_LoadPlugin(Fl_Button* o, void* v)
{((Loops*)(o->parent()->user_data()))->cb_LoadPlugin_i(o,v);}

inline void Loops::cb_ClosePlugin_i(Fl_Button* o, void* v)
{
	m_PluginPaused=o->value();
	
	for (map<int,LoopWin*>::iterator i=m_LoopWinMap.begin();
			 i!=m_LoopWinMap.end(); i++)
	{
		i->second->m_LoopGUI->StopUpdate(!o->value());
	}
}
void Loops::cb_ClosePlugin(Fl_Button* o, void* v)
{((Loops*)(o->parent()->user_data()))->cb_ClosePlugin_i(o,v);}

inline void Loops::cb_Load_i(Fl_Button* o, void* v)
{
	char *fn=fl_file_chooser("Load a Workspace", "*.lws", NULL);
		
	if (fn && fn!='\0')
	{
		ifstream inf(fn);
		
		if (inf)
		{
			ClearLoops();
			inf>>*this;
		}

		string CutFileName = fn;
		// remove the last three chars (".lws")
		CutFileName.resize(CutFileName.size()-4); 
	
		// load samples for the loops
		for (map<int,LoopWin*>::iterator i=m_LoopWinMap.begin();
			 i!=m_LoopWinMap.end(); i++)
		{
			char Number[256]; 
			sprintf(Number,"%d",i->first);
			// add on the wav file name
			string LoopFilename = CutFileName+"Loop"+string(Number)+".wav";
			i->second->m_Loop->LoadWav((char *)LoopFilename.c_str());
			
			i->second->m_LoopGUI->UpdateDataPtr();
			i->second->m_LoopGUI->UpdateValues();
		}
	}
}

void Loops::cb_Load(Fl_Button* o, void* v)
{((Loops*)(o->parent()->user_data()))->cb_Load_i(o,v);}

inline void Loops::cb_Save_i(Fl_Button* o, void* v)
{
	char *fn=fl_file_chooser("Save a Workspace", "*.lws", NULL);
		
	if (fn && fn!='\0')
	{
		ofstream of(fn);
		
		if (of)
		{			
			of<<*this;
		}
		
		string CutFileName = fn;
		// remove the last three chars (".lws")
		CutFileName.resize(CutFileName.size()-4); 

		// save samples from loops
		for (map<int,LoopWin*>::iterator i=m_LoopWinMap.begin();
			 i!=m_LoopWinMap.end(); i++)
		{
			char Number[256]; 
			sprintf(Number,"%d",i->first);
			// add on the wav file name
			string LoopFilename = CutFileName+"Loop"+string(Number)+".wav";
			i->second->m_Loop->SaveWav((char *)LoopFilename.c_str());
		}
	}
}
void Loops::cb_Save(Fl_Button* o, void* v)
{((Loops*)(o->parent()->user_data()))->cb_Save_i(o,v);}

inline void Loops::cb_SnapPoints_i(Fl_Knob* o, void* v)
{
	if(!o->value()) return;
	
	int Angle = 360/(int)o->value();

	for (map<int,LoopWin*>::iterator i=m_LoopWinMap.begin();
		 i!=m_LoopWinMap.end(); i++)
	{
		i->second->m_LoopGUI->SetSnapAngle(Angle);
	}
	
	SnapOutput->value(o->value());
}
void Loops::cb_SnapPoints(Fl_Knob* o, void* v)
{((Loops*)(o->parent()->user_data()))->cb_SnapPoints_i(o,v);}

inline void Loops::cb_Snap_i(Fl_Button* o, void* v)
{
	for (map<int,LoopWin*>::iterator i=m_LoopWinMap.begin();
		 i!=m_LoopWinMap.end(); i++)
	{
		i->second->m_LoopGUI->SetSnap(o->value());
	}
}
void Loops::cb_Snap(Fl_Button* o, void* v)
{((Loops*)(o->parent()->user_data()))->cb_Snap_i(o,v);}
	
//////////////////////////////////////////////////////////
       
int main(int argc, char **argv)
{	
	srand(time(NULL));
	SpiralLoopsInfo::Get()->LoadPrefs();
	
	Loops *loops=new Loops;
	
	Fl::visual(FL_DOUBLE|FL_RGB);
	loops->CreateWindow();
    loops->Window->show(argc, argv);
			
    for (;;) 
	{
    	if (!Fl::check()) break;
		loops->DoIdle();    	
  	}
	
	delete loops;	
	
	return 1;
}

