/*  SpiralSynth
 *  Copyleft (C) 2000 David Griffiths <dave@pawfal.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include "Output.h"
#include "Sample.h"

Output::Output(int Channels) :
m_Amp(0.5),
m_Channels(Channels)
{
	Init();
	// initialise for stereo
	m_Buffer.Allocate(SpiralInfo::BUFSIZE*m_Channels);
	m_Buffer.Zero();
}

void Output::Send(Sample &data, float LeftVolume=1, float RightVolume=1) 
{
	long temp;
	int on=0;
	for (int n=0; n<SpiralInfo::BUFSIZE; n++)
	{
		temp=static_cast<long>(data[n]*m_Amp);
		
		// clip
		if (temp<-SpiralInfo::MAXSAMPLE) temp=-SpiralInfo::MAXSAMPLE;
		if (temp>SpiralInfo::MAXSAMPLE) temp=SpiralInfo::MAXSAMPLE;

		// stereo channels
		m_Buffer.Set(on,m_Buffer[on]+temp*LeftVolume);
		if (m_Channels==2) 
		{
			on++;
			m_Buffer.Set(on,m_Buffer[on]+temp*RightVolume); 
		}
		on++;
	}
}

void Output::SendStereo(Sample &data) 
{
	if (m_Channels!=2) return;
	
	int on=0;
	long temp;
	for (int n=0; n<SpiralInfo::BUFSIZE*m_Channels; n++)
	{
		// stereo channels		
		m_Buffer.Set(on,m_Buffer[on]+data[n]*m_Amp);
		on++;
	}
}


void Output::Play()
{
	#if __BYTE_ORDER == BIG_ENDIAN
	static Sample *buf16 = new Sample[SpiralInfo::BUFSIZE*m_Channels];
	for (int n=0; n<SpiralInfo::BUFSIZE*m_Channels; n++)
	{
		buf16[n]=(Sample)((m_Buffer[n]<<8)&0xff00) | ((m_Buffer[n]>>8)&0xff);
	}
	if (SpiralInfo::REALTIMEOUT)
	{
		write(m_Dspfd,buf16,sizeof(Sample)*SpiralInfo::BUFSIZE*m_Channels);
	}
	
	if(m_Wav.Recording()) 
	{
		m_Wav.Save(SpiralInfo::BUFSIZE*m_Channels,buf16);
	}
	#else
	if (SpiralInfo::REALTIMEOUT)
	{
		write(m_Dspfd,m_Buffer.GetBuffer(),sizeof(short)*SpiralInfo::BUFSIZE*m_Channels);
	}
	
	if(m_Wav.Recording()) 
	{
		m_Wav.Save(m_Buffer);
	}
	#endif
}

void Output::ClearBuffer()
{
	m_Buffer.Zero();
}

void Output::Init()
{ 
  if (!SpiralInfo::REALTIMEOUT)
  {
  	return;
  }	
	
  int r,val;
  
  m_Dspfd = open(SpiralInfo::OUTPUTFILE.c_str(),O_WRONLY);  
  if(m_Dspfd<0) 
  	{
     fprintf(stderr,"Can't open audio driver.\n");
     exit(999);
  	}
   
  r = ioctl(m_Dspfd,SNDCTL_DSP_RESET,NULL);

  if (r>=0)
  	{
	 val=262152;
	 r=ioctl(m_Dspfd, SNDCTL_DSP_SETFRAGMENT, &val);
    }
  if(r>=0) 
  	{
	 val = 1;
     r = ioctl(m_Dspfd, SOUND_PCM_WRITE_CHANNELS, &val);
	}
  if(r>=0) 
  	{
     val = AFMT_S16_LE;
     r = ioctl(m_Dspfd,SNDCTL_DSP_SETFMT,&val);	 
  	}
  if(r>=0) 
  	{
	 if (m_Channels==2) val=1;
	 else val=0;
	
     r = ioctl(m_Dspfd,SNDCTL_DSP_STEREO,&val);
  	}
  if(r>=0) 
  	{
     val = SpiralInfo::SAMPLERATE;
     r = ioctl(m_Dspfd,SNDCTL_DSP_SPEED,&val);
  	}
  if(r<0) 
  	{
     fprintf(stderr,"Sound device did not accept settings.\n");
     exit(999);
  	}  
}

