/*
 * Copyright (c) 2009, 2010 Nhat Minh Lê
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* LINTLIBRARY */

#include <sys/cdefs.h>
#include <sys/queue.h>

#include <assert.h>
#include <ctype.h>
#include <errno.h>
#include <math.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifndef __NetBSD__
#include <regxml/compat.h>
#endif
#include <regxml/bitstring.h>
#include <regxml/buffer.h>
#include <regxml/regxml.h>

#define NAMEMAPSIZE ((size_t)1 << NAMEMAPWIDTH)
#define NAMEMAPWIDTH 8
#define MINSTACK 20

struct regxml_elem regxml_elem_fail = {
	0, 0, 0,
	/* .rxe_type = */ REGXML_XFAIL
};
struct regxml_elem regxml_elem_wait = {
	0, 0, 0,
	/* .rxe_type = */ REGXML_XWAIT
};

static const char *errmap[] = {
	/* [0] = */ NULL,
	/* [REGXML_EOF] = */ "end of file",
	/* [REGXML_NOMATCH] = */ "pattern did not match",
	/* [REGXML_CLOSING] */ NULL,
	/* [REGXML_ESYSTEM] */ NULL,
	/* [REGXML_EREGEX] = */ "regex error",
	/* [REGXML_EXML] = */ "bad XML",
	/* [REGXML_EAGAIN] = */ "more input needed",
	/* [REGXML_EWAIT] = */ "incomplete subtasks",
	/* [REGXML_ESYNTAX] = */ "pattern syntax error",
	/* [REGXML_EUSER] = */ "user function failed",
	/* [REGXML_ENOIMPL] = */ "not implemented"
};

static int valtonumber(struct regxml_vstack *);
static int valtostring(struct regxml_vstack *);
static int valtoregex(struct regxml_vstack *);
static int valequal(struct regxml_vstack *);
static int valnot(struct regxml_vstack *);
#if 0
static int valdup(struct regxml_vstack *);
#endif

static struct regxml_elem *maxelem(const struct regxml_elem *,
    const struct regxml_elem *);

static uint32_t hash(const char *, size_t);
static char *allocname(struct regxml *, const char *);
static struct regxml_elem *newelem(struct regxml *, regxml_xtype_t,
    const char *, const char *);
static int copyattrv(struct regxml_elem *, char * const *);
static void swapbitstr(regxml_bitstr_t **, regxml_bitstr_t **);
static void freeelem(struct regxml *, struct regxml_elem *);
static void freeelemrec(struct regxml *, struct regxml_elem *);
static void decref(struct regxml *, struct regxml_elem *);

static struct regxml_task *newtask(struct regxml *, struct regxml_elem *,
    struct regxml_node *);
static void freetask(struct regxml *, struct regxml_task *);

static int pushopening(struct regxml *, regxml_xtype_t,
    const char *, const char *, char * const *);
static int pushclosing(struct regxml *);

static void prune(struct regxml *);

static void returntask(struct regxml *, struct regxml_task *);
static int spawntask(struct regxml *, struct regxml_elem *,
    struct regxml_node *, struct regxml_task *);
static int flushtasks(struct regxml *);

static int expr(struct regxml_vstack *, struct regxml_elem *,
    struct regxml_node *);
static int evalexpr(struct regxml *, struct regxml_elem *,
    struct regxml_node *);
static int eval(struct regxml *, struct regxml_task *);
static int peval(struct regxml *, struct regxml_node *);

static int scopeeq(const char *, const char *);
static int scope(struct regxml *);
static int stepelem(struct regxml_elem *, regxml_xid_t,
    struct regxml_elem **, int);
static int stepin(struct regxml_elem **, regxml_xid_t *, int);
static int step(struct regxml *);
static void complongest(struct regxml *);
static int walk(struct regxml *);

int
regxml_initvstack(struct regxml_vstack *vs, struct regxml *reg, int ini)
{
	union regxml_value *tmp;

	if (ini < MINSTACK)
		ini = MINSTACK;

	tmp = malloc(ini * sizeof *tmp);
	if (tmp == NULL)
		return REGXML_ESYSTEM;

	vs->rxvs_reg = reg;
	vs->rxvs_base = tmp;
	vs->rxvs_size = ini;
	vs->rxvs_top = 0;
	return 0;
}

int
regxml_checkvstack(struct regxml_vstack *vs, int extra)
{
	union regxml_value *tmp;
	int n;

	if (vs->rxvs_size - vs->rxvs_top >= extra)
		return 0;

	for (n = vs->rxvs_size; n - vs->rxvs_top < extra; n *= 2)
		continue;
	tmp = realloc(vs->rxvs_base, n * sizeof *tmp);
	if (tmp == NULL)
		return REGXML_ESYSTEM;

	vs->rxvs_base = tmp;
	vs->rxvs_size = n;
	return 0;
}

void
regxml_freeval(struct regxml *reg, union regxml_value *val)
{
	switch (val->rxv_unit._type) {
	case REGXML_STRING:
		if (!val->rxv_string._shared)
			free(val->rxv_string._data);
		break;
	case REGXML_REGEX:
		if (!val->rxv_regex._shared) {
			regfree(val->rxv_regex._data);
			free(val->rxv_regex._data);
		}
		break;
	case REGXML_XRANGE:
		decref(reg, val->rxv_xrange._start);
		break;
	default:
		break;
	}
}

void
regxml_cleanvstack(struct regxml_vstack *vs, int k)
{
	_DIAGASSERT(vs->rxvs_top - k >= 0);
	for (; k > 0; --k)
		regxml_freeval(vs->rxvs_reg, &vs->rxvs_base[--vs->rxvs_top]);
}

void
regxml_freevstack(struct regxml_vstack *vs)
{
	regxml_cleanvstack(vs, vs->rxvs_top);
	free(vs->rxvs_base);
}

static int
valtonumber(struct regxml_vstack *vs)
{
	union regxml_value *val;
	double d;
	int r;

	val = regxml_getval(vs, -1);
	switch (val->rxv_unit._type) {
	case REGXML_UNIT:
		d = 0.0;
		break;
	case REGXML_NUMBER:
		return 0;
	case REGXML_XRANGE:
		r = valtostring(vs);
		if (r != 0)
			return r;
		/* FALLTHROUGH */
	case REGXML_STRING:
		d = strtod(val->rxv_string._data, NULL);
		break;
	case REGXML_REGEX:
		return REGXML_ENOIMPL;
	default:
		/* NOTREACHED */
		abort();
	}
	regxml_cleanvstack(vs, 1);
	regxml_pushnumber(vs, d);
	return 0;
}

static int
valtostring(struct regxml_vstack *vs)
{
	union regxml_value *val;
	char *s;
	int shared;

	val = regxml_getval(vs, -1);
	switch (val->rxv_unit._type) {
	case REGXML_UNIT:
		s = __UNCONST("");
		shared = 1;
		break;
	case REGXML_NUMBER:
		if (asprintf(&s, "%f", val->rxv_number._data) == -1)
			return REGXML_ESYSTEM;
		shared = 0;
		break;
	case REGXML_STRING:
		return 0;
	case REGXML_REGEX:
		return REGXML_ENOIMPL;
	case REGXML_XRANGE:
		/* TODO: XML intervals in expressions support. */
		return REGXML_ENOIMPL;
	default:
		/* NOTREACHED */
		abort();
	}
	regxml_cleanvstack(vs, 1);
	if (shared)
		regxml_pushsharedstring(vs, s);
	else
		regxml_pushstring(vs, s);
	return 0;
}

static int
valtoregex(struct regxml_vstack *vs)
{
	union regxml_value *val;
	char *s;
	int r;

	val = regxml_getval(vs, -1);
	switch (val->rxv_unit._type) {
	case REGXML_STRING:
		break;
	case REGXML_REGEX:
		return 0;
	default:
		r = valtostring(vs);
		if (r != 0)
			return r;
	}
	s = val->rxv_string._data;
	regxml_cleanvstack(vs, 1);
	return regxml_pushregex(vs, s, vs->rxvs_reg->rx_reflags);
}

static int
valequal(struct regxml_vstack *vs)
{
	union regxml_value *val1, *val2;
	int b;

	val1 = regxml_getval(vs, -2);
	val2 = regxml_getval(vs, -1);
	b = 0;
	if (val1->rxv_unit._type == val2->rxv_unit._type) {
		switch (val1->rxv_unit._type) {
		case REGXML_UNIT:
			b = 1;
			break;
		case REGXML_NUMBER:
			b = val1->rxv_number._data ==
			    val2->rxv_number._data;
			break;
		case REGXML_STRING:
			b = strcmp(val1->rxv_string._data,
			    val2->rxv_string._data) == 0;
			break;
		case REGXML_REGEX:
			return REGXML_ENOIMPL;
		case REGXML_XRANGE:
			return REGXML_ENOIMPL;
		default:
			/* NOTREACHED */
			abort();
		}
	}
	regxml_cleanvstack(vs, 2);
	regxml_pushbool(vs, b);
	return 0;
}

static int
valnot(struct regxml_vstack *vs)
{
	int b;

	b = regxml_gettype(vs, -1) == REGXML_UNIT;
	regxml_cleanvstack(vs, 1);
	regxml_pushbool(vs, b);
	return 0;
}

#if 0
static int
valdup(struct regxml_vstack *vs)
{
	regxml_pushsharedval(vs, regxml_getval(vs, -1));
	return 0;
}
#endif

static struct regxml_elem *
maxelem(const struct regxml_elem *el1, const struct regxml_elem *el2)
{
	const struct regxml_elem *el;

	if (el1 == NULL)
		el = el2;
	else if (el2 == NULL)
		el = el1;
	else if (el1->rxe_lowid > el2->rxe_lowid)
		el = el1;
	else
		el = el2;

	return __UNCONST(el);
}

static uint32_t
hash(const char *s, size_t w)
{
	uint32_t k;

	k = 0;
	while (*s != '\0') {
		k += *s++;
		k += k << 10;
		k ^= k >> 6;
	}
	k += k << 3;
	k ^= k >> 11;
	k += k << 15;

	return k & (((uint32_t)1 << w) - 1);
}

static char *
allocname(struct regxml *reg, const char *name)
{
	struct regxml_namepairq *head;
	struct regxml_namepair *p;
	uint32_t k;

	k = hash(name, reg->rx_namemap.rxnm_width);
	head = &reg->rx_namemap.rxnm_array[k % reg->rx_namemap.rxnm_size];
	for (p = SLIST_FIRST(head);
	    p != NULL;
	    p = SLIST_NEXT(p, rxnp_entries)) {
		if (strcmp(p->rxnp_name, name) == 0)
			return p->rxnp_name;
	}
	p = malloc(sizeof *p);
	if (p == NULL)
		return NULL;
	p->rxnp_name = strdup(name);
	if (p->rxnp_name == NULL) {
		free(p);
		return NULL;
	}
	SLIST_INSERT_HEAD(head, p, rxnp_entries);
	return p->rxnp_name;
}

static struct regxml_elem *
newelem(struct regxml *reg, regxml_xtype_t type, const char *name,
    const char *value)
{
	struct regxml_elem *el;
	regxml_id_t i;

	if (reg->rx_elempool != NULL) {
		el = reg->rx_elempool;
		reg->rx_elempool = reg->rx_elempool->rxe_next;

		/* regxml_bit_zero(el->rxe_state, reg->rx_psize); */
		regxml_bit_zero(el->rxe_cstate0, reg->rx_psize);
		regxml_bit_zero(el->rxe_nstate, reg->rx_psize);
	} else {
		el = malloc(sizeof *el);
		if (el == NULL)
			goto badelem;

		el->rxe_state = regxml_bit_alloc(reg->rx_psize);
		el->rxe_cstate0 = regxml_bit_alloc(reg->rx_psize);
		el->rxe_nstate = regxml_bit_alloc(reg->rx_psize);
		el->rxe_memory =
		    malloc(reg->rx_psize * sizeof *el->rxe_memory);

		if (el->rxe_state == NULL ||
		    el->rxe_cstate0 == NULL || el->rxe_nstate == NULL ||
		    el->rxe_memory == NULL)
			goto badarrays;
	}

	/*
	 * All elements begin with one reference which is managed
	 * specially by the matcher.
	 */
	el->rxe_refcount = 1;

	el->rxe_lowid = 0;
	el->rxe_highid = REGXML_ID_MAX;

	el->rxe_type = type;
	el->rxe_name = name != NULL ? allocname(reg, name) : NULL;
	el->rxe_value = value != NULL ? strdup(value) : NULL;

	if ((name != NULL && el->rxe_name == NULL) ||
	    (value != NULL && el->rxe_value == NULL))
		goto badstrings;

	el->rxe_parent = NULL;
	el->rxe_child0 = NULL;
	el->rxe_nchild = 0;
	el->rxe_pos = 0;

	el->rxe_prev = NULL;
	el->rxe_next = NULL;

	el->rxe_attrv = NULL;
	el->rxe_nattr = 0;

	el->rxe_lineno = reg->rx_lineno;

	for (i = 0; i < reg->rx_psize; ++i)
		el->rxe_memory[i] = NULL;

	el->rxe_cpending = 1;
	el->rxe_npending = 1;

	return el;

badstrings:
	free(el->rxe_value);
badarrays:
	free(el->rxe_state);
	free(el->rxe_cstate0);
	free(el->rxe_nstate);
	free(el->rxe_memory);
badelem:
	free(el);
	return NULL;
}

static int
copyattrv(struct regxml_elem *el, char * const *attrv)
{
	regxml_xid_t i, nattr;

	nattr = 0;
	while (attrv[nattr] != NULL && attrv[nattr+1] != NULL)
		nattr += 2;

	if (nattr == 0) {
		el->rxe_attrv = NULL;
		el->rxe_nattr = 0;
		return 0;
	}

	el->rxe_attrv = malloc((nattr + 1) * sizeof *el->rxe_attrv);
	if (el->rxe_attrv == NULL) {
		free(el);
		return REGXML_ESYSTEM;
	}
	el->rxe_nattr = nattr;

	for (i = 0; i < nattr; ++i) {
		el->rxe_attrv[i] = strdup(attrv[i]);
		if (el->rxe_attrv[i] == NULL)
			goto bad;
	}
	el->rxe_attrv[nattr] = NULL;

	return 0;

bad:
	while (i-- > 0)
		free(el->rxe_attrv[i]);
	free(el->rxe_attrv);
	return REGXML_ESYSTEM;
}

static void
swapbitstr(regxml_bitstr_t **pa, regxml_bitstr_t **pb)
{
	regxml_bitstr_t *tmp;
	tmp = *pa;
	*pa = *pb;
	*pb = tmp;
}

static void
freeelem(struct regxml *reg, struct regxml_elem *el)
{
	regxml_id_t i;

	if (el->rxe_attrv != NULL) {
		for (i = 0; i < el->rxe_nattr; ++i)
			free(el->rxe_attrv[i]);
		free(el->rxe_attrv);
		el->rxe_attrv = NULL;
	}

	free(el->rxe_value);

	el->rxe_next = reg->rx_elempool;
	reg->rx_elempool = el;
}

static void
freeelemrec(struct regxml *reg, struct regxml_elem *el)
{
	struct regxml_elem *child, *tmp;

	child = el->rxe_child0;
	while (child != NULL) {
		tmp = child;
		child = child->rxe_next;
		freeelemrec(reg, tmp);
	}
	freeelem(reg, el);
}

static void
decref(struct regxml *reg, struct regxml_elem *el)
{
	struct regxml_elem *child, *tmp;

	if (--el->rxe_refcount > 0)
		return;

	_DIAGASSERT(el->rxe_parent == NULL);

	child = el->rxe_child0;
	while (child != NULL) {
		tmp = child;
		child = child->rxe_next;
		tmp->rxe_parent = NULL;
		decref(reg, tmp);
	}

	if (el->rxe_prev != NULL)
		el->rxe_prev->rxe_next = NULL;
	if (el->rxe_next != NULL)
		el->rxe_next->rxe_prev = NULL;

	freeelem(reg, el);
}

static struct regxml_task *
newtask(struct regxml *reg, struct regxml_elem *current,
    struct regxml_node *node)
{
	struct regxml_task *task;

	if (!SIMPLEQ_EMPTY(&reg->rx_taskpool)) {
		task = SIMPLEQ_FIRST(&reg->rx_taskpool);
		SIMPLEQ_REMOVE_HEAD(&reg->rx_taskpool, rxt_entries);
	} else {
		task = malloc(sizeof *task);
		if (task == NULL)
			return NULL;
	}

	task->rxt_current = current;
	task->rxt_node = node;
	task->rxt_state = 0;
	task->rxt_return = NULL;

	return task;
}

static void
freetask(struct regxml *reg, struct regxml_task *task)
{
	SIMPLEQ_INSERT_HEAD(&reg->rx_taskpool, task, rxt_entries);
}

int
regxml_create(struct regxml **regptr, const char *s, int flags)
{
	struct regxml *reg;
	size_t i;
	int r;

	_DIAGASSERT(s != NULL);
	_DIAGASSERT(regptr != NULL);

	reg = malloc(sizeof *reg);
	if (reg == NULL)
		return REGXML_ESYSTEM;

	reg->rx_namemap.rxnm_size = NAMEMAPSIZE;
	reg->rx_namemap.rxnm_width = NAMEMAPWIDTH;
	reg->rx_namemap.rxnm_array =
	    malloc(NAMEMAPSIZE * sizeof *reg->rx_namemap.rxnm_array);
	if (reg->rx_namemap.rxnm_array == NULL) {
		r = REGXML_ESYSTEM;
		goto badmap;
	}
	for (i = 0; i < reg->rx_namemap.rxnm_size; ++i)
		SLIST_INIT(&reg->rx_namemap.rxnm_array[i]);

	r = regxml_pattern_comp(reg, s, NULL, flags);
	if (r != 0)
		goto badpat;

	r = regxml_initvstack(&reg->rx_vstack, reg, 0);
	if (r != 0)
		goto badstack;

	reg->rx_lineno = 0;

	reg->rx_elempool = NULL;
	reg->rx_tree = newelem(reg, REGXML_ROOT, NULL, NULL);
	if (reg->rx_tree == NULL) {
		r = REGXML_ESYSTEM;
		goto badpool;
	}

	reg->rx_elemid = 0;
	reg->rx_tree->rxe_lowid = reg->rx_elemid++;
	reg->rx_pointer = reg->rx_tree;
	reg->rx_start = reg->rx_tree;
	reg->rx_startid = 1;

	reg->rx_scope = NULL;
	reg->rx_inscope = 1;

	SIMPLEQ_INIT(&reg->rx_tasks);
	SIMPLEQ_INIT(&reg->rx_taskpool);

	reg->rx_reflags = flags & REGXML_ICASE ? REG_ICASE : 0;

	reg->rx_match = NULL;
	reg->rx_maxmatch = 0;
	reg->rx_smatch = 0;
	reg->rx_dofetch = 0;

	reg->rx_longest = !!(flags & REGXML_LONGEST);
	reg->rx_matchstep = !!(flags & REGXML_MATCHSTEP);

	*regptr = reg;
	return 0;

badpool:
	regxml_freevstack(&reg->rx_vstack);
badstack:
	regxml_pattern_free(reg);
badpat:
	free(reg->rx_namemap.rxnm_array);
badmap:
	free(reg);
	return r;
}

void
regxml_destroy(struct regxml *reg)
{
	struct regxml_elem *el, *tmp;
	struct regxml_namepair *p;
	struct regxml_task *task;
	size_t i;

	_DIAGASSERT(reg != NULL);

	while (!SIMPLEQ_EMPTY(&reg->rx_taskpool)) {
		task = SIMPLEQ_FIRST(&reg->rx_taskpool);
		SIMPLEQ_REMOVE_HEAD(&reg->rx_taskpool, rxt_entries);
		free(task);
	}

	freeelemrec(reg, reg->rx_tree);
	el = reg->rx_elempool;
	while (el != NULL) {
		tmp = el;
		el = el->rxe_next;

		regxml_bit_free(tmp->rxe_state);
		regxml_bit_free(tmp->rxe_cstate0);
		regxml_bit_free(tmp->rxe_nstate);
		free(tmp->rxe_memory);
		free(tmp);
	}

	free(reg->rx_scope);
	regxml_freevstack(&reg->rx_vstack);
	regxml_pattern_free(reg);

	for (i = 0; i < reg->rx_namemap.rxnm_size; ++i) {
		while (!SLIST_EMPTY(&reg->rx_namemap.rxnm_array[i])) {
			p = SLIST_FIRST(&reg->rx_namemap.rxnm_array[i]);
			SLIST_REMOVE_HEAD(&reg->rx_namemap.rxnm_array[i],
			    rxnp_entries);
			free(p->rxnp_name);
			free(p);
		}
	}
	free(reg->rx_namemap.rxnm_array);

	free(reg);
}

/* ARGSUSED */
size_t
regxml_strerror(struct regxml *reg, int code, char *s, size_t n)
{
	switch (code) {
	case REGXML_ESYSTEM:
		(void)strerror_r(errno, s, n);
		return strlen(s);
	default:
		if (code >= REGXML_NERROR || errmap[code] == NULL)
			return snprintf(s, n, "not an error");
		return snprintf(s, n, "%s", errmap[code]);
	}
}

static int
pushopening(struct regxml *reg, regxml_xtype_t type,
    const char *name, const char *value,
    char * const *attrv)
{
	struct regxml_elem *el;

	el = newelem(reg, type, name, value);
	if (el == NULL)
		return REGXML_ESYSTEM;
	el->rxe_lowid = reg->rx_elemid++;

	if (attrv != NULL && copyattrv(el, attrv) != 0) {
		free(el);
		return REGXML_ESYSTEM;
	}

	if (reg->rx_pointer->rxe_cpending) {
		el->rxe_parent = reg->rx_pointer;
		reg->rx_pointer->rxe_child0 = el;
	} else {
		reg->rx_pointer->rxe_npending = 0;
		reg->rx_pointer->rxe_next = el;
		el->rxe_prev = reg->rx_pointer;
		el->rxe_parent = reg->rx_pointer->rxe_parent;
	}

	el->rxe_pos = ++el->rxe_parent->rxe_nchild;
	reg->rx_pointer = el;

	return 0;
}

static int
pushclosing(struct regxml *reg)
{
	if (!reg->rx_pointer->rxe_cpending) {
		reg->rx_pointer->rxe_npending = 0;
		reg->rx_pointer = reg->rx_pointer->rxe_parent;
	}

	if (reg->rx_pointer->rxe_type == REGXML_ROOT)
		return REGXML_EXML;

	reg->rx_pointer->rxe_highid = reg->rx_elemid++;
	reg->rx_pointer->rxe_cpending = 0;

	return 0;
}

int
regxml_pushopening(struct regxml *reg, const char *name, char * const *attrv)
{
	_DIAGASSERT(reg != NULL);
	_DIAGASSERT(name != NULL);

	return pushopening(reg, REGXML_NODE, name, NULL, attrv);
}

int
regxml_pushclosing(struct regxml *reg)
{
	_DIAGASSERT(reg != NULL);

	return pushclosing(reg);
}

int
regxml_pushtext(struct regxml *reg, const char *value)
{
	int r;

	_DIAGASSERT(reg != NULL);
	_DIAGASSERT(value != NULL);

	r = pushopening(reg, REGXML_TEXT, NULL, value, NULL);
	if (r != 0)
		return r;
	r = pushclosing(reg);
	if (r != 0)
		return r;
	return 0;
}

int
regxml_pushcomment(struct regxml *reg, const char *value)
{
	int r;

	_DIAGASSERT(reg != NULL);
	_DIAGASSERT(value != NULL);

	r = pushopening(reg, REGXML_COMMENT, NULL, value, NULL);
	if (r != 0)
		return r;
	r = pushclosing(reg);
	if (r != 0)
		return r;
	return 0;
}

int
regxml_pushpi(struct regxml *reg, const char *name, const char *value)
{
	int r;

	_DIAGASSERT(reg != NULL);
	_DIAGASSERT(name != NULL);
	_DIAGASSERT(value != NULL);

	r = pushopening(reg, REGXML_PI, name, value, NULL);
	if (r != 0)
		return r;
	r = pushclosing(reg);
	if (r != 0)
		return r;
	return 0;
}

int
regxml_end(struct regxml *reg)
{
	_DIAGASSERT(reg != NULL);

	if (reg->rx_pointer->rxe_type != REGXML_ROOT &&
	    reg->rx_pointer->rxe_parent->rxe_type != REGXML_ROOT)
		return REGXML_EXML;

	if (reg->rx_pointer->rxe_type != REGXML_ROOT)
		reg->rx_pointer->rxe_npending = 0;

	reg->rx_tree->rxe_highid = reg->rx_elemid;
	reg->rx_tree->rxe_cpending = 0;

	return 0;
}

int
regxml_setscope(struct regxml *reg, const char *sc)
{
	_DIAGASSERT(reg != NULL);

	free(reg->rx_scope);
	if (sc == NULL) {
		reg->rx_scope = NULL;
		reg->rx_inscope = 1;
	} else if (*sc == '\0') {
		reg->rx_scope = NULL;
		reg->rx_inscope = 0;
	} else {
		reg->rx_scope = strdup(sc);
		if (reg->rx_scope == NULL)
			return REGXML_ESYSTEM;
		reg->rx_inscope = 0;
	}
	return 0;
}

static void
prune(struct regxml *reg)
{
	struct regxml_elem *base, *el1, *el2, *tmp;

	/*
	 * Find highest parent not owned externally. Everything below
	 * must not be pruned.
	 */
	base = reg->rx_start;
	for (el1 = reg->rx_start; el1->rxe_type != REGXML_ROOT;
	    el1 = el1->rxe_parent) {
		if (el1->rxe_refcount > 1)
			base = el1;
	}

	for (el1 = base; el1->rxe_type != REGXML_ROOT;
	    el1 = el1->rxe_parent) {
		el2 = el1->rxe_prev;
		while (el2 != NULL) {
			tmp = el2;
			el2 = el2->rxe_prev;
			tmp->rxe_parent = NULL;
			decref(reg, tmp);
		}
		el1->rxe_parent->rxe_child0 = el1;
	}
}

static void
returntask(struct regxml *reg, struct regxml_task *task)
{
	if (task->rxt_return != NULL) {
		++task->rxt_return->rxt_state;
		SIMPLEQ_INSERT_TAIL(&reg->rx_tasks, task->rxt_return,
		    rxt_entries);
	}
}

static int
spawntask(struct regxml *reg, struct regxml_elem *current,
    struct regxml_node *q, struct regxml_task *ret)
{
	struct regxml_elem *res;
	struct regxml_task *ts;

	res = current->rxe_memory[q->rxn_id];
	if (res != NULL) {
		if (res == REGXML_WAIT)
			return REGXML_EWAIT;
		if (ret != NULL)
			++ret->rxt_state;
		return 0;
	}

	current->rxe_memory[q->rxn_id] = REGXML_WAIT;
	ts = newtask(reg, current, q);
	if (ts == NULL)
		return REGXML_ESYSTEM;

	ts->rxt_return = ret;
	SIMPLEQ_INSERT_TAIL(&reg->rx_tasks, ts, rxt_entries);

	return REGXML_EWAIT;
}

static int
flushtasks(struct regxml *reg)
{
	struct regxml_task *task;
	int r;

	while (!SIMPLEQ_EMPTY(&reg->rx_tasks)) {
		task = SIMPLEQ_FIRST(&reg->rx_tasks);
		r = eval(reg, task);
		switch (r) {
		case 0:
			SIMPLEQ_REMOVE_HEAD(&reg->rx_tasks, rxt_entries);
			freetask(reg, task);
			break;
		case REGXML_EWAIT:
			SIMPLEQ_REMOVE_HEAD(&reg->rx_tasks, rxt_entries);
			break;
		default:
			return r;
		}
	}

	return 0;
}

/*
 * Evaluate expression and return value in the specified object.
 */
static int
expr(struct regxml_vstack *vs, struct regxml_elem *current,
    struct regxml_node *q)
{
	regxml_xid_t i;
	char *s;
	double d;
	int b;
	int r;

	r = regxml_checkvstack(vs, 2);
	if (r != 0)
		return r;

	switch (q->rxn_type) {
	case REGXML_FNODE:
		if (current->rxe_type == REGXML_NODE)
			regxml_pushsharedstring(vs, current->rxe_name);
		else
			regxml_pushunit(vs);
		break;

	case REGXML_FTEXT:
		if (current->rxe_type == REGXML_TEXT)
			regxml_pushsharedstring(vs, current->rxe_value);
		else
			regxml_pushunit(vs);
		break;

	case REGXML_FCOMMENT:
		if (current->rxe_type == REGXML_COMMENT)
			regxml_pushsharedstring(vs, current->rxe_value);
		else
			regxml_pushunit(vs);
		break;

	case REGXML_FPI:
		r = expr(vs, current, q->rxn_left);
		if (r != 0)
			return r;
		r = valtostring(vs);
		if (r != 0)
			return r;
		if (current->rxe_type == REGXML_PI &&
		    strcmp(current->rxe_name,
			regxml_getval(vs, -1)->rxv_string._data) == 0)
			regxml_retpushsharedstring(vs, 1, current->rxe_value);
		else
			regxml_retpushunit(vs, 1);
		break;

	case REGXML_FATTR:
		r = expr(vs, current, q->rxn_left);
		if (r != 0)
			return r;
		r = valtostring(vs);
		if (r != 0)
			return r;
		if (current->rxe_type == REGXML_NODE) {
			s = regxml_getval(vs, -1)->rxv_string._data;
			for (i = 0; i < current->rxe_nattr; i += 2) {
				if (strcmp(current->rxe_attrv[i], s) == 0) {
					regxml_retpushsharedstring(vs, 1,
					    current->rxe_attrv[i+1]);
					goto end;
				}
			}
		}
		regxml_retpushunit(vs, 1);
		break;

	case REGXML_FNAME:
		regxml_pushsharedstring(vs, current->rxe_name);
		break;

	case REGXML_FVALUE:
		regxml_pushsharedstring(vs, current->rxe_value);
		break;

	case REGXML_FRELPOS:
		regxml_pushnumber(vs, current->rxe_pos);
		break;

	case REGXML_ADD:
	case REGXML_SUB:
	case REGXML_MUL:
	case REGXML_DIV:
	case REGXML_MOD:
		r = expr(vs, current, q->rxn_left);
		if (r != 0)
			return r;
		r = valtonumber(vs);
		if (r != 0)
			return r;
		r = expr(vs, current, q->rxn_right);
		if (r != 0)
			return r;
		r = valtonumber(vs);
		if (r != 0)
			return r;
		switch (q->rxn_type) {
		case REGXML_ADD:
			d = regxml_getval(vs, -2)->rxv_number._data +
			    regxml_getval(vs, -1)->rxv_number._data;
			break;
		case REGXML_SUB:
			d = regxml_getval(vs, -2)->rxv_number._data -
			    regxml_getval(vs, -1)->rxv_number._data;
			break;
		case REGXML_MUL:
			d = regxml_getval(vs, -2)->rxv_number._data *
			    regxml_getval(vs, -1)->rxv_number._data;
			break;
		case REGXML_DIV:
			d = regxml_getval(vs, -2)->rxv_number._data /
			    regxml_getval(vs, -1)->rxv_number._data;
			break;
		case REGXML_MOD:
			d = fmod(regxml_getval(vs, -2)->rxv_number._data,
			    regxml_getval(vs, -1)->rxv_number._data);
			break;
		default:
			/* NOTREACHED */
			abort();
		}
		regxml_retpushnumber(vs, 2, d);
		break;

	case REGXML_LT:
	case REGXML_GT:
	case REGXML_LE:
	case REGXML_GE:
		r = expr(vs, current, q->rxn_left);
		if (r != 0)
			return r;
		r = valtonumber(vs);
		if (r != 0)
			return r;
		r = expr(vs, current, q->rxn_right);
		if (r != 0)
			return r;
		r = valtonumber(vs);
		if (r != 0)
			return r;
		switch (q->rxn_type) {
		case REGXML_LT:
			b = regxml_getval(vs, -2)->rxv_number._data <
			    regxml_getval(vs, -1)->rxv_number._data;
			break;
		case REGXML_GT:
			b = regxml_getval(vs, -2)->rxv_number._data >
			    regxml_getval(vs, -1)->rxv_number._data;
			break;
		case REGXML_LE:
			b = regxml_getval(vs, -2)->rxv_number._data <=
			    regxml_getval(vs, -1)->rxv_number._data;
			break;
		case REGXML_GE:
			b = regxml_getval(vs, -2)->rxv_number._data >=
			    regxml_getval(vs, -1)->rxv_number._data;
			break;
		default:
			/* NOTREACHED */
			abort();
		}
		regxml_retpushbool(vs, 2, b);
		break;

	case REGXML_EQ:
		r = expr(vs, current, q->rxn_left);
		if (r != 0)
			return r;
		r = expr(vs, current, q->rxn_right);
		if (r != 0)
			return r;
		(void)valequal(vs);
		break;

	case REGXML_NE:
		r = expr(vs, current, q->rxn_left);
		if (r != 0)
			return r;
		r = expr(vs, current, q->rxn_right);
		if (r != 0)
			return r;
		(void)valequal(vs);
		(void)valnot(vs);
		break;

	case REGXML_RE:
	case REGXML_NR:
		r = expr(vs, current, q->rxn_left);
		if (r != 0)
			return r;
		r = valtostring(vs);
		if (r != 0)
			return r;
		r = expr(vs, current, q->rxn_right);
		if (r != 0)
			return r;
		r = valtoregex(vs);
		if (r != 0)
			return r;
		r = regexec(regxml_getval(vs, -1)->rxv_regex._data,
		    regxml_getval(vs, -2)->rxv_string._data, 0, NULL, 0);
		if (r != 0 && r != REG_NOMATCH)
			return REGXML_EREGEX;
		if (q->rxn_type == REGXML_RE)
			regxml_retpushbool(vs, 2, r == 0);
		else
			regxml_retpushbool(vs, 2, r != 0);
		break;

	case REGXML_CONSTANT:
		regxml_pushsharedval(vs, &q->rxn_value);
		break;

	default:
		/* TODO: XML intervals in expressions support. */
		return REGXML_ENOIMPL;
	}

end:
	return 0;
}

/*
 * Evaluate expression and memoize current node if true, REGXML_FAIL
 * if false.
 */
static int
evalexpr(struct regxml *reg, struct regxml_elem *current,
    struct regxml_node *q)
{
	int r;

	r = expr(&reg->rx_vstack, current, q);
	if (r != 0)
		return r;
	if (regxml_getval(&reg->rx_vstack, -1)->rxv_unit._type == REGXML_UNIT)
		current->rxe_memory[q->rxn_id] = REGXML_FAIL;
	else
		current->rxe_memory[q->rxn_id] = current;
	regxml_cleanvstack(&reg->rx_vstack, 1);
	return 0;
}

/*
 * Evaluate a pattern node against an element in conservative mode. In
 * this mode, all pattern nodes are evaluated, whether they pertain to
 * the current element or not. If they refer to some unavailable data,
 * processing is halted and REGXML_EAGAIN or REGXML_EWAIT is returned.
 */
static int
eval(struct regxml *reg, struct regxml_task *task)
{
	struct regxml_elem *current, *el;
	struct regxml_node *q;
	int r;

	current = task->rxt_current;
	q = task->rxt_node;

	switch (q->rxn_type) {
	case REGXML_CONSTANT:
		switch (q->rxn_value.rxv_unit._type) {
		case REGXML_NUMBER:
			if (current->rxe_pos == q->rxn_value.rxv_number._data)
				current->rxe_memory[q->rxn_id] = current;
			else
				current->rxe_memory[q->rxn_id] = REGXML_FAIL;
			break;
		case REGXML_STRING:
			if (current->rxe_type == REGXML_NODE &&
			    strcmp(current->rxe_name,
				q->rxn_value.rxv_string._data) == 0)
				current->rxe_memory[q->rxn_id] = current;
			else
				current->rxe_memory[q->rxn_id] = REGXML_FAIL;
			break;
		default:
			/* XXX: Do something meaningful? */
			return REGXML_ENOIMPL;
		}
		break;

	case REGXML_SUBROOT:
		if (current->rxe_parent->rxe_type == REGXML_ROOT)
			current->rxe_memory[q->rxn_id] = current;
		else
			current->rxe_memory[q->rxn_id] = REGXML_FAIL;
		break;

	case REGXML_RELPOS1:
		if (current->rxe_pos == 1)
			current->rxe_memory[q->rxn_id] = current;
		else
			current->rxe_memory[q->rxn_id] = REGXML_FAIL;
		break;

	case REGXML_RELPOSN:
		if (current->rxe_parent->rxe_cpending)
			return REGXML_EAGAIN;
		if (current->rxe_pos == current->rxe_parent->rxe_nchild)
			current->rxe_memory[q->rxn_id] = current;
		else
			current->rxe_memory[q->rxn_id] = REGXML_FAIL;
		break;

	case REGXML_CAPTURE:
		switch (task->rxt_state) {
		case 0:
			r = spawntask(reg, current, q->rxn_left, task);
			if (r != 0)
				return r == REGXML_EAGAIN ? REGXML_EWAIT : r;
			/* FALLTHROUGH */
		case 1:
			current->rxe_memory[q->rxn_id] =
			    current->rxe_memory[q->rxn_left->rxn_id];
			if (current->rxe_memory[q->rxn_id] != REGXML_FAIL) {
				if (reg->rx_match != NULL) {
					reg->rx_match[2*q->rxn_mid] =
					    current;
					reg->rx_match[2*q->rxn_mid+1] =
					    current->rxe_memory[q->rxn_id];
				}
				reg->rx_smatch = 1;
			}
			break;
		default:
			/* NOTREACHED */
			abort();
		}
		break;

	case REGXML_ANY:
		current->rxe_memory[q->rxn_id] = current;
		break;

	case REGXML_CHILD0:
		switch (task->rxt_state) {
		case 0:
			if (current->rxe_child0 == NULL &&
			    current->rxe_cpending)
				return REGXML_EAGAIN;
			el = current->rxe_child0;
			if (el == NULL) {
				current->rxe_memory[q->rxn_id] = REGXML_FAIL;
				break;
			}
			r = spawntask(reg, el, q->rxn_left, task);
			if (r != 0)
				return r == REGXML_EAGAIN ? REGXML_EWAIT : r;
			/* FALLTHROUGH */
		case 1:
			el = current->rxe_child0;
			if (el->rxe_memory[q->rxn_left->rxn_id] ==
			    REGXML_FAIL)
				current->rxe_memory[q->rxn_id] = REGXML_FAIL;
			else
				current->rxe_memory[q->rxn_id] = current;
			break;
		default:
			/* NOTREACHED */
			abort();
		}
		break;

	case REGXML_NEXT:
		switch (task->rxt_state) {
		case 0:
			if (current->rxe_npending)
				return REGXML_EAGAIN;
			el = current->rxe_next;
			if (el == NULL) {
				current->rxe_memory[q->rxn_id] = REGXML_FAIL;
				break;
			}
			r = spawntask(reg, el, q->rxn_left, task);
			if (r != 0)
				return r == REGXML_EAGAIN ? REGXML_EWAIT : r;
			/* FALLTHROUGH */
		case 1:
			el = current->rxe_next;
			current->rxe_memory[q->rxn_id] =
			    el->rxe_memory[q->rxn_left->rxn_id];
			break;
		default:
			/* NOTREACHED */
			abort();
		}
		break;

	case REGXML_SELFL:
		switch (task->rxt_state) {
		case 0:
			r = spawntask(reg, current, q->rxn_left, task);
			if (r != 0)
				return r == REGXML_EAGAIN ? REGXML_EWAIT : r;
			/* FALLTHROUGH */
		case 1:
			current->rxe_memory[q->rxn_id] =
			    current->rxe_memory[q->rxn_left->rxn_id];
			break;
		default:
			/* NOTREACHED */
			abort();
		}
		break;

	case REGXML_SELFR:
		switch (task->rxt_state) {
		case 0:
			r = spawntask(reg, current, q->rxn_right, task);
			if (r != 0)
				return r == REGXML_EAGAIN ? REGXML_EWAIT : r;
			/* FALLTHROUGH */
		case 1:
			current->rxe_memory[q->rxn_id] =
			    current->rxe_memory[q->rxn_right->rxn_id];
			break;
		default:
			/* NOTREACHED */
			abort();
		}
		break;

	case REGXML_THEN:
		switch (task->rxt_state) {
		case 0:
			r = spawntask(reg, current, q->rxn_left, task);
			if (r != 0)
				return r == REGXML_EAGAIN ? REGXML_EWAIT : r;
			/* FALLTHROUGH */
		case 1:
			if (current->rxe_memory[q->rxn_left->rxn_id] ==
			    REGXML_FAIL) {
				current->rxe_memory[q->rxn_id] = REGXML_FAIL;
				break;
			}
			r = spawntask(reg, current, q->rxn_right, task);
			if (r != 0)
				return r == REGXML_EAGAIN ? REGXML_EWAIT : r;
			/* FALLTHROUGH */
		case 2:
			current->rxe_memory[q->rxn_id] =
			    current->rxe_memory[q->rxn_right->rxn_id];
			break;
		default:
			/* NOTREACHED */
			abort();
		}
		break;

	case REGXML_ELSE:
		switch (task->rxt_state) {
		case 0:
			r = spawntask(reg, current, q->rxn_left, task);
			if (r != 0)
				return r == REGXML_EAGAIN ? REGXML_EWAIT : r;
			/* FALLTHROUGH */
		case 1:
			if (current->rxe_memory[q->rxn_left->rxn_id] !=
			    REGXML_FAIL) {
				current->rxe_memory[q->rxn_id] =
				    current->rxe_memory[q->rxn_left->rxn_id];
				break;
			}
			r = spawntask(reg, current, q->rxn_right, task);
			if (r != 0)
				return r == REGXML_EAGAIN ? REGXML_EWAIT : r;
			/* FALLTHROUGH */
		case 2:
			current->rxe_memory[q->rxn_id] =
			    current->rxe_memory[q->rxn_right->rxn_id];
			break;
		default:
			/* NOTREACHED */
			abort();
		}
		break;

	case REGXML_OR:
		switch (task->rxt_state) {
		case 0:
			r = spawntask(reg, current, q->rxn_left, task);
			if (r != 0 && r != REGXML_EAGAIN)
				return r;
			/* FALLTHROUGH */
		case 1:
			r = spawntask(reg, current, q->rxn_right, task);
			if (r != 0 && r != REGXML_EAGAIN)
				return r;
			/* FALLTHROUGH */
		case 2:
			current->rxe_memory[q->rxn_id] =
			    maxelem(current->rxe_memory[q->rxn_left->rxn_id],
				current->rxe_memory[q->rxn_right->rxn_id]);
			break;
		default:
			/* NOTREACHED */
			abort();
		}
		break;

	default:
		r = evalexpr(reg, current, q);
		if (r != 0)
			return r;
	}

	returntask(reg, task);
	return 0;
}

/*
 * Evaluate a pattern node against an element in prefix mode. In
 * prefix mode, the matcher only evaluate (by calling eval())
 * a minimal number of pattern nodes; evaluation of pattern nodes that
 * do not need information from the current element is deferred. This
 * function updates the rxe_state, rxe_cstate0 and rxe_nstate bit
 * vectors.
 *
 * Terminal pattern nodes, i.e. nodes that are at the end of
 * a navigational chain, trigger the rx_smatch flag if they
 * match. A terminal node is more formally defined in this context as
 * any node that is not deferred, i.e. any node whose evaluation
 * terminates within the peval() cycle dealing with the current node.
 */
static int
peval(struct regxml *reg, struct regxml_node *q)
{
	struct regxml_elem *current, *result;
	int r;

	current = reg->rx_start;
	regxml_bit_clear(current->rxe_state, q->rxn_id);

	switch (q->rxn_type) {
	case REGXML_CHILD0:
		regxml_bit_set(current->rxe_cstate0, q->rxn_left->rxn_id);
		break;

	case REGXML_NEXT:
		regxml_bit_set(current->rxe_nstate, q->rxn_left->rxn_id);
		break;

	case REGXML_SELFL:
		r = peval(reg, q->rxn_left);
		if (r != 0)
			return r;
		break;

	case REGXML_SELFR:
		r = peval(reg, q->rxn_right);
		if (r != 0)
			return r;
		break;

	case REGXML_THEN:
		/*
		 * This node will need to be reevaluated if processing
		 * is halted.
		 */
		regxml_bit_set(current->rxe_state, q->rxn_id);
		r = spawntask(reg, current, q->rxn_left, NULL);
		if (r != 0)
			return r;
		if (current->rxe_memory[q->rxn_left->rxn_id] == REGXML_FAIL) {
			regxml_bit_clear(current->rxe_state, q->rxn_id);
			break;
		}
		r = peval(reg, q->rxn_right);
		if (r != 0)
			return r;
		result = current->rxe_memory[q->rxn_right->rxn_id];
		if (result != NULL && result != REGXML_FAIL)
			reg->rx_smatch = 1;
		regxml_bit_clear(current->rxe_state, q->rxn_id);
		break;

	case REGXML_ELSE:
		/*
		 * This node will need to be reevaluated if processing
		 * is halted.
		 */
		regxml_bit_set(current->rxe_state, q->rxn_id);
		r = spawntask(reg, current, q->rxn_left, NULL);
		if (r != 0)
			return r;
		if (current->rxe_memory[q->rxn_left->rxn_id] != REGXML_FAIL) {
			reg->rx_smatch = 1;
			regxml_bit_clear(current->rxe_state, q->rxn_id);
			break;
		}
		r = peval(reg, q->rxn_right);
		if (r != 0)
			return r;
		result = current->rxe_memory[q->rxn_right->rxn_id];
		if (result != NULL && result != REGXML_FAIL)
			reg->rx_smatch = 1;
		regxml_bit_clear(current->rxe_state, q->rxn_id);
		break;

	case REGXML_OR:
		regxml_bit_set(current->rxe_state, q->rxn_left->rxn_id);
		regxml_bit_set(current->rxe_state, q->rxn_right->rxn_id);
		r = peval(reg, q->rxn_left);
		if (r != 0)
			return r;
		r = peval(reg, q->rxn_right);
		if (r != 0)
			return r;
		break;

	default:
		/*
		 * This node will need to be reevaluated if processing
		 * is halted.
		 */
		regxml_bit_set(current->rxe_state, q->rxn_id);
		r = spawntask(reg, current, q, NULL);
		if (r != 0)
			return r;
		if (current->rxe_memory[q->rxn_id] != REGXML_FAIL)
			reg->rx_smatch = 1;
		regxml_bit_clear(current->rxe_state, q->rxn_id);
	}

	return 0;
}

static int
scopeeq(const char *s, const char *p)
{
	while (*p == *s && *p != '\0')
		++p, ++s;
	return *p == '\0' && (*s == '\0' || isspace((unsigned char)*s));
}

/*
 * Test and enforce scope. Return 0 if the current element is in
 * scope, REGXML_NOMATCH if not, or REGXML_EAGAIN if more data is
 * needed.
 */
static int
scope(struct regxml *reg)
{
	if (reg->rx_start->rxe_type != REGXML_PI ||
	    strcmp(reg->rx_start->rxe_name, "regxml-scope") != 0)
		return reg->rx_inscope ? 0 : REGXML_NOMATCH;
	if (!reg->rx_inscope) {
		if (reg->rx_scope == NULL ||
		    !scopeeq(reg->rx_start->rxe_value, reg->rx_scope))
			return REGXML_NOMATCH;
		else {
			reg->rx_inscope = 1;
			return REGXML_EAGAIN;
		}
	} else {
		if (reg->rx_scope == NULL ||
		    !scopeeq(reg->rx_start->rxe_value, reg->rx_scope))
			return 0;
		else {
			reg->rx_inscope = 0;
			return REGXML_EAGAIN;
		}
	}
}

/*
 * Find next element to explore when ascending the tree. The next
 * element is first next sibling of the current element or any of its
 * ancestors, in order, starting with itself.
 */
static int
stepelem(struct regxml_elem *el, regxml_xid_t nextid,
    struct regxml_elem **ptr, int dofetch)
{
	struct regxml_elem *tmp;

	for (;;) {
		if (dofetch && el->rxe_highid + 1 != nextid) {
			*ptr = el;
			return REGXML_CLOSING;
		}
		if (el->rxe_npending)
			return REGXML_EAGAIN;
		tmp = el->rxe_next;
		if (tmp != NULL) {
			el = tmp;
			break;
		}
		el = el->rxe_parent;
		if (el->rxe_type == REGXML_ROOT) {
			*ptr = el;
			return REGXML_EOF;
		}
	}
	*ptr = el;
	return 0;
}

/*
 * Find next element to explore when descending the tree.
 */
static int
stepin(struct regxml_elem **ptr, regxml_xid_t *nextidptr, int dofetch)
{
	int r;

	if (*nextidptr == (*ptr)->rxe_lowid + 1) {
		if ((*ptr)->rxe_child0 == NULL) {
			if ((*ptr)->rxe_cpending)
				return REGXML_EAGAIN;
			if (dofetch) {
				/* Next element is next sibling. */
				*nextidptr = (*ptr)->rxe_highid + 1;
				return REGXML_CLOSING;
			} else
				goto next;
		}
		*ptr = (*ptr)->rxe_child0;
	} else {
	next:
		r = stepelem(*ptr, *nextidptr, ptr, dofetch);
		if (r != 0) {
			/* Next element is next sibling. */
			*nextidptr = (*ptr)->rxe_highid + 1;
			return r;
		}
	}

	*nextidptr = (*ptr)->rxe_lowid;
	return 0;
}

/*
 * Advance rx_start pointer according to rx_startid. If a preceding
 * pass stopped due to REGXML_EWAIT or REGXML_EAGAIN, rx_start is not
 * moved.
 */
static int
step(struct regxml *reg)
{
	int r;

re:
	if (reg->rx_start->rxe_lowid != reg->rx_startid) {
		r = stepin(&reg->rx_start, &reg->rx_startid,
		    reg->rx_dofetch || reg->rx_matchstep);
		if (r != 0)
			return r;

		switch (scope(reg)) {
		case 0:
			if (reg->rx_start->rxe_prev == NULL) {
				swapbitstr(&reg->rx_start->rxe_state,
				    &reg->rx_start->rxe_parent->rxe_cstate0);
			} else {
				swapbitstr(&reg->rx_start->rxe_state,
				    &reg->rx_start->rxe_prev->rxe_nstate);
			}

			regxml_bit_set(reg->rx_start->rxe_state,
			    reg->rx_pattern->rxn_id);
			break;

		case REGXML_NOMATCH:
			break;

		case REGXML_EAGAIN:
			/* Skip this element; examine first child. */
			reg->rx_startid = reg->rx_start->rxe_lowid + 1;
			goto re;

		default:
			/* NOTREACHED */
			abort();
		}

		prune(reg);
	}

	return 0;
}

/*
 * Compute and store longest match in reg->rx_match[0] from the
 * contents of reg->rx_match.
 */
static void
complongest(struct regxml *reg)
{
	regxml_id_t id, maxid;

	reg->rx_match[0] = reg->rx_start;
	reg->rx_match[1] = reg->rx_start;
	maxid = 1;
	for (id = 3; id < reg->rx_msize; id += 2) {
		if (reg->rx_match[id] != NULL &&
		    reg->rx_match[id-1] == reg->rx_start &&
		    (reg->rx_match[1]->rxe_lowid <=
			reg->rx_match[id]->rxe_lowid)) {
			reg->rx_match[1] = reg->rx_match[id];
			maxid = id;
		}
	}
	reg->rx_maxmatch = maxid / 2;
}

/*
 * Evaluate candidate pattern nodes (i.e. those in rxe_state) against
 * the current element, computing matches and new pattern states for
 * adjacent elements (i.e. rxe_cstate0 and rxe_nstate).
 */
static int
walk(struct regxml *reg)
{
	regxml_id_t id;
	int i, r;

	/* Fetch new data if idle. */
	if (SIMPLEQ_EMPTY(&reg->rx_tasks)) {
		if (reg->rx_match != NULL) {
			for (id = 0; id < reg->rx_msize; ++id)
				reg->rx_match[id] = NULL;
		}
		reg->rx_smatch = 0;

		r = step(reg);
		if (r != 0)
			goto fail;
	}

	/* Loop on tasks generated by successive calls to peval(). */
	do {
		/* Execute tasks. */
		r = flushtasks(reg);
		if (r != 0)
			goto fail;

		/* Generate new tasks. */
		REGXML_BIT_FOREACH (i, reg->rx_start->rxe_state,
		    reg->rx_psize) {
			r = peval(reg, reg->rx_pvect[i]);
			if (r != 0 && r != REGXML_EWAIT)
				goto fail;
		} REGXML_BIT_ENDFOREACH;
	} while (!SIMPLEQ_EMPTY(&reg->rx_tasks));

	/* Next element to explore is first child. */
	reg->rx_startid = reg->rx_start->rxe_lowid + 1;

	/* Compute longest match if needed. */
	if (reg->rx_match != NULL && reg->rx_longest)
		complongest(reg);

	/* Return event. */
	if (reg->rx_dofetch)
		return REGXML_OPENING;
	else
		return reg->rx_smatch ? 0 : REGXML_NOMATCH;

fail:
	/*
	 * The current implementation is not parallel, hence the only
	 * possible cause for a REGXML_EWAIT return is input. So, we
	 * return REGXML_EAGAIN instead.
	 */
	return r == REGXML_EWAIT ? REGXML_EAGAIN : r;
}

int
regxml_match(struct regxml *reg, struct regxml_elem **match)
{
	_DIAGASSERT(reg != NULL);

	reg->rx_match = match;
	reg->rx_dofetch = 0;
	return walk(reg);
}

int
regxml_fetch(struct regxml *reg)
{
	_DIAGASSERT(reg != NULL);

	reg->rx_match = NULL;
	reg->rx_dofetch = 1;
	return walk(reg);
}

void
regxml_reset(struct regxml *reg)
{
	struct regxml_elem *el, *tmp;

	_DIAGASSERT(reg != NULL);

	el = reg->rx_tree->rxe_child0;
	while (el != NULL) {
		tmp = el;
		el = el->rxe_next;
		tmp->rxe_parent = NULL;
		decref(reg, tmp);
	}
	reg->rx_tree->rxe_child0 = NULL;

	reg->rx_elemid = 0;
	reg->rx_tree->rxe_lowid = reg->rx_elemid++;
	reg->rx_pointer = reg->rx_tree;
	reg->rx_start = reg->rx_tree;
	reg->rx_startid = 1;
}

void
regxml_hold(struct regxml_elem *start, struct regxml_elem *end)
{
	struct regxml_elem *el;

	_DIAGASSERT(start != NULL);
	_DIAGASSERT(end != NULL);

	for (el = start;; el = el->rxe_next) {
		++el->rxe_refcount;
		if (el == end)
			break;
	}
}

void
regxml_release(struct regxml *reg,
    struct regxml_elem *start, struct regxml_elem *end)
{
	struct regxml_elem *el;

	_DIAGASSERT(start != NULL);
	_DIAGASSERT(end != NULL);

	for (el = start;; el = el->rxe_next) {
		decref(reg, el);
		if (el == end)
			break;
	}
}

void
regxml_inititer(struct regxml_iter *iter,
    struct regxml_elem *start, struct regxml_elem *end)
{
	_DIAGASSERT(iter != NULL);
	_DIAGASSERT(start != NULL);

	iter->rxi_ptr = start;
	iter->rxi_end = end;
	iter->rxi_nextid = start->rxe_lowid;
}

int
regxml_fetchiter(struct regxml_iter *iter)
{
	int r;

	_DIAGASSERT(iter != NULL);

	if (iter->rxi_end != NULL &&
	    iter->rxi_nextid == iter->rxi_end->rxe_highid + 1) {
		iter->rxi_ptr = NULL;
		return REGXML_EOF;
	}

	if (iter->rxi_ptr->rxe_lowid != iter->rxi_nextid) {
		r = stepin(&iter->rxi_ptr, &iter->rxi_nextid, 1);
		if (r != 0)
			return r;
	}

	/* Next element to explore is first child. */
	iter->rxi_nextid = iter->rxi_ptr->rxe_lowid + 1;

	return REGXML_OPENING;
}
