/*
 * Copyright (c) 2004, 2005, id Quantique SA, Switzerland
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * Neither the name of id Quantique nor the names of its contributors may be
 * used to endorse or promote products derived from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>

#include "quantisio.h"
#include "quantis.h"

#define QUANTIS_BASE_NAME "/dev/qrandom/"

/* open the device file of the quantis pci card cardNumber. returns
 * the filedescriptor or the error value.
 */
static int quantisOpen(int cardNumber) {
  char fileName[sizeof(QUANTIS_BASE_NAME)+3];

  sprintf(fileName, QUANTIS_BASE_NAME"%d", cardNumber);
  return open(fileName, O_RDONLY);
}

/* call an ioctl on a quantis pci card device file. returns -1 on
 * error, 0 on success.
 */
static int quantisIoctl(int cardNumber, int request, void *arg) {
  int fd;
  int status;

  fd = quantisOpen(cardNumber);
  if (fd < 0)
    return -1;
  status = ioctl(fd, request, arg);
  close(fd);
  if (status < 0)
    return -1;
  else
    return 0;
}

int quantisCount(void) {
  int cardCount;

  /* When the device file can not be opened, we assume we have no PCI
   * cards installed.
   */
  if (quantisIoctl(0, QUANTIS_IOCTL_GET_CARD_COUNT, &cardCount) == 0)
    return cardCount;
  else
    return 0;
}

int quantisGetModules(int cardNumber) {
  int moduleMask;

  if (quantisIoctl(cardNumber, QUANTIS_IOCTL_GET_MODULES_MASK, &moduleMask) == 0)
    return moduleMask;
  else
    return 0;
}

int quantisBoardReset(int cardNumber) {
  return quantisIoctl(cardNumber, QUANTIS_IOCTL_RESET_BOARD, NULL);
}

int quantisModulesEnable(int cardNumber, int moduleMask) {
  int params[] = { moduleMask }; 
  return quantisIoctl(cardNumber, QUANTIS_IOCTL_ENABLE_MODULE, &params);
}

int quantisModulesDisable(int cardNumber, int moduleMask) {
  int params[] = { moduleMask };
  return quantisIoctl(cardNumber, QUANTIS_IOCTL_DISABLE_MODULE, &params);
}

int quantisModulesReset(int cardNumber, int moduleMask) {
  quantisModulesDisable(cardNumber, moduleMask);
  return quantisModulesEnable(cardNumber, moduleMask);
}

int quantisBoardVersion(int cardNumber) {
  int boardVersion;
  if (quantisIoctl(cardNumber, QUANTIS_IOCTL_GET_BOARD_VERSION, &boardVersion) == 0)
    return boardVersion;
  else
    return 0;
}

int quantisLibVersion(void) {
  return QUANTIS_LIB_VERSION;
}

int quantisDriverVersion(void) {
  int driverVersion;
  
  /* we try to open the first card to get the driver version. If the
   * card is not present, we return 0 to signal that there is no card,
   * as 0 is an invalid version number.
   */
  if (quantisIoctl(0, QUANTIS_IOCTL_GET_DRIVER_VERSION, &driverVersion) == 0)
    return driverVersion;
  else
    return 0;
}

int quantisRead(int cardNumber, void *buffer, unsigned int size) {
  int fd;
  int status;
  int total;

  fd = quantisOpen(cardNumber);
  if (fd < 0)
    return -1;

  /* the linux driver always reads as many bytes as we tell him, but
   * the freebsd driver writes at most as many bytes as he holds in the
   * device buffer, so we may have to call read multiples times.
   */
  for (total = 0; total < size; ) {
    status = read(fd, (unsigned char *)buffer + total,size - total);
    if (status < 0) {
      /* the read call may have been interrupted, try again */
      if (errno == EINTR)
	continue;
      close(fd);
      return -1;
    }
    total += status;
  }
  close(fd);
  
  return total;
}

int quantisSetTestMode(int cardNumber, int enable) {
  int params[] = { enable };
  return quantisIoctl(cardNumber, QUANTIS_IOCTL_SET_TEST_MODE, &params);
}

int quantisModulesStatus(int cardNumber) {
  int modulesStatus;
  if (quantisIoctl(cardNumber, QUANTIS_IOCTL_GET_MODULES_STATUS, &modulesStatus) == 0)
    return modulesStatus;
  else
    return -1;
}
