/* $Id: bpg.h,v 1.5 2008/09/11 20:58:59 agcrooks Exp $ */

/*
 * Copyright (c) 2005 Manuel Freire.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _BPG_H_
#define _BPG_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <sys/types.h>

#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>

typedef struct BPG_CTX {
	int ascii;
	uint8_t cipher;
	uint8_t hash_algo;
	uint8_t pubkey_algo;
	uint8_t keyv;
	uint8_t sigv;
	uint8_t pktv;
	char *pass;
	char *pub_uid;		/* user ID for public keys */
	char *sec_uid;		/* user ID for secret keys */
	char pub_kid[8];	/* key ID for public keys */
	char sec_kid[8];	/* key ID for secret keys */
	char *pub_path;		/* public keys directory (patch) */
	char *sec_path;		/* secret keys directory (patch) */
} BPG_CTX;

typedef struct BPG_BUF {
	unsigned char *body;
	size_t len;
	int pos;		/* current position, used in pop functions */
} BPG_BUF;

int BPG_sign(char *, char *, BPG_CTX *);
int BPG_sign_detached(char *, char *, BPG_CTX *);
int BPG_sign_k(char *, const char *, char *, BPG_CTX *);
int BPG_sign_detached_k(char *, const char *, char *, BPG_CTX *);
int BPG_do_sign(BPG_BUF *, BPG_BUF *, BPG_BUF *, BPG_CTX *);

int BPG_verify(char *, BPG_CTX *);
int BPG_verify_detached(char *, char *, BPG_CTX *);
int BPG_verify_k(char *, const char *, BPG_CTX *);
int BPG_verify_detached_k(char *, char *, const char *, BPG_CTX *);
int BPG_do_verify(BPG_BUF *, BPG_BUF *, BPG_BUF *, BPG_CTX *);
		  
int BPG_encrypt(char *, char *, BPG_CTX *);
int BPG_encrypt_k(char *, char *, char *, BPG_CTX *);
int BPG_do_encrypt(BPG_BUF *, BPG_BUF *, BPG_BUF *, BPG_CTX *);

int BPG_symmenc(char *, char *, BPG_CTX *);
int BPG_do_symmenc(BPG_BUF *, BPG_BUF *, BPG_CTX *);

int BPG_decrypt(char *, char *, BPG_CTX *);
int BPG_decrypt_k(char *, char *, char *, BPG_CTX *);
int BPG_do_decrypt(BPG_BUF *, BPG_BUF *, BPG_BUF *, BPG_CTX *);

int BPG_encrypt_sign(char *, char *, BPG_CTX *);
int BPG_encrypt_sign_k(char *, char *, char *, char *, BPG_CTX *);
int BPG_do_encrypt_sign(BPG_BUF *, BPG_BUF *, BPG_BUF *, BPG_BUF *, BPG_CTX *);

BPG_CTX * BPG_CTX_new(void);
void BPG_set_id(BPG_CTX *, const char *, const char *);
void BPG_CTX_free(BPG_CTX *);

BPG_BUF * BPG_BUF_new(size_t);
int BPG_BUF_resize(BPG_BUF *, size_t);
void BPG_BUF_free(BPG_BUF *);
void BPG_BUF_print(BPG_BUF *, FILE *);
int BPG_BUF_read(BPG_BUF *, unsigned char *, size_t);
int BPG_BUF_read_f(BPG_BUF *, FILE *);

char *BPG_get_error_string(void);

/* higher-level routines start here */

/* this structure is used to drive the helper functions */
typedef struct bpginfo_t {
	int	 detached;	/* a detached signature file? */
	int	 verbose;	/* get lippy */
	char	*pub_file;	/* public key file */
	char	*sec_file;	/* private key file */
	char	*output_file;	/* output file */
	char	*det_sig_file;	/* detached signature file */
	BPG_CTX	*opt;		/* BSD Priv Guard context */
} bpginfo_t;

int bpg_get_defaults(bpginfo_t *, char *);

int bpg_sign(bpginfo_t *, int, char **);
int bpg_verify(bpginfo_t *, int, char **);
int bpg_encrypt(bpginfo_t *, int, char **);
int bpg_symmenc(bpginfo_t *, int, char **);
int bpg_decrypt(bpginfo_t *, int, char **);
int bpg_encsig(bpginfo_t *, int, char **);

#ifdef __cplusplus
}
#endif

#endif /* !_BPG_H_ */

