/* $Id: bpg.c,v 1.13 2008/09/11 20:58:59 agcrooks Exp $ */

/*
 * Copyright (c) 2005 Manuel Freire.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <sys/cdefs.h>

#ifndef lint
__COPYRIGHT("@(#) Copyright (c) 2005 Manuel Freire. All rights reserved.");
__RCSID("$Id: bpg.c,v 1.13 2008/09/11 20:58:59 agcrooks Exp $");
#endif

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/param.h>

#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <openssl/dsa.h>
#include <openssl/engine.h>
#include <openssl/err.h>
#include <openssl/evp.h>
#include <openssl/rand.h>
#include <openssl/rsa.h>

#include <paths.h>

#include "constants.h"
#include "data.h"
#include "error.h"
#include "packet.h"
#include "bpgutil.h"

#include "bpg.h"

const char *BPG_version = PACKAGE_STRING;

#define DEFAULT_ID	"00000000"

/* seed the entropy pool with random data */
static void
initial_entropy(void)
{
#if PREDICTABLE
	/* we may want predictable data for problem recreation */
	RAND_seed("SDJA0O103UJ0WAE98RUA093UR02Q3", 20);
#else
	uint32_t	r;
	uint8_t		buf[BUFSIZ];
	int		i;

	for (i = 0 ; i < sizeof(buf) ; i += sizeof(r)) {
		r = random();
		(void) memcpy(&buf[i], &r, sizeof(r));
	}
	RAND_seed(buf, i);
#endif
}

/* Sign file `msg_p' into `dst_p'. */
int
BPG_sign(char *msg_p, char *dst_p, BPG_CTX *opt)
{
	/*
	 * FIXME: this function is a patch for using gpg in key extraction.
	 * When the bpg-key library is available it will be used instead.
 	 */
	BPG_CTX *ctx;
	char	tmp[MAXPATHLEN];
	int	tmpfd;
	int	status;
	int	ret;
	int	fd;
	
	ret = 0;

	warnx("GnuPG is used for key extraction.");
	ctx = (opt) ? opt : BPG_CTX_new();
	(void) strlcpy(tmp, "/tmp/bpg-seckey.tmp.XXXXXX", sizeof(tmp));
	tmpfd = mkstemp(tmp);
	(void) close(tmpfd);
	fd = fork();
	switch (fd) {
	case 0:
		execlp("gpg", "gpg", "-o", tmp,
		       "--export-secret-key", opt->sec_uid, NULL);
		exit(errno);
		/*FALLTHROUGH*/
	case -1:
		return -1;
	default:
		wait(&status);
		if (status) {
			ret = -1;
		}
	}
	
	if (ret == 0) {
		ret = BPG_sign_k(msg_p, tmp, dst_p, opt);
	}

	if (opt == NULL) {
		BPG_CTX_free(ctx);
	}
	(void) unlink(tmp);

	return ret;
}

/* Create a detached signature of file `msg_p' in `dst_p'. */
int
BPG_sign_detached(char *msg_p, char *dst_p, BPG_CTX *opt)
{
	/*
	 * FIXME: this function is a patch for using gpg in key extraction.
	 * When the bpg-key library is available it will be used instead.
 	 */
	BPG_CTX *ctx;
	char	tmp[MAXPATHLEN];
	int	tmpfd;
	int	status;
	int	ret;
	int	fd;
	
	ret = 0; 

	warnx("GnuPG is used for key extraction.");
	ctx = (opt) ? opt : BPG_CTX_new();

	(void) strlcpy(tmp, "/tmp/bpg-seckey.tmp.XXXXXX", sizeof(tmp));
	tmpfd = mkstemp(tmp);
	(void) close(tmpfd);

	fd = fork();
	switch (fd) {
	case 0:
		execlp("gpg", "gpg", "-o", tmp,
		       "--export-secret-key", opt->sec_uid, NULL);
		exit(errno);
		/*FALLTHROUGH*/
	case -1:
		return -1;
	default:
		wait(&status);
		if (status) {
			ret = -1;
		}
	}

	if (ret == 0) {
		ret = BPG_sign_detached_k(msg_p, tmp, dst_p, opt);
	}

	if (opt == NULL)
		BPG_CTX_free(ctx);
	(void) unlink(tmp);

	return ret;
}

/* Sign `msg_p' into `dst_p' using secret key in file `skey_p'. */
int
BPG_sign_k(char *msg_p, const char *skey_p, char *dst_p, BPG_CTX *opt)
{
	BPG_BUF	*onepass = NULL;
	BPG_BUF	*literal = NULL;
	BPG_PKT	*pkt_onepass = NULL;
	BPG_PKT	*pkt_literal = NULL;
	BPG_BUF	*msg = NULL;
	BPG_BUF	*skey = NULL;
	BPG_BUF	*dst = NULL;
	FILE	*msgfp = NULL;
	FILE	*seckeyfp = NULL;
	FILE	*destfp = NULL;
	int	ret;
	
	/* assume we're going to fail - if success, set to 0 when needed */
	ret = -1;

	if (msg_p == NULL) {
		msgfp = stdin;
	} else if ((msgfp = fopen(msg_p, "r")) == NULL) {
		return -1;
	}
	if ((seckeyfp = fopen(skey_p, "r")) == NULL) {
		goto close_and_ret;
	}
	if ((msg = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if ((skey = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if ((dst = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if ((onepass = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if ((literal = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if ((pkt_onepass = BPG_PKT_new()) == NULL) {
		goto close_and_ret;
	}
	if ((pkt_literal = BPG_PKT_new()) == NULL) {
		goto close_and_ret;
	}
	if (BPG_BUF_read_f(msg, msgfp) == -1) {
		goto close_and_ret;
	}
	if (BPG_BUF_read_f(skey, seckeyfp) == -1) {
		goto close_and_ret;
	}

	if (BPG_do_sign(msg, skey, dst, opt) == -1) {
		goto close_and_ret;
	}

	if (dst_p == NULL) {
		destfp = stdout;
	} else if ((destfp = fopen(dst_p, "w")) == NULL) {
		goto close_and_ret;
	}

	/* Write signature: One-Pass Signature + Literal Packet + Signature. */
	/* FIXME: next block is a patch I cannot fix now because of time. */
	if (BPG_ONEPASS_build(onepass, opt) == -1) {
		goto close_and_ret;
	}
	if (BPG_BUF_read(pkt_onepass->body, onepass->body, onepass->len) == -1) {
		goto close_and_ret;
	}
	BPG_PKT_build(pkt_onepass, PKTTYPE_ONEPASS, opt);
	BPG_PKT_write(pkt_onepass, onepass);
	if (BPG_LITERAL_build(msg, literal, msg_p) == -1) {
		goto close_and_ret;
	}
	if (BPG_BUF_read(pkt_literal->body, literal->body, literal->len) == -1) {
		goto close_and_ret;
	}
	BPG_PKT_build(pkt_literal, PKTTYPE_LITERAL, opt);
	BPG_PKT_write(pkt_literal, literal);

	if (fwrite(onepass->body, 1, onepass->len, destfp) != onepass->len ||
	    fwrite(literal->body, 1, literal->len, destfp) != literal->len ||
	    fwrite(dst->body, 1, dst->len, destfp) != dst->len) {
		goto close_and_ret;
	}

	ret = 0;

close_and_ret:
	if (msg_p) fclose(msgfp);
	if (seckeyfp) fclose(seckeyfp);
	if (dst_p) fclose(destfp);
	if (msg) BPG_BUF_free(msg);
	if (skey) BPG_BUF_free(skey);
	if (dst) BPG_BUF_free(dst);
	if (onepass) BPG_BUF_free(onepass);
	if (literal) BPG_BUF_free(literal);
	if (pkt_onepass) BPG_PKT_free(pkt_onepass);
	if (pkt_literal) BPG_PKT_free(pkt_literal);

	return ret;
}

/* Create a detached signature of `msg_p' in `dst_p' using key in `skey_p'. */
int
BPG_sign_detached_k(char *msg_p, const char *skey_p, char *dst_p, BPG_CTX *opt)
{
	BPG_BUF *msg = NULL;
	BPG_BUF *skey = NULL;
	BPG_BUF *dst = NULL;
	FILE	*msgfp = NULL;
	FILE	*seckeyfp = NULL;
	FILE	*destfp = NULL;
	int	 ret;

	/* assume we're going to fail - if success, set to 0 when needed */
	ret = -1;

	if (msg_p == NULL) {
		msgfp = stdin;
	} else if ((msgfp = fopen(msg_p, "r")) == NULL) {
		return -1;
	}
	if ((seckeyfp = fopen(skey_p, "r")) == NULL) {
		goto close_and_ret;
	}
	if ((msg = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if ((skey = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if ((dst = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if (BPG_BUF_read_f(msg, msgfp) == -1) {
		goto close_and_ret;
	}
	if (BPG_BUF_read_f(skey, seckeyfp) == -1) {
		goto close_and_ret;
	}

	if (BPG_do_sign(msg, skey, dst, opt) == -1) {
		goto close_and_ret;
	}

	if (dst_p == NULL) {
		destfp = stdout;
	} else if ((destfp = fopen(dst_p, "w")) == NULL) {
		goto close_and_ret;
	}

	if (fwrite(dst->body, 1, dst->len, destfp) != dst->len) {
		goto close_and_ret;
	}

	ret = 0;

close_and_ret:
	if (seckeyfp) fclose(seckeyfp);
	if (msg_p) fclose(msgfp);
	if (dst_p) fclose(destfp);

	if (msg) BPG_BUF_free(msg);
	if (skey) BPG_BUF_free(skey);
	if (dst) BPG_BUF_free(dst);

	return ret;
}

/* Write in `dst' a signature of `msg' using the secret key in `skey'. */
int
BPG_do_sign(BPG_BUF *msg, BPG_BUF *skey, BPG_BUF *dst, BPG_CTX *opt)
{
	const EVP_MD	*htype;
	EVP_MD_CTX	 hctx;
	bpg_time_t	t;
	BPG_SECKEY	*seckey = NULL;
	BPG_CTX		*ctx = NULL;
	BPG_SIG		*sig = NULL;
	BPG_BUF		*to_hash = NULL;
	BPG_BUF		*hash = NULL;
	int		 ret;
	int		 stype;
	
	ret = -1;

	ctx = (opt) ? opt : BPG_CTX_new();
	if ((seckey = BPG_SECKEY_new()) == NULL) {
		goto close_and_ret;
	}
	if ((sig = BPG_SIG_new()) == NULL) {
		goto close_and_ret;
	}
	if ((to_hash = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if ((hash = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}

	if (BPG_SECKEY_read_packet(seckey, skey->body, skey->len, ctx) == -1) {
		BPG_err(BPGERR_COULDNT_PARSE_SECKEY);
		goto close_and_ret;
	}
	stype = SIGTYPE_BINARY_DOC;	/* FIXME: other types? */
	t = bpg_time_get();

	/* Hash */
	switch (ctx->sigv) {
	case SIGVERSION_3:
		BPG_BUF_resize(to_hash, 5);
		to_hash->body[0] = stype;
		util_uint32_to_str(t, &(to_hash->body[1]));
		break;
	case SIGVERSION_4:
		BPG_err(BPGERR_SIGVERSION_4_NOT_SUPPORTED);
		goto close_and_ret;
	default:
		BPG_err(BPGERR_SIGVERSION_UNKNOWN);
		goto close_and_ret;
	}
	switch (ctx->hash_algo) {
	case HA_SHA1:
		BPG_BUF_resize(hash, SHA1_SIZE);
		if (ctx->pubkey_algo == PKA_DSA)
			htype = EVP_dss1();	/* dsa requires dss1 */
		else
			htype = EVP_sha1();
		break;
	default:
		BPG_err(BPGERR_ALGORITHM_NOT_SUPPORTED);
		goto close_and_ret;
	}
	EVP_MD_CTX_init(&hctx);	
	if (EVP_DigestInit_ex(&hctx, htype, NULL) == 0) {
		BPG_err(BPGERR_OPENSSL_DIGEST);
		goto close_and_ret;
	}
	if (EVP_DigestUpdate(&hctx, msg->body, msg->len) == 0) {
		BPG_err(BPGERR_OPENSSL_DIGEST);
		goto close_and_ret;
	}
	if (EVP_DigestUpdate(&hctx, to_hash->body, to_hash->len) == 0) {
		BPG_err(BPGERR_OPENSSL_DIGEST);
		goto close_and_ret;
	}
	if (EVP_DigestFinal_ex(&hctx, hash->body, &(hash->len)) == 0) {
		BPG_err(BPGERR_OPENSSL_DIGEST);
		goto close_and_ret;
	}
	EVP_MD_CTX_cleanup(&hctx);

	/* Sign */
	initial_entropy();
	switch (ctx->pubkey_algo) {
	case PKA_DSA:
		sig->dsa_sig = DSA_do_sign(hash->body, (int)hash->len,
			seckey->dsa);
		if (sig->dsa_sig == NULL) {
			goto close_and_ret;
		}
		break;
	default:
		BPG_err(BPGERR_ALGORITHM_NOT_SUPPORTED);
		goto close_and_ret;
	}
	
	/* Build packet */
	sig->version = ctx->sigv;
	switch (sig->version) {
	case SIGVERSION_3:
		sig->type = stype;
		sig->creation_time = t;
		(void)memcpy(sig->key_id, ctx->sec_kid, 8);
		sig->pubkey_algo = ctx->pubkey_algo;
		sig->hash_algo = ctx->hash_algo;
		(void)memcpy(sig->hash_start, hash->body, 2);
		break;
	case SIGVERSION_4:
		BPG_err(BPGERR_SIGVERSION_4_NOT_SUPPORTED);
		goto close_and_ret;
	default:
		BPG_err(BPGERR_SIGVERSION_UNKNOWN);
		goto close_and_ret;
	}
	BPG_SIG_build_packet(sig, dst, ctx);

	ret = 0;

close_and_ret:
	if (seckey) BPG_SECKEY_free(seckey);
	if (sig) BPG_SIG_free(sig);
	if (opt == NULL) BPG_CTX_free(ctx);
	if (to_hash) BPG_BUF_free(to_hash);
	if (hash) BPG_BUF_free(hash);

	return ret;
}

/* Verify signed file `smsg_p'. */
int
BPG_verify(char *smsg_p, BPG_CTX *opt)
{
	/*
	 * FIXME: this function is a patch for using gpg in key extraction.
	 * When the bpg-key library is available it will be used instead.
 	 */

	int fd, status, ret;
	BPG_CTX *ctx;
	
	warnx("GnuPG is used for key extraction.");
	ctx = (opt) ? opt : BPG_CTX_new();
	fd = fork();
	switch (fd) {
	case 0:
		execlp("gpg", "gpg", "-o", "/tmp/bpg-pubkey.tmp", "--export",
		       opt->pub_uid, NULL);
		exit(errno);
		/*FALLTHROUGH*/
	case -1:
		return -1;
	default:
		wait(&status);
		if (status)
			return -1;
	}

	ret = BPG_verify_k(smsg_p, "/tmp/bpg-pubkey.tmp", opt);

	if (opt == NULL)
		BPG_CTX_free(ctx);
	remove("/tmp/bpg-pubkey.tmp");

	return ret;
}

/* Verify detached signature `sig_p' of file `msg_p'. */
int
BPG_verify_detached(char *msg_p, char *sig_p, BPG_CTX *opt)
{
	/*
	 * FIXME: this function is a patch for using gpg in key extraction.
	 * When the bpg-key library is available it will be used instead.
 	 */

	int fd, status, ret;
	BPG_CTX *ctx;
	
	warnx("GnuPG is used for key extraction.");
	ctx = (opt) ? opt : BPG_CTX_new();
	fd = fork();
	switch (fd) {
	case 0:
		execlp("gpg", "gpg", "-o", "/tmp/bpg-pubkey.tmp", "--export",
		       opt->pub_uid, NULL);
		exit(errno);
		/*FALLTHROUGH*/
	case -1:
		return -1;
	default:
		wait(&status);
		if (status)
			return -1;
	}

	ret = BPG_verify_detached_k(msg_p, sig_p, "/tmp/bpg-pubkey.tmp", opt);

	if (opt == NULL)
		BPG_CTX_free(ctx);
	remove("/tmp/bpg-pubkey.tmp");

	return ret;
}

/* Verify signed file `smsg_p' using the public key in `pkey_p'. */
int
BPG_verify_k(char *smsg_p, const char *pkey_p, BPG_CTX *opt)
{
	BPG_BUF	*smsg = NULL;
	BPG_BUF	*msg = NULL;
	BPG_BUF	*sig = NULL;
	BPG_BUF	*pkey = NULL;
	BPG_PKT	*pkt = NULL;
	FILE	*msgfp = NULL;
	FILE	*fp = NULL;
	int	 nread;
	int	 ret;
	
	ret = -1;

	if (smsg_p == NULL) {
		msgfp = stdin;
	} else if ((msgfp = fopen(smsg_p, "r")) == NULL) {
		return -1;
	}
	if ((fp = fopen(pkey_p, "r")) == NULL) {
		goto close_and_ret;
	}
	if ((smsg = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if ((msg = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if ((sig = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if ((pkey = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if ((pkt = BPG_PKT_new()) == NULL) {
		goto close_and_ret;
	}
	if (BPG_BUF_read_f(smsg, msgfp) == -1) {
		goto close_and_ret;
	}
	if (BPG_BUF_read_f(pkey, fp) == -1) {
		goto close_and_ret;
	}

	/* Read one-pass signature (TODO: don't ignore this packet). */
	if (BPG_PKT_read(pkt, smsg->body, smsg->len) == -1) {
		goto close_and_ret;
	}
	nread = pkt->full_length;

	/* Read literal data. */
	if (BPG_PKT_read(pkt, &smsg->body[nread], smsg->len - nread) == -1) {
		goto close_and_ret;
	}
	nread += pkt->full_length;
	if (BPG_LITERAL_read(msg, pkt->body) == -1) {
		goto close_and_ret;
	}

	/* Read signature. */
	if (BPG_BUF_read(sig, &smsg->body[nread], smsg->len - nread) == -1) {
		goto close_and_ret;
	}

	ret = BPG_do_verify(msg, sig, pkey, opt);

close_and_ret:
	if (msgfp != stdin) fclose(msgfp);
	if (fp) fclose(fp);
	if (smsg) BPG_BUF_free(smsg);
	if (msg) BPG_BUF_free(msg);
	if (sig) BPG_BUF_free(sig);
	if (pkey) BPG_BUF_free(pkey);

	return ret;
}

/* Verify detached signature `sig_p' of file `msg_p' using key in `pkey_p'. */
int
BPG_verify_detached_k(char *msg_p, char *sig_p, const char *pkey_p, BPG_CTX *opt)
{
	BPG_BUF	*msg = NULL;
	BPG_BUF	*sig = NULL;
	BPG_BUF	*pkey = NULL;
	FILE	*msgfp = NULL;
	FILE	*sigfp = NULL;
	FILE	*fp = NULL;
	int	 ret;
	
	ret = -1;

	if (msg_p == NULL) {
		msgfp = stdin;
	} else if ((msgfp = fopen(msg_p, "r")) == NULL) {
		return -1;
	}
	if ((sigfp = fopen(sig_p, "r")) == NULL) {
		goto close_and_ret;
	}
	if ((fp = fopen(pkey_p, "r")) == NULL) {
		goto close_and_ret;
	}
	if ((msg = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if ((sig = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if ((pkey = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if (BPG_BUF_read_f(msg, msgfp) == -1) {
		goto close_and_ret;
	}
	if (BPG_BUF_read_f(sig, sigfp) == -1) {
		goto close_and_ret;
	}
	if (BPG_BUF_read_f(pkey, fp) == -1) {
		goto close_and_ret;
	}

	ret = BPG_do_verify(msg, sig, pkey, opt);

close_and_ret:
	if (msgfp != stdin) fclose(msgfp);
	if (sigfp) fclose(sigfp);
	if (fp) fclose(fp);
	if (msg) BPG_BUF_free(msg);
	if (sig) BPG_BUF_free(sig);
	if (pkey) BPG_BUF_free(pkey);

	return ret;
}

/* Verify signature of buffer `msg' using key in `pkey'.  `opt' is ignored. */
/*ARGSUSED3*/
int
BPG_do_verify(BPG_BUF *msg, BPG_BUF *sig, BPG_BUF *pkey, BPG_CTX *opt)
{
	const EVP_MD	*htype;
	EVP_MD_CTX	 hctx;
	BPG_PUBKEY	*pubkey = NULL;
	BPG_SIG		*signature = NULL;
	BPG_BUF		*to_hash = NULL;
	BPG_BUF		*hash = NULL;
	int		 ret;
	
	ret = -1;

	if ((signature = BPG_SIG_new()) == NULL)
		return -1;
	if ((pubkey = BPG_PUBKEY_new()) == NULL) {
		goto close_and_ret;
	}
	if ((to_hash = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if ((hash = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}

	if (BPG_SIG_read_packet(signature, sig->body, sig->len) == -1) {
		goto close_and_ret;
	}
	if (BPG_PUBKEY_read_packet(pubkey, pkey->body, pkey->len) == -1) {
		goto close_and_ret;
	}

	/* Calculate hash. */
	switch (signature->version) {
	case SIGVERSION_3:
		BPG_BUF_resize(to_hash, 5);
		to_hash->body[0] = signature->type;
		util_uint32_to_str(signature->creation_time,
		    &(to_hash->body[1]));
		break;
	case SIGVERSION_4:
		BPG_err(BPGERR_SIGVERSION_4_NOT_SUPPORTED);
		goto close_and_ret;
	default:
		BPG_err(BPGERR_SIGVERSION_UNKNOWN);
		goto close_and_ret;
	}
	switch (signature->hash_algo) {
	case HA_SHA1:
		BPG_BUF_resize(hash, SHA1_SIZE);
		if (signature->pubkey_algo == PKA_DSA)
			htype = EVP_dss1();	/* dsa requires dss1 */
		else
			htype = EVP_sha1();
		break;
	default:
		BPG_err(BPGERR_ALGORITHM_NOT_SUPPORTED);
		goto close_and_ret;
	}
	EVP_MD_CTX_init(&hctx);	
	if (EVP_DigestInit_ex(&hctx, htype, NULL) == 0) {
		BPG_err(BPGERR_OPENSSL_DIGEST);
		goto close_and_ret;
	}
	if (EVP_DigestUpdate(&hctx, msg->body, msg->len) == 0) {
		BPG_err(BPGERR_OPENSSL_DIGEST);
		goto close_and_ret;
	}
	if (EVP_DigestUpdate(&hctx, to_hash->body, to_hash->len) == 0) {
		BPG_err(BPGERR_OPENSSL_DIGEST);
		goto close_and_ret;
	}
	if (EVP_DigestFinal_ex(&hctx, hash->body, &(hash->len)) == 0) {
		BPG_err(BPGERR_OPENSSL_DIGEST);
		goto close_and_ret;
	}
	EVP_MD_CTX_cleanup(&hctx);

	/* Verification. */
	if (memcmp(hash->body, signature->hash_start, 2) != 0) {
		ret = 0;
		goto close_and_ret;
	}
	switch (signature->pubkey_algo) {
	case PKA_DSA:
		ret = DSA_do_verify(hash->body, (int)hash->len,
			signature->dsa_sig, pubkey->dsa);
			/* TODO: use openssl_evp(3) */
		break;
	default:
		BPG_err(BPGERR_ALGORITHM_NOT_SUPPORTED);
		/* make it explicit what's being returned */
		ret = -1;
		goto close_and_ret;
	}

close_and_ret:
	if (signature) BPG_SIG_free(signature);
	if (pubkey) BPG_PUBKEY_free(pubkey);
	if (to_hash) BPG_BUF_free(to_hash);
	if (hash) BPG_BUF_free(hash);

	return ret;
}

/* Encrypt file `msg_p' into `dst_p'. */
/* ARGSUSED0 */
int
BPG_encrypt(char *msg_p, char *dst_p, BPG_CTX *opt)
{

	BPG_err(BPGERR_NOT_IMPLEMENTED);
	return -1;
}

/* Encrypt `msg_p' into `dst_p' using the key in `pkey_p'. */
/* ARGSUSED0 */
int
BPG_encrypt_k(char *msg_p, char *pkey_p, char *dst_p, BPG_CTX *opt)
{

	BPG_err(BPGERR_NOT_IMPLEMENTED);
	return -1;
}

/* Encrypt buffer `msg' into `dst' using the key in `pkey'. */
/* ARGSUSED0 */
int
BPG_do_encrypt(BPG_BUF *msg, BPG_BUF *pkey, BPG_BUF *dst, BPG_CTX *opt)
{

	BPG_err(BPGERR_NOT_IMPLEMENTED);
	return -1;
}

/* Symmetrically encrypt file `msg_p' into `dst_p'. */
/* ARGSUSED0 */
int
BPG_symmenc(char *msg_p, char *dst_p, BPG_CTX *opt)
{

	BPG_err(BPGERR_NOT_IMPLEMENTED);
	return -1;
}

/* Symmetrically encrypt buffer `msg' into `dst'. */
/* ARGSUSED0 */
int
BPG_do_symmenc(BPG_BUF *msg, BPG_BUF *dst, BPG_CTX *opt)
{

	BPG_err(BPGERR_NOT_IMPLEMENTED);
	return -1;
}

/* Decrypt file `msg_p' into `dst_p'. */
/* ARGSUSED0 */
int
BPG_decrypt(char *msg_p, char *dst_p, BPG_CTX *opt)
{

	BPG_err(BPGERR_NOT_IMPLEMENTED);
	return -1;
}

/* Decrypt file `msg_p' into `dst_p' using the key in `skey_p'. */
/* ARGSUSED0 */
int
BPG_decrypt_k(char *msg_p, char *skey_p, char *dst_p, BPG_CTX *opt)
{

	BPG_err(BPGERR_NOT_IMPLEMENTED);
	return -1;
}

/* Decrypt buffer `msg' into `dst' using the key in `skey'. */
/* ARGSUSED0 */
int
BPG_do_decrypt(BPG_BUF *msg, BPG_BUF *skey, BPG_BUF *dst, BPG_CTX *opt)
{

	BPG_err(BPGERR_NOT_IMPLEMENTED);
	return -1;
}

/* Encrypt and sign file `msg_p' into `dst_p'. */
/* ARGSUSED0 */
int
BPG_encrypt_sign(char *msg_p, char *dst_p, BPG_CTX *opt)
{

	BPG_err(BPGERR_NOT_IMPLEMENTED);
	return -1;
}

/* Encrypt and sign `msg_p' into `dst_p' using keys in `pkey_p' and `skey_p'. */
/* ARGSUSED0 */
int
BPG_encrypt_sign_k(char *msg_p, char *pkey_p, char *skey_p, char *dst_p,
		   BPG_CTX *opt)
{

	BPG_err(BPGERR_NOT_IMPLEMENTED);
	return -1;
}

/* Encrypt and sign `msg' into `dst' using keys in `pkey' and `skey'. */
/* ARGSUSED0 */
int
BPG_do_encrypt_sign(BPG_BUF *msg, BPG_BUF *pkey, BPG_BUF *skey, BPG_BUF *dst,
		    BPG_CTX *opt)
{

	BPG_err(BPGERR_NOT_IMPLEMENTED);
	return -1;
}

/* free the previous contents, and set the new contents */
void
BPG_set_id(BPG_CTX *ctx, const char *type, const char *id)
{
	if (strcmp(type, "secret") == 0) {
		(void) free(ctx->sec_uid);
		ctx->sec_uid = strdup(id);
	} else {
		(void) free(ctx->pub_uid);
		ctx->pub_uid = strdup(id);
	}
}

/* Allocate a new bpg context with defaults set. */
BPG_CTX *
BPG_CTX_new(void)
{
	BPG_CTX *ctx;

	if ((ctx = malloc(sizeof(*ctx))) == NULL) {
		return NULL;
	}
	(void) memset(ctx, 0x0, sizeof(*ctx));

	ctx->cipher = BPG_DEFAULT_CIPHER;
	ctx->pubkey_algo = BPG_DEFAULT_PKA;
	ctx->hash_algo = BPG_DEFAULT_HASH;
	ctx->pktv = BPG_DEFAULT_PKTV;
	ctx->sigv = BPG_DEFAULT_SIGV;
	ctx->keyv = BPG_DEFAULT_KEYV;
	ctx->pass = NULL;
	ctx->pub_uid = strdup(DEFAULT_ID);
	ctx->sec_uid = strdup(DEFAULT_ID);
	/*ctx->pub_kid = ;*/	/* FIXME: set the default key ID. */
	/*ctx->sec_kid = ;*/	/* FIXME: set the default key ID. */
	ctx->pub_path = strdup("");
	ctx->sec_path = strdup("");
	
	return ctx;
}

/* Free a bpg context. */
void
BPG_CTX_free(BPG_CTX *ctx)
{
	if (ctx != NULL) {
		free(ctx);
	}
}

/* Allocate a new buffer. */
BPG_BUF *
BPG_BUF_new(size_t len)
{
	BPG_BUF *buf;

	if ((buf = malloc(sizeof(*buf))) == NULL) {
		return NULL;
	}
	(void) memset(buf, 0x0, sizeof(*buf));
	buf->body = NULL;
	BPG_BUF_resize(buf, len);
	buf->pos = 0;

	return buf;
}

/* Free a buffer. */ 
void
BPG_BUF_free(BPG_BUF *buf)
{

	if (buf != NULL) {
		if (buf->body != NULL)
			free(buf->body);
		free(buf);
	}
}

/* Reallocate buffer.  Buffer length is set to `len'. */
int
BPG_BUF_resize(BPG_BUF *buf, size_t len)
{

	if (buf == NULL) {
		BPG_err(BPGERR_NULL_POINTER);
		return -1;
	}

	if (len) {
		if ((buf->body = realloc(buf->body, len)) == NULL) {
			BPG_err(BPGERR_MALLOC);
			return -1;
		}
	}

	buf->len = len;

	return 0;
}

/* Print a buffer, formatted. */
void
BPG_BUF_print(BPG_BUF *buf, FILE *fp)
{
	int i;

	if (buf != NULL && fp != NULL) {
		(void)fprintf(fp, "Buffer (%d bytes):", buf->len);
		for (i = 0; i < buf->len; i++) {
			if (i % 16 == 0)
				(void)fprintf(fp, "\n  ");
			(void)fprintf(fp, "%02x ", buf->body[i]);
		}
		(void)fprintf(fp, "\n");
	}
}

/* Read `flen' bytes from `from'. */
int
BPG_BUF_read(BPG_BUF *buf, unsigned char *from, size_t flen)
{

	if (buf == NULL || from == NULL) {
		BPG_err(BPGERR_NULL_POINTER);
		return -1;
	}

	if (BPG_BUF_resize(buf, flen) == -1)
		return -1;

	memcpy(buf->body, from, flen);

	return 0;
}

/* Dumps stream `fp' into a buffer. */
int
BPG_BUF_read_f(BPG_BUF *buf, FILE *fp)
{
	size_t nread;

	if (fp == NULL || buf == NULL) {
		BPG_err(BPGERR_NULL_POINTER);
		return -1;
	}

	BPG_BUF_resize(buf, 100);
	while ((nread = fread(&buf->body[buf->len - 100], 1, 100, fp)) == 100)
		BPG_BUF_resize(buf, buf->len + 100);
	BPG_BUF_resize(buf, buf->len - 100 + nread);

	return 0;
}



/* the high-level BPG routines start here */


int
bpg_sign(bpginfo_t *info, int argc, char **argv)
{
	int	allocated;
	int	done;
	int	ret;
	int	i;

	if (info->sec_file == NULL && strcmp(info->opt->sec_uid, DEFAULT_ID) == 0) {
		warnx( "-u <identity> or -U <priv key file> is required");
		return 0;
	}
	if (info->output_file != NULL && argc > 1) {
		/* make sure we have an output file to put output in */
		warnx("Multiple files provided, but only one output file \"%s\"",
		      info->output_file);
		return 0;
	}
	for (i = 0, done = 0 ; i < argc ; i++) {
		if ((allocated = (info->output_file == NULL)) != 0) {
			char	ofile[MAXPATHLEN];

			(void) snprintf(ofile, sizeof(ofile), "%s.bpg",
							argv[i]);
			info->output_file = strdup(ofile);
		}
		if (info->verbose) {
			warnx("Signing `%s' %s.", argv[i],
				(info->detached) ? "(Detached)" : "");
			warnx("Using private key \"%s\"", info->sec_file);
		}
		if (info->detached) {
			ret = BPG_sign_detached_k(argv[i], info->sec_file,
					info->output_file, info->opt);
		} else {
			ret = BPG_sign_k(argv[i], info->sec_file,
					 info->output_file, info->opt);
		}

		if (allocated) {
			(void) free(info->output_file);
			info->output_file = NULL;
		}

		if (ret == 0) {
			done += 1;
		} else {
			warnx("%s", BPG_get_error_string());
		}
	}

	return (done == argc);
}

int
bpg_verify(bpginfo_t *info, int argc, char **argv)
{
	int ret;

	if (info->pub_file == NULL && strcmp(info->opt->pub_uid, DEFAULT_ID) == 0) {
		warnx("-r or -R is required");
		return 0;
	}

	if (argv[0] != NULL && argv[1] != NULL)
		warnx("Multiple files not yet supported, verifying `%s' only",
		      argv[0]);

	if (info->detached && info->det_sig_file == NULL) {
		warnx("-D option required.");
		return 0;
	}
	if (info->verbose) {
		warnx("Verifying `%s'.", argv[0]);
		if (info->detached) {
			warnx("(detached) with `%s'.", info->det_sig_file);
		}
	}
	if (info->detached) {
		ret = BPG_verify_detached_k(argv[0], info->det_sig_file,
				info->pub_file, info->opt);
	} else {
		ret = BPG_verify_k(argv[0], info->pub_file, info->opt);
	}

	if (ret < 0) {
		warnx("%s", BPG_get_error_string());
		return 0;
	}
	if (ret == 0) {
		warnx("***** Invalid signature of file `%s'! *****",
		      argv[0]);
		return 0;
	}
	if (info->verbose) {
		warnx("Valid signature of file \"%s\" by %s.", argv[0],
				info->opt->pub_uid);
	}
	return 1;
}

int
bpg_encrypt(bpginfo_t *info, int argc, char **argv)
{

	warnx("Encryption not yet supported");
	return 0;
}

int
bpg_symmenc(bpginfo_t *info, int argc, char **argv)
{

	warnx("Symmetric encryption not yet supported");
	return 0;
}

int
bpg_decrypt(bpginfo_t *info, int argc, char **argv)
{

	warnx("Decryption not yet supported");
	return 0;
}

int
bpg_encsig(bpginfo_t *info, int argc, char **argv)
{

	warnx("Encryption and signing not yet supported");
	return 0;
}
