/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.netbeans.modules.apisupport.project.ui.customizer;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import javax.swing.JPanel;
import org.netbeans.api.annotations.common.NonNull;
import org.netbeans.spi.project.ui.support.ProjectCustomizer;
import org.openide.util.HelpCtx;

/**
 * Provides common support for a <em>standard</em> panels in the NetBeans module
 * and suite customizers.
 *
 * @author Martin Krauskopf
 */
public abstract class NbPropertyPanel extends JPanel implements
        ModuleProperties.LazyStorage, PropertyChangeListener, HelpCtx.Provider {

    private Class<?> helpCtxClass;

    /** Property whether <code>this</code> panel is valid. */
    static final String VALID_PROPERTY = "isPanelValid"; // NOI18N
    
    /** Property for error message of this panel. */
    static final String ERROR_MESSAGE_PROPERTY = "errorMessage"; // NOI18N
    
    protected @NonNull ModuleProperties props;
    protected final ProjectCustomizer.Category category;
    
    @SuppressWarnings("LeakingThisInConstructor")
    NbPropertyPanel(@NonNull ModuleProperties props, Class<?> helpCtxClass, ProjectCustomizer.Category cat) {
        this.props = props;
        category = cat;
        props.addLazyStorage(this);
        initComponents();
        props.addPropertyChangeListener(this);
        this.helpCtxClass = helpCtxClass;
    }
    
    /**
     * This method is called whenever {@link ModuleProperties} are refreshed.
     */
    protected abstract void refresh();
    
    String getProperty(String key) {
        return props.getProperty(key);
    }
    
    void setProperty(String key, String property) {
        props.setProperty(key, property);
    }
    
    boolean getBooleanProperty(String key) {
        return props.getBooleanProperty(key);
    }
    
    void setBooleanProperty(String key, boolean property) {
        props.setBooleanProperty(key, property);
    }

    public @Override void store() { /* empty implementation */ }
    
    public @Override void propertyChange(PropertyChangeEvent evt) {
        if (ModuleProperties.PROPERTIES_REFRESHED.equals(evt.getPropertyName())) {
            refresh();
        }
    }
    
    @Override
    public void addNotify() {
        super.addNotify();
        //TODO replace with something else..
        getRootPane().putClientProperty(BasicCustomizer.LAST_SELECTED_PANEL, category.getName());
    }
    
    public @Override HelpCtx getHelpCtx() {
        return new HelpCtx(helpCtxClass);
    }
    
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents() {

        setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.CENTER, 0, 0));

    }
    // </editor-fold>//GEN-END:initComponents
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    // End of variables declaration//GEN-END:variables
    
    abstract static class Single extends NbPropertyPanel {
        Single(SingleModuleProperties props, Class<?> helpCtxClass, ProjectCustomizer.Category cat) {
            super(props, helpCtxClass, cat);
        }
        @NonNull SingleModuleProperties getProperties() {
            return (SingleModuleProperties) props;
        }
    }
    
    public abstract static class Suite extends NbPropertyPanel {
        public Suite(SuiteProperties props, Class<?> helpCtxClass, ProjectCustomizer.Category cat) {
            super(props, helpCtxClass, cat);
        }
        protected @NonNull SuiteProperties getProperties() {
            return (SuiteProperties) props;
        }
    }
    
}
