/*
 * Routines for daemon, killproc, killall5, pidof, and runlevel.
 *
 * Version:     2.0 10-Nov-2000 Fink
 *
 * Copyright 1994-2000 Werner Fink, 1996-2000 SuSE GmbH Nuernberg, Germany.
 * Copyright 2005 Werner Fink, 2005 SUSE LINUX Products GmbH, Germany.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Author:    Werner Fink <werner@suse.de>, 1994-2000
 *
 * 1998/09/29 Werner Fink: Add kernel thread handling.
 * 1999/02/24 Werner Fink: Add xread to avoid EINTR
 * 1999/08/05 Werner Fink: environment, move some inlined into libint.c
 * 2000/11/10 Werner Fink: LSB specs, logging
 */

#include <stdlib.h>
#include <stdio.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/param.h>
#include <utmp.h>
#include <pwd.h>
#include <mntent.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <syslog.h>
#include <errno.h>
#include <stdarg.h>
#include <ctype.h>
#ifdef USE_BLOGD
# include <libblogger.h>
#endif

/*
 * LSB specs:
 *
 * For start/stop and others but status
 *   0 - success
 *   1 - generic or unspecified error
 *   2 - invalid or excess argument(s)
 *   3 - unimplemented feature (e.g. "reload")
 *   4 - insufficient privilege
 *   5 - program is not installed
 *   6 - program is not configured
 *   7 - program is not running
 */
#define LSB_OK			0
#define LSB_FAILED		1
#define LSB_WRGSYN		2
#define LSB_MISSED		3	/* not used */
#define LSB_NOPERM		4
#define LSB_NOENTR		5
#define LSB_NOCONF		6	/* not used */
#define LSB_NOPROC		((flags & KSTOP) ? LSB_OK : 7 )

#define LSB_PROOF 		((errno == EPERM || errno == EACCES) ? LSB_NOPERM : LSB_FAILED )
#define LSB_PROOFX 		((errno == ENOENT) ? LSB_NOENTR : LSB_PROOF )
#define LSB_PROOFE 		((errno == EPERM || errno == EACCES) ? LSB_NOPERM : LSB_NOPROC )
/*
 * For status
 *   0 - service running
 *   1 - service dead, but /var/run/  pid  file exists
 *   2 - service dead, but /var/lock/ lock file exists
 *   3 - service not running
 */
#define LSB_STATUS_OK		((flags & KILL) ? LSB_OK     : 0 )
#define LSB_STATUS_ISDEAD	((flags & KILL) ? LSB_NOPROC : 1 )
#define LSB_STATUS_NOLOCK	2	/* not used */
#define LSB_STATUS_NOPROC	((flags & KILL) ? LSB_NOPROC : 3 )

#define WRGSYNTAX		102	/* usage etc. */
#define NOPIDREAD		101	/* trouble */

#define LSB_STATUS_PROOF	((errno == EPERM) ? LSB_NOPERM : NOPIDREAD )
#define LSB_STATUS_PROOFX	((errno == ENOENT) ? LSB_NOENTR : LSB_STATUS_PROOF )

#define LOG_OPTIONS	(LOG_ODELAY|LOG_CONS)
#define LOG_FACILITY	LOG_LOCAL7

#define PIDOF   0x0001
#define DAEMON  0x0002
#define KILL    0x0004
#define KTHREAD 0x0008
#define NZOMBIE 0x0010
#define KSHORT  0x0020
#define FLWLINK 0x0040
#define KSTOP   0x0080
#define KBASE   0x0100

#define  MAXENV  20
#define  CMDLLEN MAXNAMLEN	/* The string length of /proc/12345/cmdline\0\0 + 1 */
#ifdef _PATH_VARRUN
# define DEFPIDDIR _PATH_VARRUN
#else
# define DEFPIDDIR "/var/run/"
#endif
#define  DEFPIDEXT ".pid"
#define  DEFPIDLEN 14		/* The string length of /var/run/.pid + 1 */

typedef enum _boolean {false, true} boolean;

extern char **environ;
extern char     * newenvp[];
extern unsigned   newenvc;

extern char * we_are;
extern unsigned short stopped;
extern pid_t p_pid, p_sid, p_ppid;

/* Declare */
extern int  pidof  (const char * inname, const char * root, const unsigned short flag);
extern int  remember_pids (const char * pids, const char * inname, const char * root,
			   unsigned short flags);
extern int  verify_pidfile (const char * pid_file, const char * inname, const char * root,
			    const unsigned short flag, const boolean ignore);
extern int  check_pids (const char * inname, const char * root, const unsigned short flag);
extern void clear_pids (void);
extern void error(int stat, const char *fmt, ...);
extern void warn(const char *fmt, ...);
extern int rlstat(char ** file, struct stat *st, const unsigned short flag);
extern void init_nfs(void);
boolean check4nfs(const char * path);

/* Feature */
extern char *sys_signame [];
extern char *sys_sigalias[];
extern void init_signames();

extern int signame_to_signum (const char *sig);

/* Used in killproc.c only once to list the signal names */
static inline void list_signames(void)
{
    int n, l;

    init_signames();
    for (n = 1, l = 0; n < NSIG+1; n++) {
	if (sys_signame [n]) {
	    printf("%2d) SIG%-9s", n, sys_signame [n]);
	    l++;
	}
	if (sys_sigalias[n]) {
	    printf("%2d) SIG%-9s", n, sys_sigalias[n]);
	    l++;
	}
	if (!(l % 4))
	    putc('\n', stdout);
    }
    if (l % 4)
	putc('\n', stdout);
}

typedef struct _proc_
{
    struct _proc_ *next;	/* Pointer to next struct. */
    struct _proc_ *prev;	/* Pointer to previous st. */
    pid_t pid;			/* Process ID.             */
    pid_t sid;			/* Session ID.             */
} PROC;

extern PROC * remember;
extern PROC * doignore;

/* Inlined functions: just like macros */

static inline void *xmalloc(const size_t bytes)
{
    void *p = malloc(bytes);

    if (p == (void*)0) {
	if (stopped) kill(-1, SIGCONT);
	error(100, "malloc(): %s\n", strerror(errno));
    }
    return p;
}

static inline char *xstrdup(const char * string)
{
    char *p = strdup(string);

    if (p == (char*)0) {
	if (stopped) kill(-1, SIGCONT);
	error(100, "strdup(): %s\n", strerror(errno));
    }
    return p;
}

static inline char * base_name ( const char * full )
{
    char *basename = strrchr(full, '/');

    if (basename == (char *)0)
	basename = (char *)full;
    else
	basename++;

    return basename;
}

static inline char * swap_name ( const char * base )
{
    size_t len = strlen(base) + 2 + 1;
    char *swap = (char*)xmalloc(len);

    return strcat(strcat(strcpy(swap,"("),base),")");
}

static inline char * exe_entry ( char *buf, const char *d_name )
{
    return strcat(strcat(strcpy(buf,"/proc/"),d_name),"/exe");
}

static inline char * stat_entry ( char *buf, const char *d_name )
{
    return strcat(strcat(strcpy(buf,"/proc/"),d_name),"/stat");
}

static inline char * statm_entry ( char *buf, const char *d_name )
{
    return strcat(strcat(strcpy(buf,"/proc/"),d_name),"/statm");
}

static inline char * env_entry ( char *buf, const char *d_name )
{
    return strcat(strcat(strcpy(buf,"/proc/"),d_name),"/environ");
}

static inline char * cmd_entry ( char *buf, const char *d_name )
{
    return strcat(strcat(strcpy(buf,"/proc/"),d_name),"/cmdline");
}

static inline char * pid_entry ( char *buf, const char *pid )
{
    return strcat(strcpy(buf,"/proc/"),pid);
}

extern void addnewenv ( const char * name, const char * entry );
extern char ** runlevel(const char *file);

/* Used in startproc only once to overwrite the environment */
static inline void set_newenv(const char * fullname)
{
    char *tmp;
    char buf[_POSIX_PATH_MAX + 1];

/*
 *  Default environment for a daemon:
 *  PATH=..., HOME=..., SHELL=...
 *  We need control environment:
 *  DAEMON=fullname, PREVLEVEL=..., RUNLEVEL=...
 */
    if ( (tmp = getenv("HOME")) != (char*)0 )
	addnewenv("HOME",tmp);
    else
	addnewenv("HOME","/");

    if ( (tmp = getenv("PATH")) != (char*)0 )
	addnewenv("PATH",tmp);
    else
	addnewenv("PATH","/usr/bin:/usr/sbin:/bin:/sbin");

    if ( (tmp = getenv("SHELL")) != (char*)0 )
	addnewenv("SHELL",tmp);
    else
	addnewenv("SHELL","/bin/sh");

    if ( (tmp = getenv("RUNLEVEL")) != (char*)0 )
	addnewenv("RUNLEVEL",tmp);

    if ( (tmp = getenv("PREVLEVEL")) != (char*)0 )
	addnewenv("PREVLEVEL",tmp);
    else {
	char ** tmp = runlevel((char*)0);
	addnewenv("PREVLEVEL",tmp[0]);
	addnewenv("RUNLEVEL", tmp[1]);
    }

    environ = (char**)newenvp; /* Make new environment active */

    (void)snprintf(buf, _POSIX_PATH_MAX, "%s",fullname);
    addnewenv("DAEMON",buf);

    return;
}

/* Used in startproc only once to extend the environment */
static inline void set_environ(const char * fullname)
{
    char *tmp;
    char buf[_POSIX_PATH_MAX + 1];

    if ( (tmp = getenv("RUNLEVEL")) != (char*)0 )
	setenv("RUNLEVEL",tmp,1);
    if ( (tmp = getenv("PREVLEVEL")) != (char*)0 )
	setenv("PREVLEVEL",tmp,1);
    else {
	char ** tmp = runlevel((char*)0);
	setenv("PREVLEVEL",tmp[0],1);
	setenv("RUNLEVEL", tmp[1],1);
    }

    (void)snprintf(buf, _POSIX_PATH_MAX, "%s",fullname);
    setenv("DAEMON",buf,1);
    return;
}

/* Add a task to our remember list */
static inline void do_list(const pid_t pid, const pid_t sid, const boolean ignore)
{
    PROC * p = (PROC*)xmalloc(sizeof(PROC));
    p->pid = pid;
    p->sid = sid;
    if (ignore) {
        if (doignore)
	    doignore->prev = p;
        p->next = doignore;
        p->prev = (PROC*)0;
        doignore = p;
    } else {
        if (remember)
	    remember->prev = p;
        p->next = remember;
        p->prev = (PROC*)0;
        remember = p;
    }

    return;
}

static inline boolean check_ignore(const pid_t pid)
{
    PROC * q, * m = doignore;

    if (pid <= 1)
	goto out;

    for (q = doignore; m; q = m) {
	m = q->next;
        if (pid == q->pid)
	    return true;
    }
out:
    return false;
}

static inline void clear_ignore(void)
{
    PROC * p, *l, * n = remember;

    if (!doignore)
	return;

    for (p = remember; n; p = n) {
	l = p->prev;
	n = p->next;

	if (!check_ignore(p->pid))
	    continue;

	/* Remove this entry in remember because we ignore it */
	if (p == remember) {
	    if (n) n->prev = (PROC*)0;
	    remember = n;
	    free(p);
	} else if (l) {
	    if (n) n->prev = l;
	    l->next = n;
	    free(p);
	}
    }
}
/* libinit.h ends here */
