/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "assert-macros.h"
#include "test-util.h"

#include "gpass/crypt-stream.h"

/***********************************************************
 *
 * initialize/terminate,  setup/teardown
 *
 ***********************************************************/
static void
initialize(void)
{
    g_type_init();
}

static void
terminate(void)
{
}

static void
setup(void)
{
}

static void
teardown(void)
{
}

/***********************************************************
 *
 * test case
 *
 ***********************************************************/
static void
create_crypt_file(const guchar *data, gint data_bytes,
                  FILE **file, gchar **path)
{
    /* SHA-1 hash from "qwerty" */
    MCRYPT mcrypt_thread;
    guchar key[20] = {
        0xb1, 0xb3, 0x77, 0x3a, 0x05, 0xc0, 0xed, 0x01, 0x76, 0x78, 
        0x7a, 0x4f, 0x15, 0x74, 0xff, 0x00, 0x75, 0xf7, 0x52, 0x1e
    };
    unsigned char iv[] = { 5, 23, 1, 123, 12, 3, 54, 94 };
    guchar *copied;
    guchar *p;
    int result;

    mcrypt_thread =
        mcrypt_module_open(MCRYPT_BLOWFISH, NULL, MCRYPT_CBC, NULL);
    if (mcrypt_thread == MCRYPT_FAILED) {
        fail("failed mcrypt_module_open()");
    }
    result = mcrypt_generic_init(mcrypt_thread, key, 20, iv);
    if (result < 0) {
        fail("failed mcrypt_generic_init()");
    }
    if ((data_bytes % 8) != 0) {
        fail("not enough data");
    }
    copied = g_malloc(sizeof(guchar) * data_bytes);
    memcpy(copied, data, data_bytes);
    for (p = copied; p < copied + data_bytes; p += 8) {
        mcrypt_generic(mcrypt_thread, p, 8);
    }
    mcrypt_generic_deinit(mcrypt_thread);
    mcrypt_module_close(mcrypt_thread);
    test_util_create_temporary_file("w+", file, path);
    fwrite(copied, data_bytes, 1, *file);
    g_free(copied);
    rewind(*file);
}

START_TEST(test_read)
{
    GPassDecryptStream *decrypt;
    gchar data[16] = "0123456789012345";
    FILE *file;
    gchar *path;
    gchar buf[16];
    ssize_t read_len;
    GError *error;
    
    create_crypt_file(data, 16, &file, &path);
    gpass_decrypt_stream_open(&decrypt, file, "qwerty");
    
    error = gpass_decrypt_stream_read(decrypt, buf, 16, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY(data, buf, 16);
    ASSERT_EQUAL_INT(16, read_len);
    ASSERT_TRUE(gpass_decrypt_stream_eof(decrypt));
    
    gpass_decrypt_stream_close(decrypt);
    unlink(path);
    g_free(path);
}
END_TEST

START_TEST(test_read__each)
{
    GPassDecryptStream *decrypt;
    gchar data[16] = "0123456789012345";
    FILE *file;
    gchar *path;
    gchar buf[4];
    ssize_t read_len;
    GError *error;
    
    create_crypt_file(data, 16, &file, &path);
    gpass_decrypt_stream_open(&decrypt, file, "qwerty");

    error = gpass_decrypt_stream_read(decrypt, buf, 4, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY("0123", buf, 4);
    ASSERT_EQUAL_INT(4, read_len);
    error = gpass_decrypt_stream_read(decrypt, buf, 4, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY("4567", buf, 4);
    ASSERT_EQUAL_INT(4, read_len);
    error = gpass_decrypt_stream_read(decrypt, buf, 4, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY("8901", buf, 4);
    ASSERT_EQUAL_INT(4, read_len);
    error = gpass_decrypt_stream_read(decrypt, buf, 4, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY("2345", buf, 4);
    ASSERT_EQUAL_INT(4, read_len);
    ASSERT_TRUE(gpass_decrypt_stream_eof(decrypt));

    gpass_decrypt_stream_close(decrypt);
    unlink(path);
    g_free(path);
}
END_TEST

START_TEST(test_read__padded)
{
    GPassDecryptStream *decrypt;
    gchar data[8] = {
        '0', '1', '2', '3', '4', '5', 0x02, 0x02
    };
    FILE *file;
    gchar *path;
    gchar buf[6];
    ssize_t read_len;
    GError *error;

    create_crypt_file(data, 8, &file, &path);
    gpass_decrypt_stream_open(&decrypt, file, "qwerty");

    error = gpass_decrypt_stream_read(decrypt, buf, 6, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY(data, buf, 6);
    ASSERT_EQUAL_INT(6, read_len);
    ASSERT_TRUE(gpass_decrypt_stream_eof(decrypt));

    gpass_decrypt_stream_close(decrypt);
    unlink(path);
    g_free(path);
}
END_TEST

START_TEST(test_read__not_pad)
{
    GPassDecryptStream *decrypt;
    gchar data[8] = {
        '0', '1', '2', '3', '4', '5', 0x01, 0x03
    };
    FILE *file;
    gchar *path;
    gchar buf[6];
    ssize_t read_len;
    GError *error;

    create_crypt_file(data, 8, &file, &path);
    gpass_decrypt_stream_open(&decrypt, file, "qwerty");
    
    error = gpass_decrypt_stream_read(decrypt, buf, 8, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY(data, buf, 8);
    ASSERT_EQUAL_INT(8, read_len);
    ASSERT_TRUE(gpass_decrypt_stream_eof(decrypt));
    
    gpass_decrypt_stream_close(decrypt);
    unlink(path);
    g_free(path);
}
END_TEST

START_TEST(test_read_line)
{
    GPassDecryptStream *decrypt;
    gchar data[16] = "0123\n56789012\n4\n";
    FILE *file;
    gchar *path;
    GString *line;
    GError *error;
    
    create_crypt_file(data, 16, &file, &path);
    gpass_decrypt_stream_open(&decrypt, file, "qwerty");
    
    line = g_string_new(NULL);
    error = gpass_decrypt_stream_read_line(decrypt, &line);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY("0123\n", line->str, 5);
    ASSERT_EQUAL_INT(5, line->len);
    error = gpass_decrypt_stream_read_line(decrypt, &line);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY("56789012\n", line->str, 9);
    ASSERT_EQUAL_INT(9, line->len);
    error = gpass_decrypt_stream_read_line(decrypt, &line);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY("4\n", line->str, 2);
    ASSERT_EQUAL_INT(2, line->len);
    g_string_free(line, TRUE);
    ASSERT_TRUE(gpass_decrypt_stream_eof(decrypt));
    
    gpass_decrypt_stream_close(decrypt);
    unlink(path);
    g_free(path);
}
END_TEST

START_TEST(test_read_line__no_lf)
{
    GPassDecryptStream *decrypt;
    gchar data[8] = {
        '0', '1', '2', '3', '4', '5', 0x02, 0x02
    };
    FILE *file;
    gchar *path;
    GString *line;
    GError *error;
    
    create_crypt_file(data, 8, &file, &path);
    gpass_decrypt_stream_open(&decrypt, file, "qwerty");
    
    line = g_string_new(NULL);
    error = gpass_decrypt_stream_read_line(decrypt, &line);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY("012345", line->str, 6);
    ASSERT_EQUAL_INT(6, line->len);
    ASSERT_TRUE(gpass_decrypt_stream_eof(decrypt));
    
    gpass_decrypt_stream_close(decrypt);
    unlink(path);
    g_free(path);
}
END_TEST

START_TEST(test_write)
{
    GPassEncryptStream *encrypt;
    GPassDecryptStream *decrypt;
    FILE *file;
    gchar *path;
    gchar buf[8];
    ssize_t read_len;
    GError *error;
    
    test_util_create_temporary_file("w+", &file, &path);
    gpass_encrypt_stream_open(&encrypt, file, "qwerty");
    error = gpass_encrypt_stream_write(encrypt, "01234567", 8);
    ASSERT_NULL(error);
    gpass_encrypt_stream_close(encrypt);
    
    file = fopen(path, "r");
    gpass_decrypt_stream_open(&decrypt, file, "qwerty");
    error = gpass_decrypt_stream_read(decrypt, buf, 8, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY("01234567", buf, 8);
    ASSERT_EQUAL_INT(8, read_len);
    gpass_decrypt_stream_close(decrypt);
    
    unlink(path);
    g_free(path);
}
END_TEST

START_TEST(test_write__each)
{
    GPassEncryptStream *encrypt;
    GPassDecryptStream *decrypt;
    FILE *file;
    gchar *path;
    gchar buf[16];
    ssize_t read_len;
    GError *error;
    
    test_util_create_temporary_file("w+", &file, &path);
    gpass_encrypt_stream_open(&encrypt, file, "qwerty");
    error = gpass_encrypt_stream_write(encrypt, "0123", 4);
    ASSERT_NULL(error);
    error = gpass_encrypt_stream_write(encrypt, "4567", 4);
    ASSERT_NULL(error);
    error = gpass_encrypt_stream_write(encrypt, "8901", 4);
    ASSERT_NULL(error);
    error = gpass_encrypt_stream_write(encrypt, "2345", 4);
    ASSERT_NULL(error);
    gpass_encrypt_stream_close(encrypt);

    file = fopen(path, "r");
    gpass_decrypt_stream_open(&decrypt, file, "qwerty");
    error = gpass_decrypt_stream_read(decrypt, buf, 16, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY("0123456789012345", buf, 16);
    ASSERT_EQUAL_INT(16, read_len);
    gpass_decrypt_stream_close(decrypt);
    
    unlink(path);
    g_free(path);
}
END_TEST

START_TEST(test_write__padded)
{
    GPassEncryptStream *encrypt;
    GPassDecryptStream *decrypt;
    FILE *file;
    gchar *path;
    gchar buf[4];
    ssize_t read_len;
    GError *error;
    
    test_util_create_temporary_file("w+", &file, &path);
    gpass_encrypt_stream_open(&encrypt, file, "qwerty");
    error = gpass_encrypt_stream_write(encrypt, "0123", 4);
    ASSERT_NULL(error);
    gpass_encrypt_stream_close(encrypt);

    file = fopen(path, "r");
    gpass_decrypt_stream_open(&decrypt, file, "qwerty");
    error = gpass_decrypt_stream_read(decrypt, buf, 4, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY("0123", buf, 4);
    ASSERT_EQUAL_INT(4, read_len);
    error = gpass_decrypt_stream_read(decrypt, buf, 1, &read_len);
    ASSERT_NOT_NULL(error);
    g_error_free(error);
    gpass_decrypt_stream_close(decrypt);
    
    unlink(path);
    g_free(path);
}
END_TEST

/***********************************************************
 *
 * suite / main
 *
 ***********************************************************/
static Suite *
test_suite(void)
{
    Suite *s = suite_create("GPassCryptStream");
    TCase *tc;
    
    tc = tcase_create("functions");
    suite_add_tcase(s, tc);
    tcase_add_checked_fixture(tc, setup, teardown);
    
    tcase_add_test(tc, test_read);
    tcase_add_test(tc, test_read__each);
    tcase_add_test(tc, test_read__padded);
    tcase_add_test(tc, test_read__not_pad);
    tcase_add_test(tc, test_read_line);
    tcase_add_test(tc, test_read_line__no_lf);
    tcase_add_test(tc, test_write);
    tcase_add_test(tc, test_write__each);
    tcase_add_test(tc, test_write__padded);
    return s;
}

int
main(int argc, char *argv[])
{
    Suite *s;
    SRunner *sr;
    int nf;

    initialize();
    
    s = test_suite();
    sr = srunner_create(s);
    srunner_run_all(sr, CK_ENV);
    nf = srunner_ntests_failed(sr);
    srunner_free(sr);
    
    terminate();
    return (nf == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}
