/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "assert-macros.h"

#include "gpass/attribute.h"

/***********************************************************
 *
 * initialize/terminate,  setup/teardown
 *
 ***********************************************************/
static void
initialize(void)
{
    g_type_init();
}

static void
terminate(void)
{
}

static void
setup(void)
{
}

static void
teardown(void)
{
}

/***********************************************************
 *
 * test case
 *
 ***********************************************************/
START_TEST(test_new__integer)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_INTEGER,
                     "name", "integer",
                     "nick", "Integer Number",
                     "blurb", "The integer number", NULL);
    
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_INTEGER, attr->type);
    ASSERT_EQUAL_STRING("integer", attr->name);
    ASSERT_EQUAL_STRING("Integer Number", attr->nick);
    ASSERT_EQUAL_STRING("The integer number", attr->blurb);

    g_object_unref(attr);
}
END_TEST

START_TEST(test_set__integer)
{
    GPassAttribute *attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                                        "type", GPASS_ATTRIBUTE_TYPE_INTEGER,
                                        NULL);
    GError *error;
    
    error = gpass_attribute_set(attr, 10);
    ASSERT_NULL(error);
    ASSERT_EQUAL_INT(10, g_value_get_int(attr->value));

    error = gpass_attribute_set(attr, 3000);
    ASSERT_NULL(error);
    ASSERT_EQUAL_INT(3000, g_value_get_int(attr->value));

    g_object_unref(attr);
}
END_TEST

START_TEST(test_get__integer)
{
    GPassAttribute *attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                                        "type", GPASS_ATTRIBUTE_TYPE_INTEGER,
                                        NULL);
    gint val;
    GError *error;
    
    g_value_set_int(attr->value, 10);
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_EQUAL_INT(10, val);

    g_value_set_int(attr->value, 3000);
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_EQUAL_INT(3000, val);

    g_object_unref(attr);
}
END_TEST

START_TEST(test_new__boolean)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_BOOLEAN,
                     "name", "boolean",
                     "nick", "Boolean",
                     "blurb", "The boolean", NULL);
    
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_BOOLEAN, attr->type);
    ASSERT_EQUAL_STRING("boolean", attr->name);
    ASSERT_EQUAL_STRING("Boolean", attr->nick);
    ASSERT_EQUAL_STRING("The boolean", attr->blurb);

    g_object_unref(attr);
}
END_TEST

START_TEST(test_set__boolean)
{
    GPassAttribute *attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                                        "type", GPASS_ATTRIBUTE_TYPE_BOOLEAN,
                                        NULL);
    GError *error;
    
    error = gpass_attribute_set(attr, TRUE);
    ASSERT_NULL(error);
    ASSERT_TRUE(g_value_get_boolean(attr->value));

    error = gpass_attribute_set(attr, FALSE);
    ASSERT_NULL(error);
    ASSERT_FALSE(g_value_get_boolean(attr->value));

    g_object_unref(attr);
}
END_TEST

START_TEST(test_get__boolean)
{
    GPassAttribute *attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                                        "type", GPASS_ATTRIBUTE_TYPE_BOOLEAN,
                                        NULL);
    gboolean val;
    GError *error;
    
    g_value_set_boolean(attr->value, TRUE);
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_TRUE(val);

    g_value_set_boolean(attr->value, FALSE);
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_FALSE(val);

    g_object_unref(attr);
}
END_TEST

START_TEST(test_new__time)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_TIME,
                     "name", "time",
                     "nick", "Time",
                     "blurb", "The time", NULL);
    
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_TIME, attr->type);
    ASSERT_EQUAL_STRING("time", attr->name);
    ASSERT_EQUAL_STRING("Time", attr->nick);
    ASSERT_EQUAL_STRING("The time", attr->blurb);

    g_object_unref(attr);
}
END_TEST

START_TEST(test_set__time)
{
    GPassAttribute *attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                                        "type", GPASS_ATTRIBUTE_TYPE_TIME,
                                        NULL);
    time_t now = time(NULL);
    GError *error;
    
    error = gpass_attribute_set(attr, now);
    ASSERT_NULL(error);
    ASSERT_EQUAL_INT(now, g_value_get_ulong(attr->value));

    error = gpass_attribute_set(attr, 0);
    ASSERT_NULL(error);
    ASSERT_EQUAL_INT(0, g_value_get_ulong(attr->value));

    g_object_unref(attr);
}
END_TEST

START_TEST(test_get__time)
{
    GPassAttribute *attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                                        "type", GPASS_ATTRIBUTE_TYPE_TIME,
                                        NULL);
    time_t val;
    GError *error;
    
    g_value_set_ulong(attr->value, 10000);
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_EQUAL_INT(10000, val);

    g_value_set_ulong(attr->value, 0);
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_EQUAL_INT(0, val);

    g_object_unref(attr);
}
END_TEST

START_TEST(test_new__string)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_STRING,
                     "name", "string",
                     "nick", "String",
                     "blurb", "The string", NULL);
    
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_STRING, attr->type);
    ASSERT_EQUAL_STRING("string", attr->name);
    ASSERT_EQUAL_STRING("String", attr->nick);
    ASSERT_EQUAL_STRING("The string", attr->blurb);

    g_object_unref(attr);
}
END_TEST

START_TEST(test_set__string)
{
    GPassAttribute *attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                                        "type", GPASS_ATTRIBUTE_TYPE_STRING,
                                        NULL);
    GError *error;
    
    error = gpass_attribute_set(attr, "abcdefg");
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("abcdefg", g_value_get_string(attr->value));

    error = gpass_attribute_set(attr, "0123456");
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("0123456", g_value_get_string(attr->value));

    error = gpass_attribute_set(attr, NULL);
    ASSERT_NULL(error);
    ASSERT_NULL(g_value_get_string(attr->value));

    g_object_unref(attr);
}
END_TEST

START_TEST(test_get__string)
{
    GPassAttribute *attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                                        "type", GPASS_ATTRIBUTE_TYPE_STRING,
                                        NULL);
    const gchar *val;
    GError *error;
    
    g_value_set_string(attr->value, "abcdefg");
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("abcdefg", val);
    
    g_value_set_string(attr->value, "0123456");
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("0123456", val);

    g_value_set_string(attr->value, NULL);
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_NULL(val);
    
    g_object_unref(attr);
}
END_TEST

START_TEST(test_new__text)
{
    GPassAttribute *attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                                        "type", GPASS_ATTRIBUTE_TYPE_TEXT,
                                        "name", "text",
                                        "nick", "Text",
                                        "blurb", "The text", NULL);
    
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_TEXT, attr->type);
    ASSERT_EQUAL_STRING("text", attr->name);
    ASSERT_EQUAL_STRING("Text", attr->nick);
    ASSERT_EQUAL_STRING("The text", attr->blurb);

    g_object_unref(attr);
}
END_TEST

START_TEST(test_set__text)
{
    GPassAttribute *attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                                        "type", GPASS_ATTRIBUTE_TYPE_TEXT,
                                        NULL);
    GError *error;
    
    error = gpass_attribute_set(attr, "abc\ndef");
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("abc\ndef", g_value_get_string(attr->value));

    error = gpass_attribute_set(attr, "0123\n456");
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("0123\n456", g_value_get_string(attr->value));

    error = gpass_attribute_set(attr, NULL);
    ASSERT_NULL(error);
    ASSERT_NULL(g_value_get_string(attr->value));

    g_object_unref(attr);
}
END_TEST

START_TEST(test_get__text)
{
    GPassAttribute *attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                                        "type", GPASS_ATTRIBUTE_TYPE_TEXT,
                                        NULL);
    const gchar *val;
    GError *error;
    
    g_value_set_string(attr->value, "abc\ndefg");
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("abc\ndefg", val);
    
    g_value_set_string(attr->value, "0123\n456");
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("0123\n456", val);

    g_value_set_string(attr->value, NULL);
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_NULL(val);
    
    g_object_unref(attr);
}
END_TEST

START_TEST(test_new__password)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_PASSWORD,
                     "name", "password",
                     "nick", "Password",
                     "blurb", "The password", NULL);
    
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_PASSWORD, attr->type);
    ASSERT_EQUAL_STRING("password", attr->name);
    ASSERT_EQUAL_STRING("Password", attr->nick);
    ASSERT_EQUAL_STRING("The password", attr->blurb);

    g_object_unref(attr);
}
END_TEST

START_TEST(test_set__password)
{
    GPassAttribute *attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                                        "type", GPASS_ATTRIBUTE_TYPE_PASSWORD,
                                        NULL);
    GError *error;
    
    error = gpass_attribute_set(attr, "qwerty");
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("qwerty", g_value_get_string(attr->value));

    error = gpass_attribute_set(attr, "Eughee8ohtiaceicheud");
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("Eughee8ohtiaceicheud",
                        g_value_get_string(attr->value));

    g_object_unref(attr);
}
END_TEST

START_TEST(test_get__password)
{
    GPassAttribute *attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                                        "type", GPASS_ATTRIBUTE_TYPE_PASSWORD,
                                        NULL);
    const gchar *val;
    GError *error;
    
    g_value_set_string(attr->value, "qwerty");
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("qwerty", val);
    
    g_value_set_string(attr->value, "Eughee8ohtiaceicheud");
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("Eughee8ohtiaceicheud", val);

    g_object_unref(attr);
}
END_TEST

START_TEST(test_new__url)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_URL,
                     "name", "url",
                     "nick", "Url",
                     "blurb", "The url", NULL);
    
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_URL, attr->type);
    ASSERT_EQUAL_STRING("url", attr->name);
    ASSERT_EQUAL_STRING("Url", attr->nick);
    ASSERT_EQUAL_STRING("The url", attr->blurb);

    g_object_unref(attr);
}
END_TEST

START_TEST(test_set__url)
{
    GPassAttribute *attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                                        "type", GPASS_ATTRIBUTE_TYPE_URL,
                                        NULL);
    GError *error;
    
    error = gpass_attribute_set(attr, "qwerty");
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("qwerty", g_value_get_string(attr->value));

    error = gpass_attribute_set(attr, "Eughee8ohtiaceicheud");
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("Eughee8ohtiaceicheud",
                        g_value_get_string(attr->value));

    g_object_unref(attr);
}
END_TEST

START_TEST(test_get__url)
{
    GPassAttribute *attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                                        "type", GPASS_ATTRIBUTE_TYPE_URL,
                                        NULL);
    const gchar *val;
    GError *error;
    
    g_value_set_string(attr->value, "qwerty");
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("qwerty", val);
    
    g_value_set_string(attr->value, "Eughee8ohtiaceicheud");
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("Eughee8ohtiaceicheud", val);

    g_object_unref(attr);
}
END_TEST

START_TEST(test_new__entry_type)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_ENTRY_TYPE,
                     "name", "entry type",
                     "nick", "Entry Type",
                     "blurb", "The entry type", NULL);
    
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_ENTRY_TYPE, attr->type);
    ASSERT_EQUAL_STRING("entry type", attr->name);
    ASSERT_EQUAL_STRING("Entry Type", attr->nick);
    ASSERT_EQUAL_STRING("The entry type", attr->blurb);

    g_object_unref(attr);
}
END_TEST

START_TEST(test_set__entry_type)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_ENTRY_TYPE, NULL);
    GError *error;
    
    error = gpass_attribute_set(attr, "general");
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("general", g_value_get_string(attr->value));

    error = gpass_attribute_set(attr, "folder");
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("folder", g_value_get_string(attr->value));

    g_object_unref(attr);
}
END_TEST

START_TEST(test_get__entry_type)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_ENTRY_TYPE, NULL);
    const gchar *val;
    GError *error;
    
    g_value_set_string(attr->value, "general");
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("general", val);
    
    g_value_set_string(attr->value, "folder");
    error = gpass_attribute_get(attr, &val);
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("folder", val);

    g_object_unref(attr);
}
END_TEST

static void
check_dump_integer(GPassAttribute *attr)
{
    GString *buf = g_string_new(NULL);
    const guchar expected_data_10[] = { 0x0a };
    const guchar expected_data_524[] = { 0x8c, 0x04 };
    const guchar expected_data_50040[] = { 0xf8, 0x86, 0x03 };
    GError *error;

    buf = g_string_assign(buf, "");
    gpass_attribute_set(attr, 10);
    error = gpass_attribute_dump(attr, &buf);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY(expected_data_10, buf->str,
                        G_N_ELEMENTS(expected_data_10));

    buf = g_string_assign(buf, "");
    gpass_attribute_set(attr, 524);
    error = gpass_attribute_dump(attr, &buf);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY(expected_data_524, buf->str, 
                        G_N_ELEMENTS(expected_data_524));

    buf = g_string_assign(buf, "");
    gpass_attribute_set(attr, 50040);
    error = gpass_attribute_dump(attr, &buf);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY(expected_data_50040, buf->str, 
                        G_N_ELEMENTS(expected_data_50040));

    g_string_free(buf, TRUE);
}

START_TEST(test_dump__integer)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_INTEGER, NULL);

    check_dump_integer(attr);
    g_object_unref(attr);
}
END_TEST

START_TEST(test_dump__time)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_TIME, NULL);

    check_dump_integer(attr);
    g_object_unref(attr);
}
END_TEST

START_TEST(test_dump__boolean)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_BOOLEAN, NULL);
    GString *buf = g_string_new(NULL);
    const guchar expected_data_true[] = { 0x01 };
    const guchar expected_data_false[] = { 0x00 };
    GError *error;
    
    buf = g_string_assign(buf, "");
    gpass_attribute_set(attr, TRUE);
    error = gpass_attribute_dump(attr, &buf);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY(expected_data_true, buf->str,
                        G_N_ELEMENTS(expected_data_true));

    buf = g_string_assign(buf, "");
    gpass_attribute_set(attr, FALSE);
    error = gpass_attribute_dump(attr, &buf);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY(expected_data_false, buf->str, 
                        G_N_ELEMENTS(expected_data_false));
    
    g_string_free(buf, TRUE);
    g_object_unref(attr);
}
END_TEST

START_TEST(test_dump__string)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_STRING, NULL);
    GString *buf = g_string_new(NULL);
    const guchar expected_data_1[] = {
        0x06, 's', 't', 'r', 'i', 'n', 'g', 
    };
    const guchar expected_data_2[] = {
        0x1a,
        'a', 'b', 'c', 'd', 'e', 'f', 'g',
        'h', 'i', 'j', 'k', 'l', 'm', 'n',
        'o', 'p', 'q', 'r', 's', 't', 'u',
        'v', 'w', 'x', 'y', 'z', 
    };
    GError *error;
    
    buf = g_string_assign(buf, "");
    gpass_attribute_set(attr, "string");
    error = gpass_attribute_dump(attr, &buf);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY(expected_data_1, buf->str,
                        G_N_ELEMENTS(expected_data_1));

    buf = g_string_assign(buf, "");
    gpass_attribute_set(attr, "abcdefghijklmnopqrstuvwxyz");
    error = gpass_attribute_dump(attr, &buf);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY(expected_data_2, buf->str, 
                        G_N_ELEMENTS(expected_data_2));
    
    g_string_free(buf, TRUE);
    g_object_unref(attr);
}
END_TEST

static void
check_load_integer(GPassAttribute *attr)
{
    gint val;
    gsize read_len;
    const guchar dump_data_0[] = { 0x00 };
    const guchar dump_data_10[] = { 0x0a };
    const guchar dump_data_524[] = { 0x8c, 0x04 };
    const guchar dump_data_50040[] = { 0xf8, 0x86, 0x03 };
    const guchar dump_data_error1[] = { 0x81, 0x81, 0x81, 0x81, 0x81 };
    const guchar dump_data_error2[] = { 0x81, 0x81 };
    GError *error;

    error = gpass_attribute_load(attr, dump_data_0, 1, &read_len);
    ASSERT_NULL(error);
    gpass_attribute_get(attr, &val);
    ASSERT_EQUAL_INT(0, val);
    ASSERT_EQUAL_INT(1, read_len);

    error = gpass_attribute_load(attr, dump_data_10, 1, &read_len);
    ASSERT_NULL(error);
    gpass_attribute_get(attr, &val);
    ASSERT_EQUAL_INT(10, val);
    ASSERT_EQUAL_INT(1, read_len);

    error = gpass_attribute_load(attr, dump_data_524, 2, &read_len);
    ASSERT_NULL(error);
    gpass_attribute_get(attr, &val);
    ASSERT_EQUAL_INT(524, val);
    ASSERT_EQUAL_INT(2, read_len);

    error = gpass_attribute_load(attr, dump_data_50040, 3, &read_len);
    ASSERT_NULL(error);
    gpass_attribute_get(attr, &val);
    ASSERT_EQUAL_INT(50040, val);
    ASSERT_EQUAL_INT(3, read_len);

    error = gpass_attribute_load(attr, dump_data_error1, 5, &read_len);
    ASSERT_NOT_NULL(error);
    g_error_free(error);

    error = gpass_attribute_load(attr, dump_data_error2, 2, &read_len);
    ASSERT_NOT_NULL(error);
    g_error_free(error);
}

START_TEST(test_load__integer)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_INTEGER, NULL);

    check_load_integer(attr);
    g_object_unref(attr);
}
END_TEST

START_TEST(test_load__time)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_TIME, NULL);

    check_load_integer(attr);
    g_object_unref(attr);
}
END_TEST

START_TEST(test_load__boolean)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_BOOLEAN, NULL);
    const guchar dump_data_true[] = { 0x01 };
    const guchar dump_data_false[] = { 0x00 };
    gint read_len;
    gboolean val;
    GError *error;
    
    error = gpass_attribute_load(attr, dump_data_true, 1, &read_len);
    ASSERT_NULL(error);
    gpass_attribute_get(attr, &val);
    ASSERT_TRUE(val);
    ASSERT_EQUAL_INT(1, read_len);

    error = gpass_attribute_load(attr, dump_data_false, 1, &read_len);
    ASSERT_NULL(error);
    gpass_attribute_get(attr, &val);
    ASSERT_FALSE(val);
    ASSERT_EQUAL_INT(1, read_len);
    
    g_object_unref(attr);
}
END_TEST

START_TEST(test_load__string)
{
    GPassAttribute *attr =
        g_object_new(GPASS_TYPE_ATTRIBUTE,
                     "type", GPASS_ATTRIBUTE_TYPE_STRING, NULL);
    const guchar dump_data_1[] = {
        0x06, 's', 't', 'r', 'i', 'n', 'g'
    };
    const guchar dump_data_2[] = {
        0x1a,
        'a', 'b', 'c', 'd', 'e', 'f', 'g',
        'h', 'i', 'j', 'k', 'l', 'm', 'n',
        'o', 'p', 'q', 'r', 's', 't', 'u',
        'v', 'w', 'x', 'y', 'z'
    };
    const guchar dump_data_error1[] = {
        0x1a, 's', 'h', 'o', 'r', 't'
    };
    const guchar dump_data_error2[] = {
        0x0b, 'T', 'h', 'e', ' ', 's', 't', 'r', 'i', 'n', 'g'
    };
    gint read_len;
    const gchar *val;
    GError *error;
    
    error = gpass_attribute_load(attr, dump_data_1, G_N_ELEMENTS(dump_data_1),
                                 &read_len);
    ASSERT_NULL(error);
    gpass_attribute_get(attr, &val);
    ASSERT_EQUAL_STRING("string", val);
    ASSERT_EQUAL_INT(7, read_len);

    error = gpass_attribute_load(attr, dump_data_2, G_N_ELEMENTS(dump_data_2),
                                 &read_len);
    ASSERT_NULL(error);
    gpass_attribute_get(attr, &val);
    ASSERT_EQUAL_STRING("abcdefghijklmnopqrstuvwxyz", val);
    ASSERT_EQUAL_INT(27, read_len);

    error = gpass_attribute_load(attr, dump_data_error1,
                                 G_N_ELEMENTS(dump_data_error1), &read_len);
    ASSERT_NOT_NULL(error);
    g_error_free(error);

    error = gpass_attribute_load(attr, dump_data_error2,
                                 G_N_ELEMENTS(dump_data_error2), &read_len);
    ASSERT_NOT_NULL(error);
    g_error_free(error);

    g_object_unref(attr);
}
END_TEST

/***********************************************************
 *
 * suite / main
 *
 ***********************************************************/
static Suite *
test_suite(void)
{
    Suite *s = suite_create("GPassAttribute");
    TCase *tc;
    
    tc = tcase_create("functions");
    suite_add_tcase(s, tc);
    tcase_add_checked_fixture(tc, setup, teardown);
    
    tcase_add_test(tc, test_new__integer);
    tcase_add_test(tc, test_set__integer);
    tcase_add_test(tc, test_get__integer);
    tcase_add_test(tc, test_new__boolean);
    tcase_add_test(tc, test_set__boolean);
    tcase_add_test(tc, test_get__boolean);
    tcase_add_test(tc, test_new__time);
    tcase_add_test(tc, test_set__time);
    tcase_add_test(tc, test_get__time);
    tcase_add_test(tc, test_new__string);
    tcase_add_test(tc, test_set__string);
    tcase_add_test(tc, test_get__string);
    tcase_add_test(tc, test_new__text);
    tcase_add_test(tc, test_set__text);
    tcase_add_test(tc, test_get__text);
    tcase_add_test(tc, test_new__password);
    tcase_add_test(tc, test_set__password);
    tcase_add_test(tc, test_get__password);
    tcase_add_test(tc, test_new__url);
    tcase_add_test(tc, test_set__url);
    tcase_add_test(tc, test_get__url);
    tcase_add_test(tc, test_new__entry_type);
    tcase_add_test(tc, test_set__entry_type);
    tcase_add_test(tc, test_get__entry_type);
    tcase_add_test(tc, test_dump__integer);
    tcase_add_test(tc, test_dump__time);
    tcase_add_test(tc, test_dump__boolean);
    tcase_add_test(tc, test_dump__string);
    tcase_add_test(tc, test_load__integer);
    tcase_add_test(tc, test_load__time);
    tcase_add_test(tc, test_load__boolean);
    tcase_add_test(tc, test_load__string);
    return s;
}

int
main(int argc, char *argv[])
{
    Suite *s;
    SRunner *sr;
    int nf;

    initialize();
    
    s = test_suite();
    sr = srunner_create(s);
    srunner_run_all(sr, CK_ENV);
    nf = srunner_ntests_failed(sr);
    srunner_free(sr);
    
    terminate();
    return (nf == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}
