/* vim:set noet ts=4: */
/** 
 * scim-python
 * 
 * Copyright (c) 2007-2008 Huang Peng <shawn.p.huang@gmail.com>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: $
 */

#include "scim-python-lookup-table.h"
#include "scim-python-attribute.h"
#if Py_UNICODE_SIZE == 2
#  include <glib.h>
#endif

struct PyLookupTableObject {
	PyObject_HEAD
	/* Type-specific fields go here. */
	PyLookupTable lookup_table;
};

PyLookupTable::PyLookupTable (PyLookupTableObject *self, unsigned int page_size)
: CommonLookupTable (page_size), self (self)
{
	Py_INCREF (self);

}

PyLookupTable::PyLookupTable (PyLookupTableObject *self, unsigned int page_size, const std::vector<WideString> &labels)
: CommonLookupTable (page_size, labels), self (self)
{
	Py_INCREF (self);
}

PyLookupTable::~PyLookupTable () 
{
	Py_DECREF (self);
}

PyObject *
PyLookupTable::py_get_candidate (PyLookupTableObject *self, PyObject *args)
{
	unsigned index = 0;
	
	if (!PyArg_ParseTuple (args, "I:get_candidate", &index))
		return NULL;

	WideString candidate = self->lookup_table.get_candidate (index);
#if Py_UNICODE_SIZE == 4
	return PyUnicode_FromUnicode ((Py_UNICODE *)candidate.c_str (), candidate.length ());
#else
	gunichar2 *utf16_str = g_ucs4_to_utf16 ((gunichar *)candidate.c_str (),
						candidate.length (), NULL, NULL, NULL);
	PyObject * result = PyUnicode_FromUnicode ((Py_UNICODE *)utf16_str, candidate.length ());
	g_free (utf16_str);
	return result;
#endif
}

PyObject *
PyLookupTable::py_get_attributes (PyLookupTableObject *self, PyObject *args)
{
#if 0
	PyObject *result;

	unsigned index = 0;
	
	if (!PyArg_ParseTuple (args, "I:get_attributes", &index))
		return NULL;
	
	AttributeList atts = self->lookup_table.get_attributes (index)

	result = PyTuple_New ();	
#endif
	Py_INCREF (Py_None);
	return Py_None;	
}

PyObject *
PyLookupTable::py_clear (PyLookupTableObject *self, PyObject *args)
{
	self->lookup_table.clear ();
	Py_INCREF (Py_None);
	return Py_None;
}

PyObject *
PyLookupTable::py_page_up (PyLookupTableObject *self, PyObject *args)
{
	PyObject *result;

	if (self->lookup_table.page_up ())
		result = Py_True;
	else
		result = Py_False;

	Py_INCREF (result);
	return result;
}

PyObject *
PyLookupTable::py_page_down (PyLookupTableObject *self, PyObject *args)
{
	PyObject *result;

	if (self->lookup_table.page_down ())
		result = Py_True;
	else
		result = Py_False;

	Py_INCREF (result);
	return result;
}

PyObject *
PyLookupTable::py_append_candidate (PyLookupTableObject *self, PyObject *args)
{
	Py_UNICODE *candidate = NULL;
	PyObject *pAttrs = NULL;
	PyObject *result = Py_False;

#if Py_UNICODE_SIZE == 4
	if (!PyArg_ParseTuple (args, "u|O:append_candidate", &candidate, &pAttrs))
		return NULL;
	
	if (self->lookup_table.append_candidate (WideString ((wchar_t *)candidate),
								Attributes_FromTupleOrList (pAttrs)))
	{
		result = Py_True;
	}
#else
	int size = 0;
	gunichar *unistr = NULL;

	if (!PyArg_ParseTuple (args, "u#|O:append_candidate", &candidate, &size, &pAttrs))
		return NULL;
	unistr = g_utf16_to_ucs4 (candidate, size, NULL, NULL, NULL);

	if (self->lookup_table.append_candidate (WideString ((wchar_t *)unistr),
								Attributes_FromTupleOrList (pAttrs)))
	{
		result = Py_True;
	}
	g_free (unistr);
#endif
	
	Py_INCREF (result);
	return result;
}

PyObject *
PyLookupTable::py_get_current_page_start (PyLookupTableObject *self, PyObject *args)
{
	long index = self->lookup_table.get_current_page_start ();

	return PyInt_FromLong (index);
}

PyObject *
PyLookupTable::py_cursor_up (PyLookupTableObject *self, PyObject *args)
{
	PyObject * result;

	if (self->lookup_table.cursor_up ()) {
		result = Py_True;
	}
	else {
		result = Py_False;
	}

	Py_INCREF (result);
	return result;
}

PyObject *
PyLookupTable::py_cursor_down (PyLookupTableObject *self, PyObject *args)
{
	PyObject * result;

	if (self->lookup_table.cursor_down ()) {
		result = Py_True;
	}
	else {
		result = Py_False;
	}

	Py_INCREF (result);
	return result;
}

PyObject *
PyLookupTable::py_show_cursor (PyLookupTableObject *self, PyObject *args)
{
	unsigned int show = 1;

	if (!PyArg_ParseTuple (args, "|I:show_cursor", &show))
		return NULL;
	
	self->lookup_table.show_cursor (show);

	Py_INCREF (Py_None);
	return Py_None;
}

PyObject *
PyLookupTable::py_is_cursor_visible (PyLookupTableObject *self, PyObject *args)
{
	PyObject * result;
	if (self->lookup_table.is_cursor_visible ()) {
		result = Py_True;
	}
	else {
		result = Py_False;
	}

	Py_INCREF (result);
	return result;
}

PyObject *
PyLookupTable::py_set_cursor_pos (PyLookupTableObject *self, PyObject *args)
{
	unsigned int pos;

	if (!PyArg_ParseTuple (args, "I:set_cursor_pos", &pos))
		return NULL;
	
	self->lookup_table.set_cursor_pos (pos);

	Py_INCREF (Py_None);
	return Py_None;
}

PyObject *
PyLookupTable::py_set_cursor_pos_in_current_page (PyLookupTableObject *self, PyObject *args)
{
	unsigned int pos;

	if (!PyArg_ParseTuple (args, "I:set_cursor_pos_in_current_pos", &pos))
		return NULL;
	
	self->lookup_table.set_cursor_pos_in_current_page (pos);

	Py_INCREF (Py_None);
	return Py_None;
}

PyObject *
PyLookupTable::py_get_cursor_pos (PyLookupTableObject *self, PyObject *args)
{
	int pos;

	pos = self->lookup_table.get_cursor_pos ();

	return PyInt_FromLong (pos);
}

PyObject *
PyLookupTable::py_get_cursor_pos_in_current_page (PyLookupTableObject *self, PyObject *args)
{
	int pos;

	pos = self->lookup_table.get_cursor_pos_in_current_page ();

	return PyInt_FromLong (pos);
}

PyObject *
PyLookupTable::py_set_page_size (PyLookupTableObject *self, PyObject *args)
{
	unsigned int page_size;

	if (!PyArg_ParseTuple (args, "I:set_page_size", &page_size))
		return NULL;
	
	self->lookup_table.set_page_size (page_size);

	Py_INCREF (Py_None);
	return Py_None;
}

PyMethodDef
PyLookupTable::py_methods[] = { 
		{	"clear", (PyCFunction)PyLookupTable::py_clear, METH_NOARGS,
			"clear candidates"
		},
		{	"page_up", (PyCFunction)PyLookupTable::py_page_up, METH_NOARGS,
			"page up"
		},
		{	"page_down", (PyCFunction)PyLookupTable::py_page_down, METH_NOARGS,
			"page down"
		},
		{	"get_candidate", (PyCFunction)PyLookupTable::py_get_candidate, METH_VARARGS,
			"get candidate"
		},
		{	"get_attributes", (PyCFunction)PyLookupTable::py_get_attributes, METH_VARARGS,
			"get attributes"
		},
		{	"append_candidate", (PyCFunction)PyLookupTable::py_append_candidate, METH_VARARGS,
			"append candidate"
		},
		{	"get_current_page_start", (PyCFunction)PyLookupTable::py_get_current_page_start, METH_NOARGS,
			"return current page start index"
		},
		{	"cursor_up", (PyCFunction)PyLookupTable::py_cursor_up, METH_NOARGS,
			"cursor up"
		},
		{	"cursor_down", (PyCFunction)PyLookupTable::py_cursor_down, METH_NOARGS,
			"cursor down"
		},
		{	"show_cursor", (PyCFunction)PyLookupTable::py_show_cursor, METH_VARARGS,
			"show or hide cursor"
		},
		{	"is_cursor_visible", (PyCFunction)PyLookupTable::py_is_cursor_visible, METH_NOARGS,
			"is cursor visible"
		},
		{	"set_cursor_pos", (PyCFunction)PyLookupTable::py_set_cursor_pos, METH_VARARGS,
			"set cursor position"
		},
		{	"set_cursor_pos_in_current_page", (PyCFunction)PyLookupTable::py_set_cursor_pos_in_current_page, METH_VARARGS,
			"set cursor position in the current page"
		},
		{	"get_cursor_pos", (PyCFunction)PyLookupTable::py_get_cursor_pos, METH_VARARGS,
			"get cursor position"
		},
		{	"get_cursor_pos_in_current_page", (PyCFunction)PyLookupTable::py_get_cursor_pos_in_current_page, METH_NOARGS,
			"get cursor position in the current page"
		},
		{	"set_page_size", (PyCFunction)PyLookupTable::py_set_page_size, METH_VARARGS,
			"set page size"
		},
		{ NULL }
};

PyObject *
PyLookupTable::py_new (PyTypeObject *type, PyObject *args, PyObject *kwds)
{
	PyLookupTable *self;
	
	self = (PyLookupTable *)type->tp_alloc (type, 0);
	return (PyObject *)self;
}

int
PyLookupTable::py_init (PyLookupTableObject *self, PyObject *args, PyObject *kwds)
{
	unsigned int page_size = 10;

	if (!PyArg_ParseTuple (args, "|I:__init__", &page_size))
		return -1;

	new (&self->lookup_table) PyLookupTable (self, page_size);
		
	return 0;
}

void
PyLookupTable::py_dealloc (PyLookupTableObject *self)
{
	self->lookup_table.~LookupTable ();
	((PyObject *)self)->ob_type->tp_free (self);
}

const PyLookupTable &
PyLookupTable::from_pyobject (PyObject *object)
{
    return ((PyLookupTableObject *)object)->lookup_table;
}

PyTypeObject PyLookupTableType = {
	PyObject_HEAD_INIT (NULL)
	0,						 			/*ob_size*/
	"scim.LookupTable", 				/*tp_name*/
	sizeof (PyLookupTableObject), 		/*tp_basicsize*/
	0,						 			/*tp_itemsize*/
	(destructor)PyLookupTable::py_dealloc,	/*tp_dealloc*/
	0,			  						/*tp_print*/
	0,						 			/*tp_getattr*/
	0,									/*tp_setattr*/
	0,									/*tp_compare*/
	0,			  						/*tp_repr*/
	0,									/*tp_as_number*/
	0,			  						/*tp_as_sequence*/
	0,									/*tp_as_mapping*/
	0,			  						/*tp_hash */
	0,									/*tp_call*/
	0,		  							/*tp_str*/
	0,					   				/*tp_getattro*/
	0,									/*tp_setattro*/
	0,					 				/*tp_as_buffer*/
	Py_TPFLAGS_DEFAULT | 
	Py_TPFLAGS_BASETYPE,				/*tp_flags*/
	"PyLookupTable objects",		 	/* tp_doc */
	0,					   				/* tp_traverse */
	0,					   				/* tp_clear */
	0,					   				/* tp_richcompare */
	0,					   				/* tp_weaklistoffset */
	0,					   				/* tp_iter */
	0,					   				/* tp_iternext */
	PyLookupTable::py_methods,			/* tp_methods */
	0,			 						/* tp_members */
	0,						 			/* tp_getset */
	0,						 			/* tp_base */
	0,						 			/* tp_dict */
	0,						 			/* tp_descr_get */
	0,						 			/* tp_descr_set */
	0,									/* tp_dictoffset */
	(initproc)PyLookupTable::py_init,	/* tp_init */
	0,						 			/* tp_alloc */
	PyLookupTable::py_new,				/* tp_new */	
	PyObject_Del,				 		/* tp_free */	
};

void init_lookup_table (PyObject *module)
{
	if (PyType_Ready (&PyLookupTableType) < 0)
		return;
	
	Py_INCREF (&PyLookupTableType);
	PyModule_AddObject (module, "LookupTable", (PyObject *)&PyLookupTableType);	
}
