#!/usr/bin/awish

##############################################################
#
# apol: SELinux Policy Analysis Tools
#
# Copyright (C) 2002-2007 Tresys Technology, LLC
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#
# TCL/TK GUI for SE Linux policy analysis
# Requires tcl and tk 8.4+, with BWidget
#
# Question/comments to: selinux@tresys.com
#
# This tools is designed to analysis selinux policies.
#
# See the assoicated help file for more information.
#
##############################################################


proc tcl_config_init {} {
    set ApolTop::gui_ver 3.2
    set ApolTop::apol_install_dir \{/usr/local/share/setools-3.2\}
}
namespace eval Apol_Types {
    variable typelist {}
    variable attriblist {}
    variable opts
    variable widgets
}
proc Apol_Types::open { } {
    variable typelist {}
    variable attriblist {}
    foreach type_datum [apol_GetTypes {} 0] {
        lappend typelist [lindex $type_datum 0]
    }
    set typelist [lsort $typelist]
    foreach attrib_datum [apol_GetAttribs {} 0] {
        lappend attriblist [lindex $attrib_datum 0]
    }
    set attriblist [lsort $attriblist]
}
proc Apol_Types::close { } {
    variable widgets
    initializeVars
    set Apol_Types::typelist {}
    set Apol_Types::attriblist {}
    Apol_Widget::clearSearchResults $widgets(results)
}
proc Apol_Types::initializeVars {} {
    variable opts
    array set opts {
        types 1    types:show_attribs 1  types:show_aliases 1
        attribs 0  attribs:show_types 1  attribs:show_attribs 1
    }
}
proc Apol_Types::set_Focus_to_Text {} {
    focus $Apol_Types::widgets(results)
}
proc Apol_Types::popupTypeInfo {which ta} {
    set info_fc ""
    set index_file_loaded 0
    if {$which == "type"} {
        set info_ta [renderType [lindex [apol_GetTypes $ta] 0] 1 1]
    } else {
        set info_ta [renderAttrib [lindex [apol_GetAttribs $ta 0] 0] 1 0]
    }
    if {$ApolTop::libsefs == 1 && [Apol_File_Contexts::is_db_loaded]} {
        set info_fc [Apol_File_Contexts::get_fc_files_for_ta $which $ta]
        set index_file_loaded 1
    }
    set w .ta_infobox
    destroy $w
    toplevel $w
    wm title $w $ta
    set top_f [frame $w.top_f]
    set bot_f [frame $w.bot_f]
    set notebook [NoteBook $top_f.nb]
    set ta_info_tab [$notebook insert end ta_info_tab]
    if {$ApolTop::libsefs == 1} {
        set fc_info_tab [$notebook insert end fc_info_tab -text "Files"]
    }
    if {$which == "type"} {
        $notebook itemconfigure ta_info_tab -text "Attributes"
    } else {
        $notebook itemconfigure ta_info_tab -text "Types"
    }
    set s_ta [ScrolledWindow [$notebook getframe ta_info_tab].s_ta  -scrollbar both -auto both]
    set f_ta [text [$s_ta getframe].f -font {helvetica 10} -wrap none -width 35 -height 10 -bg white]
    $s_ta setwidget $f_ta
    if {$ApolTop::libsefs == 1} {
        if {$which != "type"} {
            set lbl [Label [$notebook getframe fc_info_tab].lbl \
                         -text "Files labeled with types that are members of this attribute:" \
                         -justify left]
        }
        set s_fc [ScrolledWindow [$notebook getframe fc_info_tab].s_fc  -scrollbar both -auto both]
        set f_fc [text [$s_fc getframe].f -font {helvetica 10} -wrap none -width 35 -height 10 -bg white]
        $s_fc setwidget $f_fc
    }
    set b_close [button $bot_f.b_close -text "Close" -command [list destroy $w]]
    pack $top_f -side top -anchor nw -fill both -expand yes
    pack $bot_f -side bottom -anchor sw -fill x
    pack $b_close -side bottom -anchor center -expand 0 -pady 5
    pack $s_ta -fill both -expand yes
    pack $notebook -fill both -expand yes -padx 4 -pady 4
    $notebook raise [$notebook page 0]
    $f_ta insert 0.0 $info_ta
    $f_ta configure -state disabled
    if {$ApolTop::libsefs == 1} {
        if {$which != "type"} {
            pack $lbl -side top -side top -anchor nw
        }
        pack $s_fc -fill both -expand yes -side top
        if {$index_file_loaded} {
            if {$info_fc != ""} {
                set num 0
                foreach item $info_fc {
                    foreach {ctxt class path} $item {}
                    $f_fc insert end "$ctxt\t     $class\t     $path\n"
                    incr num
                }
                $f_fc insert 1.0 "Number of files: $num\n\n"
            } else {
                $f_fc insert end "No files found."
            }
        } else {
            $f_fc insert 0.0 "No index file is loaded. If you would like to load an index file, go to the File Context tab."
        }
        $f_fc configure -state disabled
    }
    wm geometry $w 400x400
    wm deiconify $w
    wm protocol $w WM_DELETE_WINDOW [list destroy $w]
    raise $w
}
proc Apol_Types::search { str case_Insensitive regExpr srch_Direction } {
    variable widgets
    ApolTop::textSearch $widgets(results).tb $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_Types::searchTypes {} {
    variable widgets
    variable opts
    Apol_Widget::clearSearchResults $widgets(results)
    if {![ApolTop::is_policy_open]} {
        tk_messageBox -icon error -type ok -title "Error" -message "No current policy file is opened!"
        return
    }
    if {$opts(types) == 0 && $opts(attribs) == 0} {
        tk_messageBox -icon error -type ok -title "Error" -message "No search options provided!"
        return
    }
    set use_regexp [Apol_Widget::getRegexpEntryState $widgets(regexp)]
    set regexp [Apol_Widget::getRegexpEntryValue $widgets(regexp)]
    if {$use_regexp} {
        if {$regexp == {}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No regular expression provided."
            return
        }
    } else {
        set regexp {}
    }
    set results {}
    if {$opts(types)} {
        if {[catch {apol_GetTypes $regexp $use_regexp} types_data]} {
            tk_messageBox -icon error -type ok -title "Error" -message "Error obtaining types list:\n$types_data"
            return
        }
        append results "TYPES ([llength $types_data]):\n\n"
        foreach t [lsort -index 0 $types_data] {
            append results "[renderType $t $opts(types:show_attribs) $opts(types:show_aliases)]\n"
        }
    }
    if {$opts(attribs)} {
        if {[catch {apol_GetAttribs $regexp $use_regexp} attribs_data]} {
            tk_messageBox -icon error -type ok -title "Error" -message "Error obtaining types list:\n$attribs_data"
            return
        }
        if {$opts(types)} {
            append results "\n\n"
        }
        append results "ATTRIBUTES ([llength $attribs_data]):\n\n"
        foreach a [lsort -index 0 $attribs_data] {
            append results "[renderAttrib $a $opts(attribs:show_types) $opts(attribs:show_attribs)]\n"
        }
    }
    Apol_Widget::appendSearchResultText $widgets(results) $results
}
proc Apol_Types::renderType {type_datum show_attribs show_aliases} {
    set text ""
    foreach {type attribs aliases} $type_datum {break}
    append text "$type"
    if {$show_aliases && [llength $aliases] > 0} {
        append text " alias [list $aliases]"
    }
    if {$show_attribs} {
        append text " ([llength $attribs] attribute"
        if {[llength $attribs] != 1} {
            append text s
        }
        append text ")\n"
        foreach a [lsort $attribs] {
            append text "    $a\n"
        }
    }
    return $text
}
proc Apol_Types::renderAttrib {attrib_datum show_types show_attribs} {
    set text ""
    foreach {attrib types} $attrib_datum {break}
    append text "$attrib"
    if {$show_types} {
        append text " ([llength $types] type"
        if {[llength $types] != 1} {
            append text s
        }
        append text ")\n"
        foreach type [lsort $types] {
            append text "    $type"
            if {$show_attribs} {
                set a [lsort [lindex [apol_GetTypes $type] 0 1]]
                set idx [lsearch -sorted -exact $a $attrib]
                append text "  { [lreplace $a $idx $idx] }"
            }
            append text "\n"
        }
    }
    return $text
}
proc Apol_Types::goto_line { line_num } {
    variable widgets
    Apol_Widget::gotoLineSearchResults $widgets(results) $line_num
}
proc Apol_Types::create {nb} {
    variable opts
    variable widgets
    initializeVars
    set frame [$nb insert end $ApolTop::types_tab -text "Types"]
    set pw1   [PanedWindow $frame.pw -side top]
    set left_pane   [$pw1 add -weight 0]
    set center_pane [$pw1 add -weight 1]
    set tpane [frame $left_pane.t]
    set apane [frame $left_pane.a]
    set tbox [TitleFrame $tpane.tbox -text "Types"]
    set abox [TitleFrame $apane.abox -text "Attributes"]
    set obox [TitleFrame $center_pane.obox -text "Search Options"]
    set rbox [TitleFrame $center_pane.rbox -text "Search Results"]
    pack $obox -side top -expand 0 -fill both -padx 2
    pack $rbox -expand yes -fill both -padx 2
    pack $tbox -fill both -expand yes
    pack $abox -fill both -expand yes
    pack $pw1 -fill both -expand yes
    pack $tpane -fill both -expand 1
    pack $apane -fill both -expand 1
    set tlistbox [Apol_Widget::makeScrolledListbox [$tbox getframe].types \
                      -height 10 -width 20 -listvar Apol_Types::typelist]
    Apol_Widget::setListboxCallbacks $tlistbox \
        {{"Show Type Info" {Apol_Types::popupTypeInfo type}}}
    pack $tlistbox -expand 1 -fill both
    set alistbox [Apol_Widget::makeScrolledListbox [$abox getframe].attribs \
                      -height 5 -width 20 -listvar Apol_Types::attriblist]
    Apol_Widget::setListboxCallbacks $alistbox {{"Show Attribute Info" {Apol_Types::popupTypeInfo attrib}}}
    pack $alistbox -expand 1 -fill both
    set ofm [$obox getframe]
    set fm_types_select [frame $ofm.to]
    set fm_attribs_select [frame $ofm.ao]
    pack $fm_types_select $fm_attribs_select -side left -padx 4 -pady 2 -anchor nw
    set types_select [checkbutton $fm_types_select.type -text "Show types" -variable Apol_Types::opts(types)]
    set typeattribs [checkbutton $fm_types_select.typeattribs -text "Include attributes" \
	-variable Apol_Types::opts(types:show_attribs)]
    pack $types_select -anchor w
    pack $typeattribs -anchor w -padx 8
    trace add variable Apol_Types::opts(types) write \
        [list Apol_Types::toggleCheckbuttons $typeattribs]
    set attribs_select [checkbutton $fm_attribs_select.type -text "Show attributes" \
	-variable Apol_Types::opts(attribs)]
    set a_types [checkbutton $fm_attribs_select.types -text "Include types" \
	-variable Apol_Types::opts(attribs:show_types) -state disabled]
    set a_typeattribs [checkbutton $fm_attribs_select.typeattribs -text "Include types' attributes" \
	-variable Apol_Types::opts(attribs:show_attribs) -state disabled]
    pack $attribs_select -anchor w
    pack $a_types $a_typeattribs -anchor w -padx 8
    trace add variable Apol_Types::opts(attribs) write \
        [list Apol_Types::toggleCheckbuttons [list $a_typeattribs $a_types]]
    set widgets(regexp) [Apol_Widget::makeRegexpEntry $ofm.regexpf]
    Apol_Widget::setRegexpEntryState $widgets(regexp) 1
    pack $widgets(regexp) -side left -padx 4 -pady 2 -anchor nw
    set ok [button $ofm.ok -text OK -width 6 -command Apol_Types::searchTypes]
    pack $ok -side right -padx 5 -pady 5 -anchor ne
    set widgets(results) [Apol_Widget::makeSearchResults [$rbox getframe].results]
    pack $widgets(results) -expand yes -fill both
    return $frame
}
proc Apol_Types::toggleCheckbuttons {w name1 name2 op} {
    variable opts
    variable widgets
    if {$opts($name2)} {
        foreach x $w {
            $x configure -state normal
        }
    } else {
        foreach x $w {
            $x configure -state disabled
        }
    }
    if {!$opts(types) && !$opts(attribs)} {
        Apol_Widget::setRegexpEntryState $widgets(regexp) 0
    } else {
        Apol_Widget::setRegexpEntryState $widgets(regexp) 1
    }
}
namespace eval Apol_TE {
    variable vals
    variable widgets
    variable tabs
    variable enabled
}
proc Apol_TE::goto_line { line_num } {
    variable widgets
    variable tabs
    if {[$widgets(results) pages] != {}} {
        set raisedPage [$widgets(results) raise]
        if {$raisedPage != {}} {
            Apol_Widget::gotoLineSearchResults $tabs($raisedPage) $line_num
	}
    }
}
proc Apol_TE::search { str case_Insensitive regExpr srch_Direction } {
    variable widgets
    variable tabs
    if {[$widgets(results) pages] != {}} {
        set raisedPage [$widgets(results) raise]
        if {$raisedPage != {}} {
            ApolTop::textSearch $tabs($raisedPage).tb $str $case_Insensitive $regExpr $srch_Direction
	}
    }
}
proc Apol_TE::set_Focus_to_Text { tab } {
    variable widgets
    variable tabs
    if {[$widgets(results) pages] != {}} {
        set raisedPage [$widgets(results) raise]
        if {$raisedPage != {}} {
            focus $tabs($raisedPage)
	}
    }
}
proc Apol_TE::get_results_raised_tab {} {
    variable widgets
    $widgets(results) raise
}
proc Apol_TE::open { } {
    reinitialize_default_search_options
    reinitialize_search_widgets
    reinitialize_tabs
    variable vals
    variable enabled
    set vals(cp:classes) $Apol_Class_Perms::class_list
    set enabled(cp:classes) 1
    set enabled(cp:perms) 1
}
proc Apol_TE::close { } {
    reinitialize_tabs
    reinitialize_search_widgets
    reinitialize_default_search_options
    set enabled(cp:perms) 1
}
proc Apol_TE::reinitialize_default_search_options { } {
    variable vals
    array set vals {
        rs:allow 1       rs:type_transition 1
        rs:neverallow 1  rs:type_member 1
        rs:auditallow 1  rs:type_change 1
        rs:dontaudit 1
        oo:enabled 0
        oo:regexp 0
        ta:use_source 0
        ta:source_indirect 1
        ta:source_which source
        ta:source_sym,types 1 ta:source_sym,attribs 0
        ta:source_sym {}
        ta:use_target 0
        ta:target_indirect 1
        ta:target_sym,types 1 ta:target_sym,attribs 0
        ta:target_sym {}
        ta:use_default 0
        ta:default_sym,types 1 ta:default_sym,attribs 0
        ta:default_sym {}
        cp:classes {}
        cp:classes_selected {}
        cp:perms {}
        cp:perms_selected {}
        cp:perms_toshow all
    }
    variable enabled
    array set enabled {
        ta:use_source 1
        ta:use_target 1
        ta:use_default 1
        cp:classes 0
        cp:perms 0
    }
}
proc Apol_TE::reinitialize_tabs {} {
    variable widgets
    variable tabs
    array set tabs {
        next_result_id 1
    }
    foreach p [$widgets(results) pages 0 end] {
        delete_results $p
    }
}
proc Apol_TE::reinitialize_search_widgets {} {
    variable widgets
    $widgets(search_opts) raise typeattrib
    $widgets(cp:classes) selection clear 0 end
    $widgets(cp:perms) selection clear 0 end
}
proc Apol_TE::create {nb} {
    variable vals
    variable widgets
    reinitialize_default_search_options
    set frame [$nb insert end $ApolTop::terules_tab -text "TE Rules"]
    set pw [PanedWindow $frame.pw -side left -weights extra]
    set topf [$pw add -weight 0]
    set bottomf [$pw add -weight 1]
    pack $pw -expand 1 -fill both
    set top_leftf [frame $topf.tl]
    set widgets(search_opts) [NoteBook $topf.nb]
    set abox [frame $topf.abox]
    pack $top_leftf -side left -expand 0 -fill y
    pack $widgets(search_opts) -side left -expand 1 -fill both -padx 10
    pack $abox -side right -fill y -padx 5
    set rsbox [TitleFrame $top_leftf.rsbox -text "Rule Selection"]
    set oobox [TitleFrame $top_leftf.oobox -text "Search Options"]
    set rbox [TitleFrame $bottomf.rbox -text "Type Enforcement Rules Display"]
    pack $rsbox -side top -expand 0 -fill both
    pack $oobox -side top -expand 1 -fill both -pady 2
    pack $rbox -expand yes -fill both -padx 2
    set fm_rules [$rsbox getframe]
    set allow [checkbutton $fm_rules.allow -text "allow" \
                   -variable Apol_TE::vals(rs:allow)]
    set neverallow [checkbutton $fm_rules.neverallow -text "neverallow" \
                        -variable Apol_TE::vals(rs:neverallow)]
    set auditallow [checkbutton $fm_rules.auditallow -text "auditallow" \
                        -variable Apol_TE::vals(rs:auditallow)]
    set dontaudit [checkbutton $fm_rules.dontaudit -text "dontaudit" \
                       -variable Apol_TE::vals(rs:dontaudit)]
    set type_transition [checkbutton $fm_rules.type_transition -text "type_trans" \
                             -variable Apol_TE::vals(rs:type_transition)]
    set type_member [checkbutton $fm_rules.type_member -text "type_member" \
                         -variable Apol_TE::vals(rs:type_member)]
    set type_change [checkbutton $fm_rules.type_change -text "type_change" \
                         -variable Apol_TE::vals(rs:type_change)]
    grid $allow $type_transition -sticky w -padx 2
    grid $neverallow $type_member -sticky w -padx 2
    grid $auditallow $type_change -sticky w -padx 2
    grid $dontaudit x -sticky w -padx 2
    foreach x {allow neverallow auditallow dontaudit type_transition type_member type_change} {
        trace add variable Apol_TE::vals(rs:$x) write \
            [list Apol_TE::toggle_rule_selection]
    }
    set fm_options [$oobox getframe]
    set enabled [checkbutton $fm_options.enabled -text "Search only enabled rules" \
                     -variable Apol_TE::vals(oo:enabled)]
    set regexp [checkbutton $fm_options.regex -text "Search using regular expression" \
                    -variable Apol_TE::vals(oo:regexp)]
    pack $enabled $regexp -expand 0 -fill none -anchor w
    createTypesAttribsTab
    createClassesPermsTab
    set widgets(new) [button $abox.new -text "New Search" -width 12 \
                       -command [list Apol_TE::search_terules new]]
    set widgets(update) [button $abox.update -text "Update Search" -width 12 -state disabled \
                             -command [list Apol_TE::search_terules update]]
    set widgets(reset) [button $abox.reset -text "Reset Criteria" -width 12 \
                            -command Apol_TE::reset]
    pack $widgets(new) $widgets(update) $widgets(reset) \
        -side top -pady 5 -padx 5 -anchor ne
    $widgets(search_opts) compute_size
    set popupTab_Menu [menu .popup_terules -tearoff 0]
    set tab_menu_callbacks \
        [list {"Close Tab" Apol_TE::delete_results} \
             {"Rename Tab" Apol_TE::display_rename_tab_dialog}]
    set widgets(results) [NoteBook [$rbox getframe].results]
    $widgets(results) bindtabs <Button-1> Apol_TE::switch_to_tab
    $widgets(results) bindtabs <Button-3> \
        [list ApolTop::popup_Tab_Menu \
             %W %x %y $popupTab_Menu $tab_menu_callbacks]
    set close [button [$rbox getframe].close -text "Close Tab" \
                   -command Apol_TE::delete_current_results]
    pack $widgets(results) -expand 1 -fill both -padx 4
    pack $close -expand 0 -fill x -padx 4 -pady 2
    reinitialize_tabs
    return $frame
}
proc Apol_TE::createTypesAttribsTab {} {
    variable vals
    variable widgets
    variable enabled
    set ta_tab [$widgets(search_opts) insert end typeattrib -text "Types/Attributes"]
    set fm_source [frame $ta_tab.source]
    set fm_target [frame $ta_tab.target]
    set fm_default [frame $ta_tab.default]
    grid $fm_source $fm_target $fm_default -padx 4 -sticky ewns
    foreach i {0 1 2} {
        grid columnconfigure $ta_tab $i -weight 1 -uniform 1
    }
    grid rowconfigure $ta_tab 0 -weight 1
    create_ta_box source $fm_source "Source type/attribute" 1 1 1
    create_ta_box target $fm_target "Target type/attribute" 1 0 1
    create_ta_box default $fm_default "Default type" 0 0 0
    $widgets(search_opts) raise typeattrib
}
proc Apol_TE::create_ta_box {prefix f title has_indirect has_which has_attribs} {
    variable vals
    variable widgets
    set widgets(ta:use_${prefix}) [checkbutton $f.use -text $title \
                                       -variable Apol_TE::vals(ta:use_${prefix})]
    pack $widgets(ta:use_${prefix}) -side top -anchor w
    trace add variable Apol_TE::vals(ta:use_${prefix}) write \
        [list Apol_TE::toggle_ta_box $prefix]
    set w {}
    if {$has_attribs} {
        set helptext "Type or select a type or attribute"
    } else {
        set helptext "Type or select a type"
    }
    set widgets(ta:${prefix}_sym) [ComboBox $f.sym \
                                       -state disabled -entrybg $ApolTop::default_bg_color \
                                       -textvariable Apol_TE::vals(ta:${prefix}_sym) \
                                       -helptext $helptext -autopost 1]
    pack $widgets(ta:${prefix}_sym) -expand 0 -fill x -padx 8
    lappend w $widgets(ta:${prefix}_sym)
    if {$has_attribs} {
        set ta_frame [frame $f.ta]
        pack $ta_frame -expand 0 -anchor center -pady 2
        set types [checkbutton $ta_frame.types -text "Types" -state disabled \
                       -variable Apol_TE::vals(ta:${prefix}_sym,types)]
        set attribs [checkbutton $ta_frame.attribs -text "Attribs" -state disabled \
                         -variable Apol_TE::vals(ta:${prefix}_sym,attribs)]
        $types configure -command [list Apol_TE::toggle_ta_pushed $prefix $types]
        $attribs configure -command [list Apol_TE::toggle_ta_pushed $prefix $attribs]
        trace add variable Apol_TE::vals(ta:${prefix}_sym,attribs) write \
            [list Apol_TE::toggle_ta_sym $prefix]
        pack $types $attribs -side left -padx 2
        lappend w $types $attribs
    }
    if {$has_indirect} {
        set indirect [checkbutton $f.indirect -text "Only direct matches" \
                          -state disabled -variable Apol_TE::vals(ta:${prefix}_indirect) \
                          -onvalue 0 -offvalue 1]
        pack $indirect -anchor w -padx 8
        lappend w $indirect
    }
    if {$has_which} {
        set which_fm [frame $f.which]
        set which_source [radiobutton $which_fm.source \
                              -text "As source" -state disabled \
                              -variable Apol_TE::vals(ta:${prefix}_which) \
                              -value source]
        set which_any [radiobutton $which_fm.any \
                           -text "Any" -state disabled \
                           -variable Apol_TE::vals(ta:${prefix}_which) \
                           -value either]
        trace add variable Apol_TE::vals(ta:${prefix}_which) write \
            [list Apol_TE::toggle_which]
        pack $which_source $which_any -side left -padx 2
        pack $which_fm -anchor w -padx 6
        lappend w $which_source $which_any
    }
    trace add variable Apol_TE::vals(ta:${prefix}_sym,types) write \
            [list Apol_TE::toggle_ta_sym $prefix]
    set widgets(ta:${prefix}_widgets) $w
    trace add variable Apol_TE::enabled(ta:use_${prefix}) write \
        [list Apol_TE::toggle_enable_ta ${prefix}]
}
proc Apol_TE::toggle_rule_selection {name1 name2 op} {
    maybe_enable_default_type
    maybe_enable_perms
}
proc Apol_TE::toggle_ta_box {col name1 name2 op} {
    variable vals
    variable enabled
    if {$col == "source"} {
        maybe_enable_target_type
        maybe_enable_default_type
    }
    set enabled(ta:use_${col}) $enabled(ta:use_${col})
}
proc Apol_TE::toggle_which {name1 name2 op} {
    maybe_enable_target_type
    maybe_enable_default_type
}
proc Apol_TE::maybe_enable_target_type {} {
    variable vals
    variable enabled
    set any_set 0
    if {$enabled(ta:use_source) && $vals(ta:use_source) && $vals(ta:source_which) == "either"} {
        set any_set 1
    }
    if {!$any_set} {
        set enabled(ta:use_target) 1
    } else {
        set enabled(ta:use_target) 0
    }
}
proc Apol_TE::maybe_enable_default_type {} {
    variable vals
    variable enabled
    set typerule_set 0
    set any_set 0
    foreach x {type_transition type_member type_change} {
        if {$vals(rs:$x)} {
            set typerule_set 1
            break
        }
    }
    if {$enabled(ta:use_source) && $vals(ta:use_source) && $vals(ta:source_which) == "either"} {
        set any_set 1
    }
    if {$typerule_set && !$any_set} {
        set enabled(ta:use_default) 1
    } else {
        set enabled(ta:use_default) 0
    }
}
proc Apol_TE::toggle_enable_ta {col name1 name2 op} {
    variable vals
    variable widgets
    variable enabled
    if {$enabled(ta:use_${col})} {
        $widgets(ta:use_${col}) configure -state normal
    } else {
        $widgets(ta:use_${col}) configure -state disabled
    }
    if {$enabled(ta:use_${col}) && $vals(ta:use_${col})} {
        foreach w $widgets(ta:${col}_widgets) {
            $w configure -state normal
        }
        $widgets(ta:${col}_sym) configure -entrybg white
    } else {
        foreach w $widgets(ta:${col}_widgets) {
            $w configure -state disabled
        }
        $widgets(ta:${col}_sym) configure -entrybg $ApolTop::default_bg_color
    }
    if {($enabled(ta:use_source) && $vals(ta:use_source)) || \
            ($enabled(ta:use_target) && $vals(ta:use_target)) || \
            ($enabled(ta:use_default) && $vals(ta:use_default))} {
        $widgets(search_opts) itemconfigure typeattrib -text "Types/Attributes *"
    } else {
        $widgets(search_opts) itemconfigure typeattrib -text "Types/Attributes"
    }
}
proc Apol_TE::toggle_ta_sym {col name1 name2 op} {
    variable vals
    variable widgets
    if {!$vals(ta:${col}_sym,types) && !$vals(ta:${col}_sym,attribs)} {
        return
    }
    if {$vals(ta:${col}_sym,types) && $vals(ta:${col}_sym,attribs)} {
        set items [lsort [concat $Apol_Types::typelist $Apol_Types::attriblist]]
    } elseif {$vals(ta:${col}_sym,types)} {
        set items $Apol_Types::typelist
    } else {
        set items $Apol_Types::attriblist
    }
    $widgets(ta:${col}_sym) configure -values $items
}
proc Apol_TE::toggle_ta_pushed {col cb} {
    variable vals
    if {!$vals(ta:${col}_sym,types) && !$vals(ta:${col}_sym,attribs)} {
        $cb select
    }
}
proc Apol_TE::createClassesPermsTab {} {
    variable vals
    variable widgets
    variable enabled
    set objects_tab [$widgets(search_opts) insert end classperms -text "Classes/Permissions"]
    set fm_objs [TitleFrame $objects_tab.objs -text "Object Classes"]
    set fm_perms [TitleFrame $objects_tab.perms -text "Allow and Audit Rule Permissions"]
    pack $fm_objs -side left -expand 0 -fill both -padx 2 -pady 2
    pack $fm_perms -side left -expand 1 -fill both -padx 2 -pady 2
    set sw [ScrolledWindow [$fm_objs getframe].sw -auto both]
    set widgets(cp:classes) [listbox [$sw getframe].lb -height 5 -width 24 \
                                 -highlightthickness 0 -selectmode multiple \
                                 -exportselection 0 -state disabled \
                                 -bg $ApolTop::default_bg_color \
                                 -listvar Apol_TE::vals(cp:classes)]
    $sw setwidget $widgets(cp:classes)
    update
    grid propagate $sw 0
    bind $widgets(cp:classes) <<ListboxSelect>> \
        [list Apol_TE::toggle_cp_select classes]
    pack $sw -expand 1 -fill both
    set clear [button [$fm_objs getframe].b -text "Clear" -width 6 -state disabled \
                   -command [list Apol_TE::clear_cp_listbox $widgets(cp:classes) classes]]
    pack $clear -expand 0 -pady 2
    set widgets(cp:classes_widgets) [list $widgets(cp:classes) $clear]
    set f [$fm_perms getframe]
    set sw [ScrolledWindow $f.sw -auto both]
    set widgets(cp:perms) [listbox [$sw getframe].lb -height 5 -width 24 \
                               -highlightthickness 0 -selectmode multiple \
                               -exportselection 0 -bg white \
                               -listvar Apol_TE::vals(cp:perms)]
    $sw setwidget $widgets(cp:perms)
    update
    grid propagate $sw 0
    bind $widgets(cp:perms) <<ListboxSelect>> \
        [list Apol_TE::toggle_cp_select perms]
    set clear [button $f.clear -text "Clear" \
                   -command [list Apol_TE::clear_cp_listbox $widgets(cp:perms) perms]]
    set reverse [button $f.reverse -text "Reverse" \
                     -command [list Apol_TE::reverse_cp_listbox $widgets(cp:perms)]]
    set perm_rb_f [frame $f.rb]
    set l [label $perm_rb_f.l -text "Permissions to show:" -state disabled]
    set all [radiobutton $perm_rb_f.all -text "All" \
                       -variable Apol_TE::vals(cp:perms_toshow) -value all]
    set union [radiobutton $perm_rb_f.union -text "All for selected classes" \
                       -variable Apol_TE::vals(cp:perms_toshow) -value union]
    set intersect [radiobutton $perm_rb_f.inter -text "Common to selected classes" \
                       -variable Apol_TE::vals(cp:perms_toshow) -value intersect]
    trace add variable Apol_TE::vals(cp:perms_toshow) write \
        Apol_TE::toggle_perms_toshow
    pack $l $all $union $intersect -anchor w -padx 4
    grid $sw - $perm_rb_f -sticky nsw
    grid $clear $reverse ^ -pady 2 -sticky ew
    grid columnconfigure $f 0 -weight 0 -uniform 1 -pad 2
    grid columnconfigure $f 1 -weight 0 -uniform 1 -pad 2
    grid columnconfigure $f 2 -weight 1
    grid rowconfigure $f 0 -weight 1
    set widgets(cp:perms_widgets) \
        [list $widgets(cp:perms) $clear $reverse $l $all $union $intersect]
    trace add variable Apol_TE::vals(cp:classes_selected) write \
        [list Apol_TE::update_cp_tabname]
    trace add variable Apol_TE::vals(cp:perms_selected) write \
        [list Apol_TE::update_cp_tabname]
    trace add variable Apol_TE::enabled(cp:classes) write \
        [list Apol_TE::toggle_enable_cp classes]
    trace add variable Apol_TE::enabled(cp:perms) write \
        [list Apol_TE::toggle_enable_cp perms]
}
proc Apol_TE::toggle_enable_cp {prefix name1 name2 op} {
    variable vals
    variable widgets
    variable enabled
    if {$enabled(cp:${prefix})} {
        foreach w $widgets(cp:${prefix}_widgets) {
            $w configure -state normal
        }
        $widgets(cp:${prefix}) configure -bg white
    } else {
        foreach w $widgets(cp:${prefix}_widgets) {
            $w configure -state disabled
        }
        $widgets(cp:${prefix}) configure -bg $ApolTop::default_bg_color
    }
    set vals(cp:${prefix}_selected) $vals(cp:${prefix}_selected)
}
proc Apol_TE::maybe_enable_perms {} {
    variable vals
    variable enabled
    set avrule_set 0
    foreach x {allow neverallow auditallow dontaudit} {
        if {$vals(rs:$x)} {
            set avrule_set 1
            break
        }
    }
    if {$avrule_set} {
        set enabled(cp:perms) 1
    } else {
        set enabled(cp:perms) 0
    }
}
proc Apol_TE::toggle_perms_toshow {name1 name2 op} {
    variable vals
    variable widgets
    if {$vals(cp:perms_toshow) == "all"} {
        if {$op != "update"} {
            set vals(cp:perms) $Apol_Class_Perms::perms_list
            set vals(cp:perms_selected) {}
        }
    } elseif {$vals(cp:perms_toshow) == "union"} {
        set vals(cp:perms) {}
        set vals(cp:perms_selected) {}
        foreach class $vals(cp:classes_selected) {
            set vals(cp:perms) [lsort -unique -dictionary [concat $vals(cp:perms) [get_perms $class]]]
        }
    } else {  ;# intersection
        set vals(cp:perms) {}
        set vals(cp:perms_selected) {}
        set classes {}
        foreach i [$widgets(cp:classes) curselection] {
            lappend classes [$widgets(cp:classes) get $i]
        }
        if {$classes == {}} {
            return
        }
        set vals(cp:perms) [get_perms [lindex $classes 0]]
        foreach class [lrange $classes 1 end] {
            set this_perms [get_perms $class]
            set new_perms {}
            foreach p $vals(cp:perms) {
                if {[lsearch -exact $this_perms $p] >= 0} {
                    lappend new_perms $p
                }
            }
            set vals(cp:perms) $new_perms
        }
    }
}
proc Apol_TE::toggle_cp_select {col} {
    variable vals
    variable widgets
    set items {}
    foreach i [$widgets(cp:${col}) curselection] {
        lappend items [$widgets(cp:${col}) get $i]
    }
    set vals(cp:${col}_selected) $items
    if {$col == "classes"} {
        toggle_perms_toshow {} {} update
    }
}
proc Apol_TE::get_perms {class} {
    foreach {class common perms} [lindex [apol_GetClasses $class] 0] {break}
    if {$common != {}} {
        foreach {common p classes} [lindex [apol_GetCommons $common] 0] {break}
        lsort -dictionary -unique [concat $perms $p]
    } else {
        lsort -dictionary $perms
    }
}
proc Apol_TE::clear_cp_listbox {lb prefix} {
    variable vals
    $lb selection clear 0 end
    set vals(cp:${prefix}_selected) {}
    if {$prefix == "classes"} {
        toggle_perms_toshow {} {} update
    }
}
proc Apol_TE::reverse_cp_listbox {lb} {
    variable vals
    set old_selection [$lb curselection]
    set items {}
    for {set i 0} {$i < [$lb index end]} {incr i} {
        if {[lsearch $old_selection $i] >= 0} {
            $lb selection clear $i
        } else {
            $lb selection set $i
            lappend items [$lb get $i]
        }
    }
    set vals(cp:perms_selected) $items
}
proc Apol_TE::update_cp_tabname {name1 name2 op} {
    variable vals
    variable widgets
    variable enabled
    if {($enabled(cp:classes) && $vals(cp:classes_selected) > 0) || \
            ($enabled(cp:perms) && $vals(cp:perms_selected) > 0)} {
            $widgets(search_opts) itemconfigure classperms -text "Classes/Permissions *"
    } else {
        $widgets(search_opts) itemconfigure classperms -text "Classes/Permissions"
    }
}
proc Apol_TE::delete_results {pageID} {
    variable widgets
    variable tabs
    set curpos [$widgets(results) index $pageID]
    $widgets(results) delete $pageID
    array unset tabs $pageID:*
    array unset tabs $pageID
    if {[set next_id [$widgets(results) pages $curpos]] != {}} {
        switch_to_tab $next_id
    } elseif {$curpos > 0} {
        switch_to_tab [$widgets(results) pages [expr {$curpos - 1}]]
    } else {
        $widgets(update) configure -state disabled
    }
}
proc Apol_TE::display_rename_tab_dialog {pageID} {
    variable widgets
    variable tabs
    set d [Dialog .apol_te_tab_rename -homogeneous 1 -spacing 2 -cancel 1 \
               -default 0 -modal local -parent . -place center -separator 1 \
               -side bottom -title "Rename Results Tab"]
    $d add -text "OK" -command [list $d enddialog "ok"]
    $d add -text "Cancel" -command [list $d enddialog "cancel"]
    set f [$d getframe]
    set l [label $f.l -text "Tab name:"]
    set tabs(tab:new_name) [$widgets(results) itemcget $pageID -text]
    set e [entry $f.e -textvariable Apol_TE::tabs(tab:new_name) -width 16 -bg white]
    pack $l $e -side left -padx 2
    set retval [$d draw]
    destroy $d
    if {$retval == "ok"} {
        $widgets(results) itemconfigure $pageID -text $tabs(tab:new_name)
    }
}
proc Apol_TE::delete_current_results {} {
    variable widgets
    if {[set curid [$widgets(results) raise]] != {}} {
        delete_results $curid
    }
}
proc Apol_TE::create_new_results_tab {} {
    variable vals
    variable widgets
    variable tabs
    set i $tabs(next_result_id)
    incr tabs(next_result_id)
    set id "results$i"
    set frame [$widgets(results) insert end "$id" -text "Results $i"]
    $widgets(results) raise $id
    set tabs($id) [Apol_Widget::makeSearchResults $frame.results]
    pack $tabs($id) -expand 1 -fill both
    set tabs($id:vals) [array get vals]
    return $tabs($id)
}
proc Apol_TE::switch_to_tab {pageID} {
    variable vals
    variable widgets
    variable tabs
    if {[$Apol_TE::widgets(results) raise] == $pageID} {
        return
    }
    $widgets(results) raise $pageID
    set cur_search_opts [$widgets(search_opts) raise]
    array set tmp_vals $tabs($pageID:vals)
    set classes_selected $tmp_vals(cp:classes_selected)
    set perms_selected $tmp_vals(cp:perms_selected)
    array set vals $tabs($pageID:vals)
    reinitialize_search_widgets
    set vals(cp:classes_selected) $classes_selected
    set vals(cp:perms_selected) $perms_selected
    foreach c $classes_selected {
        $widgets(cp:classes) selection set [lsearch $vals(cp:classes) $c]
    }
    foreach p $perms_selected {
        $widgets(cp:perms) selection set [lsearch $vals(cp:perms) $p]
    }
    $widgets(search_opts) raise $cur_search_opts
}
proc Apol_TE::reset {} {
    variable enabled
    set old_classes_enabled $enabled(cp:classes)
    reinitialize_default_search_options
    reinitialize_search_widgets
    if {[set enabled(cp:classes) $old_classes_enabled]} {
        variable vals
        set vals(cp:classes) $Apol_Class_Perms::class_list
        set enabled(cp:classes) 1
        set enabled(cp:perms) 1
    }
}
proc Apol_TE::search_terules {whichButton} {
    variable vals
    variable widgets
    variable enabled
    variable tabs
    if {![ApolTop::is_policy_open]} {
        tk_messageBox -icon error -type ok -title "Error" -message "No current policy file is opened!"
        return
    }
    if {$enabled(ta:use_source) && $vals(ta:use_source) && $vals(ta:source_sym) == {}} {
        tk_messageBox -icon error -type ok -title "Error" -message "No source type/attribute was selected."
        return
    }
    if {$enabled(ta:use_target) && $vals(ta:use_target) && $vals(ta:target_sym) == {}} {
        tk_messageBox -icon error -type ok -title "Error" -message "No target type/attribute was selected."
        return
    }
    if {$enabled(ta:use_default) && $vals(ta:use_default) && $vals(ta:default_sym) == {}} {
        tk_messageBox -icon error -type ok -title "Error" -message "No default type selected."
        return
    }
    set rule_selection {}
    foreach {key value} [array get vals rs:*] {
        if {$value} {
            lappend rule_selection [string range $key 3 end]
        }
    }
    if {$rule_selection == {}} {
            tk_messageBox -icon error -type ok -title "Error" -message "At least one rule must be selected."
            return
    }
    set source_any 0
    if {$enabled(ta:use_source) && $vals(ta:use_source)} {
        if {$vals(ta:source_which) == "either"} {
            set source_any 1
        }
        set source [list $vals(ta:source_sym) $vals(ta:source_indirect) $vals(ta:source_sym,types) $vals(ta:source_sym,attribs)]
    } else {
        set source {}
    }
    if {$enabled(ta:use_target) && $vals(ta:use_target)} {
        set target [list $vals(ta:target_sym) $vals(ta:target_indirect) $vals(ta:target_sym,types) $vals(ta:target_sym,attribs)]
    } else {
        set target {}
    }
    if {$enabled(ta:use_default) && $vals(ta:use_default)} {
        set default [list $vals(ta:default_sym) 0 $vals(ta:default_sym,types) $vals(ta:default_sym,attribs)]
    } else {
        set default {}
    }
    if {$enabled(cp:classes)} {
        set classes $vals(cp:classes_selected)
    } else {
        set classes {}
    }
    if {$enabled(cp:perms)} {
        set perms $vals(cp:perms_selected)
    } else {
        set perms {}
    }
    set other_opts {}
    if {$vals(oo:enabled)} {
        lappend other_opts "only_enabled"
    }
    if {$vals(oo:regexp)} {
        lappend other_opts "regex"
    }
    if {$source_any} {
        lappend other_opts "source_any"
    }
    if {[ApolTop::is_capable "syntactic rules"]} {
        lappend other_opts "syn_search"
    }
    foreach x {new update reset} {
        $widgets($x) configure -state disabled
    }
    set tabs(searches_done) -1
    set tabs(searches_text) "Searching for TE Rules..."
    ProgressDlg .terules_busy -title "TE Rules Search" \
        -type normal -stop {} -separator 1 -parent . -maximum 2 \
        -textvariable Apol_TE::tabs(searches_text) -width 32 \
        -variable Apol_TE::tabs(searches_done)
    ApolTop::setBusyCursor
    update idletasks
    set retval [catch {apol_SearchTERules $rule_selection $other_opts \
                           $source $target $default $classes $perms} results]
    $widgets(new) configure -state normal
    $widgets(reset) configure -state normal
    if {$retval} {
        ApolTop::resetBusyCursor
        destroy .terules_busy
        tk_messageBox -icon error -type ok -title Error -message "Error searching TE rules:\n$results"
        focus -force .
    } else {
        set tabs(searches_text) "Collecting results..."
        update idletasks
        foreach {avresults teresults} $results {break}
        if {$whichButton == "new"} {
            set sr [create_new_results_tab]
        } else {
            set id [$widgets(results) raise]
            set tabs($id:vals) [array get vals]
            set sr $tabs($id)
            Apol_Widget::clearSearchResults $sr
        }
        if {![ApolTop::is_capable "syntactic rules"]} {
            set numAVs [Apol_Widget::appendSearchResultAVRules $sr 0 $avresults tabs(searches_text)]
            set numTEs [Apol_Widget::appendSearchResultTERules $sr 0 $teresults tabs(searches_text)]
        } else {
            set numAVs [Apol_Widget::appendSearchResultSynAVRules $sr 0 $avresults tabs(searches_text)]
            set numTEs [Apol_Widget::appendSearchResultSynTERules $sr 0 $teresults tabs(searches_text)]
        }
        set num_rules [expr {[lindex $numAVs 0] + [lindex $numTEs 0]}]
        set num_enabled [expr {[lindex $numAVs 1] + [lindex $numTEs 1]}]
        set num_disabled [expr {[lindex $numAVs 2] + [lindex $numTEs 2]}]
        set header "$num_rules rule"
        if {$num_rules != 1} {
            append header s
        }
        append header " match the search criteria.\n"
        append header "Number of enabled conditional rules: $num_enabled\n"
        append header "Number of disabled conditional rules: $num_disabled\n"
        Apol_Widget::appendSearchResultHeader $sr $header
        ApolTop::resetBusyCursor
        destroy .terules_busy
        focus -force .
    }
    if {[$widgets(results) pages] != {} || $retval == 0} {
        $widgets(update) configure -state normal
    }
}
proc Apol_TE::load_query_options {file_channel parentDlg} {
    variable vals
    variable widgets
    variable enabled
    reinitialize_default_search_options
    set classes_selected {}
    set perms_selected {}
    while {[gets $file_channel line] >= 0} {
        set line [string trim $line]
        if {$line == {} || [string index $line 0] == "#"} {
            continue
        }
        regexp -line -- {^(\S+)( (.+))?} $line -> key --> value
        if {$key == "cp:classes_selected"} {
            set classes_selected $value
        } elseif {$key == "cp:perms_selected"} {
            set perms_selected $value
        } else {
            set vals($key) $value
        }
    }
    reinitialize_search_widgets
    set vals(cp:classes) $Apol_Class_Perms::class_list
    set enabled(cp:classes) 1
    set enabled(cp:perms) 1
    toggle_perms_toshow -> -> reset
    set unknowns {}
    set vals(cp:classes_selected) {}
    foreach class $classes_selected {
        if {[set i [lsearch $vals(cp:classes) $class]] >= 0} {
            $widgets(cp:classes) selection set $i
            lappend vals(cp:classes_selected) $class
        } else {
            lappend unknowns $class
        }
    }
    if {[llength $unknowns] > 0} {
        tk_messageBox -icon warning -type ok -title "Invalid Object Classes" \
            -message "The following object classes do not exist in the currently loaded policy and were ignored:\n\n[join $unknowns ", "]" \
            -parent $parentDlg
    }
    toggle_perms_toshow {} {} {}
    set unknowns {}
    set vals(cp:perms_selected) {}
    foreach perm $perms_selected {
        if {[set i [lsearch $vals(cp:perms) $perm]] >= 0} {
            $widgets(cp:perms) selection set $i
            lappend vals(cp:perms_selected) $perm
        } else {
            lappend unknowns $perm
        }
    }
    if {[llength $unknowns] > 0} {
        tk_messageBox -icon warning -type ok -title "Invalid Permissions" \
            -message "The following permissions do not exist in the currently loaded policy and were ignored:\n\n[join $unknowns ", "]" \
            -parent $parentDlg
    }
}
proc Apol_TE::save_query_options {file_channel query_file} {
    variable vals
    foreach {key value} [array get vals] {
        if {$key != "cp:classes" && $key != "cp:perms"} {
            puts $file_channel "$key $value"
        }
    }
}
namespace eval Apol_Roles {
    variable widgets
    variable opts
    set opts(useType)		0
    set opts(showSelection)     all
    variable role_list		""
}
proc Apol_Roles::open { } {
    variable widgets
    variable role_list {}
    foreach r [apol_GetRoles {} {} 0] {
        lappend role_list [lindex $r 0]
    }
    set role_list [lsort $role_list]
    Apol_Widget::resetTypeComboboxToPolicy $widgets(combo_types)
}
proc Apol_Roles::close { } {
    variable widgets
    variable opts
    set opts(useType)	0
    set opts(showSelection)	all
    set Apol_Roles::role_list	""
    Apol_Widget::clearTypeCombobox $widgets(combo_types)
    Apol_Widget::clearSearchResults $widgets(resultsbox)
}
proc Apol_Roles::set_Focus_to_Text {} {
    focus $Apol_Roles::widgets(resultsbox)
}
proc Apol_Roles::popupRoleInfo {which role} {
    set role_datum [lindex [apol_GetRoles $role] 0]
    Apol_Widget::showPopupText $role [renderRole $role_datum 1]
}
proc Apol_Roles::renderRole {role_datum show_all} {
    foreach {name types dominates} $role_datum {break}
    if {!$show_all} {
        return $name
    }
    set text "$name ([llength $types] type"
    if {[llength $types] != 1} {
        append text "s"
    }
    append text ")\n"
    foreach t [lsort -dictionary $types] {
        append text "    $t\n"
    }
    return $text
}
proc Apol_Roles::search { str case_Insensitive regExpr srch_Direction } {
    variable widgets
    ApolTop::textSearch $widgets(resultsbox).tb $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_Roles::searchRoles {} {
    variable widgets
    variable opts
    Apol_Widget::clearSearchResults $widgets(resultsbox)
    if {![ApolTop::is_policy_open]} {
        tk_messageBox -icon error -type ok -title "Error" -message "No current policy file is opened!"
        return
    }
    if {$opts(useType)} {
        set type [Apol_Widget::getTypeComboboxValue $widgets(combo_types)]
        if {$type == {}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No type selected."
            return
        }
    } else {
        set type {}
    }
    if {$opts(showSelection) == "names"} {
        set show_all 0
    } else {
        set show_all 1
    }
    if {[catch {apol_GetRoles {} $type 0} roles_data]} {
        tk_messageBox -icon error -type ok -title "Error" -message $roles_data
        return
    }
    set text "ROLES:\n"
    if {[llength $roles_data] == 0} {
        append text "Search returned no results."
    } else {
        foreach r [lsort -index 0 $roles_data] {
            append text "\n[renderRole $r $show_all]"
        }
    }
    Apol_Widget::appendSearchResultText $widgets(resultsbox) $text
}
proc Apol_Roles::toggleTypeCombobox {path name1 name2 op} {
    Apol_Widget::setTypeComboboxState $path $Apol_Roles::opts(useType)
}
proc Apol_Roles::goto_line { line_num } {
    variable widgets
    Apol_Widget::gotoLineSearchResults $widgets(resultsbox) $line_num
}
proc Apol_Roles::create {nb} {
    variable widgets
    variable opts
    set frame [$nb insert end $ApolTop::roles_tab -text "Roles"]
    set pw [PanedWindow $frame.pw -side top]
    set leftf [$pw add -weight 0]
    set rightf [$pw add -weight 1]
    pack $pw -fill both -expand yes
    set rolebox [TitleFrame $leftf.rolebox -text "Roles"]
    set s_optionsbox [TitleFrame $rightf.obox -text "Search Options"]
    set resultsbox [TitleFrame $rightf.rbox -text "Search Results"]
    pack $rolebox -fill both -expand yes
    pack $s_optionsbox -padx 2 -fill both -expand 0
    pack $resultsbox -padx 2 -fill both -expand yes
    set rlistbox [Apol_Widget::makeScrolledListbox [$rolebox getframe].lb \
                      -width 20 -listvar Apol_Roles::role_list]
    Apol_Widget::setListboxCallbacks $rlistbox \
        {{"Display Role Info" {Apol_Roles::popupRoleInfo role}}}
    pack $rlistbox -fill both -expand yes
    set ofm [$s_optionsbox getframe]
    set lfm [frame $ofm.to]
    set cfm [frame $ofm.co]
    pack $lfm $cfm -side left -anchor nw -padx 4 -pady 2
    radiobutton $lfm.all_info -text "All information" \
        -variable Apol_Roles::opts(showSelection) -value all
    radiobutton $lfm.names_only -text "Names only" \
        -variable Apol_Roles::opts(showSelection) -value names
    pack $lfm.all_info $lfm.names_only -anchor w -padx 5 -pady 4
    set cb_type [checkbutton $cfm.cb -variable Apol_Roles::opts(useType) -text "Type"]
    set widgets(combo_types) [Apol_Widget::makeTypeCombobox $cfm.combo_types]
    Apol_Widget::setTypeComboboxState $widgets(combo_types) disabled
    trace add variable Apol_Roles::opts(useType) write \
        [list Apol_Roles::toggleTypeCombobox $widgets(combo_types)]
    pack $cb_type -anchor w
    pack $widgets(combo_types) -anchor w -padx 4
    button $ofm.ok -text OK -width 6 -command {Apol_Roles::searchRoles}
    pack $ofm.ok -side top -anchor e -pady 5 -padx 5
    set widgets(resultsbox) [Apol_Widget::makeSearchResults [$resultsbox getframe].sw]
    pack $widgets(resultsbox) -expand 1 -fill both
    return $frame
}
namespace eval Apol_RBAC {
    variable vals
    variable widgets
}
proc Apol_RBAC::search { str case_Insensitive regExpr srch_Direction } {
    variable widgets
    ApolTop::textSearch $widgets(results).tb $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_RBAC::set_Focus_to_Text {} {
    focus $Apol_RBAC::widgets(results)
}
proc Apol_RBAC::searchRBACs {} {
    variable vals
    variable widgets
    Apol_Widget::clearSearchResults $widgets(results)
    if {![ApolTop::is_policy_open]} {
        tk_messageBox -icon error -type ok -title "Error" -message "No current policy file is opened!"
        return
    }
    switch -- $vals(rule_selection) {
        "allow" { set rule_selection allow}
        "trans" { set rule_selection role_transition}
        "both"  { set rule_selection [list allow role_transition] }
    }
    set other_opts {}
    set source_sym {}
    if {$vals(source:use)} {
        if {$vals(source:sym) == {}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No source role selected."
            return
        }
        if {$vals(source:which) == "either"} {
            lappend other_opts "source_any"
        }
        set source_sym $vals(source:sym)
    }
    set target_sym {}
    if {$rule_selection == "allow" && $vals(target_role:use) && \
            (!$vals(source:use) || $vals(source:which) != "either")} {
        if {$vals(target_role:sym) == {}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No target role selected."
            return
        }
        set target_sym $vals(target_role:sym)
    } elseif {$rule_selection == "role_transition" && $vals(target_type:use)} {
        if {$vals(target_type:sym) == {}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No target type selected."
            return
        }
        set target_sym $vals(target_type:sym)
    }
    set default_sym {}
    if {$rule_selection == "role_transition" && $vals(default:use) && \
            (!$vals(source:use) || $vals(source:which) != "either")} {
        if {$vals(default:sym) == {}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No default role selected."
            return
        }
        set default_sym $vals(default:sym)
    }
    if {[catch {apol_SearchRBACRules $rule_selection $other_opts $source_sym $target_sym $default_sym} results]} {
        tk_messageBox -icon error -type ok -title "Error" -message "Error searching RBAC rules:\n$results"
        return
    }
    if {[llength $results] == 0} {
        set text "Search returned no results."
    } else {
        set text "[llength $results] rule"
        if {[llength $results] != 1} {
            append text s
        }
        append text " match the search criteria.\n\n"
    }
    Apol_Widget::appendSearchResultText $widgets(results) $text
    foreach r [lsort $results] {
        renderRBAC $r
    }
}
proc Apol_RBAC::renderRBAC {rule} {
    variable widgets
    foreach {rule_type source_set target_set default } $rule {
        if {[llength $source_set] > 1} {
            set source_set "\{ $source_set \}"
        }
        if {[llength $target_set] > 1} {
            set target_set "\{ $target_set \}"
        }
        if {$default != {}} {
            Apol_Widget::appendSearchResultLine $widgets(results) 0 \
                {} $rule_type $source_set $target_set $default
        } else {
            Apol_Widget::appendSearchResultLine $widgets(results) 0 \
                {} $rule_type $source_set $target_set
        }
    }
}
proc Apol_RBAC::open { } {
    variable vals
    variable widgets
    $widgets(allow:source) configure -values $Apol_Roles::role_list
    $widgets(allow:target) configure -values $Apol_Roles::role_list
    $widgets(trans:source) configure -values $Apol_Roles::role_list
    $widgets(trans:default) configure -values $Apol_Roles::role_list
    $widgets(both:source) configure -values $Apol_Roles::role_list
    set vals(target_type:types) $vals(target_type:types)
    set vals(rule_selection) allow
}
proc Apol_RBAC::close { } {
    variable widgets
    initializeVars
    $widgets(allow:source) configure -values {}
    $widgets(allow:target) configure -values {}
    $widgets(trans:source) configure -values {}
    $widgets(trans:target) configure -values {}
    $widgets(trans:default) configure -values {}
    $widgets(both:source) configure -values {}
}
proc Apol_RBAC::initializeVars {} {
    variable vals
    array set vals {
        rule_selection allow
        source:use 0
        source:sym {}
        source:which source
        target_role:use 0
        target_role:sym {}
        target_type:use 0
        target_type:sym {}
        target_type:types 1
        target_type:attribs 0
        default:use 0
        default:sym {}
    }
}
proc Apol_RBAC::goto_line { line_num } {
    variable widgets
    Apol_Widget::gotoLineSearchResults $widgets(results) $line_num
}
proc Apol_RBAC::create {nb} {
    variable vals
    variable widgets
    initializeVars
    set frame [$nb insert end $ApolTop::rbac_tab -text "RBAC Rules"]
    set topf [frame $frame.top]
    set bottomf [frame $frame.bottom]
    pack $topf -expand 0 -fill both -pady 2
    pack $bottomf -expand 1 -fill both -pady 2
    set rsbox [TitleFrame $topf.rs -text "Rule Selection"]
    set obox [TitleFrame $topf.opts -text "Search Options"]
    set dbox [TitleFrame $bottomf.results -text "RBAC Rules Display"]
    pack $rsbox -side left -expand 0 -fill both -padx 2
    pack $obox -side left -expand 1 -fill both -padx 2
    pack $dbox -expand 1 -fill both -padx 2
    set rs [$rsbox getframe]
    radiobutton $rs.allow -text allow -value allow \
        -variable Apol_RBAC::vals(rule_selection)
    radiobutton $rs.trans -text role_transition -value trans \
        -variable Apol_RBAC::vals(rule_selection)
    radiobutton $rs.both -text "allow and role_transition" -value both \
        -variable Apol_RBAC::vals(rule_selection)
    trace add variable Apol_RBAC::vals(rule_selection) write \
        [list Apol_RBAC::ruleChanged]
    pack $rs.allow $rs.trans $rs.both -side top -anchor w
    set widgets(options_pm) [PagesManager [$obox getframe].opts]
    allowCreate [$widgets(options_pm) add allow]
    transCreate [$widgets(options_pm) add trans]
    bothCreate [$widgets(options_pm) add both]
    trace add variable Apol_RBAC::vals(source:which) write Apol_RBAC::toggleRoleBox
    $widgets(options_pm) compute_size
    pack $widgets(options_pm) -expand 1 -fill both -side left
    $widgets(options_pm) raise allow
    set ok [button [$obox getframe].ok -text OK -width 6 -command Apol_RBAC::searchRBACs]
    pack $ok -side right -padx 5 -pady 5 -anchor ne
    set widgets(results) [Apol_Widget::makeSearchResults [$dbox getframe].results]
    pack $widgets(results) -expand yes -fill both
    return $frame
}
proc Apol_RBAC::ruleChanged {name1 name2 ops} {
    variable vals
    variable widgets
    Apol_Widget::clearSearchResults $widgets(results)
    $widgets(options_pm) raise $vals(rule_selection)
}
proc Apol_RBAC::allowCreate {a_f} {
    variable vals
    variable widgets
    set source [frame $a_f.source]
    set source_cb [checkbutton $source.enable -text "Source role" \
                       -variable Apol_RBAC::vals(source:use)]
    set widgets(allow:source) [ComboBox $source.cb -width 20 -state disabled \
                                   -entrybg $ApolTop::default_bg_color \
                                   -textvariable Apol_RBAC::vals(source:sym) \
                                   -helptext "Type or select a role" -autopost 1]
    set which_fm [frame $source.which]
    set which_source [radiobutton $which_fm.source \
                          -text "As source" -state disabled \
                          -variable Apol_RBAC::vals(source:which) \
                          -value source]
    set which_any [radiobutton $which_fm.any \
                       -text "As source or target" -state disabled \
                       -variable Apol_RBAC::vals(source:which) \
                       -value either]
    trace add variable Apol_RBAC::vals(source:use) write \
        [list Apol_RBAC::toggleCheckbutton $widgets(allow:source) [list $which_source $which_any]]
    pack $which_source $which_any -side top -anchor w
    pack $source_cb -side top -anchor w
    pack $widgets(allow:source) -side top -expand 0 -fill x -padx 4
    pack $which_fm -anchor w -padx 8
    pack $source -side left -padx 4 -pady 2 -expand 0 -anchor nw
    set target [frame $a_f.target]
    set widgets(allow:target_cb) [checkbutton $target.enable -text "Target role" \
                                      -variable Apol_RBAC::vals(target_role:use)]
    set widgets(allow:target) [ComboBox $target.cb -width 20 -state disabled \
                                   -entrybg $ApolTop::default_bg_color \
                                   -textvariable Apol_RBAC::vals(target_role:sym) \
                                   -helptext "Type or select a role" -autopost 1]
    trace add variable Apol_RBAC::vals(target_role:use) write \
        [list Apol_RBAC::toggleCheckbutton $widgets(allow:target) {}]
    pack $widgets(allow:target_cb) -side top -anchor w
    pack $widgets(allow:target) -side top -expand 0 -fill x -padx 4
    pack $target -side left -padx 4 -pady 2 -expand 0 -fill y
}
proc Apol_RBAC::transCreate {t_f} {
    variable vals
    variable widgets
    set source [frame $t_f.source]
    set source_cb [checkbutton $source.enable -text "Source role" \
                       -variable Apol_RBAC::vals(source:use)]
    set widgets(trans:source) [ComboBox $source.cb -width 20 -state disabled \
                                        -entrybg $ApolTop::default_bg_color \
                                        -textvariable Apol_RBAC::vals(source:sym) \
                                        -helptext "Type or select a role" -autopost 1]
    set which_fm [frame $source.which]
    set which_source [radiobutton $which_fm.source \
                          -text "As source" -state disabled \
                          -variable Apol_RBAC::vals(source:which) \
                          -value source]
    set which_any [radiobutton $which_fm.any \
                       -text "As source or default" -state disabled \
                       -variable Apol_RBAC::vals(source:which) \
                       -value either]
    trace add variable Apol_RBAC::vals(source:use) write \
        [list Apol_RBAC::toggleCheckbutton $widgets(trans:source) [list $which_source $which_any]]
    pack $which_source $which_any -side top -anchor w
    pack $source_cb -side top -anchor w
    pack $widgets(trans:source) -side top -expand 0 -fill x -padx 4
    pack $which_fm -anchor w -padx 8
    pack $source -side left -padx 4 -pady 2 -expand 0 -anchor nw
    set target [frame $t_f.target]
    set target_cb [checkbutton $target.enable -text "Target type" \
                       -variable Apol_RBAC::vals(target_type:use)]
    set widgets(trans:target) [ComboBox $target.cb -width 20 -state disabled \
                                   -entrybg $ApolTop::default_bg_color \
                                   -textvariable Apol_RBAC::vals(target_type:sym) \
                                   -helptext "Type or select a type/attribute" -autopost 1]
    set ta_frame [frame $target.ta]
    set types [checkbutton $ta_frame.types -text "Types" -state disabled \
                   -variable Apol_RBAC::vals(target_type:types)]
    set attribs [checkbutton $ta_frame.attribs -text "Attribs" -state disabled \
                   -variable Apol_RBAC::vals(target_type:attribs)]
    $types configure -command [list Apol_RBAC::toggleTAPushed $types]
    $attribs configure -command [list Apol_RBAC::toggleTAPushed $attribs]
    trace add variable Apol_RBAC::vals(target_type:types) write \
        [list Apol_RBAC::toggleTASym]
    trace add variable Apol_RBAC::vals(target_type:attribs) write \
        [list Apol_RBAC::toggleTASym]
    pack $types $attribs -side left -padx 2
    trace add variable Apol_RBAC::vals(target_type:use) write \
        [list Apol_RBAC::toggleCheckbutton $widgets(trans:target) [list $types $attribs]]
    pack $target_cb -side top -anchor w
    pack $widgets(trans:target) -side top -expand 0 -fill x -padx 4
    pack $ta_frame -anchor center -pady 2
    pack $target -side left -padx 4 -pady 2 -expand 0 -fill y
    set default [frame $t_f.default]
    set widgets(trans:default_cb) [checkbutton $default.enable -text "Default role" \
                                       -variable Apol_RBAC::vals(default:use)]
    set widgets(trans:default) [ComboBox $default.cb -width 20 -state disabled \
                                   -entrybg $ApolTop::default_bg_color \
                                   -textvariable Apol_RBAC::vals(default:sym) \
                                   -helptext "Type or select a role" -autopost 1]
    trace add variable Apol_RBAC::vals(default:use) write \
        [list Apol_RBAC::toggleCheckbutton $widgets(trans:default) {}]
    pack $widgets(trans:default_cb) -side top -anchor w
    pack $widgets(trans:default) -side top -expand 0 -fill x -padx 4
    pack $default -side left -padx 4 -pady 2 -expand 0 -fill y
}
proc Apol_RBAC::bothCreate {b_f} {
    variable vals
    variable widgets
    set source [frame $b_f.source]
    set source_cb [checkbutton $source.enable -text "Source role" \
                       -variable Apol_RBAC::vals(source:use)]
    set widgets(both:source) [ComboBox $source.cb -width 20 -state disabled \
                                   -entrybg $ApolTop::default_bg_color \
                                   -textvariable Apol_RBAC::vals(source:sym) \
                                   -helptext "Type or select a role" -autopost 1]
    set which_fm [frame $source.which]
    set which_source [radiobutton $which_fm.source \
                          -text "As source" -state disabled \
                          -variable Apol_RBAC::vals(source:which) \
                          -value source]
    set which_any [radiobutton $which_fm.any \
                       -text "Any field" -state disabled \
                       -variable Apol_RBAC::vals(source:which) \
                       -value either]
    trace add variable Apol_RBAC::vals(source:use) write \
        [list Apol_RBAC::toggleCheckbutton $widgets(both:source) [list $which_source $which_any]]
    pack $which_source $which_any -side top -anchor w
    pack $source_cb -side top -anchor w
    pack $widgets(both:source) -side top -expand 0 -fill x -padx 4
    pack $which_fm -anchor w -padx 8
    pack $source -side left -padx 4 -pady 2 -expand 0 -anchor nw
}
proc Apol_RBAC::toggleCheckbutton {cb w name1 name2 ops} {
    variable vals
    if {$vals($name2)} {
        $cb configure -state normal -entrybg white
        foreach x $w {
            $x configure -state normal
        }
    } else {
        $cb configure -state disabled -entrybg $ApolTop::default_bg_color
        foreach x $w {
            $x configure -state disabled
        }
    }
    maybeEnableTargetRole
    maybeEnableDefaultRole
}
proc Apol_RBAC::toggleRoleBox {name1 name2 ops} {
    maybeEnableTargetRole
    maybeEnableDefaultRole
}
proc Apol_RBAC::maybeEnableTargetRole {} {
    variable vals
    variable widgets
    if {$vals(source:use) && $vals(source:which) == "either"} {
        $widgets(allow:target_cb) configure -state disabled
        $widgets(allow:target) configure -state disabled -entrybg $ApolTop::default_bg_color
    } else {
        $widgets(allow:target_cb) configure -state normal
        set vals(target_role:use) $vals(target_role:use)
    }
}
proc Apol_RBAC::maybeEnableDefaultRole {} {
    variable vals
    variable widgets
    if {$vals(source:use) && $vals(source:which) == "either"} {
        $widgets(trans:default_cb) configure -state disabled
        $widgets(trans:default) configure -state disabled -entrybg $ApolTop::default_bg_color
    } else {
        $widgets(trans:default_cb) configure -state normal
        set vals(default:use) $vals(default:use)
    }
}
proc Apol_RBAC::toggleTASym {name1 name2 ops} {
    variable vals
    variable widgets
    if {!$vals(target_type:types) && !$vals(target_type:attribs)} {
        return
    }
    if {$vals(target_type:types) && $vals(target_type:attribs)} {
        set items [lsort [concat $Apol_Types::typelist $Apol_Types::attriblist]]
    } elseif {$vals(target_type:types)} {
        set items $Apol_Types::typelist
    } else {
        set items $Apol_Types::attriblist
    }
    $widgets(trans:target) configure -values $items
}
proc Apol_RBAC::toggleTAPushed {cb} {
    variable vals
    if {!$vals(target_type:types) && !$vals(target_type:attribs)} {
        $cb select
    }
}
namespace eval Apol_Users {
    variable opts
    variable users_list ""
    variable widgets
}
proc Apol_Users::search { str case_Insensitive regExpr srch_Direction } {
    variable widgets
    ApolTop::textSearch $widgets(results).tb $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_Users::set_Focus_to_Text {} {
    focus $Apol_Users::widgets(results)
}
proc Apol_Users::searchUsers {} {
    variable opts
    variable widgets
    Apol_Widget::clearSearchResults $widgets(results)
    if {![ApolTop::is_policy_open]} {
        tk_messageBox -icon error -type ok -title "Error" -message "No current policy file is opened!"
        return
    }
    if {$opts(useRole)} {
        if {$opts(role) == ""} {
            tk_messageBox -icon error -type ok -title "Error" -message "No role selected."
            return
        }
        set role $opts(role)
    } else {
        set role {}
    }
    if {$opts(enable_default)} {
        if {$opts(default_level) == {{} {}}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No default level selected."
            return
        }
        set default $opts(default_level)
    } else {
        set default {}
    }
    set range_enabled [Apol_Widget::getRangeSelectorState $widgets(range)]
    foreach {range range_type} [Apol_Widget::getRangeSelectorValue $widgets(range)] {break}
    if {$range_enabled} {
        if {$range == {{{} {}} {{} {}}}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No range selected."
            return
        }
    } else {
        set range {}
    }
    if {$opts(showSelection) == "all"} {
        set show_all 1
    } else {
        set show_all 0
    }
    if {[catch {apol_GetUsers {} $role $default $range $range_type 0} users_data]} {
	tk_messageBox -icon error -type ok -title "Error" -message "Error obtaining users list:\n$users_data"
        return
    }
    set text "USERS:\n"
    if {[llength $users_data] == 0} {
        append text "Search returned no results."
    } else {
        foreach u [lsort -index 0 $users_data] {
            append text "\n[renderUser $u $show_all]"
        }
    }
    Apol_Widget::appendSearchResultText $widgets(results) $text
}
proc Apol_Users::renderUser {user_datum show_all} {
    set text ""
    foreach {user roles default range} $user_datum {break}
    append text "$user"
    if {!$show_all} {
        return $text
    }
    if {[ApolTop::is_capable "mls"]} {
        append text " level [apol_RenderLevel $default]"
        set low [apol_RenderLevel [lindex $range 0]]
        set high [apol_RenderLevel [lindex $range 1]]
        if {$low == $high} {
            append text " range $low"
        } else {
            append text " range $low - $high"
        }
    }
    append text " ([llength $roles] role"
    if {[llength $roles] != 1} {
        append text "s"
    }
    append text ")"
    append text "\n"
    foreach r $roles {
        append text "    $r\n"
    }
    return $text
}
proc Apol_Users::open { } {
    variable users_list {}
    variable widgets
    foreach u [apol_GetUsers {} {} {} {} {} 0] {
        lappend users_list [lindex $u 0]
    }
    set users_list [lsort $users_list]
    $Apol_Users::widgets(role) configure -values $Apol_Roles::role_list
    if {[ApolTop::is_capable "mls"]} {
        Apol_Widget::setRangeSelectorCompleteState $widgets(range) normal
        $widgets(defaultCB) configure -state normal
    } else {
        Apol_Widget::clearRangeSelector $widgets(range)
        Apol_Widget::setRangeSelectorCompleteState $widgets(range) disabled
        set Apol_Users::opts(enable_default) 0
        $widgets(defaultCB) configure -state disabled
    }
}
proc Apol_Users::close { } {
    variable widgets
    initializeVars
    set Apol_Users::users_list ""
    $widgets(role) configure -values ""
    Apol_Widget::clearSearchResults $widgets(results)
    Apol_Widget::clearRangeSelector $widgets(range)
    Apol_Widget::setRangeSelectorCompleteState $widgets(range) normal
    $widgets(defaultCB) configure -state normal
}
proc Apol_Users::initializeVars {} {
    variable opts
    array set opts {
        showSelection all
        useRole 0         role {}
        enable_default 0  default_level {{} {}}
    }
}
proc Apol_Users::popupUserInfo {which user} {
    set user_datum [lindex [apol_GetUsers $user] 0]
    Apol_Widget::showPopupText $user [renderUser $user_datum 1]
}
proc Apol_Users::goto_line { line_num } {
    variable widgets
    Apol_Widget::gotoLineSearchResults $widgets(results) $line_num
}
proc Apol_Users::create {nb} {
    variable opts
    variable widgets
    initializeVars
    set frame [$nb insert end $ApolTop::users_tab -text "Users"]
    set pw1   [PanedWindow $frame.pw -side top]
    set rpane [$pw1 add -weight 0]
    set spane [$pw1 add -weight 1]
    set userbox [TitleFrame $rpane.userbox -text "Users"]
    set s_optionsbox [TitleFrame $spane.obox -text "Search Options"]
    set resultsbox [TitleFrame $spane.rbox -text "Search Results"]
    pack $pw1 -fill both -expand yes
    pack $s_optionsbox -side top -expand 0 -fill both -padx 2
    pack $userbox -fill both -expand yes
    pack $resultsbox -expand yes -fill both -padx 2
    set users_listbox [Apol_Widget::makeScrolledListbox [$userbox getframe].lb -width 20 -listvar Apol_Users::users_list]
    Apol_Widget::setListboxCallbacks $users_listbox \
        {{"Display User Info" {Apol_Users::popupUserInfo users}}}
    pack $users_listbox -fill both -expand yes
    set ofm [$s_optionsbox getframe]
    set verboseFrame [frame $ofm.verbose]
    set rolesFrame [frame $ofm.roles]
    set defaultFrame [frame $ofm.default]
    set rangeFrame [frame $ofm.range]
    pack $verboseFrame $rolesFrame $defaultFrame $rangeFrame \
        -side left -padx 4 -pady 2 -anchor nw -expand 0 -fill y
    radiobutton $verboseFrame.all_info -text "All information" \
        -variable Apol_Users::opts(showSelection) -value all
    radiobutton $verboseFrame.names_only -text "Names only" \
        -variable Apol_Users::opts(showSelection) -value names
    pack $verboseFrame.all_info $verboseFrame.names_only -anchor w -padx 5 -pady 4
    checkbutton $rolesFrame.cb -variable Apol_Users::opts(useRole) -text "Role"
    set widgets(role) [ComboBox $rolesFrame.combo -width 12 -textvariable Apol_Users::opts(role) \
                           -helptext "Type or select a role" -state disabled \
                           -autopost 1]
    trace add variable Apol_Users::opts(useRole) write \
        [list Apol_Users::toggleRolesCheckbutton $widgets(role)]
    pack $rolesFrame.cb -anchor nw
    pack $widgets(role) -padx 4
    set widgets(defaultCB) [checkbutton $defaultFrame.cb -variable Apol_Users::opts(enable_default) -text "Default MLS level"]
    set defaultDisplay [Entry $defaultFrame.display -textvariable Apol_Users::opts(default_level_display) -width 16 -editable 0]
    set defaultButton [button $defaultFrame.button -text "Select Level..." -state disabled -command [list Apol_Users::show_level_dialog]]
    trace add variable Apol_Users::opts(enable_default) write \
        [list Apol_Users::toggleDefaultCheckbutton $widgets(defaultCB) $defaultDisplay $defaultButton]
    trace add variable Apol_Users::opts(default_level) write \
        [list Apol_Users::updateDefaultDisplay $defaultDisplay]
    pack $widgets(defaultCB) -side top -anchor nw -expand 0
    pack $defaultDisplay -side top -expand 0 -fill x -padx 4
    pack $defaultButton -side top -expand 1 -fill none -padx 4 -anchor ne
    set widgets(range) [Apol_Widget::makeRangeSelector $rangeFrame.range Users]
    pack $widgets(range) -expand 1 -fill x
    button $ofm.ok -text OK -width 6 -command {Apol_Users::searchUsers}
    pack $ofm.ok -side right -pady 5 -padx 5 -anchor ne
    set widgets(results) [Apol_Widget::makeSearchResults [$resultsbox getframe].results]
    pack $widgets(results) -expand yes -fill both
    return $frame
}
proc Apol_Users::toggleRolesCheckbutton {path name1 name2 op} {
    variable opts
    if {$opts($name2)} {
	$path configure -state normal -entrybg white
    } else {
        $path configure -state disabled -entrybg $ApolTop::default_bg_color
    }
}
proc Apol_Users::toggleDefaultCheckbutton {cb display button name1 name2 op} {
    variable opts
    if {$opts($name2)} {
        $button configure -state normal
        $display configure -state normal
    } else {
        $button configure -state disabled
        $display configure -state disabled
    }
}
proc Apol_Users::show_level_dialog {} {
    set Apol_Users::opts(default_level) [Apol_Level_Dialog::getLevel $Apol_Users::opts(default_level)]
}
proc Apol_Users::updateDefaultDisplay {display name1 name2 op} {
    variable opts
    if {$opts(default_level) == {{} {}}} {
        set opts(default_level_display) ""
        $display configure -helptext {}
    } else {
        set level [apol_RenderLevel $opts(default_level)]
        if {$level == ""} {
            set opts(default_level_display) "<invalid MLS level>"
        } else {
            set opts(default_level_display) $level
        }
        $display configure -helptext $opts(default_level_display)
    }
}
namespace eval Apol_Initial_SIDS {
    variable widgets
    variable vals
}
proc Apol_Initial_SIDS::search { str case_Insensitive regExpr srch_Direction } {
    variable widgets
    ApolTop::textSearch $widgets(results).tb $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_Initial_SIDS::set_Focus_to_Text {} {
    focus $Apol_Initial_SIDS::widgets(results)
}
proc Apol_Initial_SIDS::searchSIDs {} {
    variable vals
    variable widgets
    set name {}
    set context {}
    set range_match 0
    Apol_Widget::clearSearchResults $widgets(results)
    if {![ApolTop::is_policy_open]} {
        tk_messageBox -icon error -type ok -title "Error" -message "No current policy file is opened!"
        return
    }
    if {[Apol_Widget::getContextSelectorState $widgets(context)]} {
        foreach {context range_match} [Apol_Widget::getContextSelectorValue $widgets(context)] {break}
    }
    if {[catch {apol_GetInitialSIDs $name $context $range_match} isids]} {
        tk_messageBox -icon error -type ok -title "Error" -message "Error obtaining initial SIDs list: $isids"
        return
    }
    set results "INITIAL SIDS:"
    if {[llength $isids] == 0} {
        append results "\nSearch returned no results."
    } else {
        foreach i [lsort -index 0 -dictionary $isids] {
            append results "\n[render_isid $i]"
        }
    }
    Apol_Widget::appendSearchResultText $widgets(results) $results
}
proc Apol_Initial_SIDS::open { } {
    variable vals
    set vals(items) {}
    foreach sid [lsort -index 0 -dictionary [apol_GetInitialSIDs {} {} 0]] {
        lappend vals(items) [lindex $sid 0]
    }
}
proc Apol_Initial_SIDS::close { } {
    variable vals
    variable widgets
    set vals(items) {}
    Apol_Widget::clearSearchResults $widgets(results)
    Apol_Widget::clearContextSelector $widgets(context)
}
proc Apol_Initial_SIDS::render_isid {isid {compact 0}} {
    foreach {name context} $isid {break}
    set context [apol_RenderContext $context]
    if {$compact} {
        format "sid %s %s" $name $context
    } else {
        format "sid  %-16s %s" $name $context
    }
}
proc Apol_Initial_SIDS::popupSIDInfo {sid} {
    set info [apol_GetInitialSIDs $sid]
    set text "$sid:"
    foreach s [lsort -index 0 -dictionary $info] {
        append text "\n\t[render_isid $s 1]"
    }
    Apol_Widget::showPopupText "$sid Context" $text
}
proc Apol_Initial_SIDS::goto_line { line_num } {
    variable widgets
    Apol_Widget::gotoLineSearchResults $widgets(results) $line_num
}
proc Apol_Initial_SIDS::create {nb} {
    variable widgets
    variable vals
    array set vals {
        items {}
    }
    set frame [$nb insert end $ApolTop::initial_sids_tab -text "Initial SIDs"]
    set pw [PanedWindow $frame.pw -side top -weights extra]
    set leftf [$pw add -weight 0]
    set rightf [$pw add -weight 1]
    pack $pw -fill both -expand yes
    set sids_box [TitleFrame $leftf.sids_box -text "Initial SIDs"]
    set s_optionsbox [TitleFrame $rightf.obox -text "Search Options"]
    set rslts_frame [TitleFrame $rightf.rbox -text "Search Results"]
    pack $sids_box -expand 1 -fill both
    pack $s_optionsbox -side top -expand 0 -fill both -padx 2
    pack $rslts_frame -side top -expand yes -fill both -padx 2
    set widgets(items) [Apol_Widget::makeScrolledListbox [$sids_box getframe].lb -width 20 -listvar Apol_Initial_SIDS::vals(items)]
    Apol_Widget::setListboxCallbacks $widgets(items) \
        {{"Display Initial SID Context" {Apol_Initial_SIDS::popupSIDInfo}}}
    pack $widgets(items) -expand 1 -fill both
    set f [frame [$s_optionsbox getframe].c]
    set widgets(context) [Apol_Widget::makeContextSelector $f.context "Context"]
    pack $widgets(context)
    pack $f -side left -anchor n -padx 4 -pady 2
    set ok [button [$s_optionsbox getframe].ok -text "OK" -width 6 \
                -command Apol_Initial_SIDS::searchSIDs]
    pack $ok -side right -pady 5 -padx 5 -anchor ne
    set widgets(results) [Apol_Widget::makeSearchResults [$rslts_frame getframe].results]
    pack $widgets(results) -side top -expand yes -fill both
    return $frame
}
namespace eval Apol_File_Contexts {
    variable opts
    variable widgets
	variable entry_dir
	variable entry_fn
	variable create_fc_dlg		.fc_db_create_Dlg
    variable info_button_text \
"This tab allows you to create and load a file context index.  The file
context index is an on-disk database which contains the labeling
information for an entire filesystem. Once an index has been created
you can query the database by enabling and selecting a user, type,
object class or path. A query can also use regular expressions, if
this is enabled.\n
The results of the context query show the number of results followed
by a list of the matching files. The first field is the full context
followed by the object class of the file and lastly the path."
}
proc Apol_File_Contexts::display_analysis_info {} {
    Apol_Widget::showPopupParagraph "File Contexts Information" $Apol_File_Contexts::info_button_text
} 
proc Apol_File_Contexts::search { str case_Insensitive regExpr srch_Direction } {
    variable widgets
    ApolTop::textSearch $widgets(results).tb $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_File_Contexts::set_Focus_to_Text {} {
    focus $Apol_File_Contexts::widgets(results)
}
proc Apol_File_Contexts::is_db_loaded {} {
    return $Apol_File_Contexts::opts(db_loaded)
}
proc Apol_File_Contexts::populate_combo_boxes {} {
    variable widgets
    if {[catch {apol_FC_Index_DB_Get_Items types} types]} {
        tk_messageBox -icon error -type ok -title "Error" \
            -message "Error getting types from file context database: $types.\n"
        return
    }
    $widgets(type) configure -values [lsort $types]
    if {[catch {apol_FC_Index_DB_Get_Items users} users]} {
        tk_messageBox -icon error -type ok -title "Error" \
            -message "Error getting users from file context database: $users.\n"
        return
    }
    $widgets(user) configure -values [lsort $users]
    if {[catch {apol_FC_Index_DB_Get_Items classes} classes]} {
        tk_messageBox -icon error -type ok -title "Error" \
            -message "Error getting object classes from file context database: $classes.\n"
        return
    }
    $widgets(objclass) configure -values [lsort [lreplace $classes $i $i]]
}
proc Apol_File_Contexts::open { } {
    return 0
} 
proc Apol_File_Contexts::initialize { } {
    variable opts
    variable widgets
    array set opts {
        useUser 0       user {}    useUserRegex 0
        useObjclass 0   objclass {}
        useType 0       type {}    useTypeRegx 0
        useRange 0      range {}   useRangeRegex 0   fc_is_mls 1
        usePath 0       path {}    usePathRegex 0
        showContext 1  showObjclass 1
        indexFilename {}
        db_loaded 0
    }
    $widgets(user) configure -values {}
    $widgets(type) configure -values {}
    $widgets(objclass) configure -values {}
    Apol_Widget::clearSearchResults $widgets(results)
}
proc Apol_File_Contexts::close { } {
    Apol_File_Contexts::close_fc_db
    Apol_File_Contexts::initialize
}
proc Apol_File_Contexts::get_fc_files_for_ta {which ta} {	
    if {$which == "type"} {
        set types_list $ta
    } else {
        set types_list [lindex [apol_GetAttribs $ta] 0 1]
    }
    set results [apol_Search_FC_Index_DB {} $types_list {} {} {} 0 0 0 0]
    set return_list {}
    foreach fscon $results {
        lappend return_list $fscon
    }
    return $return_list
}
proc Apol_File_Contexts::search_fc_database { } {
    variable opts
    variable widgets
    Apol_Widget::clearSearchResults $widgets(results)
    if {$opts(useUser)} {
        if {[set user [list $opts(user)]] == {{}}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No user selected."
            return
        }
    } else {
        set user {}
    }
    if {$opts(useObjclass)} {
        if {[set objclass [list $opts(objclass)]] == {{}}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No object class selected."
            return
        }
    } else {
        set objclass {}
    }
    if {$opts(useType)} {
        if {[set type [list $opts(type)]] == {{}}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No type selected."
            return
        }
    } else {
        set type {}
    }
    if {$opts(fc_is_mls) && $opts(useRange)} {
        if {[set range [list $opts(range)]] == {{}}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No MLS range selected."
            return
        }
    } else {
        set range {}
    }
    if {$opts(usePath)} {
        if {[set path [list $opts(path)]] == {{}}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No path selected."
            return
        }
    } else {
        set path {}
    }
    ApolTop::setBusyCursor
    set rt [catch {apol_Search_FC_Index_DB \
                       $user $type $objclass $range $path \
                       $opts(useUserRegex) $opts(useTypeRegex) \
                       $opts(useRangeRegex) $opts(usePathRegex)} results]
    ApolTop::resetBusyCursor
    if {$rt != 0} {
        tk_messageBox -icon error -type ok -title "Error" -message $results
        return
    }
    if {$results == {}} {
        Apol_Widget::appendSearchResultText $widgets(results) "Search returned no results."
    } else {
        set text "FILES FOUND ([llength $results]):\n\n"
        foreach fscon $results {
            foreach {ctxt class path} $fscon {break}
            if {$opts(showContext)} {
                append text [format "%-46s" $ctxt]
            }
            if {$opts(showObjclass)} {
                append text [format "  %-14s" $class]
            }
            append text "  $path\n"
	}
        Apol_Widget::appendSearchResultText $widgets(results) $text
    }
}
proc Apol_File_Contexts::display_create_db_dlg {} {
	variable entry_dir
	variable entry_fn
	variable create_fc_dlg
	variable b1_create_dlg
	variable b2_create_dlg
    set d [Dialog .filecontexts_create -title "Create Index File" \
               -default 0 -cancel 1 -modal local -parent . -separator 1]
    set create_fc_dlg $d
    set f [$d getframe]
    set lbl_fn [label $f.lbl_fn -justify left -anchor w -text "Save file:"]
    set entry_fn [entry $f.entry_fn -width 30 -bg white]
    set browse_fn [button $f.button2 -text "Browse" -width 8 -command {
        set txt [$Apol_File_Contexts::entry_fn get]
        if {[string is space $txt]} {
            set dir_name "/"
            set init_file "/"
        } elseif {![file isdirectory $txt]} {
            set dir_name [file dirname $txt]
            set init_file $txt
        } else {
            set dir_name $txt
            set init_file ""
        }
        set file_n [tk_getSaveFile \
			-title "Select File to Save..." \
			-parent $Apol_File_Contexts::create_fc_dlg \
			-initialdir $dir_name \
			-initialfile $init_file]
        if {$file_n != ""} {
            $Apol_File_Contexts::entry_fn delete 0 end
            $Apol_File_Contexts::entry_fn insert end $file_n
        }
    }]
    set lbl_dir [label $f.lbl_dir -justify left -anchor w -text "Directory to index:"]
    set entry_dir [entry $f.entry_path -width 30 -bg white]
    set browse_dir [button $f.button1 -text "Browse" -width 8 -command {
        set txt [$Apol_File_Contexts::entry_dir get]
        if {[string is space $txt]} {
            set txt "/"
        } elseif {![file isdirectory $txt]} {
            set txt [file dirname $txt]
        }
        set dir_n [tk_chooseDirectory \
                       -title "Select Directory to Index..." \
                       -parent $Apol_File_Contexts::create_fc_dlg \
                       -initialdir $txt]
        if {$dir_n != ""} {
            $Apol_File_Contexts::entry_dir delete 0 end
            $Apol_File_Contexts::entry_dir insert end $dir_n
        }
    }]
    $entry_dir insert end "/"
    grid $lbl_fn $entry_fn $browse_fn -padx 4 -pady 2 -sticky ew
    grid $lbl_dir $entry_dir $browse_dir -padx 4 -pady 2 -sticky ew
    grid columnconfigure $f 0 -weight 0 -pad 4
    grid columnconfigure $f 1 -weight 1
    grid columnconfigure $f 2 -weight 0
    $d add -text "Create" \
            -command [list Apol_File_Contexts::create_fc_db $d]
    $d add -text "Cancel"
    $d draw
    destroy $d
}
proc Apol_File_Contexts::create_and_load_fc_db {fname dir_str} {
    variable opts
	set rt [catch {apol_Create_FC_Index_File $fname $dir_str} err]
	if {$rt != 0} {
		return -code error "Error while creating the index file: $err"
	} 
	set rt [catch {apol_Load_FC_Index_File $fname} err]
	if {$rt != 0} {
		return -code error \
			"The index file was created successfully, however, there was an error while loading: $err"
	}
    Apol_File_Contexts::initialize
    set opts(fc_is_mls) [apol_FC_Is_MLS]
    set opts(indexFilename) $fname
    set opts(db_loaded) 1
    Apol_File_Contexts::populate_combo_boxes
}
proc Apol_File_Contexts::create_fc_db {dlg} {
	variable entry_dir
	variable entry_fn
    variable opts
	set fname [$entry_fn get]
	set dir_str [$entry_dir get]
    if {$fname == {} || $dir_str == {}} {
        tk_messageBox -icon error -type ok -title "Error" \
            -message "Both a filename and starting directory are needed."
        raise $dlg
        return
    }
    set opts(progressMsg) "Creating index file.. .This may take a while."
    set opts(progressVal) -1
    set progress_dlg [ProgressDlg .apol_fc_progress -parent . \
                          -textvariable Apol_File_Contexts::opts(progressMsg) \
                          -variable Apol_File_Contexts::opts(progressVal) \
                          -maximum 3 -width 45]
    ApolTop::setBusyCursor
    update idletasks
	set rt [catch {Apol_File_Contexts::create_and_load_fc_db $fname $dir_str} err]
    ApolTop::resetBusyCursor
    destroy $progress_dlg
	if {$rt != 0} {
		tk_messageBox -icon error -type ok -title "Error" \
			-message "$err\nSee stderr for more information."
            raise $dlg
		return
	} 
    $dlg enddialog 0
}
proc Apol_File_Contexts::load_fc_db { } {
    variable opts
    	set db_file [tk_getOpenFile -title "Select Index File to Load..." -parent $ApolTop::mainframe]
	if {$db_file != ""} {	
		set rt [catch {apol_Load_FC_Index_File $db_file} err]
		if {$rt != 0} {
			tk_messageBox -icon error -type ok -title "Error" -message \
				"Error loading file context database: $err\nSee stderr for more information."
			return -1
		} 
		Apol_File_Contexts::initialize
            set opts(fc_is_mls) [apol_FC_Is_MLS]
            set opts(indexFilename) $db_file
		set opts(db_loaded) 1
		Apol_File_Contexts::populate_combo_boxes
		return 1
	}
	return 0
}
proc Apol_File_Contexts::close_fc_db { } {
    variable widgets
    variable opts
	set rt [catch {apol_Close_FC_Index_DB} err]
	if {$rt != 0} {
		tk_messageBox -icon error -type ok -title "Error" \
			-message "Error closing file context database: $err.\n"
		return
	}
    Apol_Widget::clearSearchResults $widgets(results)
    set opts(db_loaded) 0
}
proc Apol_File_Contexts::goto_line { line_num } {
    variable widgets
    ApolTop::goto_line $line_num $widgets(results)
}
proc Apol_File_Contexts::create {nb} {
    variable opts
    variable widgets
    set frame [$nb insert end $ApolTop::file_contexts_tab -text "File Contexts"]
    set options_pane [frame $frame.top]
    set results_pane [frame $frame.bottom]
    pack $options_pane -expand 0 -fill x
    pack $results_pane -expand 1 -fill both -pady 2
    set status [TitleFrame $options_pane.status -text "File Context Index"]
    set status_frame [$status getframe]
    set status_buttons [ButtonBox $status_frame.bb -homogeneous 0 -padx 2]
    $status_buttons add -text "Create and Load" -width 15 \
        -command {Apol_File_Contexts::display_create_db_dlg}
    $status_buttons add -text "Load" -width 8 \
        -command {Apol_File_Contexts::load_fc_db}
    set status_text [frame $status_frame.t]
    label $status_text.l -text "Loaded Index:"
    set status1 [label $status_text.t -textvariable Apol_File_Contexts::opts(statusText)]
    set status2 [label $status_text.t2 -textvariable Apol_File_Contexts::opts(statusText2) -fg red]
    trace add variable Apol_File_Contexts::opts(indexFilename) write \
        [list Apol_File_Contexts::changeStatusLabel $status1 $status2]
    grid $status_text.l $status1 -sticky w
    grid x $status2 -sticky w -pady 2
    pack $status_buttons $status_text -side left -anchor nw -padx 2 -pady 4
    pack $status -side top -expand 0 -fill x -pady 2 -padx 2
    set optionsbox [TitleFrame $options_pane.opts -text "Search Options"]
    pack $optionsbox -fill both -expand yes -padx 2 -pady 2
    set options_frame [$optionsbox getframe]
    set show_frame [frame $options_frame.show]
    set user_frame [frame $options_frame.user]
    set objclass_frame [frame $options_frame.objclass]
    set type_frame [frame $options_frame.type]
    set range_frame [frame $options_frame.range]
    set path_frame [frame $options_frame.path]
    grid $show_frame $user_frame $objclass_frame $type_frame $range_frame $path_frame \
        -padx 2 -sticky news
    foreach idx {1 2 3 4} {
        grid columnconfigure $options_frame $idx -uniform 1 -weight 0
    }
    grid columnconfigure $options_frame 0 -weight 0 -pad 8
    grid columnconfigure $options_frame 5 -weight 0
    set show_context [checkbutton $show_frame.context \
                          -variable Apol_File_Contexts::opts(showContext) \
                          -text "Show context"]
    set show_objclass [checkbutton $show_frame.objclass \
                           -variable Apol_File_Contexts::opts(showObjclass) \
                           -text "Show object class"]
    pack $show_context $show_objclass -side top -anchor nw
    checkbutton $user_frame.enable -text "User" \
        -variable Apol_File_Contexts::opts(useUser)
    set widgets(user) [ComboBox $user_frame.box -width 8 -autopost 1 \
                           -textvariable Apol_File_Contexts::opts(user) \
                           -helptext "Type or select a user"]
    set user_regex [checkbutton $user_frame.regex \
                        -variable Apol_File_Contexts::opts(useUserRegex) \
                        -text "Regular expression"]
    trace add variable Apol_File_Contexts::opts(useUser) write \
        [list Apol_File_Contexts::toggleEnable $widgets(user) -entrybg $user_regex]
    pack $user_frame.enable -side top -anchor nw
    pack $widgets(user) -side top -anchor nw -padx 4 -expand 0 -fill x
    pack $user_regex -side top -anchor nw -padx 4
    checkbutton $objclass_frame.enable -text "Object class" \
        -variable Apol_File_Contexts::opts(useObjclass)
    set widgets(objclass) [ComboBox $objclass_frame.box -width 8 -autopost 1 \
                               -textvariable Apol_File_Contexts::opts(objclass) \
                               -helptext "Type or select an object class"]
    trace add variable Apol_File_Contexts::opts(useObjclass) write \
        [list Apol_File_Contexts::toggleEnable $widgets(objclass) -entrybg {}]
    pack $objclass_frame.enable -side top -anchor nw
    pack $widgets(objclass) -side top -anchor nw -padx 4 -expand 0 -fill x
    checkbutton $type_frame.enable -text "Type" \
        -variable Apol_File_Contexts::opts(useType)
    set widgets(type) [ComboBox $type_frame.box -width 8 -autopost 1 \
                           -textvariable Apol_File_Contexts::opts(type) \
                           -helptext "Type or select a type"]
    set type_regex [checkbutton $type_frame.regex \
                        -variable Apol_File_Contexts::opts(useTypeRegex) \
                        -text "Regular expression"]
    trace add variable Apol_File_Contexts::opts(useType) write \
        [list Apol_File_Contexts::toggleEnable $widgets(type) -entrybg $type_regex]
    pack $type_frame.enable -side top -anchor nw
    pack $widgets(type) -side top -anchor nw -padx 4 -expand 0 -fill x
    pack $type_regex -side top -anchor nw -padx 4
    set range_cb [checkbutton $range_frame.enable \
                      -variable Apol_File_Contexts::opts(useRange) -text "MLS range"]
    set range_entry [entry $range_frame.range -width 12 \
                         -textvariable Apol_File_Contexts::opts(range)]
    set range_regex [checkbutton $range_frame.regex \
                         -variable Apol_File_Contexts::opts(useRangeRegex) \
                         -text "Regular expression"]
    trace add variable Apol_File_Contexts::opts(useRange) write \
        [list Apol_File_Contexts::toggleEnable $range_entry -bg $range_regex]
    trace add variable Apol_File_Contexts::opts(fc_is_mls) write \
        [list Apol_File_Contexts::toggleRange $range_cb $range_entry $range_regex]
    pack $range_cb -side top -anchor nw
    pack $range_entry -side top -anchor nw -padx 4 -expand 0 -fill x
    pack $range_regex -side top -anchor nw -padx 4
    checkbutton $path_frame.enable \
        -variable Apol_File_Contexts::opts(usePath) -text "File path"
    set path_entry [entry $path_frame.path -width 24 -textvariable Apol_File_Contexts::opts(path)]
    set path_regex [checkbutton $path_frame.regex \
                        -variable Apol_File_Contexts::opts(usePathRegex) \
                        -text "Regular expression"]
    trace add variable Apol_File_Contexts::opts(usePath) write \
        [list Apol_File_Contexts::toggleEnable $path_entry -bg $path_regex]
    pack $path_frame.enable -side top -anchor nw
    pack $path_entry -side top -anchor nw -padx 4 -expand 0 -fill x
    pack $path_regex -side top -anchor nw -padx 4
    set action_buttons [ButtonBox $options_frame.bb -orient vertical -homogeneous 1 -pady 2]
    $action_buttons add -text OK -width 6 -command {Apol_File_Contexts::search_fc_database}
    $action_buttons add -text "Info" -width 6 -command {Apol_File_Contexts::display_analysis_info}
    grid $action_buttons -row 0 -column 6 -padx 5 -pady 5 -sticky ne
    grid columnconfigure $options_frame 6 -weight 1
    set results_frame [TitleFrame $results_pane.results -text "Matching Files"]
    set widgets(results) [Apol_Widget::makeSearchResults [$results_frame getframe].results]
    pack $widgets(results) -expand yes -fill both
    pack $results_frame -expand yes -fill both -padx 2
    initialize
    return $frame
}
proc Apol_File_Contexts::changeStatusLabel {label1 label2 name1 name2 opt} {
    variable opts
    if {$opts(indexFilename) == ""} {
        set opts(statusText) "No Index File Loaded"
        $label1 configure -fg red
        set opts(statusText2) {}
    } else {
        set opts(statusText) $opts(indexFilename)
        $label1 configure -fg black
        if {$opts(fc_is_mls)} {
            set opts(statusText2) "Database contexts include MLS ranges."
            $label2 configure -fg black
        } else {
            set opts(statusText2) "Database contexts do not include MLS ranges."
            $label2 configure -fg red
        }
    }
}
proc Apol_File_Contexts::toggleEnable {entry bgcmd regex name1 name2 op} {
    variable opts
    if {$opts($name2)} {
        $entry configure -state normal $bgcmd white
        catch {$regex configure -state normal}
    } else {
        $entry configure -state disabled $bgcmd $ApolTop::default_bg_color
        catch {$regex configure -state disabled}
    }
}
proc Apol_File_Contexts::toggleRange {cb entry regex name1 name2 op} {
    variable opts
    if {$opts(fc_is_mls)} {
        $cb configure -state normal
        if {$opts(useRange)} {
            $entry configure -state normal -bg white
            $regex configure -state normal
        }
    } else {
        $cb configure -state disabled
        $entry configure -state disabled -bg $ApolTop::default_bg_color
        $regex configure -state disabled
    }
}
namespace eval Apol_Cond_Bools {
    variable cond_bools_list {}
    variable cond_bools_defaults
    variable cond_bools_values
    variable opts
    variable widgets
}
proc Apol_Cond_Bools::search_bools {} {
    variable opts
    variable widgets
    Apol_Widget::clearSearchResults $widgets(results)
    if {![ApolTop::is_policy_open]} {
        tk_messageBox -icon error -type ok -title "Error" -message "No current policy file is opened!"
        return
    }
    set name [string trim $opts(name)]
    if {$opts(enable_bool) && $name == {}} {
        tk_messageBox -icon error -type ok -title "Error" -message "No boolean variable provided!"
        return
    }
    set results {}
    if {[catch {apol_GetBools $name $opts(use_regexp)} bools_data]} {
        tk_messageBox -icon error -type ok -title "Error" -message "Error obtaining booleans list:\n$bools_data"
        return
    }
    set results "BOOLEANS:\n"
    if {[llength $bools_data] == 0} {
        append results "Search returned no results."
    } else {
        foreach b [lsort -index 0 $bools_data] {
            append results "\n[renderBool $b $opts(show_default) $opts(show_current)]"
        }
    }
    Apol_Widget::appendSearchResultText $widgets(results) $results
}
proc Apol_Cond_Bools::renderBool {bool_datum show_default show_current} {
    variable cond_bools_defaults
    foreach {bool_name cur_value} $bool_datum {break}
    set text [format "%-28s" $bool_name]
    if {$show_default} {
        if {$cond_bools_defaults($bool_name)} {
            append text "  Default State: True "
        } else {
            append text "  Default State: False"
        }
    }
    if {$show_current} {
        if {$cur_value} {
            append text "  Current State: True "
        } else {
            append text "  Current State: False"
        }
    }
    return $text
}
proc Apol_Cond_Bools::reset_bools {} {
    variable cond_bools_defaults
    variable cond_bools_values
    array set cond_bools_values [array get cond_bools_defaults]
}
proc Apol_Cond_Bools::set_bool_value {name1 name2 op} {
    variable cond_bools_values
    apol_SetBoolValue $name2 $cond_bools_values($name2)
}
proc Apol_Cond_Bools::insert_listbox_item {bool_datum} {
    variable widgets
    variable cond_bools_values
    set subf [$widgets(listbox) getframe]
    foreach {bool_name bool_value} $bool_datum {break}
    set cond_bools_values($bool_name) $bool_value
    set rb_true [radiobutton $subf.t:$bool_name -bg white \
                     -variable Apol_Cond_Bools::cond_bools_values($bool_name) \
                     -value 1 -highlightthickness 0 -text "True"]
    set rb_false [radiobutton $subf.f:$bool_name -bg white \
                      -variable Apol_Cond_Bools::cond_bools_values($bool_name) \
                      -value 0 -highlightthickness 0 -text "False"]
    trace add variable Apol_Cond_Bools::cond_bools_values($bool_name) write \
        [list Apol_Cond_Bools::set_bool_value]
    set rb_label [label $subf.l:$bool_name -bg white -text "- $bool_name"]
    grid $rb_true $rb_false $rb_label -padx 2 -pady 5 -sticky w
}
proc Apol_Cond_Bools::search { str case_Insensitive regExpr srch_Direction } {
    variable widgets
    ApolTop::textSearch $widgets(results).tb $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_Cond_Bools::goto_line { line_num } {
    variable widgets
    Apol_Widget::gotoLineSearchResults $widgets(results) $line_num
}
proc Apol_Cond_Bools::set_Focus_to_Text {} {
    focus $Apol_Cond_Bools::widgets(results)
}
proc Apol_Cond_Bools::open {} {
    variable cond_bools_list {}
    variable cond_bools_defaults
    variable widgets
    foreach bool_datum [lsort [apol_GetBools {} 0]] {
        foreach {name value} $bool_datum {break}
        lappend cond_bools_list $name
        set cond_bools_defaults($name) $value
        insert_listbox_item $bool_datum
    }
    $widgets(listbox) xview moveto 0
    $widgets(listbox) yview moveto 0
    $widgets(listbox) configure -areaheight 0 -areawidth 0
    $widgets(combo_box) configure -values $cond_bools_list
}
proc Apol_Cond_Bools::close { } {
    variable widgets
    variable cond_bools_list {}
    variable cond_bools_defaults
    variable cond_bools_values
    initializeVars
    $widgets(combo_box) configure -values {}
    foreach w [winfo children [$widgets(listbox) getframe]] {
        destroy $w
    }
    [$widgets(listbox) getframe] configure -width 1 -height 1
    Apol_Widget::clearSearchResults $widgets(results)
    array unset cond_bools_defaults
    array unset cond_bools_values
}
proc Apol_Cond_Bools::initializeVars {} {
    variable opts
    array set opts {
        enable_bool 0
        name ""
        use_regexp 0
        show_default 1
        show_current 1
    }
}
proc Apol_Cond_Bools::create {nb} {
    variable opts
    variable widgets
    initializeVars
    set frame [$nb insert end $ApolTop::cond_bools_tab -text "Booleans"]
    set pw [PanedWindow $frame.pw -side top]
    set left_pane [$pw add -weight 0]
    set right_pane [$pw add -weight 1]
    pack $pw -expand 1 -fill both
    set cond_bools_box [TitleFrame $left_pane.cond_bools_box -text "Booleans"]
    set s_optionsbox   [TitleFrame $right_pane.obox -text "Search Options"]
    set rslts_frame    [TitleFrame $right_pane.rbox -text "Search Results"]
    pack $cond_bools_box -expand 1 -fill both
    pack $s_optionsbox -padx 2 -fill x -expand 0
    pack $rslts_frame -padx 2 -fill both -expand yes
    set left_frame [$cond_bools_box getframe]
    set sw_b [ScrolledWindow $left_frame.sw -auto both]
    set widgets(listbox) [ScrollableFrame $sw_b.listbox -bg white -width 200]
    $sw_b setwidget $widgets(listbox)
    set button_defaults [button $left_frame.button_defaults \
                             -text "Reset to Policy Defaults" \
                             -command Apol_Cond_Bools::reset_bools]
    pack $sw_b -side top -expand 1 -fill both
    pack $button_defaults -side bottom -pady 2 -expand 0 -fill x
    set ofm [$s_optionsbox getframe]
    set bool_frame [frame $ofm.bool]
    set show_frame [frame $ofm.show]
    pack $bool_frame $show_frame -side left -padx 4 -pady 2 -anchor nw
    set enable [checkbutton $bool_frame.enable \
                    -variable Apol_Cond_Bools::opts(enable_bool) \
                    -text "Boolean"]
    set widgets(combo_box) [ComboBox $bool_frame.combo_box \
                                -textvariable Apol_Cond_Bools::opts(name) \
                                -helptext "Type or select a boolean variable" \
                                -state disabled -entrybg white -autopost 1]
    set widgets(regexp) [checkbutton $bool_frame.regexp \
                             -text "Search using regular expression" \
                             -state disabled \
                             -variable Apol_Cond_Bools::opts(use_regexp)]
    trace add variable Apol_Cond_Bools::opts(enable_bool) write \
        [list Apol_Cond_Bools::toggleSearchBools]
    pack $enable -anchor w
    pack $widgets(combo_box) $widgets(regexp) -padx 4 -anchor nw -expand 0 -fill x
    set show_default [checkbutton $show_frame.show_default \
                           -variable Apol_Cond_Bools::opts(show_default) \
                          -text "Show default state"]
    set show_current [checkbutton $show_frame.show_current \
                        -variable Apol_Cond_Bools::opts(show_current) \
                        -text "Show current state"]
    pack $show_default $show_current -anchor w
    set ok_button [button $ofm.ok -text "OK" -width 6 \
                       -command Apol_Cond_Bools::search_bools]
    pack $ok_button -side right -anchor ne -padx 5 -pady 5
    set widgets(results) [Apol_Widget::makeSearchResults [$rslts_frame getframe].results]
    pack $widgets(results) -expand yes -fill both
    return $frame
}
proc Apol_Cond_Bools::toggleSearchBools {name1 name2 op} {
    variable opts
    variable widgets
    if {$opts(enable_bool)} {
        $widgets(combo_box) configure -state normal
        $widgets(regexp) configure -state normal
    } else {
        $widgets(combo_box) configure -state disabled
        $widgets(regexp) configure -state disabled
    }
}
namespace eval Apol_Cond_Rules {
    variable vals
    variable widgets
}
proc Apol_Cond_Rules::cond_rules_search {} {
    variable vals
    variable widgets
    Apol_Widget::clearSearchResults $widgets(results)
    if {![ApolTop::is_policy_open]} {
        tk_messageBox -icon error -type ok -title "Error" -message "No current policy file is opened!"
        return
    }
    set rule_selection {}
    foreach {key value} [array get vals rs:*] {
        if {$value} {
            lappend rule_selection [string range $key 3 end]
        }
    }
    if {$rule_selection == {}} {
            tk_messageBox -icon error -type ok -title "Error" -message "At least one rule must be selected."
            return
    }
    set other_opts {}
    if {$vals(use_regexp)} {
        lappend other_opts regex
    }
    set bool_name {}
    if {$vals(enable_bool)} {
        if {[set bool_name $vals(name)] == {}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No booleean selected."
            return
        }
    }
    if {[catch {apol_SearchConditionalRules $rule_selection $other_opts $bool_name} results]} {
        tk_messageBox -icon error -type ok -title "Error" -message "Error searching conditionals:\n$results"
        return
    }
    if {[llength $results] == 0} {
        set text "Search returned no results."
    } else {
        set text "[llength $results] conditional"
        if {[llength $results] != 1} {
            append text s
        }
        append text " match the search criteria.  Expressions are in Reverse Polish Notation.\n\n"
    }
    Apol_Widget::appendSearchResultText $widgets(results) $text
    set counter 1
    foreach r [lsort -index 0 $results] {
        renderConditional $r $counter
        Apol_Widget::appendSearchResultText $widgets(results) "\n\n"
        incr counter
    }
}
proc Apol_Cond_Rules::renderConditional {cond cond_number} {
    variable widgets
    foreach {cond_expr true_list false_list} $cond {break}
    set text "conditional expression $cond_number: \[ [join $cond_expr] \]\n"
    append text "\nTRUE list:\n"
    Apol_Widget::appendSearchResultText $widgets(results) $text
    if {![ApolTop::is_capable "syntactic rules"]} {
        Apol_Widget::appendSearchResultAVRules $widgets(results) 4 [lindex $true_list 0]
        Apol_Widget::appendSearchResultTERules $widgets(results) 4 [lindex $true_list 1]
    } else {
        set syn_avrules [apol_GetSynAVRules [lindex $true_list 0] {}]
        Apol_Widget::appendSearchResultSynAVRules $widgets(results) 4 $syn_avrules
        set syn_terules [apol_GetSynTERules [lindex $true_list 1]]
        Apol_Widget::appendSearchResultSynTERules $widgets(results) 4 $syn_terules
    }
    Apol_Widget::appendSearchResultText $widgets(results) "\nFALSE list:\n"
    if {![ApolTop::is_capable "source"]} {
        Apol_Widget::appendSearchResultAVRules $widgets(results) 4 [lindex $false_list 0]
        Apol_Widget::appendSearchResultTERules $widgets(results) 4 [lindex $false_list 1]
    } else {
        set syn_avrules [apol_GetSynAVRules [lindex $false_list 0] {}]
        Apol_Widget::appendSearchResultSynAVRules $widgets(results) 4 $syn_avrules
        set syn_terules [apol_GetSynTERules [lindex $false_list 1]]
        Apol_Widget::appendSearchResultSynTERules $widgets(results) 4 $syn_terules
    }
}
proc Apol_Cond_Rules::search { str case_Insensitive regExpr srch_Direction } {
    variable widgets
    ApolTop::textSearch $widgets(results).tb $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_Cond_Rules::goto_line { line_num } {
    variable widgets
    Apol_Widget::gotoLineSearchResults $widgets(results) $line_num
}
proc Apol_Cond_Rules::set_Focus_to_Text {} {
    focus $Apol_RBAC::widgets(results)
}
proc Apol_Cond_Rules::open { } {
    variable widgets
    $widgets(combo_box) configure -values $Apol_Cond_Bools::cond_bools_list
}
proc Apol_Cond_Rules::close { } {
    variable widgets
    initializeVars
    $widgets(combo_box) configure -values {}
    Apol_Widget::clearSearchResults $widgets(results)
}
proc Apol_Cond_Rules::initializeVars {} {
    variable vals
    array set vals {
        rs:allow 1       rs:type_transition 1
        rs:auditallow 1  rs:type_member 1
        rs:dontaudit 1   rs:type_change 1
        enable_bool 0
        name {}
        use_regexp 0
    }
}
proc Apol_Cond_Rules::create {nb} {
    variable vals
    variable widgets
    initializeVars
    set frame [$nb insert end $ApolTop::cond_rules_tab -text "Conditional Expressions"]
    set topf [frame $frame.top]
    set bottomf [frame $frame.bottom]
    pack $topf -expand 0 -fill both -pady 2
    pack $bottomf -expand 1 -fill both -pady 2
    set rules_box [TitleFrame $topf.rules_box -text "Rule Selection"]
    set obox [TitleFrame $topf.obox -text "Search Options"]
    set dbox [TitleFrame $bottomf.dbox -text "Conditional Expressions Display"]
    pack $rules_box -side left -expand 0 -fill both -padx 2
    pack $obox -side left -expand 1 -fill both -padx 2
    pack $dbox -expand 1 -fill both -padx 2
    set fm_rules [$rules_box getframe]
    set allow [checkbutton $fm_rules.allow -text "allow" \
                   -variable Apol_Cond_Rules::vals(rs:allow)]
    set auditallow [checkbutton $fm_rules.auditallow -text "auditallow" \
                        -variable Apol_Cond_Rules::vals(rs:auditallow)]
    set dontaudit [checkbutton $fm_rules.dontaudit -text "dontaudit" \
                       -variable Apol_Cond_Rules::vals(rs:dontaudit)]
    set type_transition [checkbutton $fm_rules.type_transition -text "type_trans" \
                             -variable Apol_Cond_Rules::vals(rs:type_transition)]
    set type_member [checkbutton $fm_rules.type_member -text "type_member" \
                         -variable Apol_Cond_Rules::vals(rs:type_member)]
    set type_change [checkbutton $fm_rules.type_change -text "type_change" \
                         -variable Apol_Cond_Rules::vals(rs:type_change)]
    grid $allow $type_transition -sticky w -padx 2
    grid $auditallow $type_member -sticky w -padx 2
    grid $dontaudit $type_change -sticky w -padx 2
    set ofm [$obox getframe]
    set bool_frame [frame $ofm.bool]
    pack $bool_frame -side left -padx 4 -pady 2 -anchor nw
    set enable [checkbutton $bool_frame.enable \
                    -variable Apol_Cond_Rules::vals(enable_bool) \
                    -text "Boolean"]
    set widgets(combo_box) [ComboBox $bool_frame.combo_box \
                                -textvariable Apol_Cond_Rules::vals(name) \
                                -helptext "Type or select a boolean variable" \
                                -state disabled -entrybg white -autopost 1]
    set widgets(regexp) [checkbutton $bool_frame.regexp \
                             -text "Search using regular expression" \
                             -state disabled \
                             -variable Apol_Cond_Rules::vals(use_regexp)]
    trace add variable Apol_Cond_Rules::vals(enable_bool) write \
        [list Apol_Cond_Rules::toggleSearchBools]
    pack $enable -anchor w
    pack $widgets(combo_box) $widgets(regexp) -padx 4 -anchor nw -expand 0 -fill x
    set ok_button [button $ofm.ok -text OK -width 6 \
                       -command Apol_Cond_Rules::cond_rules_search]
    pack $ok_button -side right -anchor ne -padx 5 -pady 5
    set widgets(results) [Apol_Widget::makeSearchResults [$dbox getframe].results]
    pack $widgets(results) -expand yes -fill both
    return $frame
}
proc Apol_Cond_Rules::toggleSearchBools {name1 name2 op} {
    variable vals
    variable widgets
    if {$vals(enable_bool)} {
        $widgets(combo_box) configure -state normal
        $widgets(regexp) configure -state normal
    } else {
        $widgets(combo_box) configure -state disabled
        $widgets(regexp) configure -state disabled
    }
}
namespace eval Apol_Class_Perms {
    variable class_list {}
    variable common_perms_list {}
    variable perms_list {}
    variable opts
    variable widgets
}
proc Apol_Class_Perms::open { } {
    variable class_list {}
    foreach class [lsort -index 0 [apol_GetClasses {} 0]] {
        lappend class_list [lindex $class 0]
    }
    variable common_perms_list {}
    foreach common [lsort -index 0 [apol_GetCommons {} 0]] {
        lappend common_perms_list [lindex $common 0]
    }
    variable perms_list {}
    foreach perm [lsort -index 0 [apol_GetPerms {} 0]] {
        lappend perms_list [lindex $perm 0]
    }
}
proc Apol_Class_Perms::close { } {
    variable class_list {}
    variable common_perms_list {}
    variable perms_list {}
    variable widgets
    initializeVars
    Apol_Widget::clearSearchResults $widgets(results)
}
proc Apol_Class_Perms::initializeVars {} {
    variable opts
    array set opts {
        classes:show 1  classes:perms 1  classes:commons 1
        commons:show 0  commons:perms 1  commons:classes 1
        perms:show 0    perms:classes 1  perms:commons 1
    }
}
proc Apol_Class_Perms::set_Focus_to_Text {} {
    focus $Apol_Class_Perms::widgets(results)
}
proc Apol_Class_Perms::goto_line { line_num } {
    variable widgets
    Apol_Widget::gotoLineSearchResults $widgets(results) $line_num
}
proc Apol_Class_Perms::popupInfo {which name} {
    if {$which == "class"} {
        set text [renderClass [lindex [apol_GetClasses $name] 0] 1 0]
    } elseif {$which == "common"} {
        set text [renderCommon [lindex [apol_GetCommons $name] 0] 1 0]
    } else {
        set text [renderPerm [lindex [apol_GetPerms $name] 0] 1 1]
    }
    Apol_Widget::showPopupText $name $text
}
proc Apol_Class_Perms::search { str case_Insensitive regExpr srch_Direction } {
    variable widgets
    ApolTop::textSearch $widgets(results).tb $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_Class_Perms::search_Class_Perms {} {
    variable opts
    variable widgets
    Apol_Widget::clearSearchResults $widgets(results)
    if {![ApolTop::is_policy_open]} {
        tk_messageBox -icon error -type ok -title "Error" -message "No current policy file is opened!"
        return
    }
    if {!$opts(classes:show) && !$opts(commons:show) && !$opts(perms:show)} {
        tk_messageBox -icon error -type ok -title "Error" -message "No search options provided!"
        return
    }
    set use_regexp [Apol_Widget::getRegexpEntryState $widgets(regexp)]
    set regexp [Apol_Widget::getRegexpEntryValue $widgets(regexp)]
    if {$use_regexp} {
        if {$regexp == {}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No regular expression provided."
            return
        }
    } else {
        set regexp {}
    }
    set results {}
    if {$opts(classes:show)} {
        if {[set classes_perms $opts(classes:perms)]} {
            set classes_commons $opts(classes:commons)
        } else {
            set classes_commons 0
        }
        if {[catch {apol_GetClasses $regexp $use_regexp} classes_data]} {
            tk_messageBox -icon error -type ok -title Error -message "Error obtaining classes list:\n$classes_data"
            return
        }
        append results "OBJECT CLASSES:\n"
        if {$classes_data == {}} {
            append results "Search returned no results.\n"
        } else {
            foreach c [lsort -index 0 $classes_data] {
                append results [renderClass $c $opts(classes:perms) $classes_commons]
            }
        }
    }
    if {$opts(commons:show)} {
        if {[catch {apol_GetCommons $regexp $use_regexp} commons_data]} {
            tk_messageBox -icon error -type ok -title Error -message "Error obtaining common permissions list:\n$commons_data"
            return
        }
        append results "\nCOMMON PERMISSIONS:  \n"
        if {$commons_data == {}} {
            append results "Search returned no results.\n"
        } else {
            foreach c [lsort -index 0 $commons_data] {
                append results [renderCommon $c $opts(commons:perms) $opts(commons:classes)]
            }
        }
    }
    if {$opts(perms:show)} {
        if {[catch {apol_GetPerms $regexp $use_regexp} perms_data]} {
            tk_messageBox -icon error -type ok -title Error -message "Error obtaining permissions list:\n$perms_data"
            return
        }
        append results "\nPERMISSIONS"
        if {$opts(perms:classes)} {
            append results "  (* means class uses permission via a common permission)"
        }
        append results ":\n"
        if {$perms_data == {}} {
            append results "Search returned no results.\n"
        } else {
            foreach p [lsort -index 0 $perms_data] {
                append results [renderPerm $p $opts(perms:classes) $opts(perms:commons)]
            }
        }
    }
    Apol_Widget::appendSearchResultText $widgets(results) [string trim $results]
}
proc Apol_Class_Perms::renderClass {class_datum show_perms expand_common} {
    foreach {class_name common_class perms_list} $class_datum {break}
    set text "$class_name\n"
    if {$show_perms} {
        foreach perm [lsort $perms_list] {
            append text "    $perm\n"
        }
        if {$common_class != {}} {
            append text "    $common_class  (common perm)\n"
            if {$expand_common} {
                foreach perm [lsort [lindex [apol_GetCommons $common_class] 0 1]] {
                    append text "        $perm\n"
                }
            }
        }
        append text \n
    }
    return $text
}
proc Apol_Class_Perms::renderCommon {common_datum show_perms show_classes} {
    foreach {common_name perms_list classes_list} $common_datum {break}
    set text "$common_name\n"
    if {$show_perms} {
        foreach perm [lsort $perms_list] {
            append text "    $perm\n"
        }
    }
    if {$show_classes} {
        append text "  Object classes that use this common permission:\n"
        foreach class [lsort $classes_list] {
            append text "      $class\n"
        }
    }
    if {$show_perms || $show_classes} {
        append text "\n"
    }
    return $text
}
proc Apol_Class_Perms::renderPerm {perm_datum show_classes show_commons} {
    foreach {perm_name classes_list commons_list} $perm_datum {break}
    set text "$perm_name\n"
    if {$show_classes} {
        append text "  object classes:\n"
        foreach common $commons_list {
            foreach class [lindex [apol_GetCommons $common] 0 2] {
                lappend classes_list ${class}*
            }
        }
        if {$classes_list == {}} {
            append text "    <none>\n"
        } else {
            foreach class [lsort -uniq $classes_list] {
                append text "    $class\n"
            }
        }
    }
    if {$show_commons} {
        append text "  common permissions:\n"
        if {$commons_list == {}} {
            append text "    <none>\n"
        } else {
            foreach common [lsort $commons_list] {
                append text "    $common\n"
            }
        }
    }
    if {$show_classes || $show_commons} {
        append text "\n"
    }
    return $text
}
proc Apol_Class_Perms::create {nb} {
    variable opts
    variable widgets
    initializeVars
    set frame [$nb insert end $ApolTop::class_perms_tab -text "Classes/Perms"]
    set pw1 [PanedWindow $frame.pw -side top]
    set left_pane   [$pw1 add -weight 0]
    set center_pane [$pw1 add -weight 1]
    set class_pane  [frame $left_pane.class]
    set common_pane [frame $left_pane.common]
    set perms_pane  [frame $left_pane.perms]
    set classes_box [TitleFrame $class_pane.tbox -text "Object Classes"]
    set common_box  [TitleFrame $common_pane.tbox -text "Common Permissions"]
    set perms_box   [TitleFrame $perms_pane.tbox -text "Permissions"]
    set options_box [TitleFrame $center_pane.obox -text "Search Options"]
    set results_box [TitleFrame $center_pane.rbox -text "Search Results"]
    pack $classes_box -fill both -expand yes
    pack $common_box -fill both -expand yes
    pack $perms_box -fill both -expand yes
    pack $options_box -padx 2 -fill both -expand 0
    pack $results_box -padx 2 -fill both -expand yes
    pack $pw1 -fill both -expand yes
    pack $class_pane $common_pane -expand 0 -fill both
    pack $perms_pane -expand 1 -fill both
    set class_listbox [Apol_Widget::makeScrolledListbox [$classes_box getframe].lb -height 8 -width 20 -listvar Apol_Class_Perms::class_list]
    Apol_Widget::setListboxCallbacks $class_listbox \
        {{"Display Object Class Info" {Apol_Class_Perms::popupInfo class}}}
    pack $class_listbox -fill both -expand yes
    set common_listbox [Apol_Widget::makeScrolledListbox [$common_box getframe].lb -height 5 -width 20 -listvar Apol_Class_Perms::common_perms_list]
    Apol_Widget::setListboxCallbacks $common_listbox \
        {{"Display Common Permission Class Info" {Apol_Class_Perms::popupInfo common}}}
    pack $common_listbox -fill both -expand yes
    set perms_listbox [Apol_Widget::makeScrolledListbox [$perms_box getframe].lb -height 10 -width 20 -listvar Apol_Class_Perms::perms_list]
    Apol_Widget::setListboxCallbacks $perms_listbox \
        {{"Display Permission Info" {Apol_Class_Perms::popupInfo perm}}}
    pack $perms_listbox -fill both -expand yes
    set ofm [$options_box getframe]
    set classesfm [frame $ofm.classes]
    set commonsfm [frame $ofm.commons]
    set permsfm [frame $ofm.perms]
    pack $classesfm $commonsfm $permsfm -side left -padx 4 -pady 2 -anchor ne
    set classes [checkbutton $classesfm.classes -text "Object classes" \
                     -variable Apol_Class_Perms::opts(classes:show)]
    set perms [checkbutton $classesfm.perms -text "Include perms" \
                   -variable Apol_Class_Perms::opts(classes:perms)]
    set commons [checkbutton $classesfm.commons -text "Expand common perms" \
                     -variable Apol_Class_Perms::opts(classes:commons)]
    trace add variable Apol_Class_Perms::opts(classes:show) write \
        [list Apol_Class_Perms::toggleCheckbuttons $perms $commons]
    trace add variable Apol_Class_Perms::opts(classes:perms) write \
        [list Apol_Class_Perms::toggleCheckbuttons $commons {}]
    pack $classes -anchor w
    pack $perms $commons -anchor w -padx 8
    set commons [checkbutton $commonsfm.commons -text "Common permissions" \
                     -variable Apol_Class_Perms::opts(commons:show)]
    set perms [checkbutton $commonsfm.perms2 -text "Include perms" \
                   -variable Apol_Class_Perms::opts(commons:perms) \
                   -state disabled]
    set classes [checkbutton $commonsfm.classes -text "Object classes" \
                     -variable Apol_Class_Perms::opts(commons:classes) \
                     -state disabled]
    trace add variable Apol_Class_Perms::opts(commons:show) write \
        [list Apol_Class_Perms::toggleCheckbuttons $perms $classes]
    pack $commons -anchor w
    pack $perms $classes -anchor w -padx 8
    set perms [checkbutton $permsfm.prems -text "Permissions" \
                   -variable Apol_Class_Perms::opts(perms:show)]
    set classes [checkbutton $permsfm.classes -text "Object classes" \
                     -variable Apol_Class_Perms::opts(perms:classes) \
                     -state disabled]
    set commons [checkbutton $permsfm.commons -text "Common perms" \
                     -variable Apol_Class_Perms::opts(perms:commons) \
                     -state disabled]
    trace add variable Apol_Class_Perms::opts(perms:show) write \
        [list Apol_Class_Perms::toggleCheckbuttons $classes $commons]
    pack $perms -anchor w
    pack $classes $commons -anchor w -padx 8
    set widgets(regexp) [Apol_Widget::makeRegexpEntry $ofm.regexp]
    pack $widgets(regexp) -side left -padx 2 -pady 2 -anchor ne
    set ok [button $ofm.ok -text OK -width 6 \
                -command Apol_Class_Perms::search_Class_Perms]
    pack $ok -side right -pady 5 -padx 5 -anchor ne
    set widgets(results) [Apol_Widget::makeSearchResults [$results_box getframe].results]
    pack $widgets(results) -expand yes -fill both
    return $frame
}
proc Apol_Class_Perms::toggleCheckbuttons {cb1 cb2 name1 name2 op} {
    variable opts
    variable widgets
    if {$opts($name2)} {
        $cb1 configure -state normal
        if {$name2 == "classes:show"} {
            if {$opts(classes:perms)} {
                $cb2 configure -state normal
            } else {
                $cb2 configure -state disabled
            }
        } elseif {$cb2 != {}} {
            $cb2 configure -state normal
        }
    } else {
        $cb1 configure -state disabled
        if {$cb2 != {}} {
            $cb2 configure -state disabled
        }
    }
    if {!$opts(classes:show) && !$opts(commons:show) && !$opts(perms:show)} {
        Apol_Widget::setRegexpEntryState $widgets(regexp) 0
    } else {
        Apol_Widget::setRegexpEntryState $widgets(regexp) 1
    }
}
namespace eval Apol_PolicyConf {
    variable textbox
}
proc Apol_PolicyConf::set_Focus_to_Text {} {
    focus $Apol_PolicyConf::textbox
    insertionMarkChanged
}
proc Apol_PolicyConf::create {nb} {
    variable textbox
    set frame [$nb insert end $ApolTop::policy_conf_tab -text "Policy Source"]
    set sw [ScrolledWindow $frame.sw -auto none]
    set textbox [text [$sw getframe].text -bg white -wrap none]
    $sw setwidget $textbox
    pack $sw -expand yes -fill both
    bind $textbox <<Insertion>> Apol_PolicyConf::insertionMarkChanged
    rename $textbox ::Apol_PolicyConf::real_text
    proc ::$textbox {cmd args} {
        switch -- $cmd {
            insert -
            delete { return }
            fakeinsert { set cmd insert }
            fakedelete { set cmd delete }
        }
        set retval [uplevel 1 ::Apol_PolicyConf::real_text $cmd $args]
        if {$cmd == "mark" && [string equal -length 10 $args "set insert"]} {
            event generate $Apol_PolicyConf::textbox <<Insertion>>
        }
        return $retval
    }
}
proc Apol_PolicyConf::insertionMarkChanged {} {
    set lpos [$Apol_PolicyConf::textbox index insert]
    foreach {line col} [split $lpos .] {break}
    set ApolTop::policyConf_lineno "Line $line"
}
proc Apol_PolicyConf::open {policy_path} {
    variable textbox
    $textbox fakedelete 0.0 end
    if {![ApolTop::is_capable "source"]} {
        $textbox fakeinsert end "The currently loaded policy is not a source policy."
    } else {
        set primary_file [lindex $policy_path 1]
        if {[catch {::open $primary_file r} f]} {
            $textbox fakeinsert end "$primary_file does not exist or could not be read by the user."
        } else {
            $textbox fakeinsert end [read $f]
            ::close $f
        }
    }
    $textbox see 0.0
    $textbox mark set insert 1.0
}
proc Apol_PolicyConf::close {} {
    variable textbox
    $textbox fakedelete 0.0 end
}
proc Apol_PolicyConf::search { str case_Insensitive regExpr srch_Direction } {
    variable textbox
    ApolTop::textSearch $textbox $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_PolicyConf::goto_line { line_num } {
    variable textbox
    ApolTop::goto_line $line_num $textbox
}
namespace eval Apol_Perms_Map {
    variable edit_dialog .apol_perms
    variable user_default_pmap_name  [file join "$::env(HOME)" ".apol_perm_mapping"]
    variable opts           ;# options for edit perm map dialog
    variable widgets
}
proc Apol_Perms_Map::loadPermMapFromFile {} {
    set pmap_name [tk_getOpenFile -title "Select Perm Map to Load" -parent .]
    if {$pmap_name != {}} {
        return [loadPermMap $pmap_name [file tail $pmap_name] 1]
    }
    return 0
}
proc Apol_Perms_Map::loadDefaultPermMap {} {
    if {![ApolTop::is_policy_open]} {
        tk_messageBox -icon error -type ok -title "Error" -message "No current policy file is opened!"
        return 0
    }
    variable user_default_pmap_name
    if {[file exists $user_default_pmap_name]} {
        set pmap_name $user_default_pmap_name
        set pmap_short "User Default Permission Map"
        set pmap_editable 1
    } else {
        set pmap_editable 0
        set policy_version [apol_GetPolicyVersionNumber]
        set pmap_name [apol_GetDefault_PermMap "apol_perm_mapping_ver${policy_version}"]
        if {$pmap_name == {}} {
            set pmap_name [apol_GetDefault_PermMap apol_perm_mapping]
            if {$pmap_name == {}} {
                 tk_messageBox -icon error -type ok -title "Error" \
                     -message "Could not locate system default perm map. You must explicitly load a perm map from file."
                return 0
            }
        }
        set pmap_short "System Default Permission Map (Read-Only)"
    }
    return [loadPermMap $pmap_name $pmap_short $pmap_editable]
}
proc Apol_Perms_Map::close {} {
    variable opts
    variable edit_dialog
    trace remove variable Apol_Perms_Map::opts(modified) write \
        Apol_Perms_Map::toggleSaveButtons
    trace remove variable Apol_Perms_Map::opts(is_saveable) write \
        Apol_Perms_Map::toggleSaveButtons
    destroy $edit_dialog
    array unset opts c:*
    array unset opts p:*
}
proc Apol_Perms_Map::is_pmap_loaded {} {
    return [apol_IsPermMapLoaded]
}
proc Apol_Perms_Map::editPermMappings {} {
    variable edit_dialog
    if {[winfo exists $edit_dialog]} {
        raise $edit_dialog
    } else {
        createEditDialog
        refreshEditDialog
    }
}
proc Apol_Perms_Map::loadPermMap {filename shortname saveable} {
    if {[catch {apol_LoadPermMap $filename} err]} {
        tk_messageBox -icon error -type ok \
            -title "Error Loading Permission Map File" -message $err
        return 0
    } elseif {$err != {}} {
        set len [llength [split $err "\n"]]
        if {$len > 5} {
            incr len -4
            set err [lrange [split $err "\n"] 0 3]
            lappend err "(plus $len more lines)"
            set err [join $err "\n"]
        }
        set message "The permission map has been loaded, but there were warnings:"
        tk_messageBox -icon warning -type ok \
            -title "Warning While Loading Permission Map" \
            -message "$message\n\n$err"
    }
    variable opts
    set opts(filename) $filename
    set opts(shortname) $shortname
    set opts(is_saveable) $saveable
    set opts(modified) 0
    variable edit_dialog
    if {[winfo exists $edit_dialog]} {
        refreshEditDialog
    }
    ApolTop::configure_edit_pmap_menu_item 1
    return 1
}
proc Apol_Perms_Map::createEditDialog {} {
    variable edit_dialog
    variable opts
    variable widgets
    set title "Edit Permissions Mappings: $opts(shortname)"
    Dialog $edit_dialog -parent . -separator 1 -title $title -modal none -cancel 3
    set topf [frame [$edit_dialog getframe].top]
    pack $topf -side top -expand 1 -fill both
    set classes_box [TitleFrame $topf.classes -text "Object Classes"]
    pack $classes_box -side left -padx 2 -pady 2 -expand 0 -fill y
    set widgets(classes) [Apol_Widget::makeScrolledListbox [$classes_box getframe].c \
                              -height 16 -width 30 -listvar Apol_Perms_Map::opts(classes)]
    bind $widgets(classes).lb <<ListboxSelect>> Apol_Perms_Map::refreshPermEdit
    pack $widgets(classes) -expand 1 -fill both
    set results_box [TitleFrame $topf.perms -text "Permission Mappings"]
    pack $results_box -side right -padx 2 -pady 2 -expand 1 -fill both
    set sw [ScrolledWindow [$results_box getframe].sw -auto both]
    set widgets(perms) [ScrollableFrame $sw.perms -bg white -width 450]
    $sw setwidget $widgets(perms)
    pack $sw -expand 1 -fill both
    set label_box [frame [$edit_dialog getframe].l]
    pack $label_box -side bottom -anchor center
    set widgets(l1) [label $label_box.l1 -fg red -text ""]
    set widgets(l2) [label $label_box.l2 -text ""]
    pack $widgets(l1) $widgets(l2) -side left
    $edit_dialog add -text "Save and Load Changes" -command Apol_Perms_Map::save -width -1
    $edit_dialog add -text "Save As..." -command Apol_Perms_Map::saveAs
    $edit_dialog add -text "Save As User Default" -command Apol_Perms_Map::saveAsDefault
    $edit_dialog add -text "Exit" -command Apol_Perms_Map::exitDialog
    trace add variable Apol_Perms_Map::opts(modified) write \
        Apol_Perms_Map::toggleSaveButtons
    trace add variable Apol_Perms_Map::opts(is_saveable) write \
        Apol_Perms_Map::toggleSaveButtons
    $edit_dialog draw
}
proc Apol_Perms_Map::refreshEditDialog {} {
    variable opts
    variable widgets
    array set opts {
        classes {}
    }
    if {[catch {apol_GetPermMap} perm_map]} {
        tk_messageBox -icon error -type ok \
            -title "Error Getting Permission Map" -message $perm_map
        return
    }
    set all_unmapped 1
    set class_index 0
    foreach class_tuple [lsort -index 0 $perm_map] {
        foreach {class perm_list} $class_tuple {break}
        set suffix {}
        set opts(c:$class) [lsort -index 0 $perm_list]
        foreach perm $opts(c:$class) {
            foreach {perm map weight} $perm {break}
            set opts(p:${class}:${perm}:map) $map
            set opts(p:${class}:${perm}:weight) $weight
            if {$map == "u"} {
                set suffix *
                set all_unmapped 0
            }
        }
        lappend opts(classes) "$class$suffix"
        if {$suffix != {}} {
            $widgets(classes).lb itemconfigure $class_index -foreground red
        }
        incr class_index
    }
    if {!$all_unmapped} {
        $widgets(l1) configure -text "*"
        $widgets(l2) configure -text " - Undefined permission mapping(s)"
    } else {
        $widgets(l1) configure -text ""
        $widgets(l2) configure -text ""
    }
    set opts(modified) $opts(modified)
    set opts(is_saveable) $opts(is_saveable)
}
proc Apol_Perms_Map::refreshPermEdit {} {
    variable opts
    variable widgets
    focus $widgets(classes).lb
    set perms [$widgets(perms) getframe]
    foreach w [winfo children $perms] {
        destroy $w
    }
    if {[set selection [$widgets(classes).lb curselection]] == {}} {
        return
    }
    set class [lindex $opts(classes) [lindex $selection 0]]
    set class [string trimright $class "*"]
    foreach perm $opts(c:$class) {
        foreach {perm map weight} $perm {break}
        if {$map != "u"} {
            set l [label $perms.$perm:l -text $perm -bg white -anchor w]
        } else {
            set l [label $perms.$perm:l -text "${perm}*" -fg red -bg white -anchor w]
        }
        set r [radiobutton $perms.$perm:r -text "Read" -value r -bg white \
                   -highlightthickness 0 \
                   -command [list Apol_Perms_Map::togglePermMap $class $perm] \
                   -variable Apol_Perms_Map::opts(p:${class}:${perm}:map)]
        set w [radiobutton $perms.$perm:w -text "Write" -value w -bg white \
                   -highlightthickness 0 \
                   -command [list Apol_Perms_Map::togglePermMap $class $perm] \
                   -variable Apol_Perms_Map::opts(p:${class}:${perm}:map)]
        set b [radiobutton $perms.$perm:b -text "Both" -value b -bg white \
                   -highlightthickness 0 \
                   -command [list Apol_Perms_Map::togglePermMap $class $perm] \
                   -variable Apol_Perms_Map::opts(p:${class}:${perm}:map)]
        set n [radiobutton $perms.$perm:n -text "None" -value n -bg white \
                   -highlightthickness 0 \
                   -command [list Apol_Perms_Map::togglePermMap $class $perm] \
                   -variable Apol_Perms_Map::opts(p:${class}:${perm}:map)]
        set l2 [label $perms.$perm:l2 -text "Weight:" -bg white -anchor e]
        set weight [spinbox $perms.$perm:weight -from 1 -to 10 -increment 1 \
                        -width 2 -bg white \
                        -command [list Apol_Perms_Map::togglePermMap $class $perm] \
                        -textvariable Apol_Perms_Map::opts(p:${class}:${perm}:weight)]
        grid $l $r $w $b $n $l2 $weight -padx 2 -sticky w -pady 4
        grid configure $l2 -ipadx 10
    }
    grid columnconfigure $perms 0 -minsize 100 -weight 1
    foreach i {1 2 3 4} {
        grid columnconfigure $perms $i -uniform 1 -weight 0
    }
    $widgets(perms) xview moveto 0
    $widgets(perms) yview moveto 0
}
proc Apol_Perms_Map::toggleSaveButtons {name1 name2 op} {
    variable opts
    variable widgets
    variable edit_dialog
    if {$opts(modified)} {
        if {$opts(is_saveable)} {
            $edit_dialog itemconfigure 0 -state normal
        } else {
            $edit_dialog itemconfigure 0 -state disabled
        }
        $edit_dialog itemconfigure 1 -state normal
        $edit_dialog itemconfigure 2 -state normal
    } else {
        $edit_dialog itemconfigure 0 -state disabled
        $edit_dialog itemconfigure 1 -state disabled
        $edit_dialog itemconfigure 2 -state disabled
    }
}
proc Apol_Perms_Map::togglePermMap {class perm} {
    variable opts
    set map $opts(p:${class}:${perm}:map)
    set weight $opts(p:${class}:${perm}:weight)
    if {[catch {apol_SetPermMap $class $perm $map $weight} err]} {
        tk_messageBox -icon error -type ok -title Error -message "Error setting permission map: $err"
    }
    set opts(modified) 1
}
proc Apol_Perms_Map::save {} {
    variable opts
    savePermMap $opts(filename) [file tail $opts(shortname)]
}
proc Apol_Perms_Map::saveAs {} {
    variable edit_dialog
    set pmap_name [tk_getSaveFile -title "Save Perm Map" -parent $edit_dialog]
    if {$pmap_name != {}} {
         savePermMap $pmap_name [file tail $pmap_name]
    }
}
proc Apol_Perms_Map::saveAsDefault {} {
    variable user_default_pmap_name
    variable opts
    savePermMap $user_default_pmap_name "User Default Permission Map"
}
proc Apol_Perms_Map::exitDialog {} {
    variable opts
    variable edit_dialog
    if {$opts(modified)} {
        set ans [tk_messageBox -icon question -type yesno -title "Exit Perm Map Editor" \
                     -parent $edit_dialog \
                     -message "There were unsaved changes to the perm map.  Exit without saving changes to the perm map?"]
        if {$ans == "no"} {
            return
        }
        foreach class $opts(classes) {
            set class [string trimright $class "*"]
            foreach perm $opts(c:$class) {
                foreach {perm map weight} $perm {break}
                if {[catch {apol_SetPermMap $class $perm $map $weight} err]} {
                    tk_messageBox -icon error -type ok -title Error -message "Error restoring permission map: $err"
                }
            }
        }
        set opts(modified) 0
    }
    Apol_Perms_Map::close  ;# invoke my close to remove traces and clear memory
}
proc Apol_Perms_Map::savePermMap {filename shortname} {
    variable opts
    variable edit_dialog
    if {[catch {apol_SavePermMap $filename} err]} {
        tk_messageBox -icon error -type ok -title Error -message "Error saving permission map: $err"
    } else {
        set opts(filename) $filename
        set opts(shortname) $shortname
        set opts(is_saveable) 1
        set opts(modified) 0
        set title "Edit Permissions Mappings: $opts(shortname)"
        $edit_dialog configure -title $title
        refreshEditDialog
        refreshPermEdit
    }
}
namespace eval Apol_Analysis {
    variable vals
    variable widgets
    variable tabs
}
proc Apol_Analysis::open {} {
    variable vals
    foreach m $vals(modules) {
        ${m}::open
    }
}
proc Apol_Analysis::close {} {
    variable vals
    variable widgets
    foreach m $vals(modules) {
        ${m}::close
    }
    reinitializeTabs
}
proc Apol_Analysis::set_Focus_to_Text { tab } {
}
proc Apol_Analysis::goto_line { line_num } {
    variable widgets
    variable tabs
    set curid [$widgets(results) raise]
    if {$curid != {}} {
        $tabs($curid:module)::gotoLine [$widgets(results) getframe $curid] \
            $line_num
    }
}
proc Apol_Analysis::search { str case_Insensitive regExpr srch_Direction } {
    variable widgets
    variable tabs
    set curid [$widgets(results) raise]
    if {$curid != {}} {
        $tabs($curid:module)::search [$widgets(results) getframe $curid] \
            $str $case_Insensitive $regExpr $srch_Direction
    }
}
proc Apol_Analysis::save_query_options {file_channel query_file} {
    variable widgets
    set m [$widgets(search_opts) raise]
    puts $file_channel $m
    ${m}::saveQuery $file_channel
}
proc Apol_Analysis::load_query_options {file_channel parentDlg} {
    variable vals
    variable widgets
    set line {}
    while {[gets $file_channel line] >= 0} {
        set line [string trim $line]
        if {$line == {} || [string index $line 0] == "#"} {
            continue
        }
        break
    }
    if {$line == {} || [set i [lsearch -exact $vals(modules) $line]] == -1} {
        tk_messageBox -icon error -type ok -title Error -message "The specified query is not a valid analysis module."
    }
    ${line}::loadQuery $file_channel
    $widgets(modules).lb selection clear 0 end
    set module [lindex $vals(modules) $i]
    $widgets(search_opts) raise $module
    $widgets(modules).lb selection set [lsearch $vals(module_names) $vals($module:name)]
}
proc Apol_Analysis::create {nb} {
    variable vals
    variable widgets
    set frame [$nb insert end $ApolTop::analysis_tab -text "Analysis"]
    set pw [PanedWindow $frame.pw -side left -weights extra]
    set topf [$pw add -weight 0]
    set bottomf [$pw add -weight 1]
    pack $pw -expand 1 -fill both
    set top_leftf [TitleFrame $topf.left -text "Analysis Type"]
    set opts_f [TitleFrame $topf.opts -text "Analysis Options"]
    set buttons_f [frame $topf.buttons]
    pack $top_leftf -side left -expand 0 -fill y -padx 2
    pack $opts_f -side left -expand 1 -fill both -padx 2
    pack $buttons_f -side right -expand 0 -anchor ne -padx 2
    set results_f [TitleFrame $bottomf.r -text "Analysis Results"]
    pack $results_f -expand 1 -fill both -padx 2
    set widgets(modules) [Apol_Widget::makeScrolledListbox [$top_leftf getframe].m \
                              -height 8 -width 24 -listvar Apol_Analysis::vals(module_names) -exportselection 0]
    $widgets(modules).lb selection set 0
    bind $widgets(modules).lb <<ListboxSelect>> Apol_Analysis::selectModule
    pack $widgets(modules) -expand 1 -fill both
    set widgets(search_opts) [PagesManager [$opts_f getframe].s]
    foreach m $vals(modules) {
        ${m}::create [$widgets(search_opts) add $m]
    }
    $widgets(search_opts) compute_size
    $widgets(search_opts) raise [lindex $vals(modules) 0]
    pack $widgets(search_opts) -expand 1 -fill both
    set widgets(new) [button $buttons_f.new -text "New Analysis" -width 12 \
                          -command [list Apol_Analysis::analyze new]]
    set widgets(update) [button $buttons_f.update -text "Update Analysis" -width 12 -state disabled \
                             -command [list Apol_Analysis::analyze update]]
    set widgets(reset) [button $buttons_f.reset -text "Reset Criteria" -width 12 \
                            -command Apol_Analysis::reset]
    set widgets(info) [button $buttons_f.info -text "Info" -width 12 \
                            -command Apol_Analysis::info]
    pack $widgets(new) $widgets(update) $widgets(reset) $widgets(info) \
        -side top -pady 5 -padx 5 -anchor ne
    set popupTab_Menu [menu .popup_analysis -tearoff 0]
    set tab_menu_callbacks \
        [list {"Close Tab" Apol_Analysis::deleteResults} \
             {"Rename Tab" Apol_Analysis::displayRenameTabDialog}]
    set widgets(results) [NoteBook [$results_f getframe].results]
    $widgets(results) bindtabs <Button-1> Apol_Analysis::switchTab
    $widgets(results) bindtabs <Button-3> \
        [list ApolTop::popup_Tab_Menu \
             %W %x %y $popupTab_Menu $tab_menu_callbacks]
    set close [button [$results_f getframe].close -text "Close Tab" \
                   -command Apol_Analysis::deleteCurrentResults]
    pack $widgets(results) -expand 1 -fill both -padx 4
    pack $close -expand 0 -fill x -padx 4 -pady 2
    reinitializeTabs
    return $frame
}
proc Apol_Analysis::registerAnalysis {mod_proc mod_name} {
    variable vals
    lappend vals(modules) $mod_proc
    lappend vals(module_names) $mod_name
    set vals($mod_proc:name) $mod_name
}
proc Apol_Analysis::createResultTab {short_name criteria} {
    variable widgets
    variable tabs
    set i $tabs(next_result_id)
    incr tabs(next_result_id)
    set m [$widgets(search_opts) raise]
    set id "results$i"
    set frame [$widgets(results) insert end $id -text "($i) $short_name"]
    $widgets(results) raise $id
    set tabs($id:module) $m
    set tabs($id:vals) $criteria
    return $frame
}
proc Apol_Analysis::setResultTabCriteria {criteria} {
    variable widgets
    variable tabs
    set id [$widgets(results) raise]
    if {$id != {}} {
        set tabs($id:vals) $criteria
    }
}
proc Apol_Analysis::selectModule {} {
    variable vals
    variable widgets
    variable tabs
    focus $widgets(modules).lb
    if {[set selection [$widgets(modules).lb curselection]] == {}} {
        return
    }
    set module [lindex $vals(modules) [lindex $selection 0]]
    $widgets(search_opts) raise $module
    set result_tab [$widgets(results) raise]
    if {$result_tab != {} && $tabs($result_tab:module) == $module} {
        $widgets(update) configure -state normal
    } else {
        $widgets(update) configure -state disabled
    }
}
proc Apol_Analysis::analyze {which_button} {
    variable vals
    variable widgets
    variable tabs
    set m [$widgets(search_opts) raise]
    set tabs(analyses_done) -1
    set tabs(analyses_text) "Performing $vals($m:name) Analysis..."
    ProgressDlg .analysis_busy -title "$vals($m:name) Analysis" \
        -type normal -stop {} -separator 1 -parent . -maximum 2 \
        -width [string length $tabs(analyses_text)] \
        -textvariable Apol_Analysis::tabs(analyses_text) \
        -variable Apol_Analysis::tabs(analyses_done)
    ApolTop::setBusyCursor
    update idletasks
    if {$which_button == "new"} {
        set retval [${m}::newAnalysis]
    } else {
        set f [$widgets(results) getframe [$widgets(results) raise]]
        if {[set retval [${m}::updateAnalysis $f]] != {}} {
            deleteCurrentResults
        }
    }
    ApolTop::resetBusyCursor
    destroy .analysis_busy
    focus -force .
    if {$retval != {}} {
        tk_messageBox -icon error -type ok -title Error -message "Error while performing analysis:\n\n$retval"
    }
    if {[$widgets(results) raise] == {}} {
        $widgets(update) configure -state disabled
    } else {
        $widgets(update) configure -state normal
    }
}
proc Apol_Analysis::reset {} {
    variable vals
    variable widgets
    set m [$widgets(search_opts) raise]
    ${m}::reset
}
proc Apol_Analysis::info {} {
    variable vals
    variable widgets
    set m [$widgets(search_opts) raise]
    Apol_Widget::showPopupParagraph $vals(${m}:name) [${m}::getInfo]
}
proc Apol_Analysis::reinitializeTabs {} {
    variable widgets
    variable tabs
    array set tabs {
        next_result_id 1
    }
    foreach p [$widgets(results) pages 0 end] {
        deleteResults $p
    }
}
proc Apol_Analysis::switchTab {pageID} {
    variable vals
    variable widgets
    variable tabs
    $widgets(update) configure -state normal
    if {[$Apol_TE::widgets(results) raise] == $pageID} {
        return
    }
    $widgets(results) raise $pageID
    set cur_search_opts [$widgets(search_opts) raise]
    set m $tabs($pageID:module)
    ${m}::switchTab $tabs($pageID:vals)
    $widgets(modules).lb selection clear 0 end
    $widgets(modules).lb selection set [lsearch $vals(module_names) $vals(${m}:name)]
    $widgets(search_opts) raise $m
}
proc Apol_Analysis::deleteResults {pageID} {
    variable widgets
    variable tabs
    set curpos [$widgets(results) index $pageID]
    $widgets(results) delete $pageID
    array unset tabs $pageID:*
    array unset tabs $pageID
    if {[set next_id [$widgets(results) pages $curpos]] != {}} {
        switchTab $next_id
    } elseif {$curpos > 0} {
        switchTab [$widgets(results) pages [expr {$curpos - 1}]]
    } else {
        $widgets(update) configure -state disabled
    }
}
proc Apol_Analysis::deleteCurrentResults {} {
    variable widgets
    if {[set curid [$widgets(results) raise]] != {}} {
        deleteResults $curid
    }
}
proc Apol_Analysis::displayRenameTabDialog {pageID} {
    variable widgets
    variable tabs
    set d [Dialog .apol_analysis_tab_rename -homogeneous 1 -spacing 2 -cancel 1 \
               -default 0 -modal local -parent . -place center -separator 1 \
               -side bottom -title "Rename Results Tab"]
    $d add -text "OK" -command [list $d enddialog "ok"]
    $d add -text "Cancel" -command [list $d enddialog "cancel"]
    set f [$d getframe]
    set l [label $f.l -text "Tab name:"]
    set tabs(tab:new_name) [$widgets(results) itemcget $pageID -text]
    set e [entry $f.e -textvariable Apol_Analysis::tabs(tab:new_name) -width 16 -bg white]
    pack $l $e -side left -padx 2
    set retval [$d draw]
    destroy $d
    if {$retval == "ok"} {
        $widgets(results) itemconfigure $pageID -text $tabs(tab:new_name)
    }
}
namespace eval Apol_Range_Dialog {
    variable dialog ""
    variable vars
}
proc Apol_Range_Dialog::getRange {{defaultRange {{{} {}}}} {parent .}} {
    variable dialog
    variable vars
    if {![winfo exists $dialog]} {
        _create_dialog $parent
    }
    set f [$dialog getframe]
    Apol_Widget::resetLevelSelectorToPolicy $f.low
    Apol_Widget::resetLevelSelectorToPolicy $f.high
    Apol_Widget::setLevelSelectorLevel $f.low [lindex $defaultRange 0]
    if {[llength $defaultRange] == 1} {
        set vars($dialog:highenable) 0
    } else {
        Apol_Widget::setLevelSelectorLevel $f.high [lindex $defaultRange 1]
        set vars($dialog:highenable) 1
    } 
    _high_enabled $dialog
    $dialog.bbox _redraw
    set retval [$dialog draw]
    if {$retval == -1 || $retval == 1} {
        return $defaultRange
    }
    _get_range $dialog
}
proc Apol_Range_Dialog::_create_dialog {parent} {
    variable dialog
    variable vars
    set dialog [Dialog .range_dialog -modal local -parent $parent \
                    -separator 1 -homogeneous 1 -title "Select Range"]
    array unset vars $dialog:*
    set f [$dialog getframe]
    set low_label [label $f.ll -text "Single level"]
    set low_level [Apol_Widget::makeLevelSelector $f.low 12]
    set high_cb [checkbutton $f.high_enable \
                     -text "High level" \
                     -variable Apol_Range_Dialog::vars($dialog:highenable) \
                     -command [list Apol_Range_Dialog::_high_enabled $dialog]]
    set high_level [Apol_Widget::makeLevelSelector $f.high 12]
    Apol_Widget::setLevelSelectorState $high_level 0
    grid $low_label $high_cb -sticky w
    grid $low_level $high_level -sticky ns
    grid columnconfigure $f 0 -weight 1 -uniform 1 -pad 4
    grid columnconfigure $f 1 -weight 1 -uniform 1 -pad 4
    $dialog add -text "Ok" -command [list Apol_Range_Dialog::_okay $dialog]
    $dialog add -text "Cancel"
}
proc Apol_Range_Dialog::_get_range {dialog} {
    variable vars
    set f [$dialog getframe]
    set low [Apol_Widget::getLevelSelectorLevel $f.low]
    if {!$vars($dialog:highenable)} {
        list $low
    } else {
        list $low [Apol_Widget::getLevelSelectorLevel $f.high]
    }
}
proc Apol_Range_Dialog::_okay {dialog} {
    set range [_get_range $dialog]
    set low [lindex $range 0]
    if {[llength $range] == 1} {
        set high $low
    } else {
        set high [lindex $range 1]
    }
    if {[catch {apol_IsValidRange [list $low $high]} val]} {
        tk_messageBox -icon error -type ok -title "Could Not Validate Range" \
            -message "The selected range is not valid.  The selected level is not part of the current policy."
    } elseif {$val == 0} {
        tk_messageBox -icon error -type ok -title "Invalid Range" \
            -message "The selected range is not valid.  The high level does not dominate the low level."
    } else {
        $dialog enddialog 0
    }
}
proc Apol_Range_Dialog::_high_enabled {dialog} {
    variable vars
    set f [$dialog getframe]
    if {$vars($dialog:highenable)} {
        $f.ll configure -text "Low level"
        Apol_Widget::setLevelSelectorState $f.high 1
    } else {
        $f.ll configure -text "Single level"
        Apol_Widget::setLevelSelectorState $f.high 0
    }
}
namespace eval Apol_Level_Dialog {
    variable dialog ""
    variable vars
}
proc Apol_Level_Dialog::getLevel {{defaultLevel {{} {}}} {parent .}} {
    variable dialog
    if {![winfo exists $dialog]} {
        _create_dialog $parent
    }
    set f [$dialog getframe]
    Apol_Widget::resetLevelSelectorToPolicy $f.level
    Apol_Widget::setLevelSelectorLevel $f.level $defaultLevel
    $dialog.bbox _redraw
    set retval [$dialog draw]
    if {$retval == -1 || $retval == 1} {
        return $defaultLevel
    }
    _get_level
}
proc Apol_Level_Dialog::_create_dialog {parent} {
    variable dialog
    variable vars
    set dialog [Dialog .level_dialog -modal local -parent $parent \
                    -separator 1 -homogeneous 1 -title "Select Level"]
    array unset vars $dialog:*
    set f [$dialog getframe]
    set label [label $f.ll -text "Level:"]
    set level [Apol_Widget::makeLevelSelector $f.level 12]
    pack $label -anchor w
    pack $level -expand 1 -fill both
    $dialog add -text "Ok" -command [list Apol_Level_Dialog::_okay $dialog]
    $dialog add -text "Cancel"
}
proc Apol_Level_Dialog::_get_level {} {
    variable dialog
    Apol_Widget::getLevelSelectorLevel [$dialog getframe].level
}
proc Apol_Level_Dialog::_okay {dialog} {
    set level [_get_level]
    if {[catch {apol_IsValidRange [list $level]} val]} {
        tk_messageBox -icon error -type ok -title "Could Not Validate Level" \
            -message "Could not validate selected level.  Make sure that the correct policy was loaded."
    } elseif {$val == 0} {
        tk_messageBox -icon error -type ok -title "Invalid Level" \
            -message "The selected level is not valid for the current policy."
    } else {
        $dialog enddialog 0
    }
}
namespace eval Apol_Range {
    variable widgets
    variable vals
}
proc Apol_Range::set_Focus_to_Text {} {
    focus $Apol_Range::widgets(results)
}
proc Apol_Range::open {} {
    variable vals
    variable widgets
    Apol_Widget::resetTypeComboboxToPolicy $widgets(source_type)
    Apol_Widget::resetTypeComboboxToPolicy $widgets(target_type)
    set vals(classes) $Apol_Class_Perms::class_list
    $widgets(classes) configure -bg white -state normal
}
proc Apol_Range::close {} {
    variable vals
    variable widgets
    Apol_Widget::clearTypeCombobox $widgets(source_type)
    Apol_Widget::clearTypeCombobox $widgets(target_type)
    set vals(classes) {}
    $widgets(classes) configure -bg $ApolTop::default_bg_color -state disabled
    Apol_Widget::clearRangeSelector $widgets(range)
    Apol_Widget::clearSearchResults $widgets(results)
    set vals(enable_source) 0
    set vals(enable_target) 0
}
proc Apol_Range::search { str case_Insensitive regExpr srch_Direction } {
    variable widgets
    ApolTop::textSearch $widgets(results).tb $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_Range::goto_line { line_num } {
    variable widgets
    Apol_Widget::gotoLineSearchResults $widgets(results) $line_num
}
proc Apol_Range::create {nb} {
    variable widgets
    variable vals
    set frame [$nb insert end $ApolTop::range_tab -text "Range Transition Rules"]
    set obox [TitleFrame $frame.obox -text "Search Options"]
    set dbox [TitleFrame $frame.dbox -text "Range Transition Rules Display"]
    pack $obox -fill x -expand 0 -padx 2 -pady 2
    pack $dbox -fill both -expand yes -padx 2 -pady 2
    set source_frame [frame [$obox getframe].source]
    set target_frame [frame [$obox getframe].target]
    set classes_frame [frame [$obox getframe].classes]
    pack $source_frame $target_frame $classes_frame -side left -padx 4 -pady 2 -expand 0 -anchor nw
    set vals(enable_source) 0
    set source_cb [checkbutton $source_frame.cb -text "Source type" \
                       -variable Apol_Range::vals(enable_source)]
    set widgets(source_type) [Apol_Widget::makeTypeCombobox $source_frame.tcb]
    Apol_Widget::setTypeComboboxState $widgets(source_type) 0
    trace add variable Apol_Range::vals(enable_source) write \
        [list Apol_Range::toggleTypeCombobox $widgets(source_type)]
    pack $source_cb -side top -expand 0 -anchor nw
    pack $widgets(source_type) -side top -expand 0 -anchor nw -padx 4
    set vals(enable_target) 0
    set target_cb [checkbutton $target_frame.cb -text "Target type" \
                       -variable Apol_Range::vals(enable_target)]
    set widgets(target_type) [Apol_Widget::makeTypeCombobox $target_frame.tcb]
    Apol_Widget::setTypeComboboxState $widgets(target_type) 0
    trace add variable Apol_Range::vals(enable_target) write \
        [list Apol_Range::toggleTypeCombobox $widgets(target_type)]
    pack $target_cb -side top -expand 0 -anchor nw
    pack $widgets(target_type) -side top -expand 0 -anchor nw -padx 4
    set l [label $classes_frame.l -text "Target Classes"]
    set sw [ScrolledWindow $classes_frame.sw -auto both]
    set widgets(classes) [listbox [$sw getframe].lb -height 5 -width 24 \
                              -highlightthickness 0 -selectmode multiple \
                              -exportselection 0 -state disabled \
                              -bg $ApolTop::default_bg_color \
                              -listvar Apol_Range::vals(classes)]
    $sw setwidget $widgets(classes)
    update
    grid propagate $sw 0
    pack $l $sw -side top -expand 0 -anchor nw
    set widgets(range) [Apol_Widget::makeRangeSelector [$obox getframe].range Rules]
    pack $widgets(range) -side left -padx 4 -pady 2 -expand 0 -anchor nw
    set ok [button [$obox getframe].ok -text "OK" -width 6 -command Apol_Range::searchRanges]
    pack $ok -side right -pady 5 -padx 5 -anchor ne
    set widgets(results) [Apol_Widget::makeSearchResults [$dbox getframe].results]
    pack $widgets(results) -expand yes -fill both
    return $frame
}
proc Apol_Range::searchRanges {} {
    variable vals
    variable widgets
    Apol_Widget::clearSearchResults $widgets(results)
    if {![ApolTop::is_policy_open]} {
        tk_messageBox -icon error -type ok -title "Error" -message "No current policy file is opened!"
        return
    }
    if {$vals(enable_source)} {
        set source [Apol_Widget::getTypeComboboxValue $widgets(source_type)]
        if {$source == {}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No source type provided."
            return
        }
    } else {
        set source {}
    }
    if {$vals(enable_target)} {
        set target [Apol_Widget::getTypeComboboxValue $widgets(target_type)]
        if {$target == {}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No target type provided."
            return
        }
    } else {
        set target {}
    }
    set classes {}
    foreach c [$widgets(classes) curselection] {
        lappend classes [$widgets(classes) get $c]
    }
    set range_enabled [Apol_Widget::getRangeSelectorState $widgets(range)]
    foreach {range range_match} [Apol_Widget::getRangeSelectorValue $widgets(range)] break
    if {$range_enabled} {
        if {$range == {{{} {}} {{} {}}}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No range provided!"
            return
        }
    } else {
        set range {}
    }
    if {[catch {apol_SearchRangeTransRules $source $target $classes $range $range_match} results]} {
        tk_messageBox -icon error -type ok -title "Error" -message "Error searching range transitions:\n$results"
        return
    }
    if {[llength $results] == 0} {
        set text "Search returned no results."
    } else {
        set text "[llength $results] rule"
        if {[llength $results] != 1} {
            append text s
        }
        append text " match the search criteria.\n\n"
    }
    Apol_Widget::appendSearchResultText $widgets(results) $text
    foreach r [lsort $results] {
        renderRangeTrans $r
    }
}
proc Apol_Range::renderRangeTrans {rule} {
    variable widgets
    foreach {source_set target_set target_class range} $rule {
        if {[llength $source_set] > 1} {
            set source_set "\{ $source_set \}"
        }
        if {[llength $target_set] > 1} {
            set target_set "\{ $target_set \}"
        }
        set text "$source_set $target_set : $target_class "
        set low [apol_RenderLevel [lindex $range 0]]
        set high [apol_RenderLevel [lindex $range 1]]
        if {$low == $high} {
            append text $low
        } else {
            append text "$low - $high"
        }
        Apol_Widget::appendSearchResultLine $widgets(results) 0 {} range_transition $text
    }
}
proc Apol_Range::toggleTypeCombobox {path name1 name2 op} {
    Apol_Widget::setTypeComboboxState $path $Apol_Range::vals($name2)
}
namespace eval Apol_MLS {
    variable widgets
    variable vals
}
proc Apol_MLS::set_Focus_to_Text {} {
    focus $Apol_MLS::widgets(results)
}
proc Apol_MLS::open {} {
    variable vals
    set vals(senslist) {}
    foreach s [lsort -integer -index 3 [apol_GetLevels {} 0]] {
        lappend vals(senslist) [lindex $s 0]
    }
    set vals(catslist) {}
    foreach c [lsort -integer -index 3 [apol_GetCats {} 0]] {
        lappend vals(catslist) [lindex $c 0]
    }
}
proc Apol_MLS::close {} {
    variable widgets
    initializeVars
    Apol_Widget::clearSearchResults $widgets(results)
}
proc Apol_MLS::initializeVars {} {
    variable vals
    array set vals {
        senslist {}      catslist {}
        enable_sens 1    show_cats_too 1
        enable_cats 0    show_sens_too 1
    }
}
proc Apol_MLS::search { str case_Insensitive regExpr srch_Direction } {
	variable widgets
	ApolTop::textSearch $widgets(results).tb $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_MLS::goto_line { line_num } {
    variable widgets
    Apol_Widget::gotoLineSearchResults $widgets(results) $line_num
}
proc Apol_MLS::create {nb} {
    variable widgets
    variable vals
    initializeVars
    set frame [$nb insert end $ApolTop::mls_tab -text "MLS"]
    set pw [PanedWindow $frame.pw -side top -weights extra]
    set leftf [$pw add -weight 0]
    set rightf [$pw add -weight 1]
    pack $pw -fill both -expand yes
    set sensbox [TitleFrame $leftf.sensbox -text "Sensitivities"]
    set catsbox [TitleFrame $leftf.catsbox -text "Categories"]
    pack $sensbox -fill both -expand 0
    pack $catsbox -fill both -expand yes
    set sensbox [Apol_Widget::makeScrolledListbox [$sensbox getframe].sens \
                     -height 10 -width 20 -listvar Apol_MLS::vals(senslist)]
    Apol_Widget::setListboxCallbacks $sensbox \
        {{"Show Sensitivity Info" {Apol_MLS::popupSensInfo}}}
    pack $sensbox -expand 1 -fill both
    set catsbox [Apol_Widget::makeScrolledListbox [$catsbox getframe].cats \
                     -height 16 -width 20 -listvar Apol_MLS::vals(catslist)]
    Apol_Widget::setListboxCallbacks $catsbox \
        {{"Show Category Info" {Apol_MLS::popupCatsInfo}}}
    pack $catsbox -expand 1 -fill both
    set optsbox [TitleFrame $rightf.optsbox -text "Search Options"]
    pack $optsbox -side top -expand 0 -fill both -padx 2
    set sensf [frame [$optsbox getframe].sensf]
    set catsf [frame [$optsbox getframe].catsf]
    pack $sensf $catsf -side left -padx 4 -pady 2 -anchor nw
    set enable_sens [checkbutton $sensf.enable -text "Sensitivities" \
                         -variable Apol_MLS::vals(enable_sens)]
    set show_cats [checkbutton $sensf.cats -text "Show levels (categories)" \
                       -variable Apol_MLS::vals(show_cats_too)]
    trace add variable Apol_MLS::vals(enable_sens) write \
        [list Apol_MLS::toggleCheckbutton $show_cats]
    pack $enable_sens -side top -anchor nw
    pack $show_cats -side top -anchor nw -padx 8
    set enable_cats [checkbutton $catsf.enable -text "Categories" \
                         -variable Apol_MLS::vals(enable_cats)]
    set show_sens [checkbutton $catsf.cats -text "Show sensitivities" \
                       -variable Apol_MLS::vals(show_sens_too) -state disabled]
    trace add variable Apol_MLS::vals(enable_cats) write \
        [list Apol_MLS::toggleCheckbutton $show_sens]
    pack $enable_cats -side top -anchor nw
    pack $show_sens -side top -anchor nw -padx 8
    set widgets(regexp) [Apol_Widget::makeRegexpEntry [$optsbox getframe].regexpf]
    pack $widgets(regexp) -side left -padx 4 -pady 2 -anchor nw
    set ok [button [$optsbox getframe].ok -text "OK" -width 6 \
                -command Apol_MLS::runSearch]
    pack $ok -side right -pady 5 -padx 5 -anchor ne
    set resultsbox [TitleFrame $rightf.resultsbox -text "Search Results"]
    pack $resultsbox -expand yes -fill both -padx 2
    set widgets(results) [Apol_Widget::makeSearchResults [$resultsbox getframe].results]
    pack $widgets(results) -side top -expand yes -fill both
    return $frame
}
proc Apol_MLS::toggleCheckbutton {path name1 name2 op} {
    variable vals
    variable widgets
    if {$vals($name2)} {
        $path configure -state normal
    } else {
        $path configure -state disabled
    }
    if {$vals(enable_sens) == 0 && $vals(enable_cats) == 0} {
        Apol_Widget::setRegexpEntryState $widgets(regexp) 0
    } else {
        Apol_Widget::setRegexpEntryState $widgets(regexp) 1
    }
}
proc Apol_MLS::popupSensInfo {sens} {
    set sens_datum [lindex [apol_GetLevels $sens] 0]
    Apol_Widget::showPopupText $sens [renderLevel $sens_datum 1]
}
proc Apol_MLS::popupCatsInfo {cats} {
    set cats_datum [lindex [apol_GetCats $cats] 0]
    Apol_Widget::showPopupText $cats [renderCats $cats_datum 1]
}
proc Apol_MLS::runSearch {} {
    variable vals
    variable widgets
    Apol_Widget::clearSearchResults $widgets(results)
    if {![ApolTop::is_policy_open]} {
        tk_messageBox -icon error -type ok -title "Error" -message "No current policy file is opened!"
        return
    }
    if {$vals(enable_sens) == 0 && $vals(enable_cats) == 0} {
        tk_messageBox -icon error -type ok -title "Error" -message "No search options provided!"
        return
    }
    set results ""
    set use_regexp [Apol_Widget::getRegexpEntryState $widgets(regexp)]
    if {$use_regexp} {
        set regexp [Apol_Widget::getRegexpEntryValue $widgets(regexp)]
        if {$regexp == {}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No regular expression provided."
            return
        }
    } else {
        set regexp {}
    }
    if {$vals(enable_sens)} {
        if {[catch {apol_GetLevels $regexp $use_regexp} level_data]} {
            tk_messageBox -icon error -type ok -title "Error" -message "Error obtaining sensitivities list:\n$level_data"
            return
        }
        append results "SENSITIVITIES (ordered by dominance from low to high):"
        if {[llength $level_data] == 0} {
            append results "\nSearch returned no results."
        } else {
            foreach l [lsort -integer -index 3 $level_data] {
                append results "\n[renderLevel $l $vals(show_cats_too)]"
            }
        }
    }
    if {$vals(enable_cats)} {
        if {$vals(enable_sens)} {
            append results "\n\n"
        }
        if {[catch {apol_GetCats $regexp $use_regexp} cats_data]} {
            tk_messageBox -icon error -type ok -title "Error" -message "Error obtaining categories list:\n$cats_data"
            return
        }
        append results "CATEGORIES (ordered by appearance within policy):"
        if {[llength $cats_data] == 0} {
            append results "\nSearch returned no results."
        } else {
            foreach c [lsort -integer -index 3 $cats_data] {
                append results "\n[renderCats $c $vals(show_sens_too)]"
            }
        }
    }
    Apol_Widget::appendSearchResultText $widgets(results) $results
}
proc Apol_MLS::renderLevel {level_datum show_level} {
    foreach {sens aliases cats dom_value} $level_datum {break}
    set text $sens
    if {[llength $aliases] > 0} {
        append text " alias \{$aliases\}"
    }
    if {$show_level} {
        append text " ([llength $cats] categor"
        if {[llength $cats] == 1} {
            append text "y)"
        } else {
            append text "ies)"
        }
        append text "\n    level [apol_RenderLevel [list $sens $cats]]\n"
    }
    return $text
}
proc Apol_MLS::renderCats {cats_datum show_sens} {
    foreach {cats aliases sens cat_value} $cats_datum {break}
    set text $cats
    if {[llength $aliases] > 0} {
        append text " alias \{$aliases\}"
    }
    append text "\n"
    if {$show_sens} {
        set sens_list {}
        foreach s $sens {
            lappend sens_list [lindex [apol_GetLevels $s] 0]
        }
        foreach s [lsort -integer -index 3 $sens_list] {
            append text "    [lindex $s 0]\n"
        }
    }
    return $text
}
namespace eval Apol_NetContexts {
    variable widgets
    variable vals
}
proc Apol_NetContexts::set_Focus_to_Text {} {
    focus $Apol_NetContexts::widgets(results)
}
proc Apol_NetContexts::open {} {
    variable vals
    portcon_open
    netifcon_open
    nodecon_open
    set vals(context_type) portcon
}
proc Apol_NetContexts::close {} {
    variable widgets
    initializeVars
    Apol_Widget::clearSearchResults $widgets(results)
    Apol_Widget::clearContextSelector $widgets(portcon:context)
    Apol_Widget::clearContextSelector $widgets(netifcon:ifcon)
    Apol_Widget::clearContextSelector $widgets(netifcon:msgcon)
    Apol_Widget::clearContextSelector $widgets(nodecon:context)
    $widgets(portcon:proto) configure -values {}
    $widgets(netifcon:dev) configure -values {}
}
proc Apol_NetContexts::initializeVars {} {
    variable vals
    array set vals {
        portcon:items {}
        portcon:proto_enable 0    portcon:proto {}
        portcon:port_enable 0     portcon:port 0
        portcon:hiport_enable 0   portcon:hiport 0
        netifcon:items {}
        netifcon:dev_enable 0     netifcon:dev {}
        nodecon:items {}
        nodecon:ip_type ipv4
        nodecon:ipv4_addr_enable 0
        nodecon:ipv4_addr0 0        nodecon:ipv4_addr1 0
        nodecon:ipv4_addr2 0        nodecon:ipv4_addr3 0
        nodecon:ipv4_mask_enable 0
        nodecon:ipv4_mask0 255      nodecon:ipv4_mask1 255
        nodecon:ipv4_mask2 255      nodecon:ipv4_mask3 255
        nodecon:ipv6_addr_enable 0  nodecon:ipv6_addr ::
        nodecon:ipv6_mask_enable 0  nodecon:ipv6_mask ::
        items {}
        context_type portcon
    }
}
proc Apol_NetContexts::search { str case_Insensitive regExpr srch_Direction } {
    variable widgets
    ApolTop::textSearch $widgets(results).tb $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_NetContexts::goto_line { line_num } {
    variable widgets
    Apol_Widget::gotoLineSearchResults $widgets(results) $line_num
}
proc Apol_NetContexts::create {nb} {
    variable widgets
    variable vals
    initializeVars
    set frame [$nb insert end $ApolTop::net_contexts_tab -text "Net Contexts"]
    set pw [PanedWindow $frame.pw -side top -weights extra]
    set leftf [$pw add -weight 0]
    set rightf [$pw add -weight 1]
    pack $pw -fill both -expand yes
    set context_box [TitleFrame $leftf.context_f -text "Context Type"]
    set context_f [$context_box getframe]
    radiobutton $context_f.portcon -text "portcon" -value portcon \
        -variable Apol_NetContexts::vals(context_type)
    radiobutton $context_f.netifcon -text "netifcon" -value netifcon \
        -variable Apol_NetContexts::vals(context_type)
    radiobutton $context_f.nodecon -text "nodecon" -value nodecon \
        -variable Apol_NetContexts::vals(context_type)
    trace add variable Apol_NetContexts::vals(context_type) write \
        {Apol_NetContexts::contextTypeChanged}
    pack $context_f.portcon $context_f.netifcon $context_f.nodecon \
        -anchor w -expand 0 -padx 4 -pady 5
    pack $context_box -anchor nw -expand 0 -fill x
    set widgets(items_tf) [TitleFrame $leftf.items_f -text "Port Contexts"]
    set widgets(items) [Apol_Widget::makeScrolledListbox [$widgets(items_tf) getframe].items \
                            -height 20 -width 20 -listvar Apol_NetContexts::vals(items)]
    Apol_Widget::setListboxCallbacks $widgets(items) \
        {{"Show Context Info" {Apol_NetContexts::popupContextInfo}}}
    pack $widgets(items) -expand 1 -fill both
    pack $widgets(items_tf) -expand 1 -fill both
    set optsbox [TitleFrame $rightf.optsbox -text "Search Options"]
    pack $optsbox -side top -expand 0 -fill both -padx 2
    set widgets(options_pm) [PagesManager [$optsbox getframe].pm]
    portcon_create [$widgets(options_pm) add portcon]
    netifcon_create [$widgets(options_pm) add netifcon]
    nodecon_create [$widgets(options_pm) add nodecon]
    $widgets(options_pm) compute_size
    pack $widgets(options_pm) -expand 1 -fill both -side left
    $widgets(options_pm) raise portcon
    set ok [button [$optsbox getframe].ok -text "OK" -width 6 \
                -command Apol_NetContexts::runSearch]
    pack $ok -side right -pady 5 -padx 5 -anchor ne
    set resultsbox [TitleFrame $rightf.resultsbox -text "Search Results"]
    pack $resultsbox -expand yes -fill both -padx 2
    set widgets(results) [Apol_Widget::makeSearchResults [$resultsbox getframe].results]
    pack $widgets(results) -side top -expand yes -fill both
    return $frame
}
proc Apol_NetContexts::popupContextInfo {value} {
    variable vals
    if {$vals(context_type) == "portcon"} {
        portcon_popup $value
    } elseif {$vals(context_type) == "netifcon"} {
        netifcon_popup $value
    } else {
        nodecon_popup $value
    }
}
proc Apol_NetContexts::contextTypeChanged {name1 name2 op} {
    variable vals
    variable widgets
    Apol_Widget::clearSearchResults $widgets(results)
    if {$vals(context_type) == "portcon"} {
        portcon_show
    } elseif {$vals(context_type) == "netifcon"} {
        netifcon_show
    } else {
        nodecon_show
    }
}
proc Apol_NetContexts::toggleCheckbutton {path name1 name2 op} {
    variable vals
    variable widgets
    if {$vals($name2)} {
        $path configure -state normal
    } else {
        $path configure -state disabled
    }
}
proc Apol_NetContexts::runSearch {} {
    variable vals
    variable widgets
    Apol_Widget::clearSearchResults $widgets(results)
    if {![ApolTop::is_policy_open]} {
        tk_messageBox -icon error -type ok -title "Error" -message "No current policy file is opened!"
        return
    }
    if {$vals(context_type) == "portcon"} {
        portcon_runSearch
    } elseif {$vals(context_type) == "netifcon"} {
        netifcon_runSearch
    } else {
        nodecon_runSearch
    }
}
proc Apol_NetContexts::portcon_open {} {
    variable vals
    variable widgets
    set vals(portcon:items) {}
    set protos {}
    foreach p [apol_GetPortcons -1 -1 {} {} 0] {
        foreach {low high proto context} $p {break}
        if {$low == $high} {
            lappend vals(portcon:items) $low
        } else {
            lappend vals(portcon:items) "$low-$high"
        }
        lappend protos $proto
    }
    set vals(portcon:items) [lsort -unique -dictionary $vals(portcon:items)]
    $widgets(portcon:proto) configure -values [lsort -unique -dictionary $protos]
}
proc Apol_NetContexts::portcon_show {} {
    variable vals
    variable widgets
    $widgets(items_tf) configure -text "Port Contexts"
    $widgets(options_pm) raise portcon
    set vals(items) $vals(portcon:items)
}
proc Apol_NetContexts::portcon_create {p_f} {
    variable widgets
    variable vals
    frame $p_f.proto
    set proto_cb [checkbutton $p_f.proto.proto_enable -text "Protocol" \
                      -variable Apol_NetContexts::vals(portcon:proto_enable)]
    set widgets(portcon:proto) [ComboBox $p_f.proto.proto -entrybg white -width 8 -state disabled \
                                    -textvariable Apol_NetContexts::vals(portcon:proto) -autopost 1]
    trace add variable Apol_NetContexts::vals(portcon:proto_enable) write \
        [list Apol_NetContexts::toggleCheckbutton $widgets(portcon:proto)]
    pack $proto_cb -side top -anchor w
    pack $widgets(portcon:proto) -side top -expand 0 -fill x -padx 4
    frame $p_f.port
    set low [frame $p_f.port.l]
    set port_cb [checkbutton $low.port_enable -text "Single Port" \
                     -variable Apol_NetContexts::vals(portcon:port_enable)]
    set widgets(portcon:port) [spinbox $low.port -bg white -width 8 \
                                   -justify right -state disabled \
                                   -from 0 -to 65535 \
                                   -validate all -vcmd [list Apol_NetContexts::portcon_limitPort %W %V %P port] \
                                   -textvariable Apol_NetContexts::vals(portcon:port)]
    set high [frame $p_f.port.h]
    set hiport_cb [checkbutton $high.hiport_enable -text "High Port" \
                       -state disabled \
                       -variable Apol_NetContexts::vals(portcon:hiport_enable)]
    set widgets(portcon:hiport) [spinbox $high.hiport -bg white -width 8 \
                                     -justify right -state disabled \
                                     -from 0 -to 65535 \
                                     -validate all -vcmd [list Apol_NetContexts::portcon_limitPort %W %V %P hiport] \
                                     -textvariable Apol_NetContexts::vals(portcon:hiport)]
    trace add variable Apol_NetContexts::vals(portcon:port_enable) write \
        [list Apol_NetContexts::portcon_toggleCheckbutton_lowport \
             $widgets(portcon:port) $hiport_cb $widgets(portcon:hiport)]
    trace add variable Apol_NetContexts::vals(portcon:hiport_enable) write \
        [list Apol_NetContexts::portcon_toggleCheckbutton_hiport $port_cb $widgets(portcon:hiport)]
    pack $port_cb -side top -anchor w -expand 0
    pack $widgets(portcon:port) -side top -expand 0 -fill x -padx 4
    pack $hiport_cb -side top -anchor w -expand 0
    pack $widgets(portcon:hiport) -side top -expand 0 -fill x -padx 4
    pack $low $high -side left -expand 0 -fill both
    frame $p_f.c
    set widgets(portcon:context) [Apol_Widget::makeContextSelector $p_f.c.context "Contexts"]
    pack $widgets(portcon:context)
    pack $p_f.proto $p_f.port $p_f.c -side left -padx 4 -pady 2 -anchor nw
}
proc Apol_NetContexts::portcon_limitPort {widget command new_port varname} {
    variable vals
    if {$command == "key"} {
        if {$new_port != "" &&
            (![string is integer $new_port] || $new_port < 0 || $new_port > 65535)} {
            return 0
        }
    } elseif {$command == "focusout"} {
        if {$new_port == ""} {
            set vals(portcon:$varname) 0
        } elseif {[string length $new_port] > 1} {
            set vals(portcon:$varname) [string trimleft $new_port " 0"]
        }
        $widget config -validate all
    }
    return 1
}
proc Apol_NetContexts::portcon_toggleCheckbutton_lowport {low high_cb high name1 name2 op} {
    variable vals
    variable widgets
    if {$vals($name2)} {
        $low configure -state normal
        $high_cb configure -state normal
        if {$vals(portcon:hiport_enable)} {
            $high configure -state normal
        }
    } else {
        $low configure -state disabled
        $high_cb configure -state disabled
        $high configure -state disabled
    }
}
proc Apol_NetContexts::portcon_toggleCheckbutton_hiport {low high name1 name2 op} {
    variable vals
    variable widgets
    if {$vals($name2)} {
        $low configure -text "Low Port"
        $high configure -state normal
    } else {
        $low configure -text "Single Port"
        $high configure -state disabled
    }
}
proc Apol_NetContexts::portcon_render {portcon} {
    foreach {loport hiport proto context} $portcon {break}
    if {$loport == $hiport} {
        set line "portcon $proto $loport "
    } else {
        set line "portcon $proto ${loport}-${hiport} "
    }
    concat $line [apol_RenderContext $context]
}
proc Apol_NetContexts::portcon_popup {port} { 
    foreach {low high} [split $port "-"] {break}
    if {$high == {}} {
        set high $low
    }
    set portcons [apol_GetPortcons $low $high]
    set text "port $port ([llength $portcons] context"
    if {[llength $portcons] != 1} {
        append text s
    }
    append text ")"
    foreach p [lsort -command portcon_sort $portcons] {
        append text "\n\t[portcon_render $p]"
    }
    Apol_Widget::showPopupText "port $port" $text
}
proc Apol_NetContexts::portcon_sort {a b} {
    foreach {loport1 hiport1 proto1 context1} $a {break}
    foreach {loport2 hiport2 proto2 context2} $b {break}
    if {$loport1 == $hiport1} {
        set singleport1 1
    } else {
        set singleport1 0
    }
    if {$loport2 == $hiport2} {
        set singleport2 1
    } else {
        set singleport2 0
    }
    if {$singleport1 && !$singleport2} {
        return -1
    } elseif {!$singleport1 && $singleport2} {
        return 1
    }
    if {$loport1 < $loport2} {
        return -1
    } elseif {$loport1 > $loport2} {
        return 1
    }
    if {$hiport1 < $hiport2} {
        return -1
    } elseif {$hiport1 > $hiport2} {
        return 1
    }
    string compare $proto1 $proto2
}
proc Apol_NetContexts::portcon_runSearch {} {
    variable vals
    variable widgets
    portcon_limitPort $widgets(portcon:port) focusout $vals(portcon:port) port
    portcon_limitPort $widgets(portcon:hiport) focusout $vals(portcon:hiport) hiport
    if {$vals(portcon:port_enable)} {
        set low $vals(portcon:port)
        set high $low
        if {$vals(portcon:hiport_enable)} {
            set high $vals(portcon:hiport)
            if {$vals(portcon:port_enable) && $high < $low} {
                tk_messageBox -icon error -type ok -title "Error" -message "The second port is not greater than the first."
                return
            }
        }
    } else {
        set low -1
        set high -1
    }
    if {$vals(portcon:proto_enable)} {
        if {[set proto $vals(portcon:proto)] == {}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No protocol selected."
            return
        }
    } else {
        set proto {}
    }
    if {[Apol_Widget::getContextSelectorState $widgets(portcon:context)]} {
        foreach {context range_match} [Apol_Widget::getContextSelectorValue $widgets(portcon:context)] {break}
    } else {
        set context {}
        set range_match 0
    }
    if {[catch {apol_GetPortcons $low $high $proto $context $range_match} portcons]} {
        tk_messageBox -icon error -type ok -title "Error" -message "Error obtaining portcons list:\n$portcons"
        return
    }
    set results "PORTCONS:"
    if {[llength $portcons] == 0} {
        append results "\nSearch returned no results."
    } else {
        foreach p [lsort -command portcon_sort $portcons] {
            append results "\n[portcon_render $p]"
        }
    }
    Apol_Widget::appendSearchResultText $widgets(results) $results
}
proc Apol_NetContexts::netifcon_open {} {
    variable vals
    variable widgets
    set ifs {}
    foreach if [lsort -unique -index 0 [apol_GetNetifcons {} {} {} {} {}]] {
        lappend ifs [lindex $if 0]
    }
    $widgets(netifcon:dev) configure -values $ifs
    set vals(netifcon:items) $ifs
}
proc Apol_NetContexts::netifcon_show {} {
    variable vals
    variable widgets
    $widgets(items_tf) configure -text "NetIF Contexts"
    $widgets(options_pm) raise netifcon
    set vals(items) $vals(netifcon:items)
}
proc Apol_NetContexts::netifcon_create {p_f} {
    variable vals
    variable widgets
    frame $p_f.dev
    set dev_cb [checkbutton $p_f.dev.dev_enable -text "Device" \
                    -variable Apol_NetContexts::vals(netifcon:dev_enable)]
    set widgets(netifcon:dev) [ComboBox $p_f.dev.dev -entrybg white -width 8 -state disabled \
                                   -textvariable Apol_NetContexts::vals(netifcon:dev) -autopost 1]
    trace add variable Apol_NetContexts::vals(netifcon:dev_enable) write \
        [list Apol_NetContexts::toggleCheckbutton $widgets(netifcon:dev)]
    pack $dev_cb -side top -anchor w
    pack $widgets(netifcon:dev) -side top -expand 0 -fill x -padx 4
    frame $p_f.ifcon
    set widgets(netifcon:ifcon) [Apol_Widget::makeContextSelector $p_f.ifcon.context "Contexts" "Interface context" -width 18]
    pack $widgets(netifcon:ifcon)
    frame $p_f.msgcon
    set widgets(netifcon:msgcon) [Apol_Widget::makeContextSelector $p_f.msgcon.context "Contexts" "Message context" -width 18]
    pack $widgets(netifcon:msgcon)
    pack $p_f.dev $p_f.ifcon $p_f.msgcon -side left -padx 4 -pady 2 -anchor nw
}
proc Apol_NetContexts::netifcon_render {netifcon_datum} {
    foreach {dev ifcon msgcon} $netifcon_datum {break}
    set line "netifcon $dev "
    append line "[apol_RenderContext $ifcon] [apol_RenderContext $msgcon]"
}
proc Apol_NetContexts::netifcon_popup {netif} {
    set netifcons [apol_GetNetifcons $netif]
    set text "network interface $netif ([llength $netifcons] context"
    if {[llength $netifcons] != 1} {
        append text s
    }
    append text ")"
    foreach n [lsort -index 0 $netifcons] {
        append text "\n\t[netifcon_render $n]"
    }
    Apol_Widget::showPopupText "interface $netif" $text
}
proc Apol_NetContexts::netifcon_runSearch {} {
    variable vals
    variable widgets
    if {$vals(netifcon:dev_enable)} {
        if {[set dev $vals(netifcon:dev)] == {}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No device selected."
            return
        }
    } else {
        set dev {}
    }
    if {[Apol_Widget::getContextSelectorState $widgets(netifcon:ifcon)]} {
        foreach {ifcon_context ifcon_range_match} [Apol_Widget::getContextSelectorValue $widgets(netifcon:ifcon)] {break}
    } else {
        set ifcon_context {}
        set ifcon_range_match 0
    }
    if {[Apol_Widget::getContextSelectorState $widgets(netifcon:msgcon)]} {
        foreach {msgcon_context msgcon_range_match} [Apol_Widget::getContextSelectorValue $widgets(netifcon:msgcon)] {break}
    } else {
        set msgcon_context {}
        set msgcon_range_match 0
    }
    if {[catch {apol_GetNetifcons $dev $ifcon_context $ifcon_range_match \
                    $msgcon_context $msgcon_range_match} netifcons]} {
        tk_messageBox -icon error -type ok -title "Error" -message "Error obtaining netifcons list:\n$netifcons"
        return
    }
    set results "NETIFCONS:"
    if {[llength $netifcons] == 0} {
        append results "\nSearch returned no results."
    } else {
        foreach n [lsort -index 0 $netifcons] {
            append results "\n[netifcon_render $n]"
        }
    }
    Apol_Widget::appendSearchResultText $widgets(results) $results
}
proc Apol_NetContexts::nodecon_open {} {
    variable vals
    variable widgets
    set vals(nodecon:items) {}
    if {[catch {apol_GetNodecons {} {} {} {} 0} nodecons]} {
        return
    }
    foreach n [lsort -command nodecon_sort $nodecons] {
        set addr [lindex $n 1]
        if {[lsearch $vals(nodecon:items) $addr] == -1} {
            lappend vals(nodecon:items) $addr
        }
    }
}
proc Apol_NetContexts::nodecon_show {} {
    variable vals
    variable widgets
    $widgets(items_tf) configure -text "Node Contexts"
    $widgets(options_pm) raise nodecon
    set vals(items) $vals(nodecon:items)
}
proc Apol_NetContexts::nodecon_create {p_f} {
    variable vals
    variable widgets
    frame $p_f.ip_type
    set ipv4_rb [radiobutton $p_f.ip_type.v4 -text "IPv4" -value ipv4 \
                     -variable Apol_NetContexts::vals(nodecon:ip_type)]
    set ipv6_rb [radiobutton $p_f.ip_type.v6 -text "IPv6" -value ipv6 \
                     -variable Apol_NetContexts::vals(nodecon:ip_type)]
    trace add variable Apol_NetContexts::vals(nodecon:ip_type) write \
        [list Apol_NetContexts::nodecon_pageChanged]
    pack $ipv4_rb $ipv6_rb -side top -anchor nw -pady 5
    frame $p_f.opts
    set widgets(nodecon:ip_pm) [PagesManager $p_f.opts.pm]
    nodecon_ipv4Create [$widgets(nodecon:ip_pm) add ipv4]
    nodecon_ipv6Create [$widgets(nodecon:ip_pm) add ipv6]
    $widgets(nodecon:ip_pm) compute_size
    pack $widgets(nodecon:ip_pm)
    $widgets(nodecon:ip_pm) raise ipv4
    frame $p_f.con
    set widgets(nodecon:context) [Apol_Widget::makeContextSelector $p_f.con.context "Contexts"]
    pack $widgets(nodecon:context)
    pack $p_f.ip_type $p_f.opts $p_f.con -side left -padx 4 -pady 2 -anchor nw
}
proc Apol_NetContexts::nodecon_ipv4Create {fv4} {
    variable widgets
    set v4addrf [frame $fv4.addr]
    set ipv4_addr_cb [checkbutton $v4addrf.enable -text "IP address" \
                          -variable Apol_NetContexts::vals(nodecon:ipv4_addr_enable)]
    set widgets(nodecon:v4addrf2) [frame $v4addrf.a]
    for {set i 0} {$i < 4} {incr i} {
        set e [entry $widgets(nodecon:v4addrf2).e$i -bg white -justify center -width 4 \
                   -state disabled \
                   -validate all -vcmd [list Apol_NetContexts::nodecon_limitAddr %W %V %P ipv4_addr$i] \
                   -textvariable Apol_NetContexts::vals(nodecon:ipv4_addr$i)]
        pack $e -side left -padx 1 -anchor center
        if {$i < 3} {
            pack [label $widgets(nodecon:v4addrf2).l$i -text "."] -side left -expand 0 -anchor s
        }
    }
    trace add variable Apol_NetContexts::vals(nodecon:ipv4_addr_enable) write \
        [list Apol_NetContexts::nodecon_toggleV4button $widgets(nodecon:v4addrf2).e]
    pack $ipv4_addr_cb -anchor w
    pack $widgets(nodecon:v4addrf2) -padx 3 -expand 0 -fill x
    set v4maskf [frame $fv4.mask]
    set ipv4_mask_cb [checkbutton $v4maskf.enable -text "Mask" \
                          -variable Apol_NetContexts::vals(nodecon:ipv4_mask_enable)]
    set widgets(nodecon:v4maskf2) [frame $v4maskf.m]
    for {set i 0} {$i < 4} {incr i} {
        set e [entry $widgets(nodecon:v4maskf2).e$i -bg white -justify center -width 4 \
                   -state disabled \
                   -validate all -vcmd [list Apol_NetContexts::nodecon_limitAddr %W %V %P ipv4_mask$i] \
                   -textvariable Apol_NetContexts::vals(nodecon:ipv4_mask$i)]
        pack $e -side left -padx 1 -anchor center
        if {$i < 3} {
            pack [label $widgets(nodecon:v4maskf2).l$i -text "."] -side left -expand 0 -anchor s
        }
    }
    trace add variable Apol_NetContexts::vals(nodecon:ipv4_mask_enable) write \
        [list Apol_NetContexts::nodecon_toggleV4button $widgets(nodecon:v4maskf2).e]
    pack $ipv4_mask_cb -anchor w
    pack $widgets(nodecon:v4maskf2) -padx 3 -expand 0 -fill x
    pack $v4addrf $v4maskf -padx 4 -pady 2 -anchor nw
}
proc Apol_NetContexts::nodecon_ipv6Create {fv6} {
    set v6addrf [frame $fv6.addr]
    set ipv4_addr_cb [checkbutton $v6addrf.enable -text "IP address" \
                          -variable Apol_NetContexts::vals(nodecon:ipv6_addr_enable)]
    set e [entry $v6addrf.addr -bg white -width 28 -state disabled \
               -textvariable Apol_NetContexts::vals(nodecon:ipv6_addr)]
    trace add variable Apol_NetContexts::vals(nodecon:ipv6_addr_enable) write \
        [list Apol_NetContexts::toggleCheckbutton $e]
    pack $ipv4_addr_cb -anchor w
    pack $e -padx 4 -expand 0 -fill x
    set v6maskf [frame $fv6.mask]
    set ipv6_mask_cb [checkbutton $v6maskf.enable -text "Mask" \
                          -variable Apol_NetContexts::vals(nodecon:ipv6_mask_enable)]
    set e [entry $v6maskf.addr -bg white -width 28 -state disabled \
               -textvariable Apol_NetContexts::vals(nodecon:ipv6_mask)]
    trace add variable Apol_NetContexts::vals(nodecon:ipv6_mask_enable) write \
        [list Apol_NetContexts::toggleCheckbutton $e]
    pack $ipv6_mask_cb -anchor w
    pack $e -padx 4 -expand 0 -fill x
    pack $v6addrf $v6maskf -padx 4 -pady 2 -anchor w
}
proc Apol_NetContexts::nodecon_pageChanged {name1 name2 op} {
    variable vals
    variable widgets
    $widgets(nodecon:ip_pm) raise $vals(nodecon:ip_type)
}
proc Apol_NetContexts::nodecon_limitAddr {widget command new_addr varname} {
    variable vals
    if {$command == "key"} {
        if {$new_addr != "" &&
            (![string is integer $new_addr] || $new_addr < 0 || $new_addr > 255)} {
            return 0
        }
    } elseif {$command == "focusout"} {
        if {$new_addr == ""} {
            set vals(nodecon:$varname) 0
        } elseif {[string length $new_addr] > 1} {
            set vals(nodecon:$varname) [string trimleft $new_addr " 0"]
        }
        after idle [list $widget config -validate all]
    }
    return 1
}
proc Apol_NetContexts::nodecon_toggleV4button {path name1 name2 op} {
    variable vals
    if {$vals($name2)} {
        for {set i 0} {$i < 4} {incr i} {
            ${path}${i} configure -state normal
        }
    } else {
        for {set i 0} {$i < 4} {incr i} {
            ${path}${i} configure -state disabled
        }
    }
}
proc Apol_NetContexts::nodecon_render {nodecon} {
    foreach {iptype addr mask context} $nodecon {break}
    return "nodecon $addr $mask [apol_RenderContext $context]"
}
proc Apol_NetContexts::nodecon_popup {nodecon} {
    if {[catch {apol_GetNodecons $nodecon} nodecons]} {
        tk_messageBox -icon error -type ok -title "Error" -message "Error obtaining nodecons list:\n$nodecons"
        return
    }
    set text "nodecon $nodecon ([llength $nodecons] context"
    if {[llength $nodecons] != 1} {
        append text s
    }
    append text ")"
    foreach n [lsort -index 0 $nodecons] {
        append text "\n\t[nodecon_render $n]"
    }
    Apol_Widget::showPopupText "address $nodecon" $text
}
proc Apol_NetContexts::nodecon_sort {a b} {
    foreach {t1 a1 m1 c1} $a {break}
    foreach {t2 a2 m2 c2} $b {break}
    if {$t1 == "ipv4" && $t2 == "ipv6"} {
        return -1
    } elseif {$t1 == "ipv6" && $t1 == "ipv4"} {
        return 0
    }
    if {[set x [string compare $a1 $a2]] != 0} {
        return $x
    }
    string compare $m1 $m2
}
proc Apol_NetContexts::nodecon_runSearch {} {
    variable vals
    variable widgets
    set addr {}
    set mask {}
    set context {}
    set range_match 0
    if {$vals(nodecon:ip_type) == "ipv4"} {
        foreach i {0 1 2 3} {
            nodecon_limitAddr $widgets(nodecon:v4addrf2).e$i focusout $vals(nodecon:ipv4_addr$i) ipv4_addr$i
            nodecon_limitAddr $widgets(nodecon:v4maskf2).e$i focusout $vals(nodecon:ipv4_mask$i) ipv4_mask$i
        }
        if {$vals(nodecon:ipv4_addr_enable)} {
            set addr [format "%d.%d.%d.%d" \
                          $vals(nodecon:ipv4_addr0) $vals(nodecon:ipv4_addr1) \
                          $vals(nodecon:ipv4_addr2) $vals(nodecon:ipv4_addr3)]
        }
        if {$vals(nodecon:ipv4_mask_enable)} {
            set mask [format "%d.%d.%d.%d" \
                          $vals(nodecon:ipv4_mask0) $vals(nodecon:ipv4_mask1) \
                          $vals(nodecon:ipv4_mask2) $vals(nodecon:ipv4_mask3)]
        }
    } else {
        if {$vals(nodecon:ipv6_addr_enable)} {
            if {[set addr $vals(nodecon:ipv6_addr)] == {}} {
                tk_messageBox -icon error -type ok -title "Error" -message "No IPV6 address provided."
                return
            }
        }
        if {$vals(nodecon:ipv6_mask_enable)} {
            if {[set mask $vals(nodecon:ipv6_mask)] == {}} {
                tk_messageBox -icon error -type ok -title "Error" -message "No IPV6 address provided."
                return
            }
        }
    }
    if {[Apol_Widget::getContextSelectorState $widgets(nodecon:context)]} {
        foreach {context range_match} [Apol_Widget::getContextSelectorValue $widgets(nodecon:context)] {break}
    }
    if {[catch {apol_GetNodecons $addr $mask $vals(nodecon:ip_type) $context $range_match} nodecons]} {
        tk_messageBox -icon error -type ok -title "Error" -message "Error obtaining nodecons list:\n$nodecons"
        return
    }
    set results "NODECONS:"
    if {[llength $nodecons] == 0} {
        append results "\nSearch returned no results."
    } else {
        foreach n [lsort -command nodecon_sort $nodecons] {
            append results "\n[nodecon_render $n]"
        }
    }
    Apol_Widget::appendSearchResultText $widgets(results) $results    
}
namespace eval Apol_Context_Dialog {
    variable dialog ""
    variable vars
}
proc Apol_Context_Dialog::getContext {{defaultContext {{} {} {} {{{} {}}}}} {parent .}} {
    variable dialog
    variable vars
    if {![winfo exists $dialog]} {
        _create_dialog $parent
    }
    array set vars [list $dialog:low_enable 0  $dialog:high_enable 0]
    foreach {user role type range} $defaultContext {break}
    $vars($dialog:user_box) configure -values $Apol_Users::users_list
    set vars($dialog:user) $user
    if {$user == {}} {
        set vars($dialog:user_enable) 0
    } else {
        set vars($dialog:user_enable) 1
    }
    $vars($dialog:role_box) configure -values $Apol_Roles::role_list
    set vars($dialog:role) $role
    if {$role == {}} {
        set vars($dialog:role_enable) 0
    } else {
        set vars($dialog:role_enable) 1
    }
    Apol_Widget::resetTypeComboboxToPolicy $vars($dialog:type_box)
    Apol_Widget::setTypeComboboxValue $vars($dialog:type_box) $type
    if {$type == {}} {
        set vars($dialog:type_enable) 0
    } else {
        set vars($dialog:type_enable) 1
    }
    Apol_Widget::resetLevelSelectorToPolicy $vars($dialog:low_level)
    Apol_Widget::resetLevelSelectorToPolicy $vars($dialog:high_level)
    if {[ApolTop::is_capable "mls"]} {
        if {[llength $range] == 1} {
            if {[lindex $range 0] == {{} {}}} {
                set vars($dialog:low_enable) 0
            } else {
                set vars($dialog:low_enable) 1
                Apol_Widget::setLevelSelectorLevel $vars($dialog:low_level) [lindex $range 0]
            }
            set vars($dialog:high_enable) 0
        } else {
            set vars($dialog:low_enable) 1
            Apol_Widget::setLevelSelectorLevel $vars($dialog:low_level) [lindex $range 0]
            set vars($dialog:high_enable) 1
            Apol_Widget::setLevelSelectorLevel $vars($dialog:high_level) [lindex $range 1]
        }
        $vars($dialog:low_cb) configure -state normal
    } else {
        set vars($dialog:low_enable) 0
        set vars($dialog:high_enable) 0
        $vars($dialog:low_cb) configure -state disabled
    }
    $dialog.bbox _redraw
    set retval [$dialog draw]
    if {$retval == -1 || $retval == 1} {
        return $defaultContext
    }
    _get_context $dialog
}
proc Apol_Context_Dialog::_create_dialog {parent} {
    variable dialog
    variable vars
    set dialog [Dialog .context_dialog -modal local -parent $parent \
                    -separator 1 -homogeneous 1 -title "Select Context"]
    array unset vars $dialog:*
    set f [$dialog getframe]
    set left_f [frame $f.left]
    set user_f [frame $left_f.user]
    set vars($dialog:user_cb) [checkbutton $user_f.enable -text "User" \
                                  -variable Apol_Context_Dialog::vars($dialog:user_enable)]
    set vars($dialog:user_box) [ComboBox $user_f.user -entrybg white -width 12 \
                                   -textvariable Apol_Context_Dialog::vars($dialog:user) -autopost 1]
    trace add variable Apol_Context_Dialog::vars($dialog:user_enable) write \
        [list Apol_Context_Dialog::_user_changed $dialog]
    pack $vars($dialog:user_cb) -anchor nw
    pack $vars($dialog:user_box) -anchor nw -padx 4 -expand 0 -fill x
    set role_f [frame $left_f.role]
    set vars($dialog:role_cb) [checkbutton $role_f.enable -text "Role" \
                                 -variable Apol_Context_Dialog::vars($dialog:role_enable)]
    set vars($dialog:role_box) [ComboBox $role_f.role -entrybg white -width 12 \
                                  -textvariable Apol_Context_Dialog::vars($dialog:role) -autopost 1]
    trace add variable Apol_Context_Dialog::vars($dialog:role_enable) write \
        [list Apol_Context_Dialog::_role_changed $dialog]
    pack $vars($dialog:role_cb) -anchor nw
    pack $vars($dialog:role_box) -anchor nw -padx 4 -expand 0 -fill x
    set type_f [frame $left_f.type]
    set vars($dialog:type_cb) [checkbutton $type_f.enable -text "Type" \
                                   -variable Apol_Context_Dialog::vars($dialog:type_enable)]
    set vars($dialog:type_box) [Apol_Widget::makeTypeCombobox $type_f.type]
    pack $vars($dialog:type_cb) -anchor nw
    pack $vars($dialog:type_box) -anchor nw -padx 4 -expand 0 -fill x
    trace add variable Apol_Context_Dialog::vars($dialog:type_enable) write \
        [list Apol_Context_Dialog::_type_changed $dialog]
    pack $user_f $role_f $type_f -side top -expand 1 -fill x
    set mlsbox [TitleFrame $f.mlsbox -text "MLS Range"]
    set mls_f [$mlsbox getframe]
    set vars($dialog:low_cb) [checkbutton $mls_f.low_cb -text "Single Level" \
                                  -variable Apol_Context_Dialog::vars($dialog:low_enable)]
    set vars($dialog:low_level) [Apol_Widget::makeLevelSelector $mls_f.low 8]
    trace add variable Apol_Context_Dialog::vars($dialog:low_enable) write \
        [list Apol_Context_Dialog::_low_changed $dialog]
    set vars($dialog:high_cb) [checkbutton $mls_f.high_cb \
                                   -text "High Level" \
                                   -variable Apol_Context_Dialog::vars($dialog:high_enable)]
    set vars($dialog:high_level) [Apol_Widget::makeLevelSelector $mls_f.high 8]
    trace add variable Apol_Context_Dialog::vars($dialog:high_enable) write \
        [list Apol_Context_Dialog::_high_changed $dialog]
    grid $vars($dialog:low_cb) $vars($dialog:high_cb) -sticky w
    grid $vars($dialog:low_level) $vars($dialog:high_level) -sticky nsew
    grid columnconfigure $mls_f 0 -weight 1 -uniform 1 -pad 2
    grid columnconfigure $mls_f 1 -weight 1 -uniform 1 -pad 2
    grid rowconfigure $mls_f 1 -weight 1
    pack $left_f $mlsbox -side left -expand 1 -fill both
    $dialog add -text "Ok" -command [list Apol_Context_Dialog::_okay $dialog]
    $dialog add -text "Cancel"
}
proc Apol_Context_Dialog::_okay {dialog} {
    variable vars
    if {$vars($dialog:user_enable)} {
        if {[set user $vars($dialog:user)] == {}} {
            tk_messageBox -icon error -type ok -title "Could Not Validate Context" \
                -message "No user was selected."
            return
        }
    } else {
        set user {}
    }
    if {$vars($dialog:role_enable)} {
        if {[set role $vars($dialog:role)] == {}} {
            tk_messageBox -icon error -type ok -title "Could Not Validate Context" \
                -message "No role was selected."
            return
        }
    } else {
        set role {}
    }
    set type [Apol_Widget::getTypeComboboxValue $vars($dialog:type_box)]
    if {$vars($dialog:type_enable)} {
        if {$type == {}} {
            tk_messageBox -icon error -type ok -title "Could Not Validate Context" \
                -message "No type was selected."
            return
        }
    } else {
        set type {}
    }
    if {$vars($dialog:low_enable)} {
        set range [_get_range $dialog]
        set low [lindex $range 0]
        if {$low == {{} {}}} {
            tk_messageBox -icon error -type ok -title "Could Not Validate Context" \
                -message "No level was selected."
            return
        }
    } else {
        set range {}
    }
    if {![ApolTop::is_policy_open]} {
        tk_messageBox -icon error -type ok -title "Could Not Validate Context" \
            -message "The context could not be validated because no policy is open."
        return
    }
    if {[catch {apol_IsValidPartialContext [list $user $role $type $range]} val]} {
        tk_messageBox -icon error -type ok -title "Could Not Validate Context" \
            -message "The context could not be validated:\n$val"
        return
    } elseif {$val == 0} {
        tk_messageBox -icon error -type ok -title "Could Not Validate Context" \
            -message "The selected context is not valid for the current policy."
        return
    }
    $dialog enddialog 0
}
proc Apol_Context_Dialog::_get_context {dialog} {
    variable vars
    set context {}
    if {$vars($dialog:user_enable)} {
        lappend context $vars($dialog:user)
    } else {
        lappend context {}
    }
    if {$vars($dialog:role_enable)} {
        lappend context $vars($dialog:role)
    } else {
        lappend context {}
    }
    if {$vars($dialog:type_enable)} {
        lappend context [Apol_Widget::getTypeComboboxValueAndAttrib $vars($dialog:type_box)]
    } else {
        lappend context {}
    }
    lappend context [_get_range $dialog]
}
proc Apol_Context_Dialog::_get_range {dialog} {
    variable vars
    if {!$vars($dialog:low_enable)} {
        return {{{} {}}}
    }
    set low [Apol_Widget::getLevelSelectorLevel $vars($dialog:low_level)]
    if {!$vars($dialog:high_enable)} {
        list $low
    } else {
        list $low [Apol_Widget::getLevelSelectorLevel $vars($dialog:high_level)]
    }
}
proc Apol_Context_Dialog::_user_changed {dialog name1 name2 op} {
    variable vars
    if {$vars($dialog:user_enable)} {
        $vars($dialog:user_box) configure -state normal
    } else {
        $vars($dialog:user_box) configure -state disabled
    }
}
proc Apol_Context_Dialog::_role_changed {dialog name1 name2 op} {
    variable vars
    if {$vars($dialog:role_enable)} {
        $vars($dialog:role_box) configure -state normal
    } else {
        $vars($dialog:role_box) configure -state disabled
    }
}
proc Apol_Context_Dialog::_type_changed {dialog name1 name2 op} {
    variable vars
    if {$vars($dialog:type_enable)} {
        Apol_Widget::setTypeComboboxState $vars($dialog:type_box) 1
    } else {
        Apol_Widget::setTypeComboboxState $vars($dialog:type_box) 0
    }
}
proc Apol_Context_Dialog::_low_changed {dialog name1 name2 op} {
    variable vars
    if {$vars($dialog:low_enable)} {
        $vars($dialog:high_cb) configure -state normal
        Apol_Widget::setLevelSelectorState $vars($dialog:low_level) 1
        if {$vars($dialog:high_enable)} {
            Apol_Widget::setLevelSelectorState $vars($dialog:high_level) 1
        }
    } else {
        $vars($dialog:high_cb) configure -state disabled
        Apol_Widget::setLevelSelectorState $vars($dialog:low_level) 0
        Apol_Widget::setLevelSelectorState $vars($dialog:high_level) 0
    }
}
proc Apol_Context_Dialog::_high_changed {dialog name1 name2 op} {
    variable vars
    if {$vars($dialog:high_enable)} {
        $vars($dialog:low_cb) configure -text "Low Level"
        Apol_Widget::setLevelSelectorState $vars($dialog:high_level) 1
    } else {
        $vars($dialog:low_cb) configure -text "Single Level"
        Apol_Widget::setLevelSelectorState $vars($dialog:high_level) 0
    }
}
namespace eval Apol_Widget {
    variable vars
}
proc Apol_Widget::makeContextSelector {path rangeMatchText {enableText "Context"} args} {
    variable vars
    array unset vars $path:*
    set vars($path:context) {{} {} {} {{{} {}}}}
    set vars($path:context_rendered) {}
    set vars($path:search_type) "exact"
    set f [frame $path]
    set context_frame [frame $f.context]
    set context2_frame [frame $f.context2]
    pack $context_frame $context2_frame -side left -expand 0 -anchor nw
    if {$enableText != {}} {
        set vars($path:enable) 0
        set context_cb [checkbutton $context_frame.enable -text $enableText \
                          -variable Apol_Widget::vars($path:enable)]
        pack $context_cb -side top -expand 0 -anchor nw
        trace add variable Apol_Widget::vars($path:enable) write [list Apol_Widget::_toggle_context_selector $path $context_cb]
    }
    set context_display [eval Entry $context_frame.display -textvariable Apol_Widget::vars($path:context_rendered) -width 26 -editable 0 $args]
    set context_button [button $context_frame.button -text "Select Context..." -state disabled -command [list Apol_Widget::_show_context_dialog $path]]
    trace add variable Apol_Widget::vars($path:context) write [list Apol_Widget::_update_context_display $path]
    pack $context_display -side top -expand 1 -fill x -anchor nw
    pack $context_button -side top -expand 0 -anchor ne
    if {$enableText != {}} {
        pack configure $context_display -padx 4
        pack configure $context_button -padx 4
    }
    set range_label [label $context2_frame.label -text "MLS range matching:" \
                         -state disabled]
    set range_exact [radiobutton $context2_frame.exact -text "Exact matches" \
                         -state disabled \
                         -value exact -variable Apol_Widget::vars($path:search_type)]
    set range_subset [radiobutton $context2_frame.subset -text "$rangeMatchText containing range" \
                          -state disabled \
                          -value subset -variable Apol_Widget::vars($path:search_type)]
    set range_superset [radiobutton $context2_frame.superset -text "$rangeMatchText within range" \
                            -state disabled \
                            -value superset -variable Apol_Widget::vars($path:search_type)]
    pack $range_label $range_exact $range_subset $range_superset \
        -side top -expand 0 -anchor nw
    return $f
}
proc Apol_Widget::setContextSelectorState {path newState} {
    if {$newState == 0 || $newState == "disabled"} {
        set new_state disabled
    } else {
        set new_state normal
    }
    foreach w {display button} {
        $path.context.$w configure -state $new_state
    }
    if {![ApolTop::is_capable "mls"]} {
        set new_state disabled
    }
    foreach w {label exact subset superset} {
        $path.context2.$w configure -state $new_state
    }
}
proc Apol_Widget::clearContextSelector {path} {
    set Apol_Widget::vars($path:context) {{} {} {} {{{} {}}}}
    set Apol_Widget::vars($path:search_type) exact
    catch {set Apol_Widget::vars($path:enable) 0}
}
proc Apol_Widget::getContextSelectorState {path} {
    return $Apol_Widget::vars($path:enable)
}
proc Apol_Widget::getContextSelectorValue {path} {
    variable vars
    set c $vars($path:context)
    lset c 2 [lindex $c 2 0]
    list $c $vars($path:search_type)
}
proc Apol_Widget::_toggle_context_selector {path cb name1 name2 op} {
    if {$Apol_Widget::vars($path:enable)} {
        Apol_Widget::setContextSelectorState $path normal
    } else {
        Apol_Widget::setContextSelectorState $path disabled
    }
}
proc Apol_Widget::_show_context_dialog {path} {
    set Apol_Widget::vars($path:context) [Apol_Context_Dialog::getContext $Apol_Widget::vars($path:context)]
}
proc Apol_Widget::_update_context_display {path name1 name2 op} {
    variable vars
    set display $path.context.display
    foreach {user role type range} $vars($path:context) {break}
    set context ""
    if {$user == ""} {
        lappend context "*"
    } else {
        lappend context $user
    }
    if {$role == ""} {
        lappend context "*"
    } else {
        lappend context $role
    }
    if {$type == ""} {
        lappend context "*"
    } else {
        lappend context [lindex $type 0]
    }
    if {[ApolTop::is_capable "mls"]} {
        if {$range == {} || $range == {{{} {}}}} {
            lappend context "*"
        } else {
            if {[catch {apol_RenderLevel [lindex $range 0]} level]} {
                set level "?"
            }
            if {[llength $range] > 1 && [lindex $range 0] != [lindex $range 1]} {
                if {[catch {apol_RenderLevel [lindex $range 1]} high]} {
                    append level " - ?"
                } else {
                    append level " - $high"
                }
            }
            if {$level == ""} {
                lappend context "*"
            } else {
                lappend context $level
            }
        }
    }
    set vars($path:context_rendered) [join $context ":"]
    $display configure -helptext $vars($path:context_rendered)
}
namespace eval Apol_FSContexts {
    variable widgets
    variable vals
}
proc Apol_FSContexts::set_Focus_to_Text {} {
    focus $Apol_FSContexts::widgets(results)
}
proc Apol_FSContexts::open {} {
    variable vals
    genfscon_open
    fsuse_open
    set vals(context_type) genfscon
}
proc Apol_FSContexts::close {} {
    variable widgets
    initializeVars
    Apol_Widget::clearSearchResults $widgets(results)
    Apol_Widget::clearContextSelector $widgets(genfscon:context)
    Apol_Widget::clearContextSelector $widgets(fsuse:context)
    $widgets(genfscon:fs) configure -values {}
    $widgets(fsuse:type) configure -values {}
    $widgets(fsuse:fs) configure -values {}
}
proc Apol_FSContexts::initializeVars {} {
    variable vals
    array set vals {
        genfscon:items {}
        genfscon:fs_enable 0     genfscon:fs {}
        genfscon:path_enable 0   genfscon:path {}
        fsuse:items {}
        fsuse:type_enable 0  fsuse:type {}
        fsuse:fs_enable 0    fsuse:fs {}
        items {}
        context_type genfscon
    }
}
proc Apol_FSContexts::search { str case_Insensitive regExpr srch_Direction } {
    variable widgets
    ApolTop::textSearch $widgets(results).tb $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_FSContexts::goto_line { line_num } {
    variable widgets
    Apol_Widget::gotoLineSearchResults $widgets(results) $line_num
}
proc Apol_FSContexts::create {nb} {
    variable widgets
    variable vals
    initializeVars
    set frame [$nb insert end $ApolTop::fs_contexts_tab -text "FS Contexts"]
    set pw [PanedWindow $frame.pw -side top -weights extra]
    set leftf [$pw add -weight 0]
    set rightf [$pw add -weight 1]
    pack $pw -fill both -expand yes
    set context_box [TitleFrame $leftf.context_f -text "Context Type"]
    set context_f [$context_box getframe]
    radiobutton $context_f.genfscon -text "genfscon" -value genfscon \
        -variable Apol_FSContexts::vals(context_type)
    radiobutton $context_f.fsuse -text "fs_use" -value fsuse \
        -variable Apol_FSContexts::vals(context_type)
    trace add variable Apol_FSContexts::vals(context_type) write \
        {Apol_FSContexts::contextTypeChanged}
    pack $context_f.genfscon $context_f.fsuse \
        -anchor w -expand 0 -padx 4 -pady 5
    pack $context_box -expand 0 -fill x
    set widgets(items_tf) [TitleFrame $leftf.items_f -text "GenFS Contexts"]
    set widgets(items) [Apol_Widget::makeScrolledListbox [$widgets(items_tf) getframe].items \
                            -height 20 -width 20 -listvar Apol_FSContexts::vals(items)]
    Apol_Widget::setListboxCallbacks $widgets(items) \
        {{"Show Context Info" {Apol_FSContexts::popupContextInfo}}}
    pack $widgets(items) -expand 1 -fill both
    pack $widgets(items_tf) -expand 1 -fill both
    set optsbox [TitleFrame $rightf.optsbox -text "Search Options"]
    pack $optsbox -side top -expand 0 -fill both -padx 2
    set widgets(options_pm) [PagesManager [$optsbox getframe].pm]
    genfscon_create [$widgets(options_pm) add genfscon]
    fsuse_create [$widgets(options_pm) add fsuse]
    $widgets(options_pm) compute_size
    pack $widgets(options_pm) -expand 1 -fill both -side left
    $widgets(options_pm) raise genfscon
    set ok [button [$optsbox getframe].ok -text "OK" -width 6 \
                -command Apol_FSContexts::runSearch]
    pack $ok -side right -pady 5 -padx 5 -anchor ne
    set resultsbox [TitleFrame $rightf.resultsbox -text "Search Results"]
    pack $resultsbox -expand yes -fill both -padx 2
    set widgets(results) [Apol_Widget::makeSearchResults [$resultsbox getframe].results]
    pack $widgets(results) -side top -expand yes -fill both
    return $frame
}
proc Apol_FSContexts::popupContextInfo {value} {
    variable vals
    if {$vals(context_type) == "genfscon"} {
        genfscon_popup $value
    } else {
        fsuse_popup $value
    }
}
proc Apol_FSContexts::contextTypeChanged {name1 name2 op} {
    variable vals
    variable widgets
    Apol_Widget::clearSearchResults $widgets(results)
    if {$vals(context_type) == "genfscon"} {
        genfscon_show
    } else {
        fsuse_show
    }
}
proc Apol_FSContexts::toggleCheckbutton {path name1 name2 op} {
    variable vals
    variable widgets
    if {$vals($name2)} {
        $path configure -state normal
    } else {
        $path configure -state disabled
    }
}
proc Apol_FSContexts::runSearch {} {
    variable vals
    variable widgets
    Apol_Widget::clearSearchResults $widgets(results)
    if {![ApolTop::is_policy_open]} {
        tk_messageBox -icon error -type ok -title "Error" -message "No current policy file is opened!"
        return
    }
    if {$vals(context_type) == "genfscon"} {
        genfscon_runSearch
    } else {
        fsuse_runSearch
    }
}
proc Apol_FSContexts::fscontext_sort {a b} {
    if {[set z [string compare [lindex $a 0] [lindex $b 0]]] != 0} {
        return $z
    }
    if {[set z [string compare [lindex $a 1] [lindex $b 1]]] != 0} {
        return $z
    }
    return 0
}
proc Apol_FSContexts::genfscon_open {} {
    variable vals
    variable widgets
    set fstypes {}
    foreach genfs [lsort -unique -index 0 [apol_GetGenFSCons {} {} {} 0]] {
        lappend fstypes [lindex $genfs 0]
    }
    set vals(genfscon:items) $fstypes
    $widgets(genfscon:fs) configure -values $fstypes
}
proc Apol_FSContexts::genfscon_show {} {
    variable vals
    variable widgets
    $widgets(items_tf) configure -text "GenFS Contexts"
    $widgets(options_pm) raise genfscon
    set vals(items) $vals(genfscon:items)
}
proc Apol_FSContexts::genfscon_create {p_f} {
    variable widgets
    variable vals
    set fs [frame $p_f.fs]
    set fs_cb [checkbutton $fs.fs_enable -text "Filesystem" \
                   -variable Apol_FSContexts::vals(genfscon:fs_enable)]
    set widgets(genfscon:fs) [ComboBox $fs.fs -entrybg white -width 12 -state disabled \
                                  -textvariable Apol_FSContexts::vals(genfscon:fs) -autopost 1]
    trace add variable Apol_FSContexts::vals(genfscon:fs_enable) write \
        [list Apol_FSContexts::toggleCheckbutton $widgets(genfscon:fs)]
    pack $fs_cb -side top -anchor w
    pack $widgets(genfscon:fs) -side top -expand 0 -fill x -padx 4
    set p [frame $p_f.p]
    set p_cb [checkbutton $p.p_enable -text "Path" \
                   -variable Apol_FSContexts::vals(genfscon:path_enable)]
    set widgets(genfscon:path) [entry $p.path -bg white -width 24 \
                                    -state disabled \
                                    -textvariable Apol_FSContexts::vals(genfscon:path)]
    trace add variable Apol_FSContexts::vals(genfscon:path_enable) write \
        [list Apol_FSContexts::toggleCheckbutton $widgets(genfscon:path)]
    pack $p_cb -side top -anchor w
    pack $widgets(genfscon:path) -side top -expand 0 -fill x -padx 4
    frame $p_f.c
    set widgets(genfscon:context) [Apol_Widget::makeContextSelector $p_f.c.context "Contexts"]
    pack $widgets(genfscon:context)
    pack $fs $p $p_f.c -side left -anchor n -padx 4 -pady 2
}
proc Apol_FSContexts::genfscon_render {genfscon {compact 0}} {
    foreach {fstype path objclass context} $genfscon {break}
    set context [apol_RenderContext $context]
    if {$objclass != "any"} {
        if {$compact} {
            format "genfscon %s %s -t%s %s" $fstype $path $objclass $context
        } else {
            format "genfscon  %-12s %-24s -t%-4s %s" \
                $fstype $path $objclass $context
        }
    } else {
        if {$compact} {
            format "genfscon %s %s %s" $fstype $path $context
        } else {
            format "genfscon  %-12s %-24s %s" \
                $fstype $path $context
        }
    }
}
proc Apol_FSContexts::genfscon_popup {fstype} {
    set genfscons [apol_GetGenFSCons $fstype]
    set text "genfs filesystem $fstype ([llength $genfscons] context"
    if {[llength $genfscons] != 1} {
        append text s
    }
    append text ")"
    foreach g [lsort -index 1 -dictionary $genfscons] {
        append text "\n\t[genfscon_render $g 1]"
    }
    Apol_Widget::showPopupText "filesystem $fstype" $text
}
proc Apol_FSContexts::genfscon_runSearch {} {
    variable vals
    variable widgets
    set fstype {}
    set path {}
    set context {}
    set range_match 0
    if {$vals(genfscon:fs_enable) && [set fstype $vals(genfscon:fs)] == {}} {
        tk_messageBox -icon error -type ok -title "Error" -message "No filesystem selected."
        return
    }
    if {$vals(genfscon:path_enable) && [set path $vals(genfscon:path)] == {}} {
        tk_messageBox -icon error -type ok -title "Error" -message "No path given."
        return
    }
    if {[Apol_Widget::getContextSelectorState $widgets(genfscon:context)]} {
        foreach {context range_match} [Apol_Widget::getContextSelectorValue $widgets(genfscon:context)] {break}
    }
    if {[catch {apol_GetGenFSCons $fstype $path $context $range_match} genfscons]} {
        tk_messageBox -icon error -type ok -title "Error" -message "Error obtaining genfscons list: $genfscons"
        return
    }
    set results "GENFSCONS:"
    if {[llength $genfscons] == 0} {
        append results "\nSearch returned no results."
    } else {
        foreach g [lsort -command fscontext_sort $genfscons] {
            append results "\n[genfscon_render $g]"
        }
    }
    Apol_Widget::appendSearchResultText $widgets(results) $results
}
proc Apol_FSContexts::fsuse_open {} {
    variable vals
    variable widgets
    set fsuses [apol_GetFSUses {} {} {} 0]
    set behavs {}
    foreach fsuse [lsort -unique -index 0 $fsuses] {
        lappend behavs [lindex $fsuse 0]
    }
    $widgets(fsuse:type) configure -values $behavs
    set fstypes {}
    foreach fsuse [lsort -unique -index 1 $fsuses] {
        lappend fstypes [lindex $fsuse 1]
    }
    $widgets(fsuse:fs) configure -values $fstypes
    set vals(fsuse:items) $fstypes
}
proc Apol_FSContexts::fsuse_show {} {
    variable vals
    variable widgets
    $widgets(items_tf) configure -text "fs_use Contexts"
    $widgets(options_pm) raise fsuse
    set vals(items) $vals(fsuse:items)
}
proc Apol_FSContexts::fsuse_create {p_f} {
    variable widgets
    variable vals
    set t [frame $p_f.t]
    set type_cb [checkbutton $t.type_enable -text "Statement type" \
                   -variable Apol_FSContexts::vals(fsuse:type_enable)]
    set widgets(fsuse:type) [ComboBox $t.type -entrybg white -width 12 -state disabled \
                                  -textvariable Apol_FSContexts::vals(fsuse:type) -autopost 1]
    trace add variable Apol_FSContexts::vals(fsuse:type_enable) write \
        [list Apol_FSContexts::toggleCheckbutton $widgets(fsuse:type)]
    pack $type_cb -side top -anchor w
    pack $widgets(fsuse:type) -side top -expand 0 -fill x -padx 4
    set fs [frame $p_f.fs]
    set fs_cb [checkbutton $fs.fs_enable -text "Filesystem" \
                   -variable Apol_FSContexts::vals(fsuse:fs_enable)]
    set widgets(fsuse:fs) [ComboBox $fs.fs -entrybg white -width 12 -state disabled \
                                  -textvariable Apol_FSContexts::vals(fsuse:fs) -autopost 1]
    trace add variable Apol_FSContexts::vals(fsuse:fs_enable) write \
        [list Apol_FSContexts::toggleCheckbutton $widgets(fsuse:fs)]
    pack $fs_cb -side top -anchor w
    pack $widgets(fsuse:fs) -side top -expand 0 -fill x -padx 4
    frame $p_f.c
    set widgets(fsuse:context) [Apol_Widget::makeContextSelector $p_f.c.context "Contexts"]
    pack $widgets(fsuse:context)
    pack $t $fs $p_f.c -side left -anchor n -padx 4 -pady 2
}
proc Apol_FSContexts::fsuse_render {fsuse} {
    foreach {behav fstype context} $fsuse {break}
    if {$behav == "fs_use_psid"} {
        format "%-13s %s;" $behav $fstype
    } else {
        format "%-13s %-10s %s;" $behav $fstype [apol_RenderContext $context]
    }
}
proc Apol_FSContexts::fsuse_popup {fs} {
    set fsuses [apol_GetFSUses $fs]
    set text "fs_use $fs ([llength $fsuses] context"
    if {[llength $fsuses] != 1} {
        append text s
    }
    append text ")"
    foreach u [lsort -index 1 -dictionary $fsuses] {
        append text "\n\t[fsuse_render $u]"
    }
    Apol_Widget::showPopupText $fs $text
}
proc Apol_FSContexts::fsuse_runSearch {} {
    variable vals
    variable widgets
    set behavior {}
    set fstype {}
    set context {}
    set range_match 0
    if {$vals(fsuse:type_enable) && [set behavior $vals(fsuse:type)] == {}} {
            tk_messageBox -icon error -type ok -title "Error" -message "No fs_use statement type selected."
            return
    }
    if {$vals(fsuse:fs_enable) && [set fstype $vals(fsuse:fs)] == {}} {
        tk_messageBox -icon error -type ok -title "Error" -message "No filesystem selected."
        return
    }
    if {[Apol_Widget::getContextSelectorState $widgets(fsuse:context)]} {
        foreach {context range_match} [Apol_Widget::getContextSelectorValue $widgets(fsuse:context)] {break}
    }
    if {[catch {apol_GetFSUses $fstype $behavior $context $range_match} fsuses]} {
        tk_messageBox -icon error -type ok -title "Error" -message "Error obtaining fs_use list: $fsuses"
        return
    }
    set results "FS_USES:"
    if {[llength $fsuses] == 0} {
        append results "\nSearch returned no results."
    } else {
        foreach u [lsort -command fscontext_sort $fsuses] {
            append results "\n[fsuse_render $u]"
        }
    }
    Apol_Widget::appendSearchResultText $widgets(results) $results
}
namespace eval Apol_Widget {
    variable menuPopup {}
    variable infoPopup {}
    variable infoPopup2 {}
    variable vars
}
proc Apol_Widget::makeScrolledListbox {path args} {
    set sw [ScrolledWindow $path -scrollbar both -auto both]
    set lb [eval listbox $sw.lb $args -bg white -highlightthickness 0]
    $sw setwidget $lb
    update
    grid propagate $sw 0
    bind $lb <<ListboxSelect>> [list focus $lb]
    bind $lb <Key> [list Apol_Widget::_listbox_key $lb %K]
    return $sw
}
proc Apol_Widget::setListboxCallbacks {path callback_list} {
    set lb [getScrolledListbox $path]
    bind $lb <Double-Button-1> [eval list Apol_Widget::_listbox_double_click $lb [lindex $callback_list 0 1]]
    variable menuPopup
    if {![winfo exists $menuPopup]} {
        set menuPopup [menu .apol_widget_menu_popup]
    }
    set lb [getScrolledListbox $path]
    bind $lb <Button-3> [list ApolTop::popup_listbox_Menu %W %x %y $menuPopup $callback_list $lb]
}
proc Apol_Widget::getScrolledListbox {path} {
    return $path.lb
}
proc Apol_Widget::setScrolledListboxState {path newState} {
    if {$newState == 0 || $newState == "disabled"} {
        $path.lb configure -state disabled
    } else {
        $path.lb configure -state normal
    }
}
proc Apol_Widget::makeTypeCombobox {path args} {
    variable vars
    array unset vars $path:*
    set vars($path:type) ""
    set vars($path:attribenable) 0
    set vars($path:attrib) ""
    set f [frame $path]
    set type_box [eval ComboBox $f.tb -helptext {{Type or select a type}} \
                      -textvariable Apol_Widget::vars($path:type) \
                      -entrybg white -width 20 -autopost 1 $args]
    pack $type_box -side top -expand 1 -fill x
    set attrib_width [expr {[$type_box cget -width] - 4}]
    set attrib_enable [checkbutton $f.ae \
                           -anchor w -text "Filter by attribute"\
                           -variable Apol_Widget::vars($path:attribenable) \
                           -command [list Apol_Widget::_attrib_enabled $path]]
    set attrib_box [ComboBox $f.ab -autopost 1 -entrybg white -width $attrib_width \
                        -textvariable Apol_Widget::vars($path:attrib)]
    trace add variable Apol_Widget::vars($path:attrib) write [list Apol_Widget::_attrib_changed $path]
    pack $attrib_enable -side top -expand 0 -fill x -anchor sw -padx 5 -pady 2
    pack $attrib_box -side top -expand 1 -fill x -padx 9
    _attrib_enabled $path
    return $f
}
proc Apol_Widget::resetTypeComboboxToPolicy {path} {
    $path.tb configure -values $Apol_Types::typelist
    $path.ab configure -values $Apol_Types::attriblist
}
proc Apol_Widget::clearTypeCombobox {path} {
    variable vars
    set vars($path:attribenable) 0
    set vars($path:attrib) ""
    set vars($path:type) ""
    $path.tb configure -values {}
    $path.ab configure -values {}
    _attrib_enabled $path
}
proc Apol_Widget::getTypeComboboxValue {path} {
    string trim $Apol_Widget::vars($path:type)
}
proc Apol_Widget::getTypeComboboxValueAndAttrib {path} {
    variable vars
    if {$vars($path:attribenable)} {
        list $vars($path:type) $vars($path:attrib)
    } else {
        set vars($path:type)
    }
}
proc Apol_Widget::setTypeComboboxValue {path type} {
    variable vars
    if {[llength $type] <= 1} {
        set vars($path:type) $type
        set vars($path:attribenable) 0
        set vars($path:attrib) ""
    } else {
        set vars($path:type) [lindex $type 0]
        set vars($path:attribenable) 1
        set vars($path:attrib) [lindex $type 1]
    }
    _attrib_enabled $path
}
proc Apol_Widget::setTypeComboboxState {path newState} {
    variable vars
    if {$newState == 0 || $newState == "disabled"} {
        $path.tb configure -state disabled
        $path.ae configure -state disabled
        $path.ab configure -state disabled
    } else {
        $path.tb configure -state normal
        $path.ae configure -state normal
        if {$vars($path:attribenable)} {
            $path.ab configure -state normal
        }
    }
}
proc Apol_Widget::makeLevelSelector {path catSize args} {
    variable vars
    array unset vars $path:*
    set vars($path:sens) ""
    set vars($path:cats) {}
    set f [frame $path]
    set sens_box [eval ComboBox $f.sens $args \
                      -textvariable Apol_Widget::vars($path:sens) \
                      -entrybg white -width 16 -autopost 1]
    trace add variable Apol_Widget::vars($path:sens) write [list Apol_Widget::_sens_changed $path]
    pack $sens_box -side top -expand 0 -fill x
    set cats_label [label $f.cl -text "Categories:"]
    pack $cats_label -side top -anchor sw -pady 2 -expand 0
    set cats [makeScrolledListbox $f.cats -width 16 -height $catSize \
                  -listvariable Apol_Widget::vars($path:cats) \
                  -selectmode extended -exportselection 0]
    pack $cats -side top -expand 1 -fill both
    set reset [button $f.reset -text "Clear Categories" \
                   -command [list [getScrolledListbox $cats] selection clear 0 end]]
    pack $reset -side top -anchor center -pady 2
    return $f
}
proc Apol_Widget::getLevelSelectorLevel {path} {
    variable vars
    if {[catch {apol_GetLevels $vars($path:sens)} l] || $l == {}} {
        set sens $vars($path:sens)
    } else {
        set sens [lindex $l 0 0]
    }
    set sl [getScrolledListbox $path.cats]
    set cats {}
    foreach idx [$sl curselection] {
        lappend cats [$sl get $idx]
    }
    list $sens $cats
}
proc Apol_Widget::setLevelSelectorLevel {path level} {
    variable vars
    set sens [lindex $level 0]
    set cats [lindex $level 1]
    set sens_list [$path.sens cget -values]
    if {[lsearch -exact $sens_list $sens] != -1} {
        set vars($path:sens) $sens
        set cats_list $vars($path:cats)
        set first_idx -1
        set listbox [getScrolledListbox $path.cats]
        foreach cat $cats {
            if {[set idx [lsearch -exact $cats_list $cat]] != -1} {
                $listbox selection set $idx
                if {$first_idx == -1 || $idx < $first_idx} {
                    set first_idx $idx
                }
            }
        }
        incr first_idx -1
        $listbox yview scroll $first_idx units
    }
}
proc Apol_Widget::resetLevelSelectorToPolicy {path} {
    variable vars
    set vars($path:sens) ""
    if {[catch {apol_GetLevels {} 0} level_data]} {
        $path.sens configure -values {}
    } else {
        set vals {}
        foreach l [lsort -integer -index 3 $level_data] {
            lappend vals [lindex $l 0]
        }
        $path.sens configure -values $vals
    }
}
proc Apol_Widget::clearLevelSelector {path} {
    variable vars
    set vars($path:sens) ""
    $path.sens configure -values {}
}
proc Apol_Widget::setLevelSelectorState {path newState} {
    if {$newState == 0 || $newState == "disabled"} {
        set newState disabled
    } else {
        set newState normal
    }
    $path.sens configure -state $newState
    $path.cl configure -state $newState
    $path.reset configure -state $newState
    setScrolledListboxState $path.cats $newState
}
proc Apol_Widget::makeRegexpEntry {path args} {
    variable vars
    array unset vars $path:*
    set vars($path:enable_regexp) 0
    set f [frame $path]
    set cb [checkbutton $f.cb -text "Search using regular expression" \
                -variable Apol_Widget::vars($path:enable_regexp)]
    set regexp [eval entry $f.entry $args \
                    -textvariable Apol_Widget::vars($path:regexp) \
                    -width 32 -state disabled -bg $ApolTop::default_bg_color]
    trace add variable Apol_Widget::vars($path:enable_regexp) write \
        [list Apol_Widget::_toggle_regexp_check_button $regexp]
    pack $cb -side top -anchor nw
    pack $regexp -side top -padx 4 -anchor nw -expand 0 -fill x
    return $f
}
proc Apol_Widget::setRegexpEntryState {path newState} {
    variable vars
    if {$newState == 0 || $newState == "disabled"} {
        set vars($path:enable_regexp) 0
        $path.cb configure -state disabled
    } else {
        $path.cb configure -state normal
    }
}
proc Apol_Widget::setRegexpEntryValue {path newState newValue} {
    variable vars
    set old_state [$path.cb cget -state]
    set vars($path:enable_regexp) $newState
    set vars($path:regexp) $newValue
    $path.cb configure -state $old_state
}
proc Apol_Widget::getRegexpEntryState {path} {
    return $Apol_Widget::vars($path:enable_regexp)
}
proc Apol_Widget::getRegexpEntryValue {path} {
    return $Apol_Widget::vars($path:regexp)
}
proc Apol_Widget::makeSearchResults {path args} {
    variable vars
    array unset vars $path:*
    set sw [ScrolledWindow $path -scrollbar both -auto both]
    set tb [eval text $sw.tb $args -bg white -wrap none -state disabled -font $ApolTop::text_font]
    set vars($path:cursor) [$tb cget -cursor]
    $tb tag configure linenum -foreground blue -underline 1
    $tb tag configure selected -foreground red -underline 1
    $tb tag configure enabled -foreground green -underline 1
    $tb tag configure disabled -foreground red -underline 1
    $tb tag bind linenum <Button-1> [list Apol_Widget::_hyperlink $path %x %y]
    $tb tag bind linenum <Enter> [list $tb configure -cursor hand2]
    $tb tag bind linenum <Leave> [list $tb configure -cursor $Apol_Widget::vars($path:cursor)]
    $sw setwidget $tb
    return $sw
}
proc Apol_Widget::clearSearchResults {path} {
    $path.tb configure -state normal
    $path.tb delete 0.0 end
    $path.tb configure -state disabled
}
proc Apol_Widget::appendSearchResultHeader {path header} {
    $path.tb configure -state normal
    $path.tb insert 1.0 "$header\n"
    $path.tb configure -state disabled
}
proc Apol_Widget::appendSearchResultText {path text} {
    $path.tb configure -state normal
    $path.tb insert end $text
    $path.tb configure -state disabled
}
proc Apol_Widget::appendSearchResultLine {path indent cond_info line_type args} {
    $path.tb configure -state normal
    $path.tb insert end [string repeat " " $indent]
    set text [join [concat $line_type $args]]
    $path.tb insert end "[string trim $text];"
    if {$cond_info != {}} {
        if {[lindex $cond_info 0] == "enabled"} {
            $path.tb insert end "  \[" {} "Enabled" enabled "\]"
        } else {
            $path.tb insert end "  \[" {} "Disabled" disabled "\]"
        }
    }
    $path.tb insert end "\n"
    $path.tb configure -state disabled
}
proc Apol_Widget::appendSearchResultAVRules {path indent rule_list {varname {}}} {
    set curstate [$path.tb cget -state]
    $path.tb configure -state normal
    set rules {}
    if {$varname != {}} {
        upvar $varname progressvar
        set progressvar "Sorting [llength $rule_list] semantic av rule(s)..."
        update idletasks
    }
    set rules [lsort -command apol_RenderAVRuleComp [lsort -unique $rule_list]]
    if {$varname != {}} {
        set progressvar "Rendering [llength $rules] semantic av rule(s)..."
        update idletasks
    }
    set num_enabled 0
    set num_disabled 0
    foreach r $rules {
        $path.tb insert end [string repeat " " $indent]
        foreach {rule_type source_set target_set class perms cond_info} [apol_RenderAVRule $r] {break}
        set text [list "$rule_type $source_set $target_set" {}]
        if {[llength $perms] > 1} {
            set perms "\{$perms\}"
        }
        lappend text " : $class $perms;" {}
        eval $path.tb insert end $text
        if {$cond_info != {}} {
            if {[lindex $cond_info 0] == "enabled"} {
                $path.tb insert end "  \[" {} "Enabled" enabled "\]"
                incr num_enabled
            } else {
                $path.tb insert end "  \[" {} "Disabled" disabled "\]"
                incr num_disabled
            }
        }
        $path.tb insert end "\n"
    }
    $path.tb configure -state $curstate
    list [llength $rules] $num_enabled $num_disabled
}
proc Apol_Widget::appendSearchResultSynAVRules {path indent rules {varname {}}} {
    set curstate [$path.tb cget -state]
    $path.tb configure -state normal
    if {$varname != {}} {
        upvar $varname progressvar
        set progressvar "Rendering [llength $rules] syntactic av rule(s)..."
        update idletasks
    }
    set num_enabled 0
    set num_disabled 0
    if {[ApolTop::is_capable "line numbers"]} {
        set do_linenums 1
    } else {
        set do_linenums 0
    }
    foreach r $rules {
        $path.tb insert end [string repeat " " $indent]
        foreach {rule_type source_set target_set class perms line_num cond_info} [apol_RenderSynAVRule $r] {break}
        if {$do_linenums} {
            set text [list \[ {} \
                          $line_num linenum \
                          "\] " {} \
                          $rule_type {}]
        } else {
            set text [list $rule_type {}]
        }
        set source_set [_render_typeset $source_set]
        set target_set [_render_typeset $target_set]
        if {[llength $class] > 1} {
            set class "\{$class\}"
        }
        lappend text " $source_set $target_set" {}
        if {[llength $perms] > 1} {
            set perms "\{$perms\}"
        }
        lappend text " : $class $perms;" {}
        eval $path.tb insert end $text
        if {$cond_info != {}} {
            if {[lindex $cond_info 0] == "enabled"} {
                $path.tb insert end "  \[" {} "Enabled" enabled "\]"
                incr num_enabled
            } else {
                $path.tb insert end "  \[" {} "Disabled" disabled "\]"
                incr num_disabled
            }
        }
        $path.tb insert end "\n"
    }
    $path.tb configure -state $curstate
    list [llength $rules] $num_enabled $num_disabled
}
proc Apol_Widget::appendSearchResultTERules {path indent rule_list {varname {}}} {
    set curstate [$path.tb cget -state]
    $path.tb configure -state normal
    if {$varname != {}} {
        upvar $varname progressvar
        set progressvar "Sorting [llength $rule_list] semantic type rule(s)..."
        update idletasks
    }
    set rules [lsort -command apol_RenderTERuleComp [lsort -unique $rule_list]]
    if {$varname != {}} {
        set progressvar "Rendering [llength $rules] semantic type rule(s)..."
        update idletasks
    }
    set num_enabled 0
    set num_disabled 0
    foreach r $rules {
        $path.tb insert end [string repeat " " $indent]
        foreach {rule_type source_set target_set class default_type cond_info} [apol_RenderTERule $r] {break}
        set text [list "$rule_type $source_set $target_set" {}]
        lappend text " : $class $default_type;" {}
        eval $path.tb insert end $text
        if {$cond_info != {}} {
            if {[lindex $cond_info 0] == "enabled"} {
                $path.tb insert end "  \[" {} "Enabled" enabled "\]"
                incr num_enabled
            } else {
                $path.tb insert end "  \[" {} "Disabled" disabled "\]"
                incr num_disabled
            }
        }
        $path.tb insert end "\n"
    }
    $path.tb configure -state $curstate
    list [llength $rules] $num_enabled $num_disabled
}
proc Apol_Widget::appendSearchResultSynTERules {path indent rules {varname {}}} {
    set curstate [$path.tb cget -state]
    $path.tb configure -state normal
    if {$varname != {}} {
        upvar $varname progressvar
        set progressvar "Rendering [llength $rules] syntactic type rule(s)..."
        update idletasks
    }
    set num_enabled 0
    set num_disabled 0
    if {[ApolTop::is_capable "line numbers"]} {
        set do_linenums 1
    } else {
        set do_linenums 0
    }
    foreach r $rules {
        $path.tb insert end [string repeat " " $indent]
        foreach {rule_type source_set target_set class default_type line_num cond_info} [apol_RenderSynTERule $r] {break}
        if {$do_linenums} {
            set text [list \[ {} \
                          $line_num linenum \
                          "\] " {} \
                          $rule_type {}]
        } else {
            set text [list $rule_type {}]
        }
        set source_set [_render_typeset $source_set]
        set target_set [_render_typeset $target_set]
        if {[llength $class] > 1} {
            set class "\{$class\}"
        }
        lappend text " $source_set $target_set" {}
        lappend text " : $class $default_type;" {}
        eval $path.tb insert end $text
        if {$cond_info != {}} {
            if {[lindex $cond_info 0] == "enabled"} {
                $path.tb insert end "  \[" {} "Enabled" enabled "\]"
                incr num_enabled
            } else {
                $path.tb insert end "  \[" {} "Disabled" disabled "\]"
                incr num_disabled
            }
        }
        $path.tb insert end "\n"
    }
    $path.tb configure -state $curstate
    list [llength $rules] $num_enabled $num_disabled
}
proc Apol_Widget::gotoLineSearchResults {path line_num} {
    if {![string is integer -strict $line_num]} {
        tk_messageBox -icon error -type ok -title "Invalid line number" \
            -message "$line_num is not a valid line number."
        return
    }
    set textbox $path.tb
    $textbox tag remove sel 0.0 end
    $textbox mark set insert ${line_num}.0
    $textbox see ${line_num}.0
    $textbox tag add sel $line_num.0 $line_num.end
    focus $textbox
}
proc Apol_Widget::showPopupText {title info} {
    variable infoPopup
    if {![winfo exists $infoPopup]} {
        set infoPopup [toplevel .apol_widget_info_popup]
        wm withdraw $infoPopup
        set sw [ScrolledWindow $infoPopup.sw -scrollbar both -auto horizontal]
        set text [text [$sw getframe].text -font {helvetica 10} -wrap none -width 35 -height 10]
        $sw setwidget $text
        pack $sw -expand 1 -fill both
        set b [button $infoPopup.close -text "Close" -command [list destroy $infoPopup]]
        pack $b -side bottom -expand 0 -pady 5
        wm geometry $infoPopup 250x200+50+50
        update
        grid propagate $sw 0
    }
    wm title $infoPopup $title
    set text [$infoPopup.sw getframe].text
    $text configure -state normal
    $text delete 1.0 end
    $text insert 0.0 $info
    $text configure -state disabled
    wm deiconify $infoPopup
    raise $infoPopup
}
proc Apol_Widget::showPopupParagraph {title info} {
    variable infoPopup2
    if {![winfo exists $infoPopup2]} {
        set infoPopup2 [Dialog .apol_widget_info_popup2 -modal none -parent . \
                            -transient false -cancel 0 -default 0 -separator 1]
        $infoPopup2 add -text "Close" -command [list destroy $infoPopup2]
        set sw [ScrolledWindow [$infoPopup2 getframe].sw -auto both -scrollbar both]
        $sw configure -relief sunken
        set text [text [$sw getframe].text -font $ApolTop::text_font \
                      -wrap none -width 75 -height 25 -bg white]
        $sw setwidget $text
        update
        grid propagate $sw 0
        pack $sw -expand 1 -fill both -padx 4 -pady 4
        $infoPopup2 draw
    } else {
        raise $infoPopup2
        wm deiconify $infoPopup2
    }
    $infoPopup2 configure -title $title
    set text [[$infoPopup2 getframe].sw getframe].text
    $text configure -state normal
    $text delete 1.0 end
    $text insert 0.0 $info
    $text configure -state disabled
}
proc Apol_Widget::_listbox_key {listbox key} {
    if {[string length $key] == 1} {
        set values [set ::[$listbox cget -listvar]]
        set x [lsearch $values $key*]
        if {$x >= 0} {
            set curvalue [$listbox get active]
            set curindex [$listbox curselection]
            if {$curindex != "" && [string index $curvalue 0] == $key} {
                set new_x [expr {$curindex + 1}]
                if {[string index [lindex $values $new_x] 0] != $key} {
                    set new_x $x
                }
            } else {
                set new_x $x
            }
            $listbox selection clear 0 end
            $listbox selection set $new_x
            $listbox activate $new_x
            $listbox see $new_x
        }
        event generate $listbox <<ListboxSelect>>
    }
}
proc Apol_Widget::_listbox_double_click {listbox callback_func args} {
    eval $callback_func $args [$listbox get active]
}
proc Apol_Widget::_attrib_enabled {path} {
    variable vars
    if {$vars($path:attribenable)} {
        $path.ab configure -state normal
        _filter_type_combobox $path $vars($path:attrib)
    } else {
        $path.ab configure -state disabled
        _filter_type_combobox $path ""
    }
}
proc Apol_Widget::_attrib_changed {path name1 name2 op} {
    variable vars
    if {$vars($path:attribenable)} {
        _filter_type_combobox $path $vars($name2)
    }
}
proc Apol_Widget::_attrib_validate {path} {
}
proc Apol_Widget::_filter_type_combobox {path attribvalue} {
    variable vars
    if {$attribvalue != ""} {
        set typesList [lsort [lindex [apol_GetAttribs $attribvalue] 0 1]]
        if {$typesList == {}} {
            return
        }
    } else {
        set typesList $Apol_Types::typelist
    }
    if {[lsearch -exact $typesList $vars($path:type)] == -1} {
        set vars($path:type) ""
    }
    $path.tb configure -values $typesList
}
proc Apol_Widget::_sens_changed {path name1 name2 op} {
    variable vars
    [getScrolledListbox $path.cats] selection clear 0 end
    set vars($path:cats) {}
    if {![catch {apol_GetLevels $vars($path:sens)} level_data]} {
        set vars($path:cats) [concat $vars($path:cats) [lindex $level_data 0 2]]
    }
}
proc Apol_Widget::_toggle_regexp_check_button {path name1 name2 op} {
    if {$Apol_Widget::vars($name2)} {
        $path configure -state normal -bg white
    } else {
        $path configure -state disabled -bg $ApolTop::default_bg_color
    }
}
proc Apol_Widget::_hyperlink {path x y} {
    set tb $path.tb
    set range [$tb tag prevrange linenum "@$x,$y + 1 char"]
    $tb tag add selected [lindex $range 0] [lindex $range 1]
    set line_num [$tb get [lindex $range 0] [lindex $range 1]]
    $ApolTop::notebook raise $ApolTop::policy_conf_tab
    Apol_PolicyConf::goto_line $line_num
}
proc Apol_Widget::_render_typeset {typeset} {
    if {[llength $typeset] > 1} {
        if {[lindex $typeset 0] == "~"} {
            set typeset "~\{[lrange $typeset 1 end]\}"
        } else {
            set typeset "\{$typeset\}"
        }
    } else {
        set typeset
    }
}
namespace eval Apol_Widget {
    variable vars
}
proc Apol_Widget::makeRangeSelector {path rangeMatchText {enableText "MLS range"} args} {
    variable vars
    array unset vars $path:*
    set vars($path:range) {{{} {}}}
    set vars($path:range_rendered) {}
    set vars($path:search_type) "exact"
    set f [frame $path]
    set range_frame [frame $f.range]
    set range2_frame [frame $f.range2]
    pack $range_frame $range2_frame -side left -expand 0 -anchor nw
    if {$enableText != {}} {
        set vars($path:enable) 0
        set range_cb [checkbutton $range_frame.enable -text $enableText \
                          -variable Apol_Widget::vars($path:enable)]
        pack $range_cb -side top -expand 0 -anchor nw
        trace add variable Apol_Widget::vars($path:enable) write [list Apol_Widget::_toggle_range_selector $path $range_cb]
    }
    set range_display [eval Entry $range_frame.display -textvariable Apol_Widget::vars($path:range_rendered) -width 20 -editable 0 $args]
    set range_button [button $range_frame.button -text "Select Range..." -state disabled -command [list Apol_Widget::_show_mls_range_dialog $path]]
    trace add variable Apol_Widget::vars($path:range) write [list Apol_Widget::_update_range_display $path]
    pack $range_display -side top -expand 1 -fill x -anchor nw
    pack $range_button -side top -expand 0 -anchor ne
    if {$enableText != {}} {
        pack configure $range_display -padx 4
        pack configure $range_button -padx 4
    }
    set range_label [label $range2_frame.label -text "Range matching:" \
                         -state disabled]
    set range_exact [radiobutton $range2_frame.exact -text "Exact matches" \
                         -state disabled \
                         -value exact -variable Apol_Widget::vars($path:search_type)]
    set range_subset [radiobutton $range2_frame.subset -text "$rangeMatchText containing range" \
                          -state disabled \
                          -value subset -variable Apol_Widget::vars($path:search_type)]
    set range_superset [radiobutton $range2_frame.superset -text "$rangeMatchText within range" \
                            -state disabled \
                            -value superset -variable Apol_Widget::vars($path:search_type)]
    pack $range_label $range_exact $range_subset $range_superset \
        -side top -expand 0 -anchor nw
    return $f
}
proc Apol_Widget::setRangeSelectorState {path newState} {
    if {$newState == 0 || $newState == "disabled"} {
        set new_state disabled
    } else {
        set new_state normal
    }
    foreach w {display button} {
        $path.range.$w configure -state $new_state
    }
    foreach w {label exact subset superset} {
        $path.range2.$w configure -state $new_state
    }
}
proc Apol_Widget::setRangeSelectorCompleteState {path newState} {
    if {$newState == 0 || $newState == "disabled"} {
        set new_state disabled
    } else {
        set new_state normal
    }
    catch {$path.range.enable configure -state $new_state}
}
proc Apol_Widget::clearRangeSelector {path} {
    set Apol_Widget::vars($path:range) {{{} {}}}
    set Apol_Widget::vars($path:search_type) exact
    catch {set Apol_Widget::vars($path:enable) 0}
}
proc Apol_Widget::getRangeSelectorState {path} {
    return $Apol_Widget::vars($path:enable)
}
proc Apol_Widget::getRangeSelectorValue {path} {
    variable vars
    if {[llength $vars($path:range)] == 1} {
        set range [list [lindex $vars($path:range) 0] [lindex $vars($path:range) 0]]
    } else {
        set range $vars($path:range)
    }
    list $range $vars($path:search_type)
}
proc Apol_Widget::_toggle_range_selector {path cb name1 name2 op} {
    if {$Apol_Widget::vars($path:enable)} {
        Apol_Widget::setRangeSelectorState $path normal
    } else {
        Apol_Widget::setRangeSelectorState $path disabled
    }
}
proc Apol_Widget::_show_mls_range_dialog {path} {
    set Apol_Widget::vars($path:range) [Apol_Range_Dialog::getRange $Apol_Widget::vars($path:range)]
}
proc Apol_Widget::_update_range_display {path name1 name2 op} {
    variable vars
    set display $path.range.display
    set low_level [lindex $vars($path:range) 0]
    if {[llength $vars($path:range)] == 1} {
        set high_level $low_level
    } else {
        set high_level [lindex $vars($path:range) 1]
    }
    if {$low_level == {{} {}} && $high_level == {{} {}}} {
        set vars($path:range_rendered) {}
        $display configure -helptext {}
    } else {
        set low_level [apol_RenderLevel $low_level]
        set high_level [apol_RenderLevel $high_level]
        if {$low_level == "" || $high_level == ""} {
            set vars($path:range_rendered) "<invalid MLS range>"
        } else {
            if {$low_level == $high_level} {
                set vars($path:range_rendered) $low_level
            } else {
                set vars($path:range_rendered) "$low_level - $high_level"
            }
        }
        $display configure -helptext $vars($path:range_rendered)
    }
}
namespace eval Apol_Analysis_domaintrans {
    variable vals
    variable widgets
    Apol_Analysis::registerAnalysis "Apol_Analysis_domaintrans" "Domain Transition"
}
proc Apol_Analysis_domaintrans::open {} {
    variable vals
    variable widgets
    Apol_Widget::resetTypeComboboxToPolicy $widgets(type)
    set vals(targets:inc) $Apol_Types::typelist
    set vals(targets:inc_displayed) $Apol_Types::typelist
    foreach c $Apol_Class_Perms::class_list {
        set vals(classes:$c) [lsort [apol_GetAllPermsForClass $c]]
        set vals(classes:$c:enable) 1
    }
}
proc Apol_Analysis_domaintrans::close {} {
    variable widgets
    reinitializeVals
    reinitializeWidgets
    Apol_Widget::clearTypeCombobox $widgets(type)
}
proc Apol_Analysis_domaintrans::getInfo {} {
    return "A forward domain transition analysis will determine all (target)
domains to which a given (source) domain may transition.  For a
forward domain transition to be allowed, multiple forms of access must
be granted:
\n    (1) source domain must have process transition permission for
        target domain,
    (2) source domain must have file execute permission for some
        entrypoint type,
    (3) target domain must have file entrypoint permission for the
        same entrypoint type, and,
    (4) for policies version 15 or later, either a type_transition
        rule or a setexec permission for the source domain.
\nA reverse domain transition analysis will determine all (source)
domains that can transition to a given (target) domain.  For a reverse
domain transition to be allowed, three forms of access must be
granted:
\n    (1) target domain must have process transition permission from the
        source domain,
    (2) target domain must have file entrypoint permission to some
        entrypoint type, and
    (3) source domain must have file execute permission to the same
        entrypoint type.
\nThe results are presented in tree form.  Open target children domains
to perform another domain transition analysis on that domain.
\nFor additional help on this topic select \"Domain Transition Analysis\"
from the help menu."
}
proc Apol_Analysis_domaintrans::create {options_frame} {
    variable vals
    variable widgets
    reinitializeVals
    set dir_tf [TitleFrame $options_frame.dir -text "Direction"]
    pack $dir_tf -side left -padx 2 -pady 2 -expand 0 -fill y
    set dir_forward [radiobutton [$dir_tf getframe].forward -text "Forward" \
                         -variable Apol_Analysis_domaintrans::vals(dir) -value forward]
    set dir_reverse [radiobutton [$dir_tf getframe].reverse -text "Reverse" \
                         -variable Apol_Analysis_domaintrans::vals(dir) -value reverse]
    pack $dir_forward $dir_reverse -anchor w
    trace add variable Apol_Analysis_domaintrans::vals(dir) write \
        Apol_Analysis_domaintrans::toggleDirection
    set req_tf [TitleFrame $options_frame.req -text "Required Parameters"]
    pack $req_tf -side left -padx 2 -pady 2 -expand 0 -fill y
    set l [label [$req_tf getframe].l -textvariable Apol_Analysis_domaintrans::vals(type:label)]
    pack $l -anchor w
    set widgets(type) [Apol_Widget::makeTypeCombobox [$req_tf getframe].type]
    pack $widgets(type)
    set filter_tf [TitleFrame $options_frame.filter -text "Optional Result Filters"]
    pack $filter_tf -side left -padx 2 -pady 2 -expand 1 -fill both
    set access_f [frame [$filter_tf getframe].access]
    pack $access_f -side left -anchor nw
    set widgets(access_enable) [checkbutton $access_f.enable -text "Use access filters" \
                                    -variable Apol_Analysis_domaintrans::vals(access:enable)]
    pack $widgets(access_enable) -anchor w
    set widgets(access) [button $access_f.b -text "Access Filters" \
                             -command Apol_Analysis_domaintrans::createAccessDialog \
                             -state disabled]
    pack $widgets(access) -anchor w -padx 4
    trace add variable Apol_Analysis_domaintrans::vals(access:enable) write \
        Apol_Analysis_domaintrans::toggleAccessSelected
    set widgets(regexp) [Apol_Widget::makeRegexpEntry [$filter_tf getframe].end]
    $widgets(regexp).cb configure -text "Filter result types using regular expression"
    pack $widgets(regexp) -side left -anchor nw -padx 8
}
proc Apol_Analysis_domaintrans::newAnalysis {} {
    if {[set rt [checkParams]] != {}} {
        return $rt
    }
    if {[catch {analyze} results]} {
        return $results
    }
    set f [createResultsDisplay]
    if {[catch {renderResults $f $results} rt]} {
        Apol_Analysis::deleteCurrentResults
        return $rt
    }
    return {}
}
proc Apol_Analysis_domaintrans::updateAnalysis {f} {
    if {[set rt [checkParams]] != {}} {
        return $rt
    }
    if {[catch {analyze} results]} {
        return $results
    }
    clearResultsDisplay $f
    if {[catch {renderResults $f $results} rt]} {
        return $rt
    }
    return {}
}
proc Apol_Analysis_domaintrans::reset {} {
    reinitializeVals
    reinitializeWidgets
}
proc Apol_Analysis_domaintrans::switchTab {query_options} {
    variable vals
    variable widgets
    array set vals $query_options
    if {$vals(type:attrib) != {}} {
        Apol_Widget::setTypeComboboxValue $widgets(type) [list $vals(type) $vals(type:attrib)]
    } else {
        Apol_Widget::setTypeComboboxValue $widgets(type) $vals(type)
    }
    Apol_Widget::setRegexpEntryValue $widgets(regexp) $vals(regexp:enable) $vals(regexp)
}
proc Apol_Analysis_domaintrans::saveQuery {channel} {
    variable vals
    variable widgets
    foreach {key value} [array get vals] {
        switch -- $key {
            targets:inc_displayed -
            classes:perms_displayed -
            search:regexp -
            search:object_types -
            search:classperm_perms {
            }
            default {
                puts $channel "$key $value"
            }
        }
    }
    set type [Apol_Widget::getTypeComboboxValueAndAttrib $widgets(type)]
    puts $channel "type [lindex $type 0]"
    puts $channel "type:attrib [lindex $type 1]"
    set use_regexp [Apol_Widget::getRegexpEntryState $widgets(regexp)]
    set regexp [Apol_Widget::getRegexpEntryValue $widgets(regexp)]
    puts $channel "regexp:enable $use_regexp"
    puts $channel "regexp $regexp"
}
proc Apol_Analysis_domaintrans::loadQuery {channel} {
    variable vals
    set targets_inc {}
    while {[gets $channel line] >= 0} {
        set line [string trim $line]
        if {$line == {} || [string index $line 0] == "#"} {
            continue
        }
        set key {}
        set value {}
        regexp -line -- {^(\S+)( (.+))?} $line -> key --> value
        if {$key == "targets:inc"} {
            lappend targets_inc $value
        } elseif {[regexp -- {^classes:(.+)} $key -> class]} {
            set c($class) $value
        } else {
            set vals($key) $value
        }
    }
    open
    set vals(targets:inc) {}
    foreach s $targets_inc {
        set i [lsearch $Apol_Types::typelist $s]
        if {$i >= 0} {
            lappend vals(targets:inc) $s
        }
    }
    foreach class_key [array names c] {
        if {[regexp -- {^([^:]+):enable} $class_key -> class]} {
            if {[lsearch $Apol_Class_Perms::class_list $class] >= 0} {
                set vals(classes:$class:enable) $c($class_key)
            }
        } else {
            set class $class_key
            set old_p $vals(classes:$class)
            set new_p {}
            foreach p $c($class) {
                if {[lsearch $old_p $p] >= 0} {
                    lappend new_p $p
                }
            }
            set vals(classes:$class) [lsort -uniq $new_p]
        }
    }
    reinitializeWidgets
}
proc Apol_Analysis_domaintrans::gotoLine {tab line_num} {
    set searchResults [$tab.right getframe].res
    Apol_Widget::gotoLineSearchResults $searchResults $line_num
}
proc Apol_Analysis_domaintrans::search {tab str case_Insensitive regExpr srch_Direction } {
    set textbox [$tab.right getframe].res.tb
    ApolTop::textSearch $textbox $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_Analysis_domaintrans::reinitializeVals {} {
    variable vals
    array set vals {
        dir forward
        type:label "Source domain"
        type {}  type:attrib {}
        regexp:enable 0
        regexp {}
        access:enable 0
        targets:inc {}   targets:inc_displayed {}
        targets:attribenable 0  targets:attrb {}
    }
    array unset vals classes:*
    array unset vals search:*
    foreach c $Apol_Class_Perms::class_list {
        set vals(classes:$c) [lsort [apol_GetAllPermsForClass $c]]
        set vals(classes:$c:enable) 1
    }
}
proc Apol_Analysis_domaintrans::reinitializeWidgets {} {
    variable vals
    variable widgets
    if {$vals(type:attrib) != {}} {
        Apol_Widget::setTypeComboboxValue $widgets(type) [list $vals(type) $vals(type:attrib)]
    } else {
        Apol_Widget::setTypeComboboxValue $widgets(type) $vals(type)
    }
    Apol_Widget::setRegexpEntryValue $widgets(regexp) $vals(regexp:enable) $vals(regexp)
}
proc Apol_Analysis_domaintrans::toggleDirection {name1 name2 op} {
    variable vals
    if {$vals(dir) == "forward"} {
        set vals(type:label) "Source domain"
    } elseif {$vals(dir) == "reverse"} {
        set vals(type:label) "Target domain"
    }
    maybeEnableAccess
}
proc Apol_Analysis_domaintrans::toggleAccessSelected {name1 name2 op} {
    maybeEnableAccess
}
proc Apol_Analysis_domaintrans::maybeEnableAccess {} {
    variable vals
    variable widgets
    if {$vals(dir) == "forward"} {
        $widgets(access_enable) configure -state normal
        if {$vals(access:enable)} {
            $widgets(access) configure -state normal
        } else {
            $widgets(access) configure -state disabled
        }
    } else {
        $widgets(access_enable) configure -state disabled
        $widgets(access) configure -state disabled
    }
}
proc Apol_Analysis_domaintrans::createAccessDialog {} {
    destroy .domaintrans_adv
    set d [Dialog .domaintrans_adv -modal local -separator 1 -title "Domain Transition Access Filter" -parent .]
    $d add -text "Close"
    createAccessTargets [$d getframe]
    createAccessClasses [$d getframe]
    $d draw
}
proc Apol_Analysis_domaintrans::createAccessTargets {f} {
    variable vals
    set type_f [frame $f.targets]
    pack $type_f -side left -expand 0 -fill both -padx 4 -pady 4
    set l1 [label $type_f.l1 -text "Included Object Types"]
    pack $l1 -anchor w
    set targets [Apol_Widget::makeScrolledListbox $type_f.targets -height 10 -width 24 \
                 -listvar Apol_Analysis_domaintrans::vals(targets:inc_displayed) \
                 -selectmode extended -exportselection 0]
    set targets_lb [Apol_Widget::getScrolledListbox $targets]
    bind $targets_lb <<ListboxSelect>> \
        [list Apol_Analysis_domaintrans::selectTargetListbox $targets_lb]
    pack $targets -expand 0 -fill both
    set bb [ButtonBox $type_f.bb -homogeneous 1 -spacing 4]
    $bb add -text "Include All" \
        -command [list Apol_Analysis_domaintrans::includeAllItems $targets_lb targets]
    $bb add -text "Ignore All" \
        -command [list Apol_Analysis_domaintrans::ignoreAllItems $targets_lb targets]
    pack $bb -pady 4
    set attrib [frame $type_f.a]
    pack $attrib
    set attrib_enable [checkbutton $attrib.ae -anchor w \
                           -text "Filter by attribute" \
                           -variable Apol_Analysis_domaintrans::vals(targets:attribenable)]
    set attrib_box [ComboBox $attrib.ab -autopost 1 -entrybg white -width 16 \
                        -values $Apol_Types::attriblist \
                        -textvariable Apol_Analysis_domaintrans::vals(targets:attrib)]
    $attrib_enable configure -command \
        [list Apol_Analysis_domaintrans::attribEnabled $attrib_box $targets_lb]
    trace remove variable Apol_Analysis_domaintrans::vals(targets:attrib) write \
        [list Apol_Analysis_domaintrans::attribChanged $targets_lb]
    trace add variable Apol_Analysis_domaintrans::vals(targets:attrib) write \
        [list Apol_Analysis_domaintrans::attribChanged $targets_lb]
    pack $attrib_enable -side top -expand 0 -fill x -anchor sw -padx 5 -pady 2
    pack $attrib_box -side top -expand 1 -fill x -padx 10
    attribEnabled $attrib_box $targets_lb
    if {[set anchor [lindex [lsort [$targets_lb curselection]] 0]] != {}} {
        $targets_lb selection anchor $anchor
        $targets_lb see $anchor
    }
}
proc Apol_Analysis_domaintrans::selectTargetListbox {lb} {
    variable vals
    for {set i 0} {$i < [$lb index end]} {incr i} {
        set t [$lb get $i]
        if {[$lb selection includes $i]} {
            lappend vals(targets:inc) $t
        } else {
            if {[set j [lsearch $vals(targets:inc) $t]] >= 0} {
                set vals(targets:inc) [lreplace $vals(targets:inc) $j $j]
            }
        }
    }
    set vals(targets:inc) [lsort -uniq $vals(targets:inc)]
    focus $lb
}
proc Apol_Analysis_domaintrans::includeAllItems {lb varname} {
    variable vals
    $lb selection set 0 end
    set displayed [$lb get 0 end]
    set vals($varname:inc) [lsort -uniq [concat $vals($varname:inc) $displayed]]
}
proc Apol_Analysis_domaintrans::ignoreAllItems {lb varname} {
    variable vals
    $lb selection clear 0 end
    set displayed [$lb get 0 end]
    set inc {}
    foreach t $vals($varname:inc) {
        if {[lsearch $displayed $t] == -1} {
            lappend inc $t
        }
    }
    set vals($varname:inc) $inc
}
proc Apol_Analysis_domaintrans::attribEnabled {cb lb} {
    variable vals
    if {$vals(targets:attribenable)} {
        $cb configure -state normal
        filterTypeLists $vals(targets:attrib) $lb
    } else {
        $cb configure -state disabled
        filterTypeLists "" $lb
    }
}
proc Apol_Analysis_domaintrans::attribChanged {lb name1 name2 op} {
    variable vals
    if {$vals(targets:attribenable)} {
        filterTypeLists $vals(targets:attrib) $lb
    }
}
proc Apol_Analysis_domaintrans::filterTypeLists {attrib lb} {
    variable vals
    $lb selection clear 0 end
    if {$attrib != ""} {
        set vals(targets:inc_displayed) [lsort [lindex [apol_GetAttribs $attrib] 0 1]]
    } else {
        set vals(targets:inc_displayed) $Apol_Types::typelist
    }
    foreach t $vals(targets:inc) {
        if {[set i [lsearch $vals(targets:inc_displayed) $t]] >= 0} {
            $lb selection set $i $i
        }
    }
}
proc Apol_Analysis_domaintrans::createAccessClasses {f} {
    variable vals
    variable widgets
    set lf [frame $f.left]
    pack $lf -side left -expand 0 -fill both -padx 4 -pady 4
    set l1 [label $lf.l -text "Included Object Classes"]
    pack $l1 -anchor w
    set rf [frame $f.right]
    pack $rf -side left -expand 0 -fill both -padx 4 -pady 4
    set l2 [label $rf.l]
    pack $l2 -anchor w
    set classes [Apol_Widget::makeScrolledListbox $lf.classes -height 10 -width 24 \
                     -listvar Apol_Class_Perms::class_list \
                     -selectmode extended -exportselection 0]
    set classes_lb [Apol_Widget::getScrolledListbox $classes]
    pack $classes -expand 1 -fill both
    set cbb [ButtonBox $lf.cbb -homogeneous 1 -spacing 4]
    $cbb add -text "Include All" \
        -command [list Apol_Analysis_domaintrans::includeAllClasses $classes_lb]
    $cbb add -text "Ignore All" \
        -command [list Apol_Analysis_domaintrans::ignoreAllClasses $classes_lb]
    pack $cbb -pady 4 -expand 0
    set perms [Apol_Widget::makeScrolledListbox $rf.perms -height 10 -width 24 \
                     -listvar Apol_Analysis_domaintrans::vals(classes:perms_displayed) \
                     -selectmode extended -exportselection 0]
    set perms_lb [Apol_Widget::getScrolledListbox $perms]
    pack $perms -expand 1 -fill both
    set pbb [ButtonBox $rf.pbb -homogeneous 1 -spacing 4]
    $pbb add -text "Include All" \
        -command [list Apol_Analysis_domaintrans::includeAllPerms $classes_lb $perms_lb]
    $pbb add -text "Ignore All" \
        -command [list Apol_Analysis_domaintrans::ignoreAllPerms $classes_lb $perms_lb]
    pack $pbb -pady 4 -expand 0
    bind $classes_lb <<ListboxSelect>> \
        [list Apol_Analysis_domaintrans::selectClassListbox $l2 $classes_lb $perms_lb]
    bind $perms_lb <<ListboxSelect>> \
        [list Apol_Analysis_domaintrans::selectPermListbox $classes_lb $perms_lb]
    foreach class_key [array names vals classes:*:enable] {
        if {$vals($class_key)} {
            regexp -- {^classes:([^:]+):enable} $class_key -> class
            set i [lsearch $Apol_Class_Perms::class_list $class]
            $classes_lb selection set $i $i
        }
    }
    if {[set anchor [lindex [lsort [$classes_lb curselection]] 0]] != {}} {
        $classes_lb selection anchor $anchor
        $classes_lb see $anchor
    }
    set vals(classes:perms_displayed) {}
    selectClassListbox $l2 $classes_lb $perms_lb
}
proc Apol_Analysis_domaintrans::selectClassListbox {perm_label lb plb} {
    variable vals
    for {set i 0} {$i < [$lb index end]} {incr i} {
        set c [$lb get $i]
        set vals(classes:$c:enable) [$lb selection includes $i]
    }
    if {[set class [$lb get anchor]] == {}} {
        $perm_label configure -text "Permissions"
        return
    }
    $perm_label configure -text "Permissions for $class"
    set vals(classes:perms_displayed) [lsort [apol_GetAllPermsForClass $class]]
    $plb selection clear 0 end
    foreach p $vals(classes:$class) {
        set i [lsearch $vals(classes:perms_displayed) $p]
        $plb selection set $i
    }
    if {[set anchor [lindex [lsort [$plb curselection]] 0]] != {}} {
        $plb selection anchor $anchor
        $plb see $anchor
    }
    focus $lb
}
proc Apol_Analysis_domaintrans::includeAllClasses {lb} {
    variable vals
    $lb selection set 0 end
    foreach c $Apol_Class_Perms::class_list {
        set vals(classes:$c:enable) 1
    }
}
proc Apol_Analysis_domaintrans::ignoreAllClasses {lb} {
    variable vals
    $lb selection clear 0 end
    foreach c $Apol_Class_Perms::class_list {
        set vals(classes:$c:enable) 0
    }
}
proc Apol_Analysis_domaintrans::selectPermListbox {lb plb} {
    variable vals
    set class [$lb get anchor]
    set p {}
    foreach i [$plb curselection] {
        lappend p [$plb get $i]
    }
    set vals(classes:$class) $p
    focus $plb
}
proc Apol_Analysis_domaintrans::includeAllPerms {lb plb} {
    variable vals
    set class [$lb get anchor]
    $plb selection set 0 end
    set vals(classes:$class) $vals(classes:perms_displayed)
}
proc Apol_Analysis_domaintrans::ignoreAllPerms {lb plb} {
    variable vals
    set class [$lb get anchor]
    $plb selection clear 0 end
    set vals(classes:$class) {}
}
proc Apol_Analysis_domaintrans::checkParams {} {
    variable vals
    variable widgets
    if {![ApolTop::is_policy_open]} {
        return "No current policy file is opened!"
    }
    set type [Apol_Widget::getTypeComboboxValueAndAttrib $widgets(type)]
    if {[lindex $type 0] == {}} {
        return "No type was selected."
    }
    set vals(type) [lindex $type 0]
    set vals(type:attrib) [lindex $type 1]
    set use_regexp [Apol_Widget::getRegexpEntryState $widgets(regexp)]
    set regexp [Apol_Widget::getRegexpEntryValue $widgets(regexp)]
    if {$use_regexp && $regexp == {}} {
            return "No regular expression provided."
    }
    set vals(regexp:enable) $use_regexp
    set vals(regexp) $regexp
    if {$vals(dir) == "forward" && $vals(access:enable)} {
        set classperm_pairs {}
        foreach class $Apol_Class_Perms::class_list {
            if {$vals(classes:$class:enable) == 0} {
                continue
            }
            if {$vals(classes:$class) == {}} {
                return "No permissions were selected for class $class."
            }
            foreach perm $vals(classes:$class) {
                lappend classperm_pairs [list $class $perm]
            }
        }
        if {$vals(targets:inc) == {}} {
            return "No object types were selected."
        }
        if {$classperm_pairs == {}} {
            return "No object classes were selected."
        }
        set vals(search:object_types) $vals(targets:inc)
        set vals(search:classperm_pairs) $classperm_pairs
    } else {
        set vals(search:object_types) {}
        set vals(search:classperm_pairs) {}
    }
    if {$vals(regexp:enable)} {
        set vals(search:regexp) $vals(regexp)
    } else {
        set vals(search:regexp) {}
    }
    return {}  ;# all parameters passed, now ready to do search
}
proc Apol_Analysis_domaintrans::analyze {} {
    variable vals
    apol_DomainTransitionAnalysis $vals(dir) $vals(type) $vals(search:object_types) $vals(search:classperm_pairs) $vals(search:regexp)
}
proc Apol_Analysis_domaintrans::analyzeMore {tree node analysis_args} {
    set new_start [$tree itemcget $node -text]
    if {[$tree itemcget [$tree parent $node] -text] == $new_start} {
        return {}
    }
    foreach {dir orig_type object_types classperm_pairs regexp} $analysis_args {break}
    apol_DomainTransitionAnalysis $dir $new_start $object_types $classperm_pairs $regexp
}
proc Apol_Analysis_domaintrans::createResultsDisplay {} {
    variable vals
    set f [Apol_Analysis::createResultTab "Domain Trans" [array get vals]]
    if {$vals(dir) == "forward"} {
        set tree_title "Forward Domain Transition"
    } else {
        set tree_title "Reverse Domain Transition"
    }
    set tree_tf [TitleFrame $f.left -text $tree_title]
    pack $tree_tf -side left -expand 0 -fill y -padx 2 -pady 2
    set sw [ScrolledWindow [$tree_tf getframe].sw -auto both]
    set tree [Tree [$sw getframe].tree -width 24 -redraw 1 -borderwidth 0 \
                  -highlightthickness 0 -showlines 1 -padx 0 -bg white]
    $sw setwidget $tree
    pack $sw -expand 1 -fill both
    set res_tf [TitleFrame $f.right -text "Domain Transition Results"]
    pack $res_tf -side left -expand 1 -fill both -padx 2 -pady 2
    set res [Apol_Widget::makeSearchResults [$res_tf getframe].res]
    $res.tb tag configure title -font {Helvetica 14 bold}
    $res.tb tag configure title_type -foreground blue -font {Helvetica 14 bold}
    $res.tb tag configure subtitle -font {Helvetica 10 bold}
    $res.tb tag configure num -foreground blue -font {Helvetica 10 bold}
    pack $res -expand 1 -fill both
    $tree configure -selectcommand [list Apol_Analysis_domaintrans::treeSelect $res]
    $tree configure -opencmd [list Apol_Analysis_domaintrans::treeOpen $tree]
    return $f
}
proc Apol_Analysis_domaintrans::treeSelect {res tree node} {
    if {$node != {}} {
        $res.tb configure -state normal
        $res.tb delete 0.0 end
        set data [$tree itemcget $node -data]
        if {[string index $node 0] == "f" || [string index $node 0] == "r"} {
            renderResultsDTA $res $tree $node [lindex $data 1]
        } else {
            eval $res.tb insert end $data
        }
        $res.tb configure -state disabled
    }
}
proc Apol_Analysis_domaintrans::treeOpen {tree node} {
    foreach {search_crit results} [$tree itemcget $node -data] {break}
    if {([string index $node 0] == "f" || [string index $node 0] == "r") && $search_crit != {}} {
        ApolTop::setBusyCursor
        update idletasks
        set retval [catch {analyzeMore $tree $node $search_crit} new_results]
        ApolTop::resetBusyCursor
        if {$retval} {
            tk_messageBox -icon error -type ok -title "Domain Transition Analysis" -message "Could not perform additional analysis:\n\n$new_results"
        } else {
            $tree itemconfigure $node -data [list {} $results]
            createResultsNodes $tree $node $new_results $search_crit
        }
    }
}
proc Apol_Analysis_domaintrans::clearResultsDisplay {f} {
    variable vals
    set tree [[$f.left getframe].sw getframe].tree
    set res [$f.right getframe].res
    $tree delete [$tree nodes root]
    Apol_Widget::clearSearchResults $res
    Apol_Analysis::setResultTabCriteria [array get vals]
}
proc Apol_Analysis_domaintrans::renderResults {f results} {
    variable vals
    set tree [[$f.left getframe].sw getframe].tree
    set res [$f.right getframe].res
    $tree insert end root top -text $vals(type) -open 1 -drawcross auto
    set top_text [renderTopText]
    $tree itemconfigure top -data $top_text
    set search_crit [list $vals(dir) $vals(type) $vals(search:object_types) $vals(search:classperm_pairs) $vals(search:regexp)]
    createResultsNodes $tree top $results $search_crit
    $tree selection set top
    $tree opentree top 0
    update idletasks
    $tree see top
}
proc Apol_Analysis_domaintrans::renderTopText {} {
    variable vals
    if {$vals(dir) == "forward"} {
        set top_text [list "Forward Domain Transition Analysis: Starting Type: " title]
    } else {
        set top_text [list "Reverse Domain Transition Analysis: Starting Type: " title]
    }
    lappend top_text $vals(type) title_type \
        "\n\n" title
    if {$vals(dir) == "forward"} {
        lappend top_text \
"This tab provides the results of a forward domain transition analysis
starting from the source domain type above.  The results of this
analysis are presented in tree form with the root of the tree (this
node) being the start point for the analysis.
\nEach child node in the tree represents a TARGET DOMAIN TYPE.  A target
domain type is a domain to which the source domain may transition.
You can follow the domain transition tree by opening each subsequent
generation of children in the tree.\n" {}
    } else {
        lappend top_text \
"This tab provides the results of a reverse domain transition analysis
given the target domain type above.  The results of this analysis are
presented in tree form with the root of the tree (this node) being the
target point of the analysis.
\nEach child node in the tree represents a source DOMAIN TYPE.  A source
domain type is a domain that can transition to the target domain.  You
can follow the domain transition tree by opening each subsequent
generation of children in the tree.\n" {}
    }
    lappend top_text \
"\nNOTE: For any given generation, if the parent and the child are the
same, you cannot open the child. This avoids cyclic analyses.
\nThe criteria that defines an allowed domain transition are:
\n1) There must be at least one rule that allows TRANSITION access for
   PROCESS objects between the SOURCE and TARGET domain types.
\n2) There must be at least one FILE TYPE that allows the TARGET type
   ENTRYPOINT access for FILE objects.
\n3) There must be at least one FILE TYPE that meets criterion 2) above
   and allows the SOURCE type EXECUTE access for FILE objects.
\nThe information window shows all the rules and file types that meet
these criteria for each target domain type.
\nFUTURE NOTE: In the future we also plan to show the type_transition
rules that provide for a default domain transitions.  While such rules
cause a domain transition to occur by default, they do not allow it.
Thus, associated type_transition rules are not truly part of the
definition of allowed domain transition" {}
}
proc Apol_Analysis_domaintrans::createResultsNodes {tree parent_node results search_crit} {
    set dir [lindex $search_crit 0]
    foreach r $results {
        foreach {source target intermed proctrans entrypoint execute setexec type_trans access_list} $r {break}
        if {$dir == "forward"} {
            set key $target
            set node f:\#auto
        } else {
            set key $source
            set node r:\#auto
        }
        foreach p $proctrans {
            lappend types($key) $p
        }
        if {[info exists types($key:setexec)]} {
            set types($key:setexec) [concat $types($key:setexec) $setexec]
        } else {
            set types($key:setexec) $setexec
        }
        lappend types($key:inter) $intermed
        foreach e $entrypoint {
            lappend types($key:inter:$intermed:entry) $e
        }
        foreach e $execute {
            lappend types($key:inter:$intermed:exec) $e
        }
        if {[info exists types($key:inter:$intermed:type_trans)]} {
            set types($key:inter:$intermed:type_trans) [concat $types($key:inter:$intermed:type_trans) $type_trans]
        } else {
            set types($key:inter:$intermed:type_trans) $type_trans
        }
        if {[info exists types($key:access)]} {
            set types($key:access) [concat $types($key:access) $access_list]
        } else {
            set types($key:access) $access_list
        }
    }
    foreach key [lsort [array names types]] {
        if {[string first : $key] != -1} {
            continue
        }
        set ep {}
        set proctrans [lsort -uniq $types($key)]
        set setexec [lsort -uniq $types($key:setexec)]
        foreach intermed [lsort -uniq $types($key:inter)] {
            lappend ep [list $intermed \
                            [lsort -uniq $types($key:inter:$intermed:entry)] \
                            [lsort -uniq $types($key:inter:$intermed:exec)] \
                            [lsort -uniq $types($key:inter:$intermed:type_trans)]]
        }
        set access_list [lsort -uniq $types($key:access)]
        set data [list $proctrans $setexec $ep $access_list]
        $tree insert end $parent_node $node -text $key -drawcross allways \
            -data [list $search_crit $data]
    }
}
proc Apol_Analysis_domaintrans::renderResultsDTA {res tree node data} {
    set parent_name [$tree itemcget [$tree parent $node] -text]
    set name [$tree itemcget $node -text]
    foreach {proctrans setexec ep access_list} $data {break}
    if {[string index $node 0] == "f"} {
        set header [list "Domain transition from " title \
                        $parent_name title_type \
                        " to " title \
                        $name title_type]
    } else {
        set header [list "Domain transition from " title \
                        $name title_type \
                        " to " title \
                        $parent_name title_type]
    }
    eval $res.tb insert end $header
    $res.tb insert end "\n\n" title_type
    $res.tb insert end "Process Transition Rules: " subtitle \
        [llength $proctrans] num \
        "\n" subtitle
    Apol_Widget::appendSearchResultAVRules $res 6 $proctrans
    if {[llength $setexec] > 0} {
        $res.tb insert end "\n" {} \
            "Setexec Rules: " subtitle \
            [llength $setexec] num \
            "\n" subtitle
        Apol_Widget::appendSearchResultAVRules $res 6 $setexec
    }
    $res.tb insert end "\nEntry Point File Types: " subtitle \
        [llength $ep] num
    foreach e [lsort -index 0 $ep] {
        foreach {intermed entrypoint execute type_trans} $e {break}
        $res.tb insert end "\n      $intermed\n" {} \
            "            " {} \
            "File Entrypoint Rules: " subtitle \
            [llength $entrypoint] num \
            "\n" subtitle
        Apol_Widget::appendSearchResultAVRules $res 12 $entrypoint
        $res.tb insert end "\n" {} \
            "            " {} \
            "File Execute Rules: " subtitle \
            [llength $execute] num \
            "\n" subtitle
        Apol_Widget::appendSearchResultAVRules $res 12 $execute
        if {[llength $type_trans] > 0} {
            $res.tb insert end "\n" {} \
                "            " {} \
                "Type_transition Rules: " subtitle \
                [llength $type_trans] num \
                "\n" subtitle
            Apol_Widget::appendSearchResultTERules $res 12 $type_trans
        }
    }
    if {[llength $access_list] > 0} {
        $res.tb insert end "\n" {} \
            "The access filters you specified returned the following rules: " subtitle \
            [llength $access_list] num \
            "\n" subtitle
        Apol_Widget::appendSearchResultAVRules $res 6 $access_list
    }
}
namespace eval Apol_Analysis_directflow {
    variable vals
    variable widgets
    Apol_Analysis::registerAnalysis "Apol_Analysis_directflow" "Direct Information Flow"
}
proc Apol_Analysis_directflow::open {} {
    variable vals
    variable widgets
    Apol_Widget::resetTypeComboboxToPolicy $widgets(type)
    set vals(classes:selected) $Apol_Class_Perms::class_list
    Apol_Widget::setScrolledListboxState $widgets(classes) normal
    set classes_lb [Apol_Widget::getScrolledListbox $widgets(classes)]
    $classes_lb selection set 0 end
    toggleClasses {} {} {}
}
proc Apol_Analysis_directflow::close {} {
    variable widgets
    reinitializeVals
    reinitializeWidgets
    Apol_Widget::clearTypeCombobox $widgets(type)
}
proc Apol_Analysis_directflow::getInfo {} {
    return "This analysis generates the results of a Direct Information Flow
analysis beginning from the starting type selected.  The results of
the analysis are presented in tree form with the root of the tree
being the start point for the analysis.
\nEach child node in the tree represents a type in the current policy
for which there is a direct information flow to or from its parent
node.  If 'in' was selected then the information flow is from the
child to the parent.  If 'out' was selected then information flows
from the parent to the child.
\nThe results of the analysis may be optionally filtered by object class
selection or an end type regular expression.
\nNOTE: For any given generation, if the parent and the child are the
same, the child cannot be opened.  This avoids cyclic analyses.
\nFor additional help on this topic select \"Information Flow Analysis\"
from the help menu."
}
proc Apol_Analysis_directflow::create {options_frame} {
    variable vals
    variable widgets
    reinitializeVals
    set dir_tf [TitleFrame $options_frame.mode -text "Direction"]
    pack $dir_tf -side left -padx 2 -pady 2 -expand 0 -fill y
    set dir_in [radiobutton [$dir_tf getframe].in -text In -value in \
                    -variable Apol_Analysis_directflow::vals(dir)]
    set dir_out [radiobutton [$dir_tf getframe].out -text Out -value out \
                     -variable Apol_Analysis_directflow::vals(dir)]
    set dir_either [radiobutton [$dir_tf getframe].either -text Either -value either \
                        -variable Apol_Analysis_directflow::vals(dir)]
    set dir_both [radiobutton [$dir_tf getframe].both -text Both -value both \
                         -variable Apol_Analysis_directflow::vals(dir)]
    pack $dir_in $dir_out $dir_either $dir_both -anchor w
    set req_tf [TitleFrame $options_frame.req -text "Required Parameters"]
    pack $req_tf -side left -padx 2 -pady 2 -expand 0 -fill y
    set l [label [$req_tf getframe].l -text "Starting type"]
    pack $l -anchor w
    set widgets(type) [Apol_Widget::makeTypeCombobox [$req_tf getframe].type]
    pack $widgets(type)
    set filter_tf [TitleFrame $options_frame.filter -text "Optional Result Filters"]
    pack $filter_tf -side left -padx 2 -pady 2 -expand 1 -fill both
    set class_f [frame [$filter_tf getframe].class]
    pack $class_f -side left -anchor nw
    set class_enable [checkbutton $class_f.enable -text "Filter by object class" \
                          -variable Apol_Analysis_directflow::vals(classes:enable)]
    pack $class_enable -anchor w
    set widgets(classes) [Apol_Widget::makeScrolledListbox $class_f.classes \
                              -height 6 -width 24 \
                              -listvar Apol_Class_Perms::class_list \
                              -selectmode multiple -exportselection 0]
    set classes_lb [Apol_Widget::getScrolledListbox $widgets(classes)]
    bind $classes_lb <<ListboxSelect>> \
        [list Apol_Analysis_directflow::selectClassesListbox $classes_lb]
    pack $widgets(classes) -padx 4 -expand 0 -fill both
    trace add variable Apol_Analysis_directflow::vals(classes:enable) write \
        Apol_Analysis_directflow::toggleClasses
    Apol_Widget::setScrolledListboxState $widgets(classes) disabled
    set classes_bb [ButtonBox $class_f.bb -homogeneous 1 -spacing 4]
    $classes_bb add -text "Include All" \
        -command [list Apol_Analysis_directflow::includeAll $classes_lb]
    $classes_bb add -text "Exclude All"  \
        -command [list Apol_Analysis_directflow::excludeAll $classes_lb]
    pack $classes_bb -pady 4
    set widgets(regexp) [Apol_Widget::makeRegexpEntry [$filter_tf getframe].end]
    $widgets(regexp).cb configure -text "Filter result types using regular expression"
    pack $widgets(regexp) -side left -anchor nw -padx 8
}
proc Apol_Analysis_directflow::newAnalysis {} {
    if {[set rt [checkParams]] != {}} {
        return $rt
    }
    if {[catch {analyze} results]} {
        return $results
    }
    set f [createResultsDisplay]
    if {[catch {renderResults $f $results} rt]} {
        Apol_Analysis::deleteCurrentResults
        return $rt
    }
    return {}
}
proc Apol_Analysis_directflow::updateAnalysis {f} {
    if {[set rt [checkParams]] != {}} {
        return $rt
    }
    if {[catch {analyze} results]} {
        return $results
    }
    clearResultsDisplay $f
    if {[catch {renderResults $f $results} rt]} {
        return $rt
    }
    return {}
}
proc Apol_Analysis_directflow::reset {} {
    reinitializeVals
    reinitializeWidgets
}
proc Apol_Analysis_directflow::switchTab {query_options} {
    variable vals
    variable widgets
    array set vals $query_options
    reinitializeWidgets
}
proc Apol_Analysis_directflow::saveQuery {channel} {
    variable vals
    variable widgets
    foreach {key value} [array get vals] {
        puts $channel "$key $value"
    }
    set type [Apol_Widget::getTypeComboboxValueAndAttrib $widgets(type)]
    puts $channel "type [lindex $type 0]"
    puts $channel "type:attrib [lindex $type 1]"
    set use_regexp [Apol_Widget::getRegexpEntryState $widgets(regexp)]
    set regexp [Apol_Widget::getRegexpEntryValue $widgets(regexp)]
    puts $channel "regexp:enable $use_regexp"
    puts $channel "regexp $regexp"
}
proc Apol_Analysis_directflow::loadQuery {channel} {
    variable vals
    set classes {}
    while {[gets $channel line] >= 0} {
        set line [string trim $line]
        if {$line == {} || [string index $line 0] == "#"} {
            continue
        }
        set key {}
        set value {}
        regexp -line -- {^(\S+)( (.+))?} $line -> key --> value
        switch -- $key {
            classes:selected {
                set classes $value
            }
            default {
                set vals($key) $value
            }
        }
    }
    open
    set vals(classes:selected) {}
    foreach c $classes {
        set i [lsearch $Apol_Class_Perms::class_list $c]
        if {$i >= 0} {
            lappend vals(classes:selected) $c
        }
    }
    set vals(classes:selected) [lsort $vals(classes:selected)]
    reinitializeWidgets
}
proc Apol_Analysis_directflow::gotoLine {tab line_num} {
    set searchResults [$tab.right getframe].res
    Apol_Widget::gotoLineSearchResults $searchResults $line_num
}
proc Apol_Analysis_directflow::search {tab str case_Insensitive regExpr srch_Direction } {
    set textbox [$tab.right getframe].res.tb
    ApolTop::textSearch $textbox $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_Analysis_directflow::reinitializeVals {} {
    variable vals
    array set vals {
        dir in
        type {}  type:attrib {}
        classes:enable 0
        classes:selected {}
        regexp:enable 0
        regexp {}
    }
}
proc Apol_Analysis_directflow::reinitializeWidgets {} {
    variable vals
    variable widgets
    if {$vals(type:attrib) != {}} {
        Apol_Widget::setTypeComboboxValue $widgets(type) [list $vals(type) $vals(type:attrib)]
    } else {
        Apol_Widget::setTypeComboboxValue $widgets(type) $vals(type)
    }
    Apol_Widget::setRegexpEntryValue $widgets(regexp) $vals(regexp:enable) $vals(regexp)
    Apol_Widget::setScrolledListboxState $widgets(classes) enabled
    set classes_lb [Apol_Widget::getScrolledListbox $widgets(classes)]
    $classes_lb selection clear 0 end
    foreach c $vals(classes:selected) {
        set i [lsearch $Apol_Class_Perms::class_list $c]
        $classes_lb selection set $i $i
    }
    toggleClasses {} {} {}
}
proc Apol_Analysis_directflow::toggleClasses {name1 name2 op} {
    variable vals
    variable widgets
    if {$vals(classes:enable)} {
        Apol_Widget::setScrolledListboxState $widgets(classes) enabled
    } else {
        Apol_Widget::setScrolledListboxState $widgets(classes) disabled
    }
}
proc Apol_Analysis_directflow::selectClassesListbox {lb} {
    variable vals
    for {set i 0} {$i < [$lb index end]} {incr i} {
        set t [$lb get $i]
        if {[$lb selection includes $i]} {
            lappend vals(classes:selected) $t
        } else {
            if {[set j [lsearch $vals(classes:selected) $t]] >= 0} {
                set vals(classes:selected) [lreplace $vals(classes:selected) $j $j]
            }
        }
    }
    set vals(classes:selected) [lsort -uniq $vals(classes:selected)]
    focus $lb
}
proc Apol_Analysis_directflow::includeAll {lb} {
    variable vals
    $lb selection set 0 end
    set vals(classes:selected) $Apol_Class_Perms::class_list
}
proc Apol_Analysis_directflow::excludeAll {lb} {
    variable vals
    $lb selection clear 0 end
    set vals(classes:selected) {}
}
proc Apol_Analysis_directflow::checkParams {} {
    variable vals
    variable widgets
    if {![ApolTop::is_policy_open]} {
        return "No current policy file is opened!"
    }
    set type [Apol_Widget::getTypeComboboxValueAndAttrib $widgets(type)]
    if {[lindex $type 0] == {}} {
        return "No type was selected."
    }
    set vals(type) [lindex $type 0]
    set vals(type:attrib) [lindex $type 1]
    set use_regexp [Apol_Widget::getRegexpEntryState $widgets(regexp)]
    set regexp [Apol_Widget::getRegexpEntryValue $widgets(regexp)]
    if {$use_regexp && $regexp == {}} {
            return "No regular expression provided."
    }
    set vals(regexp:enable) $use_regexp
    set vals(regexp) $regexp
    if {$vals(classes:enable) && $vals(classes:selected) == {}} {
        return "At least one object class must be included."
    }
    if {![Apol_Perms_Map::is_pmap_loaded]} {
        if {![Apol_Perms_Map::loadDefaultPermMap]} {
            return "This analysis requires that a permission map is loaded."
	}
    }
    return {}  ;# all parameters passed, now ready to do search
}
proc Apol_Analysis_directflow::analyze {} {
    variable vals
    set classes {}
    if {$vals(classes:enable)} {
        foreach c $vals(classes:selected) {
            foreach p [apol_GetAllPermsForClass $c] {
                lappend classes [list $c $p]
            }
        }
    }
    if {$vals(regexp:enable)} {
        set regexp $vals(regexp)
    } else {
        set regexp {}
    }
    apol_DirectInformationFlowAnalysis $vals(dir) $vals(type) $classes $regexp
}
proc Apol_Analysis_directflow::analyzeMore {tree node} {
    set new_start [$tree itemcget $node -text]
    if {[$tree itemcget [$tree parent $node] -text] == $new_start} {
        return {}
    }
    set g [lindex [$tree itemcget top -data] 0]
    apol_DirectInformationFlowMore $g $new_start
}
proc Apol_Analysis_directflow::createResultsDisplay {} {
    variable vals
    set f [Apol_Analysis::createResultTab "Direct Flow" [array get vals]]
    set tree_tf [TitleFrame $f.left -text "Direct Information Flow Tree"]
    pack $tree_tf -side left -expand 0 -fill y -padx 2 -pady 2
    set sw [ScrolledWindow [$tree_tf getframe].sw -auto both]
    set tree [Tree [$sw getframe].tree -width 24 -redraw 1 -borderwidth 0 \
                  -highlightthickness 0 -showlines 1 -padx 0 -bg white]
    $sw setwidget $tree
    pack $sw -expand 1 -fill both
    set res_tf [TitleFrame $f.right -text "Direct Information Flow Results"]
    pack $res_tf -side left -expand 1 -fill both -padx 2 -pady 2
    set res [Apol_Widget::makeSearchResults [$res_tf getframe].res]
    $res.tb tag configure title -font {Helvetica 14 bold}
    $res.tb tag configure title_type -foreground blue -font {Helvetica 14 bold}
    $res.tb tag configure subtitle -font {Helvetica 10 bold}
    $res.tb tag configure subtitle_dir -foreground blue -font {Helvetica 10 bold}
    pack $res -expand 1 -fill both
    $tree configure -selectcommand [list Apol_Analysis_directflow::treeSelect $res]
    $tree configure -opencmd [list Apol_Analysis_directflow::treeOpen $tree]
    bind $tree <Destroy> [list Apol_Analysis_directflow::treeDestroy $tree]
    return $f
}
proc Apol_Analysis_directflow::treeSelect {res tree node} {
    if {$node != {}} {
        $res.tb configure -state normal
        $res.tb delete 0.0 end
        set data [$tree itemcget $node -data]
        if {[string index $node 0] == "x"} {
            renderResultsDirectFlow $res $tree $node [lindex $data 1]
        } else {
            eval $res.tb insert end [lindex $data 1]
        }
        $res.tb configure -state disabled
    }
}
proc Apol_Analysis_directflow::treeOpen {tree node} {
    foreach {is_expanded results} [$tree itemcget $node -data] {break}
    if {[string index $node 0] == "x" && !$is_expanded} {
        ApolTop::setBusyCursor
        update idletasks
        set retval [catch {analyzeMore $tree $node} new_results]
        ApolTop::resetBusyCursor
        if {$retval} {
            tk_messageBox -icon error -type ok -title "Direct Information Flow" -message "Could not perform additional analysis:\n\n$new_results"
        } else {
            $tree itemconfigure $node -data [list 1 $results]
            createResultsNodes $tree $node $new_results 1
        }
    }
}
proc Apol_Analysis_directflow::treeDestroy {tree} {
    set graph_handler [lindex [$tree itemcget top -data] 0]
    apol_InformationFlowDestroy $graph_handler
}
proc Apol_Analysis_directflow::clearResultsDisplay {f} {
    variable vals
    set tree [[$f.left getframe].sw getframe].tree
    set res [$f.right getframe].res
    set graph_handler [lindex [$tree itemcget top -data] 0]
    apol_InformationFlowDestroy $graph_handler
    $tree delete [$tree nodes root]
    Apol_Widget::clearSearchResults $res
    Apol_Analysis::setResultTabCriteria [array get vals]
}
proc Apol_Analysis_directflow::renderResults {f results} {
    variable vals
    set graph_handler [lindex $results 0]
    set results_list [lrange $results 1 end]
    set tree [[$f.left getframe].sw getframe].tree
    set res [$f.right getframe].res
    $tree insert end root top -text $vals(type) -open 1 -drawcross auto
    set top_text [renderTopText]
    $tree itemconfigure top -data [list $graph_handler $top_text]
    createResultsNodes $tree top $results_list 1
    $tree selection set top
    $tree opentree top 0
    update idletasks
    $tree see top
}
proc Apol_Analysis_directflow::renderTopText {} {
    variable vals
    set top_text [list "Direct Information Flow Analysis: Starting type: " title]
    lappend top_text $vals(type) title_type \
        "\n\n" title \
        "This tab provides the results of a Direct Information Flow analysis
beginning from the starting type selected above.  The results of the
analysis are presented in tree form with the root of the tree (this
node) being the start point for the analysis.
\nEach child node in the tree represents a type in the current policy
for which there is a direct information flow to or from (depending on
your selection above) its parent node.
\nNOTE: For any given generation, if the parent and the child are the
same, you cannot open the child.  This avoids cyclic analyses."
}
proc Apol_Analysis_directflow::createResultsNodes {tree parent_node results do_expand} {
    set all_targets {}
    foreach r $results {
        foreach {flow_dir source target rules} $r {break}
        if {!$do_expand} {
            set target [lindex $results 0 2]
        }
        lappend all_targets $target
        foreach r $rules {
            set class [apol_RenderAVRuleClass $r]
            lappend classes($target) $class
            lappend classes($target:$class) $r
        }
        set dir($target:$flow_dir) 1
    }
    foreach t [lsort -uniq $all_targets] {
        if {[info exists dir($t:both)] ||
            ([info exists dir($t:in)] && [info exists dir($t:out)])} {
            set flow_dir "both"
        } elseif {[info exists dir($t:in)]} {
            set flow_dir "in"
        } else {
            set flow_dir "out"
        }
        set rules {}
        foreach c [lsort -uniq $classes($t)] {
            lappend rules [list $c [lsort -uniq $classes($t:$c)]]
        }
        set data [list $flow_dir $rules]
        $tree insert end $parent_node x\#auto -text $t -drawcross allways \
            -data [list 0 $data]
    }
}
proc Apol_Analysis_directflow::renderResultsDirectFlow {res tree node data} {
    set parent_name [$tree itemcget [$tree parent $node] -text]
    set name [$tree itemcget $node -text]
    foreach {flow_dir classes} $data {break}
    switch -- $flow_dir {
        both {
            $res.tb insert end "Information flows both into and out of " title \
                $parent_name title_type \
                " from/to " title \
                $name title_type
        }
        in {
            $res.tb insert end "Information flows into " title \
                $parent_name title_type \
                " from " title \
                $name title_type
        }
        out {
            $res.tb insert end "Information flows out of " title \
                $parent_name title_type \
                " to " title \
                $name title_type
        }
    }
    $res.tb insert end "\n\n" title_type \
        "Objects classes for " subtitle \
        [string toupper $flow_dir] subtitle_dir \
        " flows:\n" subtitle
    foreach c $classes {
        foreach {class_name rules} $c {break}
        $res.tb insert end "      " {} \
            $class_name\n subtitle
        Apol_Widget::appendSearchResultAVRules $res 12 $rules
    }
}
namespace eval Apol_Analysis_transflow {
    variable vals
    variable widgets
    Apol_Analysis::registerAnalysis "Apol_Analysis_transflow" "Transitive Information Flow"
}
proc Apol_Analysis_transflow::open {} {
    variable vals
    variable widgets
    Apol_Widget::resetTypeComboboxToPolicy $widgets(type)
    set vals(intermed:inc) $Apol_Types::typelist
    set vals(intermed:inc_all) $Apol_Types::typelist
    set vals(classes:displayed) {}
    foreach class $Apol_Class_Perms::class_list {
        foreach perm [apol_GetAllPermsForClass $class] {
            set vals(perms:$class:$perm) 1
        }
        lappend vals(classes:displayed) $class
    }
}
proc Apol_Analysis_transflow::close {} {
    variable widgets
    reinitializeVals
    reinitializeWidgets
    Apol_Widget::clearTypeCombobox $widgets(type)
}
proc Apol_Analysis_transflow::getInfo {} {
    return "This analysis generates the results of a Transitive Information Flow
analysis beginning from the starting type selected.  The results of
the analysis are presented in tree form with the root of the tree
being the start point for the analysis.
\nEach child node in the tree represents a type in the current policy
for which there is a transitive information flow to or from its parent
node.  If flow 'To' is selected the information flows from the child
to the parent.  If flow 'From' is selected then information flows from
the parent to the child.
\nThe results of the analysis may be optionally filtered by object
classes and/or permissions, intermediate types, or an end type regular
expression.
\nNOTE: For any given generation, if the parent and the child are the
same, the child cannot be opened.  This avoids cyclic analyses.
\nFor additional help on this topic select \"Information Flow Analysis\"
from the help menu."
}
proc Apol_Analysis_transflow::create {options_frame} {
    variable vals
    variable widgets
    reinitializeVals
    set dir_tf [TitleFrame $options_frame.dir -text "Direction"]
    pack $dir_tf -side left -padx 2 -pady 2 -expand 0 -fill y
    set dir_to [radiobutton [$dir_tf getframe].to -text "To" \
                    -variable Apol_Analysis_transflow::vals(dir) -value to]
    set dir_from [radiobutton [$dir_tf getframe].from -text "From" \
                      -variable Apol_Analysis_transflow::vals(dir) -value from]
    pack $dir_to $dir_from -anchor w
    set req_tf [TitleFrame $options_frame.req -text "Required Parameters"]
    pack $req_tf -side left -padx 2 -pady 2 -expand 0 -fill y
    set l [label [$req_tf getframe].l -text "Starting type"]
    pack $l -anchor w
    set widgets(type) [Apol_Widget::makeTypeCombobox [$req_tf getframe].type]
    pack $widgets(type)
    set filter_tf [TitleFrame $options_frame.filter -text "Optional Result Filters"]
    pack $filter_tf -side left -padx 2 -pady 2 -expand 1 -fill both
    set advanced_f [frame [$filter_tf getframe].advanced]
    pack $advanced_f -side left -anchor nw
    set widgets(advanced_enable) [checkbutton $advanced_f.enable -text "Use advanced filters" \
                                      -variable Apol_Analysis_transflow::vals(advanced:enable)]
    pack $widgets(advanced_enable) -anchor w
    set widgets(advanced) [button $advanced_f.b -text "Advanced Filters" \
                               -command Apol_Analysis_transflow::createAdvancedDialog \
                               -state disabled]
    pack $widgets(advanced) -anchor w -padx 4
    trace add variable Apol_Analysis_transflow::vals(advanced:enable) write \
        Apol_Analysis_transflow::toggleAdvancedSelected
    set widgets(regexp) [Apol_Widget::makeRegexpEntry [$filter_tf getframe].end]
    $widgets(regexp).cb configure -text "Filter result types using regular expression"
    pack $widgets(regexp) -side left -anchor nw -padx 8
}
proc Apol_Analysis_transflow::newAnalysis {} {
    if {[set rt [checkParams]] != {}} {
        return $rt
    }
    if {[catch {analyze} results]} {
        return $results
    }
    set f [createResultsDisplay]
    if {[catch {renderResults $f $results} rt]} {
        Apol_Analysis::deleteCurrentResults
        return $rt
    }
    return {}
}
proc Apol_Analysis_transflow::updateAnalysis {f} {
    if {[set rt [checkParams]] != {}} {
        return $rt
    }
    if {[catch {analyze} results]} {
        return $results
    }
    clearResultsDisplay $f
    if {[catch {renderResults $f $results} rt]} {
        return $rt
    }
    return {}
}
proc Apol_Analysis_transflow::reset {} {
    reinitializeVals
    reinitializeWidgets
    open
}
proc Apol_Analysis_transflow::switchTab {query_options} {
    variable vals
    variable widgets
    array set vals $query_options
    reinitializeWidgets
}
proc Apol_Analysis_transflow::saveQuery {channel} {
    variable vals
    variable widgets
    foreach {key value} [array get vals] {
        switch -glob -- $key {
            find_more:* -
            intermed:inc* -
            intermed:exc -
            classes:title {}
            classes:displayed {}
            perms:* {
                if {$value == 0} {
                    puts $channel "$key $value"
                }
            }
            default {
                puts $channel "$key $value"
            }
        }
    }
    set type [Apol_Widget::getTypeComboboxValueAndAttrib $widgets(type)]
    puts $channel "type [lindex $type 0]"
    puts $channel "type:attrib [lindex $type 1]"
    set use_regexp [Apol_Widget::getRegexpEntryState $widgets(regexp)]
    set regexp [Apol_Widget::getRegexpEntryValue $widgets(regexp)]
    puts $channel "regexp:enable $use_regexp"
    puts $channel "regexp $regexp"
}
proc Apol_Analysis_transflow::loadQuery {channel} {
    variable vals
    set intermed_exc {}
    set perms_disabled {}
    while {[gets $channel line] >= 0} {
        set line [string trim $line]
        if {$line == {} || [string index $line 0] == "#"} {
            continue
        }
        set key {}
        set value {}
        regexp -line -- {^(\S+)( (.+))?} $line -> key --> value
        switch -glob -- $key {
            intermed:exc_all {
                set intermed_exc $value
            }
            perms:* {
                set perms_disabled [concat $perms_disabled $key $value]
            }
            default {
                set vals($key) $value
            }
        }
    }
    open
    set vals(intermed:exc_all) {}
    set vals(intermed:exc) {}
    foreach t $intermed_exc {
        set i [lsearch $vals(intermed:inc_all) $t]
        if {$i >= 0} {
            lappend vals(intermed:exc_all) $t
            lappend vals(intermed:exc) $t
            set vals(intermed:inc_all) [lreplace $vals(intermed:inc_all) $i $i]
            set i [lsearch $vals(intermed:inc) $t]
            set vals(intermed:inc) [lreplace $vals(intermed:inc) $i $i]
        }
    }
    set vals(intermed:exc_all) [lsort $vals(intermed:exc_all)]
    set vals(intermed:exc) [lsort $vals(intermed:exc)]
    foreach {key value} $perms_disabled {
        if {[info exists vals($key)]} {
            set vals($key) $value
        }
    }
    set vals(classes:displayed) {}
    foreach class $Apol_Class_Perms::class_list {
        set all_disabled 1
        foreach perm_key [array names vals perms:$class:*] {
            if {$vals($perm_key)} {
                set all_disabled 0
                break
            }
        }
        if {$all_disabled} {
            lappend vals(classes:displayed) "$class (excluded)"
        } else {
            lappend vals(classes:displayed) $class
        }
    }
    reinitializeWidgets
}
proc Apol_Analysis_transflow::gotoLine {tab line_num} {
    set searchResults [$tab.right getframe].res
    Apol_Widget::gotoLineSearchResults $searchResults $line_num
}
proc Apol_Analysis_transflow::search {tab str case_Insensitive regExpr srch_Direction } {
    set textbox [$tab.right getframe].res.tb
    ApolTop::textSearch $textbox $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_Analysis_transflow::reinitializeVals {} {
    variable vals
    array set vals {
        dir to
        type {}  type:attrib {}
        regexp:enable 0
        regexp {}
        advanced:enable 0
        classes:title {}
        classes:displayed {}
        classes:threshold_enable 0
        classes:threshold 1
        intermed:inc {}   intermed:inc_all {}
        intermed:exc {}   intermed:exc_all {}
        intermed:attribenable 0  intermed:attrib {}
        find_more:hours 0   find_more:minutes 0   find_more:seconds 30
        find_more:limit 20
    }
    array unset vals perms:*
    foreach class $Apol_Class_Perms::class_list {
        foreach perm [apol_GetAllPermsForClass $class] {
            set vals(perms:$class:$perm) 1
        }
    }
}
proc Apol_Analysis_transflow::reinitializeWidgets {} {
    variable vals
    variable widgets
    if {$vals(type:attrib) != {}} {
        Apol_Widget::setTypeComboboxValue $widgets(type) [list $vals(type) $vals(type:attrib)]
    } else {
        Apol_Widget::setTypeComboboxValue $widgets(type) $vals(type)
    }
    Apol_Widget::setRegexpEntryValue $widgets(regexp) $vals(regexp:enable) $vals(regexp)
}
proc Apol_Analysis_transflow::toggleAdvancedSelected {name1 name2 op} {
    variable vals
    variable widgets
    if {$vals(advanced:enable)} {
        $widgets(advanced) configure -state normal
    } else {
        $widgets(advanced) configure -state disabled
    }
}
proc Apol_Analysis_transflow::createAdvancedDialog {} {
    destroy .transflow_adv
    variable vals
    if {[ApolTop::is_policy_open] && ![Apol_Perms_Map::is_pmap_loaded]} {
        if {![Apol_Perms_Map::loadDefaultPermMap]} {
            return "This analysis requires that a permission map is loaded."
	}
    }
    set d [Dialog .transflow_adv -modal local -separator 1 -title "Transitive Information Flow Advanced Filters" -parent .]
    $d add -text "Close"
    set tf [TitleFrame [$d getframe].classes -text "Filter By Object Class Permissions"]
    pack $tf -side top -expand 1 -fill both -padx 2 -pady 4
    createClassFilter [$tf getframe]
    set tf [TitleFrame [$d getframe].types -text "Filter By Intermediate Types"]
    pack $tf -side top -expand 1 -fill both -padx 2 -pady 4
    createIntermedFilter [$tf getframe]
    set inc [$tf getframe].inc
    set exc [$tf getframe].exc
    set attrib [frame [$tf getframe].a]
    grid $attrib - -
    set attrib_enable [checkbutton $attrib.ae -anchor w \
                           -text "Filter by attribute" \
                           -variable Apol_Analysis_transflow::vals(intermed:attribenable)]
    set attrib_box [ComboBox $attrib.ab -autopost 1 -entrybg white -width 16 \
                        -values $Apol_Types::attriblist \
                        -textvariable Apol_Analysis_transflow::vals(intermed:attrib)]
    $attrib_enable configure -command \
        [list Apol_Analysis_transflow::attribEnabled $attrib_box]
    trace remove variable Apol_Analysis_transflow::vals(intermed:attrib) write \
        [list Apol_Analysis_transflow::attribChanged]
    trace add variable Apol_Analysis_transflow::vals(intermed:attrib) write \
        [list Apol_Analysis_transflow::attribChanged]
    pack $attrib_enable -side top -expand 0 -fill x -anchor sw -padx 5 -pady 2
    pack $attrib_box -side top -expand 1 -fill x -padx 10
    attribEnabled $attrib_box
    $d draw
}
proc Apol_Analysis_transflow::createClassFilter {f} {
    variable vals
    set l1 [label $f.l1 -text "Object Classes"]
    set l [label $f.l]
    set vals(classes:title) "Permissions"
    set l2 [label $f.l2 -textvariable Apol_Analysis_transflow::vals(classes:title)]
    grid $l1 $l $l2 -sticky w
    set classes [Apol_Widget::makeScrolledListbox $f.c -selectmode extended \
                     -height 16 -width 30 -listvar Apol_Analysis_transflow::vals(classes:displayed)]
    set sw [ScrolledWindow $f.sw -auto both]
    set perms [ScrollableFrame $sw.perms -bg white -height 200 -width 300]
    $sw setwidget $perms
    bind $classes.lb <<ListboxSelect>> \
        [list Apol_Analysis_transflow::refreshPerm $classes $perms]
    grid $classes x $sw -sticky nsew
    update
    grid propagate $sw 0
    set bb [ButtonBox $f.bb -homogeneous 1 -spacing 4]
    $bb add -text "Include All Perms" -width 16 -command [list Apol_Analysis_transflow::setAllPerms $classes $perms 1]
    $bb add -text "Exclude All Perms" -width 16 -command [list Apol_Analysis_transflow::setAllPerms $classes $perms 0]
    grid ^ x $bb -pady 4
    set f [frame $f.f]
    grid ^ x $f
    grid configure $f -sticky ew
    set cb [checkbutton $f.cb -text "Exclude permissions that have weights below this threshold:" \
                -variable Apol_Analysis_transflow::vals(classes:threshold_enable)]
    set weight [spinbox $f.threshold -from 1 -to 10 -increment 1 \
                    -width 2 -bg white -justify right \
                    -textvariable Apol_Analysis_transflow::vals(classes:threshold)]
    trace remove variable Apol_Analysis_transflow::vals(classes:threshold_enable) write \
        [list Apol_Analysis_transflow::thresholdChanged $weight]
    trace add variable Apol_Analysis_transflow::vals(classes:threshold_enable) write \
        [list Apol_Analysis_transflow::thresholdChanged $weight]
    pack $cb $weight -side left
    thresholdChanged $weight {} {} {}
    grid columnconfigure $f 0 -weight 0
    grid columnconfigure $f 1 -weight 0 -pad 4
    grid columnconfigure $f 2 -weight 1
}
proc Apol_Analysis_transflow::refreshPerm {classes perms} {
    variable vals
    focus $classes.lb
    if {[$classes.lb curselection] == {}} {
        return
    }
    set pf [$perms getframe]
    foreach w [winfo children $pf] {
        destroy $w
    }
    foreach {class foo} [$classes.lb get anchor] {break}
    set i [$classes.lb index anchor]
    set vals(classes:title) "Permissions for $class"
    if {[catch {apol_GetPermMap $class} perm_map_list]} {
        tk_messageBox -icon error -type ok \
            -title "Error Getting Permission Map" -message $perm_map_list
        return
    }
    foreach perm_key [lsort [array names vals perms:$class:*]] {
        foreach {foo bar perm} [split $perm_key :] {break}
        set j [lsearch -glob [lindex $perm_map_list 0 1] "$perm *"]
        set weight [lindex $perm_map_list 0 1 $j 2]
        set l [label $pf.$perm:l -text $perm -bg white -anchor w]
        set inc [radiobutton $pf.$perm:i -text "Include" -value 1 -bg white \
                     -highlightthickness 0 \
                     -command [list Apol_Analysis_transflow::togglePerm $class $i] \
                     -variable Apol_Analysis_transflow::vals(perms:$class:$perm)]
        set exc [radiobutton $pf.$perm:e -text "Exclude" -value 0 -bg white \
                     -highlightthickness 0 \
                     -command [list Apol_Analysis_transflow::togglePerm $class $i] \
                     -variable Apol_Analysis_transflow::vals(perms:$class:$perm)]
        set w [label $pf.$perm:w -text "Weight: $weight" -bg white]
        grid $l $inc $exc $w -padx 2 -sticky w -pady 4
        grid configure $w -ipadx 10
    }
    grid columnconfigure $pf 0 -minsize 100 -weight 1
    foreach i {1 2} {
        grid columnconfigure $pf $i -uniform 1 -weight 0
    }
    $perms xview moveto 0
    $perms yview moveto 0
}
proc Apol_Analysis_transflow::togglePerm {class i} {
    variable vals
    set all_disabled 1
    foreach perm_key [array names vals perms:$class:*] {
        if {$vals($perm_key)} {
            set all_disabled 0
            break
        }
    }
    if {$all_disabled} {
        set vals(classes:displayed) [lreplace $vals(classes:displayed) $i $i "$class (excluded)"]
    } else {
        set vals(classes:displayed) [lreplace $vals(classes:displayed) $i $i $class]
    }
}
proc Apol_Analysis_transflow::setAllPerms {classes perms newValue} {
    variable vals
    foreach i [$classes.lb curselection] {
        foreach {class foo} [split [$classes.lb get $i]] {break}
        foreach perm_key [array names vals perms:$class:*] {
            set vals($perm_key) $newValue
        }
        if {$newValue == 1} {
            set vals(classes:displayed) [lreplace $vals(classes:displayed) $i $i $class]
        } else {
            set vals(classes:displayed) [lreplace $vals(classes:displayed) $i $i "$class (excluded)"]
        }
    }
}
proc Apol_Analysis_transflow::thresholdChanged {w name1 name2 op} {
    variable vals
    if {$vals(classes:threshold_enable)} {
        $w configure -state normal
    } else {
        $w configure -state disabled
    }
}
proc Apol_Analysis_transflow::createIntermedFilter {f} {
    set l1 [label $f.l1 -text "Included Intermediate Types"]
    set l2 [label $f.l2 -text "Excluded Intermediate Types"]
    grid $l1 x $l2 -sticky w
    set inc [Apol_Widget::makeScrolledListbox $f.inc -height 10 -width 24 \
                 -listvar Apol_Analysis_transflow::vals(intermed:inc) \
                 -selectmode extended -exportselection 0]
    set exc [Apol_Widget::makeScrolledListbox $f.exc -height 10 -width 24 \
                 -listvar Apol_Analysis_transflow::vals(intermed:exc) \
                 -selectmode extended -exportselection 0]
    set inc_lb [Apol_Widget::getScrolledListbox $inc]
    set exc_lb [Apol_Widget::getScrolledListbox $exc]
    set bb [ButtonBox $f.bb -homogeneous 1 -orient vertical -spacing 4]
    $bb add -text "-->" -width 10 -command [list Apol_Analysis_transflow::moveToExclude $inc_lb $exc_lb]
    $bb add -text "<--" -width 10 -command [list Apol_Analysis_transflow::moveToInclude $inc_lb $exc_lb]
    grid $inc $bb $exc -sticky nsew
    set inc_bb [ButtonBox $f.inc_bb -homogeneous 1 -spacing 4]
    $inc_bb add -text "Select All" -command [list $inc_lb selection set 0 end]
    $inc_bb add -text "Unselect" -command [list $inc_lb selection clear 0 end]
    set exc_bb [ButtonBox $f.exc_bb -homogeneous 1 -spacing 4]
    $exc_bb add -text "Select All" -command [list $exc_lb selection set 0 end]
    $exc_bb add -text "Unselect" -command [list $exc_lb selection clear 0 end]
    grid $inc_bb x $exc_bb -pady 4
    grid columnconfigure $f 0 -weight 1 -uniform 0 -pad 2
    grid columnconfigure $f 1 -weight 0 -pad 8
    grid columnconfigure $f 2 -weight 1 -uniform 0 -pad 2
}
proc Apol_Analysis_transflow::moveToExclude {inc exc} {
    variable vals
    if {[set selection [$inc curselection]] == {}} {
        return
    }
    foreach i $selection {
        lappend types [$inc get $i]
    }
    set vals(intermed:exc) [lsort [concat $vals(intermed:exc) $types]]
    set vals(intermed:exc_all) [lsort [concat $vals(intermed:exc_all) $types]]
    foreach t $types {
        set i [lsearch $vals(intermed:inc) $t]
        set vals(intermed:inc) [lreplace $vals(intermed:inc) $i $i]
        set i [lsearch $vals(intermed:inc_all) $t]
        set vals(intermed:inc_all) [lreplace $vals(intermed:inc_all) $i $i]
    }
    $inc selection clear 0 end
    $exc selection clear 0 end
}
proc Apol_Analysis_transflow::moveToInclude {inc exc} {
    variable vals
    if {[set selection [$exc curselection]] == {}} {
        return
    }
    foreach i $selection {
        lappend types [$exc get $i]
    }
    set vals(intermed:inc) [lsort [concat $vals(intermed:inc) $types]]
    set vals(intermed:inc_all) [lsort [concat $vals(intermed:inc_all) $types]]
    foreach t $types {
        set i [lsearch $vals(intermed:exc) $t]
        set vals(intermed:exc) [lreplace $vals(intermed:exc) $i $i]
        set i [lsearch $vals(intermed:exc_all) $t]
        set vals(intermed:exc_all) [lreplace $vals(intermed:exc_all) $i $i]
    }
    $inc selection clear 0 end
    $exc selection clear 0 end
}
proc Apol_Analysis_transflow::attribEnabled {cb} {
    variable vals
    if {$vals(intermed:attribenable)} {
        $cb configure -state normal
        filterTypeLists $vals(intermed:attrib)
    } else {
        $cb configure -state disabled
        filterTypeLists ""
    }
}
proc Apol_Analysis_transflow::attribChanged {name1 name2 op} {
    variable vals
    if {$vals(intermed:attribenable)} {
        filterTypeLists $vals(intermed:attrib)
    }
}
proc Apol_Analysis_transflow::filterTypeLists {attrib} {
    variable vals
    if {$attrib != ""} {
        set typesList [lindex [apol_GetAttribs $attrib] 0 1]
        set vals(intermed:inc) {}
        set vals(intermed:exc) {}
        foreach t $typesList {
            if {[lsearch $vals(intermed:inc_all) $t] >= 0} {
                lappend vals(intermed:inc) $t
            }
            if {[lsearch $vals(intermed:exc_all) $t] >= 0} {
                lappend vals(intermed:exc) $t
            }
        }
        set vals(intermed:inc) [lsort $vals(intermed:inc)]
        set vals(intermed:exc) [lsort $vals(intermed:exc)]
    } else {
        set vals(intermed:inc) $vals(intermed:inc_all)
        set vals(intermed:exc) $vals(intermed:exc_all)
    }
}
proc Apol_Analysis_transflow::checkParams {} {
    variable vals
    variable widgets
    if {![ApolTop::is_policy_open]} {
        return "No current policy file is opened!"
    }
    set type [Apol_Widget::getTypeComboboxValueAndAttrib $widgets(type)]
    if {[lindex $type 0] == {}} {
        return "No type was selected."
    }
    set vals(type) [lindex $type 0]
    set vals(type:attrib) [lindex $type 1]
    set use_regexp [Apol_Widget::getRegexpEntryState $widgets(regexp)]
    set regexp [Apol_Widget::getRegexpEntryValue $widgets(regexp)]
    if {$use_regexp && $regexp == {}} {
            return "No regular expression provided."
    }
    set vals(regexp:enable) $use_regexp
    set vals(regexp) $regexp
    if {![Apol_Perms_Map::is_pmap_loaded]} {
        if {![Apol_Perms_Map::loadDefaultPermMap]} {
            return "This analysis requires that a permission map is loaded."
	}
    }
    if {$vals(advanced:enable)} {
        if {$vals(intermed:inc_all) == {}} {
            return "At least one intermediate type must be selected."
        }
        if {[lsearch $vals(intermed:exc_all) $vals(type)] >= 0} {
            return "The starting type is on the excluded intermediate types list"
        }
        set num_perms 0
        foreach perm_key [array names vals perms:*] {
            if {$vals($perm_key)} {
                set num_perms 1
                break
            }
        }
        if {$num_perms == 0} {
            return "At least one permissions must be enabled."
        }
    }
    return {}  ;# all parameters passed, now ready to do search
}
proc Apol_Analysis_transflow::analyze {} {
    variable vals
    if {$vals(regexp:enable)} {
        set regexp $vals(regexp)
    } else {
        set regexp {}
    }
    set threshold {}
    if {$vals(advanced:enable)} {
        set intermed $vals(intermed:inc_all)
        set classperms {}
        foreach perm_key [array names vals perms:*] {
            if {$vals($perm_key)} {
                foreach {foo class perm} [split $perm_key :] {break}
                lappend classperms [list $class $perm]
            }
        }
        if {$vals(classes:threshold_enable)} {
            set threshold $vals(classes:threshold)
        }
    } else {
        set intermed {}
        set classperms {}
    }
    apol_TransInformationFlowAnalysis $vals(dir) $vals(type) $intermed $classperms $threshold $regexp
}
proc Apol_Analysis_transflow::analyzeMore {tree node} {
    set new_start [$tree itemcget $node -text]
    if {[$tree itemcget [$tree parent $node] -text] == $new_start} {
        return {}
    }
    set g [lindex [$tree itemcget top -data] 0]
    apol_TransInformationFlowMore $g $new_start
}
proc Apol_Analysis_transflow::createResultsDisplay {} {
    variable vals
    set f [Apol_Analysis::createResultTab "Trans Flow" [array get vals]]
    set tree_tf [TitleFrame $f.left -text "Transitive Information Flow Tree"]
    pack $tree_tf -side left -expand 0 -fill y -padx 2 -pady 2
    set sw [ScrolledWindow [$tree_tf getframe].sw -auto both]
    set tree [Tree [$sw getframe].tree -width 24 -redraw 1 -borderwidth 0 \
                  -highlightthickness 0 -showlines 1 -padx 0 -bg white]
    $sw setwidget $tree
    pack $sw -expand 1 -fill both
    set res_tf [TitleFrame $f.right -text "Transitive Information Flow Results"]
    pack $res_tf -side left -expand 1 -fill both -padx 2 -pady 2
    set res [Apol_Widget::makeSearchResults [$res_tf getframe].res]
    $res.tb tag configure title -font {Helvetica 14 bold}
    $res.tb tag configure title_type -foreground blue -font {Helvetica 14 bold}
    $res.tb tag configure find_more -underline 1
    $res.tb tag configure subtitle -font {Helvetica 10 bold}
    $res.tb tag configure num -foreground blue -font {Helvetica 10 bold}
    $res.tb tag bind find_more <Button-1> [list Apol_Analysis_transflow::findMore $res $tree]
    $res.tb tag bind find_more <Enter> [list $res.tb configure -cursor hand2]
    $res.tb tag bind find_more <Leave> [list $res.tb configure -cursor {}]
    pack $res -expand 1 -fill both
    $tree configure -selectcommand [list Apol_Analysis_transflow::treeSelect $res]
    $tree configure -opencmd [list Apol_Analysis_transflow::treeOpen $tree]
    bind $tree <Destroy> [list Apol_Analysis_transflow::treeDestroy $tree]
    return $f
}
proc Apol_Analysis_transflow::treeSelect {res tree node} {
    if {$node != {}} {
        $res.tb configure -state normal
        $res.tb delete 0.0 end
        set data [$tree itemcget $node -data]
        if {[string index $node 0] == "y"} {
            renderResultsTransFlow $res $tree $node [lindex $data 1]
        } else {
            eval $res.tb insert end [lindex $data 1]
        }
        $res.tb configure -state disabled
    }
}
proc Apol_Analysis_transflow::treeOpen {tree node} {
    foreach {is_expanded results} [$tree itemcget $node -data] {break}
    if {[string index $node 0] == "y" && !$is_expanded} {
        ApolTop::setBusyCursor
        update idletasks
        set retval [catch {analyzeMore $tree $node} new_results]
        ApolTop::resetBusyCursor
        if {$retval} {
            tk_messageBox -icon error -type ok -title "Transitive Information Flow" -message "Could not perform additional analysis:\n\n$new_results"
        } else {
            $tree itemconfigure $node -data [list 1 $results]
            createResultsNodes $tree $node $new_results 1
        }
    }
}
proc Apol_Analysis_transflow::treeDestroy {tree} {
    set graph_handler [lindex [$tree itemcget top -data] 0]
    apol_InformationFlowDestroy $graph_handler
}
proc Apol_Analysis_transflow::clearResultsDisplay {f} {
    variable vals
    set tree [[$f.left getframe].sw getframe].tree
    set res [$f.right getframe].res
    set graph_handler [lindex [$tree itemcget top -data] 0]
    apol_InformationFlowDestroy $graph_handler
    $tree delete [$tree nodes root]
    Apol_Widget::clearSearchResults $res
    Apol_Analysis::setResultTabCriteria [array get vals]
}
proc Apol_Analysis_transflow::renderResults {f results} {
    variable vals
    set graph_handler [lindex $results 0]
    set results_list [lrange $results 1 end]
    set tree [[$f.left getframe].sw getframe].tree
    set res [$f.right getframe].res
    $tree insert end root top -text $vals(type) -open 1 -drawcross auto
    set top_text [renderTopText]
    $tree itemconfigure top -data [list $graph_handler $top_text]
    createResultsNodes $tree top $results_list 1
    $tree selection set top
    $tree opentree top 0
    update idletasks
    $tree see top
}
proc Apol_Analysis_transflow::renderTopText {} {
    variable vals
    set top_text [list "Transitive Information Flow Analysis: Starting type: " title]
    lappend top_text $vals(type) title_type \
        "\n\n" title \
        "This tab provides the results of a Transitive Information Flow
analysis beginning from the starting type selected above.  The results
of the analysis are presented in tree form with the root of the tree
(this node) being the start point for the analysis.
\nEach child node in the tree represents a type in the current policy
for which there is a transitive information flow to or from (depending
on your selection above) its parent node.
\nNOTE: For any given generation, if the parent and the child are the
same, you cannot open the child.  This avoids cyclic analyses." {}
}
proc Apol_Analysis_transflow::createResultsNodes {tree parent_node results do_expand} {
    set all_targets {}
    foreach r $results {
        foreach {flow_dir source target length steps} $r {break}
        if {!$do_expand} {
            set target [lindex $results 0 2]
        }
        lappend all_targets $target
        lappend paths($target) [list $length $steps]
    }
    foreach t [lsort -uniq $all_targets] {
        set sorted_paths {}
        foreach path [lsort -uniq [lsort -index 0 -integer $paths($t)]] {
            if {$flow_dir == "to"} {
                set p {}
                foreach step [lindex $path 1] {
                    set p [concat [list $step] $p]
                }
                lappend sorted_paths $p
            } else {
                lappend sorted_paths [lindex $path 1]
            }
        }
        set data [list $flow_dir $sorted_paths]
        $tree insert end $parent_node y\#auto -text $t -drawcross allways \
            -data [list 0 $data]
    }
}
proc Apol_Analysis_transflow::renderResultsTransFlow {res tree node data} {
    set parent_name [$tree itemcget [$tree parent $node] -text]
    set name [$tree itemcget $node -text]
    foreach {flow_dir paths} $data {break}
    switch -- $flow_dir {
        to {
            $res.tb insert end "Information flows to " title \
                $parent_name title_type \
                " from " title \
                $name title_type
        }
        from {
            $res.tb insert end "Information flows from " title \
                $parent_name title_type \
                " to " title \
                $name title_type
        }
    }
    $res.tb insert end "  (" title \
        "Find more flows" {title_type find_more} \
        ")\n\n" title \
        "Apol found the following number of information flows: " subtitle \
        [llength $paths] num \
        "\n" subtitle
    set path_num 1
    foreach path $paths {
        $res.tb insert end "\n" {}
        renderPath $res $path_num $path
        incr path_num
    }
}
proc Apol_Analysis_transflow::renderPath {res path_num path} {
    $res.tb insert end "Flow " subtitle \
        $path_num num \
        " requires " subtitle \
        [llength $path] num \
        " steps(s).\n" subtitle \
        "    " {}
    $res.tb insert end [lindex $path 0 0] subtitle \
        " -> " {} \
        [lindex $path 0 1] subtitle
    foreach step [lrange $path 1 end] {
        $res.tb insert end " -> " {} \
            [lindex $step 1] subtitle
    }
    $res.tb insert end \n {}
    foreach steps $path {
        set rules [lindex $steps 3]
        Apol_Widget::appendSearchResultAVRules $res 6 [lindex $rules 0]
        Apol_Widget::appendSearchResultAVRules $res 10 [lrange $rules 1 end]
    }
}
proc Apol_Analysis_transflow::findMore {res tree} {
    set node [$tree selection get]
    set start [$tree itemcget [$tree parent $node] -text]
    set end [$tree itemcget $node -text]
    set d [Dialog .trans_more -cancel 1 -default 0 -modal local -parent . \
               -separator 1 -title "Find More Flows"]
    $d add -text Find -command [list Apol_Analysis_transflow::verifyFindMore $d]
    $d add -text Cancel
    set f [$d getframe]
    set l1 [label $f.l1 -text "Source: $start"]
    set l2 [label $f.l2 -text "Target: $end"]
    set time_f [frame $f.time]
    set path_f [frame $f.path]
    pack $l1 $l2 $time_f $path_f -anchor w -padx 8 -pady 4
    set t1 [label $time_f.t1 -text "Time limit: "]
    set e1 [entry $time_f.e1 -textvariable Apol_Analysis_transflow::vals(find_more:hours) -width 5 -justify right -bg white]
    set t2 [label $time_f.t2 -text "Hour(s)  "]
    set e2 [entry $time_f.e2 -textvariable Apol_Analysis_transflow::vals(find_more:minutes) -width 5 -justify right -bg white]
    set t3 [label $time_f.t3 -text "Minute(s)  "]
    set e3 [entry $time_f.e3 -textvariable Apol_Analysis_transflow::vals(find_more:seconds) -width 5 -justify right -bg white]
    set t4 [label $time_f.t4 -text "Second(s)  "]
    pack $t1 $e1 $t2 $e2 $t3 $e3 $t4 -side left
    set t1 [label $path_f.t1 -text "Limit by these number of flows: "]
    set e1 [entry $path_f.e1 -textvariable Apol_Analysis_transflow::vals(find_more:limit) -width 5 -justify right -bg white]
    pack $t1 $e1 -side left
    set retval [$d draw]
    destroy .trans_more
    if {$retval == 0} {
        set graph_handler [lindex [$tree itemcget top -data] 0]
        if {[catch {apol_TransInformationFurtherPrepare $graph_handler $start $end} err]} {
            tk_messageBox -icon error -type ok -title "Find More Flows" -message "Could not prepare infoflow graph:\n$err"
        } else {
            doFindMore $res $tree $node
        }
    }
}
proc Apol_Analysis_transflow::verifyFindMore {d} {
    variable vals
    set message {}
    if {[set hours [string trim $vals(find_more:hours)]] == {}} {
        set hours 0
    }
    if {[set minutes [string trim $vals(find_more:minutes)]] == {}} {
        set minutes 0
    }
    if {[set seconds [string trim $vals(find_more:seconds)]] == {}} {
        set seconds 0
    }
    set path_limit [string trim $vals(find_more:limit)]
    if {![string is integer $hours] || $hours > 24 || $hours < 0} {
        set message "Invalid hours limit input.  Must be between 0-24 inclusive."
    } elseif {![string is integer $minutes] || $minutes > 59 || $minutes < 0} {
        set message "Invalid minutes limit input.  Must be between 0-59 inclusive."
    } elseif {![string is integer $seconds] || $seconds > 59 || $seconds < 0} {
        set message "Invalid seconds limit input.  Must be between 0-59 inclusive."
    } elseif {$path_limit == {} && $hours == 0 && $minutes == 0 && $seconds == 0} {
        set message "You must specify a time limit."
    } elseif {$path_limit != {} && (![string is integer $path_limit] || $path_limit < 0)} {
        set message "Number of flows cannot be less than 1."
    }
    if {$message != {}} {
        tk_messageBox -icon error -type ok -title "Find More Flows" -message $message
    } else {
        $d enddialog 0
    }
}
proc Apol_Analysis_transflow::doFindMore {res tree node} {
    variable vals
    if {[set hours [string trim $vals(find_more:hours)]] == {}} {
        set hours 0
    }
    if {[set minutes [string trim $vals(find_more:minutes)]] == {}} {
        set minutes 0
    }
    if {[set seconds [string trim $vals(find_more:seconds)]] == {}} {
        set seconds 0
    }
    set path_limit [string trim $vals(find_more:limit)]
    if {$hours != 0 || $minutes != 0 || $seconds != 0} {
        set time_limit [expr {$hours * 3600 + $minutes * 60 + $seconds}]
        set time_limit_str [format " elapsed out of %02d:%02d:%02d" $hours $minutes $seconds]
    } else {
        set time_limit {}
        set time_limit_str {}
    }
    if {$path_limit != {}} {
        set path_limit_str " out of $path_limit"
    } else {
        set path_limit 0
        set path_limit_str {}
    }
    set vals(find_more:abort) 0
    set vals(find_more:searches_text) {}
    set vals(find_more:searches_done) -1
    set d [ProgressDlg .trans_domore -parent . -title "Find Results" \
               -width 40 -height 5 \
               -textvariable Apol_Analysis_transflow::vals(find_more:searches_text) \
               -variable Apol_Analysis_transflow::vals(find_more:searches_done) \
               -stop Stop \
               -command [list set Apol_Analysis_transflow::vals(find_more:abort) 1]]
    set graph_handler [lindex [$tree itemcget top -data] 0]
    set start_time [clock seconds]
    set elapsed_time 0
    set results {}
    set path_found 0
    while {1} {
        set elapsed_time [expr {[clock seconds] - $start_time}]
        set vals(find_more:searches_text) "Finding more flows:\n\n"
        append vals(find_more:searches_text) "    Time: [clock format $elapsed_time -format "%H:%M:%S" -gmt 1]$time_limit_str\n\n"
        append vals(find_more:searches_text) "    Flows: found $path_found$path_limit_str"
        update
        if {[catch {apol_TransInformationFurtherNext $graph_handler} r]} {
            tk_messageBox -icon error -type ok -title "Find More Flows" -message "Could not find more flows:\n$results"
            break
        }
        set results [lsort -unique [concat $results $r]]
        set path_found [llength $results]
        if {($time_limit != {} && $elapsed_time >= $time_limit) || \
                ($path_limit != 0 && $path_found > $path_limit) || \
                $vals(find_more:abort)} {
            break
        }
    }
    set vals(find_more:searches_text) "Rendering $path_found flow(s)."
    update idletasks
    $res.tb configure -state normal
    $res.tb delete 0.0 end
    set parent_name [$tree itemcget [$tree parent $node] -text]
    set name [$tree itemcget $node -text]
    set flow_dir [lindex [$tree itemcget $node -data] 1 0]
    switch -- $flow_dir {
        to {
            $res.tb insert end "More information flows to " title \
                $parent_name title_type \
                " from " title \
                $name title_type
        }
        from {
            $res.tb insert end "More information flows from " title \
                $parent_name title_type \
                " to " title \
                $name title_type
        }
    }
    $res.tb insert end "  (" title \
        "Find more flows" {title_type find_more} \
        ")\n\n" title \
        "Time: " subtitle \
        [clock format $elapsed_time -format "%H:%M:%S" -gmt 1] subtitle \
        [format " out of %02d:%02d:%02d" $hours $minutes $seconds] subtitle \
        "\n\nApol found the following number of information flows: " subtitle \
        $path_found num \
        " out of " subtitle \
        $path_limit num \
        "\n" subtitle
    set path_num 1
    foreach r [lsort -index 3 -integer $results] {
        set path [lindex $r 4]
        if {$flow_dir == "to"} {
            set p {}
            foreach step $path {
                set p [concat [list $step] $p]
            }
            set sorted_path $p
        } else {
            set sorted_path $path
        }
        $res.tb insert end "\n" {}
        renderPath $res $path_num $sorted_path
        incr path_num
    }
    $res.tb configure -state disabled
    destroy $d
}
namespace eval Apol_Analysis_relabel {
    variable vals
    variable widgets
    Apol_Analysis::registerAnalysis "Apol_Analysis_relabel" "Direct Relabel"
}
proc Apol_Analysis_relabel::open {} {
    variable vals
    variable widgets
    Apol_Widget::resetTypeComboboxToPolicy $widgets(type)
    set vals(classes:inc) {}
    foreach class $Apol_Class_Perms::class_list {
        set perms [apol_GetAllPermsForClass $class]
        if {[lsearch $perms "relabelto"] >= 0 && [lsearch $perms "relabelfrom"] >= 0} {
            lappend vals(classes:inc) $class
        }
    }
    set vals(subjects:inc) $Apol_Types::typelist
    set vals(subjects:inc_all) $Apol_Types::typelist
}
proc Apol_Analysis_relabel::close {} {
    variable widgets
    reinitializeVals
    reinitializeWidgets
    Apol_Widget::clearTypeCombobox $widgets(type)
}
proc Apol_Analysis_relabel::getInfo {} {
    return "Direct relabel analysis is designed to facilitate querying a policy
for both potential changes to object labels and relabel privileges
granted to a subject. These two modes are respectively called Object
Mode and Subject Mode.
\nOBJECT MODE
In object mode the user specifies a starting or ending type and either
To, From, or Both. When To is selected all types to which the starting
type can be relabeled will be displayed. When From is selected all
types from which the ending type can be relabeled will be
displayed. Both will, obviously, do both analyses.
\nSUBJECT MODE
In subject mode the user specifies only a subject type. Two lists of
types will be displayed corresponding to all of the types To which the
subject can relabel and From which the subject can relabel.
\nOPTIONAL RESULT FILTERS
Results may be filtered in several ways. The end types resulting from
a query may be filtered by regular expression. The Advanced Filters
provide the option of selecting which object classes to include in the
analysis and which types to include as subjects of relabeling
operations. Note, excluded subjects are ignored in subject mode
because only the selected subject type is used as a subject."
}
proc Apol_Analysis_relabel::create {options_frame} {
    variable vals
    variable widgets
    reinitializeVals
    set mode_tf [TitleFrame $options_frame.mode -text "Mode"]
    pack $mode_tf -side left -padx 2 -pady 2 -expand 0 -fill y
    set object_mode [radiobutton [$mode_tf getframe].object \
                         -text "Object" -value "object" \
                         -variable Apol_Analysis_relabel::vals(mode)]
    pack $object_mode -anchor w
    set widgets(mode:to) [checkbutton [$mode_tf getframe].to \
                                -text "To" \
                                -variable Apol_Analysis_relabel::vals(mode:to)]
    $widgets(mode:to) configure -command \
        [list Apol_Analysis_relabel::toggleToFromPushed $widgets(mode:to)]
    set widgets(mode:from) [checkbutton [$mode_tf getframe].from \
                                -text "From" \
                                -variable Apol_Analysis_relabel::vals(mode:from)]
    $widgets(mode:from) configure -command \
        [list Apol_Analysis_relabel::toggleToFromPushed $widgets(mode:from)]
    pack $widgets(mode:to) $widgets(mode:from) -anchor w -padx 8
    set subject_mode [radiobutton [$mode_tf getframe].subject \
                          -text "Subject" -value "subject" \
                          -variable Apol_Analysis_relabel::vals(mode)]
    pack $subject_mode -anchor w -pady 4
    trace add variable Apol_Analysis_relabel::vals(mode) write \
        Apol_Analysis_relabel::toggleModeSelected
    set req_tf [TitleFrame $options_frame.req -text "Required Parameters"]
    pack $req_tf -side left -padx 2 -pady 2 -expand 0 -fill y
    set l [label [$req_tf getframe].l -textvariable Apol_Analysis_relabel::vals(type:label)]
    pack $l -anchor w
    set widgets(type) [Apol_Widget::makeTypeCombobox [$req_tf getframe].type]
    pack $widgets(type)
    set filter_tf [TitleFrame $options_frame.filter -text "Optional Result Filters"]
    pack $filter_tf -side left -padx 2 -pady 2 -expand 1 -fill both
    set advanced_f [frame [$filter_tf getframe].advanced]
    pack $advanced_f -side left -anchor nw
    set access_enable [checkbutton $advanced_f.enable -text "Use advanced filters" \
                           -variable Apol_Analysis_relabel::vals(advanced_enable)]
    pack $access_enable -anchor w
    set widgets(advanced) [button $advanced_f.adv -text "Advanced Filters" \
                               -command Apol_Analysis_relabel::createAdvancedDialog \
                               -state disabled]
    pack $widgets(advanced) -anchor w -padx 4
    trace add variable Apol_Analysis_relabel::vals(advanced_enable) write \
        Apol_Analysis_relabel::toggleAdvancedSelected
    set widgets(regexp) [Apol_Widget::makeRegexpEntry [$filter_tf getframe].end]
    $widgets(regexp).cb configure -text "Filter result types using regular expression"
    pack $widgets(regexp) -side left -anchor nw -padx 8
}
proc Apol_Analysis_relabel::newAnalysis {} {
    if {[set rt [checkParams]] != {}} {
        return $rt
    }
    if {[catch {analyze} results]} {
        return $results
    }
    set f [createResultsDisplay]
    if {[catch {renderResults $f $results} rt]} {
        Apol_Analysis::deleteCurrentResults
        return $rt
    }
    return {}
}
proc Apol_Analysis_relabel::updateAnalysis {f} {
    if {[set rt [checkParams]] != {}} {
        return $rt
    }
    if {[catch {analyze} results]} {
        return $results
    }
    clearResultsDisplay $f
    if {[catch {renderResults $f $results} rt]} {
        return $rt
    }
    return {}
}
proc Apol_Analysis_relabel::reset {} {
    reinitializeVals
    reinitializeWidgets
    open
}
proc Apol_Analysis_relabel::switchTab {query_options} {
    variable vals
    variable widgets
    array set vals $query_options
    reinitializeWidgets
}
proc Apol_Analysis_relabel::saveQuery {channel} {
    variable vals
    variable widgets
    foreach {key value} [array get vals] {
        if {$key != "classes:inc" && \
                $key != "subjects:inc_all" && $key != "subjects:inc" && \
                $key != "subjects:exc"} {
            puts $channel "$key $value"
        }
    }
    set type [Apol_Widget::getTypeComboboxValueAndAttrib $widgets(type)]
    puts $channel "type [lindex $type 0]"
    puts $channel "type:attrib [lindex $type 1]"
    set use_regexp [Apol_Widget::getRegexpEntryState $widgets(regexp)]
    set regexp [Apol_Widget::getRegexpEntryValue $widgets(regexp)]
    puts $channel "regexp:enable $use_regexp"
    puts $channel "regexp $regexp"
}
proc Apol_Analysis_relabel::loadQuery {channel} {
    variable vals
    set classes_exc {}
    set subjects_exc {}
    while {[gets $channel line] >= 0} {
        set line [string trim $line]
        if {$line == {} || [string index $line 0] == "#"} {
            continue
        }
        set key {}
        set value {}
        regexp -line -- {^(\S+)( (.+))?} $line -> key --> value
        switch -- $key {
            classes:exc {
                set classes_exc $value
            }
            subjects:exc_all {
                set subjects_exc $value
            }
            default {
                set vals($key) $value
            }
        }
    }
    open
    set vals(classes:exc) {}
    foreach c $classes_exc {
        set i [lsearch $vals(classes:inc) $c]
        if {$i >= 0} {
            lappend vals(classes:exc) $c
            set vals(classes:inc) [lreplace $vals(classes:inc) $i $i]
        }
    }
    set vals(classes:exc) [lsort $vals(classes:exc)]
    set vals(subjects:exc_all) {}
    set vals(subjects:exc) {}
    foreach s $subjects_exc {
        set i [lsearch $vals(subjects:inc_all) $s]
        if {$i >= 0} {
            lappend vals(subjects:exc_all) $s
            lappend vals(subjects:exc) $s
            set vals(subjects:inc_all) [lreplace $vals(subjects:inc_all) $i $i]
            set i [lsearch $vals(subjects:inc) $s]
            set vals(subjects:inc) [lreplace $vals(subjects:inc) $i $i]
        }
    }
    set vals(subjects:exc_all) [lsort $vals(subjects:exc_all)]
    set vals(subjects:exc) [lsort $vals(subjects:exc)]
    reinitializeWidgets
}
proc Apol_Analysis_relabel::gotoLine {tab line_num} {
    set searchResults [$tab.right getframe].res
    Apol_Widget::gotoLineSearchResults $searchResults $line_num
}
proc Apol_Analysis_relabel::search {tab str case_Insensitive regExpr srch_Direction } {
    set textbox [$tab.right getframe].res.tb
    ApolTop::textSearch $textbox $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_Analysis_relabel::reinitializeVals {} {
    variable vals
    array set vals {
        mode object
        mode:to 1
        mode:from 0
        type:label "Starting type"
        type {}  type:attrib {}
        regexp:enable 0
        regexp {}
        advanced_enable 0
        classes:inc {}  classes:exc {}
        subjects:inc {}  subjects:inc_all {}
        subjects:exc {}  subjects:exc_all {}
        subjects:attribenable 0 subjects:attrib {}
    }
}
proc Apol_Analysis_relabel::reinitializeWidgets {} {
    variable vals
    variable widgets
    if {$vals(type:attrib) != {}} {
        Apol_Widget::setTypeComboboxValue $widgets(type) [list $vals(type) $vals(type:attrib)]
    } else {
        Apol_Widget::setTypeComboboxValue $widgets(type) $vals(type)
    }
    Apol_Widget::setRegexpEntryValue $widgets(regexp) $vals(regexp:enable) $vals(regexp)
    updateTypeLabel
}
proc Apol_Analysis_relabel::toggleModeSelected {name1 name2 op} {
    variable vals
    variable widgets
    if {$vals(mode) == "object"} {
        $widgets(mode:to) configure -state normal
        $widgets(mode:from) configure -state normal
    } else {
        $widgets(mode:to) configure -state disabled
        $widgets(mode:from) configure -state disabled
    }
    updateTypeLabel
}
proc Apol_Analysis_relabel::toggleToFromPushed {cb} {
    variable vals
    if {!$vals(mode:to) && !$vals(mode:from)} {
        $cb select
    }
    updateTypeLabel
}
proc Apol_Analysis_relabel::updateTypeLabel {} {
    variable vals
    if {$vals(mode) == "subject"} {
        set vals(type:label) "Subject"
    } elseif {$vals(mode:to) && $vals(mode:from)} {
        set vals(type:label) "Starting/ending type"
    } elseif {$vals(mode:from)} {
        set vals(type:label) "Ending type"
    } else {
        set vals(type:label) "Starting type"
    }
}
proc Apol_Analysis_relabel::toggleAdvancedSelected {name1 name2 op} {
    variable vals
    variable widgets
    if {$vals(advanced_enable)} {
        $widgets(advanced) configure -state normal
    } else {
        $widgets(advanced) configure -state disabled
    }
}
proc Apol_Analysis_relabel::createAdvancedDialog {} {
    destroy .relabel_analysis_adv
    variable vals
    set d [Dialog .relabel_analysis_adv -modal local -separator 1 -title "Direct Relabel Advanced Filters" -parent .]
    $d add -text "Close"
    set tf [TitleFrame [$d getframe].objs -text "Filter By Object Classes"]
    pack $tf -side top -expand 1 -fill both -padx 2 -pady 4
    createAdvancedFilter [$tf getframe] "Object Classes" classes 0
    set l [label [$tf getframe].l -text "Only showing object classes that have both 'relabelto' and 'relabelfrom' permissions."]
    grid $l - - -padx 4 -pady 2
    set tf [TitleFrame [$d getframe].types -text "Filter By Subject Types"]
    pack $tf -side top -expand 1 -fill both -padx 2 -pady 4
    if {$vals(mode) == "object"} {
        createAdvancedFilter [$tf getframe] "Subject Types" subjects 0
    } else {
        createAdvancedFilter [$tf getframe] "Subject Types" subjects 1
    }
    set inc [$tf getframe].inc
    set exc [$tf getframe].exc
    set attrib [frame [$tf getframe].a]
    grid $attrib - -
    set attrib_enable [checkbutton $attrib.ae -anchor w \
                           -text "Filter by attribute" \
                           -variable Apol_Analysis_relabel::vals(subjects:attribenable)]
    set attrib_box [ComboBox $attrib.ab -autopost 1 -entrybg white -width 16 \
                        -values $Apol_Types::attriblist \
                        -textvariable Apol_Analysis_relabel::vals(subjects:attrib)]
    $attrib_enable configure -command \
        [list Apol_Analysis_relabel::attribEnabled $attrib_box]
    trace remove variable Apol_Analysis_relabel::vals(subjects:attrib) write \
        [list Apol_Analysis_relabel::attribChanged]
    trace add variable Apol_Analysis_relabel::vals(subjects:attrib) write \
        [list Apol_Analysis_relabel::attribChanged]
    pack $attrib_enable -side top -expand 0 -fill x -anchor sw -padx 5 -pady 2
    pack $attrib_box -side top -expand 1 -fill x -padx 10
    attribEnabled $attrib_box
    if {$vals(mode) == "subject"} {
        $attrib_enable configure -state disabled
        $attrib_box configure -state disabled
    }
    $d draw
}
proc Apol_Analysis_relabel::createAdvancedFilter {f title varname disabled} {
    set l1 [label $f.l1 -text "Included $title"]
    set l2 [label $f.l2 -text "Excluded $title"]
    grid $l1 x $l2 -sticky w
    set inc [Apol_Widget::makeScrolledListbox $f.inc -height 10 -width 24 \
                 -listvar Apol_Analysis_relabel::vals($varname:inc) \
                 -selectmode extended -exportselection 0]
    set exc [Apol_Widget::makeScrolledListbox $f.exc -height 10 -width 24 \
                 -listvar Apol_Analysis_relabel::vals($varname:exc) \
                 -selectmode extended -exportselection 0]
    set inc_lb [Apol_Widget::getScrolledListbox $inc]
    set exc_lb [Apol_Widget::getScrolledListbox $exc]
    set bb [ButtonBox $f.bb -homogeneous 1 -orient vertical -spacing 4]
    $bb add -text "-->" -width 10 -command [list Apol_Analysis_relabel::moveToExclude $varname $inc_lb $exc_lb]
    $bb add -text "<--" -width 10 -command [list Apol_Analysis_relabel::moveToInclude $varname $inc_lb $exc_lb]
    grid $inc $bb $exc -sticky nsew
    set inc_bb [ButtonBox $f.inc_bb -homogeneous 1 -spacing 4]
    $inc_bb add -text "Select All" -command [list $inc_lb selection set 0 end]
    $inc_bb add -text "Unselect" -command [list $inc_lb selection clear 0 end]
    set exc_bb [ButtonBox $f.exc_bb -homogeneous 1 -spacing 4]
    $exc_bb add -text "Select All" -command [list $exc_lb selection set 0 end]
    $exc_bb add -text "Unselect" -command [list $exc_lb selection clear 0 end]
    grid $inc_bb x $exc_bb -pady 4
    grid columnconfigure $f 0 -weight 1 -uniform 0 -pad 2
    grid columnconfigure $f 1 -weight 0 -pad 8
    grid columnconfigure $f 2 -weight 1 -uniform 0 -pad 2
    if {$disabled} {
        foreach w [list $l1 $l2 $bb $inc_bb $exc_bb] {
            $w configure -state disabled
        }
        Apol_Widget::setScrolledListboxState $inc disabled
        Apol_Widget::setScrolledListboxState $exc disabled
    }
}
proc Apol_Analysis_relabel::moveToExclude {varname inc exc} {
    variable vals
    if {[set selection [$inc curselection]] == {}} {
        return
    }
    foreach i $selection {
        lappend perms [$inc get $i]
    }
    set vals($varname:exc) [lsort [concat $vals($varname:exc) $perms]]
    if {$varname == "subjects"} {
        set vals(subjects:exc_all) [lsort [concat $vals(subjects:exc_all) $perms]]
    }
    foreach p $perms {
        set i [lsearch $vals($varname:inc) $p]
        set vals($varname:inc) [lreplace $vals($varname:inc) $i $i]
        if {$varname == "subjects"} {
            set i [lsearch $vals(subjects:inc_all) $p]
            set vals(subjects:inc_all) [lreplace $vals(subjects:inc_all) $i $i]
        }
    }
    $inc selection clear 0 end
    $exc selection clear 0 end
}
proc Apol_Analysis_relabel::moveToInclude {varname inc exc} {
    variable vals
    if {[set selection [$exc curselection]] == {}} {
        return
    }
    foreach i $selection {
        lappend perms [$exc get $i]
    }
    set vals($varname:inc) [lsort [concat $vals($varname:inc) $perms]]
    if {$varname == "subjects"} {
        set vals(subjects:inc_all) [lsort [concat $vals(subjects:inc_all) $perms]]
    }
    foreach p $perms {
        set i [lsearch $vals($varname:exc) $p]
        set vals($varname:exc) [lreplace $vals($varname:exc) $i $i]
        if {$varname == "subjects"} {
            set i [lsearch $vals(subjects:exc_all) $p]
            set vals(subjects:exc_all) [lreplace $vals(subjects:exc_all) $i $i]
        }
    }
    $inc selection clear 0 end
    $exc selection clear 0 end
}
proc Apol_Analysis_relabel::attribEnabled {cb} {
    variable vals
    if {$vals(subjects:attribenable)} {
        $cb configure -state normal
        filterTypeLists $vals(subjects:attrib)
    } else {
        $cb configure -state disabled
        filterTypeLists ""
    }
}
proc Apol_Analysis_relabel::attribChanged {name1 name2 op} {
    variable vals
    if {$vals(subjects:attribenable)} {
        filterTypeLists $vals(subjects:attrib)
    }
}
proc Apol_Analysis_relabel::filterTypeLists {attrib} {
    variable vals
    if {$attrib != ""} {
        set typesList [lindex [apol_GetAttribs $attrib] 0 1]
        set vals(subjects:inc) {}
        set vals(subjects:exc) {}
        foreach t $typesList {
            if {[lsearch $vals(subjects:inc_all) $t] >= 0} {
                lappend vals(subjects:inc) $t
            }
            if {[lsearch $vals(subjects:exc_all) $t] >= 0} {
                lappend vals(subjects:exc) $t
            }
        }
        set vals(subjects:inc) [lsort $vals(subjects:inc)]
        set vals(subjects:exc) [lsort $vals(subjects:exc)]
    } else {
        set vals(subjects:inc) $vals(subjects:inc_all)
        set vals(subjects:exc) $vals(subjects:exc_all)
    }
}
proc Apol_Analysis_relabel::checkParams {} {
    variable vals
    variable widgets
    if {![ApolTop::is_policy_open]} {
        return "No current policy file is opened!"
    }
    set type [Apol_Widget::getTypeComboboxValueAndAttrib $widgets(type)]
    if {[lindex $type 0] == {}} {
        return "No type was selected."
    }
    set vals(type) [lindex $type 0]
    set vals(type:attrib) [lindex $type 1]
    set use_regexp [Apol_Widget::getRegexpEntryState $widgets(regexp)]
    set regexp [Apol_Widget::getRegexpEntryValue $widgets(regexp)]
    if {$use_regexp && $regexp == {}} {
            return "No regular expression provided."
    }
    set vals(regexp:enable) $use_regexp
    set vals(regexp) $regexp
    if {$vals(advanced_enable)} {
        if {$vals(classes:inc) == {}} {
            return "At least one object class must be included."
        }
        if {$vals(mode) == "object" && $vals(subjects:inc_all) == {}} {
            return "At least one subject type must be included."
        }
    }
    return {}  ;# all parameters passed, now ready to do search
}
proc Apol_Analysis_relabel::analyze {} {
    variable vals
    if {$vals(mode) == "object"} {
        if {$vals(mode:to) && $vals(mode:from)} {
            set mode "both"
        } elseif {$vals(mode:to)} {
            set mode "to"
        } else {
            set mode "from"
        }
    } else {
        set mode "subject"
    }
    if {$vals(advanced_enable) && $vals(classes:exc) != {}} {
        set classes $vals(classes:inc)
    } else {
        set classes {}
    }
    if {$vals(advanced_enable) && $vals(subjects:exc) != {}} {
        set subjects $vals(subjects:inc)
    } else {
        set subjects {}
    }
    if {$vals(regexp:enable)} {
        set regexp $vals(regexp)
    } else {
        set regexp {}
    }
    apol_RelabelAnalysis $mode $vals(type) $classes $subjects $regexp
}
proc Apol_Analysis_relabel::createResultsDisplay {} {
    variable vals
    set f [Apol_Analysis::createResultTab "Relabel" [array get vals]]
    if {$vals(mode) == "object"} {
        if {$vals(mode:to) && $vals(mode:from)} {
            set tree_title "Type $vals(type) relabels to/from"
        } elseif {$vals(mode:to)} {
            set tree_title "Type $vals(type) relabels to"
        } else {
            set tree_title "Type $vals(type) relabels from"
        }
    } else {
        set tree_title "Subject $vals(type) relabels"
    }
    set tree_tf [TitleFrame $f.left -text $tree_title]
    pack $tree_tf -side left -expand 0 -fill y -padx 2 -pady 2
    set sw [ScrolledWindow [$tree_tf getframe].sw -auto both]
    set tree [Tree [$sw getframe].tree -width 24 -redraw 1 -borderwidth 0 \
                  -highlightthickness 0 -showlines 1 -padx 0 -bg white]
    $sw setwidget $tree
    pack $sw -expand 1 -fill both
    set res_tf [TitleFrame $f.right -text "Relabeling Results"]
    pack $res_tf -side left -expand 1 -fill both -padx 2 -pady 2
    set res [Apol_Widget::makeSearchResults [$res_tf getframe].res]
    $res.tb tag configure title -font {Helvetica 14 bold}
    $res.tb tag configure title_type -foreground blue -font {Helvetica 14 bold}
    $res.tb tag configure num -font {Helvetica 12 bold}
    $res.tb tag configure type_tag -foreground blue -font {Helvetica 12 bold}
    pack $res -expand 1 -fill both
    $tree configure -selectcommand [list Apol_Analysis_relabel::treeSelect $res]
    return $f
}
proc Apol_Analysis_relabel::treeSelect {res tree node} {
    if {$node != {}} {
        $res.tb configure -state normal
        $res.tb delete 0.0 end
        set data [$tree itemcget $node -data]
        if {[string index $node 0] == "o"} {
            renderResultsRuleObject $res $tree $node $data
        } elseif {[string index $node 0] == "s"} {
            renderResultsRuleSubject $res $tree $node $data
        } else {
            eval $res.tb insert end $data
        }
        $res.tb configure -state disabled
    }
}
proc Apol_Analysis_relabel::clearResultsDisplay {f} {
    variable vals
    set tree [[$f.left getframe].sw getframe].tree
    set res [$f.right getframe].res
    $tree delete [$tree nodes root]
    Apol_Widget::clearSearchResults $res
    Apol_Analysis::setResultTabCriteria [array get vals]
}
proc Apol_Analysis_relabel::renderResults {f results} {
    variable vals
    set tree [[$f.left getframe].sw getframe].tree
    set res [$f.right getframe].res
    $tree insert end root top -text $vals(type) -open 1 -drawcross auto
    if {$vals(mode) == "object"} {
        set top_text [renderResultsObject $results $tree]
    } else {  ;# subject mode
        set top_text [renderResultsSubject $results $tree]
    }
    $tree itemconfigure top -data $top_text
    $tree selection set top
    $tree opentree top
    update idletasks
    $tree see top
}
proc Apol_Analysis_relabel::renderResultsObject {results tree} {
    variable vals
    if {$vals(mode:from) && $vals(mode:to)} {
        set dir both
    } elseif {$vals(mode:to)} {
        set dir to
    } else {
        set dir from
    }
    foreach r [lsort -index 0 $results] {
        foreach {type to from both} $r {break}
        set pairs [lsort -unique [concat $to $from $both]]
        $tree insert end top o:$dir:\#auto -text $type -data $pairs
    }
    set top_text [list "Direct Relabel Analysis: " title]
    switch -- $dir {
        both { lappend top_text "Starting/Ending Type: " title }
        to   { lappend top_text "Ending Type: " title }
        from { lappend top_text "Starting Type: " title }
    }
    lappend top_text $vals(type) title_type \
        "\n\n" title \
        $vals(type) type_tag
    if {[llength $results]} {
        switch -- $dir {
            both { lappend top_text " can be relabeled to and from " {} }
            to   { lappend top_text " can be relabeled to " {} }
            from { lappend top_text " can be relabeled from " {} }
        }
        lappend top_text [llength $results] num \
            " type(s).\n\n" {} \
            "This tab provides the results of a Direct Relabel Analysis beginning\n" {}
        switch -- $dir {
            both { lappend top_text "with the starting/ending" {} }
            to   { lappend top_text "with the starting" {} }
            from { lappend top_text "with the ending" {} }
        }
        lappend top_text " type above. The results of the analysis are\n" {} \
            "presented in tree form with the root of the tree (this node) being the\n" {} \
            "starting point for the analysis.\n\n" {} \
            "Each child node in the tree represents a type in the current policy\n" {} \
            "to/from which relabeling is allowed (depending on you selection\n" {} \
            "above)." {}
    } else {
        switch -- $dir {
            both { lappend top_text " cannot be relabeled to/from any type." {} }
            to   { lappend top_text " cannot be relabeled to any type." {} }
            from { lappend top_text " cannot be relabeled from any type." {} }
        }
    }
}
proc Apol_Analysis_relabel::renderResultsRuleObject {res tree node data} {
    set header [list [$tree itemcget top -text] title_type]
    lappend header " can be relabeled:\n" {}
    eval $res.tb insert end $header
    set dir [lindex [split $node :] 1]
    set target_type [$tree itemcget $node -text]
    foreach rule_pairs $data {
        set class [apol_RenderAVRuleClass [lindex $rule_pairs 0]]
        lappend classes($class) $rule_pairs
    }
    foreach key [lsort [array names classes]] {
        $res.tb configure -state normal
        $res.tb insert end "\n$key:\n" title
        foreach rule_pairs [lsort -index 2 $classes($key)] {
            foreach {a_rule b_rule intermed} $rule_pairs {break}
            if {$dir == "to" || $dir == "from"} {
                set dir_string $dir
            } else {
                set a_perms [apol_RenderAVRulePerms $a_rule]
                set b_perms [apol_RenderAVRulePerms $b_rule]
                if {[lsearch $a_perms "relabelto"] >= 0 && \
                        [lsearch $a_perms "relabelfrom"] >= 0 && \
                        [lsearch $b_perms "relabelto"] >= 0 && \
                        [lsearch $b_perms "relabelfrom"] >= 0} {
                    set dir_string "to and from"
                } elseif {[lsearch $a_perms "relabelto"] >= 0 &&
                          [lsearch $b_perms "relabelfrom"] >= 0} {
                    set dir_string "to"
                } else {
                    set dir_string "from"
                }
            }
            $res.tb configure -state normal
            $res.tb insert end "\n  $dir_string " num \
                $target_type type_tag \
                " by " {} \
                $intermed type_tag \
                "\n" {}
            Apol_Widget::appendSearchResultAVRules $res 6 $a_rule
            if {$a_rule != $b_rule} {
                Apol_Widget::appendSearchResultAVRules $res 6 $b_rule
            }
        }
    }
}
proc Apol_Analysis_relabel::renderResultsSubject {results tree} {
    variable vals
    set to_count 0
    set from_count 0
    foreach r $results {
        foreach {type to from both} $r {break}
        foreach rule [concat $to $both] {
            lappend to_types($type) [lindex $rule 0]  ;# second half of pair is NULL
        }
        foreach rule [concat $from $both] {
            lappend from_types($type) [lindex $rule 0]
        }
    }
    set to_count [llength [array names to_types]]
    if {$to_count} {
        set to_text [list $vals(type) title_type " can relabel to " {} ]
        lappend to_text $to_count num \
            " type(s). Open the subtree of this item to view the list of types." {}
        $tree insert end top to -text "To" -data $to_text -drawcross auto
        foreach type [lsort [array names to_types]] {
            set rules [lsort -unique $to_types($type)]
            $tree insert end to s\#auto -text $type -data [list to $rules]
        }
    }
    set from_count [llength [array names from_types]]
    if {$from_count} {
        set from_text [list $vals(type) title_type " can relabel from " {} ]
        lappend from_text $from_count num \
            " type(s). Open the subtree of this item to view the list of types." {}
        $tree insert end top from -text "From" -data $from_text -drawcross auto
        foreach type [lsort [array names from_types]] {
            set rules [lsort -unique $from_types($type)]
            $tree insert end from s\#auto -text $type -data [list from $rules]
        }
    }
    set top_text [list "Direct Relabel Analysis: Subject: " title]
    lappend top_text $vals(type) title_type \
        "\n\n" title \
        $vals(type) type_tag
    if {$to_count + $from_count} {
        lappend top_text " can relabel to " {} \
            $to_count num \
            " type(s) and relabel from " {} \
            $from_count num \
            " type(s).\n\n" {} \
            "This tab provides the results of a Direct Relabel Analysis for the\n" {} \
            "subject above. The results of the analysis are presented in tree form\n" {} \
            "with the root of the tree (this node) being the starting point for the\n" {} \
            "analysis.\n\n" {} \
            "Each child node in the To and From subtrees represents a type in the\n" {} \
            "current policy which the chosen subject can relabel." {}
    } else {
        lappend top_text " does not relabel to or from any type as a subject." {}
    }
}
proc Apol_Analysis_relabel::renderResultsRuleSubject {res tree node data} {
    foreach {dir rules} $data {break}
    set header [list [$tree itemcget top -text] title_type]
    lappend header " can relabel $dir " {} \
        [$tree itemcget $node -text] title_type \
        "\n\n" {}
    eval $res.tb insert end $header
    Apol_Widget::appendSearchResultAVRules $res 0 $rules
}
namespace eval Apol_Analysis_tra {
    variable vals
    variable widgets
    Apol_Analysis::registerAnalysis "Apol_Analysis_tra" "Types Relationship Summary"
}
proc Apol_Analysis_tra::open {} {
    variable widgets
    Apol_Widget::resetTypeComboboxToPolicy $widgets(typeA)
    Apol_Widget::resetTypeComboboxToPolicy $widgets(typeB)
}
proc Apol_Analysis_tra::close {} {
    variable widgets
    reinitializeVals
    reinitializeWidgets
    Apol_Widget::clearTypeCombobox $widgets(typeA)
    Apol_Widget::clearTypeCombobox $widgets(typeB)
}
proc Apol_Analysis_tra::getInfo {} {
    return "The types relationship summary analysis in Apol is a convenience
mechanism to allow a user to quickly do several queries and analyses
already in present in Apol to understand the relationship between two
types.  This is meant to quickly display the relationship between two
types and therefore does not include all of the options present in the
standard queries and analyses.
\nFor additional help on this topic select \"Types Relationship Summary
Analysis\" from the help menu."
}
proc Apol_Analysis_tra::create {options_frame} {
    variable vals
    variable widgets
    reinitializeVals
    set req_tf [TitleFrame $options_frame.req -text "Required Parameters"]
    pack $req_tf -side left -padx 2 -pady 2 -expand 0 -fill y
    set fA [frame [$req_tf getframe].fA]
    pack $fA -side left -anchor nw -padx 2
    set lA [label $fA.l -text "Type A"]
    pack $lA -anchor w
    set widgets(typeA) [Apol_Widget::makeTypeCombobox $fA.t -width 19]
    pack $widgets(typeA)
    set fB [frame [$req_tf getframe].fB]
    pack $fB -side left -anchor nw -padx 2
    set lB [label $fB.l -text "Type B"]
    pack $lB -anchor w
    set widgets(typeB) [Apol_Widget::makeTypeCombobox $fB.t -width 19]
    pack $widgets(typeB)
    set basic_tf [TitleFrame $options_frame.basic -text "Basic Relationships"]
    pack $basic_tf -side left -padx 2 -pady 2 -expand 0 -fill y
    foreach {t v} {"Common attributes" attribs \
                       "Common roles" roles \
                       "Common users" users \
                       "Similar access to resources" similars \
                       "Dissimilar access to resources" dissimilars
                       "TE allow rules" allows \
                       "Type transition/change rules" typerules} {
        set cb [checkbutton [$basic_tf getframe].$v -text $t \
                    -variable Apol_Analysis_tra::vals(run:$v)]
        pack $cb -anchor w
    }
    set an_tf [TitleFrame $options_frame.an -text "Analysis Relationships"]
    pack $an_tf -side left -padx 2 -pady 2 -expand 1 -fill both
    foreach {t v} {"Direct flows between A and B" direct \
                       "Transitive flows A -> B" transAB \
                       "Transitive flows B -> A" transBA \
                       "Domain transitions A -> B" domainAB \
                       "Domain transitions B -> A" domainBA} {
        set cb [checkbutton [$an_tf getframe].$v -text $t \
                    -variable Apol_Analysis_tra::vals(run:$v)]
        pack $cb -anchor w
    }
}
proc Apol_Analysis_tra::newAnalysis {} {
    if {[set rt [checkParams]] != {}} {
        return $rt
    }
    if {[catch {analyze} results]} {
        return $results
    }
    set f [createResultsDisplay]
    if {[catch {renderResults $f $results} rt]} {
        Apol_Analysis::deleteCurrentResults
        return $rt
    }
    return {}
}
proc Apol_Analysis_tra::updateAnalysis {f} {
    if {[set rt [checkParams]] != {}} {
        return $rt
    }
    if {[catch {analyze} results]} {
        return $results
    }
    clearResultsDisplay $f
    if {[catch {renderResults $f $results} rt]} {
        return $rt
    }
    return {}
}
proc Apol_Analysis_tra::reset {} {
    reinitializeVals
    reinitializeWidgets
}
proc Apol_Analysis_tra::switchTab {query_options} {
    variable vals
    variable widgets
    array set vals $query_options
    reinitializeWidgets
}
proc Apol_Analysis_tra::saveQuery {channel} {
    variable vals
    variable widgets
    foreach {key value} [array get vals] {
        puts $channel "$key $value"
    }
    set type [Apol_Widget::getTypeComboboxValueAndAttrib $widgets(typeA)]
    puts $channel "typeA [lindex $type 0]"
    puts $channel "typeA:attrib [lindex $type 1]"
    set type [Apol_Widget::getTypeComboboxValueAndAttrib $widgets(typeB)]
    puts $channel "typeB [lindex $type 0]"
    puts $channel "typeB:attrib [lindex $type 1]"
}
proc Apol_Analysis_tra::loadQuery {channel} {
    variable vals
    set classes_exc {}
    set subjects_exc {}
    while {[gets $channel line] >= 0} {
        set line [string trim $line]
        if {$line == {} || [string index $line 0] == "#"} {
            continue
        }
        set key {}
        set value {}
        regexp -line -- {^(\S+)( (.+))?} $line -> key --> value
        set vals($key) $value
    }
    reinitializeWidgets
}
proc Apol_Analysis_tra::gotoLine {tab line_num} {
    set searchResults [$tab.right getframe].res
    Apol_Widget::gotoLineSearchResults $searchResults $line_num
}
proc Apol_Analysis_tra::search {tab str case_Insensitive regExpr srch_Direction } {
    set textbox [$tab.right getframe].res.tb
    ApolTop::textSearch $textbox $str $case_Insensitive $regExpr $srch_Direction
}
proc Apol_Analysis_tra::reinitializeVals {} {
    variable vals
    array set vals {
        typeA {}  typeA:attrib {}
        typeB {}  typeB:attrib {}
        run:attribs 1
        run:roles 1
        run:users 1
        run:similars 0
        run:dissimilars 0
        run:allows 0
        run:typerules 0
        run:direct 0
        run:transAB 0
        run:transBA 0
        run:domainAB 0
        run:domainBA 0
    }
}
proc Apol_Analysis_tra::reinitializeWidgets {} {
    variable vals
    variable widgets
    if {$vals(typeA:attrib) != {}} {
        Apol_Widget::setTypeComboboxValue $widgets(typeA) [list $vals(typeA) $vals(typeA:attrib)]
    } else {
        Apol_Widget::setTypeComboboxValue $widgets(typeA) $vals(typeA)
    }
    if {$vals(typeB:attrib) != {}} {
        Apol_Widget::setTypeComboboxValue $widgets(typeB) [list $vals(typeB) $vals(typeB:attrib)]
    } else {
        Apol_Widget::setTypeComboboxValue $widgets(typeB) $vals(typeB)
    }
}
proc Apol_Analysis_tra::checkParams {} {
    variable vals
    variable widgets
    if {![ApolTop::is_policy_open]} {
        return "No current policy file is opened!"
    }
    set type [Apol_Widget::getTypeComboboxValueAndAttrib $widgets(typeA)]
    if {[lindex $type 0] == {}} {
        return "No type was selected for type A."
    }
    set vals(typeA) [lindex $type 0]
    set vals(typeA:attrib) [lindex $type 1]
    set type [Apol_Widget::getTypeComboboxValueAndAttrib $widgets(typeB)]
    if {[lindex $type 0] == {}} {
        return "No type was selected for type B."
    }
    set vals(typeB) [lindex $type 0]
    set vals(typeB:attrib) [lindex $type 1]
    set analysis_selected 0
    foreach key [array names vals run:*] {
        if {$vals($key)} {
            if {($key == "run:direct" || [string match run:trans* $key]) && \
                    ![Apol_Perms_Map::is_pmap_loaded]} {
                if {![Apol_Perms_Map::loadDefaultPermMap]} {
                    return "This analysis requires that a permission map is loaded."
                }
            }
            set analysis_selected 1
        }
    }
    if {!$analysis_selected} {
        return "At least one analysis must be selected."
    }
    return {}  ;# all parameters passed, now ready to do search
}
proc Apol_Analysis_tra::analyze {} {
    variable vals
    set analyses {}
    foreach key [array names vals run:*] {
        if {$vals($key)} {
            lappend analyses [lindex [split $key :] 1]
        }
    }
    apol_TypesRelationshipAnalysis $vals(typeA) $vals(typeB) $analyses
}
proc Apol_Analysis_tra::createResultsDisplay {} {
    variable vals
    set f [Apol_Analysis::createResultTab "Types Relationship" [array get vals]]
    set tree_tf [TitleFrame $f.left -text "Types Relationship Results"]
    pack $tree_tf -side left -expand 0 -fill y -padx 2 -pady 2
    set sw [ScrolledWindow [$tree_tf getframe].sw -auto both]
    set tree [Tree [$sw getframe].tree -width 24 -redraw 1 -borderwidth 0 \
                  -highlightthickness 0 -showlines 1 -padx 0 -bg white]
    $sw setwidget $tree
    pack $sw -expand 1 -fill both
    set res_tf [TitleFrame $f.right -text "Types Relationship Information"]
    pack $res_tf -side left -expand 1 -fill both -padx 2 -pady 2
    set res [Apol_Widget::makeSearchResults [$res_tf getframe].res]
    $res.tb tag configure title -font {Helvetica 14 bold}
    $res.tb tag configure title_type -foreground blue -font {Helvetica 14 bold}
    $res.tb tag configure subtitle -font {Helvetica 10 bold}
    $res.tb tag configure subtitle_dir -foreground blue -font {Helvetica 10 bold}
    $res.tb tag configure num -foreground blue -font {Helvetica 10 bold}
    pack $res -expand 1 -fill both
    update
    grid propagate $sw 0
    $tree configure -selectcommand [list Apol_Analysis_tra::treeSelect $res]
    return $f
}
proc Apol_Analysis_tra::treeSelect {res tree node} {
    if {$node != {}} {
        $res.tb configure -state normal
        $res.tb delete 0.0 end
        set data [$tree itemcget $node -data]
        set name [$tree itemcget $node -text]
        if {[set parent [$tree parent $node]] != "root"} {
            set parent_name [$tree itemcget $parent -text]
            set parent_data [$tree itemcget $parent -data]
        }
        switch -glob -- $node {
            pre:* {
                eval $res.tb insert end $data
            }
            simtitle {
                showSimilarTitle $res $data
            }
            sim:* {
                showSimilar $res $name $parent_data $data
            }
            distitle {
                showDissimilarTitle $res $data
            }
            dissubtitle* {
                showDissimilarSubtitle $res $data
            }
            dis:* {
                showDissimilar $res $name $parent_name $data
            }
            allow {
                showAllows $res $data
            }
            typerules {
                showTypeRules $res $data
            }
            x* {
                showDirectFlow $res $data
            }
            y* {
                showTransFlow $res $data
            }
            f:* {
                showDTA $res $data
            }
        }
        $res.tb configure -state disabled
    }
}
proc Apol_Analysis_tra::clearResultsDisplay {f} {
    variable vals
    set tree [[$f.left getframe].sw getframe].tree
    set res [$f.right getframe].res
    $tree delete [$tree nodes root]
    Apol_Widget::clearSearchResults $res
    Apol_Analysis::setResultTabCriteria [array get vals]
}
proc Apol_Analysis_tra::renderResults {f results} {
    variable vals
    set tree [[$f.left getframe].sw getframe].tree
    set res [$f.right getframe].res
    foreach {attribs roles users similars dissimilars allows typerules \
        dirflow transAB transBA domsAB domsBA} $results {break}
    if {$vals(run:attribs)} {
        renderCommon Attributes $tree $attribs
    }
    if {$vals(run:roles)} {
        renderCommon Roles $tree $roles
    }
    if {$vals(run:users)} {
        renderCommon Users $tree $users
    }
    if {$vals(run:similars)} {
        renderSimilars $tree $similars
    }
    if {$vals(run:dissimilars)} {
        renderDissimilars $tree $dissimilars
    }
    if {$vals(run:allows)} {
        renderAllows $tree $allows
    }
    if {$vals(run:typerules)} {
        renderTypeRules $tree $typerules
    }
    if {$vals(run:direct)} {
        renderDirectFlow $tree $dirflow
    }
    if {$vals(run:transAB)} {
        renderTransFlow 0 $tree $transAB
    }
    if {$vals(run:transBA)} {
        renderTransFlow 1 $tree $transBA
    }
    if {$vals(run:domainAB)} {
        renderDTA 0 $tree $domsAB
    }
    if {$vals(run:domainBA)} {
        renderDTA 1 $tree $domsBA
    }
    set first_node [$tree nodes root 0]
    $tree selection set $first_node
    update idletasks
    $tree see $first_node
}
proc Apol_Analysis_tra::renderCommon {title tree names} {
    set text [list "Common $title ([llength $names]):\n\n" title]
    foreach n [lsort $names] {
        lappend text "$n\n" {}
    }
    $tree insert end root pre:$title -text "Common $title" -data $text
}
proc Apol_Analysis_tra::renderSimilars {tree results} {
    variable vals
    foreach {simA simB} $results {break}
    set data [list $vals(typeA) $vals(typeB) [llength $simA]]
    $tree insert end root simtitle -text "Similar access to resources" -data $data -drawcross allways
    foreach accessA [lsort -index 0 $simA] accessB [lsort -index 0 $simB] {
        set type [lindex $accessA 0]
        set rulesA [lindex $accessA 1]
        set rulesB [lindex $accessB 1]
        $tree insert end simtitle sim:$type -text $type -data [list $rulesA $rulesB]
    }
}
proc Apol_Analysis_tra::showSimilarTitle {res data} {
    foreach {typeA typeB numTypes} $data {break}
    $res.tb insert end $typeA title_type \
        " and " title \
        $typeB title_type \
        " access $numTypes common type(s).\n\n" title \
        "Open the subtree for this item to see the list of common types that
can be accessed.  You may then select a type from the subtree to see
the allow rules which provide the access." {}
}
proc Apol_Analysis_tra::showSimilar {res name parent_data data} {
    foreach {typeA typeB} $parent_data {rulesA rulesB} $data {break}
    $res.tb insert end $typeA title_type \
        " accesses " title \
        $name title_type \
        ":\n\n" title
    Apol_Widget::appendSearchResultAVRules $res 2 $rulesA
    $res.tb insert end "\n" title \
        $typeB title_type \
        " accesses " title \
        $name title_type \
        ":\n\n" title
    Apol_Widget::appendSearchResultAVRules $res 2 $rulesB
}
proc Apol_Analysis_tra::renderDissimilars {tree results} {
    variable vals
    foreach {disA disB} $results {break}
    set data [list $vals(typeA) $vals(typeB)]
    $tree insert end root distitle -text "Dissimilar access to resources" -data $data
    set data [list $vals(typeA) $vals(typeB) [llength $disA]]
    $tree insert end distitle dissubtitleA -text $vals(typeA) -data $data -drawcross allways
    foreach access [lsort -index 0 $disA] {
        set type [lindex $access 0]
        set rules [lindex $access 1]
        $tree insert end dissubtitleA dis:$type -text $type -data $rules
    }
    set data [list $vals(typeB) $vals(typeA) [llength $disB]]
    $tree insert end distitle dissubtitleB -text $vals(typeB) -data $data -drawcross allways
    foreach access [lsort -index 0 $disB] {
        set type [lindex $access 0]
        set rules [lindex $access 1]
        $tree insert end dissubtitleB dis:$type -text $type -data $rules
    }
}
proc Apol_Analysis_tra::showDissimilarTitle {res data} {
    foreach {typeA typeB} $data {break}
    $res.tb insert end "Dissimilar access between " title \
        $typeA title_type \
        " and " title \
        $typeB title_type \
        ".\n\n" title \
        "Open the subtree for this item to access individual subtrees of types
which can be accessed by one type but not the other.  You may then
select a type from a subtree to see the allow rules which provide the
access." {}
}
proc Apol_Analysis_tra::showDissimilarSubtitle {res data} {
    foreach {one_type other_type numTypes} $data {break}
    $res.tb insert end $one_type title_type \
        " accesss $numTypes type(s) to which " title \
        $other_type title_type \
        " does not have access.\n\n" title \
        "Open the subtree for this item to see the list of types.  You may then
select a type from the subtree to see the allow rules which provide
the access." {}
}
proc Apol_Analysis_tra::showDissimilar {res name parent_name data} {
    $res.tb insert end $parent_name title_type \
        " accesses " title \
        $name title_type \
        ":\n\n" title
    Apol_Widget::appendSearchResultAVRules $res 2 $data
}
proc Apol_Analysis_tra::renderAllows {tree rules} {
    $tree insert end root allow -text "TE Allow Rules" -data $rules
}
proc Apol_Analysis_tra::showAllows {res data} {
    $res.tb insert end "TE Allow Rules ([llength $data]):\n\n" title
    Apol_Widget::appendSearchResultAVRules $res 2 $data
}
proc Apol_Analysis_tra::renderTypeRules {tree rules} {
    $tree insert end root typerules -text "Type Transition/Change Rules" -data $rules
}
proc Apol_Analysis_tra::showTypeRules {res data} {
    $res.tb insert end "Type transition/change rules ([llength $data]):\n\n" title
    foreach r $data {
         Apol_Widget::appendSearchResultTERules $res 2 $r
    }
}
proc Apol_Analysis_tra::renderDirectFlow {tree dirflows} {
    if {$dirflows == {}} {
        $tree insert end root pre:direct
        set node [$tree nodes root end]
        set data [list "No direct information flows found." title]
    } else {
        variable vals
        Apol_Analysis_directflow::createResultsNodes $tree root $dirflows 0
        set node [$tree nodes root end]
        set data [list $vals(typeA) $vals(typeB) [$tree itemcget $node -data]]
    }
    $tree itemconfigure $node -text "Direct Flows Between A and B" -data $data -drawcross auto
}
proc Apol_Analysis_tra::showDirectFlow {res data} {
    foreach {parent_name name data} $data {break}
    foreach {flow_dir classes} [lindex $data 1] {break}
    $res.tb insert end "Information flows both into and out of " title \
        $parent_name title_type \
        " from/to " title \
        $name title_type
    $res.tb insert end "\n\n" title_type \
        "Objects classes for " subtitle \
        [string toupper $flow_dir] subtitle_dir \
        " flows:\n" subtitle
    foreach c $classes {
        foreach {class_name rules} $c {break}
        $res.tb insert end "      " {} \
            $class_name\n subtitle
        Apol_Widget::appendSearchResultAVRules $res 12 $rules
    }
}
proc Apol_Analysis_tra::renderTransFlow {dir tree transflows} {
    variable vals
    if {$dir == 0} {
        set title2 "A->B"
        set data [list $vals(typeB) $vals(typeA)]
    } else {
        set title2 "B->A"
        set data [list $vals(typeA) $vals(typeB)]
    }
    if {$transflows == {}} {
        $tree insert end root pre:trans$dir
        set node [$tree nodes root end]
        set data [list "No transitive information flows found." title]
    } else {
        Apol_Analysis_transflow::createResultsNodes $tree root $transflows 0
        set node [$tree nodes root end]
        lappend data [$tree itemcget $node -data]
    }
    $tree itemconfigure $node -text "Transitive Flows $title2" -data $data -drawcross auto
}
proc Apol_Analysis_tra::showTransFlow {res data} {
    foreach {parent_name name data} $data {break}
    foreach {flow_dir paths} [lindex $data 1] {break}
    $res.tb insert end "Information flows from " title \
        $name title_type \
        " to " title \
        $parent_name title_type
    $res.tb insert end "\n\n" title \
        "Apol found the following number of information flows: " subtitle \
        [llength $paths] num \
        "\n" subtitle
    set path_num 1
    foreach path $paths {
        $res.tb insert end "\n" {}
        Apol_Analysis_transflow::renderPath $res $path_num $path
        incr path_num
    }
}
proc Apol_Analysis_tra::renderDTA {dir tree dta} {
    variable vals
    if {$dir == 0} {
        set title2 "A->B"
        set data [list $vals(typeA) $vals(typeB)]
    } else {
        set title2 "B->A"
        set data [list $vals(typeB) $vals(typeA)]
    }
    if {$dta == {}} {
        $tree insert end root pre:dta$dir
        set node [$tree nodes root end]
        set data [list "No domain transitions found." title]
    } else {
        Apol_Analysis_domaintrans::createResultsNodes $tree root $dta forward
        set node [$tree nodes root end]
        lappend data [$tree itemcget $node -data]
    }
    $tree itemconfigure $node -text "Domain Transitions $title2" -data $data -drawcross auto
}
proc Apol_Analysis_tra::showDTA {res data} {
    foreach {parent_name name data} $data {break}
    foreach {proctrans setexec ep access_list} [lindex $data 1] {break}
    set header [list "Domain transition from " title \
                    $parent_name title_type \
                    " to " title \
                    $name title_type]
    eval $res.tb insert end $header
    $res.tb insert end "\n\n" title_type
    $res.tb insert end "Process Transition Rules: " subtitle \
        [llength $proctrans] num \
        "\n" subtitle
    Apol_Widget::appendSearchResultAVRules $res 6 $proctrans
    if {[llength $setexec] > 0} {
        $res.tb insert end "\n" {} \
            "Setexec Rules: " subtitle \
            [llength $setexec] num \
            "\n" subtitle
        Apol_Widget::appendSearchResultAVRules $res 6 $setexec
    }
    $res.tb insert end "\nEntry Point File Types: " subtitle \
        [llength $ep] num
    foreach e [lsort -index 0 $ep] {
        foreach {intermed entrypoint execute type_trans} $e {break}
        $res.tb insert end "\n      $intermed\n" {} \
            "            " {} \
            "File Entrypoint Rules: " subtitle \
            [llength $entrypoint] num \
            "\n" subtitle
        Apol_Widget::appendSearchResultAVRules $res 12 $entrypoint
        $res.tb insert end "\n" {} \
            "            " {} \
            "File Execute Rules: " subtitle \
            [llength $execute] num \
            "\n" subtitle
        Apol_Widget::appendSearchResultAVRules $res 12 $execute
        if {[llength $type_trans] > 0} {
            $res.tb insert end "\n" {} \
                "            " {} \
                "Type_transition Rules: " subtitle \
                [llength $type_trans] num \
                "\n" subtitle
            Apol_Widget::appendSearchResultTERules $res 12 $type_trans
        }
    }
    if {[llength $access_list] > 0} {
        $res.tb insert end "\n" {} \
            "The access filters you specified returned the following rules: " subtitle \
            [llength $access_list] num \
            "\n" subtitle
        Apol_Widget::appendSearchResultAVRule $res 6 $access_list
    }
}
namespace eval Apol_Open_Policy_Dialog {
    variable dialog {}
    variable widgets
    variable vars
}
proc Apol_Open_Policy_Dialog::getPolicyPath {defaultPath} {
    variable dialog
    variable vars
    array unset vars
    _create_dialog .
    if {$defaultPath == {}} {
        set defaultPath [list monolithic {} {}]
    }
    set vars(path_type) [lindex $defaultPath 0]
    if {[set vars(primary_file) [lindex $defaultPath 1]] != {}} {
        $dialog itemconfigure 0 -state normal
    }
    set vars(last_module) $vars(primary_file)
    set vars(mod_names) {}
    set vars(mod_vers) {}
    set vars(mod_paths) {}
    foreach m [lindex $defaultPath 2] {
        if {[catch {apol_GetModuleInfo $m} info]} {
            tk_messageBox -icon error -type ok -title "Open Module" -message $info
        } else {
            foreach {name vers} $info {break}
            lappend vars(mod_names) $name
            lappend vars(mod_vers) $vers
            lappend vars(mod_paths) $m
            set vars(last_module) $m
        }
    }
    $dialog.bbox _redraw
    $dialog draw
    destroy $dialog
}
proc Apol_Open_Policy_Dialog::_create_dialog {parent} {
    variable dialog
    variable widgets
    variable vars
    destroy $dialog
    set dialog [Dialog .open_policy_dialog -modal local -parent $parent \
                    -cancel 1 \
                    -separator 1 -homogeneous 1 -title "Open Policy"]
    set f [$dialog getframe]
    set policy_type_f [frame $f.policy_type]
    pack $policy_type_f -padx 4 -pady 4 -expand 0 -anchor w
    set l [label $policy_type_f.l -text "Policy Type:"]
    set mono_cb [radiobutton $policy_type_f.mono -text "Monolithic policy" \
                     -value monolithic \
                     -variable Apol_Open_Policy_Dialog::vars(path_type)]
    set mod_cb [radiobutton $policy_type_f.mod -text "Modular policy" \
                    -value modular \
                    -variable Apol_Open_Policy_Dialog::vars(path_type)]
    pack $l -anchor w
    pack $mono_cb $mod_cb -anchor w -padx 8
    set primary_f [frame $f.primary]
    pack $primary_f -padx 4 -pady 8 -expand 0 -fill x
    set widgets(main_label) [label $primary_f.l -text "Policy Filename:"]
    pack $widgets(main_label) -anchor w
    frame $primary_f.f
    pack $primary_f.f -expand 1 -fill x
    set e [entry $primary_f.f.e -width 32 -bg white \
               -textvariable Apol_Open_Policy_Dialog::vars(primary_file) \
               -validate key \
               -vcmd [list Apol_Open_Policy_Dialog::validateEntryKey %P]]
    set b [button $primary_f.f.b -text "Browse" \
               -command Apol_Open_Policy_Dialog::browsePrimary]
    pack $e -side left -expand 1 -fill x -padx 4
    pack $b -side right -expand 0 -padx 4
    set modules_f [frame $f.modules]
    pack $modules_f -pady 4 -padx 4 -expand 1 -fill both
    set mod_list_f [frame $modules_f.mods -relief sunken]
    pack $mod_list_f -side left -expand 1 -fill both -padx 4
    set mlabel [label $mod_list_f.ml -text "Module:"]
    set vlabel [label $mod_list_f.vl -text "Version:"]
    set plabel [label $mod_list_f.pl -text "Path:"]
    grid $mlabel $vlabel $plabel x -sticky w
    set dis_bg [$mlabel cget -bg]
    set ml [listbox $mod_list_f.mods -height 6 -width 10 \
                -listvariable Apol_Open_Policy_Dialog::vars(mod_names)]
    set vl [listbox $mod_list_f.vers -height 6 -width 4 \
                -listvariable Apol_Open_Policy_Dialog::vars(mod_vers)]
    set pl [listbox $mod_list_f.paths -height 6 -width 24 \
                -listvariable Apol_Open_Policy_Dialog::vars(mod_paths)]
    set sb [scrollbar $mod_list_f.sb -orient vertical \
                -command [list Apol_Open_Policy_Dialog::multiscroll yview]]
    grid $ml $vl $pl $sb -sticky nsew
    set widgets(bb) [ButtonBox $modules_f.bb -homogeneous 1 -orient vertical -pady 2]
    $widgets(bb) add -text "Add" -command Apol_Open_Policy_Dialog::browseModule
    $widgets(bb) add -text "Remove" -command Apol_Open_Policy_Dialog::removeModule -state disabled
    $widgets(bb) add -text "Import" -command Apol_Open_Policy_Dialog::importList
    $widgets(bb) add -text "Export" -command Apol_Open_Policy_Dialog::exportList -state disabled
    pack $widgets(bb) -side right -expand 0 -anchor n -padx 4 -pady 10
    set widgets(listboxes) [list $ml $vl $pl]
    set widgets(scrollbar) $sb
    foreach lb $widgets(listboxes) {
        $lb configure -yscrollcommand Apol_Open_Policy_Dialog::multiyview \
                -relief groove -bg white -exportselection 0
        bind $lb <<ListboxSelect>> \
            [list Apol_Open_Policy_Dialog::multiselect $lb]
    }
    trace add variable Apol_Open_Policy_Dialog::vars(path_type) write \
        [list Apol_Open_Policy_Dialog::togglePathType \
             [list $mlabel $vlabel $plabel] $dis_bg]
    $dialog add -text "Ok" -command Apol_Open_Policy_Dialog::tryOpenPolicy \
        -state disabled
    $dialog add -text "Cancel"
}
proc Apol_Open_Policy_Dialog::validateEntryKey {newvalue} {
    variable vars
    variable dialog
    variable widgets
    if {$newvalue == {}} {
        $dialog itemconfigure 0 -state disabled
        $widgets(bb) itemconfigure 3 -state disabled
    } else {
        $dialog itemconfigure 0 -state normal
        if {$vars(path_type) == "modular"} {
            $widgets(bb) itemconfigure 3 -state normal
        } else {
            $widgets(bb) itemconfigure 3 -state disabled
        }
    }
    return 1
}
proc Apol_Open_Policy_Dialog::togglePathType {labels disabled_bg name1 name2 op} {
    variable vars
    variable widgets
    if {$vars(path_type) == "modular"} {
        set state normal
        set bg white
        $widgets(main_label) configure -text "Base Filename:"
    } else {
        set state disabled
        set bg $disabled_bg
        $widgets(main_label) configure -text "Policy Filename:"
    }
    foreach w $labels {
        $w configure -state $state
    }
    foreach w $widgets(listboxes) {
        $w configure -state $state -bg $bg
    }
    $widgets(bb) configure -state $state
    if {$state == "normal" && [[lindex $widgets(listboxes) 0] curselection] > 0} {
        $widgets(bb) itemconfigure 1 -state normal
    } else {
        $widgets(bb) itemconfigure 1 -state disabled
    }
    if {$state == "normal" && $vars(primary_file) != {}} {
        $widgets(bb) itemconfigure 3 -state normal
    } else {
        $widgets(bb) itemconfigure 3 -state disabled
    }
}
proc Apol_Open_Policy_Dialog::browsePrimary {} {
    variable vars
    variable dialog
    if {$vars(path_type) == "monolithic"} {
        set title "Open Monolithic Policy"
    } else {
        set title "Open Modular Policy"
    }
    set f [tk_getOpenFile -initialdir [file dirname $vars(primary_file)] \
               -initialfile $vars(primary_file) -parent $dialog -title $title]
    if {$f != {}} {
        set vars(primary_file) $f
        $dialog itemconfigure 0 -state normal
    }
}
proc Apol_Open_Policy_Dialog::browseModule {} {
    variable vars
    variable dialog
    set paths [tk_getOpenFile -initialdir [file dirname $vars(last_module)] \
                   -initialfile $vars(last_module) -parent $dialog \
                   -title "Open Module" -multiple 1]
    if {$paths == {}} {
        return
    }
    foreach f $paths {
        addModule $f
    }
}
proc Apol_Open_Policy_Dialog::addModule {f} {
    variable vars
    variable widgets
    if {[lsearch $vars(mod_paths) $f] >= 0} {
        tk_messageBox -icon error -type ok -title "Open Module" -message "Module $f was already added."
        return
    }
    if {[catch {apol_GetModuleInfo $f} info]} {
        tk_messageBox -icon error -type ok -title "Open Module" -message $info
    } else {
        foreach {name vers} $info {break}
        set vars(mod_names) [lsort [concat $vars(mod_names) $name]]
        set i [lsearch $vars(mod_names) $name]
        set vars(mod_vers) [linsert $vars(mod_vers) $i $vers]
        set vars(mod_paths) [linsert $vars(mod_paths) $i $f]
        foreach lb $widgets(listboxes) {
            $lb selection clear 0 end
            $lb selection set $i
        }
        [lindex $widgets(listboxes) 0] see $i
        set vars(last_module) $f
        $widgets(bb) itemconfigure 1 -state normal
    }
}
proc Apol_Open_Policy_Dialog::removeModule {} {
    variable widgets
    set i [[lindex $widgets(listboxes) 0] curselection]
    if {[llength $i] > 0} {
        foreach lb $widgets(listboxes) {
            $lb delete [lindex $i 0]
        }
    }
    $widgets(bb) itemconfigure 1 -state disabled
}
proc Apol_Open_Policy_Dialog::importList {} {
    variable vars
    variable dialog
    variable widgets
    set f [tk_getOpenFile -initialdir [file dirname $vars(primary_file)] \
               -parent $dialog -title "Import Policy List"]
    if {$f == {}} {
        return
    }
    if {[catch {apol_OpenPolicyList $f} ppath]} {
        tk_messageBox -icon error -type ok -title "Import Policy List" \
            -message "Error importing policy list $f: $ppath"
        return
    }
    foreach lb $widgets(listboxes) {
        $lb delete 0 end
    }
    foreach {path_type base mods} $ppath {break}
    set vars(path_type) $path_type
    set vars(primary_file) $base
    validateEntryKey $base
    if {$path_type == "modular"} {
        foreach m $mods {
            addModule $m
        }
    }
}
proc Apol_Open_Policy_Dialog::exportList {} {
    variable vars
    variable dialog
    set f [tk_getSaveFile -parent $dialog -title "Export Policy List"]
    if {$f == {}} {
        return
    }
    set path [list $vars(path_type) $vars(primary_file) $vars(mod_paths)]
    if {[catch {apol_SavePolicyList $path $f} err]} {
        tk_messageBox -icon error -type ok -title "Export Policy List" \
            -message "Error exporting policy list $f: $err"
    }
}
proc Apol_Open_Policy_Dialog::multiscroll {args} {
    variable widgets
    foreach lb $widgets(listboxes) {
        eval $lb $args
    }
}
proc Apol_Open_Policy_Dialog::multiselect {lb} {
    variable widgets
    set sellist [$lb curselection]
    set enable_remove 0
    foreach lb $widgets(listboxes) {
        $lb selection clear 0 end
        foreach item $sellist {
            $lb selection set $item
            set enable_remove 1
        }
    }
    if {$enable_remove} {
        $widgets(bb) itemconfigure 1 -state normal
    }
}
proc Apol_Open_Policy_Dialog::multiyview {args} {
    variable widgets
    eval $widgets(scrollbar) set $args
    multiscroll yview moveto [lindex $args 0]
}
proc Apol_Open_Policy_Dialog::tryOpenPolicy {} {
    variable dialog
    variable vars
    set path [list $vars(path_type) $vars(primary_file) $vars(mod_paths)]
    if {[ApolTop::openPolicyFile $path] == 0} {
        $dialog enddialog {}
    }
}
namespace eval ApolTop {
    variable status {}
    variable policy_version_string {}
    variable last_policy_path {}
    variable policyConf_lineno {}
    variable polstats
    variable policy_stats_summary {}
    variable gui_ver
    variable apol_install_dir
    variable copyright_date "2001-2007"
    variable recent_files {}
    variable max_recent_files	5
    variable dot_apol_file	"[file join "$::env(HOME)" ".apol"]"
    variable goto_line_num
    variable prevCursor		arrow
    variable default_bg_color
    set default_bg_color	[. cget -background]
    variable text_font		""
    variable title_font		""
    variable dialog_font	""
    variable general_font	""
    variable query_file_ext	".qf"
    variable top_width          1000
    variable top_height         700
    variable libsefs		0
    variable helpDlg
    set helpDlg .apol_helpDlg
    variable searchDlg .apol_find_dialog
    variable goto_Dialog .apol_goto_dialog
    variable options_Dialog .apol_options_dialog
    variable mainframe
    variable textbox_policyConf
    variable searchDlg_entryBox
    variable gotoDlg_entryBox
    variable notebook
    variable components_nb
    variable rules_nb
    variable mls_tabs {}  ;# list of notebook tabs that are only for MLS
    variable searchString	""
    variable case_sensitive     0
    variable regExpr		0
    variable srch_Direction	"down"
    variable policy_is_open	0
    variable tabName_prefix	"Apol_"
    variable components_tab	"Apol_Components"
    variable types_tab		"Apol_Types"
    variable class_perms_tab	"Apol_Class_Perms"
    variable roles_tab		"Apol_Roles"
    variable users_tab		"Apol_Users"
    variable cond_bools_tab	"Apol_Cond_Bools"
    variable mls_tab            "Apol_MLS"
    variable initial_sids_tab	"Apol_Initial_SIDS"
    variable net_contexts_tab	"Apol_NetContexts"
    variable fs_contexts_tab	"Apol_FSContexts"
    variable rules_tab		"Apol_Rules"
    variable terules_tab	"Apol_TE"
    variable cond_rules_tab	"Apol_Cond_Rules"
    variable rbac_tab		"Apol_RBAC"
    variable range_tab		"Apol_Range"
    variable file_contexts_tab	"Apol_File_Contexts"
    variable analysis_tab	"Apol_Analysis"
    variable policy_conf_tab	"Apol_PolicyConf"
    variable tab_names {
        Types Class_Perms Roles Users Cond_Bools MLS Initial_SIDS NetContexts FSContexts
        TE Cond_Rules RBAC Range
        File_Contexts
        Analysis
        PolicyConf
    }
    variable tk_msgBox_Wait
    variable show_fake_attrib_warning 1
}
proc ApolTop::is_policy_open {} {
    return $ApolTop::policy_is_open
}
proc ApolTop::get_install_dir {} {
    return $ApolTop::apol_install_dir
}
proc ApolTop::get_toplevel_dialog {} {
    return $ApolTop::mainframe
}
proc ApolTop::is_capable {capability} {
    if {![is_policy_open]} {
        return 0;
    }
    return [apol_IsCapable $capability]
}
proc ApolTop::load_fc_index_file {} {
    set rt [Apol_File_Contexts::load_fc_db]
    if {$rt == 1} {
        ApolTop::configure_load_index_menu_item 1
    }
    return 0
}
proc ApolTop::create_fc_index_file {} {
    Apol_File_Contexts::display_create_db_dlg
    return 0
}
proc ApolTop::load_perm_map_fileDlg {} {
    if {[Apol_Perms_Map::loadPermMapFromFile]} {
        ApolTop::configure_edit_pmap_menu_item 1
    }
}
proc ApolTop::load_default_perm_map_Dlg {} {
    if {[Apol_Perms_Map::loadDefaultPermMap]} {
        ApolTop::configure_edit_pmap_menu_item 1
    }
}
proc ApolTop::configure_edit_pmap_menu_item {enable} {
    variable mainframe
    if {$enable} {
        [$mainframe getmenu pmap_menu] entryconfigure last -state normal -label "Edit Perm Map..."
    } else {
        [$mainframe getmenu pmap_menu] entryconfigure last -state disabled -label "Edit Perm Map... (Not loaded)"
    }
}
proc ApolTop::configure_load_index_menu_item {enable} {
    variable mainframe
    if {$enable} {
        [$mainframe getmenu fc_index_menu] entryconfigure last -label "Load Index..."
    } else {
        [$mainframe getmenu fc_index_menu] entryconfigure last -label "Load Index... (Not loaded)"
    }
}
proc ApolTop::popup_listbox_Menu { global x y popup callbacks list_box} {
    focus -force $list_box
    set selected_item [$list_box get active]
    if {$selected_item == ""} {
        return
    }
    set gx [winfo rootx $global]
    set gy [winfo rooty $global]
    set cmx [expr $gx + $x]
    set cmy [expr $gy + $y]
    $popup delete 0 end
    foreach callback $callbacks {
        $popup add command -label "[lindex $callback 0]" -command "[lindex $callback 1] $selected_item"
    }
    tk_popup $popup $cmx $cmy
}
proc ApolTop::popup_Tab_Menu { window x y popupMenu callbacks page } {
    if {$page == ""} {
        return
    }
    set gx [winfo rootx $window]
    set gy [winfo rooty $window]
    set cmx [expr $gx + $x]
    set cmy [expr $gy + $y]
    $popupMenu delete 0 end
    foreach callback $callbacks {
        $popupMenu add command -label [lindex $callback 0] -command [list [lindex $callback 1] $page]
    }
    tk_popup $popupMenu $cmx $cmy
}
proc ApolTop::set_Focus_to_Text { tab } {
    variable components_nb
    variable rules_nb
    variable file_contexts_tab
    $ApolTop::mainframe setmenustate Disable_SearchMenu_Tag normal
    $ApolTop::mainframe setmenustate Disable_LoadQuery_Tag normal
    set ApolTop::policyConf_lineno {}
    switch -exact -- $tab \
        $ApolTop::components_tab {
            $ApolTop::mainframe setmenustate Disable_SaveQuery_Tag disabled
            ApolTop::set_Focus_to_Text [$components_nb raise]
        } \
        $ApolTop::rules_tab {
            ApolTop::set_Focus_to_Text [$rules_nb raise]
        } \
        $ApolTop::terules_tab {
            $ApolTop::mainframe setmenustate Disable_SaveQuery_Tag normal
            set raisedPage [Apol_TE::get_results_raised_tab]
            if {$raisedPage != ""} {
                Apol_TE::set_Focus_to_Text $raisedPage
            } else {
                focus [$ApolTop::rules_nb getframe $ApolTop::terules_tab]
            }
        } \
        $ApolTop::analysis_tab {
            $ApolTop::mainframe setmenustate Disable_SaveQuery_Tag normal
        } \
        default {
            $ApolTop::mainframe setmenustate Disable_SaveQuery_Tag disabled
            ${tab}::set_Focus_to_Text
        }
}
proc ApolTop::textSearch { w str case_Insensitive regExpr srch_Direction } {
    if {$str == {}} {
        return
    }
    set case_opt {}
    set regExpr_opt {}
    set direction_opt {}
    if {$case_Insensitive} {
        set case_opt "-nocase"
    }
    if {$regExpr} {
        set regExpr_opt "-regexp"
    }
    if { $srch_Direction == "down" } {
        set direction_opt "-forward"
        set cur_srch_pos [$w index insert]
    } else {
        set direction_opt "-backward"
        set cur_srch_pos [lindex [$w tag ranges sel] 0]
    }
    if { $cur_srch_pos == "" } {
        set cur_srch_pos "1.0"
    }
    $w tag remove sel 0.0 end
    set cur_srch_pos [eval [list $w search -count cur_srch_pos_length] $case_opt $regExpr_opt $direction_opt [list -- $str $cur_srch_pos]]
    if {$cur_srch_pos == {}} {
        set ApolTop::tk_msgBox_Wait  \
            [tk_messageBox -parent $ApolTop::searchDlg -icon warning -type ok -title "Search Failed" -message \
                 "Search string not found."]
        vwait ApolTop::tk_msgBox_Wait
    } else {
        $w mark set insert "$cur_srch_pos + $cur_srch_pos_length char"
        $w tag add sel $cur_srch_pos "$cur_srch_pos + $cur_srch_pos_length char"
        $w see $cur_srch_pos
    }
}
proc ApolTop::search {} {
    variable searchString
    variable case_sensitive
    variable regExpr
    variable srch_Direction
    variable notebook
    variable components_nb
    variable rules_nb
    variable components_tab
    variable rules_tab
    variable policy_conf_tab
    variable analysis_tab
    variable file_contexts_tab
    if {$case_sensitive} {
        set insens 0
    } else {
        set insens 1
    }
    set raised_tab [$notebook raise]
    switch -- $raised_tab \
        $policy_conf_tab {
            ${policy_conf_tab}::search $searchString $insens $regExpr $srch_Direction
        } \
        $analysis_tab {
            ${analysis_tab}::search $searchString $insens $regExpr $srch_Direction
        } \
        $rules_tab {
            [$rules_nb raise]::search $searchString $insens $regExpr $srch_Direction
        } \
        $components_tab {
            [$components_nb raise]::search $searchString $insens $regExpr $srch_Direction
        } \
        $file_contexts_tab {
            ${file_contexts_tab}::search $searchString $insens $regExpr $srch_Direction
        } \
        default {
            puts "Invalid raised tab!"
        }
}
proc ApolTop::load_query_info {} {
    variable notebook
    variable rules_tab
    variable terules_tab
    variable analysis_tab
    variable rules_nb
    variable mainframe
    set query_file ""
    set types {
        {"Query files"		{$ApolTop::query_file_ext}}
    }
    set query_file [tk_getOpenFile -filetypes $types -title "Select Query to Load..." \
                        -defaultextension $ApolTop::query_file_ext -parent $mainframe]
    if {$query_file != ""} {
        if {[file exists $query_file] == 0 } {
            tk_messageBox -icon error -type ok -title "Error" \
                -message "File $query_file does not exist." -parent $mainframe
            return -1
        }
        set rt [catch {set f [::open $query_file]} err]
        if {$rt != 0} {
            tk_messageBox -icon error -type ok -title "Error" \
                -message "Cannot open $query_file: $err"
            return -1
        }
        gets $f line
        set query_id [string trim $line]
        while {[eof $f] != 1} {
            if {$query_id == "" || [string compare -length 1 $query_id "#"] == 0} {
                gets $f line
                set query_id [string trim $line]
                continue
            }
            break
        }
        switch -- $query_id \
            $analysis_tab {
                set rt [catch {${analysis_tab}::load_query_options $f $mainframe} err]
                if {$rt != 0} {
                    tk_messageBox -icon error -type ok -title "Error" \
                        -message "$err"
                    return -1
                }
                $notebook raise $analysis_tab
            } \
            $terules_tab {
                if {[string equal [$rules_nb raise] $ApolTop::terules_tab]} {
                    set rt [catch {${ApolTop::terules_tab}::load_query_options $f $mainframe} err]
                    if {$rt != 0} {
                        tk_messageBox -icon error -type ok -title "Error" \
                            -message "$err"
                        return -1
                    }
                    $notebook raise $rules_tab
                    $rules_nb raise $ApolTop::terules_tab
                }
            } \
            default {
                tk_messageBox -icon error -type ok -title "Error" \
                    -message "Invalid query ID."
            }
        ApolTop::set_Focus_to_Text [$notebook raise]
        ::close $f
    }
}
proc ApolTop::save_query_info {} {
    variable notebook
    variable rules_tab
    variable terules_tab
    variable analysis_tab
    variable rules_nb
    variable mainframe
    set raised_tab [$notebook raise]
    if {![string equal $raised_tab $analysis_tab] && ![string equal $raised_tab $rules_tab]} {
        tk_messageBox -icon error -type ok -title "Save Query Error" \
            -message "You cannot save a query from this tab! \
			You can only save from the Policy Rules->TE Rules tab and the Analysis tab."
        return -1
    }
    if {[string equal $raised_tab $rules_tab] && ![string equal [$rules_nb raise] $terules_tab]} {
        tk_messageBox -icon error -type ok -title "Save Query Error" \
            -message "You cannot save a query from this tab! \
			You can only save from the Policy Rules->TE Rules tab and the Analysis tab."
        return -1
    }
    set query_file ""
    set types {
        {"Query files"		{$ApolTop::query_file_ext}}
    }
    set query_file [tk_getSaveFile -title "Save Query As?" \
                        -defaultextension $ApolTop::query_file_ext \
                        -filetypes $types -parent $mainframe]
    if {$query_file != ""} {
        set rt [catch {set f [::open $query_file w+]} err]
        if {$rt != 0} {
            return -code error $err
        }
        switch -- $raised_tab \
            $analysis_tab {
                puts $f "$analysis_tab"
                set rt [catch {${analysis_tab}::save_query_options $f $query_file} err]
                if {$rt != 0} {
                    ::close $f
                    tk_messageBox -icon error -type ok -title "Save Query Error" \
                        -message "$err"
                    return -1
                }
            } \
            $rules_tab {
                if {[string equal [$rules_nb raise] $terules_tab]} {
                    puts $f "$terules_tab"
                    set rt [catch {${terules_tab}::save_query_options $f $query_file} err]
                    if {$rt != 0} {
                        ::close $f
                        tk_messageBox -icon error -type ok -title "Save Query Error" \
                            -message "$err"
                        return -1
                    }
                }
            } \
            default {
                ::close $f
                tk_messageBox -icon error -type ok -title "Save Query Error" \
                    -message "You cannot save a query from this tab!"
                return -1
            }
        ::close $f
    }
}
proc ApolTop::displayFindDialog {} {
    variable searchDlg
    variable searchDlg_entryBox
    if {[winfo exists $searchDlg]} {
        raise $searchDlg
        focus $searchDlg_entryBox
        $searchDlg_entryBox selection range 0 end
        return
    }
    Dialog $searchDlg -title "Find" -separator 0 -parent . \
        -side right -default 0 -cancel 1 -modal none -homogeneous 1
    set top_frame [frame [$searchDlg getframe].top]
    set bottom_frame [frame [$searchDlg getframe].bottom]
    pack $top_frame -expand 1 -fill both -padx 10 -pady 5
    pack $bottom_frame -expand 0 -fill both -padx 10 -pady 5
    set entry_label [label $top_frame.l -text "Find:" -anchor e]
    set searchDlg_entryBox [entry $top_frame.e -bg white \
                                -textvariable ApolTop::searchString -width 16]
    pack $entry_label -side left -expand 0 -padx 10
    pack $searchDlg_entryBox -side left -expand 1 -fill x
    set options_frame [frame $bottom_frame.opts]
    pack $options_frame -side left -padx 5
    set options_case [checkbutton $options_frame.case -text "Match case" \
                          -variable ApolTop::case_sensitive]
    set options_regex [checkbutton $options_frame.regex -text "Regular expression" \
                           -variable ApolTop::regExpr]
    pack $options_case -anchor w
    pack $options_regex -anchor w
    set dir_frame [TitleFrame $bottom_frame.dir -text Direction]
    pack $dir_frame -side left
    set dir_up [radiobutton [$dir_frame getframe].up -text Up \
                    -variable ApolTop::srch_Direction -value up]
    set dir_down [radiobutton [$dir_frame getframe].down -text Down \
                      -variable ApolTop::srch_Direction -value down]
    pack $dir_up $dir_down -side left
    $searchDlg add -text "Find Next" -command ApolTop::search
    $searchDlg add -text "Cancel" -command [list destroy $searchDlg]
    $searchDlg_entryBox selection range 0 end
    focus $searchDlg_entryBox
    $searchDlg draw
    wm resizable $searchDlg 0 0
}
proc ApolTop::goto_line { line_num textBox } {
    if {[string is integer -strict $line_num] != 1} {
        tk_messageBox -icon error \
            -type ok  \
            -title "Invalid Line Number" \
            -message "$line_num is not a valid line number."
    } else {
	$textBox tag remove sel 0.0 end
	$textBox mark set insert ${line_num}.0
	$textBox see ${line_num}.0
	$textBox tag add sel $line_num.0 $line_num.end
	focus $textBox
    }
}
proc ApolTop::goto {dialog} {
    variable goto_line_num
    variable notebook
    variable components_nb
    variable rules_nb
    variable components_tab
    variable rules_tab
    variable policy_conf_tab
    variable analysis_tab
    variable file_contexts_tab
    if {[string is integer -strict $goto_line_num] != 1} {
        tk_messageBox -icon error -type ok -parent $dialog \
            -title "Invalid Line Number" \
            -message "$goto_line_num is not a valid line number."
    } else {
	set raised_tab [$notebook raise]
	switch -- $raised_tab \
            $policy_conf_tab {
                ${policy_conf_tab}::goto_line $goto_line_num
            } \
            $analysis_tab {
                ${analysis_tab}::goto_line $goto_line_num
            } \
            $rules_tab {
                [$rules_nb raise]::goto_line $goto_line_num
            } \
            $components_tab {
                [$components_nb raise]::goto_line $goto_line_num
            } \
            $file_contexts_tab {
                ${file_contexts_tab}::goto_line $goto_line_num
            } \
            default {
                return -code error
            }
        destroy $dialog
    }
}
proc ApolTop::displayGotoDialog {} {
    variable goto_Dialog
    variable gotoDlg_entryBox
    if {[winfo exists $goto_Dialog]} {
        raise $goto_Dialog
        focus $gotoDlg_entryBox
        $gotoDlg_entryBox selection range 0 end
        return
    }
    Dialog $goto_Dialog -title "Goto Line" -separator 0 -parent . \
        -default 0 -cancel 1 -modal none -homogeneous 1
    set top_frame [$goto_Dialog getframe]
    set entry_label [label $top_frame.l -text "Goto Line:" -anchor e]
    set gotoDlg_entryBox [entry $top_frame.e -bg white \
                              -textvariable ApolTop::goto_line_num -width 10]
    pack $entry_label -side left -padx 5 -pady 5
    pack $gotoDlg_entryBox -side left -padx 5 -pady 5 -expand 1 -fill x
    $goto_Dialog add -text "OK" -command [list ApolTop::goto $goto_Dialog]
    $goto_Dialog add -text "Cancel" -command [list destroy $goto_Dialog]
    $gotoDlg_entryBox selection range 0 end
    focus $gotoDlg_entryBox
    $goto_Dialog draw
    wm resizable $goto_Dialog 0 0
}
proc ApolTop::check_libsefs {} {
    set ApolTop::libsefs [apol_IsLibsefs_BuiltIn]
}
proc ApolTop::create { } {
    variable notebook
    variable mainframe
    variable components_nb
    variable rules_nb
    set descmenu {
	"&File" {} file 0 {
	    {command "&Open..." {} "Open a new policy" {Ctrl o} -command ApolTop::openPolicy}
	    {command "&Close" {} "Close current polocy" {Ctrl w} -command ApolTop::closePolicy}
	    {separator}
	    {command "&Quit" {} "Quit policy analysis tool" {Ctrl q} -command ApolTop::apolExit}
	    {separator}
	    {cascade "&Recent Files" {} recent 0 {}}
	}
	"&Search" {} search 0 {
	    {command "&Find..." {Disable_SearchMenu_Tag} "Find text in current buffer" {Ctrl f} -command ApolTop::displayFindDialog}
	    {command "&Goto Line..." {Disable_SearchMenu_Tag} "Goto a line in current buffer" {Ctrl g} -command ApolTop::displayGotoDialog}
	}
	"&Query" {} query 0 {
	    {command "&Load Query..." {Disable_LoadQuery_Tag} "Load query criteria " {} -command ApolTop::load_query_info}
	    {command "&Save Query..." {Disable_SaveQuery_Tag} "Save current query criteria" {} -command ApolTop::save_query_info}
	    {separator}
	    {command "&Policy Summary" {Disable_Summary} "Display summary statistics" {} -command ApolTop::popupPolicyStats}
	}
	"&Advanced" all options 0 {
	    {cascade "&Permission Mappings" {Perm_Map_Tag} pmap_menu 0 {
                {command "Load Default Perm Map" {} "Load the default permission map" {} -command ApolTop::load_default_perm_map_Dlg}
                {command "Load Perm Map from File" {} "Load a permission map from a file" {} -command ApolTop::load_perm_map_fileDlg}
                {separator}
                {command "Edit Perm Map... (Not loaded)" {} "Edit currently loaded permission map" {} -command Apol_Perms_Map::editPermMappings}
            }}
        }
	"&Help" {} helpmenu 0 {
	    {command "&General Help" {} "Show help on using apol" {} -command {ApolTop::helpDlg Help apol_help.txt}}
	    {command "&Domain Transition Analysis" {} "Show help on domain transitions" {} -command {ApolTop::helpDlg "Domain Transition Analysis Help" domaintrans_help.txt}}
	    {command "&Information Flow Analysis" {} "Show help on information flows" {} -command {ApolTop::helpDlg "Information Flow Analysis Help" infoflow_help.txt}}
	    {command "Direct &Relabel Analysis" {} "Show help on file relabeling" {} -command {ApolTop::helpDlg "Relabel Analysis Help" file_relabel_help.txt}}
	    {command "&Types Relationship Summary Analysis" {} "Show help on types relationships" {} -command {ApolTop::helpDlg "Types Relationship Summary Analysis Help" types_relation_help.txt}}
	    {separator}
	    {command "&About apol" {} "Show copyright information" {} -command ApolTop::aboutBox}
	}
    }
    set mainframe [MainFrame .mainframe -menu $descmenu -textvariable ApolTop::status]
    $mainframe addindicator -textvariable ApolTop::policyConf_lineno -width 14
    $mainframe addindicator -textvariable ApolTop::policy_stats_summary -width 88
    $mainframe addindicator -textvariable ApolTop::policy_version_string -width 28
    $ApolTop::mainframe setmenustate Disable_SearchMenu_Tag disabled
    $ApolTop::mainframe setmenustate Perm_Map_Tag disabled
    $ApolTop::mainframe setmenustate Disable_SaveQuery_Tag disabled
    $ApolTop::mainframe setmenustate Disable_LoadQuery_Tag disabled
    $ApolTop::mainframe setmenustate Disable_Summary disabled
    set frame    [$mainframe getframe]
    set notebook [NoteBook $frame.nb]
    set components_frame [$notebook insert end $ApolTop::components_tab -text "Policy Components"]
    set rules_frame [$notebook insert end $ApolTop::rules_tab -text "Policy Rules"]
    if {$ApolTop::libsefs == 1} {
        Apol_File_Contexts::create $notebook
    }
    Apol_Analysis::create $notebook
    Apol_PolicyConf::create $notebook
    set components_nb [NoteBook $components_frame.components_nb]
    set rules_nb [NoteBook $rules_frame.rules_nb]
    variable mls_tabs
    Apol_Types::create $components_nb
    Apol_Class_Perms::create $components_nb
    Apol_Roles::create $components_nb
    Apol_Users::create $components_nb
    Apol_Cond_Bools::create $components_nb
    Apol_MLS::create $components_nb
    lappend mls_tabs [list $components_nb [$components_nb pages end]]
    Apol_Initial_SIDS::create $components_nb
    Apol_NetContexts::create $components_nb
    Apol_FSContexts::create $components_nb
    Apol_TE::create $rules_nb
    Apol_Cond_Rules::create $rules_nb
    Apol_RBAC::create $rules_nb
    Apol_Range::create $rules_nb
    lappend mls_tabs [list $rules_nb [$rules_nb pages end]]
    $components_nb compute_size
    pack $components_nb -fill both -expand yes -padx 4 -pady 4
    $components_nb raise [$components_nb page 0]
    $components_nb bindtabs <Button-1> { ApolTop::set_Focus_to_Text }
    $rules_nb compute_size
    pack $rules_nb -fill both -expand yes -padx 4 -pady 4
    $rules_nb raise [$rules_nb page 0]
    $rules_nb bindtabs <Button-1> { ApolTop::set_Focus_to_Text }
    $notebook compute_size
    pack $notebook -fill both -expand yes -padx 4 -pady 4
    $notebook raise [$notebook page 0]
    $notebook bindtabs <Button-1> { ApolTop::set_Focus_to_Text }
    pack $mainframe -fill both -expand yes
    return 0
}
proc ApolTop::writeInitFile { } {
    variable dot_apol_file
    variable recent_files
    variable text_font
    variable title_font
    variable dialog_font
    variable general_font
    if {[catch {open $dot_apol_file w+} f]} {
        tk_messageBox -icon error -type ok -title "Error" \
            -message "Could not open $dot_apol_file for writing: $f"
        return
    }
    puts $f "recent_files"
    puts $f [llength $recent_files]
    foreach recent $recent_files {
        puts $f $recent
    }
    puts $f "\n"
    puts $f "# Font format: family ?size? ?style? ?style ...?"
    puts $f "# Possible values for the style arguments are as follows:"
    puts $f "# normal bold roman italic underline overstrike\n#\n#"
    puts $f "# NOTE: When configuring fonts, remember to remove the following "
    puts $f "# \[window height\] and \[window width\] entries before starting apol. "
    puts $f "# Not doing this may cause widgets to be obscured when running apol."
    puts $f "\[general_font\]"
    if {$general_font == {}} {
        puts $f "Helvetica 10"
    } else {
        puts $f "$general_font"
    }
    puts $f "\[title_font\]"
    if {$title_font == {}} {
        puts $f "Helvetica 10 bold italic"
    } else {
        puts $f "$title_font"
    }
    puts $f "\[dialog_font\]"
    if {$dialog_font == {}} {
        puts $f "Helvetica 10"
    } else {
        puts $f "$dialog_font"
    }
    puts $f "\[text_font\]"
    if {$text_font == {}} {
        puts $f "fixed"
    } else {
        puts $f "$text_font"
    }
    puts $f "\[window_height\]"
    puts $f [winfo height .]
    puts $f "\[window_width\]"
    puts $f [winfo width .]
    puts $f "\[show_fake_attrib_warning\]"
    variable show_fake_attrib_warning
    puts $f $show_fake_attrib_warning
    puts $f "\[max_recent_files\]"
    variable max_recent_files
    puts $f $max_recent_files
    close $f
}
proc ApolTop::readInitFile { } {
    variable dot_apol_file
    variable recent_files
    if {![file exists $dot_apol_file]} {
        return
    }
    if {[catch {open $dot_apol_file r} f]} {
        tk_messageBox -icon error -type ok -title "Error opening configuration file" \
            -message "Cannot open $dot_apol_file: $f"
        return
    }
    while {![eof $f]} {
        set option [string trim [gets $f]]
        if {$option == {} || [string compare -length 1 $option "\#"] == 0} {
            continue
        }
        set value [string trim [gets $f]]
        if {[eof $f]} {
            puts "EOF reached while reading $option"
            break
        }
        if {$value == {}} {
            puts "Empty value for option $option"
            continue
        }
        switch -- $option {
            "\[window_height\]" {
                if {[string is integer -strict $value] != 1} {
                    puts "window_height was not given as an integer and is ignored"
                    break
                }
                variable top_height $value
            }
            "\[window_width\]" {
                if {[string is integer -strict $value] != 1} {
                    puts "window_width was not given as an integer and is ignored"
                    break
                }
                variable top_width $value
            }
            "\[title_font\]" {
                variable title_font $value
            }
            "\[dialog_font\]" {
                variable dialog_font $value
            }
            "\[text_font\]" {
                variable text_font $value
            }
            "\[general_font\]" {
                variable general_font $value
            }
            "\[show_fake_attrib_warning\]" {
                variable show_fake_attrib_warning $value
            }
            "\[max_recent_files\]" {
                if {[string is integer -strict $value] != 1} {
                    puts "max_recent_files was not given as an integer and is ignored"
                } else {
                    if {$value < 2} {
                        variable max_recent_files 2
                    } else {
                        variable max_recent_files $value
                    }
                }
            }
            "recent_files" {
                if {[string is integer -strict $value] != 1} {
                    puts "number of recent files was not given as an integer and is ignored"
                    continue;
                } elseif {$value < 0} {
                    puts "number of recent was less than 0 and is ignored"
                    continue
                }
                while {$value > 0} {
                    incr value -1
                    set line [gets $f]
                    if {[eof $f]} {
                        puts "EOF reached trying to read recent files."
                        break
                    }
                    if {[llength $line] == 1} {
                        set line [list monolithic $line {}]
                    }
                    lappend recent_files $line
                }
            }
        }
    }
    close $f
}
proc ApolTop::addRecent {path} {
    variable recent_files
    variable max_recent_files
    if {[lsearch $recent_files $path] >= 0} {
        return
    }
    set recent_files [lrange [concat [list $path] $recent_files] 0 [expr {$max_recent_files - 1}]]
    buildRecentFilesMenu
}
proc ApolTop::buildRecentFilesMenu {} {
    variable mainframe
    variable recent_files
    variable max_recent_files
    set recent_menu [$mainframe getmenu recent]
    $recent_menu delete 0 $max_recent_files
    foreach r $recent_files {
        foreach {path_type primary_file modules} $r {break}
        if {$path_type == "monolithic"} {
            set label $primary_file
        } else {
            set label "$primary_file + [llength $modules] module"
            if {[llength $modules] != 1} {
                append label "s"
            }
        }
        $recent_menu add command -label $label \
            -command [list ApolTop::openPolicyFile $r]
    }
}
proc ApolTop::helpDlg {title file_name} {
    set help_dir [apol_GetHelpDir "$file_name"]
    set helpfile [file join $help_dir $file_name]
    if {[catch {open $helpfile} f]} {
        set info $f
    } else {
        set info [read $f]
        close $f
    }
    Apol_Widget::showPopupParagraph $title $info
}
proc ApolTop::setBusyCursor {} {
    variable prevCursor
    set prevCursor [. cget -cursor]
    . configure -cursor watch
}
proc ApolTop::resetBusyCursor {} {
    variable prevCursor
    . configure -cursor $prevCursor
}
proc ApolTop::popupPolicyStats {} {
    variable polstats
    set classes $polstats(classes)
    set common_perms $polstats(common_perms)
    set perms $polstats(perms)
    if {![regexp -- {^([^\(]+) \(([^,]+), ([^\)]+)} $ApolTop::policy_version_string -> policy_version policy_type policy_mls_type]} {
        set policy_version $ApolTop::policy_version_string
        set policy_type "unknown"
        set policy_mls_type "unknown"
    }
    set policy_version [string trim $policy_version]
    destroy .polstatsbox
    set dialog [Dialog .polstatsbox -separator 1 -title "Policy Summary" \
                    -modal none -parent .]
    $dialog add -text Close -command [list destroy $dialog]
    set w [$dialog getframe]
    label $w.title -text "Policy Summary Statistics"
    set f [frame $w.summary]
    label $f.l -justify left -text "    Policy Version:\n    Policy Type:\n    MLS Status:"
    label $f.r -justify left -text "$policy_version\n$policy_type\n$policy_mls_type"
    grid $f.l $f.r -sticky w
    grid configure $f.r -padx 30
    grid $w.title - -sticky w -padx 8
    grid $f - -sticky w -padx 8
    grid [Separator $w.sep] - -sticky ew -pady 5
    set f [frame $w.left]
    set i 0
    foreach {title block} {
        "Number of Classes and Permissions" {
            "Object Classes" classes
            "Common Perms" common_perms
            "Permissions" perms
        }
        "Number of Types and Attributes" {
            "Types" types
            "Attributes" attribs
        }
        "Number of Type Enforcement Rules" {
            "allow" teallow
            "neverallow" neverallow
            "auditallow" auditallow
            "dontaudit" dontaudit
            "type_transition" tetrans
            "type_member" temember
            "type_change" techange
        }
        "Number of Roles" {
            "Roles" roles
        }
        "Number of RBAC Rules" {
            "allow" roleallow
            "role_transition" roletrans
        }
    } {
        set ltext "$title:"
        set rtext {}
        foreach {l r} $block {
            append ltext "\n    $l:"
            append rtext "\n$polstats($r)"
        }
        label $f.l$i -justify left -text $ltext
        label $f.r$i -justify left -text $rtext
        grid $f.l$i $f.r$i -sticky w -padx 4 -pady 2
        incr i
    }
    set i 0
    set g [frame $w.right]
    foreach {title block} {
        "Number of Users" {
            "Users" users
        }
        "Number of Booleans" {
            "Bools" cond_bools
        }
        "Number of MLS Components" {
            "Sensitivities" sens
            "Categories" cats
        }
        "Number of MLS Rules" {
            "range_transition" rangetrans
        }
        "Number of Initial SIDs" {
            "SIDs" sids
        }
        "Number of OContexts" {
            "PortCons" portcons
            "NetIfCons" netifcons
            "NodeCons" nodecons
            "GenFSCons" genfscons
            "fs_use statements" fs_uses
        }
    } {
        set ltext "$title:"
        set rtext {}
        foreach {l r} $block {
            append ltext "\n    $l:"
            append rtext "\n$polstats($r)"
        }
        label $g.l$i -justify left -text $ltext
        label $g.r$i -justify left -text $rtext
        grid $g.l$i $g.r$i -sticky w -padx 4 -pady 2
        incr i
    }
    grid $f $g -sticky nw -padx 4
    $dialog draw
}
proc ApolTop::showPolicyStats {} {
    variable polstats
    variable policy_stats_summary
    if {[catch {apol_GetStats} pstats]} {
        tk_messageBox -icon error -type ok -title "Error" -message $pstats
        return
    }
    array unset polstats
    array set polstats $pstats
    set policy_stats_summary ""
    append policy_stats_summary "Classes: $polstats(classes)   "
    append policy_stats_summary "Perms: $polstats(perms)   "
    append policy_stats_summary "Types: $polstats(types)   "
    append policy_stats_summary "Attribs: $polstats(attribs)   "
    set num_te_rules [expr {$polstats(teallow) + $polstats(neverallow) +
                            $polstats(auditallow) + $polstats(dontaudit) +
                            $polstats(tetrans) + $polstats(temember) +
                            $polstats(techange)}]
    append policy_stats_summary "TE rules: $num_te_rules   "
    append policy_stats_summary "Roles: $polstats(roles)   "
    append policy_stats_summary "Users: $polstats(users)"
}
proc ApolTop::aboutBox {} {
    if {[winfo exists .apol_about]} {
        raise .apol_about
    } else {
        variable gui_ver
        variable copyright_date
        variable apol_icon
        Dialog .apol_about -cancel 0 -default 0 -image $apol_icon \
            -modal none -parent . -separator 1 -title "About apol"
        set f [.apol_about getframe]
        set l1 [label $f.l1 -text "apol $gui_ver" -height 2]
        foreach {name size} [$l1 cget -font] {break}
        incr size 6
        $l1 configure -font [list $name $size bold]
        set l2 [label $f.l2 -text "Security Policy Analysis Tool for Security Enhanced Linux\nCopyright (c) $copyright_date Tresys Technology, LLC\nhttp://oss.tresys.com/projects/setools"]
        pack $l1 $l2
        .apol_about add -text "Close" -command [list destroy .apol_about]
        .apol_about draw
    }
}
proc ApolTop::closePolicy {} {
    variable policy_version_string {}
    variable policy_is_open
    variable policy_stats_summary {}
    wm title . "SELinux Policy Analysis"
    variable tab_names
    foreach tab $tab_names {
        Apol_${tab}::close
    }
    Apol_Perms_Map::close
    ApolTop::set_Focus_to_Text [$ApolTop::notebook raise]
    if {[catch {apol_ClosePolicy} err]} {
        tk_messageBox -icon error -type ok -title "Error closing policy" \
            -message "There was an error closing the policy: $err."
    }
    set policy_is_open 0
    $ApolTop::mainframe setmenustate Disable_SearchMenu_Tag disabled
    $ApolTop::mainframe setmenustate Perm_Map_Tag disabled
    $ApolTop::mainframe setmenustate Disable_SaveQuery_Tag disabled
    $ApolTop::mainframe setmenustate Disable_LoadQuery_Tag disabled
    $ApolTop::mainframe setmenustate Disable_Summary disabled
    ApolTop::enable_source_policy_tab
    ApolTop::enable_disable_conditional_widgets 1
    set_mls_tabs_state normal
    ApolTop::configure_edit_pmap_menu_item 0
}
proc ApolTop::open_apol_tabs {policy_path} {
    variable tab_names
    foreach tab $tab_names {
        if {$tab == "PolicyConf"} {
            Apol_PolicyConf::open $policy_path
        } else {
            Apol_${tab}::open
        }
    }
}
proc ApolTop::enable_disable_conditional_widgets {enable} {
    set tab [$ApolTop::notebook raise]
    switch -exact -- $tab \
        $ApolTop::components_tab {
            if {[$ApolTop::components_nb raise] == $ApolTop::cond_bools_tab} {
                if {$enable} {
                    $ApolTop::components_nb raise $ApolTop::cond_bools_tab
                } else {
                    set name [$ApolTop::components_nb pages 0]
                    $ApolTop::components_nb raise $name
                }
            }
        } \
        $ApolTop::rules_tab {
            if {[$ApolTop::rules_nb raise] == $ApolTop::cond_rules_tab} {
                if {$enable} {
                    $ApolTop::rules_nb raise $ApolTop::cond_rules_tab
                } else {
                    set name [$ApolTop::rules_nb pages 0]
                    $ApolTop::rules_nb raise $name
                }
            }
        } \
        default {
        }
    if {$enable} {
        $ApolTop::components_nb itemconfigure $ApolTop::cond_bools_tab -state normal
        $ApolTop::rules_nb itemconfigure $ApolTop::cond_rules_tab -state normal
    } else {
        $ApolTop::components_nb itemconfigure $ApolTop::cond_bools_tab -state disabled
        $ApolTop::rules_nb itemconfigure $ApolTop::cond_rules_tab -state disabled
    }
}
proc ApolTop::enable_source_policy_tab {} {
    $ApolTop::notebook itemconfigure $ApolTop::policy_conf_tab -state normal
}
proc ApolTop::disable_source_policy_tab {} {
    if {[$ApolTop::notebook raise] == $ApolTop::policy_conf_tab} {
        set name [$ApolTop::notebook pages 0]
        $ApolTop::notebook raise $name
    }
    $ApolTop::notebook itemconfigure $ApolTop::policy_conf_tab -state disabled
}
proc ApolTop::set_mls_tabs_state {new_state} {
    variable mls_tabs
    foreach tab $mls_tabs {
        foreach {notebook page} $tab break
        set current_tab [$notebook raise]
        $notebook itemconfigure $page -state $new_state
        if {$current_tab == $page && $new_state == "disabled"} {
            $notebook raise [$notebook pages 0]
        }
    }
}
proc ApolTop::set_initial_open_policy_state {} {
    if {![ApolTop::is_capable "conditionals"]} {
        ApolTop::enable_disable_conditional_widgets 0
    }
    if {![ApolTop::is_capable "source"]} {
        ApolTop::disable_source_policy_tab
    }
    if {![ApolTop::is_capable "mls"]} {
        set_mls_tabs_state disabled
    }
    if {![ApolTop::is_capable "attribute names"] && \
            [llength $::Apol_Types::attriblist] > 0 && \
            $ApolTop::show_fake_attrib_warning} {
        set d [Dialog .fake_attribute_dialog -modal local -parent . \
                   -title "Warning - Attribute Names" -separator 1]
        $d add -text "OK"
        set f [$d getframe]
        label $f.l -text "Warning: Apol has generated attribute names because\nthe original names were not preserved in the policy." -justify left
        checkbutton $f.cb -text "Show this message again next time." \
            -variable ApolTop::show_fake_attrib_warning
        pack $f.l $f.cb -padx 10 -pady 10
        $d draw
        destroy $d
    }
    ApolTop::set_Focus_to_Text [$ApolTop::notebook raise]
    $ApolTop::mainframe setmenustate Perm_Map_Tag normal
    $ApolTop::mainframe setmenustate Disable_Summary normal
    $ApolTop::mainframe setmenustate Disable_SearchMenu_Tag normal
}
proc ApolTop::openPolicyFile {path} {
    variable policy_version_string
    variable policy_is_open
    ApolTop::closePolicy
    set policy_is_open 0
    set primary_file [lindex $path 1]
    variable openDialogText "$primary_file:\n    Opening policy."
    variable openDialogVal -1
    if {[set dialog_width [string length $primary_file]] < 32} {
        set dialog_width 32
    }
    ProgressDlg .apol_policy_open -title "Open Policy" \
        -type normal -stop {} -separator 1 -parent . -maximum 2 \
        -width $dialog_width -textvariable ApolTop::openDialogText \
        -variable ApolTop::openDialogVal
    set orig_Cursor [. cget -cursor]
    . configure -cursor watch
    update idletasks
    after idle ApolTop::doOpenIdle
    set retval [catch {apol_OpenPolicy $path} err]
    . configure -cursor $orig_Cursor
    destroy .apol_policy_open
    if {$retval} {
        tk_messageBox -icon error -type ok -title "Open Policy" \
            -message "The selected file does not appear to be a valid SELinux Policy.\n\n$err"
        return -1
    }
    if {[catch {apol_GetPolicyVersionString} policy_version_string]} {
        tk_messageBox -icon error -type ok -title "Open Policy" -message "Could not determine policy version:\n$policy_version_string"
        return -1
    }
    ApolTop::showPolicyStats
    set policy_is_open 1
    if {[catch {open_apol_tabs $path} err]} {
        set policy_is_open 0
        tk_messageBox -icon error -type ok -title "Open Policy" -message $err
        return -1
    }
    if {[catch {set_initial_open_policy_state} err]} {
        set policy_is_open 0
        tk_messageBox -icon error -type ok -title "Open Policy" -message $err
        return -1
    }
    addRecent $path
    variable last_policy_path $path
    wm title . "SELinux Policy Analysis - $primary_file"
    return 0
}
proc ApolTop::doOpenIdle {} {
    variable openDialogText
    if {[set infoString [apol_GetInfoString]] != {}} {
        set openDialogText [lindex [split $openDialogText "\n"] 0]
        append openDialogText "\n    $infoString"
        update idletasks
        after idle ApolTop::doOpenIdle
    }
}
proc ApolTop::openPolicy {} {
    variable last_policy_path
    Apol_Open_Policy_Dialog::getPolicyPath $last_policy_path
}
proc ApolTop::apolExit { } {
    variable policy_is_open
    if {$policy_is_open} {
        ApolTop::closePolicy
    }
    if {$ApolTop::libsefs == 1} {
        Apol_File_Contexts::close
    }
    ApolTop::writeInitFile
    exit
}
proc ApolTop::load_fonts { } {
    variable title_font
    variable dialog_font
    variable general_font
    variable text_font
    tk scaling -displayof . 1.0
    if {$general_font == ""} {
        set general_font "Helvetica 10"
    }
    option add *Font $general_font
    if {$title_font == {}} {
        set title_font "Helvetica 10 bold italic"
    }
    option add *TitleFrame.l.font $title_font
    if {$dialog_font == {}} {
        set dialog_font "Helvetica 10"
    }
    option add *Dialog*font $dialog_font
    option add *Dialog*TitleFrame.l.font $title_font
    if {$text_font == ""} {
        set text_font "fixed"
    }
    option add *text*font $text_font
}
proc ApolTop::main {} {
    variable top_width
    variable top_height
    variable notebook
    tcl_config_init
    rename send {}
    if {[catch {package require BWidget}]} {
        tk_messageBox -icon error -type ok -title "Missing BWidget package" -message \
            "Missing BWidget package.  Ensure that your installed version of Tcl/Tk includes BWidget, which can be found at http://sourceforge.net/projects/tcllib."
        exit -1
    }
    set rt [catch {package require apol} err]
    if {$rt != 0 } {
        tk_messageBox -icon error -type ok -title "Missing SELinux package" -message \
            "Missing the SELinux package.  This script will not work correctly using the generic TK wish program.  You must either use the apol executable or the awish	interpreter."
        exit -1
    }
    wm withdraw .
    wm title . "SELinux Policy Analysis"
    wm protocol . WM_DELETE_WINDOW ApolTop::apolExit
    set rt [catch {ApolTop::check_libsefs} err]
    if {$rt != 0} {
        tk_messageBox -icon error -type ok -title "Error" -message "$err"
        return
    }
    catch {tcl_patch_bwidget}
    ApolTop::load_fonts
    ApolTop::readInitFile
    ApolTop::create
    bind . <Button-1> {focus %W}
    bind . <Button-2> {focus %W}
    bind . <Button-3> {focus %W}
    ApolTop::buildRecentFilesMenu
    set icon_file [file join [apol_GetHelpDir apol.gif] apol.gif]
    if {![catch {image create photo -file $icon_file} icon]} {
        wm iconphoto . -default $icon
    }
    variable apol_icon $icon
    set ApolTop::top_width [$notebook cget -width]
    set ApolTop::top_height [$notebook cget -height]
    wm geom . ${top_width}x${top_height}
    wm deiconify .
    raise .
    focus .
}
ApolTop::main
