/*
** PHREL
** $Id: thread.c,v 1.25 2006/04/14 05:12:28 sella Exp $
** Copyright (c) 2005,2006 James M. Sella. All Rights Reserved.
** Released under the GPL Version 2 License.
** http://www.digitalgenesis.com
*/

static const char rcsid[] = "$Id: thread.c,v 1.25 2006/04/14 05:12:28 sella Exp $";

#include "thread.h"

#include "trap.h"
#include "util.h"
#include "phrel.h"
#include "prefs.h"
#include "data_hash.h"

#include <snmptrap.h>

#include <sched.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <signal.h>
#include <pthread.h>
#include <sys/wait.h>
#include <sys/types.h>

/* Global variables */
extern struct args_t args;
extern int running, restart;
extern pid_t pid;
extern time_t startup_time;
extern pthread_mutex_t data_mutex;
extern struct hash_key_t hash_key[HASH_WIDTH];

void data_thread() {
	char buf[1024], buf2[32];
	time_t next, stat_next = 0;
	unsigned int i, pps, count = 0;
	struct data_t *d_ptr;
	struct hash_entry_t **entry_h = NULL;
	FILE *fd = NULL;

	/* Insert sync chain */
	chain_sync_add();

	while (running) {
		next = time(NULL) + args.check_interval;
		while (running && next > time(NULL)) {
			usleep(750000);
		}

		/* Verify sync chain exists. */
		if (chain_sync_check() == 0) {
			syslog(LOG_INFO, "out of sync with iptables - restarting");

			snmptrap_sync_lost();

			running = 0;
			restart = 1;

			continue;
		} else if (args.debug >= DEBUG_INTERNAL) {
			syslog(LOG_INFO, "in sync with iptables");
		}

		/* Open stats file, if needed. */
		if (args.stats != NULL && time(NULL) > stat_next) {
			stat_next = time(NULL) + args.stats_interval;
			if (args.debug >= DEBUG_INTERNAL) {
				syslog(LOG_INFO, "opening stats file %s", args.stats);
			}

			if ((fd = fopen(args.stats, "w")) == NULL) {
				syslog(LOG_INFO, "failed to open stats file %s - disabling stats", args.stats);

				/* Disable stats file. */
				free(args.stats);
				args.stats = NULL;
			} else if (hash_stats(buf, sizeof(buf)) == 0) {
				if (fwrite(buf, 1, strlen(buf), fd) == 0) {
					syslog(LOG_INFO, "failed write to stats file %s - closing file", args.stats);
					/* Write failed, close stats file. */
					fclose(fd);
					fd = NULL;
				}
			}
		}

		if (args.debug >= DEBUG_INTERNAL) {
			syslog(LOG_INFO, "starting to walk hash...");
		}

		for (i = 0; i < HASH_WIDTH && running; i++) {
			pthread_mutex_lock(&data_mutex);

			if ((entry_h = &hash_key[i].entry) != NULL) {
				while (*entry_h != NULL) {
					d_ptr = (*entry_h)->data;

					pps = calc_pps(d_ptr);
					process_threshold(d_ptr, pps);

					if (args.debug >= DEBUG_DUMP2) {
						inet_ntop(AF_INET, &(d_ptr->ip), buf2, sizeof(buf2));
						syslog(LOG_INFO, "checking if %s is stale", buf2);
					}

					if (fd != NULL) {
						inet_ntop(AF_INET, &(d_ptr->ip), buf2, sizeof(buf2));
						snprintf(buf, sizeof(buf), "%s\t%8u pps %8u threshold %8u rate\n", buf2, pps, d_ptr->threshold, d_ptr->rate);
						fwrite(buf, 1, strlen(buf), fd);
					}

					/* Move to next item. */
					entry_h = &(*entry_h)->next;

					if (check_stale(d_ptr) == 1) {
						if (args.debug >= DEBUG_DUMP1) {
							inet_ntop(AF_INET, &(d_ptr->ip), buf2, sizeof(buf2));
							syslog(LOG_INFO, "removing %s from hash (stale)", buf2);
						}

						hash_del(&(d_ptr->ip)); /* Invalidates (*entry_h). */

						/* Reset (*entry_h), since hash_del invalidates the pointer. */
						entry_h = &hash_key[i].entry;
					}
				}
			}

			pthread_mutex_unlock(&data_mutex);

			/* Don't peg CPU. Loop doesn't need to be completed super quickly. */
			if ((count++ % (HASH_WIDTH / 7)) == 0) { /* 7 times per cycle. */
#ifdef _POSIX_PRIORITY_SCHEDULING
	   		sched_yield();
#else
				usleep(0);
#endif
			}
		}

		if (args.debug >= DEBUG_INTERNAL) {
			syslog(LOG_INFO, "completed walking hash");
		}

		/* Close stats file. */
		if (fd != NULL) {
			if (args.debug >= DEBUG_INTERNAL) {
				syslog(LOG_INFO, "closing stats file %s", args.stats);
			}

			fclose(fd);
			fd = NULL;
		}
	}

	/* Delete sync chain */
	chain_sync_del();

	if (args.debug >= DEBUG_INTERNAL) {
		syslog(LOG_INFO, "existing thread");
	}

	pthread_exit(NULL);
}

void cleanup() {
	int i;
	struct data_t *d_ptr;
	struct hash_entry_t **entry_h = NULL;

	/* Cleanup for shutdown. Stop filtering. */
	if (args.debug >= DEBUG_INTERNAL) {
		syslog(LOG_INFO, "cleaning up iptables...");
	}

	pthread_mutex_lock(&data_mutex);
	for (i = 0; i < HASH_WIDTH; i++) {
		if ((entry_h = &hash_key[i].entry) != NULL) {
			while (*entry_h != NULL) {
				d_ptr = (*entry_h)->data;

				if (d_ptr->threshold > 0) {
					chain_del(d_ptr);
				}

				entry_h = &(*entry_h)->next;
			}
		}

	}
	pthread_mutex_unlock(&data_mutex);

	if (args.debug >= DEBUG_INTERNAL) {
		syslog(LOG_INFO, "completed cleanup");
	}
}

int check_stale(struct data_t *data) {
	time_t cur;
	unsigned int i;

	/* If there is a threshold set, its not stale. */
	if (data->threshold > 0) {
		return 0;
	}

	cur = time(NULL);
	for (i = 0; i < args.max_samples; i++) {
		/* If recently updated, don't remove. */
		if (data->last_update[i] + args.stale_interval >= cur && data->last_update[i] != 0) {
			return 0;
		}
	}

	return 1;
}

void process_threshold(struct data_t* data, unsigned int pps) {
	char buf[32], buf2[32];
	struct threshold_t *t;
	time_t cur = time(NULL);
	unsigned int threshold, prev_threshold;
	unsigned int rate, prev_rate;
	unsigned int burst, prev_burst;
	unsigned int decay, prev_decay;

	/* Capture previous threshold, rate, burst and decay. */
	prev_threshold = data->threshold;
	prev_rate = data->rate;
	prev_burst = data->burst;
	prev_decay = data->decay;

	/* Clear threshold, rate and decay if decay has expired. */
	if (data->threshold > 0 && cur > data->decay) {
		data->threshold = 0;
		data->rate = 0;
		data->burst = 0;
		data->decay = 0;
	}

	/* Check for new threshold crossings. */
	t = args.threshold;
	while (t != NULL) {
		if (pps >= t->threshold && (t->threshold >= data->threshold || cur > data->decay)) {
			data->threshold = t->threshold;
			data->rate = (t->rate == USE_DEFAULT) ? args.rate : t->rate;
			data->burst = (t->burst == USE_DEFAULT) ? args.burst : t->burst;
			data->decay = (t->decay == USE_DEFAULT) ? cur + args.decay : cur + t->decay;
			if (args.debug >= DEBUG_DUMP2) {
				inet_ntop(AF_INET, &(data->ip), buf2, sizeof(buf2));
				syslog(LOG_INFO, "setting %s to %u (%ld - %ld) = %ld", buf2, data->threshold, data->decay, cur, data->decay - cur);
			}
		}
		t = t->next;
	}

	/* Handle threshold changes */
	if (prev_threshold != data->threshold) {
		if (data->threshold > 0) {
			snprintf(buf, sizeof(buf), "%u", data->rate);
		} else {
			strncpy(buf, "none", sizeof(buf));
		}

		inet_ntop(AF_INET, &(data->ip), buf2, sizeof(buf2));
		syslog(LOG_INFO, "changing threshold for %s from %u to %u (pps %u, rate %s, burst %u, decay %ld)", buf2, prev_threshold, data->threshold, pps, buf, data->burst, (data->decay > 0) ? data->decay - cur : 0);

		if (data->threshold > prev_threshold) {
			snmptrap_threshold_increase(data, pps, prev_threshold);
		} else {
			snmptrap_threshold_decrease(data, pps, prev_threshold);
		}

		if (data->threshold > 0) {
			if (chain_add(data) == 0) { /* Handle failed insert */
				syslog(LOG_INFO, "failed to insert chain for %s - removing partial insert", buf2);

				chain_del(data);

				/* Clear crossed threshold */
				data->threshold = 0;
				data->rate = 0;
				data->burst = 0;
				data->decay = 0;
			}
		}

		if (prev_threshold > 0) {
			/* FIXME: We could save/restore using a memcpy with a tmp data_t struct. */
			/* Capture new threshold and rate */
			threshold = data->threshold;
			rate = data->rate;
			burst = data->burst;
			decay = data->decay;

			/* Use previous and delete rule. */
			data->threshold = prev_threshold;
			data->rate = prev_rate;
			data->burst = prev_burst;
			data->decay = prev_decay;
			chain_del(data);

			/* Restore new threshold and rate. */
			data->threshold = threshold;
			data->rate = rate;
			data->burst = burst;
			data->decay = decay;
		}
	}
}

int chain_add(struct data_t* data) {
	int i, ret;
	char cmd[4][512], ip_buf[32], chain[MAX_CHAIN_LEN];

	inet_ntop(AF_INET, &(data->ip), ip_buf, sizeof(ip_buf));

	snprintf(chain, sizeof(chain), "%s-%u-%u", ip_buf, data->threshold, pid);

	syslog(LOG_INFO, "inserting chain %s", chain);

	/* Create new chain. */
	snprintf(cmd[0], sizeof(cmd[0]), "%s -N %s > /dev/null 2>&1", args.iptables, chain);

	/* Add rules to new chain. */
	if (data->rate > 0) {
		snprintf(cmd[1], sizeof(cmd[1]), "%s -A %s -m limit --limit %u/sec --limit-burst %u -j RETURN > /dev/null 2>&1", args.iptables, chain, data->rate, data->burst);
	} else {
		cmd[1][0] = '\0';
	}
	snprintf(cmd[2], sizeof(cmd[2]), "%s -A %s -j %s > /dev/null 2>&1", args.iptables, chain, args.jump);

	/* Insert new chain into INPUT */
	snprintf(cmd[3], sizeof(cmd[3]), "%s -I INPUT -j %s -s %s > /dev/null 2>&1", args.iptables, chain, ip_buf);

	/* Loop though commands running them. */
	for (i = 0; i < 4; i++) {
		/* Skip empty commands. */
		if (cmd[i][0] == '\0') {
			continue;
		}

		if (args.debug >= DEBUG_INTERNAL) {
			syslog(LOG_INFO, "exec: %s", cmd[i]);
		}

		/* Are we in test mode? */
		if (args.test == 0) {
			ret = system(cmd[i]); /* FIXME: Use exec() instead. */
		} else {
			ret = system(""); /* FIXME: Use exec() instead. */
		}

		/* Was the command successful? */
		if (WIFSIGNALED(ret) && (WTERMSIG(ret) == SIGINT || WTERMSIG(ret) == SIGQUIT)) {
			syslog(LOG_ERR, "failed to execute command: %s", cmd[i]);

			return 0;
		}

		if (WEXITSTATUS(ret) != 0) {
			syslog(LOG_ERR, "command not successful: %s", cmd[i]);

			return 0;
		}
	}

	/* Success */
	return 1;
}

int chain_del(struct data_t* data) {
	int i, ret;
	char cmd[3][512], ip_buf[32], chain[MAX_CHAIN_LEN];

	inet_ntop(AF_INET, &(data->ip), ip_buf, sizeof(ip_buf));

	snprintf(chain, sizeof(chain), "%s-%u-%u", ip_buf, data->threshold, pid);

	syslog(LOG_INFO, "deleting chain %s", chain);

	/* Delete chain from Input */
	snprintf(cmd[0], sizeof(cmd[0]), "%s -D INPUT -j %s -s %s > /dev/null 2>&1", args.iptables, chain, ip_buf);

	/* Flush Chain */
	snprintf(cmd[1], sizeof(cmd[1]), "%s -F %s > /dev/null 2>&1", args.iptables, chain);

	/* Delete Chain */
	snprintf(cmd[2], sizeof(cmd[2]), "%s -X %s > /dev/null 2>&1", args.iptables, chain);

	/* Loop though commands running them. */
	for (i = 0; i < 3; i++) {
		/* Skip empty commands. */
		if (cmd[i][0] == '\0') {
			continue;
		}

		if (args.debug >= DEBUG_INTERNAL) {
			syslog(LOG_INFO, "exec: %s", cmd[i]);
		}

		/* Are we in test mode? */
		if (args.test == 0) {
			ret = system(cmd[i]); /* FIXME: Use exec() instead. */
		} else {
			ret = system(""); /* FIXME: Use exec() instead. */
		}

		/* Was the command successful? */
		if (WIFSIGNALED(ret) && (WTERMSIG(ret) == SIGINT || WTERMSIG(ret) == SIGQUIT)) {
			syslog(LOG_ERR, "failed to execute command: %s", cmd[i]);

			return 0;
		}

		if (WEXITSTATUS(ret) != 0) {
			syslog(LOG_ERR, "command not successful: %s", cmd[i]);
		}
	}

	/* Success */
	return 1;
}

int chain_sync_add() {
	int ret;
	char cmd[512], chain[MAX_CHAIN_LEN];

	snprintf(chain, sizeof(chain), "%s-sync-check-%u", PROGRAM_NAME, pid);

	syslog(LOG_INFO, "inserting chain %s", chain);

	/* Create new chain. */
	snprintf(cmd, sizeof(cmd), "%s -N %s > /dev/null 2>&1", args.iptables, chain);

	if (args.debug >= DEBUG_INTERNAL) {
		syslog(LOG_INFO, "exec: %s", cmd);
	}

	/* Are we in test mode? */
	if (args.test == 0) {
		ret = system(cmd); /* FIXME: Use exec() instead. */
	} else {
		ret = system(""); /* FIXME: Use exec() instead. */
	}

	/* Was the command successful? */
	if (WIFSIGNALED(ret) && (WTERMSIG(ret) == SIGINT || WTERMSIG(ret) == SIGQUIT)) {
		syslog(LOG_ERR, "failed to execute command: %s", cmd);

		return 0;
	}

	if (WEXITSTATUS(ret) != 0) {
		syslog(LOG_ERR, "command not successful: %s", cmd);
	}

	/* Success */
	return 1;
}

int chain_sync_check() {
	int ret;
	char cmd[512], chain[MAX_CHAIN_LEN];

	snprintf(chain, sizeof(chain), "%s-sync-check-%u", PROGRAM_NAME, pid);

	if (args.debug >= DEBUG_INTERNAL) {
		syslog(LOG_INFO, "checking chain %s", chain);
	}

	/* Create new chain. */
	snprintf(cmd, sizeof(cmd), "%s -L %s > /dev/null 2>&1", args.iptables, chain);

	if (args.debug >= DEBUG_INTERNAL) {
		syslog(LOG_INFO, "exec: %s", cmd);
	}

	/* Are we in test mode? */
	if (args.test == 0) {
		ret = system(cmd); /* FIXME: Use exec() instead. */
	} else {
		ret = system(""); /* FIXME: Use exec() instead. */
	}

	/* Was the command successful? */
	if (WIFSIGNALED(ret) && (WTERMSIG(ret) == SIGINT || WTERMSIG(ret) == SIGQUIT)) {
		syslog(LOG_ERR, "failed to execute command: %s", cmd);

		return 0;
	}

	if (WEXITSTATUS(ret) != 0) {
		syslog(LOG_ERR, "chain %s not found", chain);

		return 0;
	}

	/* Success */
	return 1;
}

int chain_sync_del() {
	int ret;
	char cmd[512], chain[MAX_CHAIN_LEN];

	snprintf(chain, sizeof(chain), "%s-sync-check-%u", PROGRAM_NAME, pid);

	syslog(LOG_INFO, "deleting chain %s", chain);

	/* Delete Chain */
	snprintf(cmd, sizeof(cmd), "%s -X %s > /dev/null 2>&1", args.iptables, chain);

	if (args.debug >= DEBUG_INTERNAL) {
		syslog(LOG_INFO, "exec: %s", cmd);
	}

	/* Are we in test mode? */
	if (args.test == 0) {
		ret = system(cmd); /* FIXME: Use exec() instead. */
	} else {
		ret = system(""); /* FIXME: Use exec() instead. */
	}

	/* Was the command successful? */
	if (WIFSIGNALED(ret) && (WTERMSIG(ret) == SIGINT || WTERMSIG(ret) == SIGQUIT)) {
		syslog(LOG_ERR, "failed to execute command: %s", cmd);

		return 0;
	}

	if (WEXITSTATUS(ret) != 0) {
		syslog(LOG_ERR, "command not successful: %s", cmd);
	}

	/* Success */
	return 1;
}

int calc_pps(struct data_t* data) {
	unsigned int i, ret_val, sum = 0, max = 0;
	time_t cur = time(NULL);
	char buf[32];

	if (args.debug >= DEBUG_DUMP2) {
		inet_ntop(AF_INET, &(data->ip), buf, sizeof(buf));
		syslog(LOG_INFO, "calculating pps for %s", buf);
	}

	for (i = 0; i < args.max_samples; i++) {
		/* Only count sample if it was updated within interval seconds. */
		if (data->last_update[i] + args.interval >= cur && data->last_update[i] != cur) {
			if (args.debug >= DEBUG_DUMP2) {
				/* Used data. Was updated within interval seconds. */
				syslog(LOG_INFO, "+sample[%02u] = %u (last: %lu sec)", i, data->samples[i], (data->last_update[i] > 0) ? cur - data->last_update[i]: 0);
			}

			if (data->samples[i] > max) {
				max = data->samples[i];
			}

			sum += data->samples[i];
		} else {
			if (args.debug >= DEBUG_DUMP2) {
				/* Unused data. Too old... was not updated within last interval seconds. */
				syslog(LOG_INFO, " sample[%02u] = %u (last: %ld sec [expired])", i, data->samples[i], (data->last_update[i] > 0) ? cur - data->last_update[i]: 0);
			}
		}
	}

	switch (args.algo) {
		case PPS_ALGO_AVG:
			ret_val = (sum / args.interval);

			break;

		case PPS_ALGO_SUM:
			ret_val = sum;

			break;

		case PPS_ALGO_MAX:
			ret_val = max;

			break;

		default:
			ret_val = 0;
	}

	if (args.debug >= DEBUG_DUMP1) {
		inet_ntop(AF_INET, &(data->ip), buf, sizeof(buf));
		syslog(LOG_INFO, "%s has %u %s PPS (%u packets in %u sec)", buf, ret_val, algo_names[args.algo], sum, args.interval);
	}

	return ret_val;
}

/*
	Create New:
	/sbin/iptables -N phreld-207.205.66.55-45

	Delete Chain
	/sbin/iptables -X phreld-207.205.66.55-45

	Append Rules onto existing chain (drop all):
	/sbin/iptables -A phreld-207.205.66.55-45 -j DROP

	Append Rules onto existing chain (rate limit):
	/sbin/iptables -A phreld-207.205.66.55-45 -p icmp -m limit --limit 600/min -j RETURN
	/sbin/iptables -A phreld-207.205.66.55-45 -p icmp -j DROP

	Insert chain into INPUT
	/sbin/iptables -I INPUT -j phreld-207.205.66.55-45

	Remove chain from INPUT
	/sbin/iptables -D INPUT -j phreld-207.205.66.55-45

*/

/*
** Local Variables:
** c-basic-offset: 3
** tab-width: 3
** End:
** vim: noet ts=3 sw=3
*/
