=begin
  gettext.rb - GetText module

  Copyright (C) 2001-2005  Masao Mutoh
  Copyright (C) 2001-2003  Masahiro Sakai

      Masao Mutoh       <mutoh@highway.ne.jp>
      Masahiro Sakai    <s01397ms@sfc.keio.ac.jp>

  You may redistribute it and/or modify it under the same
  license terms as Ruby.

  $Id: gettext.rb,v 1.2 2005/08/20 17:05:34 mutoh Exp $
=end

require 'rbconfig'
require 'gettext/version'
require 'gettext/mo'
require 'gettext/locale'
require 'gettext/textdomain'

module GetText
  class NoboundTextDomainError < RuntimeError
  end

  @@__textdomain = Hash.new
  @@__textdomain_key = Hash.new

  @@output_charset = nil

  def bindtextdomain(domainname, path = nil, locale = nil, charset = nil)
    locale ||= Locale.get
    if @@output_charset
      charset = @@output_charset
    else
      charset ||= ENV["OUTPUT_CHARSET"] ? ENV["OUTPUT_CHARSET"] : Locale.codeset
      @@output_charset = charset
    end
    src = callersrc
    $stderr.print "bind the domain '#{domainname}' to '#{src}'. charset is #{charset}\n" if $DEBUG
    textdomain = @@__textdomain[src]
    unless textdomain
      textdomain = @@__textdomain_key[domainname]
      @@__textdomain[src] = textdomain
    end
    if ! textdomain or ! textdomain.same_property?(domainname, path, locale, charset)
      textdomain = TextDomain.new(domainname, path, locale, charset)
      @@__textdomain_key[domainname] = textdomain
      @@__textdomain[src] = textdomain
    end
    @@__textdomain[src]
  end

  def textdomain(domainname)
    src = callersrc
    textdomain = @@__textdomain_key[domainname]
    raise NoboundTextDomainError, "#{domainname} is not bound." unless textdomain
    @@__textdomain[src] = textdomain
    $stderr.print "Use the domain '#{domainname}' to '#{src}'. charset is #{textdomain.charset}\n" if $DEBUG
    @@__textdomain[src]
  end
  
  def gettext(msgid)
    textdomain = @@__textdomain[callersrc]
    textdomain ? textdomain.gettext(msgid) : msgid
  end
  
  def ngettext(msgid, msgid_plural, n)
    textdomain = @@__textdomain[callersrc]
    textdomain ? textdomain.ngettext(msgid, msgid_plural, n) : msgid
  end
  
  def N_(msgid)
    msgid
  end

  def sgettext(msgid, div = '|')
    textdomain = @@__textdomain[callersrc]
    if textdomain
      msg = textdomain.gettext(msgid)
    else
      msg = msgid
    end
    if msg == msgid
      if index = msg.rindex(div)
	msg = msg[(index + 1)..-1]
      end
    end
    msg
  end

  def callersrc
    caller(2)[0].sub(/:\d+(?::in \`\S+\')?\Z/, '')
  end

  def locale=(locale)
    textdomain = @@__textdomain[callersrc]
    textdomain.set_locale(locale) if textdomain
    locale
  end

  def output_charset=(charset)
    @@output_charset = charset
  end

  def output_charset
    @@output_charset
  end

  def charset=(cs)
    textdomain = @@__textdomain[callersrc]
    textdomain.set_charset(cs) if textdomain
  end
  
  alias :setlocale :locale=
  alias :set_locale :locale=
  alias :set_output_charset :output_charset=
  alias :_ :gettext
  alias :n_ :ngettext
  alias :s_ :sgettext

  module_function :bindtextdomain, :textdomain, 
    :N_, :gettext, :_, :ngettext, :n_, 
    :sgettext, :s_, :callersrc,
    :setlocale, :set_locale, :locale=, :charset=,
    :set_output_charset, :output_charset=, :output_charset
end
