/* mg-canvas-entity.c
 *
 * Copyright (C) 2002 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "mg-canvas.h"
#include "mg-canvas-entity.h"
#include "mg-canvas-field.h"
#include <libmergeant/mg-field.h>
#include <libmergeant/mg-db-table.h>
#include <libmergeant/mg-query.h>
#include <libmergeant/mg-entity.h>

static void mg_canvas_entity_class_init (MgCanvasEntityClass * class);
static void mg_canvas_entity_init       (MgCanvasEntity * drag);
static void mg_canvas_entity_dispose    (GObject   * object);
static void mg_canvas_entity_finalize   (GObject   * object);

static void mg_canvas_entity_set_property (GObject              *object,
					   guint                 param_id,
					   const GValue         *value,
					   GParamSpec           *pspec);
static void mg_canvas_entity_get_property (GObject              *object,
					   guint                 param_id,
					   GValue               *value,
					   GParamSpec           *pspec);

enum
{
	PROP_0,
	PROP_ENTITY,
	PROP_SCALE
};

struct _MgCanvasEntityPrivate
{
	MgEntity           *entity;

	/* UI building information */
        GSList             *field_items; /* list of GnomeCanvasItem for the fields */
	gint                init_font_size;
        GnomeCanvasItem    *title;
	gdouble            *field_ypos; /* array for each field's Y position in this canvas group */

	/* presentation parameters */
        gdouble             x_text_space;
        gdouble             y_text_space;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *entity_parent_class = NULL;

guint
mg_canvas_entity_get_type (void)
{
	static GType type = 0;

        if (!type) {
		static const GTypeInfo info = {
			sizeof (MgCanvasEntityClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) mg_canvas_entity_class_init,
			NULL,
			NULL,
			sizeof (MgCanvasEntity),
			0,
			(GInstanceInitFunc) mg_canvas_entity_init
		};		

		type = g_type_register_static (MG_CANVAS_ITEM_TYPE, "MgCanvasEntity", &info, 0);
	}

	return type;
}

	

static void
mg_canvas_entity_class_init (MgCanvasEntityClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	entity_parent_class = g_type_class_peek_parent (class);

	object_class->dispose = mg_canvas_entity_dispose;
	object_class->finalize = mg_canvas_entity_finalize;

	/* Properties */
	object_class->set_property = mg_canvas_entity_set_property;
	object_class->get_property = mg_canvas_entity_get_property;

	g_object_class_install_property
                (object_class, PROP_ENTITY,
                 g_param_spec_pointer ("entity", NULL, NULL, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property 
		(object_class, PROP_SCALE,
		 g_param_spec_double ("scale", NULL, NULL, 0., G_MAXDOUBLE, 0., G_PARAM_WRITABLE));
}

static void
mg_canvas_entity_init (MgCanvasEntity * entity)
{
	entity->priv = g_new0 (MgCanvasEntityPrivate, 1);
	entity->priv->entity = NULL;
	entity->priv->field_ypos = NULL;

	entity->priv->x_text_space = 3.;
	entity->priv->y_text_space = 3.;
}

static void clean_items (MgCanvasEntity *ce);
static void create_items (MgCanvasEntity *ce);
static void entity_nullified_cb (MgEntity *ent, MgCanvasEntity *ce);
static void entity_changed_cb (MgEntity *ent, MgCanvasEntity *ce);

static void
mg_canvas_entity_dispose (GObject   * object)
{
	MgCanvasEntity *ce;
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_CANVAS_ENTITY (object));

	ce = MG_CANVAS_ENTITY (object);
	clean_items (ce);
	if (ce->priv->entity) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->entity),
						      G_CALLBACK (entity_nullified_cb), ce);
		g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->entity),
						      G_CALLBACK (entity_changed_cb), ce);
		ce->priv->entity = NULL;
	}

	/* for the parent class */
	entity_parent_class->dispose (object);
}


static void
mg_canvas_entity_finalize (GObject   * object)
{
	MgCanvasEntity *ce;
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_CANVAS_ENTITY (object));

	ce = MG_CANVAS_ENTITY (object);
	if (ce->priv) {
		if (ce->priv->field_items)
			g_slist_free (ce->priv->field_items);
		if (ce->priv->field_ypos)
			g_free (ce->priv->field_ypos);

		g_free (ce->priv);
		ce->priv = NULL;
	}

	/* for the parent class */
	entity_parent_class->finalize (object);
}

static void 
mg_canvas_entity_set_property    (GObject              *object,
				  guint                 param_id,
				  const GValue         *value,
				  GParamSpec           *pspec)
{
	MgCanvasEntity *ce;
	gpointer ptr;
	gdouble scale;
	GSList *list;
	PangoFontDescription *font_desc, *font_copy;

	ce = MG_CANVAS_ENTITY (object);

	switch (param_id) {
	case PROP_ENTITY:
		ptr = g_value_get_pointer (value);
		if (ptr == ce->priv->entity)
			return;

		if (ce->priv->entity) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->entity),
							      G_CALLBACK (entity_nullified_cb), ce);
			g_signal_handlers_disconnect_by_func (G_OBJECT (ce->priv->entity),
							      G_CALLBACK (entity_changed_cb), ce);
			ce->priv->entity = NULL;
			clean_items (ce);
		}
		if (ptr) {
			g_return_if_fail (IS_MG_ENTITY (ptr));
			ce->priv->entity = ptr;
			g_signal_connect (G_OBJECT (ce->priv->entity), "nullified",
					  G_CALLBACK (entity_nullified_cb), ce);
			/* Signals to keep the display up to date */
			g_signal_connect (G_OBJECT (ce->priv->entity), "changed",
					  G_CALLBACK (entity_changed_cb), ce);

			create_items (ce);
		}
		break;
	case PROP_SCALE:
		scale = g_value_get_double (value);
		list = ce->priv->field_items;
		while (list) {
			g_object_set (G_OBJECT (list->data), "scale", scale, NULL);
			list = g_slist_next (list);
		}
		g_object_get (G_OBJECT (ce->priv->title), "font-desc", &font_desc, NULL);
		font_copy = pango_font_description_copy (font_desc);
		pango_font_description_set_size (font_copy, scale * ce->priv->init_font_size);
		g_object_set (G_OBJECT (ce->priv->title), "font-desc", font_copy, NULL);
		pango_font_description_free (font_copy);
	}
}

static void 
mg_canvas_entity_get_property    (GObject              *object,
				    guint                 param_id,
				    GValue               *value,
				    GParamSpec           *pspec)
{
	TO_IMPLEMENT;
}

static void
entity_nullified_cb (MgEntity *ent, MgCanvasEntity *ce)
{
	gtk_object_destroy (GTK_OBJECT (ce));
}

static void
entity_changed_cb (MgEntity *ent, MgCanvasEntity *ce)
{
	create_items (ce);
}

/* 
 * destroy any existing GnomeCanvasItem obejcts 
 */
static void 
clean_items (MgCanvasEntity *ce)
{
	/* destroy all the items in the group */
	while (GNOME_CANVAS_GROUP (ce)->item_list) 
		gtk_object_destroy (GTK_OBJECT (GNOME_CANVAS_GROUP (ce)->item_list->data));

	ce->priv->title = NULL;

	/* free the fields positions */
	if (ce->priv->field_ypos) {
		g_free (ce->priv->field_ypos);
		ce->priv->field_ypos = NULL;
	}
}

static int button_item_event (GnomeCanvasItem *ci, GdkEvent *event, MgCanvasEntity *ce);
static void field_item_destroy_cb (MgCanvasField *field, MgCanvasEntity *ce);

/*
 * create new GnomeCanvasItem objects
 */
static void 
create_items (MgCanvasEntity *ce)
{
	GnomeCanvasItem *item;
	const gchar *cstr;
	double x1, y1, x2, y2;
	GSList *list, *tmplist;
	gdouble x, sqsize, radius;
	gdouble title_text_height, title_text_width, total_width;
	gint fieldn;

	/* WARNING: the text items (GNOME_TYPE_CANVAS_TEXT) are first drawn without taking care of the real zoom factor =>
	 * we must take that into account when using the text sizes */
	gdouble scale = GNOME_CANVAS_ITEM (ce)->canvas->pixels_per_unit;

	clean_items (ce);

	/* Title of the Table or query */
	ce->priv->init_font_size = pango_font_description_get_size 
		(GTK_WIDGET (GNOME_CANVAS_ITEM (ce)->canvas)->style->font_desc);
	cstr = mg_base_get_name (MG_BASE (ce->priv->entity));
	item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (ce),
				      GNOME_TYPE_CANVAS_TEXT,
				      "font-desc", GTK_WIDGET (GNOME_CANVAS_ITEM (ce)->canvas)->style->font_desc,
				      "weight", PANGO_WEIGHT_BOLD,
				      "text", cstr,
				      "x", ce->priv->x_text_space, 
				      "y", ce->priv->y_text_space,
				      "fill_color", "black",
				      "justification", GTK_JUSTIFY_RIGHT, 
				      "anchor", GTK_ANCHOR_NORTH_WEST, 
				      NULL);
	gnome_canvas_item_get_bounds (item, &x1, &y1, &x2, &y2);
	ce->priv->title = item;

	/* Getting text metrics */
	title_text_height = (y2 - y1) * scale;
	title_text_width = (x2 - x1) * scale;
	total_width = title_text_width + 2 * ce->priv->x_text_space;

	/* fields */
	sqsize = title_text_height * 0.6;
	list = mg_entity_get_visible_fields (ce->priv->entity);

	ce->priv->field_ypos = g_new0 (gdouble, g_slist_length (list) + 1);
	ce->priv->field_ypos [0] = title_text_height + 3* ce->priv->y_text_space;

	tmplist = list;
	fieldn = 0;
	while (tmplist) {
		item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (ce),
					      MG_CANVAS_FIELD_TYPE,
					      "x", 0.,
					      "y", ce->priv->field_ypos[fieldn],
					      "field", tmplist->data,
					      NULL);	
		ce->priv->field_items = g_slist_append (ce->priv->field_items, item);
		
		g_signal_connect (G_OBJECT (item), "destroy",
				  G_CALLBACK (field_item_destroy_cb), ce);

		gnome_canvas_item_get_bounds (item, &x1, &y1, &x2, &y2);
		ce->priv->field_ypos[fieldn+1] = y2;

		if (x2 - x1 > total_width)
			total_width = x2 - x1;

		tmplist = g_slist_next (tmplist);
		fieldn++;
	}
	g_slist_free (list);

	/* "button" to close the MgCanvasEntity  */
	sqsize = title_text_height * 0.8;
	x = total_width + ce->priv->x_text_space;
	if (x - sqsize - 2*ce->priv->x_text_space < title_text_width)
		x = title_text_width + sqsize + 2*ce->priv->x_text_space;

	item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (ce),
				      GNOME_TYPE_CANVAS_RECT,
				      "x1", x - sqsize,
				      "y1", ce->priv->y_text_space,
				      "x2", x,
				      "y2", ce->priv->y_text_space + sqsize,
				      "fill_color", "white",
				      "outline_color", "black",
				      "width_units", 1.0,
				      NULL);
	gnome_canvas_item_raise_to_top (item);
	g_signal_connect (G_OBJECT (item),"event",
			  G_CALLBACK (button_item_event), ce);

	if (x - ce->priv->x_text_space > total_width)
		total_width = x - ce->priv->x_text_space;
	
	cstr = MG_CANVAS_ENTITY_COLOR;
	if (IS_MG_DB_TABLE (ce->priv->entity))
		cstr = MG_CANVAS_DB_TABLE_COLOR;
	if (IS_MG_QUERY (ce->priv->entity))
		cstr = MG_CANVAS_QUERY_COLOR;

	radius = sqsize * .2;
	item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (ce),
				      GNOME_TYPE_CANVAS_ELLIPSE,
				      "x1", x - sqsize/2. - radius,
				      "y1", ce->priv->y_text_space + sqsize/2. - radius,
				      "x2", x - sqsize/2. + radius,
				      "y2", ce->priv->y_text_space + sqsize/2. + radius,
				      "fill_color", cstr,
				      "outline_color", "black",
				      "width_units", 1.0,
				      NULL);
	gnome_canvas_item_raise_to_top (item);
	g_signal_connect (G_OBJECT (item),"event",
			  G_CALLBACK (button_item_event), ce);

	/* Top little frame */
	item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (ce),
				      GNOME_TYPE_CANVAS_RECT,
				      "x1", (double) 0,
				      "y1", (double) 0,
				      "x2", total_width + 2. * ce->priv->x_text_space,
				      "y2", title_text_height + 2 * ce->priv->y_text_space,
				      "outline_color", "black",
				      "fill_color", cstr, 
				      "width_units", 1.0, 
				      NULL);
	gnome_canvas_item_lower_to_bottom (item);

	/* Outline frame */
	item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (ce),
				      GNOME_TYPE_CANVAS_RECT,
				      "x1", (double) 0,
				      "y1", (double) 0,
				      "x2", total_width + 2. * ce->priv->x_text_space,
				      "y2", ce->priv->field_ypos[fieldn] + 1,
				      "outline_color", "black",
				      "fill_color", "white",
				      "width_units", 1.0, NULL);
	gnome_canvas_item_lower_to_bottom (item);

	total_width += 2. * ce->priv->x_text_space;

	/* setting the fields' background width to be the same for all */
	tmplist = ce->priv->field_items;
	while (tmplist) {
		g_object_set (G_OBJECT (tmplist->data), "width", total_width, NULL);
		tmplist = g_slist_next (tmplist);
	}

	/* make sure the scale is correctly applied to text items */
	g_object_set (G_OBJECT (ce), "scale", scale, NULL);
}

static void
field_item_destroy_cb (MgCanvasField *field, MgCanvasEntity *ce)
{
	g_assert (g_slist_find (ce->priv->field_items, field));
	ce->priv->field_items = g_slist_remove (ce->priv->field_items, field);
}

static void popup_delete_cb (GtkMenuItem *mitem, MgCanvasEntity *ce);
static int 
button_item_event (GnomeCanvasItem *ci, GdkEvent *event, MgCanvasEntity *ce)
{
	gboolean done = TRUE;
	GtkWidget *menu, *entry;

	switch (event->type) {
	case GDK_BUTTON_PRESS:
		menu = gtk_menu_new ();
		entry = gtk_menu_item_new_with_label (_("Remove"));
		g_signal_connect (G_OBJECT (entry), "activate", G_CALLBACK (popup_delete_cb), ce);
		gtk_menu_append (GTK_MENU (menu), entry);
		gtk_widget_show (entry);
		gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
				NULL, NULL, ((GdkEventButton *)event)->button,
				((GdkEventButton *)event)->time);
		done = FALSE;
		break;
	default:
		done = FALSE;
		break;
	}

	return done;	
}

static void
popup_delete_cb (GtkMenuItem *mitem, MgCanvasEntity *ce)
{
	MgGraphItem *gitem;

	gitem = mg_canvas_item_get_graph_item (MG_CANVAS_ITEM (ce));
	mg_base_nullify (MG_BASE (gitem));
}

/**
 * mg_canvas_entity_get_field_item
 * @ce: a #MgCanvasEntity object
 * @field: a #MgField object
 *
 * Get the #MgCanvasField object representing @field
 * in @ce.
 *
 * Returns: the corresponding #MgCanvasField
 */
MgCanvasField *
mg_canvas_entity_get_field_item (MgCanvasEntity *ce, MgField *field)
{
	GSList *fields;
	gint pos;

	g_return_val_if_fail (ce && IS_MG_CANVAS_ENTITY (ce), NULL);
	g_return_val_if_fail (ce->priv, NULL);
	g_return_val_if_fail (ce->priv->entity, NULL);

	fields = mg_entity_get_visible_fields (ce->priv->entity);
	pos = g_slist_index (fields, field);
	g_return_val_if_fail (pos >= 0, NULL);

	return g_slist_nth_data (ce->priv->field_items, pos);
}


/**
 * mg_canvas_entity_get_field_ypos
 * @ce: a #MgCanvasEntity object
 * @field: a #MgField object
 *
 * Get the Y position of the #MgCanvasField object representing @field
 * in @ce, in @ce's coordinates.
 *
 * Returns: the Y coordinate.
 */
gdouble
mg_canvas_entity_get_field_ypos (MgCanvasEntity *ce, MgField *field)
{
	gint pos;

	g_return_val_if_fail (ce && IS_MG_CANVAS_ENTITY (ce), 0.);
	g_return_val_if_fail (ce->priv, 0.);
	g_return_val_if_fail (ce->priv->entity, 0.);
	g_return_val_if_fail (ce->priv->field_ypos, 0.);

	pos = mg_entity_get_field_index (ce->priv->entity, field);
	g_return_val_if_fail (pos >= 0, 0.);
	return (ce->priv->field_ypos[pos+1] + ce->priv->field_ypos[pos]) / 2.;
}
