/***************************************************************************
 *   Copyright (C) 2005 Novell, Inc.                                       *
 *   Copyright (C) 2006 Debajyoti Bera                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA           *
 ***************************************************************************/

#include <qlabel.h>
#include <qlayout.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qtooltip.h>
#include <qwhatsthis.h>
#include <qvbuttongroup.h>
#include <assert.h>
#include <qcombobox.h>
#include <qtoolbutton.h>
#include <qtextedit.h>
#include <qtimer.h>

#include <kdirselectdialog.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kpopupmenu.h>
#include <kwinmodule.h>
#include <kregexpeditorinterface.h>
#include <kparts/componentfactory.h>
#include <kmessagebox.h>
#include <kurlrequester.h>
#include <klineedit.h>
#include <kpassivepopup.h>
#include <kprocess.h>

#include "configdialog.h"
#include "select_privacy_resource.h"

extern "C" {
#include <glib.h>
#include <beagle/beagle.h>
}

ConfigDialog::ConfigDialog( KGlobalAccel *accel )
    : KDialogBase( Tabbed, i18n("Configure"),
                    Ok | Cancel,
                    Ok, 0L, "config dialog" )
{
    QFrame *w = 0L; // the parent for the widgets

    w = addVBoxPage( i18n("&Search") );
    searchWidget = new SearchWidget( w, accel, "search widget" );

    w = addVBoxPage( i18n("&Indexing") );
    indexingWidget = new IndexingWidget( w, "indexing widget" );

    w = addVBoxPage( i18n("&Backends") );
    backendsWidget = new BackendsWidget( w, "backends widget" );

    w = addVBoxPage( i18n("&Daemon Status") );
    statusWidget = new StatusWidget( w, "status widget" );
}


ConfigDialog::~ConfigDialog()
{
}

int ConfigDialog::getMaxResultsDisplayed() const
{
   return searchWidget->maxResultsDisplayed->value();
}

void ConfigDialog::setMaxResultsDisplayed(int value)
{
   searchWidget->maxResultsDisplayed->setValue(value);
}

int ConfigDialog::getDefaultSortOrder() const
{
   return searchWidget->combo_order->currentItem();
}

void ConfigDialog::setDefaultSortOrder(int order)
{
   searchWidget->combo_order->setCurrentItem(order);
}

bool ConfigDialog::getStartBeagle() const
{
   return searchWidget->startBeagle->isChecked();
}

void ConfigDialog::setStartBeagle(bool value)
{
   searchWidget->startBeagle->setChecked(value);
}

void ConfigDialog::commitShortcuts()
{
   searchWidget->keysWidget->commitChanges();
}

bool ConfigDialog::getIndexHome() const
{
   return indexingWidget->indexHome->isChecked();
}

void ConfigDialog::setIndexHome(bool value)
{
   indexingWidget->indexHome->setChecked(value);
}

QStringList ConfigDialog::getRoots() const
{
   QStringList ret;
   for (int i=0;i<indexingWidget->general_list->childCount();i++)
     ret << indexingWidget->general_list->itemAtIndex(i)->text(0);
   return ret;
}

void ConfigDialog::setRoots(QStringList list)
{
   indexingWidget->general_list->clear();
   indexingWidget->remove_general->setEnabled(false);
   for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it ) {
     new KListViewItem(indexingWidget->general_list,*it);
   }
}

QStringList ConfigDialog::getTypes() const
{
   QStringList ret;
   for (int i=0;i<indexingWidget->privacy_list->childCount();i++)
     ret << indexingWidget->privacy_list->itemAtIndex(i)->text(0);
   return ret;
}

QStringList ConfigDialog::getValues() const
{
   QStringList ret;
   for (int i=0;i<indexingWidget->privacy_list->childCount();i++)
     ret << indexingWidget->privacy_list->itemAtIndex(i)->text(1);
   return ret;
}

void ConfigDialog::setExcludes(QStringList types, QStringList values)
{
   indexingWidget->privacy_list->clear();
   indexingWidget->remove_privacy->setEnabled(false);
   QStringList::Iterator it_values = values.begin();
   for ( QStringList::Iterator it_types = types.begin(); it_types != types.end(); ++it_types ) {
     new KListViewItem(indexingWidget->privacy_list,*it_types,*it_values);
     it_values++;
   }
}

void ConfigDialog::addAvailableBackends(QStringList availableBackends)
{
   backendsWidget->listview->clear();
   for ( QStringList::Iterator it_backends = availableBackends.begin(); it_backends != availableBackends.end(); ++it_backends ) {
     QCheckListItem *item = new QCheckListItem(backendsWidget->listview,*it_backends,QCheckListItem::CheckBox);
     item->setOn(true);
   }
}

void ConfigDialog::setDisabledBackends(QStringList disabledBackends) 
{
   for ( QStringList::Iterator it_backends = disabledBackends.begin(); it_backends != disabledBackends.end(); ++it_backends ) {
     QListViewItem *item = backendsWidget->listview->findItem(*it_backends,0);
     if (item)
       ((QCheckListItem*)item)->setOn(false);
   }
}

QStringList ConfigDialog::getDisabledBackends()
{
   QStringList ret;
   QListViewItemIterator it_items(backendsWidget->listview);
   while (it_items.current())
   {
     if (!((QCheckListItem*)it_items.current())->isOn())
       ret << it_items.current()->text(0);
     it_items++;
   }
   return ret;
}

/////////////////////////////////////////


SearchWidget::SearchWidget( QWidget *parent, KGlobalAccel *accel, const char *name )
    : QVBox( parent, name )
{
    setSpacing(KDialog::spacingHint());

    // General
    QGroupBox* gb_general = new QGroupBox(0, Qt::Vertical, i18n("General"), this);
    gb_general->setFlat(true);
    QVBoxLayout* gb_general_layout = new QVBoxLayout( gb_general->layout(), KDialog::spacingHint() );

    startBeagle = new QCheckBox(i18n("Start search && and indexing services automatically"), gb_general);
    gb_general_layout->addWidget( startBeagle);

    QHBoxLayout* qh_layout = new QHBoxLayout(gb_general_layout);
    QLabel* label_order = new QLabel(i18n("Default result sort order:"), gb_general);
    qh_layout->addWidget(label_order);
    combo_order = new QComboBox(gb_general);
    combo_order->insertItem(i18n("Relevance"));
    combo_order->insertItem(i18n("Name"));
    combo_order->insertItem(i18n("Date Modified"));
    QWhatsThis::add(combo_order,i18n("Define the default sort order at startup. You can switch the result sort order with the result list context menu."));
    label_order->setBuddy(combo_order);
    qh_layout->addWidget(combo_order);

    QHBoxLayout* qh_number = new QHBoxLayout(gb_general_layout);
    QLabel* label_number = new QLabel(i18n( "Maximum number of results displayed:" ), gb_general);
    qh_number->addWidget(label_number);
    maxResultsDisplayed = new QSpinBox( 2, 100, 1, gb_general );
    QWhatsThis::add(maxResultsDisplayed,i18n("Define how many results shall be displayed on one result page."));
    label_number->setBuddy(maxResultsDisplayed);
    qh_number->addWidget(maxResultsDisplayed);

    // Global Shortcuts
    QGroupBox* gb_keys = new QGroupBox(0, Qt::Vertical, i18n("Global Shortcuts"), this);
    QVBoxLayout* gb_keys_layout = new QVBoxLayout( gb_keys->layout(), KDialog::spacingHint() );
    gb_keys->setFlat(true);

    keysWidget = new KKeyChooser( accel, gb_keys );
    gb_keys_layout->addWidget(keysWidget);

    // Add some spacing at the end
    QWidget *dummy = new QWidget( this );
    setStretchFactor( dummy, 1 );
}

SearchWidget::~SearchWidget()
{
}

/////////////////////////////////////////

IndexingWidget::IndexingWidget( QWidget *parent, const char *name )
    : QVBox( parent, name )
{
    setSpacing(KDialog::spacingHint());

    // General
    QGroupBox* gb_general = new QGroupBox(0, Qt::Vertical, i18n("General"), this);
    gb_general->setFlat(true);
    QVBoxLayout* gb_general_layout = new QVBoxLayout( gb_general->layout(), KDialog::spacingHint() );

    QHBoxLayout* qh_index = new QHBoxLayout(gb_general_layout);
    indexHome = new QCheckBox(i18n("Index my home folder"), gb_general);
    qh_index->addWidget( indexHome );
    qh_index->addStretch(1);

    QPushButton* add_applications = new QPushButton( i18n("Add Application Paths"), gb_general );
    qh_index->addWidget( add_applications );
    connect( add_applications, SIGNAL( clicked() ), this, SLOT( slotAddApplications() ) );


    QLabel* label_general = new QLabel(i18n( "Add any additional folder to be included for indexing." ), gb_general);
    gb_general_layout->addWidget(label_general);

    QGridLayout* grid_general = new QGridLayout( gb_general_layout, 1, 1, KDialog::spacingHint());

    general_list = new KListView( gb_general );
    general_list->addColumn( i18n( "Name" ) );
    general_list->setResizeMode( KListView::LastColumn );
    general_list->setFullWidth( true );
    grid_general->addMultiCellWidget( general_list, 0, 2, 0, 0 );
    label_general->setBuddy(general_list);

    QPushButton* add_general = new QPushButton( i18n("Add..."), gb_general );
    grid_general->addWidget( add_general, 0, 1 );

    remove_general = new QPushButton( i18n("Remove"), gb_general );
    remove_general->setEnabled(false);
    grid_general->addWidget( remove_general, 1, 1 );

    QSpacerItem* general_spacer = new QSpacerItem( 20, 30, QSizePolicy::Minimum, QSizePolicy::Preferred );
    grid_general->addItem( general_spacer, 2, 1 );

    connect( add_general, SIGNAL( clicked() ), this, SLOT( slotAddSearch() ) );
    connect( remove_general, SIGNAL( clicked() ), this, SLOT( slotRemoveSearch() ) );
    connect( general_list, SIGNAL( selectionChanged(QListViewItem*) ), this, SLOT( slotSearchSelectionChanged(QListViewItem*) ) );

    // Privacy
    QGroupBox* gb_privacy = new QGroupBox(0, Qt::Vertical, i18n("Privacy"), this);
    gb_privacy->setFlat(true);
    QVBoxLayout* gb_privacy_layout = new QVBoxLayout( gb_privacy->layout(), KDialog::spacingHint() );

    QLabel* label_privacy = new QLabel(i18n( "Specify any resource, such as folder or pattern, you wish to exclude from indexing." ), gb_privacy);
    gb_privacy_layout->addWidget(label_privacy);

    QGridLayout* grid_privacy = new QGridLayout( gb_privacy_layout, 1, 1, KDialog::spacingHint());

    privacy_list = new KListView( gb_privacy );
    privacy_list->addColumn( i18n( "Type" ) );
    privacy_list->addColumn( i18n( "Name" ) );
    privacy_list->setResizeMode( KListView::LastColumn );
    privacy_list->setFullWidth( true );
    privacy_list->setAllColumnsShowFocus( true );
    grid_privacy->addMultiCellWidget( privacy_list, 0, 2, 0, 0 );
    label_privacy->setBuddy(privacy_list);

    QPushButton* add_privacy = new QPushButton( i18n("Add..."), gb_privacy );
    grid_privacy->addWidget( add_privacy, 0, 1 );

    remove_privacy = new QPushButton( i18n("Remove"), gb_privacy );
    remove_privacy->setEnabled(false);
    grid_privacy->addWidget( remove_privacy, 1, 1 );

    QSpacerItem* privacy_spacer = new QSpacerItem( 20, 30, QSizePolicy::Minimum, QSizePolicy::Preferred );
    grid_privacy->addItem( privacy_spacer, 2, 1 );

    connect( add_privacy, SIGNAL( clicked() ), this, SLOT( slotAddPrivacy() ) );
    connect( remove_privacy, SIGNAL( clicked() ), this, SLOT( slotRemovePrivacy() ) );
    connect( privacy_list, SIGNAL( selectionChanged(QListViewItem*) ), this, SLOT( slotPrivacySelectionChanged(QListViewItem*) ) );

    // Add some spacing at the end
    QWidget *dummy = new QWidget( this );
    setStretchFactor( dummy, 1 );
}

IndexingWidget::~IndexingWidget()
{
}

void IndexingWidget::slotAddSearch()
{
    KURL path = KDirSelectDialog::selectDirectory( QDir::home().absPath(), true, this, i18n("Select Folder"));
    new KListViewItem(general_list,path.path());
}

void IndexingWidget::slotSearchSelectionChanged(QListViewItem* item)
{
    remove_general->setEnabled( item );
}

void IndexingWidget::slotRemoveSearch()
{
    QListViewItem *item = general_list->currentItem();
    if (item &&
      KMessageBox::warningContinueCancel(this, i18n("<qt>Do you really want to remove this folder from the list of folders to be included for indexing?</qt>"),i18n("Remove Folder"),KStdGuiItem::del()) == KMessageBox::Continue)
    {
      delete item;
      remove_general->setEnabled( general_list->childCount() );
    }
}

void IndexingWidget::slotAddApplications()
{
   QString xdgdata = getenv("XDG_DATA_DIRS");
   if (xdgdata.isEmpty())
     xdgdata = "/usr/local/share/:/usr/share/";

   QStringList xdgappsdata = QStringList::split(':', xdgdata+":"+QDir::home().absPath()+"/.kde/share/");

   QStringList currentitems;
   for (int i=0;i<general_list->childCount();i++)
     currentitems << general_list->itemAtIndex(i)->text(0);

   for ( QStringList::Iterator it = xdgappsdata.begin(); it != xdgappsdata.end(); ++it ) {
     QString newitem = QString((*it)+"/applications").replace("//","/");
     if (currentitems.find(newitem)==currentitems.end())
       new KListViewItem(general_list,newitem);
   }

}

void IndexingWidget::slotAddPrivacy()
{
    KDialogBase dlg(this, 0, true, i18n("Add Resource"), KDialogBase::Ok | KDialogBase::Cancel);
    AddPrivacyResource w(&dlg);
    dlg.setMainWidget(&w);
    connect(w.radioButtonFolder, SIGNAL(toggled(bool)), w.folderRequester, SLOT(setEnabled(bool)));
    connect(w.radioButtonFolder, SIGNAL(toggled(bool)), w.patternEdit, SLOT(setDisabled(bool)));
    connect(w.radioButtonPattern, SIGNAL(toggled(bool)), w.patternEdit, SLOT(setEnabled(bool)));
    connect(w.radioButtonPattern, SIGNAL(toggled(bool)), w.folderRequester, SLOT(setDisabled(bool)));
    w.folderRequester->setCaption(i18n("Select Folder"));
    w.radioButtonFolder->setChecked(true);
    w.folderRequester->setMode(KFile::Directory | KFile::LocalOnly);
    if (dlg.exec())
    {
       if ( w.radioButtonFolder->isChecked()) {
         if (!w.folderRequester->lineEdit()->text().isEmpty())
           new KListViewItem(privacy_list,"Path",w.folderRequester->lineEdit()->text());
       }
       else
         if (!w.patternEdit->text().isEmpty())
           new KListViewItem(privacy_list,"Pattern",w.patternEdit->text());
    }
}

void IndexingWidget::slotRemovePrivacy()
{
    QListViewItem *item = privacy_list->currentItem();
    if (item &&
      KMessageBox::warningContinueCancel(this, i18n("<qt>Do you really want to remove this item from the list of data to be excluded from indexing?</qt>"),i18n("Remove Item"),KStdGuiItem::del()) == KMessageBox::Continue)
    {
      delete item;
      remove_privacy->setEnabled( privacy_list->childCount() );
    }
}

void IndexingWidget::slotPrivacySelectionChanged(QListViewItem* item)
{
    remove_privacy->setEnabled( item );
}

/////////////////////////////////////////

BackendsWidget::BackendsWidget( QWidget *parent, const char *name )
    : QVBox( parent, name )
{
    setSpacing(KDialog::spacingHint());

    new QLabel(i18n( "Select which of the available Beagle backends you want to have enabled." ), this);

    listview = new KListView(this);
    listview->addColumn(i18n("Backends"));
    listview->setResizeMode( QListView::LastColumn );
    listview->setFullWidth( true );
}

BackendsWidget::~BackendsWidget()
{
}

///////////////////////////////////////////
//

StatusWidget::StatusWidget( QWidget *parent, const char *name )
    : QVBox (parent, name)
{
    setSpacing(KDialog::spacingHint());

    QHBox *control_box = new QHBox (this);
    control_box->setSpacing (3);

    label_control = new QLabel (control_box);
    pb_control = new KPushButton (control_box);
    connect (pb_control, SIGNAL (clicked ()), this, SLOT (controlPressed ()) );

    status_box = new QGroupBox (1, Qt::Horizontal, this);

    version_label = new	QLabel (status_box);

    status_area = new QTextEdit (status_box);
    status_area->setReadOnly (true);

    index_info_box = new QTextEdit (status_box);
    index_info_box->setReadOnly (true);

    QHBox *footer_box = new QHBox (this);

    // Add some spacing to left
    QWidget *dummy = new QWidget( footer_box );
    setStretchFactor( dummy, 1 );
    pb_refresh = new KPushButton (i18n("Refresh Status"), footer_box);
    connect (pb_refresh, SIGNAL (clicked()), this, SLOT (refreshStatus ()) );

    g_type_init ();
    refreshStatus ();
}

StatusWidget::~StatusWidget ()
{
}

bool StatusWidget::refreshDaemonStatus ()
{
    gboolean is_running = beagle_util_daemon_is_running ();
    if (is_running) {
	label_control->setText (i18n("Beagle service is currently running. Click here to stop."));
	pb_control->setText (i18n("Stop"));
	last_status = true;
    } else {
	label_control->setText (i18n("Beagle service is currently stopped. Click here to start."));
	pb_control->setText (i18n("Start"));
	last_status = false;
    }
    return is_running;
}

void StatusWidget::refreshStatus ()
{
    pb_refresh->setDisabled (TRUE);
    bool is_running = refreshDaemonStatus ();

    status_box->setTitle ( QString ("[%1] ").arg (QDateTime::currentDateTime ().toString ()) );
    if (! is_running) {
	version_label->setText (i18n("Service not started."));
	pb_refresh->setDisabled (FALSE);
	status_area->clear ();
	index_info_box->clear ();
	return;
    }

    BeagleClient *client = beagle_client_new (NULL);
    BeagleDaemonInformationRequest *request = beagle_daemon_information_request_new ();
    BeagleResponse *response = beagle_client_send_request (client, BEAGLE_REQUEST (request), NULL);

    version_label->setText ( i18n ("Beagle service version: %1\n").arg (beagle_daemon_information_response_get_version (BEAGLE_DAEMON_INFORMATION_RESPONSE (response))));

    status_area->append (i18n("Current status:\n"));
    status_area->append (" "); // cheating
    status_area->append ( beagle_daemon_information_response_get_human_readable_status (BEAGLE_DAEMON_INFORMATION_RESPONSE (response)));

    index_info_box->append (i18n("Index information:"));
    index_info_box->append (" ");
    index_info_box->append ( beagle_daemon_information_response_get_index_information (BEAGLE_DAEMON_INFORMATION_RESPONSE (response)));

    g_object_unref (request);
    g_object_unref (response);
    //g_print ("%s\n", beagle_daemon_information_response_get_human_readable_status  (BEAGLE_DAEMON_INFORMATION_RESPONSE (response)));
    g_object_unref (client);

    pb_refresh->setDisabled (FALSE);
}

void StatusWidget::controlPressed ()
{
    pb_control->setDisabled (TRUE);
    if (last_status) {
	if (stopBeagle ())
	    QTimer::singleShot (1000, this, SLOT (verifyStatus ()));
    } else {
	if (startBeagle ())
	    QTimer::singleShot (5000, this, SLOT (verifyStatus ()));
    }
}

void StatusWidget::verifyStatus ()
{
    pb_control->setEnabled (TRUE);
    refreshDaemonStatus ();
}

bool StatusWidget::stopBeagle ()
{
    gboolean is_running = beagle_util_daemon_is_running ();
    if (! is_running) {
	KPassivePopup::message (i18n("Beagle service was already stopped."), this);
	return false;
    }

    BeagleClient *client = beagle_client_new (NULL);
    BeagleShutdownRequest *request;
    BeagleResponse *response;

    request = beagle_shutdown_request_new ();
    response = beagle_client_send_request (client, BEAGLE_REQUEST (request), NULL);
    g_object_unref (client);

    return true;
}

bool StatusWidget::startBeagle ()
{
    gboolean is_running = beagle_util_daemon_is_running ();
    if (is_running) {
	KPassivePopup::message (i18n("Beagle service already running."), this);
	return false;
    }

    KProcess *proc = new KProcess;
    *proc << "beagled";
    *proc << "--indexing-delay 2";
    if (!proc->start()) {
	KPassivePopup::message (i18n("Could not start beagle service."), this);
	return false;
    }

    return true;
}

#include "configdialog.moc"
