/* picture.c - 2004/11/21 */
/*
 *  EasyTAG - Tag editor for MP3 and Ogg Vorbis files
 *  Copyright (C) 2000-2003  Jerome Couderc <j.couderc@ifrance.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <config.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdk.h>
#include <glib/gi18n-lib.h>
#include <sys/types.h>
#include <fcntl.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>

#include "picture.h"
#include "easytag.h"
#include "misc.h"
#include "setting.h"
#include "msgbox.h"
#include "charset.h"


/****************
 * Declarations *
 ****************/


/**************
 * Prototypes *
 **************/

void Tag_Area_Picture_Drag_Data (GtkWidget *widget, GdkDragContext *dc, 
                                 gint x, gint y, GtkSelectionData *selection_data,
                                 guint info, guint t, gpointer data);
void Picture_Selection_Changed_cb (GtkTreeSelection *selection, gpointer data);
void Picture_Load_Filename (gchar *filename, gpointer user_data);

void Picture_Add_Button_Clicked         (GObject *object);
void Picture_Properties_Button_Clicked  (GObject *object);
void Picture_Save_Button_Clicked        (GObject *object);
void Picture_Clear_Button_Clicked       (GObject *object);

gint Picture_Format (Picture *pic);
const gchar *Picture_Format_String (gint format);
const gchar *Picture_Type_String   (gint type);
gchar *Picture_Info (Picture *pic);
void PictureEntry_Clear  (void);
void PictureEntry_Update (Picture *pic, gint select);

Picture *Picture_Allocate (void);
Picture *Picture_Copy_One (const Picture *pic);
Picture *Picture_Copy (const Picture *pic);
void     Picture_Free (Picture *pic);
Picture *Picture_Load_File_Data (const gchar *filename);
gboolean Picture_Save_File_Data (const Picture *pic, const gchar *filename);

gboolean Picture_Entry_View_Button_Pressed (GtkTreeView *treeview, GdkEventButton *event, gpointer data);
gboolean Picture_Entry_View_Key_Pressed    (GtkTreeView *treeview, GdkEvent *event, gpointer data);


/*************
 * Functions *
 *************/

void Tag_Area_Picture_Drag_Data (GtkWidget *widget, GdkDragContext *dc, 
                                 gint x, gint y, GtkSelectionData *selection_data,
                                 guint info, guint t, gpointer data)
{
    GtkTreeSelection *selection;
    gchar **uri_list, **uri;

    gtk_drag_finish(dc, TRUE, FALSE, t);

    if (info != TARGET_URI_LIST)
        return;
    if (!selection_data)
        return;

    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(PictureEntryView));
    gtk_tree_selection_unselect_all(selection);
    
    uri = uri_list = g_strsplit(selection_data->data, "\r\n", 0);
    while (*uri && strlen(*uri))
    {
        //Picture *pic;
        gchar *filename;

        filename = g_filename_from_uri(*uri, 0, 0);
        if (filename)
        {
            Picture_Load_Filename(filename,NULL);
            /*pic = Picture_Load_File_Data(filename);
            g_free(filename);
            if (pic)
                PictureEntry_Update(pic, 1);*/
        }
        uri++;
    }
    g_strfreev(uri_list);
}

void Picture_Selection_Changed_cb (GtkTreeSelection *selection, gpointer data)
{
    //if (gtk_tree_selection_count_selected_rows(GTK_TREE_SELECTION(selection)) == 1)
    //{
        gtk_widget_set_sensitive(GTK_WIDGET(PictureSaveButton), TRUE);
        gtk_widget_set_sensitive(GTK_WIDGET(PicturePropertiesButton), TRUE);
    //}else
    //{
    //    gtk_widget_set_sensitive(GTK_WIDGET(PictureSaveButton), FALSE);
    //    gtk_widget_set_sensitive(GTK_WIDGET(PicturePropertiesButton), FALSE);
    //}
}

void Picture_Clear_Button_Clicked (GObject *object)
{
    GList *paths, *refs = NULL, *node = NULL;
    GtkTreeSelection *selection;
    GtkTreeModel *model;
    GtkTreeIter iter;
    gpointer proxy;
    gint n = 0;
    
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(PictureEntryView));
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(PictureEntryView));
    paths = gtk_tree_selection_get_selected_rows(selection, 0);
    proxy = g_object_newv(G_TYPE_OBJECT, 0, NULL);

    for (node = paths; node; node = node->next)
    {
        refs = g_list_append(refs, gtk_tree_row_reference_new_proxy(proxy, model, node->data));
        gtk_tree_path_free(node->data);
    }
    g_list_free(paths);

    for (node = refs; node; node = node->next)
    {
        GtkTreePath *path = gtk_tree_row_reference_get_path(node->data);
        Picture *pic;
      
        gtk_tree_model_get_iter(model, &iter, path);
        
        gtk_tree_model_get(model, &iter, PICTURE_COLUMN_DATA, &pic,-1);
        Picture_Free(pic);
        
        gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
        gtk_tree_row_reference_deleted(proxy, path);
        gtk_tree_path_free(path);
        gtk_tree_row_reference_free(node->data);
        n++;
    }
    g_list_free(refs);
    
    if (!n)
        // Delete all if no one was selected.
        PictureEntry_Clear();
}

void Picture_Load_Filename (gchar *filename, gpointer user_data)
{
    Picture *pic;
    gchar *filename_utf8;
    
    filename_utf8 = filename_to_display(filename);
    pic = Picture_Load_File_Data(filename_utf8);
    
    // By default, set the filename in the description
    pic->description = g_path_get_basename(filename_utf8);
    
    g_free(filename_utf8);
    
    if (pic)
        PictureEntry_Update(pic, 1);
}

/*
 * To add an image in the list -> call a FileSelectionWindow
 */
void Picture_Add_Button_Clicked (GObject *object)
{
    GtkWidget *FileSelectionWindow;
    GtkFileFilter *filter;
    GtkWindow *parent_window = NULL;
    static gchar *init_dir = NULL;

    parent_window = (GtkWindow *) gtk_widget_get_toplevel(GTK_WIDGET(object));
    if (!GTK_WIDGET_TOPLEVEL(parent_window))
    {
        g_warning("Could not get parent window\n");
        return;
    }


    FileSelectionWindow = gtk_file_chooser_dialog_new(_("Add pictures"),
                                                      parent_window,
                                                      GTK_FILE_CHOOSER_ACTION_OPEN,
                                                      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                                      GTK_STOCK_OPEN, GTK_RESPONSE_OK,
                                                      NULL);

    // Add files filters
    // "All files" filter
    filter = gtk_file_filter_new ();
    gtk_file_filter_set_name(GTK_FILE_FILTER(filter), _("All Files"));
    gtk_file_filter_add_pattern(GTK_FILE_FILTER(filter), "*");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(FileSelectionWindow), GTK_FILE_FILTER(filter));

    // "PNG and JPEG" filter
    filter = gtk_file_filter_new ();
    gtk_file_filter_set_name(GTK_FILE_FILTER(filter), _("PNG and JPEG"));
    gtk_file_filter_add_mime_type(GTK_FILE_FILTER(filter), "image/jpeg");
    gtk_file_filter_add_mime_type(GTK_FILE_FILTER(filter), "image/png");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER (FileSelectionWindow), GTK_FILE_FILTER(filter));
    // Make this filter the default
    gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(FileSelectionWindow),  GTK_FILE_FILTER(filter));

    // Set window position
    if (MESSAGE_BOX_POSITION_NONE)
        gtk_window_set_position(GTK_WINDOW(FileSelectionWindow),GTK_WIN_POS_NONE);
    else if (MESSAGE_BOX_POSITION_CENTER)
        gtk_window_set_position(GTK_WINDOW(FileSelectionWindow),GTK_WIN_POS_CENTER);
    else if (MESSAGE_BOX_POSITION_MOUSE)
        gtk_window_set_position(GTK_WINDOW(FileSelectionWindow),GTK_WIN_POS_MOUSE);
    
    gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(FileSelectionWindow), TRUE);
    gtk_dialog_set_default_response(GTK_DIALOG(FileSelectionWindow), GTK_RESPONSE_CANCEL);
    if (init_dir)
        gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(FileSelectionWindow),init_dir);

    if (gtk_dialog_run(GTK_DIALOG(FileSelectionWindow)) == GTK_RESPONSE_OK)
    {
        GtkTreeSelection *selection;
        GSList *list;

        selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(PictureEntryView));
        gtk_tree_selection_unselect_all(selection);
    
        list = gtk_file_chooser_get_filenames(GTK_FILE_CHOOSER(FileSelectionWindow));
        g_slist_foreach(list, (GFunc) Picture_Load_Filename, 0);
        g_slist_free(list);
        
        // Save the directory selected for initialize next time
        g_free(init_dir);
        init_dir = gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(FileSelectionWindow));
    }
    gtk_widget_destroy(FileSelectionWindow);
}


void Picture_Properties_Button_Clicked (GObject *object)
{
    GtkWidget *ScrollWindowPictureTypes, *PictureTypesWindow;
    GtkWidget *type, *label, *desc;
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;
    GtkTreeSelection *selection;
    GtkListStore *store;
    GtkTreeIter type_iter_to_select, iter;
    GtkTreeModel *model;
    GtkWindow *parent_window = NULL;
    GList *selection_list = NULL;
    gint i;
    gint selection_nbr, selection_i = 1;
    gint picture_types[] = 
        {
            PICTURE_TYPE_OTHER,
            PICTURE_TYPE_FILE_ICON,
            PICTURE_TYPE_OTHER_FILE_ICON,
            PICTURE_TYPE_FRONT_COVER,
            PICTURE_TYPE_BACK_COVER,
            PICTURE_TYPE_LEAFLET_PAGE,
            PICTURE_TYPE_MEDIA,
            PICTURE_TYPE_LEAD_ARTIST_LEAD_PERFORMER_SOLOIST,
            PICTURE_TYPE_ARTIST_PERFORMER,
            PICTURE_TYPE_CONDUCTOR,
            PICTURE_TYPE_BAND_ORCHESTRA,
            PICTURE_TYPE_COMPOSER,
            PICTURE_TYPE_LYRICIST_TEXT_WRITER,
            PICTURE_TYPE_RECORDING_LOCATION,
            PICTURE_TYPE_DURING_RECORDING,
            PICTURE_TYPE_DURING_PERFORMANCE,
            PICTURE_TYPE_MOVIDE_VIDEO_SCREEN_CAPTURE,
            PICTURE_TYPE_A_BRIGHT_COLOURED_FISH,
            PICTURE_TYPE_ILLUSTRATION,
            PICTURE_TYPE_BAND_ARTIST_LOGOTYPE,
            PICTURE_TYPE_PUBLISHER_STUDIO_LOGOTYPE
        };


    parent_window = (GtkWindow *) gtk_widget_get_toplevel(GTK_WIDGET(object));
    if (!GTK_WIDGET_TOPLEVEL(parent_window))
    {
        g_warning("Could not get parent window\n");
        return;
    }

    model          = gtk_tree_view_get_model(GTK_TREE_VIEW(PictureEntryView));
    selection      = gtk_tree_view_get_selection(GTK_TREE_VIEW(PictureEntryView));
    selection_list = gtk_tree_selection_get_selected_rows(selection, NULL);
    selection_nbr  = gtk_tree_selection_count_selected_rows(GTK_TREE_SELECTION(selection));
    while (selection_list)
    {
        GtkTreePath *path = selection_list->data;
        Picture *pic;
        GtkTreeSelection *selectiontype;
        gchar *title;
        
        
        // Get corresponding picture
        gtk_tree_model_get_iter(GTK_TREE_MODEL(model), &iter, path);
        gtk_tree_model_get(GTK_TREE_MODEL(model), &iter, PICTURE_COLUMN_DATA, &pic, -1);

        title = g_strdup_printf(_("Picture Properties %d/%d"),selection_i++,selection_nbr);
        PictureTypesWindow = gtk_dialog_new_with_buttons(title,
                                                         parent_window,
                                                         GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                                         GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
                                                         GTK_STOCK_OK, GTK_RESPONSE_OK,
                                                         NULL);
        g_free(title);
        
        // Set window position
        if (MESSAGE_BOX_POSITION_NONE)
            gtk_window_set_position(GTK_WINDOW(PictureTypesWindow),GTK_WIN_POS_NONE);
        else if (MESSAGE_BOX_POSITION_CENTER)
            gtk_window_set_position(GTK_WINDOW(PictureTypesWindow),GTK_WIN_POS_CENTER);
        else if (MESSAGE_BOX_POSITION_MOUSE)
            gtk_window_set_position(GTK_WINDOW(PictureTypesWindow),GTK_WIN_POS_MOUSE);

        gtk_dialog_set_default_response(GTK_DIALOG(PictureTypesWindow), GTK_RESPONSE_OK);
    
        ScrollWindowPictureTypes = gtk_scrolled_window_new(NULL, NULL);
        gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(ScrollWindowPictureTypes),
                                       GTK_POLICY_AUTOMATIC,
                                       GTK_POLICY_AUTOMATIC);
        store = gtk_list_store_new(PICTURE_TYPE_COLUMN_COUNT, G_TYPE_STRING, G_TYPE_INT);
        type = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
        gtk_container_add(GTK_CONTAINER(ScrollWindowPictureTypes), type);
        
        renderer = gtk_cell_renderer_text_new();
        column = gtk_tree_view_column_new();
        gtk_tree_view_column_pack_start(column, renderer, FALSE);
        gtk_tree_view_column_set_title(column, _("Picture Type"));
        gtk_tree_view_column_set_attributes(column, renderer,
                                            "text", PICTURE_TYPE_COLUMN_TEXT,
                                            NULL);
        gtk_tree_view_append_column(GTK_TREE_VIEW(type), column);
        gtk_widget_set_size_request(type, 256, 256);
        gtk_box_pack_start(GTK_BOX(GTK_DIALOG(PictureTypesWindow)->vbox),ScrollWindowPictureTypes,TRUE,TRUE,0);
    
        // Load pictures types
        for (i = 0; i < sizeof(picture_types)/sizeof(picture_types[0]); i++)
        {
            GtkTreeIter itertype;
            
            gtk_list_store_append(store, &itertype);
            gtk_list_store_set(store, &itertype,
                               PICTURE_TYPE_COLUMN_TEXT, _(Picture_Type_String(picture_types[i])),
                               PICTURE_TYPE_COLUMN_TYPE_CODE, picture_types[i],
                               -1);
            // Line to select by default
            if (pic->type == picture_types[i])
                type_iter_to_select = itertype;
        }
        // Select the line by default
        selectiontype = gtk_tree_view_get_selection(GTK_TREE_VIEW(type));
        gtk_tree_selection_select_iter(selectiontype, &type_iter_to_select);

        // Description
        label = gtk_label_new(_("Picture Description:"));
        gtk_box_pack_start(GTK_BOX(GTK_DIALOG(PictureTypesWindow)->vbox),label,FALSE,FALSE,4);
    
        // Entry for the description
        desc = gtk_entry_new();
        gtk_box_pack_start(GTK_BOX(GTK_DIALOG(PictureTypesWindow)->vbox),desc,FALSE,FALSE,0);
        if (pic->description)
            gtk_entry_set_text(GTK_ENTRY(desc), pic->description);
    
        gtk_widget_show_all(PictureTypesWindow);
        
        if (gtk_dialog_run(GTK_DIALOG(PictureTypesWindow)) == GTK_RESPONSE_OK)
        {
            GtkTreeModel *modeltype;
            GtkTreeIter itertype;
            
            modeltype     = gtk_tree_view_get_model(GTK_TREE_VIEW(type));
            selectiontype = gtk_tree_view_get_selection(GTK_TREE_VIEW(type));
            if (gtk_tree_selection_get_selected(selectiontype, &modeltype, &itertype))
            {
                gchar *buffer;
                gint t;
                
                gtk_tree_model_get(modeltype, &itertype, 1, &t, -1); // Get cell 1 value
                pic->type = t;
    
                buffer = g_strdup(gtk_entry_get_text(GTK_ENTRY(desc)));
                Strip_String(buffer);
                if (pic->description)
                    g_free(pic->description);
                if ( g_utf8_strlen(buffer, -1) > 0 )
                {
                    pic->description = buffer;
                }else
                {
                    pic->description = 0;
                    g_free(buffer);
                }
                
                // Update value in the Picture view
                gtk_list_store_set(GTK_LIST_STORE(model), &iter,
                                   PICTURE_COLUMN_TEXT,
                                   Picture_Info(pic),
                                   -1);
            }
        }
        gtk_widget_destroy(PictureTypesWindow);
        
        if (!selection_list->next) break;
        selection_list = selection_list->next;
    }
}


/************
void save_selected_pic (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gchar *filename)
{
    Picture *pic;
    
    gtk_tree_model_get(model, iter, PICTURE_COLUMN_DATA, &pic, -1);
    Picture_Save_File_Data(pic, filename);
}

void get_selected_pic (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, GtkTreeIter *iter_set)
{
    *iter_set = *iter;
}

void Picture_Properties_Button_Clicked (GObject *object)
{
    GtkWidget *dialog, *type, *label, *desc;
    GtkWidget *ScrollWindowPictureTypes;
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;
    GtkTreeSelection *selection;
    GtkListStore *store;
    Picture *pic;
    GtkTreeModel *model2;
    GtkTreeIter iter2, default_type_iter;
    GtkWindow *parent_window = NULL;
    
    gint picture_types[] = 
        {
            PICTURE_TYPE_OTHER,
            PICTURE_TYPE_FILE_ICON,
            PICTURE_TYPE_OTHER_FILE_ICON,
            PICTURE_TYPE_FRONT_COVER,
            PICTURE_TYPE_BACK_COVER,
            PICTURE_TYPE_LEAFLET_PAGE,
            PICTURE_TYPE_MEDIA,
            PICTURE_TYPE_LEAD_ARTIST_LEAD_PERFORMER_SOLOIST,
            PICTURE_TYPE_ARTIST_PERFORMER,
            PICTURE_TYPE_CONDUCTOR,
            PICTURE_TYPE_BAND_ORCHESTRA,
            PICTURE_TYPE_COMPOSER,
            PICTURE_TYPE_LYRICIST_TEXT_WRITER,
            PICTURE_TYPE_RECORDING_LOCATION,
            PICTURE_TYPE_DURING_RECORDING,
            PICTURE_TYPE_DURING_PERFORMANCE,
            PICTURE_TYPE_MOVIDE_VIDEO_SCREEN_CAPTURE,
            PICTURE_TYPE_A_BRIGHT_COLOURED_FISH,
            PICTURE_TYPE_ILLUSTRATION,
            PICTURE_TYPE_BAND_ARTIST_LOGOTYPE,
            PICTURE_TYPE_PUBLISHER_STUDIO_LOGOTYPE
        };
    gint i;
    
    parent_window = (GtkWindow *) gtk_widget_get_toplevel(GTK_WIDGET(object));
    if (!GTK_WIDGET_TOPLEVEL(parent_window))
    {
        g_warning("Could not get parent window\n");
        return;
    }

    model2 = gtk_tree_view_get_model(GTK_TREE_VIEW(PictureEntryView));
    gtk_tree_selection_selected_foreach(gtk_tree_view_get_selection(GTK_TREE_VIEW(PictureEntryView)),
                                        (GtkTreeSelectionForeachFunc) get_selected_pic, &iter2);
    gtk_tree_model_get(model2, &iter2, PICTURE_COLUMN_DATA, &pic, -1);
    
    dialog = gtk_dialog_new_with_buttons(_("Picture Properties"),
                                         parent_window,
                                         GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                         GTK_STOCK_CANCEL,
                                         GTK_RESPONSE_REJECT,
                                         GTK_STOCK_OK,
                                         GTK_RESPONSE_ACCEPT,
                                         NULL);

    // Set window position
    if (MESSAGE_BOX_POSITION_NONE)
        gtk_window_set_position(GTK_WINDOW(dialog),GTK_WIN_POS_NONE);
    else if (MESSAGE_BOX_POSITION_CENTER)
        gtk_window_set_position(GTK_WINDOW(dialog),GTK_WIN_POS_CENTER);
    else if (MESSAGE_BOX_POSITION_MOUSE)
        gtk_window_set_position(GTK_WINDOW(dialog),GTK_WIN_POS_MOUSE);

    ScrollWindowPictureTypes = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(ScrollWindowPictureTypes),
                                   GTK_POLICY_AUTOMATIC,
                                   GTK_POLICY_AUTOMATIC);
    store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_INT);
    type = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
    gtk_container_add(GTK_CONTAINER(ScrollWindowPictureTypes), type);
    
    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new();
    gtk_tree_view_column_pack_start(column, renderer, FALSE);
    gtk_tree_view_column_set_title(column, _("Picture Type"));
    gtk_tree_view_column_set_attributes(column, renderer,
                                        "text", 0,
                                        NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(type), column);
    gtk_widget_set_size_request(type, 256, 256);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),ScrollWindowPictureTypes,TRUE,TRUE,0);

    // Load pictures types
    for (i = 0; i < sizeof(picture_types)/sizeof(picture_types[0]); i++)
    {
          GtkTreeIter iter;
          
          gtk_list_store_append(store, &iter);
          gtk_list_store_set(store, &iter,
                             0, _(Picture_Type_String(picture_types[i])),
                             1, picture_types[i],
                             -1);
          if (pic->type == picture_types[i])
              default_type_iter = iter;
    }

    label = gtk_label_new(_("Picture Description:"));
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),label,FALSE,FALSE,4);

    // Entry for the description
    desc = gtk_entry_new();
    if (pic->description)
        gtk_entry_set_text(GTK_ENTRY(desc), pic->description);
    
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),desc,FALSE,FALSE,0);

    gtk_widget_show_all(dialog);
    gtk_tree_selection_select_iter(gtk_tree_view_get_selection(GTK_TREE_VIEW(type)),
                                   &default_type_iter);
    
    if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT)
    {
        GtkTreeModel *model;
        GtkTreeIter iter;
        
        model = gtk_tree_view_get_model(GTK_TREE_VIEW(type));
        selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(type));
        if (gtk_tree_selection_get_selected(selection, &model, &iter))
        {
            gchar *buffer;
            gint t;
            
            gtk_tree_model_get(model, &iter, 1, &t, -1);
            pic->type = t;

            buffer = g_strdup(gtk_entry_get_text(GTK_ENTRY(desc)));
            Strip_String(buffer);
            if (pic->description)
                g_free(pic->description);
            if ( g_utf8_strlen(buffer, -1) > 0 )
            {
                pic->description = buffer;
            }else
            {
                pic->description = 0;
                g_free(buffer);
            }
            
            gtk_list_store_set(GTK_LIST_STORE(model2), &iter2,
                               PICTURE_TEXT_COLUMN,
                               Picture_Info(pic),
                               -1);
        }
    }
    gtk_widget_destroy(dialog);
}

void Picture_Save_Button_Clicked (GObject *object)
{
    GtkWidget *FileSelectionWindow;
    GtkWindow *parent_window = NULL;
    static gchar *init_dir = NULL;

    parent_window = (GtkWindow*) gtk_widget_get_toplevel(GTK_WIDGET(object));
    if (!GTK_WIDGET_TOPLEVEL(parent_window))
    {
        g_warning("Could not get parent window\n");
        return;
    }

    FileSelectionWindow = gtk_file_chooser_dialog_new(_("Save picture"),
                                                      parent_window,
                                                      GTK_FILE_CHOOSER_ACTION_SAVE,
                                                      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                                      GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
                                                      NULL);
    // Set the default folder if defined
    if (init_dir)
        gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(FileSelectionWindow),init_dir);

    if (gtk_dialog_run(GTK_DIALOG(FileSelectionWindow)) == GTK_RESPONSE_ACCEPT)
    {
        gchar *filename, *filename_utf8;
    
        filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(FileSelectionWindow));
        filename_utf8 = filename_to_display(filename);
        gtk_tree_selection_selected_foreach(gtk_tree_view_get_selection(GTK_TREE_VIEW(PictureEntryView)),
            (GtkTreeSelectionForeachFunc) save_selected_pic, filename_utf8);
        g_free(filename_utf8);

        // Save the directory selected for initialize next time
        g_free(init_dir);
        init_dir = gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(FileSelectionWindow));
    }
    gtk_widget_destroy(FileSelectionWindow);
}
*********/
void Picture_Save_Button_Clicked (GObject *object)
{
    GtkWidget *FileSelectionWindow;
    GtkFileFilter *filter;
    GtkWindow *parent_window = NULL;
    static gchar *init_dir = NULL;

    GtkTreeSelection *selection;
    GList *selection_list = NULL;
    GtkTreeModel *model;
    gint selection_nbr, selection_i = 1;


    parent_window = (GtkWindow*) gtk_widget_get_toplevel(GTK_WIDGET(object));
    if (!GTK_WIDGET_TOPLEVEL(parent_window))
    {
        g_warning("Could not get parent window\n");
        return;
    }

    model          = gtk_tree_view_get_model(GTK_TREE_VIEW(PictureEntryView));
    selection      = gtk_tree_view_get_selection(GTK_TREE_VIEW(PictureEntryView));
    selection_list = gtk_tree_selection_get_selected_rows(selection, NULL);
    selection_nbr  = gtk_tree_selection_count_selected_rows(GTK_TREE_SELECTION(selection));
    while (selection_list)
    {
        GtkTreePath *path = selection_list->data;
        GtkTreeIter iter;
        Picture *pic;
        gchar *title;
        
        // Get corresponding picture
        gtk_tree_model_get_iter(GTK_TREE_MODEL(model), &iter, path);
        gtk_tree_model_get(GTK_TREE_MODEL(model), &iter, PICTURE_COLUMN_DATA, &pic, -1);

        title = g_strdup_printf(_("Save picture %d/%d"),selection_i++,selection_nbr);
        FileSelectionWindow = gtk_file_chooser_dialog_new(title,
                                                          parent_window,
                                                          GTK_FILE_CHOOSER_ACTION_SAVE,
                                                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                                          GTK_STOCK_SAVE, GTK_RESPONSE_OK,
                                                          NULL);
        g_free(title);
        
        // Add files filters
        // "All files" filter
        filter = gtk_file_filter_new ();
        gtk_file_filter_set_name(GTK_FILE_FILTER(filter), _("All Files"));
        gtk_file_filter_add_pattern(GTK_FILE_FILTER(filter), "*");
        gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(FileSelectionWindow), GTK_FILE_FILTER(filter));
    
        // "PNG and JPEG" filter
        filter = gtk_file_filter_new ();
        gtk_file_filter_set_name(GTK_FILE_FILTER(filter), _("PNG and JPEG"));
        gtk_file_filter_add_mime_type(GTK_FILE_FILTER(filter), "image/jpeg");
        gtk_file_filter_add_mime_type(GTK_FILE_FILTER(filter), "image/png");
        gtk_file_chooser_add_filter(GTK_FILE_CHOOSER (FileSelectionWindow), GTK_FILE_FILTER(filter));
        // Make this filter the default
        gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(FileSelectionWindow),  GTK_FILE_FILTER(filter));
        
        // Set window position
        if (MESSAGE_BOX_POSITION_NONE)
            gtk_window_set_position(GTK_WINDOW(FileSelectionWindow),GTK_WIN_POS_NONE);
        else if (MESSAGE_BOX_POSITION_CENTER)
            gtk_window_set_position(GTK_WINDOW(FileSelectionWindow),GTK_WIN_POS_CENTER);
        else if (MESSAGE_BOX_POSITION_MOUSE)
            gtk_window_set_position(GTK_WINDOW(FileSelectionWindow),GTK_WIN_POS_MOUSE);
        
        gtk_dialog_set_default_response(GTK_DIALOG(FileSelectionWindow), GTK_RESPONSE_OK);

        // Set the default folder if defined
        if (init_dir)
            gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(FileSelectionWindow),init_dir);
        
        // Suggest a filename to the user
        if ( pic->description && strlen(pic->description) )
        {
            gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(FileSelectionWindow), pic->description); //filename in UTF8
        }else
        {
            gchar *image_name = NULL;
            switch (Picture_Format(pic))
            {
                case PICTURE_FORMAT_JPEG :
                    image_name = g_strdup("image_name.jpg");
                    break;
                case PICTURE_FORMAT_PNG :
                    image_name = g_strdup("image_name.png");
                    break;
                case PICTURE_FORMAT_UNKNOWN :
                    image_name = g_strdup("image_name.ext");
                    break;
            }
            gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(FileSelectionWindow), image_name); //filename in UTF8
            g_free(image_name);
        }
        
        if (gtk_dialog_run(GTK_DIALOG(FileSelectionWindow)) == GTK_RESPONSE_OK)
        {
            FILE *file;
            gchar *filename, *filename_utf8;
        
            // Save the directory selected for initialize next time
            g_free(init_dir);
            init_dir = gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(FileSelectionWindow));

            filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(FileSelectionWindow));
            filename_utf8 = filename_to_display(filename);
            
            // Warn user if the file already exists, else saves directly
            if ( (file=fopen(filename_utf8,"r"))!=NULL )
            {
                gchar *msg;
                GtkWidget *msgbox;
                gint button;

                fclose(file);
                
                msg = g_strdup_printf(_("The following file already exists :\n'%s'\n"
                    "Do you want to overwrite?"),filename_utf8);
                msgbox = msg_box_new (_("Save file..."),msg,GTK_STOCK_DIALOG_QUESTION,BUTTON_NO,BUTTON_YES,0);
                g_free(msg);
                msg_box_hide_check_button(MSG_BOX(msgbox));
                button = msg_box_run(MSG_BOX(msgbox));
                gtk_widget_destroy(msgbox);
                
                if (button == BUTTON_YES)
                {
                    Picture_Save_File_Data(pic, filename_utf8);
                }
            }else
            {
                Picture_Save_File_Data(pic, filename_utf8);
            }
            g_free(filename_utf8);
        }
        gtk_widget_destroy(FileSelectionWindow);

        if (!selection_list->next) break;
        selection_list = selection_list->next;
    }
}


/* FIXME: Possibly use gnome_vfs_get_mime_type_for_buffer. */
gint Picture_Format (Picture *pic)
{
    if (pic->data && pic->size > 2
    &&  pic->data[0] == 0xff 
    &&  pic->data[1] == 0xd8)
        return PICTURE_FORMAT_JPEG;
        
    if (pic->data && pic->size > 8
    &&  pic->data[0] == 0x89 
    &&  pic->data[1] == 0x50
    &&  pic->data[2] == 0x4e
    &&  pic->data[3] == 0x47
    &&  pic->data[4] == 0x0d
    &&  pic->data[5] == 0x0a
    &&  pic->data[6] == 0x1a
    &&  pic->data[7] == 0x0a)
        return PICTURE_FORMAT_PNG;
        
    return PICTURE_FORMAT_UNKNOWN;
}

const gchar *Picture_Format_String (gint format)
{
    switch (format)
    {
        case PICTURE_FORMAT_JPEG:
            return _("JPEG image");
        case PICTURE_FORMAT_PNG:
            return _("PNG image");
        default:
            return _("Unknown image");
    }
}

const gchar *Picture_Type_String (gint type)
{
    switch (type)
    {
        case PICTURE_TYPE_OTHER:
            return _("Other");
        case PICTURE_TYPE_FILE_ICON:
            return _("32x32 pixel PNG file icon");
        case PICTURE_TYPE_OTHER_FILE_ICON:
            return _("Other file icon");
        case PICTURE_TYPE_FRONT_COVER:
            return _("Cover (front)");
        case PICTURE_TYPE_BACK_COVER:
            return _("Cover (back)");
        case PICTURE_TYPE_LEAFLET_PAGE:
            return _("Leaflet page");
        case PICTURE_TYPE_MEDIA:
            return _("Media (e.g. label side of CD)");
        case PICTURE_TYPE_LEAD_ARTIST_LEAD_PERFORMER_SOLOIST:
            return _("Lead artist/lead performer/soloist");
        case PICTURE_TYPE_ARTIST_PERFORMER:
            return _("Artist/performer");
        case PICTURE_TYPE_CONDUCTOR:
            return _("Conductor");
        case PICTURE_TYPE_BAND_ORCHESTRA:
            return _("Band/Orchestra");
        case PICTURE_TYPE_COMPOSER:
            return _("Composer");
        case PICTURE_TYPE_LYRICIST_TEXT_WRITER:
            return _("Lyricist/text writer");
        case PICTURE_TYPE_RECORDING_LOCATION:
            return _("Recording location");
        case PICTURE_TYPE_DURING_RECORDING:
            return _("During recording");
        case PICTURE_TYPE_DURING_PERFORMANCE:
            return _("During performance");
        case PICTURE_TYPE_MOVIDE_VIDEO_SCREEN_CAPTURE:
            return _("Movie/video screen capture");
        case PICTURE_TYPE_A_BRIGHT_COLOURED_FISH:
            return _("A bright coloured fish");
        case PICTURE_TYPE_ILLUSTRATION:
            return _("Illustration");
        case PICTURE_TYPE_BAND_ARTIST_LOGOTYPE:
            return _("Band/Artist logotype");
        case PICTURE_TYPE_PUBLISHER_STUDIO_LOGOTYPE:
            return _("Publisher/studio logotype");
        default:
            return _("Unknown picture type");
    }
}

gchar *Picture_Info (Picture *pic)
{
    gchar *format, *desc, *type, *r;
    GString *s;

    format = (gchar *)Picture_Format_String(Picture_Format(pic));
    
    if (pic->description)
        desc = pic->description;
    else
        desc = "";

    type = (gchar *)Picture_Type_String(pic->type);

    s = g_string_new(0);
    g_string_sprintf(s, "%s (%s - %dx%d %s)\n%s: %s\n%s: %s",
                     format, 
                     Convert_Size_1((gfloat)pic->size),
                     pic->width, pic->height, _("pixels"),
                     _("Type"), type,
                     _("Description"), desc);
    r = s->str;
    g_string_free(s, FALSE);
    return r;
}

void PictureEntry_Clear (void)
{
    GtkListStore *picture_store;
    GtkTreeModel *model;
    GtkTreeIter iter;
    Picture *pic;
    
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(PictureEntryView));
    if (gtk_tree_model_get_iter_first(model, &iter))
    {
        do
        {
            gtk_tree_model_get(model, &iter, PICTURE_COLUMN_DATA, &pic,-1);
            Picture_Free(pic);
        } while (gtk_tree_model_iter_next(model, &iter));
    }
    
    picture_store = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(PictureEntryView)));
    if (picture_store)
        gtk_list_store_clear(picture_store);
}

void PictureEntry_Update (Picture *pic, gint select)
{
    GdkPixbufLoader *loader = 0;
    
    if (!pic->data)
    {
        PictureEntry_Clear();
        return;
    }
    
    loader = gdk_pixbuf_loader_new();
    if (loader)
    {
        if (gdk_pixbuf_loader_write(loader, pic->data, pic->size, 0))
        {
            GtkTreeSelection *selection;
            GdkPixbuf *pixbuf;
            
            selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(PictureEntryView));
            
            pixbuf = gdk_pixbuf_loader_get_pixbuf(loader);
            if (pixbuf)
            {
                GtkListStore *picture_store;
                GtkTreeIter iter1;
                GdkPixbuf *scaled_pixbuf;
                gint scaled_pixbuf_width;
                gint scaled_pixbuf_height;
                
                // Keep aspect ratio of the picture
                pic->width = gdk_pixbuf_get_width (pixbuf);
                pic->height = gdk_pixbuf_get_height (pixbuf);
                if (pic->width > pic->height)
                {
                    scaled_pixbuf_width  = 96;
                    scaled_pixbuf_height = 96 * pic->height / pic->width;
                }else
                {
                    scaled_pixbuf_width = 96 * pic->width / pic->height;
                    scaled_pixbuf_height = 96;
                }
                
                scaled_pixbuf = gdk_pixbuf_scale_simple(pixbuf, 
                                    scaled_pixbuf_width, scaled_pixbuf_height,
                                    //GDK_INTERP_NEAREST); // Lower quality but better speed
                                    GDK_INTERP_BILINEAR);
                gdk_pixbuf_unref(pixbuf);
            
                picture_store = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(PictureEntryView)));
                gtk_list_store_append(picture_store, &iter1);
                gtk_list_store_set(picture_store, &iter1,
                                   PICTURE_COLUMN_PIC, scaled_pixbuf,
                                   PICTURE_COLUMN_TEXT, Picture_Info(pic),
                                   PICTURE_COLUMN_DATA, Picture_Copy_One(pic),
                                   -1);
                if (select)
                    gtk_tree_selection_select_iter(selection, &iter1);
                gdk_pixbuf_unref(scaled_pixbuf);
            }
        }
    }
    gdk_pixbuf_loader_close(loader, 0);
    
    // Do also for next picture
    if (pic->next)
        PictureEntry_Update(pic->next, select);
}


Picture *Picture_Allocate (void)
{
    Picture *pic = g_malloc0(sizeof(Picture));
    return pic;
}

Picture *Picture_Copy_One (const Picture *pic)
{
    Picture *pic2;
    
    if (!pic)
        return 0;
    pic2 = Picture_Allocate();
    pic2->type = pic->type;
    if (pic->description)
        pic2->description = g_strdup(pic->description);
    if (pic->data)
    {
        pic2->size = pic->size;
        pic2->data = g_malloc(pic2->size);
        memcpy(pic2->data, pic->data, pic->size);
    }
    return pic2;
}

Picture *Picture_Copy (const Picture *pic)
{
    Picture *pic2 = Picture_Copy_One(pic);
    if (pic->next)
        pic2->next = Picture_Copy(pic->next);
    return pic2;
}

void Picture_Free (Picture *pic)
{
    if (!pic)
        return;
    if (pic->next)
        Picture_Free(pic->next);
    if (pic->description)
        g_free(pic->description);
    if (pic->data)
        g_free(pic->data);
    g_free(pic);
    pic = (Picture *)NULL;
}

#define LOAD_BUFFER_SIZE 4096

Picture *Picture_Load_File_Data (const gchar *filename)
{
    Picture *pic;
    gchar *buffer = 0;
    size_t size = 0;
    gint fd;
    
    fd = open(filename, O_RDONLY);
    if (fd == -1)
        return FALSE;
    
    do
    {
        ssize_t r;
        gchar *p;
        
        p = g_realloc(buffer, size + LOAD_BUFFER_SIZE);
        if (!p)
            goto fail;
        buffer = p;
        
        r = read(fd, buffer + size, LOAD_BUFFER_SIZE);
        if (r == -1)
            goto fail;
        if (!r)
            break;
        
        size += r;
    } while (1);
    close(fd);
    
    pic = Picture_Allocate();
    pic->size = size;
    pic->data = buffer;
    return pic;

    fail:
        if (buffer)
            g_free(buffer);
        close(fd);
        return 0;
}

/*
 * Save picture data to a file (jpeg, png)
 */ 
gboolean Picture_Save_File_Data (const Picture *pic, const gchar *filename)
{
    gint fd;
      
    fd = open(filename, O_WRONLY|O_CREAT|O_TRUNC, 0666);
    if (fd == -1)
        return FALSE;
    
    if (write(fd, pic->data, pic->size) != pic->size)
    {
        close(fd);
        return FALSE;
    }
    
    close(fd);
    return TRUE;
}

/*
 * If double clicking the PictureEntryView :
 *  - over a selected row : opens properties window
 *  - over an empty area : open the adding window
 */
gboolean Picture_Entry_View_Button_Pressed (GtkTreeView *treeview, GdkEventButton *event, gpointer data)
{
    if (event->type==GDK_2BUTTON_PRESS && event->button==1)
    {
        GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(PictureEntryView));
        
        if (gtk_tree_selection_count_selected_rows(GTK_TREE_SELECTION(selection)) != 0)
            Picture_Properties_Button_Clicked(G_OBJECT(PicturePropertiesButton));
        else
            Picture_Add_Button_Clicked(G_OBJECT(PictureAddButton));

        return TRUE;
    }
    
    return FALSE;
}


/* 
 * Key press into picture entry
 *   - Delete = delete selected picture files
 */
gboolean Picture_Entry_View_Key_Pressed (GtkTreeView *treeview, GdkEvent *event, gpointer data)
{
    GdkEventKey *kevent;

    kevent = (GdkEventKey *)event;
    if (event && event->type==GDK_KEY_PRESS)
    {
        switch(kevent->keyval)
        {
            case GDK_Delete:
                Picture_Clear_Button_Clicked(NULL);
                return TRUE;
        }
    }

    return FALSE;
}
