#! /usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Copyright 2022, Nils Hilbricht, Germany ( https://www.hilbricht.net )

This file is part of the Laborejo Software Suite ( https://www.laborejo.org ),

This application is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import logging; logger = logging.getLogger(__name__); logger.info("import")

#Standard Library
from collections import Counter

#Third Party
from PyQt5 import QtCore, QtGui, QtWidgets

#Our Qt
from .instrument import InstrumentTreeController, GuiLibrary, GuiInstrument

#Engine
from engine.config import * #imports METADATA
import engine.api as api



fakeLibMostUsed = {}
fakeLibMostUsed["tarFilePath"] = ""
fakeLibMostUsed["id"] = 0  #If this is selected in the GUI it will select the default lib in the main instrument view, which is wrong but safe.
fakeLibMostUsed["name"] = QtCore.QCoreApplication.translate("Favorites", "Top 10 - Most Used")
fakeLibMostUsed["description"] = ""
fakeLibMostUsed["license"] = ""
fakeLibMostUsed["vendor"] = ""
fakeLibMostUsed["version"] = "1"


class FavoritesTreeController(InstrumentTreeController):

    def buildTree(self, data:dict):
        """
        Only shows some of the instruments,
        depending on the current filter-mode
        """
        #Reset everything except our cached data.
        self.treeWidget.clear() #will delete the C++ objects. We need to delete the PyQt objects ourselves, like so:
        self.guiLibraries = {} # idKey : GuiLibrary idKey is a tuple with second value -1, which would be the instrument.
        self.guiInstruments = {} # idKey : GuiInstrument

        if data:
            self._cachedData = data
        else:
            assert self._cachedData
            data = self._cachedData

        #Add all available instruments to the database, but don't use them yet
        for libraryId, libraryDict in data.items():
            for instrumentdId, instrumentDict in libraryDict.items():
                if instrumentdId == "library":
                   continue

                gi = GuiInstrument(parentTreeController=self, instrumentDict=instrumentDict)

                self.guiInstruments[instrumentDict["idKey"]] = gi
                if instrumentDict["idKey"] in self._cachedLastInstrumentStatus:
                    gi.updateStatus(self._cachedLastInstrumentStatus[instrumentDict["idKey"]])


        mostUsed = GuiLibrary(parentTreeController=self, libraryDict=fakeLibMostUsed)
        self.treeWidget.addTopLevelItem(mostUsed)
        mostUsed.setExpanded(True)
        #mostRecent = GuiLibrary(parentTreeController=self, libraryDict=libraryDict["library"])

        top10 = self.getTop10FavoriteInstruments() # Return an ordered list of double tuple ((libid, instid), counter)
        for placement, ((libId, instId), counter) in enumerate(top10):
                if not (libId, instId) in self.guiInstruments:
                    continue
                gi = self.guiInstruments[(libId, instId)]
                mostUsed.addChild(gi)
                gi.injectWidgets() #only possible after gi.init() was done and item inserted.
                #Misuse the id as top 10-Counter
                gi.setData(gi.columns.index("idKey"), 0, str(placement+1).zfill(2)) #0 is the data role, just standard display text. We combine both IDs to a float number for sorting. If used with setData instead of setText Qt will know how to sort 11 before 1000

        self._adjustColumnSize()

    def getTop10FavoriteInstruments(self)->list:
        """Return an ordered list of double tuple ((libid, instid), counter)"""
        settings = QtCore.QSettings("LaborejoSoftwareSuite", METADATA["shortName"])
        if settings.contains("favoriteInstruments"):
            database = settings.value("favoriteInstruments", type=dict)
            return Counter(database).most_common(10) #python already knows how to create that
        else:
            return []
