#! /usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Copyright 2022, Nils Hilbricht, Germany ( https://www.hilbricht.net )

This file is part of the Laborejo Software Suite ( https://www.laborejo.org ),

This application is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
import logging; logging.info("import {}".format(__file__))

#Standard Library Modules

#Third Party Modules
from PyQt5 import QtWidgets, QtCore, QtGui

#Template Modules

#Our modules
import engine.api as api


class AuditionerMidiInputComboController(object):

    def __init__(self, parentMainWindow):
        self.parentMainWindow = parentMainWindow
        self.comboBox = parentMainWindow.ui.auditionerMidiInputComboBox
        self.volumeDial = parentMainWindow.ui.auditionerVolumeDial
        self.volumeDial.setMaximum(0)
        self.volumeDial.setMinimum(-21)
        self.volumeDial.valueChanged.connect(self._sendVolumeChangeToEngine)
        self.volumeDial.enterEvent = lambda ev: self.parentMainWindow.statusBar().showMessage((QtCore.QCoreApplication.translate("Auditioner", "Use mousewheel to change the Auditioner Volume")))
        self.volumeDial.leaveEvent = lambda ev: self.parentMainWindow.statusBar().showMessage("")
        self.wholePanel = parentMainWindow.ui.auditionerWidget
        self.currentInstrumentLabel =  parentMainWindow.ui.auditionerCurrentInstrument_label
        self.defaultText = QtCore.QCoreApplication.translate("Auditioner", "Double click on an instrument to load it into the Auditioner")
        self.currentInstrumentLabel.setText(self.defaultText)

        #if not api.isStandaloneMode():
            #self.wholePanel.hide()
            #return

        self.wholePanel.show() #explicit is better than implicit
        self.originalShowPopup = self.comboBox.showPopup
        self.comboBox.showPopup = self.showPopup
        self.comboBox.activated.connect(self._newPortChosen)

        api.callbacks.startLoadingAuditionerInstrument.append(self.callback_startLoadingAuditionerInstrument)
        api.callbacks.auditionerInstrumentChanged.append(self.callback_auditionerInstrumentChanged)
        api.callbacks.auditionerVolumeChanged.append(self.callback__auditionerVolumeChanged)

    def callback_startLoadingAuditionerInstrument(self, idkey):
        self.parentMainWindow.qtApp.setOverrideCursor(QtCore.Qt.WaitCursor) #reset in self.callback_auditionerInstrumentChanged
        self.currentInstrumentLabel.setText(QtCore.QCoreApplication.translate("Auditioner", "…loading…"))
        self.parentMainWindow.qtApp.processEvents() #actually show the label and cursor

    def callback_auditionerInstrumentChanged(self, exportMetadata:dict):
        if exportMetadata is None:
            self.currentInstrumentLabel.setText(self.defaultText)
        else:
            self.parentMainWindow.qtApp.restoreOverrideCursor() #We assume the cursor was set to a loading animation
            key = exportMetadata["idKey"]
            t = f"➜ [{key[0]}-{key[1]}] {exportMetadata['name']}"
            self.currentInstrumentLabel.setText(t)

    def callback__auditionerVolumeChanged(self, value:float):
        self.volumeDial.setValue(int(value))
        self.parentMainWindow.statusBar().showMessage(QtCore.QCoreApplication.translate("Auditioner", "Auditioner Volume: {}").format(value))

    def _sendVolumeChangeToEngine(self, newValue):
        self.volumeDial.blockSignals(True)
        api.setAuditionerVolume(newValue)
        self.volumeDial.blockSignals(False)


    def _newPortChosen(self, index:int):
        assert self.comboBox.currentIndex() == index
        api.connectAuditionerPort(self.comboBox.currentText())

    def showPopup(self):
        """When the combobox is opened quickly update the port list before showing it"""
        self._fill()
        self.originalShowPopup()

    def _fill(self):
        self.comboBox.clear()
        availablePorts = api.getAvailableAuditionerPorts()
        self.comboBox.addItem("") # Not only a more visible seaparator than the Qt one, but also doubles as "disconnect"
        self.comboBox.addItems(availablePorts["hardware"])
        #self.comboBox.insertSeparator(len(availablePorts["hardware"])+1)
        self.comboBox.addItem("") # Not only a more visible seaparator than the Qt one, but also doubles as "disconnect"
        self.comboBox.addItems(availablePorts["software"])
