/*
 * Copyright (C) 2014 Guitarix project MOD project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * --------------------------------------------------------------------------
 */


#include <cstdlib>
#include <cmath>
#include <iostream>
#include <cstring>
#include <unistd.h>

///////////////////////// MACRO SUPPORT ////////////////////////////////

#define __rt_func __attribute__((section(".rt.text")))
#define __rt_data __attribute__((section(".rt.data")))

///////////////////////// FAUST SUPPORT ////////////////////////////////

#define FAUSTFLOAT float
#ifndef N_
#define N_(String) (String)
#endif
#define max(x, y) (((x) > (y)) ? (x) : (y))
#define min(x, y) (((x) < (y)) ? (x) : (y))

#define always_inline inline __attribute__((always_inline))

template <int32_t N> inline float faustpower(float x)
{
  return powf(x, N);
}
template <int32_t N> inline double faustpower(double x)
{
  return pow(x, N);
}
template <int32_t N> inline int32_t faustpower(int32_t x)
{
  return faustpower<N/2>(x) * faustpower<N-N/2>(x);
}
template <>      inline int32_t faustpower<0>(int32_t x)
{
  return 1;
}
template <>      inline int32_t faustpower<1>(int32_t x)
{
  return x;
}

////////////////////////////// LOCAL INCLUDES //////////////////////////

#include "gx_saturate.h"        // define struct PortIndex
#include "gx_pluginlv2.h"   // define struct PluginLV2
#include "saturate.cc"    // dsp class generated by faust -> dsp2cc

////////////////////////////// PLUG-IN CLASS ///////////////////////////

namespace saturate {

class Gx_saturate_
{
private:
  // pointer to buffer
  float*      output;
  float*      input;
  // pointer to dsp class
  PluginLV2*  saturate;
  // private functions
  inline void run_dsp_(uint32_t n_samples);
  inline void connect_(uint32_t port,void* data);
  inline void init_dsp_(uint32_t rate);
  inline void connect_all__ports(uint32_t port, void* data);
  inline void activate_f();
  inline void clean_up();
  inline void deactivate_f();

public:
  // LV2 Descriptor
  static const LV2_Descriptor descriptor;
  // static wrapper to private functions
  static void deactivate(LV2_Handle instance);
  static void cleanup(LV2_Handle instance);
  static void run(LV2_Handle instance, uint32_t n_samples);
  static void activate(LV2_Handle instance);
  static void connect_port(LV2_Handle instance, uint32_t port, void* data);
  static LV2_Handle instantiate(const LV2_Descriptor* descriptor,
                                double rate, const char* bundle_path,
                                const LV2_Feature* const* features);
  Gx_saturate_();
  ~Gx_saturate_();
};

// constructor
Gx_saturate_::Gx_saturate_() :
  output(NULL),
  input(NULL),
  saturate(saturate::plugin()) {};

// destructor
Gx_saturate_::~Gx_saturate_()
{
  // just to be sure the plug have given free the allocated mem
  // it didn't hurd if the mem is already given free by clean_up()
  if (saturate->activate_plugin !=0)
    saturate->activate_plugin(false, saturate);
  // delete DSP class
  saturate->delete_instance(saturate);
};

///////////////////////// PRIVATE CLASS  FUNCTIONS /////////////////////

void Gx_saturate_::init_dsp_(uint32_t rate)
{
  saturate->set_samplerate(rate, saturate); // init the DSP class
}

// connect the Ports used by the plug-in class
void Gx_saturate_::connect_(uint32_t port,void* data)
{
  switch ((PortIndex)port)
    {
    case EFFECTS_OUTPUT:
      output = static_cast<float*>(data);
      break;
    case EFFECTS_INPUT:
      input = static_cast<float*>(data);
      break;
    default:
      break;
    }
}

void Gx_saturate_::activate_f()
{
  // allocate the internal DSP mem
  if (saturate->activate_plugin !=0)
    saturate->activate_plugin(true, saturate);
}

void Gx_saturate_::clean_up()
{
  // delete the internal DSP mem
  if (saturate->activate_plugin !=0)
    saturate->activate_plugin(false, saturate);
}

void Gx_saturate_::deactivate_f()
{
  // delete the internal DSP mem
  if (saturate->activate_plugin !=0)
    saturate->activate_plugin(false, saturate);
}

void Gx_saturate_::run_dsp_(uint32_t n_samples)
{
  saturate->mono_audio(static_cast<int>(n_samples), input, output, saturate);
}

void Gx_saturate_::connect_all__ports(uint32_t port, void* data)
{
  // connect the Ports used by the plug-in class
  connect_(port,data); 
  // connect the Ports used by the DSP class
  saturate->connect_ports(port,  data, saturate);
}

////////////////////// STATIC CLASS  FUNCTIONS  ////////////////////////

LV2_Handle 
Gx_saturate_::instantiate(const LV2_Descriptor* descriptor,
                            double rate, const char* bundle_path,
                            const LV2_Feature* const* features)
{
  // init the plug-in class
  Gx_saturate_ *self = new Gx_saturate_();
  if (!self)
    {
      return NULL;
    }

  self->init_dsp_((uint32_t)rate);

  return (LV2_Handle)self;
}

void Gx_saturate_::connect_port(LV2_Handle instance, 
                                   uint32_t port, void* data)
{
  // connect all ports
  static_cast<Gx_saturate_*>(instance)->connect_all__ports(port, data);
}

void Gx_saturate_::activate(LV2_Handle instance)
{
  // allocate needed mem
  static_cast<Gx_saturate_*>(instance)->activate_f();
}

void Gx_saturate_::run(LV2_Handle instance, uint32_t n_samples)
{
  // run dsp
  static_cast<Gx_saturate_*>(instance)->run_dsp_(n_samples);
}

void Gx_saturate_::deactivate(LV2_Handle instance)
{
  // free allocated mem
  static_cast<Gx_saturate_*>(instance)->deactivate_f();
}

void Gx_saturate_::cleanup(LV2_Handle instance)
{
  // well, clean up after us
  Gx_saturate_* self = static_cast<Gx_saturate_*>(instance);
  self->clean_up();
  delete self;
}

const LV2_Descriptor Gx_saturate_::descriptor =
{
  GXPLUGIN_URI "#_saturate_",
  Gx_saturate_::instantiate,
  Gx_saturate_::connect_port,
  Gx_saturate_::activate,
  Gx_saturate_::run,
  Gx_saturate_::deactivate,
  Gx_saturate_::cleanup,
  NULL
};


} // end namespace saturate

////////////////////////// LV2 SYMBOL EXPORT ///////////////////////////

extern "C"
LV2_SYMBOL_EXPORT
const LV2_Descriptor*
lv2_descriptor(uint32_t index)
{
  switch (index)
    {
    case 0:
      return &saturate::Gx_saturate_::descriptor;
    default:
      return NULL;
    }
}

///////////////////////////// FIN //////////////////////////////////////
