/*
 * Fonteditfs - A full-screen console font editor.
 * Copyright (C) 2002, 2003 Uri Shaked <uri@keves.org>.
 * Portions of the code were donated by amir shalem <amir@boom.org.il>.
 * Homepage: http://fonteditfs.sourceforge.net/
 *
 * Licensed under the BSD license:
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY ITS AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * $Id: font.c,v 1.2 2003/08/12 22:35:12 uri Exp $
 * Font I/O routines.
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include "font.h"

char font_error[BUFSIZ];

int font_uuload(char *file, char *fontbuf, int len) {
    FILE *f;
    char line[BUFSIZ], buf[len];
    int flag, i, j, off, linelen;
    f = fopen(file, "r");
    if (!f) {
	snprintf(font_error, sizeof(font_error),
		 "open: %s", strerror(errno));
	return -1;
    }

    // macro to decode a single char
#define UUDEC(ch) (((ch) - ' ') & 077)
    // find begin line.
    flag = 0;
    while (!feof(f)) {
	if (!fgets(line, sizeof(line), f))
	    break;
	if (sscanf(line, "begin %o %512s", &i, buf) == 2) {
	    flag++;
	    break;
	}
    }
    if (!flag) {
	snprintf(font_error, sizeof(font_error),
	    "not a uuencoded file (no begin)");
        fclose(f);
	return -1;
    }
    
    j = 0;
    for (;;) {
	if (!fgets(line, sizeof(line), f)) {
	    snprintf(font_error, sizeof(font_error),
		"uudecode: unexpected end of file");
	    return -2;
	}
	linelen = UUDEC(line[0]);
	if (linelen <= 0)
	    break;
	for (i = 0; i < linelen; i++) {
	    // decode the ith character
	    off = i / 3 * 4 + 1;
	    if (i % 3 == 0)
		buf[j++] = UUDEC(line[off]) << 2 | UUDEC(line[off + 1]) >> 4;
	    else if (i % 3 == 1)
		buf[j++] = UUDEC(line[off + 1]) << 4 | UUDEC(line[off + 2]) >> 2;
	    else
		buf[j++] = UUDEC(line[off + 2]) << 6 | UUDEC(line[off + 3]);
	    if (j == len) {
		// we've read enough data.
		memcpy(fontbuf, buf, len);
		fclose(f);
		return 0;
	    }
	}
    }
    if (j < len) {
        snprintf(font_error, sizeof(font_error), "file is too short");
	return -2;
    }
    memcpy(fontbuf, buf, len);
    fclose(f);
    return 0;
}

int font_load(char *file, char *fontbuf, int len) {
    char buf[len];
    int fd, rc;
    fd = open(file, O_RDONLY);
    if (fd < 0) {
	snprintf(font_error, sizeof(font_error),
		 "open: %s", strerror(errno));
	return -1;
    }
    rc = read(fd, buf, len);
    if (rc < sizeof(fontbuf)) {
	if (rc < 0) {
            snprintf(font_error, sizeof(font_error),
		     "read: %s", strerror(errno));
	    return -1;
	}
	if (rc == 0) {
            snprintf(font_error, sizeof(font_error),
	         "file is empty");
	    return -1;
	}
        snprintf(font_error, sizeof(font_error),
		"file is too short");
	return -1;
    }
    memcpy(fontbuf, buf, len);
    close(fd);
    return 0;
}

int font_save(char *filename, char *fontbuf, int len) {
    int fd, rc;
    fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, 0644);
    if (fd < 0) {
	snprintf(font_error, sizeof(font_error),
		 "open: %s", strerror(errno));
	return -1;
    }
    rc = write(fd, fontbuf, len);
    if (rc < sizeof(fontbuf)) {
        snprintf(font_error, sizeof(font_error),
		 "write: %s", strerror(errno));
	return -1;
    }
    close(fd);
    return 0;
}
